"SVAR2" <-
function(x, Ra = NULL, Rb = NULL, ra = NULL, rb = NULL, start = NULL, max.iter = 100, conv.crit = 0.1e-6, maxls = 1.0, lrtest = TRUE){
  if (!class(x) == "varest"){
    stop("\nPlease, provide an object of class 'varest',\n generated by function 'VAR()' as input for 'x'.\n")
  }
  ##
  ## Some preliminary settings and computations
  ##
  obs <- x$obs
  Sigma <- crossprod(x$resid) / obs
  n <- x$K
  nsq <- n^2
  ##
  ## Checking the SVAR-type and pre-setting vectors da and db
  ##
  if ((is.null(Ra)) && (is.null(Rb))){
    stop("\nAt least one matrix, either 'Ra' or 'Rb', must be non-null.\n")
  }
  if ((is.null(Ra)) && !(is.null(Rb))){
    bfree <- ncol(Rb)
    afree <- 0
    ra <- c(diag(n))
    svartype <- "B-model"
  } else if ((is.null(Rb)) && !(is.null(Ra))){
    afree <- ncol(Ra)
    bfree <- 0
    rb <- c(diag(n))
    svartype <- "A-model"
  } else {
    afree <- ncol(Ra)
    bfree <- ncol(Rb)
    svartype <- "AB-model"
  }
  l <- afree + bfree
  ##
  ## Defining the S matrix
  ##
  R <- matrix(0, nrow = 2*nsq, ncol = l)
  if(identical(afree, 0)){
    R[(nsq+1):(2*nsq), 1:l] <- Rb
  } else if(identical(bfree, 0)){
    R[1:nsq, 1:l] <- Ra
  } else if((!(is.null(afree)) && (!(is.null(bfree))))){
    R[1:nsq, 1:afree] <- Ra
    R[(nsq+1):(2*nsq), (afree+1):l] <- Rb
  }
  ##
  ## Defining the d vector
  ##
  r <- c(ra, rb)
  ##
  ## Commutation matrix Kkk and I_K^2 and I_K
  ##
  Kkk <- diag(nsq)[, c(sapply(1:n, function(i) seq(i, nsq, n)))]
  IK2 <- diag(nsq)
  IK <- diag(n)
  ##
  ## identification
  ##
  ifelse(is.null(start), gamma <- start <- rnorm(l), gamma <- start)
  vecab <- R %*% gamma + r
  A <- matrix(vecab[1:nsq], nrow = n, ncol = n)
  B <- matrix(vecab[(nsq + 1):(2*nsq)], nrow = n, ncol = n)
  v1 <- (IK2 + Kkk) %*% kronecker(t(solve(A) %*% B), solve(B))
  v2 <- -1.0 * (IK2 + Kkk) %*% kronecker(IK, solve(B))
  v <- cbind(v1, v2)
  idmat <- v %*% R
  ms <- t(v) %*% v
  auto <- eigen(ms)$values
  rni <- 0
  for(i in 1:l){
    if(auto[i] < 0.1e-10) rni <- rni + 1
  }
  if(identical(rni, 0)){
    if(identical(l, as.integer(n*(n + 1)/2))){
      ident <- paste("The", svartype, "is just identified.")
    } else {
      ident <- paste("The", svartype, "is over identified.")
    }
  } else {
    ident <- paste("The", svartype, "is unidentified. The non-identification rank is", rni, ".")
    stop(ident)
  }
  ##
  ## Scoring algorithm
  ##
  iters <- 0
  cvcrit <- conv.crit + 1.0
  while(cvcrit > conv.crit){
    z <- gamma
    vecab <- R %*% gamma + r
    A <- matrix(vecab[1:nsq], nrow = n, ncol = n)
    B <- matrix(vecab[(nsq + 1):(2*nsq)], nrow = n, ncol = n)
    Binv <- solve(B)
    Btinv <- solve(t(B))
    BinvA <- Binv %*% A
    infvecab.mat1 <- rbind(kronecker(solve(BinvA), Btinv), -1 * kronecker(IK, Btinv))
    infvecab.mat2 <- IK2 + Kkk
    infvecab.mat3 <- cbind(kronecker(t(solve(BinvA)), Binv), -1 * kronecker(IK, Binv))
    infvecab <- obs * (infvecab.mat1 %*% infvecab.mat2 %*% infvecab.mat3)
    infgamma <- t(R) %*% infvecab %*% R
    infgammainv <- solve(infgamma)
    scorevecBinvA <- obs * c(solve(t(BinvA))) - obs *(kronecker(Sigma, IK) %*% c(BinvA))
    scorevecAB.mat <- rbind(kronecker(IK, Btinv), -1.0 * kronecker(BinvA, Btinv))
    scorevecAB <- scorevecAB.mat %*% scorevecBinvA
    scoregamma <- t(R) %*% scorevecAB
    direction <- infgammainv %*% scoregamma
    length <- max(abs(direction))
    ifelse(length > maxls, lambda <- maxls/length, lambda <- 1.0)
    gamma <- gamma + lambda * direction    
    iters <- iters + 1
    z <- z - gamma
    cvcrit <- max(abs(z))
    if(iters >= max.iter){
      warning(paste("Convergence not achieved after", iters, "iterations. Convergence value:", cvcrit, "."))
      break
    }
  }
  vecab <- R %*% gamma + r
  colnames(A) <- colnames(x$y)
  rownames(A) <- colnames(A)
  colnames(B) <- colnames(A)
  rownames(B) <- colnames(A)
  ##
  ## Standard errors
  ##
  abSigma <- sqrt(diag((R %*% solve(infgamma) %*% t(R))))
  ASigma <- matrix(abSigma[1:nsq], nrow = n, ncol = n)
  BSigma <- matrix(abSigma[(nsq+1):(2*nsq)], nrow = n, ncol = n)
  colnames(ASigma) <- colnames(A)
  rownames(ASigma) <- rownames(A)
  colnames(BSigma) <- colnames(A)
  rownames(BSigma) <- rownames(A)
  Sigma.U <- solve(A) %*% B %*% t(B) %*% t(solve(A))
  ##
  ## LR Overidentification test
  ##
  LRover <- NULL
  if(lrtest){
    degrees <- 2 * n^2 - l - 2 * n^2 + 0.5 * n * (n + 1)
    if(identical(degrees, 0)){
      warning(paste("The", svartype, "is just identified. No test possible."))
    } else {
      rSigma <- solve(A) %*% B %*% t(B) %*% t(solve(A))
      det1 <- det(rSigma)
      det2 <- det(Sigma)
      STATISTIC <- (log(det1) - log(det2)) * obs
      names(STATISTIC) <- "Chi^2"
      PARAMETER <- degrees
      names(PARAMETER) <- "df"
      PVAL <- 1 - pchisq(STATISTIC, df = PARAMETER)
      METHOD <- "LR overidentification"
      LRover <- list(statistic = STATISTIC, parameter = PARAMETER, p.value = PVAL, method = METHOD, data.name = deparse(substitute(x)))
      class(LRover) <- "htest"
    }
  }
  result <- list(A = A, Ase = ASigma, B = B, Bse = BSigma, LRIM = NULL, Sigma.U = Sigma.U * 100, LR = LRover, opt = NULL, start = start, type = svartype, var = x, call = match.call())
  class(result) <- "svarest"
  return(result)
}
