% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forecast.R
\name{forecast}
\alias{forecast}
\title{Train an univariate time series forecasting model and make forecasts}
\usage{
forecast(
  timeS,
  h,
  lags = NULL,
  method = "knn",
  param = NULL,
  efa = NULL,
  tuneGrid = NULL,
  preProcess = list("additive")
)
}
\arguments{
\item{timeS}{A time series of class \code{ts} or a numeric vector.}

\item{h}{A positive integer. Number of values to be forecast into the future,
i.e., forecast horizon.}

\item{lags}{An integer vector, in increasing order, expressing the lags used
as autoregressive variables. If the default value (\code{NULL}) is provided, a
suitable vector is chosen.}

\item{method}{A string indicating the method used for training and
forecasting. Allowed values are:
\itemize{
\item \code{"knn"}: k-nearest neighbors (the default)
\item \code{"rt"}: regression trees
\item \code{"mt"}:  model trees
\item \code{"bagging"}
\item \code{"rf"}: random forest.
}

See details for a brief explanation of the models. It is also possible to
use your own regression model, in that case a function explaining how to
build your model must be provided, see the vignette for further details.}

\item{param}{A list with parameters for the underlying function that builds
the model. If the default value (\code{NULL}) is provided, the model is fitted
with its default parameters. See details for the functions used to train the
models.}

\item{efa}{A character value indicating the kind of method used to estimate
the forecast accuracy of the model using the time series. If the default
value (\code{NULL}) is provided, no estimation is done. Possible values are
\code{"rolling"} and \code{"fixed"}, indicating if rolling or fixed origin evaluation
is done. To estimate forecast accuracy the last \code{h} values of the time
series are used as test set and the previous values as training set.}

\item{tuneGrid}{A data frame with possible tuning values. The columns are
named the same as the tuning parameters. The estimation of forecast accuracy
is done as explained for the \code{efa} parameter. Rolling or fixed origin
evaluation is done according to the value of the \code{efa} parameter (fixed if
NULL). The best combination of parameters is used to train the model with
all the historical values of the time series.}

\item{preProcess}{A list indicating the preprocessings or transformations.
Currently, the length of the list must be 1 (only one preprocessing). If
\code{NULL} no preprocessing is applied. The element of the list is a character
value indicating what transformation is applied. By default (\code{"additive"})
an additive transformation is done. It is also possible a multiplicative
transformation (\code{"multiplicative"}). These transformations are recommended
if the time series has a trend. Also, taking differences is allowed using
the \code{\link[=differences]{differences()}} function.}
}
\value{
An S3 object of class \code{utsf}, basically a list with, at least, the
following components: \item{\code{ts}}{The time series being forecast.}
\item{\code{features}}{A data frame with the features of the training set. The
column names of the data frame indicate the autoregressive lags.}
\item{\code{targets}}{A vector with the targets of the training set.}
\item{\code{lags}}{An integer vector with the autoregressive lags.}
\item{\code{model}}{The regression model used recursively to make the forecast.}
\item{\code{pred}}{An object of class \code{ts} and length \code{h} with the forecast.}
\item{\code{efa}}{This component is included if forecast accuracy is estimated.
A vector with estimates of forecast accuracy according to different
forecast accuracy measures.}
\item{\code{tuneGrid}}{This component is included if the tuneGrid parameter has
been used. A data frame in which each row contains estimates of forecast
accuracy for a combination of tuning parameters.}
}
\description{
This function trains a model from the historical values of a time series using
an autoregressive approach: the targets are the historical values and the
features of the targets their lagged values. Then, the trained model is used
to predict the future values of the series using a recursive strategy.
}
\details{
The functions used to build and train the model are:
\itemize{
\item KNN: In this case no model is built and the function \code{\link[FNN:knn.reg]{FNN::knn.reg()}} is
used to predict the future values of the time series.
\item Regression trees: Function \code{\link[rpart:rpart]{rpart::rpart()}} to build the model and the
method \code{\link[rpart:predict.rpart]{rpart::predict.rpart()}} associated with the trained model to forecast
the future values of the time series.
\item Model trees: Function \code{\link[Cubist:cubist.default]{Cubist::cubist()}} to build the model and the
method \code{\link[Cubist:predict.cubist]{Cubist::predict.cubist()}} associated with the trained model to
forecast the future values of the time series.
\item Bagging: Function \code{\link[ipred:bagging]{ipred::bagging()}} to build the model and the
method \code{\link[ipred:predict.bagging]{ipred::predict.regbagg()}} associated with the trained model to
forecast the future values of the time series.
\item Random forest: Function \code{\link[ranger:ranger]{ranger::ranger()}} to build the model and the
method \code{\link[ranger:predict.ranger]{ranger::predict.ranger()}} associated with the trained model to
forecast the future values of the time series.
}
}
\examples{
## Forecast time series using k-nearest neighbors
f <- forecast(AirPassengers, h = 12, method = "knn")
f$pred
library(ggplot2)
autoplot(f)

## Using k-nearest neighbors changing the default k value
forecast(AirPassengers, h = 12, method = "knn", param = list(k = 5))$pred

## Using your own regression model

# Function to build the regression model
my_knn_model <- function(X, y) {
  structure(list(X = X, y = y), class = "my_knn")
}
# Function to predict a new example
predict.my_knn <- function(object, new_value) {
  FNN::knn.reg(train = object$X, test = new_value, y = object$y)$pred
}
forecast(AirPassengers, h = 12, method = my_knn_model)$pred

## Estimating forecast accuracy of the model
f <- forecast(UKgas, h = 4, lags = 1:4, method = "rf", efa = "rolling")
f$efa

## Estimating forecast accuracy of different tuning parameters
f <- forecast(UKgas, h = 4, lags = 1:4, method = "knn", tuneGrid = expand.grid(k = 1:5))
f$tuneGrid

## Forecasting a trending series
# Without any preprocessing or transformation
f <- forecast(airmiles, h = 4, method = "knn", preProcess = NULL)
autoplot(f)

# Applying the additive transformation (default)
f <- forecast(airmiles, h = 4, method = "knn")
autoplot(f)
}
