context("Misc. utility functions")

test_that("We can concatenate units if they have the same unit", {
  x <- 1:4 * make_unit("m")
  y <- 5:8 * make_unit("m")
  z <- c(x, y)
  
  expect_equal(length(z), length(x) + length(y))
  expect_equal(x, z[1:4])
  expect_equal(y, z[1:4 + 4])
})

test_that("We can't concatenate units if they have different units", {
  x <- 1:4 * make_unit("m")
  y <- 5:8 * make_unit("s")
  expect_error(c(x, y))
})

test_that("We can concatenate units if their units can be converted", {
  x <- 1:4 * make_unit("m")
  y <- 5:8 * make_unit("km")
  z <- c(x, y)
  
  expect_equal(length(z), length(x) + length(y))
  expect_equal(as.character(units(z)), "m")
  expect_equal(x, z[1:4])
  expect_equal(as_units(y, units(make_unit("m"))), z[1:4 + 4])
})

test_that("We can use diff on a units object", {
  x = 1:10 * make_unit("m")
  y = rep(1,9) * make_unit("m")
  expect_equal(diff(x), y)
})

test_that("type_sum is available for units objects", {
  library(tibble)
  expect_equal(type_sum(make_unit("m")), "units")
})

test_that("parse_unit works", {
  kg = make_unit("kg")
  m = make_unit("m")
  s = make_unit("s")
  u0 = kg/m/m/s
  u = parse_unit("kg m-2 s-1")
  expect_equal(u, u0)
  J = make_unit("J")
  u0 = with(ud_units, kg*kg*kg*m*m*J/s/s/s/s/s)
  u = parse_unit("kg3 m2 s-5 J")
  expect_equal(u, u0)
})

test_that("deparse_unit works", {
  str = "kg m-2 s-1"
  u = parse_unit(str)
  str0 = deparse_unit(u)
  expect_equal(str, str0)
})

test_that("we can provide a symbol to as_units and make it look in ud_units", {
  five_ha <- as_units(5, ha) # ha pulled from ud_units
  expect_equal(as.numeric(five_ha), 5)
  expect_equal(units(five_ha), units(ud_units$ha))
  
  ha <- make_unit("m") # make sure that user-defined units overrule
  five_ha <- as_units(5, ha) # ha pulled from ud_units
  expect_equal(as.numeric(five_ha), 5)
  expect_equal(units(five_ha), units(ud_units$m))
  
})

test_that("set_units(x, u) is a short form for x * with(ud_units, u)", {
  expect_equal(set_units(1:10, m/s), 1:10 * with(ud_units, m/s)) # not identical - why?
  x = set_units(1:5, m/s)
  y = x
  units(y) = set_units(1, km/h)
  expect_identical(y, set_units(x, km/h))
})

test_that("rep.units works", {
  expect_equal(rep(set_units(1:2, m/s), 2), set_units(c(1,2,1,2), m/s))
})

test_that("set_units works with symbols in character data, and resolves names", {
  skip_on_os("windows") # encoding issue with degree:

  deg = "°C"
  expect_equal(set_units(1:2, deg), set_units(1:2, "degree_C"))
  expect_equal(set_units(1:2, deg), set_units(1:2, "degree_Celsius"))
  expect_equal(set_units(1:2, "degree_C"), set_units(1:2, "degree_Celsius"))
  expect_equal(set_units(1:2, degree_C), set_units(1:2, degree_Celsius))
  expect_equal(set_units(1:2, deg), set_units(1:2, degree_Celsius))
  x = set_units(1:3, km)
  y <- set_units(x, "meter")
  expect_equal(y, set_units(c(1000,2000,3000), m))
})

test_that("all.equal works", {
  expect_true(all.equal(set_units(1, m/s), set_units(3.6, km/h)))
  expect_true(set_units(1, m/s) == set_units(3.6, km/h))
  expect_true(all.equal(set_units(3.6, km/h), set_units(1, m/s)))
  expect_false(set_units(3.6, km/h) == set_units(1, m/s))
})
