#' Accessing original forecast and observation data for triptych objects
#'
#' @param x An object from which the relevant information should be extracted.
#' @param ... Additional arguments passed to other methods.
#'
#' @return
#' For `forecasts()`: A tibble of the original forecasts in long format.
#'
#' For `observations()`: A vector of the observations.
#' 
#' @examples
#' data(ex_binary, package = "triptych")
#' tr <- triptych(ex_binary)
#' 
#' forecasts(tr)
#' observations(tr) 
#'
#' @seealso [estimates()], [regions()]
#' @name accessors
NULL

#' @rdname accessors
#' @export
forecasts <- function(x, ...) {
  UseMethod("forecasts")
}
#' @rdname accessors
#' @export
observations <- function(x, ...) {
  UseMethod("observations")
}

#' Accessing diagnostic estimate data
#'
#' @param x An object from which the estimate information should be extracted.
#' @param at A vector of thresholds where `x` should be evaluated.
#' @param ... Additional arguments passed to other methods.
#'
#' @return
#' A tibble with the relevant information describing
#'   the diagnostic estimate
#'   (Murphy curve, reliability curve, ROC curve, score decomposition)
#'   for all supplied forecasting methods.
#'   
#' For a Murphy curve, a tibble with columns: `forecast`, `knot` (the threshold value), `limit` ("left" or "right" in `knot`, only present if `at = NULL`), `mean_score`.
#' 
#' For a reliability curve, a tibble with columns: `forecast`, `CEP`, `x` (the knots of the isotonic regression estimate).
#' 
#' For a ROC curve, a tibble with columns: `forecast`, `FAR` (false alarm rate), `HR` (hit rate).
#' 
#' For a MCBDSC decomposition, a tibble with columns: `forecast`, `mean_score`, `MCB` (miscalibration), `DSC` (discrimination), `UNC` (uncertainty).
#'   
#' @examples
#' data(ex_binary, package = "triptych")
#' tr <- triptych(ex_binary)
#' 
#' estimates(tr$murphy)
#' estimates(tr$reliability)
#' estimates(tr$roc)
#' estimates(tr$mcbdsc)
#'
#' @seealso [regions()], [forecasts()], [observations()]
#' @export
estimates <- function(x, at, ...) {
  UseMethod("estimates")
}

#' Accessing confidence/consistency region data
#'
#' @param x An object from which the region information should be extracted.
#' @param ... Additional arguments passed to other methods.
#'
#' @return
#' A tibble with the relevant information for the
#'   uncertainty quantification of the chosen diagnostic
#'   (Murphy curve, reliability curve, ROC curve, score decomposition)
#'   for all supplied forecasting methods.
#'   
#' For a Murphy curve, a tibble with columns: `forecast`, `threshold`, `lower`, `upper`, `method`, `level`.
#' 
#' For a reliability curve, a tibble with columns: `forecast`, `x` (forecast values), `lower`, `upper`, `method`, `level`.
#' 
#' For a ROC curve, a tibble with columns: `forecast`, `FAR` (false alarm rate), `HR` (hit rate), `method`, `level`.
#' This tibble is twice as long as those for Murphy and reliability curves,
#' since the FAR-HR pairs are ordered to describe a polygon, generated by pointwise confidence
#' intervals along diagonal lines with slope \eqn{-\pi_0/\pi_1}.
#' Here, \eqn{\pi_1 = 1 - \pi_0} is the unconditional event probability.
#'   
#' @examples
#' data(ex_binary, package = "triptych")
#' 
#' # Bootstrap resampling is expensive
#' # (the number of bootstrap samples is small to keep execution times short)
#' 
#' tr <- triptych(ex_binary) |>
#'   dplyr::slice(1, 9) |>
#'   add_confidence(level = 0.9, method = "resampling_cases", n_boot = 20)
#' 
#' regions(tr$murphy)
#' regions(tr$reliability)
#' regions(tr$roc)
#'
#' @seealso [estimates()], [forecasts()], [observations()]
#' @export
regions <- function(x, ...) {
  UseMethod("regions")
}

has_regions <- function(x, ...) {
  UseMethod("has_regions")
}

has_compatible_observations <- function(x, y) {
  isTRUE(all.equal(observations(x), observations(y)))
}

eval_diag <- function(x, at, ...) {
  UseMethod("eval_diag")
}
