% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linalg.R
\name{linalg_eig}
\alias{linalg_eig}
\title{Computes the eigenvalue decomposition of a square matrix if it exists.}
\usage{
linalg_eig(A)
}
\arguments{
\item{A}{(Tensor): tensor of shape \verb{(*, n, n)} where \code{*} is zero or more batch dimensions
consisting of diagonalizable matrices.}
}
\value{
A list \verb{(eigenvalues, eigenvectors)} which corresponds to \teqn{\Lambda} and \teqn{V} above.
\code{eigenvalues} and \code{eigenvectors} will always be complex-valued, even when \code{A} is real. The eigenvectors
will be given by the columns of \code{eigenvectors}.
}
\description{
Letting \teqn{\mathbb{K}} be \teqn{\mathbb{R}} or \teqn{\mathbb{C}},
the \strong{eigenvalue decomposition} of a square matrix
\teqn{A \in \mathbb{K}^{n \times n}} (if it exists) is defined as
}
\details{
\Sexpr[results=rd, stage=build]{torch:::math_to_rd("
A = V \\\\operatorname{diag}(\\\\Lambda) V^{-1}\\\\mathrlap{\\\\qquad V \\\\in \\\\mathbb{C}^{n \\\\times n}, \\\\Lambda \\\\in \\\\mathbb{C}^n}
")}

This decomposition exists if and only if \teqn{A} is \code{diagonalizable}_.
This is the case when all its eigenvalues are different.
Supports input of float, double, cfloat and cdouble dtypes.
Also supports batches of matrices, and if \code{A} is a batch of matrices then
the output has the same batch dimensions.
}
\note{
The eigenvalues and eigenvectors of a real matrix may be complex.
}
\section{Warning}{

\itemize{
\item This function assumes that \code{A} is \code{diagonalizable}_ (for example, when all the
eigenvalues are different). If it is not diagonalizable, the returned
eigenvalues will be correct but \teqn{A \neq V \operatorname{diag}(\Lambda)V^{-1}}.
\item The eigenvectors of a matrix are not unique, nor are they continuous with respect to
\code{A}. Due to this lack of uniqueness, different hardware and software may compute
different eigenvectors.
This non-uniqueness is caused by the fact that multiplying an eigenvector by a
non-zero number produces another set of valid eigenvectors of the matrix.
In this implmentation, the returned eigenvectors are normalized to have norm
\code{1} and largest real component.
\item Gradients computed using \code{V} will only be finite when \code{A} does not have repeated eigenvalues.
Furthermore, if the distance between any two eigenvalues is close to zero,
the gradient will be numerically unstable, as it depends on the eigenvalues
\teqn{\lambda_i} through the computation of
\teqn{\frac{1}{\min_{i \neq j} \lambda_i - \lambda_j}}.
}
}

\examples{
if (torch_is_installed()) {
a <- torch_randn(2, 2)
wv <- linalg_eig(a)
}
}
\seealso{
\itemize{
\item \code{\link[=linalg_eigvals]{linalg_eigvals()}} computes only the eigenvalues. Unlike \code{\link[=linalg_eig]{linalg_eig()}}, the gradients of
\code{\link[=linalg_eigvals]{linalg_eigvals()}} are always numerically stable.
\item \code{\link[=linalg_eigh]{linalg_eigh()}} for a (faster) function that computes the eigenvalue decomposition
for Hermitian and symmetric matrices.
\item \code{\link[=linalg_svd]{linalg_svd()}} for a function that computes another type of spectral
decomposition that works on matrices of any shape.
\item \code{\link[=linalg_qr]{linalg_qr()}} for another (much faster) decomposition that works on matrices of
any shape.
}

Other linalg: 
\code{\link{linalg_cholesky_ex}()},
\code{\link{linalg_cholesky}()},
\code{\link{linalg_det}()},
\code{\link{linalg_eigh}()},
\code{\link{linalg_eigvalsh}()},
\code{\link{linalg_eigvals}()},
\code{\link{linalg_householder_product}()},
\code{\link{linalg_inv_ex}()},
\code{\link{linalg_inv}()},
\code{\link{linalg_lstsq}()},
\code{\link{linalg_matrix_norm}()},
\code{\link{linalg_matrix_power}()},
\code{\link{linalg_matrix_rank}()},
\code{\link{linalg_multi_dot}()},
\code{\link{linalg_norm}()},
\code{\link{linalg_pinv}()},
\code{\link{linalg_qr}()},
\code{\link{linalg_slogdet}()},
\code{\link{linalg_solve_triangular}()},
\code{\link{linalg_solve}()},
\code{\link{linalg_svdvals}()},
\code{\link{linalg_svd}()},
\code{\link{linalg_tensorinv}()},
\code{\link{linalg_tensorsolve}()},
\code{\link{linalg_vector_norm}()}
}
\concept{linalg}
