% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maps.R
\name{createMap}
\alias{createMap}
\title{Locate map, geocode data, then plot both.}
\usage{
createMap(data, maptype = "terrain", mapColor = c("color", "bw"),
  source = c("google", "osm", "stamen", "cloudmade"), location = NULL,
  locator = "center", boxBorderMargin = 10, zoom = NULL,
  locationName = NULL, lonName = "LONGITUDE", latName = "LATITUDE",
  metricName = NULL, metrics = metricName, labelName = NULL,
  scaleRange = c(1, 6), shape = 19, shapeColour = "red",
  shapeAlpha = 0.5, shapeStroke = 0.5, scaleSize = TRUE,
  textColour = "black", textFamily = "mono", textFace = "plain",
  textSize = 4, facet = NULL, ncol = 1, facetScales = "fixed",
  geocodeFun = memoise::memoise(geocode), getmapFun = get_map,
  urlonly = FALSE, api_key = NULL, baseSize = 12, baseFamily = "sans",
  title = NULL, legendPosition = "right", metricGuides = c("legend",
  "colorbar"), defaultTheme = theme_bw(base_size = baseSize),
  themeExtra = NULL)
}
\arguments{
\item{data}{data frame with artifacts and their locations and metric(s) to be placed 
on the map. If location name is provided (with \code{locationName}) then it is used to 
gecode artifacts first. If not location then longitude and latitude must be provided.
It is caller's responsibility adjust locations with value of \code{zoom} parameter to
fit artifacts on the map.}

\item{maptype}{map theme as defined in \code{\link{get_map}}. options available are 'terrain', 
'satellite', 'roadmap', and 'hybrid'}

\item{mapColor}{color (\code{'color'}) or black-and-white (\code{'bw'})}

\item{source}{Google Maps ('google'), OpenStreetMap ('osm'), Stamen Maps ('stamen'), or 
CloudMade maps ('cloudmade')}

\item{location}{location of the map: longitude/latitude pair (in that order), or 
left/bottom/right/top bounding box: 'center' uses 2 value vector for the center of the map, 
while 'box' uses 4 value vector as left/bottom/right/top. If missing then function will 
derive map location using parameter \code{locator} and the \code{data}.}

\item{locator}{in absence of \code{location} specifies how to use data to determine map 
location: when 'center' then function averages out data point longitude and latitude values 
to get approximate cneter for the map; when 'box' it will use min/max of longitude and 
latitude values to determine bounding box: left/bottom/right/top. 
If parameter \code{locationName} is specified then function will geocode values from this 
column first. If paramter \code{locationName} is missing then it assumes that data is already 
geocoded and stored in the columns with the names \code{lonName} and \code{latName}.}

\item{boxBorderMargin}{margin size in percent of box sizes to increase box when computed 
from data locations.}

\item{zoom}{map zoom as defined in \code{\link{get_map}}: an integer from 3 (continent) 
to 21 (building), default value 10 (city). Properly setting \code{zoom} for each map is 
responsibility of a caller. Zoom is optional when using bounding box location specification.}

\item{locationName}{vector of the column names with address or name to geocode its location (find 
latitude and longitude) using \code{\link{geocode}} (see package \pkg{ggmap}). 
When \code{locationName} is specified then parameters \code{lonName} and \code{latName} are ignored. 
Multiple column names are used in order of appearance: geocoding tries 1st column's values first, 
then, for the data points that didn't get resolved, it tries the 2d column's values, and so on.}

\item{lonName}{name of the column with longitude value. This value (in combination with value 
from column \code{latName}) is used to place each data point on the map. This parameter is 
ignored if \code{locationName} is defined.}

\item{latName}{name of the column with latitude value. This value (in combination with value 
from column \code{lonName}) is used to place each data point on the map. This parameter is 
ignored if \code{locationName} is defined.}

\item{metricName}{(deprecated) Use parameter \code{metrics} instead.}

\item{metrics}{character vector of column names with metric values to scale shapes placed on map. First 
metric corresponds to the size (or area depending on \code{scaleSize}), second to the fill gradient.
See also \code{scaleSize} and \code{shapeStroke}.}

\item{labelName}{name of the column to use for the artifact label text when displaying data.}

\item{scaleRange}{a numeric vector of lenght 2 that specifies the minimum and maximum size 
of the plotting symbol after transformation (see parameter \code{range} of \code{\link{scale_size}}).}

\item{shape}{type of shape to use.}

\item{shapeColour}{color of metric artifacts placed on map.}

\item{shapeAlpha}{transparency of an artifact shape expressed as a fraction between 0 (complete 
transparency) and 1 (complete opacity).}

\item{shapeStroke}{border width of an artifact shape. Remember, that in \code{ggplot2} \code{size} and \code{stroke}
are additive so a point with \code{size = 5} and \code{stroke = 5} will have a diameter of 10mm. \code{\link{createMap}} 
maps \code{metrics[[1]]} to shape size.}

\item{scaleSize}{logical if TRUE then scale artifact shapes by size (radius), otherwise scale shape's 
area (artifact shapes scaling always uses \code{metrics[[1]]} values).}

\item{textColour}{color of artifact labels on map.}

\item{textFamily}{font family (when available) to use for artfiact labels.}

\item{textFace}{font style to apply to artifact labels: 'plain' (default), 'bold', 'italic', or 
'bold.italic'.}

\item{textSize}{font size of artifact labels.}

\item{facet}{name of a column to divide plot into facets for specificed parameter (defualt is 
NULL - no facets). If facet is single value then facet wrap applied (see \code{\link{facet_wrap}}), 
otherwise facet grid (see \code{\link{facet_grid}} with 1st 2 values of the vector.}

\item{ncol}{number of facet columns (applies when single facet column supplied only - see 
parameter \code{facet}).}

\item{facetScales}{Are scales shared across all facets: "fixed" - all are the same, "free_x" - vary 
across rows (x axis), "free_y" - vary across columns (Y axis) (default), "free" - both rows and 
columns (see in \code{facet_wrap} parameter \code{scales}).}

\item{geocodeFun}{geocode function. Default is \code{\link{geocode}} but due to Google API 
restrictions use memoised version, e.g. \code{memoise(geocode)}, instead (see package \pkg{memoise}).}

\item{getmapFun}{get map function. Defayult is \code{\link{get_map}} but due to map APIs restrictions 
use memoised version, e.g. \code{memose(get_map)}, instead (see package \pkg{memoise}).}

\item{urlonly}{return url only.}

\item{api_key}{an api key for cloudmade maps.}

\item{baseSize}{base font size.}

\item{baseFamily}{base font family.}

\item{title}{plot title.}

\item{legendPosition}{the position of metric guide ("left", "right", "bottom", "top", or two-element 
numeric vector; "none" is no legend).}

\item{metricGuides}{list or vector with names of guide objects, or objects themselves, for up to 2 metrics. Typical 
guides are \code{"legend"} or \code{"colorbar"} names and \code{\link[ggplot2]{guide_legend}} or
\code{\link[ggplot2]{guide_colorbar}} objects.}

\item{defaultTheme}{plot theme to use, default is \code{theme_bw}.}

\item{themeExtra}{any additional \code{ggplot2} theme attributes to add.}
}
\description{
createMap is a smart function that places data artifact on the map. If 
necessary it geocodes the data, locates map that fits all data artifacts, 
and plots the map with the data shapes sized and colored using metrics.
}
\details{
Geocoding:
If parameter \code{locationName} is missing then no geocoding is possible.
In that case parameters \code{lonName} and \code{latName} must contain 
names of columns with longitude and latitude information assigned to 
each data artifact (data point). 
If parameter \code{locationName} is defined then geocoding attempts 
to use values from the column with this name. Function \code{geocodeFun}
specifies geocoding function (with default \code{\link{geocode}}
from \link{ggmap} package). To speed up processing and avoid hitting 
global limit on Google Map API use memoised version of this function:
\code{memoise(geocode)} (see \code{\link[memoise]{memoise}}).

Map Locating:
Function operates in 2 modes: explicit map location mode and implicit mode.
In explicit mode value \code{location} locates the map using one
of two supported formats. If it is a 2-value vector then it contains a 
center of the map. If it is 4-value vector then it contains bounding box
coordinates: left/bottom/right/top.
In implicit mode, when \code{location} is missing, fuction uses parameters
\code{locator} and \code{data} to locate the map. If \code{locator} is
equal to \code{'center'} then it centers map by averaging longitude and
latitude values of all data artifacts. If \code{locator} is equal to \code{'box'}
then it determines min/max values of longitutude and latitude of all data 
artifacts and locates the map by corresponding bounding box.
Note that both modes support require explicit parameter \code{zoom} if 
applicable.

Map Types: variety of map avaiable are from several public sources: google, 
OpenStreetMap, Stamen, and CloudMade maps. The options and terms for each are 
different. For example, not all sources support both color and black-and-white options, 
or map types terrain, satellite, roadmap or hybrid.  
Note that in most cases by using Google source you are agreeing to the Google Maps API 
Terms of Service at https://developers.google.com/maps/terms.

Shapes: data artifacts are shapes placed over the map. Their size and fill are scaled using
values in \code{metrics} columns and their location is determined either by 
geocoding values from \code{locationName} column or with longitude and latitude values
stored in \code{lonName} and \code{latName} columns.

Labels: If \code{labelName} is specified then column with such name contains text 
labels to place on the map (using the same locations as for the shapes).
}
\examples{
if(interactive()){
# initialize connection to Lahman baseball database in Aster 
conn = odbcDriverConnect(connection="driver={Aster ODBC Driver};
                         server=<dbhost>;port=2406;database=<dbname>;uid=<user>;pwd=<pw>")

data = computeAggregates(channel = conn, "teams_enh",
                aggregates = c("min(name) name", "min(park) park", "avg(rank) rank", 
                               "avg(attendance) attendance"),
                by = c("name || ', ' || park teamname", "lgid", "teamid", "decadeid"))
               
geocodeFun = memoise::memoise(ggmap::geocode)
getMapFun = memoise::memoise(ggmap::get_map)

createMap(data=data[data$decadeid>=2000,], 
          source = "stamen", maptype = "watercolor", zoom=4, 
          facet=c("lgid", "decadeid"),
          locationName=c('teamname','name'), 
          metrics=c('rank', 'attendance'), shape = 21,
          labelName='name', shapeColour="blue", scaleRange = c(2,12), textColour="black",
          title='Game Attendance by Decade and League (yearly, 2000-2012)',
          geocodeFun=geocodeFun, getmapFun = getMapFun)
}
}

