% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rtmvnorm.R
\name{rtmvnorm}
\alias{rtmvnorm}
\title{Random Generation for Truncated Multivariate Normal}
\usage{
rtmvnorm(
  mean,
  sigma,
  blc = NULL,
  lower,
  upper,
  init = NULL,
  burn = 10,
  n = NULL
)
}
\arguments{
\item{mean}{\code{n x p} matrix of means. The number of rows is the number 
of observations. The number of columns is the dimension of the problem.}

\item{sigma}{\code{p x p} covariance matrix.}

\item{blc}{\code{m x p} matrix of coefficients for linear inequality 
constraints. If \code{NULL}, the \code{p x p} identity matrix will be used.}

\item{lower}{\code{n x m} or \code{1 x m} matrix of lower bounds for 
truncation.}

\item{upper}{\code{n x m} or \code{1 x m} matrix of upper bounds for 
truncation.}

\item{init}{\code{n x p} or \code{1 x p} matrix of initial values. 
If \code{NULL}, default initial values will be generated.}

\item{burn}{number of burn-in iterations. Defaults to 10.}

\item{n}{number of random samples when \code{mean} is a vector.}
}
\value{
Returns an \code{n x p} matrix of random numbers following the 
  specified truncated multivariate normal distribution.
}
\description{
Draws from truncated multivariate normal distribution subject to 
linear inequality constraints represented by a matrix.
}
\examples{
# Example 1: full rank blc
d = 3;
rho = 0.9;
sigma = matrix(0, d, d);
sigma = rho^abs(row(sigma) - col(sigma));
blc = diag(1,d);
n = 1000;
mean = matrix(rep(1:d,n), nrow=n, ncol=d, byrow=TRUE);
set.seed(1203)
result = rtmvnorm(mean, sigma, blc, -1, 1, burn=50)
apply(result, 2, summary)

# Example 2: use the alternative form of input
set.seed(1203)
result = rtmvnorm(mean=1:d, sigma, blc, -1, 1, burn=50, n=1000)
apply(result, 2, summary)

# Example 3: non-full rank blc
d = 3;
rho = 0.5;
sigma = matrix(0, d, d);
sigma = rho^abs(row(sigma) - col(sigma));
blc = matrix(c(1,1,1,0,1,0,1,0,1), ncol=d);
n = 100;
mean = matrix(rep(1:d,n), nrow=n, ncol=d, byrow=TRUE);
set.seed(1228)
result = rtmvnorm(mean, sigma, blc, -1, 1, burn=10)
apply(result, 2, summary)

# Example 4: non-full rank blc, alternative form of input
set.seed(1228)
result = rtmvnorm(mean=1:d, sigma, blc, -1, 1, burn=10, n=100)
apply(result, 2, summary) 

# Example 5: means, lower, or upper bounds differ across samples
d = 3;
rho = 0.5;
sigma = matrix(0, d, d);
sigma = rho^abs(row(sigma) - col(sigma));
blc = matrix(c(1,0,1,1,1,0), ncol=d, byrow=TRUE)
n = 100;
set.seed(3084)
mean = matrix(runif(n*d), nrow=n, ncol=d);
result = rtmvnorm(mean, sigma, blc, -1, 1, burn=50)
apply(result, 2, summary)

}
