% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binaryirt.R
\name{binary_irt}
\alias{binary_irt}
\title{Binary (Dichotomous) Item Response Theory Estimation}
\usage{
binary_irt(data, model = "2PL", method = "EM", control = list())
}
\arguments{
\item{data}{A N x J data.frame of dichotomous responses (0/1).}

\item{model}{String. "Rasch", "2PL" (2-Parameter Logistic), or "3PL" (3-Parameter Logistic).}

\item{method}{String. "EM" (Marginal Maximum Likelihood via Expectation-Maximization) or "MLE" (Joint Maximum Likelihood).}

\item{control}{A \code{list} of control parameters for the estimation algorithm:
\itemize{
\item \code{max_iter}: Maximum number of EM iterations (default = 100).
\item \code{converge_tol}: Convergence criterion for parameter change (default = 1e-4).
\item \code{theta_range}: Numeric vector of length 2 specifying the integration
grid bounds (default = c(-4, 4)).
\item \code{quad_points}: Number of quadrature points (default = 21).
\item \code{verbose}: Logical; if \code{TRUE}, prints progress to console.
}}
}
\value{
A list containing:
\itemize{
\item \code{item_params}: A data frame of estimated item parameters (discrimination, difficulty, guessing) and their standard errors.
\item \code{person_params}: A data frame of estimated person abilities (theta) and standard errors.
\item \code{model_fit}: A data frame containing fit statistics such as Akaike’s Information Criterion (AIC), the Bayesian Information Criterion (BIC), and Log-Likelihood.
\item \code{settings}: A list of control parameters used in the estimation.
}
}
\description{
Estimates item and person parameters for binary item response models
using either Marginal Maximum Likelihood or Joint Maximum Likelihood.
}
\examples{
  # # Simulate data
  set.seed(123)
  N <- 500; J <- 10
  true_theta <- rnorm(N)
  true_b <- seq(-2, 2, length.out=J)
  true_a <- runif(J, 0.8, 1.2)
  data_mat <- matrix(NA, N, J)
  for(i in 1:N) {
    p <- 1 / (1 + exp(-true_a * (true_theta[i] - true_b)))
   data_mat[i,] <- rbinom(J, 1, p)
  }
  df <- as.data.frame(data_mat)
  names(df) <- paste0("Q", 1:J)
  # # Run Function
  res <- binary_irt(df, model="2PL", method="EM")
  # View Results
  head(res$item_params)
  head(res$person_params)
  print(res$model_fit)
  \donttest{
  # --- Example 2: With Package Data ---
  data("ela1", package = "tirt")
  # Subset the first 30 columns (must use the object name 'data_binary')
  df <- ela1[, 1:30]
  # Run Function on package data
  real_res <- binary_irt(df, model="2PL", method="EM")
  head(real_res$item_params)
  }
}
