% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recipes-step_smooth.R
\name{step_smooth}
\alias{step_smooth}
\alias{tidy.step_smooth}
\title{Smoothing Transformation using Loess}
\usage{
step_smooth(
  recipe,
  ...,
  period = 30,
  span = NULL,
  degree = 2,
  names = NULL,
  role = "predictor",
  trained = FALSE,
  columns = NULL,
  skip = FALSE,
  id = rand_id("smooth")
)

\method{tidy}{step_smooth}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more numeric columns to be smoothed.
See \code{\link[recipes:selections]{recipes::selections()}} for more details.
For the \code{tidy} method, these are not currently used.}

\item{period}{The number of periods to include in the local smoothing.
Similar to window size for a moving average.
See details for an explanation \code{period} vs \code{span} specification.}

\item{span}{The span is a percentage of data to be included
in the smoothing window. Period is preferred for shorter windows
to fix the window size.
See details for an explanation \code{period} vs \code{span} specification.}

\item{degree}{The degree of the polynomials to be used.
Set to 2 by default for 2nd order polynomial.}

\item{names}{An optional character string that is the same
length of the number of terms selected by \code{terms}. These will be
the names of the \strong{new columns} created by the step.
\itemize{
\item If \code{NULL}, existing columns are transformed.
\item If not \code{NULL}, new columns will be created.
}}

\item{role}{For model terms created by this step, what analysis
role should they be assigned?. By default, the function assumes
that the new variable columns created by the original variables
will be used as predictors in a model.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{columns}{A character string of variables that will be
used as inputs. This field is a placeholder and will be
populated once \code{recipes::prep()} is used.}

\item{skip}{A logical. Should the step be skipped when the recipe is
baked by bake.recipe()? While all operations are baked when prep.recipe()
is run, some operations may not be able to be conducted on new data
(e.g. processing the outcome variable(s)). Care should be taken when
using skip = TRUE as it may affect the computations for subsequent operations.}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_smooth} object.}
}
\value{
For \code{step_smooth}, an updated version of recipe with
the new step added to the sequence of existing steps (if any).
For the \code{tidy} method, a tibble with columns \code{terms}
(the selectors or variables selected), \code{value} (the feature
names).
}
\description{
\code{step_smooth} creates a a \emph{specification} of a recipe
step that will apply local polynomial regression
to one or more a Numeric column(s). The effect is smoothing the time series
\strong{similar to a moving average without creating missing values or using partial smoothing.}
}
\details{
\strong{Smoother Algorithm}
This function is a \code{recipe} specification that wraps the \code{stats::loess()}
with a modification to set a fixed \code{period} rather than a percentage of
data points via a \code{span}.

\strong{Why Period vs Span?}
The \code{period} is fixed whereas the \code{span} changes as the number of observations change.

\strong{When to use Period?}
The effect of using a \code{period} is similar to a Moving Average where the Window Size
is the \emph{\strong{Fixed Period}}. This helps when you are trying to smooth local trends.
If you want a 30-day moving average, specify \code{period = 30}.

\strong{When to use Span?}
Span is easier to specify when you want a \emph{\strong{Long-Term Trendline}} where the
window size is unknown. You can specify \code{span = 0.75} to locally regress
using a window of 75\% of the data.

\strong{Warning - Using Span with New Data}
When using span on New Data, the number of observations is likely different than
what you trained with. This means the trendline / smoother can be vastly different
than the smoother you trained with.

\strong{Solution to Span with New Data}
Don't use \code{span}. Rather, use \code{period} to fix the window size.
This ensures that new data includes the same number of observations in the local
polynomial regression (loess) as the training data.
}
\examples{
library(recipes)
library(tidyverse)
library(tidyquant)
library(timetk)

# Training Data
FB_tbl <- FANG \%>\%
    filter(symbol == "FB") \%>\%
    select(symbol, date, adjusted)

# New Data - Make some fake new data next 90 time stamps
new_data <- FB_tbl \%>\%
    tail(90) \%>\%
    mutate(date = date \%>\% tk_make_future_timeseries(length_out = 90))

# ---- PERIOD ----

# Create a recipe object with a step_smooth()
rec_smooth_period <- recipe(adjusted ~ ., data = FB_tbl) \%>\%
    step_smooth(adjusted, period = 30)

# Bake the recipe object - Applies the Loess Transformation
training_data_baked <- bake(prep(rec_smooth_period), FB_tbl)

# "Period" Effect on New Data
new_data_baked <- bake(prep(rec_smooth_period), new_data)

# Smoother's fit on new data is very similar because
# 30 days are used in the new data regardless of the new data being 90 days
training_data_baked \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(color = "red", data = new_data_baked)

# ---- SPAN ----

# Create a recipe object with a step_smooth
rec_smooth_span <- recipe(adjusted ~ ., data = FB_tbl) \%>\%
    step_smooth(adjusted, span = 0.03)

# Bake the recipe object - Applies the Loess Transformation
training_data_baked <- bake(prep(rec_smooth_span), FB_tbl)

# "Period" Effect on New Data
new_data_baked <- bake(prep(rec_smooth_span), new_data)

# Smoother's fit is not the same using span because new data is only 90 days
# and 0.03 x 90 = 2.7 days
training_data_baked \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(color = "red", data = new_data_baked)

# ---- NEW COLUMNS ----
# Use the `names` argument to create new columns instead of overwriting existing

rec_smooth_names <- recipe(adjusted ~ ., data = FB_tbl) \%>\%
    step_smooth(adjusted, period = 30, names = "adjusted_smooth_30") \%>\%
    step_smooth(adjusted, period = 180, names = "adjusted_smooth_180") \%>\%
    step_smooth(adjusted, span = 0.75, names = "long_term_trend")

bake(prep(rec_smooth_names), FB_tbl) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line(alpha = 0.5) +
    geom_line(aes(y = adjusted_smooth_30), color = "red", size = 1) +
    geom_line(aes(y = adjusted_smooth_180), color = "blue", size = 1) +
    geom_line(aes(y = long_term_trend), color = "orange", size = 1)



}
\seealso{
Time Series Analysis:
\itemize{
\item Engineered Features: \code{\link[=step_timeseries_signature]{step_timeseries_signature()}}, \code{\link[=step_holiday_signature]{step_holiday_signature()}}, \code{\link[=step_fourier]{step_fourier()}}
\item Diffs & Lags \code{\link[=step_diff]{step_diff()}}, \code{recipes::step_lag()}
\item Smoothing: \code{\link[=step_slidify]{step_slidify()}}, \code{\link[=step_smooth]{step_smooth()}}
\item Variance Reduction: \code{\link[=step_box_cox]{step_box_cox()}}
\item Imputation: \code{\link[=step_ts_impute]{step_ts_impute()}}, \code{\link[=step_ts_clean]{step_ts_clean()}}
\item Padding: \code{\link[=step_ts_pad]{step_ts_pad()}}
}

Main Recipe Functions:
\itemize{
\item \code{recipes::recipe()}
\item \code{recipes::prep()}
\item \code{recipes::bake()}
}
}
\concept{moving_windows}
\concept{preprocessing}
\keyword{datagen}
