% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roll_lag.R
\name{roll_lag}
\alias{roll_lag}
\alias{roll_lag.default}
\alias{roll_lag.ts}
\alias{roll_lag.zoo}
\alias{roll_diff}
\alias{roll_diff.default}
\alias{roll_diff.ts}
\alias{roll_diff.zoo}
\alias{diff_}
\title{Fast rolling grouped lags and differences}
\usage{
roll_lag(x, n = 1L, ...)

\method{roll_lag}{default}(x, n = 1L, g = NULL, fill = NULL, ...)

\method{roll_lag}{ts}(x, n = 1L, g = NULL, fill = NULL, ...)

\method{roll_lag}{zoo}(x, n = 1L, g = NULL, fill = NULL, ...)

roll_diff(x, n = 1L, ...)

\method{roll_diff}{default}(x, n = 1L, g = NULL, fill = NULL, differences = 1L, ...)

\method{roll_diff}{ts}(x, n = 1L, g = NULL, fill = NULL, differences = 1L, ...)

\method{roll_diff}{zoo}(x, n = 1L, g = NULL, fill = NULL, differences = 1L, ...)

diff_(
  x,
  n = 1L,
  differences = 1L,
  order = NULL,
  run_lengths = NULL,
  fill = NULL
)
}
\arguments{
\item{x}{A vector or data frame.}

\item{n}{Lag. This will be recycled to match the length of x and can be negative.}

\item{...}{Arguments passed onto appropriate method.}

\item{g}{Grouping vector. This can be a vector, data frame or \code{GRP} object.}

\item{fill}{Value to fill the first \code{n} elements.}

\item{differences}{Number indicating the number of times to recursively apply
the differencing algorithm. If \code{length(n) == 1}, i.e the lag is a scalar integer,
an optimised method is used which avoids recursion entirely.
If \code{length(n) != 1} then simply recursion is used.}

\item{order}{Optionally specify an ordering with which to
apply the lags/differences.
This is useful for example when applying lags chronologically
using an unsorted time variable.}

\item{run_lengths}{Optional integer vector of run lengths that defines
the size of each lag run. For example, supplying \code{c(5, 5)} applies
lags to the first 5 elements and then essentially resets the bounds and
applies lags to the next 5 elements as if
they were an entirely separate and standalone vector. \cr
This is particularly useful in conjunction with the order argument to
perform a by-group lag.}
}
\value{
A vector the same length as \code{x}.
}
\description{
Inspired by 'collapse', \code{roll_lag} and \code{roll_diff} operate similarly to
\code{flag} and \code{fdiff}.
}
\details{
While these may not be as fast the 'collapse' equivalents,
they are adequately fast and efficient. \cr
A key difference between \code{roll_lag} and \code{flag} is that \code{g} does not need
to be sorted for the result to be correct. \cr
Furthermore, a vector of lags can be supplied for a custom rolling lag.

\code{roll_diff()} silently returns \code{NA} when there is integer overflow.
Both \code{roll_lag()} and \code{roll_diff()} apply recursively to list elements.
}
\examples{
library(timeplyr)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
x <- 1:10

roll_lag(x) # Lag
roll_lag(x, -1) # Lead
roll_diff(x) # Lag diff
roll_diff(x, -1) # Lead diff

# Using cheapr::lag_sequence()
# Differences lagged at 5, first 5 differences are compared to x[1]
roll_diff(x, cheapr::lag_sequence(length(x), 5, partial = TRUE))

# Like diff() but x/y instead of x-y
quotient <- function(x, n = 1L){
  x / roll_lag(x, n)
}
# People often call this a growth rate
# but it's just a percentage difference
# See ?roll_growth_rate for growth rate calculations
quotient(1:10)
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
