##' @name lrSS
##' @export
##' @title Sample size required to show difference in survival by log-rank test
##' given prior information about Kaplan-Meier estimate
##' 
##' @description No. of events required in a two-group trial
##' (with one binary covariate) for a two-sided log-rank test to detect a given hazard ratio.
##' \cr
##' This is calculated by:
##' \deqn{ n=  \frac{ ({Z_{\frac{\alpha}{2}}+Z_{\beta}) }^2}{ p(1-p) \log^2{HR} }}{
##' n = ( Z(alpha/2)+Z(beta) )^2 / (log(HR^2)p(1-p))}
##' Where \eqn{Z} refers to the corresponding Z-value from the standard normal distribution.
##' \cr
##' This default calculation requires that the subjects be followed until \emph{all}
##' have experienced the event of interest (typically death).
##' If this is not likely to be the case, then a more informed estimate may be
##' generated by dividing \eqn{n} by the overall probability of death
##' occurring by the end of the study.
##' \cr
##' This may be generated with prior information about \eqn{\hat{S}}{S}
##' and median survival times (for the \emph{control} group \eqn{B};
##' group \eqn{A} is the \emph{experimental} group).
##' \cr
##' Given accrual time \eqn{a} and follow-up time \eqn{f}, Simpsons rule can be used
##' to give an estimate of the proportion of patients that will die in group \eqn{B}:
##' \deqn{ d_B = 1-\frac{1}{6}[\hat{S}_B(f) + 4\hat{S}_B(f+0.5a) + \hat{S}_B(f+a)] }{
##' dB = 1 - 1/6( S(f) +4S(f+0.5a) +S(f+a) )}
##' Given median survival time \eqn{t},
##' the proportion of patients expected to  die in group \eqn{B} is:
##' \deqn{ d_B = [1 - \frac{ e^{ \frac{-0.69f}{t}} (1 - e^{ \frac{-0.69f}{t}})}{\frac{0.69a}{t}}] }{
##' dB = 1 - ( exp(-0.69f/t) [1-exp(-0.69f/t)] / 0.69a/t )}
##' 
##' @param HR Hazard Ratio. Ratio of hazard with treatment to that without.
##' @param alpha Significance level \eqn{\alpha}{alpha}, two-tailed
##' @param beta Power is \eqn{1-\beta}{1-beta}
##' @param p Proportion of subjects allocated to one group. Needs to be in range \eqn{0-1}.
##' \cr
##' Arbitrary - can be either of the two groups.
##' @param Sp Prior Kaplan-Meier estimate of survival (given no intervention)
##' @param tp Prior times corresponding to survival estimates.
##' \cr
##' There must be one time corresponding to each of: \eqn{f, 0.5*a+f, a+f}.
##' @param mtp Median time, prior. (Prior median survival time).
##' @param a Accrue. Time period for which subjects accrued.
##' @param f Follow-up. Time period for which subjects followed-up.
##' @return If any of \code{Sp}, \code{tp}, \code{mtp} \code{a} or \code{f} are missing,
##' will return the number of subjects required (with \emph{no} prior information).
##' Otherwise, returns a \code{list} with the following values:
##' \describe{
##'   \item{n}{number of subjects required (with no prior information)}
##'   \item{pS}{with prior Kaplan-Meier estimates: \describe{
##'     \item{dB}{probability death in group B (that with prior information)}
##'     \item{dA}{probability death in group A (new treatment)}
##'     \item{p}{overall probability of death}
##'     \item{n}{number of subjects required}
##'   }}
##'   \item{pM}{with prior median survival time estimates: \describe{
##'     \item{dB}{probability death in group B (that with prior information)}
##'     \item{dA}{probability death in group A (new treatment)}
##'     \item{p}{overall probability of death}
##'     \item{n}{number of subjects required}
##'   }}
##' }
##' @source Schoenfeld D, 1983. Sample-size formula for the proportional-hazards regression model.
##' \emph{Biometrics}. (39):499-503. \href{http://www.jstor.org/stable/2531021}{JSTOR}
##' @note
##' Assumes there are two groups and one intervention (covariate) which is present or absent.
##' The values in second example are taken from Schoenfelds paper, except for \code{mtp}.
##' @seealso
##' \code{\link{btumors}}
##' @examples
##' lrSS(HR=1.3, alpha=0.05)
##' data(btumors)
##' m1 <- mean(rep(btumors[,"ms"],btumors[,"n"]))
##' lrSS(HR=1.5, Sp=c(0.43, 0.2, 0.11), tp=c(1, 2, 3), mtp=m1, a=2, f=1)
##' 
lrSS <- function(HR, alpha=0.1, beta=0.2, p=0.5,
                 Sp, tp, mtp, a, f
                 ){
    stopifnot(p<1 & p>0)
### convert significance levels to Z values from normal distribution
    Za <- stats::qnorm(1-alpha/2)
    Zb <- stats::qnorm(1-beta)
    n1 <- (Za + Zb)^2 / (log(HR)^2 * p * (1 - p))
### if other values missing then return n only
    if (any(missing(Sp), missing(tp), missing(mtp), missing(a), missing(f))) return (n1)
###
    if (!length(Sp)==length(tp)) stop ("Must one survival estimate for each time point")
    err1 <- paste("Survival estimates must correspond to the following times:",
                  f, 0.5 * a + f, a + f, sep=" ")
    if(!all(c(f, a + f, 0.5 * a + f) %in% tp)) stop (err1)
### survival matrix
    Smat <- cbind(Sp, tp)
### from Simpsons rule
### proportion dying on treatment B
    dB1 <- 1-(1/6)*(Smat[, "Sp"][Smat[, "tp"]==f]
                   + 4 * Smat[, "Sp"][Smat[, "tp"]==0.5 * a + f]
                   + Smat[, "Sp"][Smat[, "tp"]==a+f] )
### proportion dying on treatment A
    dA1 <- 1 - (1 - dB1)^(1 / HR)
### total proportion dying
    dTot1 <- p * dA1 + (1 - p)*dB1
    n2 <- n1/dTot1
### if survival exponential, proportion dying on treatment B
    dB2 <- 1 - (exp(-0.69 * f / mtp)*(1 - exp(-0.69 * a / mtp))) / (0.69 * a / mtp)
### calculate median time on treatment A from hazard ratio
    mtpA2 <- HR * mtp
    dA2 <- 1 - (exp(-0.69 * f / mtpA2) * (1 - exp(-0.69 * a / mtpA2)))  / (0.69 * a / mtpA2)
    dTot2 <- p * dA2 + (1 - p) * dB2
    n3 <- n1/dTot2
### result
    res1 <- list(
        n=n1,
        pS=list(dB=dB1, dA=dA1, p=dTot1, n=n2),
        pM=list(dB=dB2, dA=dA2, p=dTot2, n=n3)
        )
    return(res1)
}
