% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgr_cusum.R
\name{cgr_cusum}
\alias{cgr_cusum}
\title{Continuous time Generalized Rapid response CUSUM (CGR-CUSUM)}
\usage{
cgr_cusum(data, coxphmod, cbaseh, ctimes, h, stoptime, C, pb = FALSE,
  ncores = 1, cmethod = "memory", dependencies, detection = "upper",
  assist, maxtheta = Inf)
}
\arguments{
\item{data}{A \code{data.frame} with rows representing subjects and the
following named columns: \describe{
\item{\code{entrytime}:}{time of entry into study (numeric);}
\item{\code{survtime}:}{time from entry until event (numeric);}
\item{\code{censorid}:}{censoring indicator (0 = right censored, 1 = observed),
(integer).}
} and optionally additional covariates used for risk-adjustment.}

\item{coxphmod}{A Cox proportional hazards regression model as
produced by
the function \code{\link[survival:coxph]{coxph()}}. Suggested: \cr
\code{coxph(Surv(survtime, censorid) ~ covariates, data = data)}. \cr
Alternatively, a list with the following elements:
\describe{
\item{\code{formula}:}{a \code{\link[stats:formula]{formula()}} in the form \code{~ covariates};}
\item{\code{coefficients}:}{a named vector specifying risk adjustment coefficients
for covariates. Names must be the same as in \code{formula} and colnames of \code{data}.}
}}

\item{cbaseh}{A function that returns the unadjusted cumulative
baseline hazard \eqn{H_0(t)}{H_0(t)}. If \code{cbaseh} is missing but
\code{coxphmod} has been
specified as a survival object, this baseline hazard rate will be determined
using the provided \code{coxphmod}.}

\item{ctimes}{(optional): Vector of construction times at which the value of the chart should be
determined. When not specified, the chart is constructed at all failure times.}

\item{h}{(optional): Value of the control limit. The chart will only be
constructed until the value of the control limit has been reached or
surpassed.}

\item{stoptime}{(optional): Time after which the value of the chart should no
longer be determined. Default = max(failure time). Useful when ctimes
has not been specified.}

\item{C}{(optional): A numeric value indicating how long after entering the study
patients should no longer influence the value of the chart. This is
equivalent to right-censoring every observation at time \code{entrytime} + C.}

\item{pb}{(optional): A boolean indicating whether a progress bar should
be shown. Default is \code{FALSE}.}

\item{ncores}{number of cores to use to parallelize the computation of the
CGR-CUSUM chart. If ncores = 1 (default), no parallelization is done. You
can use \code{\link[parallel:detectCores]{detectCores()}} to check how many
cores are available on your computer.}

\item{cmethod}{Method to calculate chart values. One of the following:
\itemize{
\item \code{"memory"} (default): matrix formulation of the problem
(faster for high volume/long time construction)
\item \code{"CPU"} calculates the value of the CGR-CUSUM for every
time point from scratch. Recommended for small data volume
(lower initialization time).
}}

\item{dependencies}{(optional): When \code{ncores > 1}, specify a list of
variables/functions/other dependencies to be exported to the core clusters
for parallel computation.}

\item{detection}{Should an \code{"upper"} or \code{"lower"} CGR-CUSUM be
constructed. Upper CUSUMs can be used to monitor for an increase in the
failure rate, while lower CUSUMs can be used to monitor for a decrease in the
failure rate.}

\item{assist}{(optional): Output of the function \code{\link[success:parameter_assist]{parameter_assist()}}}

\item{maxtheta}{(optional): Maximum value of maximum likelihood estimate for
parameter \eqn{\theta}{\theta}. Default is \code{Inf}.}
}
\value{
An object of class "cgrcusum" containing:
\itemize{
\item \code{CGR}: a \code{data.frame} with named columns:
\describe{
\item{\code{time}:}{times at which chart is constructed;}
\item{\code{value}:}{value of the chart at corresponding times;}
\item{\code{exp_theta_t}:}{value of MLE \eqn{e^{\theta_t}}{e^(\theta_t)};}
\item{\code{S_nu}}{time from which patients are considered for constructing the chart.}
}
\item \code{call}: the call used to obtain output;
\item \code{stopind}: indicator for whether the chart was stopped by
the control limit;
\item \code{h}: Specified value for the control limit.
}
}
\description{
This function performs the CGR-CUSUM procedure
described in Gomon et al. (in review). For detection purposes, it suffices
to determine the value of the chart at the times of failure. This can be
achieved by leaving \code{ctimes} unspecified.
The function has two vital parameters, at least one of which must be specified:
\itemize{
\item{\code{coxphmod}: }{Cox proportional hazards model to be used for
risk-adjustment. If \code{cbaseh} is not specified, it will be determined
from \code{coxphmod} numerically.}
\item{\code{cbaseh}: }{The cumulative baseline hazard rate to use for chart
construction. If specified with \code{coxphmod} missing, no risk-adjustment
will be performed}
}
}
\details{
The CGR-CUSUM can be used to test for a change of unknown positive fixed size \eqn{\theta}{\theta}
in the subject-specific hazard rate from \eqn{h_i(t)}{h_i(t)} to \eqn{h_i(t) e^\theta}{h_i(t) exp(\theta)}
starting from some unknown subject \eqn{\nu}{\nu}. The starting time of the first subject
who had an increase in failure rate as well as the estimated increase in the
hazard rate are shown in the output.
The CGR-CUSUM is determined as
\deqn{\max_{1 \leq \nu \leq n} \left( \hat{\theta}_{\geq \nu}(t) N_{\geq \nu}(t) - \left( \exp\left( \hat{\theta}_{\geq \nu}(t) \right) - 1 \right) \Lambda_{\geq \nu}(t)\right),}{max{1<=\nu<=n} (\theta_{>=\nu}(t)N_{>=\nu}(t)) - (exp(\theta_{>=\nu}(t))-1) \Lambda_{>=\nu}(t)),}
where  \deqn{N(\geq \nu)(t) = \sum_{i \geq \nu} N_i(t),}{N_{>=\nu}(t) = \sum_{i>=\nu} N_i(t),}
with \eqn{N_i(t)}{N_i(t)} the counting process for the failure at time \eqn{t}{t} of subject \eqn{i}{i}
and \deqn{\Lambda_{\geq \nu}(t) = \sum_{i \geq \nu} \Lambda_i(t),}{\Lambda_{>=\nu}(t) = \sum_{i>=\nu}\Lambda_i(t),}
where \eqn{\Lambda_i(t)}{\Lambda_i(t)} is the cumulative intensity of subject \eqn{i}{i} at time \eqn{t}{t}.
}
\examples{
require(survival)
#Select only the data of the first year of the first hospital in the surgerydat data set
tdat <- subset(surgerydat, unit == 1 & entrytime < 365)

#We know that the cumulative baseline hazard in the data set is
#Exponential(0.01). If you don't know the cumulative baseline, we suggest
#leaving the cbaseh argument empty and determining a coxphmod (see help)
tcbaseh <- function(t) chaz_exp(t, lambda = 0.01)

#Determine a risk-adjustment model using a Cox proportional hazards model.
#Outcome (survival) is regressed on the available covariates:
exprfit <- as.formula("Surv(survtime, censorid) ~ age + sex + BMI")
tcoxmod <- coxph(exprfit, data= surgerydat)

#Determine the values of the chart
cgr <- cgr_cusum(data = tdat, coxphmod = tcoxmod, cbaseh = tcbaseh, pb = TRUE)
#plot the CGR-CUSUM (exact hazard)
plot(cgr)
\donttest{
#Alternatively, cbaseh can be left empty when specifying coxphmod through coxph()
cgr_cox <- cgr_cusum(data = tdat, coxphmod = tcoxmod, pb = TRUE)
#plot the CGR-CUSUM (estimated hazard from coxph)
plot(cgr_cox)
}
}
\seealso{
\code{\link[success]{plot.cgrcusum}}, \code{\link[success]{runlength.cgrcusum}}

Other quality control charts: 
\code{\link{bernoulli_cusum}()},
\code{\link{bk_cusum}()},
\code{\link{funnel_plot}()}
}
\author{
Daniel Gomon
}
\concept{quality control charts}
