% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/stringdist.R
\name{stringdist}
\alias{stringdist}
\alias{stringdistmatrix}
\title{Compute distance metrics between strings}
\usage{
stringdist(a, b, method = c("osa", "lv", "dl", "hamming", "lcs", "qgram",
  "cosine", "jaccard", "jw", "soundex"), useBytes = FALSE, weight = c(d = 1,
  i = 1, s = 1, t = 1), maxDist = Inf, q = 1, p = 0)

stringdistmatrix(a, b, method = c("osa", "lv", "dl", "hamming", "lcs",
  "qgram", "cosine", "jaccard", "jw", "soundex"), useBytes = FALSE,
  weight = c(d = 1, i = 1, s = 1, t = 1), maxDist = Inf, q = 1, p = 0,
  useNames = FALSE, ncores = 1, cluster = NULL)
}
\arguments{
\item{a}{R object (target); will be converted by \code{as.character}.}

\item{b}{R object (source); will be converted by \code{as.character}.}

\item{method}{Method for distance calculation. The default is \code{"osa"} (see details).}

\item{useBytes}{Perform byte-wise comparison. \code{useBytes=TRUE} is faster but may yield different
results depending on character encoding. See also below, under ``encoding issues''.}

\item{weight}{For \code{method='osa'} or \code{'dl'}, the penalty for deletion, insertion, substitution and transposition, in that order.
When \code{method='lv'}, the penalty for transposition is ignored. When \code{method='jw'}, the weights associated with characters
of \code{a}, characters from \code{b} and the transposition weight, in that order.
Weights must be positive and not exceed 1. \code{weight} is
ignored completely when \code{method='hamming'}, \code{'qgram'}, \code{'cosine'}, \code{'Jaccard'}, or \code{'lcs'}.}

\item{maxDist}{[DEPRECATED AND WILL BE REMOVED] Currently kept for backward compatibility. It does not
offer any speed gain. (In fact, it currently slows things down when set to anything different from \code{Inf}).}

\item{q}{Size of the \eqn{q}-gram; must be nonnegative. Only applies to \code{method='qgram'}, \code{'jaccard'} or \code{'cosine'}.}

\item{p}{Penalty factor for Jaro-Winkler distance. The valid range for \code{p} is \code{0 <= p <= 0.25}.
 If \code{p=0} (default), the Jaro-distance is returned. Applies only to \code{method='jw'}.}

\item{useNames}{Use input vectors as row and column names?}

\item{ncores}{Number of cores to use. If \code{ncores>1}, a local cluster is
created using \code{\link[parallel]{makeCluster}}. Parallelisation is over \code{b}, so
the speed gain by parallelisation is highest when \code{b} has less elements than \code{a}.}

\item{cluster}{(Optional) a custom cluster, created with
\code{\link[parallel]{makeCluster}}. If \code{cluster} is not \code{NULL},
\code{ncores} is ignored.}
}
\value{
For \code{stringdist},  a vector with string distances of size \code{max(length(a),length(b))}.
 For \code{stringdistmatrix}, a \code{length(a)xlength(b)} \code{matrix}. The returned distance is
 nonnegative if it can be computed, \code{NA} if any of the two argument strings is \code{NA} and \code{Inf}
 when \code{maxDist} is exceeded or, in case of the hamming distance, when the two compared strings have different length.
}
\description{
Compute distance metrics between strings
}
\section{Details}{

\code{stringdist} computes pairwise string distances between elements of \code{character} vectors
\code{a} and \code{b}, where the vector with less elements is recycled.

\code{stringdistmatrix} computes the string distance matrix with rows according to
\code{a} and columns according to \code{b}.

Currently, the following distance metrics are supported:
\tabular{ll}{
   \code{osa} \tab Optimal string aligment, (restricted Damerau-Levenshtein distance).\cr
   \code{lv} \tab Levenshtein distance (as in R's native \code{\link[utils]{adist}}).\cr
   \code{dl} \tab Full Damerau-Levenshtein distance.\cr
   \code{hamming}  \tab Hamming distance (\code{a} and \code{b} must have same nr of characters).\cr
   \code{lcs} \tab Longest common substring distance.\cr
   \code{qgram} \tab \eqn{q}-gram distance. \cr
   \code{cosine} \tab cosine distance between \eqn{q}-gram profiles \cr
   \code{jaccard} \tab Jaccard distance between \eqn{q}-gram profiles \cr
   \code{jw} \tab Jaro, or Jaro-Winker distance.\cr
   \code{soundex} \tab Distance based on soundex encoding (see below)
}

A short description of these algorithms is proveded \link[=stringdist-package]{here}, or
see the \href{http://journal.r-project.org/archive/2014-1/loo.pdf}{R Journal Paper} (external link) for
more formal descriptions.
}

\section{Encoding issues}{

If \code{bytes=FALSE}, input strings are re-encoded to \code{utf8} an then to \code{integer}
vectors prior to the distance calculation (since the underlying \code{C}-code expects \code{unsigned int}s).
This double conversion is necessary as it seems the only way to
reliably convert (possibly multibyte) characters to integers on all systems
supported by \code{R}. \code{R}'s native \code{\link[utils]{adist}} function does this as well.

If \code{bytes=TRUE}, the input strings are treated as if each byte was a
single character. This may be significantly faster since it avoids
conversion of \code{utf8} to integer with \code{\link[base]{utf8ToInt}} (up to a factor of 3, for strings of 5-25 characters).
However, results may depend on the (possibly multibyte)  character encoding scheme
and note that \code{R}'s internal encoding scheme is OS-dependent.
If you're sure that all your input is \code{ASCII},  you can safely set
\code{useBytes=TRUE} to profit from the speed gain on any platform.

See base \code{R}'s \code{\link[base]{Encoding}} and
\code{\link[base]{iconv}} documentation for details on how \code{R} handles
character encoding.
}

\section{Unicode normalisation}{

In \code{utf-8}, the same (accented) character may be represented as several byte sequences. For example, an u-umlaut
can be represented with a single byte code or as a byte code representing \code{'u'} followed by a modifier byte code
that adds the umlaut. The \href{http://cran.r-project.org/web/packages/stringi/}{stringi} package
of Gagolevski and Tartanus offers unicode normalisation tools.
}

\section{Paralellization}{

The \code{stringdistmatrix} function uses \code{\link[parallel]{makeCluster}} to create a local cluster and compute the
distance matrix in parallel when \code{ncores>1}. The cluster is terminated after the matrix has been computed.
As the cluster is local, the \code{ncores} parameter should not be larger than the number
of cores on your machine. Use \code{\link[parallel]{detectCores}} to check the number of cores available. Alternatively,
you can create a cluster using \code{\link[parallel]{makeCluster}} and pass that to \code{stringdistmatrix} (through the \code{cluster} argument.
This allows you to reuse the cluster setup for other calculations.
There is overhead in creating clusters, so creating the cluster yourself is a good choice if you want to call \code{stringdistmatrix}
multiple times, for example in a loop.
}

\section{Acknowledgement}{

The code for the full Damerau-Levenshtein distance was adapted from Nick Logan's
\href{https://github.com/ugexe/Text--Levenshtein--Damerau--XS/blob/master/damerau-int.c}{public github repository}.
}

\section{Citation}{

If you would like to cite this package, please cite the \href{http://journal.r-project.org/archive/2014-1/loo.pdf}{R Journal Paper}:
\itemize{
\item{M.P.J. van der Loo (2014). The \code{stringdist} package for approximate string matching.
 R Journal 6(1) pp 111-122}
}
Or use \code{citation('stringdist')} to get a bibtex item.
}

\section{other}{


\itemize{

\item{Raffael Vogler wrote a nice
\href{http://www.joyofdata.de/blog/comparison-of-string-distance-algorithms/}{blog}
comparing different string distances in this package.
}

}
}
\examples{

# Simple example using optimal string alignment
stringdist("ca","abc")

# The same example using Damerau-Levenshtein distance (multiple editing of substrings allowed)
stringdist("ca","abc",method="dl")

# string distance matching is case sensitive:
stringdist("ABC","abc")

# so you may want to normalize a bit:
stringdist(tolower("ABC"),"abc")

# stringdist recycles the shortest argument:
stringdist(c('a','b','c'),c('a','c'))

# stringdistmatrix gives the distance matrix (by default for optimal string alignment):
stringdist(c('a','b','c'),c('a','c'))

# different edit operations may be weighted; e.g. weighted substitution:
stringdist('ab','ba',weight=c(1,1,1,0.5))

# Non-unit weights for insertion and deletion makes the distance metric asymetric
stringdist('ca','abc')
stringdist('abc','ca')
stringdist('ca','abc',weight=c(0.5,1,1,1))
stringdist('abc','ca',weight=c(0.5,1,1,1))

# Hamming distance is undefined for 
# strings of unequal lengths so stringdist returns Inf
stringdist("ab","abc",method="h")
# For strings of eqal length it counts the number of unequal characters as they occur
# in the strings from beginning to end
stringdist("hello","HeLl0",method="h")

# The lcm (longest common substring) distance returns the number of 
# characters that are not part of the lcs.
#
# Here, the lcs is either 'a' or 'b' and one character cannot be paired:
stringdist('ab','ba',method="lcs")
# Here the lcs is 'surey' and 'v', 'g' and one 'r' of 'surgery' are not paired
stringdist('survey','surgery',method="lcs")


# q-grams are based on the difference between occurrences of q consecutive characters
# in string a and string b.
# Since each character abc occurs in 'abc' and 'cba', the q=1 distance equals 0:
stringdist('abc','cba',method='qgram',q=1)

# since the first string consists of 'ab','bc' and the second 
# of 'cb' and 'ba', the q=2 distance equals 4 (they have no q=2 grams in common):
stringdist('abc','cba',method='qgram',q=2)

# Wikipedia has the following example of the Jaro-distance. 
stringdist('MARTHA','MATHRA',method='jw')
# Note that stringdist gives a  _distance_ where wikipedia gives the corresponding 
# _similarity measure_. To get the wikipedia result:
1 - stringdist('MARTHA','MATHRA',method='jw')

# The corresponding Jaro-Winkler distance can be computed by setting p=0.1
stringdist('MARTHA','MATHRA',method='jw',p=0.1)
# or, as a similarity measure
1 - stringdist('MARTHA','MATHRA',method='jw',p=0.1)

# This gives distance 1 since Euler and Gauss translate to different soundex codes.
stringdist('Euler','Gauss',method='soundex')
# Euler and Ellery translate to the same code and have distance 0
stringdist('Euler','Ellery',method='soundex')





}

