#' @export structure.plot
#' 
#' @title Plot STRUCTURE Results
#' @description Plot Q-matrix from a call to \code{\link{structure.run}} or \code{\link{clumpp.run}}.
#' 
#' @param q.mat matrix or data.frame of assignment probabilities.
#' @param pop.col column number identifying original population designations.
#' @param prob.col column number of first assignment probabilities to first group. 
#' It is assumed that the remainder of columns (\code{prob.col:ncol(q.mat)}) contain
#' all assignment probabilities, and thus \emph{k} = \code{ncol(q.mat) - prob.col + 1}.
#' @param sort.probs logical. Sort individuals by probabilities within populations? 
#' If \code{FALSE} individuals will be plotted as in \code{q.mat}.
#' @param label.pops logical. Label the populations on the plot? If \code{FALSE}, then
#' population labels are omitted so the user can customize their format.
#' @param col colors to use for each group.
#' @param ... optional arguments to be passed to \code{\link[graphics]{barplot}}.
#' 
#' @return invisibly, a list containing:
#' \tabular{ll}{
#'   \code{q.mat} \tab the sorted matrix or data.frame of assignment probabilities used in the plot.\cr
#'   \code{bar.centers} \tab a vector of the centers of bars for each individual.\cr
#'   \code{pop.ticks} \tab a vector of the tick marks separating populations.\cr
#'  }
#' 
#' @author Eric Archer \email{eric.archer@@noaa.gov}
#' 
#' @seealso \code{\link{structure}}, \code{\link{clumpp}}

structure.plot <- function(q.mat, pop.col = 3, prob.col = 4, sort.probs = TRUE, label.pops = TRUE,
                           col = NULL, ...) {  
  # sort q.mat within strata by probability
  prob.cols <- prob.col:ncol(q.mat)
  if(sort.probs) {
    q.mat <- do.call(rbind, by(q.mat, list(q.mat[, pop.col]), function(x) {
      prob.median <- sapply(prob.cols, function(i) median(x[, i]))
      pop.order <- prob.cols[order(prob.median, decreasing = TRUE)]
      order.list <- lapply(pop.order, function(i) x[, i])
      order.list$decreasing = TRUE
      x[do.call(order, order.list), ]
    }, simplify = FALSE))
  }
  q.mat <- q.mat[order(q.mat[, pop.col]), ]
  
  # make sure probs sum to 1 and get matrix
  q.mat[, prob.cols] <- prop.table(as.matrix(q.mat[, prob.cols]), 1)
  assign.mat <- t(q.mat[, prob.cols])
  
  # create barplot
  if(is.null(col)) col <- rainbow(length(prob.cols))
  bp <- barplot(assign.mat, axisnames = FALSE, col = col, ...)
  tx <- tapply(bp, q.mat[, pop.col], min) + 0.1
  tx <- c(tx - tx[1], max(bp) + tx[1])
  axis(1, at = tx, labels = FALSE)
  if(label.pops) {
    lbl.x <- sapply(1:(length(tx) - 1), function(i) tx[i] + (tx[i + 1] - tx[i]) / 2)
    names(lbl.x) <- names(tx)[1:(length(tx) - 1)]
    mtext(names(lbl.x), side = 1, at = lbl.x)
  }
  
  invisible(list(q.mat = q.mat, bar.centers = bp, pop.ticks = tx))
}