% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/markedIPPP_mix.R
\name{rMIPPP_cond_mark}
\alias{rMIPPP_cond_mark}
\title{Generate a Marked Poisson point process (conditional on mark)}
\usage{
rMIPPP_cond_mark(lambda = 500, params = c(0.5, 0.5),
  mark_distr_choice = 0, truncate = FALSE, discrete_mark = TRUE,
  win = owin(c(-3, 3), c(-3, 3)), bigwin, open_new_window = FALSE,
  grayscale = FALSE, show_plots = TRUE)
}
\arguments{
\item{lambda}{Average number of mark values
observed over the window. This is the
total number of points observed (on the average).}

\item{params}{Parameters for the mark
distribution. The value depends on the \code{mark_distr_choice}
parameter, e.g., \code{params} is a vector
of probabilities if the mark
distribution is discrete
(\code{mark_distr_choice=0}).}

\item{mark_distr_choice}{A number indicating which
mark distribution to use. In the
discrete mark case, the mark distribution
is discrete over the marks \code{1:length(params)} with
corresponding probabilities in \code{params}.
The continuous mark case has not been implemented yet.}

\item{truncate}{Logical variable indicating whether or not we
normalize the densities of the mixture components
to have all their mass within the window defined
in the window \code{win}. This affects the mixture model for the
intensity surface of the ground process.}

\item{discrete_mark}{Logical flag indicating whether the mark is discrete or not.
Default is TRUE. For continuous marks set this to FALSE.}

\item{win}{Object of type \code{\link[spatstat]{owin}} defining the window of observation.}

\item{bigwin}{Object of type \code{\link[spatstat]{owin}}. If supplied, this will be the
window of observation, even if the pattern is generated over \code{win}. Useful if we
do not truncate (\code{truncate=FALSE}) and we want better presentation of the generated MIPPP.}

\item{open_new_window}{Open a new window for a plot.}

\item{grayscale}{Logical to request plots in grayscale.}

\item{show_plots}{Logical variable requesting to produce exploratory plots of the
Marked IPPP intensity surface and generated point pattern for each mark.}
}
\value{
A list containing the following components:
\item{groundsurfs}{A list of \code{intensity_surface} objects containing the surfaces of the ground processes (one for each discrete mark value).}
\item{groundPPs}{A list of \code{\link[spatstat]{ppp}} objects containing the locations of the ground processes (one for each discrete mark value).}
\item{genMPP}{The generated point pattern as an object of class \code{\link[spatstat]{ppp}} and \code{sppmix}. The member \code{$marks} contains the marks at each of the generated locations.}
\item{mark_distr_choice}{The choice of mark distribution. Same as the supplied parameter.}
\item{params}{The default or supplied parameter \code{params}.}
}
\description{
This function generates realizations (point patterns) from
a given Marked IPPP via conditioning of the joint intensity surface
on its marked component. See details for the choice of models
for the mark distribution. For each mark value we obtain a ground
process. There processes are
standard IPPP (unmarked) with mixture intensity surfaces. The mark
distribution is responsible for the number of events in the point pattern.

For examples see

\url{http://faculty.missouri.edu/~micheasa/sppmix/sppmix_all_examples.html
#rMIPPP_cond_mark}
}
\details{
For discrete marks, we assume that the joint intensity function of a
marked point pattern \code{N=[s,m]} with \code{n}
events is of the form:

\code{intensity(s,m)=lambda*M(m|theta1)*g(s(m)|theta2(m))}

where \code{m} denotes a mark and \code{s=s(m)}
a location with mark \code{m}, lambda a parameter
with the interpretation as the average number of events
over the window of observation, and \code{M} the mark distribution and
\code{g} the ground intensity are proper densities.

In order to simulate from this Marked IPPP
we first simulate the number of events
and their marks from an IPPP with
intensity \code{lambda*M(m|theta1)}, and then generate
the ground intensities for each mark. Marks are assumed to be
independnet of each other and the mixture parameters
describing each ground process are also assumed to be independent
of each other.

The continuous mark case will be implemented in future releases.

See Micheas (2014) for more details on
Marked IPPP models via conditioning arguments.
}
\examples{
\donttest{
# Create a marked point pattern; use randomization and 2 discrete uniform
# marks (default values)
newMPP=rMIPPP_cond_mark(bigwin = spatstat::owin(c(-10,10),c(-10,10)))
newMPP$params
plot(newMPP$genMPP, showmarks=TRUE)+add_title("Marked Poisson point pattern",
 n=newMPP$genMPP$n, nmarks=2)
plotmix_2d(newMPP$groundsurfs[[1]], newMPP$groundPPs[[1]])+ add_title(
 "Poisson point pattern for mark 1", n=newMPP$genMPP$n, m=newMPP$groundsurfs[[1]]$m)
plotmix_2d(newMPP$groundsurfs[[2]], newMPP$groundPPs[[2]])+ add_title(
 "Poisson point pattern for mark 2", n=newMPP$genMPP$n, m=newMPP$groundsurfs[[2]]$m)}

}
\references{
Hierarchical Bayesian Modeling of Marked Non-Homogeneous Poisson Processes with finite mixtures and inclusion of covariate information. Micheas, A.C. (2014). Journal of Applied Statistics, 41, 12, 2596-2615, DOI: 10.1080/02664763.2014.922167.
}
\seealso{
\code{\link{plotmix_2d}}
}
\author{
Sakis Micheas
}
