#
#   envelope.R
#
#   computes simulation envelopes 
#
#   $Revision: 2.50 $  $Date: 2013/08/14 02:50:59 $
#

envelope <- function(Y, fun, ...) {
  UseMethod("envelope")
}

  # .................................................................
  #     A 'simulation recipe' contains the following variables
  #
  #  type = Type of simulation
  #           "csr": uniform Poisson process
  #           "rmh": simulated realisation of fitted Gibbs or Poisson model 
  #          "kppm": simulated realisation of fitted cluster model 
  #          "expr": result of evaluating a user-supplied expression
  #          "list": user-supplied list of point patterns
  #
  #  expr = expression that is repeatedly evaluated to generate simulations
  #
  #    envir = environment in which to evaluate the expression `expr'
  #
  #    'csr' = TRUE iff the model is (known to be) uniform Poisson
  #
  #    pois  = TRUE if model is known to be Poisson
  #  
  # ...................................................................

simulrecipe <- function(type, expr, envir, csr, pois=csr) {
  if(csr && !pois) warning("Internal error: csr=TRUE but pois=FALSE")
  out <- list(type=type,
              expr=expr,
              envir=envir,
              csr=csr,
              pois=pois)
  class(out) <- "simulrecipe"
  out
}


envelope.ppp <-
  function(Y, fun=Kest, nsim=99, nrank=1, ..., 
           simulate=NULL, verbose=TRUE, clipdata=TRUE, 
           transform=NULL, global=FALSE, ginterval=NULL,
           savefuns=FALSE, savepatterns=FALSE, nsim2=nsim,
           VARIANCE=FALSE, nSD=2,
           Yname=NULL, maxnerr=nsim, do.pwrong=FALSE,
           envir.simul=NULL) {
  cl <- short.deparse(sys.call())
  if(is.null(Yname)) Yname <- short.deparse(substitute(Y))
  if(is.null(fun)) fun <- Kest
  envir.user <- if(!is.null(envir.simul)) envir.simul else parent.frame()
  envir.here <- sys.frame(sys.nframe())

  if(is.null(simulate)) {
    # ...................................................
    # Realisations of complete spatial randomness
    # will be generated by rpoispp
    # Data pattern X is argument Y
    # Data pattern determines intensity of Poisson process
    X <- Y
    sY <- summary(Y, checkdup=FALSE)
    Yintens <- sY$intensity
    Ywin <- Y$window
    # expression that will be evaluated
    simexpr <- 
      if(!is.marked(Y)) {
        # unmarked point pattern
        expression(rpoispp(Yintens, win=Ywin))
      } else {
        # marked point pattern
        Ymarx <- marks(Y, dfok=FALSE)
        expression({A <- rpoispp(Yintens, win=Ywin);
                    A %mark% sample(Ymarx, A$n, replace=TRUE)})
      }
    # evaluate in THIS environment
    simrecipe <- simulrecipe(type = "csr",
                             expr = simexpr,
                             envir = envir.here,
                             csr   = TRUE,
                             pois  = TRUE)
  } else {
    # ...................................................
    # Simulations are determined by 'simulate' argument
    # Processing is deferred to envelopeEngine
    simrecipe <- simulate
    # Data pattern is argument Y
    X <- Y
  }
  envelopeEngine(X=X, fun=fun, simul=simrecipe,
                 nsim=nsim, nrank=nrank, ..., 
                 verbose=verbose, clipdata=clipdata,
                 transform=transform, global=global, ginterval=ginterval,
                 savefuns=savefuns, savepatterns=savepatterns, nsim2=nsim2,
                 VARIANCE=VARIANCE, nSD=nSD,
                 Yname=Yname, maxnerr=maxnerr, cl=cl,
                 envir.user=envir.user, do.pwrong=do.pwrong)
}

envelope.ppm <- 
  function(Y, fun=Kest, nsim=99, nrank=1, ..., 
           simulate=NULL, verbose=TRUE, clipdata=TRUE, 
           start=NULL,
           control=update(default.rmhcontrol(Y), nrep=nrep), nrep=1e5, 
           transform=NULL, global=FALSE, ginterval=NULL,
           savefuns=FALSE, savepatterns=FALSE, nsim2=nsim,
           VARIANCE=FALSE, nSD=2,
           Yname=NULL, maxnerr=nsim, do.pwrong=FALSE,
           envir.simul=NULL) {
  cl <- short.deparse(sys.call())
  if(is.null(Yname)) Yname <- short.deparse(substitute(Y))
  if(is.null(fun)) fun <- Kest
  envir.user <- if(!is.null(envir.simul)) envir.simul else parent.frame()
  envir.here <- sys.frame(sys.nframe())

  # Extract data pattern X from fitted model Y
  X <- data.ppm(Y)
  
  if(is.null(simulate)) {
    # ...................................................
    # Simulated realisations of the fitted model Y
    # will be generated
    pois <- is.poisson(Y)
    csr <- is.stationary(Y) && pois
    type <- if(csr) "csr" else "rmh"
    # Set up parameters for rmh
    rmodel <- rmhmodel(Y, verbose=FALSE)
    if(is.null(start))
      start <- list(n.start=X$n)
    rstart <- rmhstart(start)
    rcontr <- rmhcontrol(control)
    # pre-digest arguments
    rmhinfolist <- rmh(rmodel, rstart, rcontr, preponly=TRUE, verbose=FALSE)
    # expression that will be evaluated
    simexpr <- expression(rmhEngine(rmhinfolist, verbose=FALSE))
    envir <- envir.here
    # evaluate in THIS environment
    simrecipe <- simulrecipe(type  = type,
                             expr  = simexpr,
                             envir = envir.here,
                             csr   = csr,
                             pois  = pois)
  } else {
    # ...................................................
    # Simulations are determined by 'simulate' argument
    # Processing is deferred to envelopeEngine
    simrecipe <- simulate
  }
  envelopeEngine(X=X, fun=fun, simul=simrecipe,
                 nsim=nsim, nrank=nrank, ..., 
                 verbose=verbose, clipdata=clipdata,
                 transform=transform, global=global, ginterval=ginterval,
                 savefuns=savefuns, savepatterns=savepatterns, nsim2=nsim2,
                 VARIANCE=VARIANCE, nSD=nSD,
                 Yname=Yname, maxnerr=maxnerr, cl=cl,
                 envir.user=envir.user, do.pwrong=do.pwrong)
}

envelope.kppm <-
  function(Y, fun=Kest, nsim=99, nrank=1, ..., 
           simulate=NULL, verbose=TRUE, clipdata=TRUE, 
           transform=NULL, global=FALSE, ginterval=NULL,
           savefuns=FALSE, savepatterns=FALSE, nsim2=nsim,
           VARIANCE=FALSE, nSD=2, Yname=NULL, maxnerr=nsim,
           do.pwrong=FALSE, envir.simul=NULL)
{
  cl <- short.deparse(sys.call())
  if(is.null(Yname)) Yname <- short.deparse(substitute(Y))
  if(is.null(fun)) fun <- Kest
  envir.user <- if(!is.null(envir.simul)) envir.simul else parent.frame()
  envir.here <- sys.frame(sys.nframe())
  
  # Extract data pattern X from fitted model Y
  X <- Y$X
  
  if(is.null(simulate)) {
    # Simulated realisations of the fitted model Y
    # will be generated using simulate.kppm
    kmodel <- Y
    # expression that will be evaluated
    simexpr <- expression(simulate(kmodel)[[1]])
    # evaluate in THIS environment
    simrecipe <- simulrecipe(type = "kppm",
                             expr = simexpr,
                             envir = envir.here,
                             csr   = FALSE,
                             pois  = FALSE)
  } else {
    # ...................................................
    # Simulations are determined by 'simulate' argument
    # Processing is deferred to envelopeEngine
    simrecipe <- simulate
  }
  envelopeEngine(X=X, fun=fun, simul=simrecipe,
                 nsim=nsim, nrank=nrank, ..., 
                 verbose=verbose, clipdata=clipdata,
                 transform=transform, global=global, ginterval=ginterval,
                 savefuns=savefuns, savepatterns=savepatterns, nsim2=nsim2,
                 VARIANCE=VARIANCE, nSD=nSD,
                 Yname=Yname, maxnerr=maxnerr, cl=cl,
                 envir.user=envir.user, do.pwrong=do.pwrong)

}

## .................................................................
##   Engine for simulating and computing envelopes
## .................................................................
#
#  X is the data point pattern, which could be ppp, pp3, ppx etc
#  X determines the class of pattern expected from the simulations
#

envelopeEngine <-
  function(X, fun, simul,
           nsim=99, nrank=1, ..., 
           verbose=TRUE, clipdata=TRUE, 
           transform=NULL, global=FALSE, ginterval=NULL,
           savefuns=FALSE, savepatterns=FALSE,
           saveresultof=NULL,
           weights=NULL,
           nsim2=nsim,
           VARIANCE=FALSE, nSD=2,
           Yname=NULL, maxnerr=nsim, internal=NULL, cl=NULL,
           envir.user=envir.user,
           expected.arg="r",
           do.pwrong=FALSE) {
  #
  envir.here <- sys.frame(sys.nframe())
  
  # ----------------------------------------------------------
  # Determine Simulation
  # ----------------------------------------------------------
  
  # Identify class of patterns to be simulated, from data pattern X
  Xclass <- if(is.ppp(X)) "ppp" else
            if(is.pp3(X)) "pp3" else
            if(is.ppx(X)) "ppx" else
            stop("Unrecognised class of point pattern")
  Xobjectname <- paste("point pattern of class", sQuote(Xclass))

  # Option to use weighted average
  if(use.weights <- !is.null(weights)) {
    # weight can be either a numeric vector or a function
    if(is.numeric(weights)) {
      compute.weights <- FALSE
      weightfun <- NULL
    } else if(is.function(weights)) {
      compute.weights <- TRUE
      weightfun <- weights
      weights <- NULL  
    } else stop("weights should be either a function or a numeric vector")
  } else compute.weights <- FALSE
    
  # Undocumented option to generate patterns only.
  patterns.only <- identical(internal$eject, "patterns")
  evaluate.fun <- !patterns.only

  # Undocumented option to evaluate 'something' for each pattern
  if(savevalues <- !is.null(saveresultof)) {
    stopifnot(is.function(saveresultof))
    SavedValues <- list()
  }

  # Identify type of simulation from argument 'simul'
  if(inherits(simul, "simulrecipe")) {
    # ..................................................
    # simulation recipe is given
    simtype <- simul$type
    simexpr <- simul$expr
    envir   <- simul$envir
    csr     <- simul$csr
    pois    <- simul$pois
  } else {
    # ...................................................
    # simulation is specified by argument `simulate' to envelope()
    simulate <- simul
    # which should be an expression, or a list of point patterns,
    # or an envelope object.
    csr <- FALSE
    # override
    if(!is.null(icsr <- internal$csr)) csr <- icsr
    pois <- csr
    model <- NULL
    if(inherits(simulate, "envelope")) {
      # envelope object: see if it contains stored point patterns
      simpat <- attr(simulate, "simpatterns")
      if(!is.null(simpat))
        simulate <- simpat
      else
        stop(paste("The argument", sQuote("simulate"),
                   "is an envelope object but does not contain",
                   "any saved point patterns."))
    }
    if(is.expression(simulate)) {
      # The user-supplied expression 'simulate' will be evaluated repeatedly
      simtype <- "expr"
      simexpr <- simulate
      envir <- envir.user
    } else if(is.list(simulate) &&
              (   (is.ppp(X) && all(unlist(lapply(simulate, is.ppp))))
               || (is.pp3(X) && all(unlist(lapply(simulate, is.pp3))))
               || (is.ppx(X) && all(unlist(lapply(simulate, is.ppx)))))) {
      # The user-supplied list of point patterns will be used
      simtype <- "list"
      SimDataList <- simulate
      # expression that will be evaluated
      simexpr <- expression(SimDataList[[i]])
      envir <- envir.here
      # ensure that `i' is defined
      i <- 1
      # any messages?
      if(!is.null(mess <- attr(simulate, "internal"))) {
        # determine whether these point patterns are realisations of CSR
        csr <- !is.null(mc <- mess$csr) && mc
      }
    } else stop(paste(sQuote("simulate"),
                      "should be an expression, or a list of point patterns"))
  }
  # -------------------------------------------------------------------
  # Determine clipping window
  # ------------------------------------------------------------------

  if(clipdata) {
    # Generate one realisation
    Xsim <- eval(simexpr, envir=envir)
    if(!inherits(Xsim, Xclass))
      switch(simtype,
             csr=stop(paste("Internal error:", Xobjectname, "not generated")),
             rmh=stop(paste("Internal error: rmh did not return an",
               Xobjectname)),
             kppm=stop(paste("Internal error: simulate.kppm did not return an",
               Xobjectname)),
             expr=stop(paste("Evaluating the expression", sQuote("simulate"),
               "did not yield an", Xobjectname)),
             list=stop(paste("Internal error: list entry was not an",
               Xobjectname)),
             stop(paste("Internal error:", Xobjectname, "not generated"))
             )
    # Extract window
    clipwin <- Xsim$window
    if(!is.subset.owin(clipwin, X$window))
      warning("Window containing simulated patterns is not a subset of data window")
  }
  
  # ------------------------------------------------------------------
  # Summary function to be applied 
  # ------------------------------------------------------------------

  if(evaluate.fun) {
    if(is.null(fun))
      stop("Internal error: fun is NULL")

    # Name of function, for error messages
    fname <- if(is.name(substitute(fun))) short.deparse(substitute(fun)) else
             if(is.character(fun)) fun else "fun"
    fname <- sQuote(fname)

    # R function to apply
    if(is.character(fun)) {
      gotfun <- try(get(fun, mode="function"))
      if(inherits(gotfun, "try-error"))
        stop(paste("Could not find a function named", sQuote(fun)))
      fun <- gotfun
    } else if(!is.function(fun)) 
      stop(paste("unrecognised format for function", fname))
    fargs <- names(formals(fun))
    if(!any(c(expected.arg, "...") %in% fargs))
      stop(paste(fname, "should have",
                 ngettext(length(expected.arg), "an argument", "arguments"),
                 "named", commasep(sQuote(expected.arg)),
                 "or a", sQuote("..."), "argument"))
    usecorrection <- any(c("correction", "...") %in% fargs)
  }
  
  # ---------------------------------------------------------------------
  # validate other arguments
  if((nrank %% 1) != 0)
    stop("nrank must be an integer")
  if((nsim %% 1) != 0)
    stop("nsim must be an integer")
  stopifnot(nrank > 0 && nrank < nsim/2)

  rgiven <- any(expected.arg %in% names(list(...)))

  if(tran <- !is.null(transform)) {
    stopifnot(is.expression(transform))
    transform.funX <- dotexpr.to.call(transform, "funX", "eval.fv")
    transform.funXsim <- dotexpr.to.call(transform, "funXsim", "eval.fv")
    # 'transform.funX' and 'transform.funXsim' are unevaluated calls to eval.fv
    
    # .... old code ....
    
    # 'transform' is an expression 
#    aa <- substitute(substitute(tt, list(.=as.name("funX"))),
#                     list(tt=transform))
    # 'aa' is a language expression invoking 'substitute'
#    bb <- eval(parse(text=deparse(aa)))
    # 'bb' is an expression obtained by replacing "." by "funX" 
#    transform.funX <- as.call(bb)
#    transform.funX[[1]] <- as.name("eval.fv")
    # 'transform.funX' is an unevaluated call to eval.fv
#    aa <- substitute(substitute(tt, list(.=as.name("funXsim"))),
#                     list(tt=transform))
#    bb <- eval(parse(text=deparse(aa)))
#    transform.funXsim <- as.call(bb)
#    transform.funXsim[[1]] <- as.name("eval.fv")
  }
  if(!is.null(ginterval)) 
    stopifnot(is.numeric(ginterval) && length(ginterval) == 2)
    
  # ---------------------------------------------------------------------
  # Evaluate function for data pattern X
  # ------------------------------------------------------------------
  if(evaluate.fun) {
    Xarg <- if(!clipdata) X else X[clipwin]
    corrx <- if(usecorrection) list(correction="best") else NULL
    funX <- do.call(fun,
                    resolve.defaults(list(Xarg),
                                     list(...),
                                     corrx))
                                     
    if(!inherits(funX, "fv"))
      stop(paste("The function", fname,
                 "must return an object of class", sQuote("fv")))

    argname <- fvnames(funX, ".x")
    valname <- fvnames(funX, ".y")
    has.theo <- "theo" %in% fvnames(funX, "*")
    csr.theo <- csr && has.theo

    if(tran) {
      # extract only the recommended value
      if(csr.theo) 
        funX <- funX[, c(argname, valname, "theo")]
      else
        funX <- funX[, c(argname, valname)]
      # apply the transformation to it
      funX <- eval(transform.funX)
    }
    
    rvals <- funX[[argname]]
    fX    <- funX[[valname]]

    # default domain over which to maximise
    alim <- attr(funX, "alim")
    if(global && is.null(ginterval))
      ginterval <- if(rgiven) range(rvals) else alim
  }
  
  #--------------------------------------------------------------------
  # Determine number of simulations
  # ------------------------------------------------------------------
  #
  ## determine whether dual simulations are required
  ## (one set of simulations to calculate the theoretical mean,
  ##  another independent set of simulations to obtain the critical point.)
  dual <- (global && !csr.theo && !VARIANCE)
  Nsim <- if(!dual) nsim else (nsim + nsim2)

  # if taking data from a list of point patterns,
  # check there are enough of them
  if(simtype == "list" && Nsim > length(SimDataList))
    stop(paste("Number of simulations",
               paren(if(!dual)
                     paste(nsim) else
                     paste(nsim, "+", nsim2, "=", Nsim)
                     ),
               "exceeds number of point pattern datasets supplied"))

  # Undocumented secret exit
  # ------------------------------------------------------------------
  if(patterns.only) {
    # generate simulated realisations and return only these patterns
    if(verbose) {
      action <- if(simtype == "list") "Extracting" else "Generating"
      descrip <- switch(simtype,
                        csr = "simulations of CSR",
                        rmh = paste("simulated realisations of fitted",
                          if(pois) "Poisson" else "Gibbs",
                          "model"),
                        kppm = "simulated realisations of fitted cluster model",
                        expr = "simulations by evaluating expression",
                        list = "point patterns from list",
                        "simulated realisations")
      explan <- if(dual) paren(paste(nsim2, "to estimate the mean and",
                                     nsim, "to calculate envelopes")) else ""
      cat(paste(action, Nsim, descrip, explan, "...\n"))
    }
    XsimList <- list()
  # start simulation loop 
    for(i in 1:Nsim) {
      if(verbose) progressreport(i, Nsim)
      Xsim <- eval(simexpr, envir=envir)
      if(!inherits(Xsim, Xclass))
        switch(simtype,
               csr={
                 stop(paste("Internal error:", Xobjectname, "not generated"))
               },
               rmh={
                 stop(paste("Internal error: rmh did not return an",
                            Xobjectname))
               },
               kppm={
                 stop(paste("Internal error: simulate.kppm did not return an",
                            Xobjectname))
               },
               expr={
                 stop(paste("Evaluating the expression", sQuote("simulate"),
                            "did not yield an", Xobjectname))
               },
               list={
                 stop(paste("Internal error: list entry was not an",
                            Xobjectname))
               },
               stop(paste("Internal error:", Xobjectname, "not generated"))
               )
      XsimList[[i]] <- Xsim
    }
    if(verbose) {
      cat(paste("Done.\n"))
      flush.console()
    }
    attr(XsimList, "internal") <- list(csr=csr)
    return(XsimList)
  }
  
  # capture main decision parameters
  EnvelopeInfo <-  list(call=cl,
                        Yname=Yname,
                        valname=valname,
                        csr=csr,
                        csr.theo=csr.theo,
                        pois=pois,
                        simtype=simtype,
                        nrank=nrank,
                        nsim=nsim,
                        Nsim=Nsim,
                        global=global,
                        dual=dual,
                        nsim2=nsim2,
                        VARIANCE=VARIANCE,
                        nSD=nSD,
                        use.weights=use.weights,
                        do.pwrong=do.pwrong)

  # ----------------------------------------
  ######### SIMULATE #######################
  # ----------------------------------------

  if(verbose) {
    action <- if(simtype == "list") "Extracting" else "Generating"
    descrip <- switch(simtype,
                      csr = "simulations of CSR",
                      rmh = paste("simulated realisations of fitted",
                        if(pois) "Poisson" else "Gibbs",
                        "model"),
                      kppm = "simulated realisations of fitted cluster model",
                      expr = "simulations by evaluating expression",
                      list = "point patterns from list",
                      "simulated patterns")
    explan <- if(dual) paren(paste(nsim2, "to estimate the mean and",
                                   nsim, "to calculate envelopes")) else ""
    cat(paste(action, Nsim, descrip, explan, "...\n"))
  }
  # determine whether simulated point patterns should be saved
  catchpatterns <- savepatterns && simtype != "list"
  Caughtpatterns <- list()
  # allocate space for computed function values
  nrvals <- length(rvals)
  simvals <- matrix(, nrow=nrvals, ncol=Nsim)
  # allocate space for weights to be computed
  if(compute.weights)
    weights <- numeric(Nsim)
  
  # inferred values of function argument 'r' or equivalent parameters
  if(identical(expected.arg, "r")) {
    # Kest, etc
    inferred.r.args <- list(r=rvals)
  } else if(identical(expected.arg, c("rmax", "nrval"))) {
    # K3est, etc
    inferred.r.args <- list(rmax=max(rvals), nrval=length(rvals))
  } else
  stop(paste("Don't know how to infer values of", commasep(expected.arg)))
    
  # arguments for function
  funargs <-
    resolve.defaults(inferred.r.args,
                     list(...),
                     if(usecorrection) list(correction="best") else NULL)
  
  # start simulation loop
  nerr <- 0
  for(i in 1:Nsim) {
    ok <- FALSE
    # safely generate a random pattern and apply function
    while(!ok) {
      Xsim <- eval(simexpr, envir=envir)
      # check valid point pattern
      if(!inherits(Xsim, Xclass))
        switch(simtype,
               csr=stop(paste("Internal error:", Xobjectname, "not generated")),
               rmh=stop(paste("Internal error: rmh did not return an",
                 Xobjectname)),
               kppm=stop(paste("Internal error:",
                 "simulate.kppm did not return an",
                 Xobjectname)),
               expr=stop(paste("Evaluating the expression", sQuote("simulate"),
                 "did not yield an", Xobjectname)),
               list=stop(paste("Internal error: list entry was not an",
                 Xobjectname)),
               stop(paste("Internal error:", Xobjectname, "not generated"))
               )
      if(catchpatterns)
        Caughtpatterns[[i]] <- Xsim
      if(savevalues)
        SavedValues[[i]] <- saveresultof(Xsim)
      if(compute.weights) {
        wti <- weightfun(Xsim)
        if(!is.numeric(wti))
          stop("weightfun did not return a numeric value")
        if(length(wti) != 1)
          stop("weightfun should return a single numeric value")
        weights[i] <- wti
      }
      # apply function safely
      funXsim <- try(do.call(fun, append(list(Xsim), funargs)))

      ok <- !inherits(funXsim, "try-error")
      
      if(!ok) {
        nerr <- nerr + 1
        if(nerr > maxnerr)
          stop("Exceeded maximum number of errors")
        cat("[retrying]\n")
      } 
    }

    # sanity checks
    if(i == 1) {
      if(!inherits(funXsim, "fv"))
        stop(paste("When applied to a simulated pattern, the function",
                   fname, "did not return an object of class",
                   sQuote("fv")))
      argname.sim <- fvnames(funXsim, ".x")
      valname.sim <- fvnames(funXsim, ".y")
      if(argname.sim != argname)
        stop(paste("The objects returned by", fname,
                   "when applied to a simulated pattern",
                   "and to the data pattern",
                   "are incompatible. They have different argument names",
                   sQuote(argname.sim), "and", sQuote(argname), 
                   "respectively"))
      if(valname.sim != valname)
        stop(paste("When", fname, "is applied to a simulated pattern",
                   "it provides an estimate named", sQuote(valname.sim), 
                   "whereas the estimate for the data pattern is named",
                   sQuote(valname),
                   ". Try using the argument", sQuote("correction"),
                   "to make them compatible"))
      rfunX    <- with(funX,    ".x")
      rfunXsim <- with(funXsim, ".x")
      if(!identical(rfunX, rfunXsim))
        stop(paste("When", fname, "is applied to a simulated pattern,",
                   "the values of the argument", sQuote(argname.sim),
                   "are different from those used for the data."))
    }

    if(tran) {
      # extract only the recommended value
      if(csr.theo) 
        funXsim <- funXsim[, c(argname, valname, "theo")]
      else
        funXsim <- funXsim[, c(argname, valname)]
      # apply the transformation to it
      funXsim <- eval(transform.funXsim)
    }

    # extract the values for simulation i
    simvals.i <- funXsim[[valname]]
    if(length(simvals.i) != nrvals)
      stop("Vectors of function values have incompatible lengths")
      
    simvals[ , i] <- funXsim[[valname]]
    if(verbose)
      progressreport(i, Nsim)
  }
  ##  end simulation loop
  
  if(verbose) {
    cat("\nDone.\n")
    flush.console()
  }

  # ...........................................................
  # save functions and/or patterns if so commanded

  if(savefuns) {
    alldata <- cbind(rvals, simvals)
    simnames <- paste("sim", 1:nsim, sep="")
    colnames(alldata) <- c("r", simnames)
    alldata <- as.data.frame(alldata)
    SimFuns <- fv(alldata,
                  argu="r",
                  ylab=attr(funX, "ylab"),
                  valu="sim1",
                  fmla= deparse(. ~ r),
                  alim=attr(funX, "alim"),
                  labl=names(alldata),
                  desc=c("distance argument r",
                    paste("Simulation ", 1:nsim, sep="")),
                  fname=attr(funX, "fname"))
    fvnames(SimFuns, ".") <- simnames
  } 
  if(savepatterns)
    SimPats <- if(simtype == "list") SimDataList else Caughtpatterns

  ######### COMPUTE ENVELOPES #######################

  etype <- if(global) "global" else if(VARIANCE) "variance" else "pointwise"
  if(dual) {
    jsim <- 1:nsim
    jsim.mean <- nsim + 1:nsim2
  } else {
    jsim <- jsim.mean <- NULL
  }

  result <- envelope.matrix(simvals, funX=funX,
                            jsim=jsim, jsim.mean=jsim.mean,
                            type=etype, csr=csr, use.theory=csr.theo,
                            nrank=nrank, ginterval=ginterval, nSD=nSD,
                            Yname=Yname, do.pwrong=do.pwrong,
                            weights=weights)

  # tack on envelope information
  attr(result, "einfo") <- EnvelopeInfo

  # tack on functions and/or patterns if so commanded   
  if(savefuns)
    attr(result, "simfuns") <- SimFuns
  if(savepatterns) {
    attr(result, "simpatterns") <- SimPats
    attr(result, "datapattern") <- X
  }
  # save function weights 
  if(use.weights)
    attr(result, "weights") <- weights

  # undocumented - tack on values of some other quantity
  if(savevalues) {
    attr(result, "simvalues") <- SavedValues
    attr(result, "datavalue") <- saveresultof(X)
  }
  return(result)
}


plot.envelope <- function(x, ...) {
  shade.given <- ("shade" %in% names(list(...)))
  shade.implied <- !is.null(fvnames(x, ".s"))
  if(!(shade.given || shade.implied)) {
    # ensure x has default 'shade' attribute
    # (in case x was produced by an older version of spatstat)
    if(all(c("lo", "hi") %in% colnames(x)))
      fvnames(x, ".s") <- c("lo", "hi")
    else warning("Unable to determine shading for envelope")
  }
  NextMethod("plot")
}

print.envelope <- function(x, ...) {
  e <- attr(x, "einfo")
  g <- e$global
  csr <- e$csr
  pois <- e$pois
  if(is.null(pois)) pois <- csr
  simtype <- e$simtype
  nr <- e$nrank
  nsim <- e$nsim
  V <- e$VARIANCE
  fname <- deparse(attr(x, "ylab"))
  type <- if(V) paste("Pointwise", e$nSD, "sigma") else
          if(g) "Simultaneous" else "Pointwise" 
  cat(paste(type, "critical envelopes for", fname, "\n"))
  cat(paste("and observed value for", sQuote(e$Yname), "\n"))
  if(!is.null(valname <- e$valname))
    cat(paste("Edge correction:",
              dQuote(valname), "\n"))
  # determine *actual* type of simulation
  descrip <-
    if(csr) "simulations of CSR"
    else if(!is.null(simtype)) {
      switch(simtype,
             csr="simulations of CSR",
             rmh=paste("simulations of fitted",
               if(pois) "Poisson" else "Gibbs",
               "model"),
             kppm="simulations of fitted cluster model",
             expr="evaluations of user-supplied expression",
             list="point pattern datasets in user-supplied list",
             funs="columns of user-supplied data")
    } else "simulations of fitted model"
  #  
  cat(paste("Obtained from", nsim, descrip, "\n"))
  #
  if(!is.null(e$dual) && e$dual) 
    cat(paste("Theoretical (i.e. null) mean value of", fname,
              "estimated from a separate set of",
              e$nsim2, "simulations\n"))
  if(!is.null(attr(x, "simfuns"))) 
    cat("(All simulated function values are stored)\n")
  if(!is.null(attr(x, "simpatterns"))) 
    cat("(All simulated point patterns are stored)\n")
  alpha <- if(g) { nr/(nsim+1) } else { 2 * nr/(nsim+1) }
  if(!V) {
    # significance interpretation!
    cat(paste("Significance level of",
              if(!g) "pointwise",
              "Monte Carlo test:",
              paste(if(g) nr else 2 * nr,
                    "/", nsim+1, sep=""),
              "=", alpha, "\n"))
  }
  if(!is.null(pwrong <- attr(x, "pwrong"))) {
    cat(paste("\t[Estimated significance level of pointwise excursions:",
              paste("pwrong=", signif(pwrong, 3), "]\n", sep="")))
  }
  cat("\n")
  NextMethod("print")
}
                  
summary.envelope <- function(object, ...) {
  e <- attr(object, "einfo")
  g <- e$global
  nr <- e$nrank
  nsim <- e$nsim
  csr <- e$csr
  pois <- e$pois
  if(is.null(pois)) pois <- csr
  has.theo <- "theo" %in% fvnames(object, "*")
  csr.theo <- csr && has.theo
  simtype <- e$simtype
  fname <- deparse(attr(object, "ylab"))
  V <- e$VARIANCE
  type <- if(V) paste("Pointwise", e$nSD, "sigma") else
          if(g) "Simultaneous" else "Pointwise" 
  cat(paste(type, "critical envelopes for", fname, "\n"))
  # determine *actual* type of simulation
  descrip <-
    if(csr) "simulations of CSR"
    else if(!is.null(simtype)) {
      switch(simtype,
             csr="simulations of CSR",
             rmh=paste("simulations of fitted",
               if(pois) "Poisson" else "Gibbs",
               "model"),
             kppm="simulations of fitted cluster model",
             expr="evaluations of user-supplied expression",
             list="point pattern datasets in user-supplied list",
             funs="columns of user-supplied data",
             "simulated point patterns")
    } else "simulations of fitted model"
  #  
  cat(paste("Obtained from", nsim, descrip, "\n"))
  #
  if(!is.null(attr(object, "simfuns")))
    cat(paste("(All", nsim, "simulated function values",
              "are stored in attr(,", dQuote("simfuns"), ") )\n"))
  if(!is.null(attr(object, "simpatterns")))
    cat(paste("(All", nsim, "simulated point patterns",
              "are stored in attr(,", dQuote("simpatterns"), ") )\n"))
  #
  if(V) {
    # nSD envelopes
    cat(paste("Envelopes computed as sample mean plus/minus",
              e$nSD, "sample standard deviations\n"))
  } else {
    # critical envelopes
    lo.ord <- if(nr == 1) "minimum" else paste(ordinal(nr), "smallest")
    hi.ord <- if(nr == 1) "maximum" else paste(ordinal(nsim-nr+1), "largest")
    if(g) 
      cat(paste("Envelopes computed as",
                if(csr.theo) "theoretical curve" else "mean of simulations",
                "plus/minus", hi.ord,
                "simulated value of maximum absolute deviation\n"))
    else {
      cat(paste("Upper envelope: pointwise", hi.ord,"of simulated curves\n"))
      cat(paste("Lower envelope: pointwise", lo.ord,"of simulated curves\n"))
    }
    alpha <- if(g) { nr/(nsim+1) } else { 2 * nr/(nsim+1) }
    cat(paste("Significance level of Monte Carlo test:",
              paste(if(g) nr else 2 * nr,
                    "/", nsim+1, sep=""),
              "=", alpha, "\n"))
  } 
  cat(paste("Data:", e$Yname, "\n"))
  return(invisible(NULL))
}
  

# envelope.matrix

# core functionality to compute envelope values

# theory = funX[["theo"]]
# observed = fX

envelope.matrix <- function(Y, ...,
                            rvals=NULL, observed=NULL, theory=NULL, 
                            funX=NULL,
                            nsim=NULL, nsim2=NULL,
                            jsim=NULL, jsim.mean=NULL,
                            type=c("pointwise", "global", "variance"),
                            csr=FALSE, use.theory = csr, 
                            nrank=1, ginterval=NULL, nSD=2,
                            savefuns=FALSE,
                            check=TRUE,
                            Yname=NULL,
                            do.pwrong=FALSE,
                            weights=NULL,
                            precomputed=NULL) {
  if(is.null(Yname))
    Yname <- short.deparse(substitute(Y))

  type <- match.arg(type)

  if(!is.null(funX))
    stopifnot(is.fv(funX))

  pwrong <- NULL
  use.weights <- !is.null(weights)
  cheat <- !is.null(precomputed)

  if(is.null(rvals) && is.null(observed) && !is.null(funX)) {
    # assume funX is summary function for observed data
    rvals <- with(funX, .x)
    observed <- with(funX, .y)
    theory <- if(use.theory && "theo" %in% names(funX)) with(funX, theo) else NULL
  } else if(check) {
    # validate vectors of data
    if(is.null(rvals)) stop("rvals must be supplied")
    if(is.null(observed)) stop("observed must be supplied")
    if(!is.null(Y)) stopifnot(length(rvals) == nrow(Y))
    stopifnot(length(observed) == length(rvals))
  }

  if(use.theory) {
    use.theory <- !is.null(theory)
    if(use.theory && check) stopifnot(length(theory) == length(rvals))
  }

  simvals <- Y
  fX <- observed

  atr <- if(!is.null(funX)) attributes(funX) else
         list(alim=range(rvals),
              ylab=quote(f(r)),
              yexp=quote(f(r)),
              fname="f")

  if(!cheat) {
    # ................   standard calculation .....................
    # validate weights
    if(use.weights) 
      check.nvector(weights, ncol(simvals), 
                    things="simulated functions", naok=TRUE)

    # determine numbers of columns used
      Ncol <- ncol(simvals)
      if(Ncol < 2)
        stop("Need at least 2 columns of function values")
      
      if(is.null(jsim) && !is.null(nsim)) {
        # usual case - 'nsim' determines 'jsim'
        if(nsim > Ncol)
          stop(paste(nsim, "simulations are not available; only",
                     Ncol, "columns provided"))
        jsim <- 1:nsim
        if(!is.null(nsim2)) {
          # 'nsim2' determines 'jsim.mean'
          if(nsim + nsim2 > Ncol)
            stop(paste(nsim, "+", nsim2, "=", nsim+nsim2, 
                       "simulations are not available; only",
                       Ncol, "columns provided"))
          jsim.mean <- nsim + 1:nsim2
        }
      }
      
      restrict.columns <- !is.null(jsim)
      dual <- !is.null(jsim.mean)

  } else {
    # ................ precomputed values ..................
    # validate weights
    if(use.weights) 
      check.nvector(weights, nsim,
                    things="simulations", naok=TRUE)
    restrict.columns <- FALSE
    dual <- FALSE
  }

  shadenames <- NULL
  
  switch(type,
         pointwise = {
           # ....... POINTWISE ENVELOPES ...............................
           if(cheat) {
             stopifnot(checkfields(precomputed, c("lo", "hi")))
             lo <- precomputed$lo
             hi <- precomputed$hi
           } else {
             simvals[is.infinite(simvals)] <- NA
             if(restrict.columns) {
               simvals <- simvals[,jsim]
               if(use.weights) weights <- weights[jsim]
             }
             nsim <- ncol(simvals)
             nsim.mean <- NULL
             if(nrank == 1) {
               lohi <- apply(simvals, 1, range)
             } else {
               lohi <- apply(simvals, 1,
                             function(x, n) { sort(x)[n] },
                             n=c(nrank, nsim-nrank+1))
             }
             lo <- lohi[1,]
             hi <- lohi[2,]
           }
           lo.name <- paste("lower pointwise envelope of %s from simulations")
           hi.name <- paste("upper pointwise envelope of %s from simulations")
           #
           if(use.theory) {
             results <- data.frame(r=rvals,
                                   obs=fX,
                                   theo=theory,
                                   lo=lo,
                                   hi=hi)
           } else {
             m <- if(cheat) precomputed$mmean else 
                  if(!use.weights) apply(simvals, 1, mean, na.rm=TRUE) else
                  apply(simvals, 1, weighted.mean, w=weights, na.rm=TRUE)
             results <- data.frame(r=rvals,
                                   obs=fX,
                                   mmean=m,
                                   lo=lo,
                                   hi=hi)
           }
           shadenames <- c("lo", "hi")
           if(do.pwrong) {
             # estimate the p-value for the 'wrong test'
             if(cheat) {
               pwrong <- precomputed$pwrong
               do.pwrong <- !is.null(pwrong) && !badprobability(pwrong, FALSE)
             } else {
               dataranks <- t(apply(simvals, 1, rank, ties.method="random"))
               is.signif <- (dataranks <= nrank) | (dataranks >= nsim-nrank+1)
               is.signif.somewhere <- apply(is.signif, 2, any)
               pwrong <- sum(is.signif.somewhere)/nsim
             }
           }
         },
         global = {
           # ..... SIMULTANEOUS ENVELOPES ..........................
           if(cheat) {
             # ... use precomputed values ..
             stopifnot(checkfields(precomputed, c("lo", "hi")))
             lo <- precomputed$lo
             hi <- precomputed$hi
             if(use.theory) {
               reference <- theory
             } else {
               stopifnot(checkfields(precomputed, "mmean"))
               reference <- precomputed$mmean
             }
             nsim.mean <- NULL
             domain <- rep.int(TRUE, length(rvals))
           } else {
             # ... normal case: compute envelopes from simulations
             if(!is.null(ginterval)) {
               domain <- (rvals >= ginterval[1]) & (rvals <= ginterval[2])
               funX <- funX[domain, ]
               simvals <- simvals[domain, ]
             } else domain <- rep.int(TRUE, length(rvals))
             simvals[is.infinite(simvals)] <- NA
             if(use.theory) {
               reference <- theory[domain]
               if(restrict.columns) {
                 simvals <- simvals[, jsim]
                 if(use.weights) weights <- weights[jsim]
               }
               nsim.mean <- NULL
             } else if(dual) {
               # Estimate the mean from one set of columns
               # Form envelopes from another set of columns
               simvals.mean <- simvals[, jsim.mean]
               reference <- mmean <-
                 if(!use.weights) apply(simvals.mean, 1, mean, na.rm=TRUE) else
                 apply(simvals.mean, 1, weighted.mean, w=weights[jsim.mean],
                       na.rm=TRUE)
               nsim.mean <- ncol(simvals.mean)
               # retain only columns used for envelope
               simvals <- simvals[, jsim]
             } else {
               # Compute the mean and envelopes using the same data
               if(restrict.columns) {
                 simvals <- simvals[, jsim]
                 if(use.weights) weights <- weights[jsim]
               }
               reference <- mmean <-
                 if(!use.weights) apply(simvals.mean, 1, mean, na.rm=TRUE) else
                 apply(simvals.mean, 1, weighted.mean, w=weights, na.rm=TRUE)
               nsim.mean <- NULL
             }
             nsim <- ncol(simvals)
             # compute max absolute deviations
             deviations <- sweep(simvals, 1, reference)
             suprema <- apply(abs(deviations), 2, max, na.rm=TRUE)
             # ranked deviations
             dmax <- sort(suprema)[nsim-nrank+1]
             # simultaneous bands
             lo <- reference - dmax
             hi <- reference + dmax
           }
           
           lo.name <- "lower critical boundary for %s"
           hi.name <- "upper critical boundary for %s"

           if(use.theory)
             results <- data.frame(r=rvals[domain],
                                   obs=fX[domain],
                                   theo=reference,
                                   lo=lo,
                                   hi=hi)
           else
             results <- data.frame(r=rvals[domain],
                                   obs=fX[domain],
                                   mmean=reference,
                                   lo=lo,
                                   hi=hi)
           shadenames <- c("lo", "hi")
           if(do.pwrong)
             warning(paste("Argument", sQuote("do.pwrong=TRUE"), "ignored;",
                           "it is not relevant to global envelopes"))
         },
         variance={
           # ....... POINTWISE MEAN, VARIANCE etc ......................
           if(cheat) {
             # .... use precomputed values ....
             stopifnot(checkfields(precomputed, c("Ef", "varf")))
             Ef   <- precomputed$Ef
             varf <- precomputed$varf
           } else {
             # .... normal case: compute from simulations
             simvals[is.infinite(simvals)] <- NA
             if(restrict.columns) {
               simvals <- simvals[, jsim]
               if(use.weights) weights <- weights[jsim]
             }
             nsim <- ncol(simvals)
             if(!use.weights) {
               Ef   <- apply(simvals, 1, mean, na.rm=TRUE)
               varf <- apply(simvals, 1, var,  na.rm=TRUE)
             } else {
               Ef   <- apply(simvals, 1, weighted.mean, w=weights, na.rm=TRUE)
               varf <- apply(simvals, 1, weighted.var,  w=weights, na.rm=TRUE)
             }
           }
           nsim.mean <- NULL
           # derived quantities
           sd <- sqrt(varf)
           stdres <- (fX-Ef)/sd
           stdres[!is.finite(stdres)] <- NA
           # critical limits
           lo <- Ef - nSD * sd
           hi <- Ef + nSD * sd
           lo.name <- paste("lower", nSD, "sigma critical limit for %s")
           hi.name <- paste("upper", nSD, "sigma critical limit for %s")
           # confidence interval 
           loCI <- Ef - nSD * sd/sqrt(nsim)
           hiCI <- Ef + nSD * sd/sqrt(nsim)
           loCI.name <- paste("lower", nSD, "sigma confidence bound",
                              "for mean of simulated %s")
           hiCI.name <- paste("upper", nSD, "sigma confidence bound",
                              "for mean of simulated %s")

           # put together
           if(use.theory) {
             results <- data.frame(r=rvals,
                                   obs=fX,
                                   theo=theory,
                                   lo=lo,
                                   hi=hi)
             shadenames <- c("lo", "hi")
             morestuff <- data.frame(mmean=Ef,
                                     var=varf,
                                     res=fX-Ef,
                                     stdres=stdres,
                                     loCI=loCI,
                                     hiCI=hiCI)
             mslabl <- c("bar(%s)(r)",
                         "paste(var,%s)(r)",
                         "paste(res,%s)(r)",
                         "paste(stdres,%s)(r)",
                         "%s[loCI](r)", "%s[hiCI](r)")
             wted <- if(use.weights) "weighted " else NULL
             msdesc <- c(paste0(wted, "sample mean of %s from simulations"),
                         paste0(wted, "sample variance of %s from simulations"),
                         "raw residual",
                         "standardised residual",
                         loCI.name, hiCI.name)
           } else {
             results <- data.frame(r=rvals,
                                   obs=fX,
                                   mmean=Ef,
                                   lo=lo,
                                   hi=hi)
             shadenames <- c("lo", "hi")
             morestuff <- data.frame(var=varf,
                                     res=fX-Ef,
                                     stdres=stdres,
                                     loCI=loCI,
                                     hiCI=hiCI)
             mslabl <- c("paste(var,%s)(r)",
                         "paste(res,%s)(r)",
                         "paste(stdres,%s)(r)",
                         "%s[loCI](r)", "%s[hiCI](r)")
             msdesc <- c(paste0(if(use.weights) "weighted " else NULL,
                                "sample variance of %s from simulations"),
                         "raw residual",
                         "standardised residual",
                         loCI.name, hiCI.name)
           }
           if(do.pwrong) {
             # estimate the p-value for the 'wrong test'
             if(cheat) {
               pwrong <- precomputed$pwrong
               do.pwrong <- !is.null(pwrong) && !badprobability(pwrong, FALSE)
             } else {
               is.signif <- (simvals < lo) | (simvals > hi)
               is.signif.somewhere <- apply(is.signif, 2, any)
               pwrong <- sum(is.signif.somewhere)/nsim
             }
           }
         }
         )

  ############  WRAP UP #########################

  if(use.theory) {
    # reference is computed curve `theo'
    reflabl <- "%s[theo](r)"
    refdesc <- paste0("theoretical value of %s", if(csr) " for CSR" else NULL)
  } else {
    # reference is sample mean of simulations
    reflabl <- "bar(%s)(r)"
    refdesc <- paste0(if(use.weights) "weighted " else NULL,
                      "sample mean of %s from simulations")
  }
  
  result <- fv(results,
               argu="r",
               ylab=atr$ylab,
               valu="obs",
               fmla= deparse(. ~ r),
               alim=atr$alim,
               labl=c("r", "%s[obs](r)",
                 reflabl,
                 "%s[lo](r)", "%s[hi](r)"),
               desc=c("distance argument r",
                 "observed value of %s for data pattern",
                 refdesc, lo.name, hi.name),
               fname=atr$fname,
               yexp =atr$yexp)

  # columns to be plotted by default
  dotty <- c("obs", if(use.theory) "theo" else "mmean", "hi", "lo")

  if(type == "variance") {
    # add more stuff
    result <- bind.fv(result, morestuff, mslabl, msdesc)
    if(use.theory) dotty <- c(dotty, "mmean")
  }

  fvnames(result, ".") <- dotty
  fvnames(result, ".s") <- shadenames

  unitname(result) <- unitname(funX)
  class(result) <- c("envelope", class(result))

  # tack on envelope information
  attr(result, "einfo") <- list(global = (type =="global"),
                                csr = csr,
                                use.theory = use.theory,
                                csr.theo = csr && use.theory,
                                simtype = "funs",
                                nrank = nrank,
                                nsim = nsim,
                                VARIANCE = (type == "variance"),
                                nSD = nSD,
                                valname = NULL,
                                dual = dual,
                                nsim = nsim,
                                nsim2 = nsim.mean,
                                Yname = Yname,
                                do.pwrong=do.pwrong,
                                use.weights=use.weights)

  # tack on saved functions
  if(savefuns) {
    alldata <- cbind(rvals, simvals)
    simnames <- paste("sim", 1:nsim, sep="")
    colnames(alldata) <- c("r", simnames)
    alldata <- as.data.frame(alldata)
    SimFuns <- fv(alldata,
                   argu="r",
                   ylab=atr$ylab,
                   valu="sim1",
                   fmla= deparse(. ~ r),
                   alim=atr$alim,
                   labl=names(alldata),
                   desc=c("distance argument r",
                     paste("Simulation ", 1:nsim, sep="")))
    fvnames(SimFuns, ".") <- simnames
    attr(result, "simfuns") <- SimFuns
  }
  if(do.pwrong)
    attr(result, "pwrong") <- pwrong
  if(use.weights)
    attr(result, "weights") <- weights
  return(result)
}


envelope.envelope <- function(Y, fun=NULL, ...,
                              transform=NULL, global=FALSE, VARIANCE=FALSE) {

  Yname <- short.deparse(substitute(Y))

  stopifnot(inherits(Y, "envelope"))
  Yorig <- Y

  csr <- list(...)$internal$csr
  if(is.null(csr))
    csr <- attr(Y, "einfo")$csr
  
  X  <- attr(Y, "datapattern")
  sf <- attr(Y, "simfuns")
  sp <- attr(Y, "simpatterns")
  wt <- attr(Y, "weights")
  
  if(is.null(fun) && is.null(sf)) {
    # No simulated functions - must compute them from simulated patterns
    if(is.null(sp))
      stop(paste("Cannot compute envelope:",
                 "Y does not contain simulated functions",
                 "(was not generated with savefuns=TRUE)",
                 "and does not contain simulated patterns",
                 "(was not generated with savepatterns=TRUE)"))
    # set default fun=Kest
    fun <- Kest
  }
  
  if(!is.null(fun)) {
    # apply new function 
    # point patterns are required
    if(is.null(sp))
      stop(paste("Object Y does not contain simulated point patterns",
                 "(attribute", dQuote("simpatterns"), ");",
                 "cannot apply a new", sQuote("fun")))
    if(is.null(X))
      stop(paste("Cannot apply a new", sQuote("fun"),
                 "; object Y generated by an older version of spatstat"))
    result <- do.call(envelope,
                      resolve.defaults(list(Y=X, fun=fun, simulate=sp),
                                       list(...),
                                       list(transform=transform,
                                            Yname=Yname,
                                            nsim=length(sp),
                                            weights=wt),
                                       .StripNull=TRUE))
  } else {
    # compute new envelope with existing simulated functions
    if(is.null(sf)) 
      stop(paste("Y does not contain a", dQuote("simfuns"), "attribute",
                 "(it was not generated with savefuns=TRUE)"))

    if(!is.null(transform)) {
      # Apply transformation to Y and sf
      stopifnot(is.expression(transform))
      cc <- dotexpr.to.call(transform, "Y", "eval.fv")
      Y <- eval(cc)
      cc <- dotexpr.to.call(transform, "sf", "eval.fv")
      sf <- eval(cc)
    }

    # extract simulated function values
    df <- as.data.frame(sf)
    rname <- fvnames(sf, ".x")
    df <- df[, (names(df) != rname)]

    # interface with 'envelope.matrix'
    etype <- if(global) "global" else if(VARIANCE) "variance" else "pointwise"
    result <- do.call(envelope.matrix,
                      resolve.defaults(list(Y=as.matrix(df)),
                                       list(...),
                                       list(type=etype,
                                            csr=csr,
                                            funX=Y, 
                                            Yname=Yname,
                                            weights=wt),
                                       .StripNull=TRUE))
  }

  if(!is.null(transform)) {
    # post-process labels
    labl <- attr(result, "labl")
    dnames <- colnames(result)
    dnames <- dnames[dnames %in% fvnames(result, ".")]
    # expand "."
    ud <- as.call(lapply(c("cbind", dnames), as.name))
    expandtransform <- eval(substitute(substitute(tr, list(.=ud)),
                                       list(tr=transform[[1]])))
    # compute new labels 
    attr(result, "fname") <- attr(Yorig, "fname")
    mathlabl <- as.character(fvlegend(result, expandtransform))
    # match labels to columns
    evars <- all.vars(expandtransform)
    used.dotnames <- evars[evars %in% dnames]
    mathmap <- match(colnames(result), used.dotnames)
    okmath <- !is.na(mathmap)
    # update appropriate labels
    labl[okmath] <- mathlabl[mathmap[okmath]]
    attr(result, "labl") <- labl
  }
  
  # Tack on envelope info
  copyacross <- c("Yname", "csr.theo", "simtype")
  attr(result, "einfo")[copyacross] <- attr(Yorig, "einfo")[copyacross]
  attr(result, "einfo")$csr <- csr
  # Save data
  
  return(result)
}

pool <- function(...) {
  UseMethod("pool")
}

pool.envelope <- function(..., savefuns=FALSE, savepatterns=FALSE) {
  Yname <- short.deparse(sys.call())
  if(nchar(Yname) > 60) Yname <- paste(substr(Yname, 1, 40), "[..]")
  Elist <- unname(list(...))
  nE <-  length(Elist)
  if(nE == 0) return(NULL)
  # ........ validate envelopes .....................
  # All arguments must be envelopes
  notenv <- !unlist(lapply(Elist, inherits, what="envelope"))
  if(any(notenv)) {
    n <- sum(notenv)
    why <- paste(ngettext(n, "Argument", "Arguments"),
                 commasep(which(notenv)),
                 ngettext(n, "does not", "do not"),
                 "belong to the class",
                 dQuote("envelope"))
    stop(why)
  }
  # Only one envelope?
  if(nE == 1)
    return(Elist[[1]])
  # Envelopes must be compatible
  ok <- do.call(compatible, Elist)
  if(!ok)
    stop("Envelopes are not compatible")
  # ... reconcile parameters in different envelopes .......
  eilist <- lapply(Elist, attr, which="einfo")
  global    <- resolveEinfo(eilist, "global",   FALSE)
  VARIANCE  <- resolveEinfo(eilist, "VARIANCE", FALSE)
  simtype   <- resolveEinfo(eilist, "simtype",  "funs",
          "Envelopes were generated using different types of simulation")
  csr         <- resolveEinfo(eilist, "csr", FALSE, NULL)
  csr.theo    <- resolveEinfo(eilist, "csr.theo", FALSE, NULL)
  use.weights <- resolveEinfo(eilist, "use.weights" , FALSE,
     "Weights were used in some, but not all, envelopes: they will be ignored")
  #
  weights <-
    if(use.weights) unlist(lapply(Elist, attr, which="weights")) else NULL
  type <- if(global) "global" else if(VARIANCE) "variance" else "pointwise"
    
  # ........ validate saved functions .....................
  if(savefuns || !VARIANCE) {
    # Individual simulated functions are required
    SFlist <- lapply(Elist, attr, which="simfuns")
    isnul <- unlist(lapply(SFlist, is.null))
    if(any(isnul)) {
      n <- sum(isnul)
      comply <- if(!VARIANCE) "compute the envelope:" else
                "save the simulated functions:"
      why <- paste("Cannot", comply,
                   ngettext(n, "argument", "arguments"),
                   commasep(which(isnul)),
                   ngettext(n, "does not", "do not"),
                   "contain a", dQuote("simfuns"), "attribute",
                   "(not generated with savefuns=TRUE)")
      stop(why)
    }
    # Simulated functions must be the same function
    fnames <- unique(unlist(lapply(SFlist, attr, which="fname")))
    if(length(fnames) > 1)
      stop(paste("Envelope objects contain values",
                 "of different functions:",
                 commasep(sQuote(fnames))))
    # vectors of r values must be identical
    rlist <- lapply(SFlist, function(z) { with(z, .x) })
    rvals <- rlist[[1]]
    samer <- unlist(lapply(rlist, identical, y=rvals))
    if(!all(samer))
      stop(paste("Simulated function values are not compatible",
                 "(different values of function argument)"))
  }
  # compute pooled envelope
  switch(type,
         global = ,
         pointwise = {
           # assemble function values into one matrix
           getsimvals <- function(z) {
             rname <- fvnames(z, ".x")
             as.matrix(as.data.frame(z)[, names(z) != rname])
           }
           matlist <- lapply(SFlist, getsimvals)
           bigmat <- do.call(cbind, matlist)
           # ..... ready to compute
           result <- envelope(bigmat, funX=Elist[[1]],
                              type=type, csr=csr, Yname=Yname,
                              weights=weights,
                              savefuns=savefuns)
         },
         variance = {
           # Pool sample means and variances
           nsims <- unlist(lapply(eilist, getElement, name="nsim"))
           mmeans <- lapply(Elist, getElement, name="mmean")
           vars   <- lapply(Elist, getElement, name="var")
           mmeans <- matrix(unlist(mmeans), ncol=nE)
           vars   <- matrix(unlist(vars),   ncol=nE)
           if(!use.weights) {
             w.mean <- nsims
             d.mean <- sum(nsims)
             w.var  <- nsims - 1
             d.var  <- sum(nsims) - 1
           } else {
             weightlist <- lapply(Elist, attr, which="weights")
             w.mean <- unlist(lapply(weightlist, sum))
             d.mean <- sum(w.mean)
             ssw <- unlist(lapply(weightlist, function(x) {sum((x/sum(x))^2)}))
             w.var  <- w.mean * (1 - ssw)
             d.var <-  d.mean * (1 - sum(ssw))
           }
           poolmmean <- as.numeric(mmeans %*% matrix(w.mean/d.mean, ncol=1))
           within <- vars %*% matrix(w.var, ncol=1)
           between <- ((mmeans - poolmmean[])^2) %*% matrix(w.mean, ncol=1)
           poolvar <- as.numeric((within + between)/d.var)
           # feed precomputed data to envelope.matrix
           pc <- list(Ef=poolmmean[],
                      varf=poolvar[])
           nsim <- sum(nsims)
           result <- envelope.matrix(NULL, funX=Elist[[1]],
                                     type=type, csr=csr, Yname=Yname,
                                     weights=weights,
                                     savefuns=savefuns,
                                     nsim=nsim,
                                     precomputed=pc)
         })
  
  # ..............saved patterns .....................
  if(savepatterns) {
    SPlist <- lapply(Elist, attr, which="simpatterns")
    isnul <- unlist(lapply(SPlist, is.null))
    if(any(isnul)) {
      n <- sum(isnul)
      why <- paste("Cannot save the simulated patterns:",
                   ngettext(n, "argument", "arguments"),
                   commasep(which(isnul)),
                   ngettext(n, "does not", "do not"),
                   "contain a", dQuote("simpatterns"), "attribute",
                   "(not generated with savepatterns=TRUE)")
      warning(why)
    } else {
      attr(result, "simpatterns") <- Reduce(SPlist, append)
    }
  }

  dotnames   <- lapply(Elist, fvnames, a=".")
  dn <- dotnames[[1]]
  if(all(unlist(lapply(dotnames, identical, y=dn))))
    fvnames(result, ".") <- dn
  
  shadenames <- lapply(Elist, fvnames, a=".s")
  sh <- shadenames[[1]]
  if(all(unlist(lapply(shadenames, identical, y=sh))))
    fvnames(result, ".s") <- sh
  
  return(result)
}

# resolve matching entries in different envelope objects
#   x is a list of envelope info objects

resolveEinfo <- function(x, what, fallback, warn) {
  y <- unique(unlist(lapply(x, getElement, name=what)))
  if(length(y) == 1)
    return(y)
  if(missing(warn))
    warn <- paste("Envelopes were generated using different values",
                  "of argument", paste(sQuote(what), ";", sep=""),
                  "reverting to default value")
  if(!is.null(warn))
    warning(warn, call.=FALSE)
  return(fallback)
}

