# Copyright (C) 2010 by 52 North Initiative for Geospatial Open Source Software GmbH, Contact: info@52north.org
# This program is free software; you can redistribute and/or modify it under the terms of the GNU General Public License version 2 as published by the Free Software Foundation. This program is distributed WITHOUT ANY WARRANTY; even without the implied WARRANTY OF MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with this program (see gpl-2.0.txt). If not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA or visit the Free Software Foundation web page, http://www.fsf.org.
# Author: Daniel Nuest (daniel.nuest@uni-muenster.de)
# Project: sos4R - visit the project web page, http://www.nordholmen.net/sos4r
library("sos4R")

##############################################################################
# Coastlab SOS - http://www.coastlab.org/
#
# This SOS provides COSYNA data, see the project website at
# http://www.hzg.de/institute/coastal_research/structure/operational_systems/KOK/projects/ICON/index.html
#
# GEOSS description:
# https://geossregistries.info/geosspub/service_details_ns.jsp?serviceId=urn:uuid:b7e0c9d4-9c4a-428c-adb9-fc0efebc9798
#
# Data disclaimer: http://www.coastlab.org/Disclaimer.html
#
# Web Portal: http://kofserver2.hzg.de/codm/
# You can also plot the data here: http://tsdata.hzg.de/index.cgi?seite=plot_form
#
coastlab <- SOS(url = "http://kopc02.gkss.de/sos/sos.py",
		method = SosSupportedConnectionMethods()[["GET"]])

coastlab.off <- sosOfferings(coastlab)
names(coastlab.off)

jade1 <- coastlab.off[["Pile_Jade1"]]
sosObservedProperties(jade1)

jade1.watertemp <- getObservation(sos = coastlab, offering = jade1,
		observedProperty = list("WaterTemperature"), verbose = TRUE,
		eventTime = sosCreateEventTimeList(sosCreateTimePeriod(
						sos = coastlab,
						begin = as.POSIXct(Sys.time() - 3600 * 24 * 180),
						end = as.POSIXct(Sys.time()))))

# TODO unhandled response document, it contains om:resultDefinition ...
#
#<?xml version="1.0" encoding="UTF-8"?>
#		<om:Observation xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:swe="http://www.opengis.net/swe/0" xmlns:gml="http://www.opengis.net/gml" xmlns:om="http://www.opengis.net/om" xmlns:xlink="http://www.w3.org/1999/xlink" xsi:schemaLocation="http://www.opengis.net/om http://amb25.stccmop.org/schemas/sos/current/sosGetObservation.xsd" gml:id="WaterTemperature">
#		<gml:description>None</gml:description>
#		<gml:name>Pile_Jade1</gml:name>
#		<gml:location>
#		<!-- geometry containing all tuples in this observation -->
#		<gml:Envelope>
#		<gml:lowerCorner srsName="urn:ogc:def:crs:EPSG:6.5:4326">53.516566 8.188217</gml:lowerCorner>
#		<gml:upperCorner srsName="urn:ogc:def:crs:EPSG:6.5:4326">53.516566 8.188217</gml:upperCorner>
#		</gml:Envelope>
#		</gml:location>
#		<!--  Time of response- use TimePeriod for a series of data  -->
#		<!--  or TimeInstant for a single measurement  -->
#		
#		<gml:TimePeriod gml:id="DATA_TIME">
#		<gml:beginPosition>2010-09-19T13:21:38</gml:beginPosition>
#		<gml:endPosition>2011-03-18T12:21:38</gml:endPosition>
#		</gml:TimePeriod>
#		
#		<!-- Procedure -->
#		<om:procedure/>
#		<!-- the property measured -->
#		<om:observedProperty xlink:href="WaterTemperature"/>
#		<!-- Feature Of Interest -->
#		<om:featureOfInterest xlink:href="urn:bodyofwater"/>
#		
#		<!-- Result Structure and Encoding -->
#		<om:resultDefinition>
#		<swe:DataBlockDefinition>
#		<swe:components name="WaterTemperature">
#		<swe:DataRecord>
#		<swe:field name="time">
#		<swe:Time definition="urn:ogc:def:phenomenon:time:iso8601"/>
#		
#		</swe:field>
#		<swe:field name="latitude">
#		<swe:Quantity definition="urn:ogc:def:phenomenon:latitude:wgs84">
#		<swe:uom code="deg"/>
#		</swe:Quantity>
#		</swe:field> 
#		<swe:field name="longitude"> 
#		<swe:Quantity definition="urn:ogc:def:phenomenon:longitude:wgs84">
#		<swe:uom code="deg"/>
#		</swe:Quantity>
#		
#		</swe:field>
#		<swe:field name="depth">
#		<swe:Quantity definition="cf:depth">
#		<swe:uom code="urn:ogc:unit:meter"/>
#		</swe:Quantity>
#		</swe:field>
#		<swe:field name="WaterTemperature">
#		<swe:Quantity definition="WaterTemperature">
#		<swe:uom xlink:href="urn:mm.def:units#degrees C"/>
#		</swe:Quantity>
#		
#		</swe:field>
#		</swe:DataRecord>
#		
#		</swe:components>
#		<swe:encoding>
#		<swe:AsciiBlock tokenSeparator="," blockSeparator="|" decimalSeparator="."/>
#		</swe:encoding>
#		</swe:DataBlockDefinition>
#		</om:resultDefinition>
#		
#		<om:result>2010-09-19T13:30:00Z,53.516566,8.188217,None,14.21|2010-09-19T13:40:00Z,53.516566,8.188217,None,14.21|2010-09-19T13:50:00Z,53.516566,8.188217,None,14.21|2010-09-19T14:00:00Z,53.516566,8.188217,None,14.21|2010-09-19T14:10:00Z,53.516566,8.188217,None,14.2|2010-09-19T14:20:00Z,53.516566,8.188217,None,14.2|2010-09-19T14:30:00Z,53.516566,8.188217,None,14.2|2010-09-19T14:40:00Z,53.516566,8.188217,None,14.19|2010-09-19T14:50:00Z,53.516566,8.188217,None,14.19|2010-09-19T15:00:00Z,53.516566,8.188217,None,14.18|2010-09-19T15:10:00Z,53.516566,8.188217,None,14.17|2010-09-19T15:20:00Z,53.516566,8.188217,None,14.16|2010-09-19T15:30:00Z,53.516566,8.188217,None,14.17|2010-09-19T15:40:00Z,53.516566,8.188217,None,14.13|2010-09-19T15:50:00Z,53.516566,8.188217,None,14.09|2010-09-19T16:00:00Z,53.516566,8.188217,None,14.05|2010-09-19T16:10:00Z,53.516566,8.188217,None,14.04|2010-09-19T16:20:00Z,53.516566,8.188217,None,14.05|2010-09-19T16:30:00Z,53.516566,8.188217,None,14.0|2010-09-19T16:40:00Z,53.516566,8.188217,None,13.74|2010-09-19T16:50:00Z,53.516566,8.188217,None,13.79|2010-09-19T17:00:00Z,53.516566,8.188217,None,13.94|2010-09-19T17:10:00Z,53.516566,8.188217,None,14.05|2010-09-19T17:20:00Z,53.516566,8.188217,None,14.04|2010-09-19T17:30:00Z,53.516566,8.188217,None,13.73|2010-09-19T17:40:00Z,53.516566,8.188217,None,14.02|2010-09-19T17:50:00Z,53.516566,8.188217,None,13.86|2010-09-19T18:00:00Z,53.516566,8.188217,None,13.98|2010-09-19T18:10:00Z,53.516566,8.188217,None,14.15|2010-09-19T18:20:00Z,53.516566,8.188217,None,14.03|2010-09-19T18:30:00Z,53.516566,8.188217,None,13.77|2010-09-19T18:40:00Z,53.516566,8.188217,None,13.76|2010-09-19T18:50:00Z,53.516566,8.188217,None,13.9|2010-09-19T19:00:00Z,53.516566,8.188217,None,13.97|2010-09-19T19:10:00Z,53.516566,8.188217,None,14.04|2010-09-19T19:20:00Z,53.516566,8.188217,None,14.14|2010-09-19T19:30:00Z,53.516566,8.188217,None,14.2|2010-09-19T19:40:00Z,53.516566,8.188217,None,14.24|2010-09-19T19:50:00Z,53.516566,8.188217,None,14.25|2010-09-19T20:00:00Z,53.516566,8.188217,None,14.26|2010-09-19T20:10:00Z,53.516566,8.188217,None,14.27|2010-09-19T20:20:00Z,53.516566,8.188217,None,14.3|2010-09-19T20:30:00Z,53.516566,8.188217,None,14.34|2010-09-19T20:40:00Z,53.516566,8.188217,None,14.37|2010-09-19T20:50:00Z,53.516566,8.188217,None,14.39|2010-09-19T21:00:00Z,53.516566,8.188217,None,14.4|2010-09-19T21:10:00Z,53.516566,8.188217,None,14.41|2010-09-19T21:20:00Z,53.516566,8.188217,None,14.42|2010-09-19T21:30:00Z,53.516566,8.188217,None,14.42|2010-09-19T21:40:00Z,53.516566,8.188217,None,14.42|2010-09-19T21:50:00Z,53.516566,8.188217,None,14.4|2010-09-19T22:00:00Z,53.516566,8.188217,None,14.37|2010-09-19T22:10:00Z,53.516566,8.188217,None,14.31|2010-09-19T22:20:00Z,53.516566,8.188217,None,14.22|2010-09-19T22:30:00Z,53.516566,8.188217,None,14.17|2010-09-19T22:40:00Z,53.516566,8.188217,None,14.09|2010-09-19T22:50:00Z,53.516566,8.188217,None,13.93|2010-09-19T23:00:00Z,53.516566,8.188217,None,13.79|2010-09-19T23:10:00Z,53.516566,8.188217,None,13.91|2010-09-19T23:20:00Z,53.516566,8.188217,None,13.91|2010-09-19T23:30:00Z,53.516566,8.188217,None,13.88|2010-09-19T23:40:00Z,53.516566,8.188217,None,13.9|2010-09-19T23:50:00Z,53.516566,8.188217,None,13.92|2010-09-20T00:00:00Z,53.516566,8.188217,None,13.93|2010-09-20T00:10:00Z,53.516566,8.188217,None,13.93|2010-09-20T00:20:00Z,53.516566,8.188217,None,13.96|2010-09-20T00:30:00Z,53.516566,8.188217,None,13.99|2010-09-20T00:40:00Z,53.516566,8.188217,None,14.02|2010-09-20T00:50:00Z,53.516566,8.188217,None,14.02|2010-09-20T01:00:00Z,53.516566,8.188217,None,14.01|2010-09-20T01:10:00Z,53.516566,8.188217,None,13.99|2010-09-20T01:20:00Z,53.516566,8.188217,None,13.94|2010-09-20T01:30:00Z,53.516566,8.188217,None,13.9|2010-09-20T01:40:00Z,53.516566,8.188217,None,13.84|2010-09-20T01:50:00Z,53.516566,8.188217,None,13.82|2010-09-20T02:00:00Z,53.516566,8.188217,None,13.8|2010-09-20T02:10:00Z,53.516566,8.188217,None,13.79|2010-09-20T02:20:00Z,53.516566,8.188217,None,13.78|2010-09-20T02:30:00Z,53.516566,8.188217,None,13.78|2010-09-20T02:40:00Z,53.516566,8.188217,None,13.77|2010-09-20T02:50:00Z,53.516566,8.188217,None,13.78|2010-09-20T03:00:00Z,53.516566,8.188217,None,13.8|2010-09-20T03:10:00Z,53.516566,8.188217,None,13.83|2010-09-20T03:20:00Z,53.516566,8.188217,None,13.84|2010-09-20T03:30:00Z,53.516566,8.188217,None,13.79|2010-09-20T03:40:00Z,53.516566,8.188217,None,13.75|2010-09-20T03:50:00Z,53.516566,8.188217,None,13.76|2010-09-20T04:00:00Z,53.516566,8.188217,None,13.83|2010-09-20T04:10:00Z,53.516566,8.188217,None,13.89|2010-09-20T04:20:00Z,53.516566,8.188217,None,13.87|2010-09-20T04:30:00Z,53.516566,8.188217,None,13.86|2010-09-20T04:40:00Z,53.516566,8.188217,None,13.84|2010-09-20T04:50:00Z,53.516566,8.188217,None,13.78|2010-09-20T05:00:00Z,53.516566,8.188217,None,13.76|2010-09-20T05:10:00Z,53.516566,8.188217,None,13.71|2010-09-20T05:20:00Z,53.516566,8.188217,None,13.94|2010-09-20T05:30:00Z,53.516566,8.188217,None,13.97|2010-09-20T05:40:00Z,53.516566,8.188217,None,13.95|2010-09-20T05:50:00Z,53.516566,8.188217,None,13.88|2010-09-20T06:00:00Z,53.516566,8.188217,None,13.72|2010-09-20T06:10:00Z,53.516566,8.188217,None,13.85|2010-09-20T06:20:00Z,53.516566,8.188217,None,13.71|2010-09-20T06:30:00Z,53.516566,8.188217,None,13.72|2010-09-20T06:40:00Z,53.516566,8.188217,None,13.75|2010-09-20T06:50:00Z,53.516566,8.188217,None,13.8|2010-09-20T07:00:00Z,53.516566,8.188217,None,13.86|2010-09-20T07:10:00Z,53.516566,8.188217,None,13.95|2010-09-20T07:20:00Z,53.516566,8.188217,None,14.03|2010-09-20T07:30:00Z,53.516566,8.188217,None,14.05|2010-09-20T07:40:00Z,53.516566,8.188217,None,14.06|2010-09-20T07:50:00Z,53.516566,8.188217,None,14.13|2010-09-20T08:00:00Z,53.516566,8.188217,None,14.19|2010-09-20T08:10:00Z,53.516566,8.188217,None,14.26|2010-09-20T08:20:00Z,53.516566,8.188217,None,14.32|2010-09-20T08:30:00Z,53.516566,8.188217,None,14.36|2010-09-20T08:40:00Z,53.516566,8.188217,None,14.38|2010-09-20T08:50:00Z,53.516566,8.188217,None,14.4|2010-09-20T09:00:00Z,53.516566,8.188217,None,14.41|2010-09-20T09:10:00Z,53.516566,8.188217,None,14.42|2010-09-20T09:20:00Z,53.516566,8.188217,None,14.43|2010-09-20T09:30:00Z,53.516566,8.188217,None,14.45|2010-09-20T09:40:00Z,53.516566,8.188217,None,14.47|2010-09-20T09:50:00Z,53.516566,8.188217,None,14.48|2010-09-20T10:00:00Z,53.516566,8.188217,None,14.48|2010-09-20T10:10:00Z,53.516566,8.188217,None,14.46|2010-09-20T10:20:00Z,53.516566,8.188217,None,14.42|2010-09-20T10:30:00Z,53.516566,8.188217,None,14.39|2010-09-20T10:40:00Z,53.516566,8.188217,None,14.32|2010-09-20T10:50:00Z,53.516566,8.188217,None,14.15|2010-09-20T11:00:00Z,53.516566,8.188217,None,13.99|2010-09-20T11:10:00Z,53.516566,8.188217,None,13.79|2010-09-20T11:20:00Z,53.516566,8.188217,None,13.82|2010-09-20T11:30:00Z,53.516566,8.188217,None,13.9|2010-09-20T11:40:00Z,53.516566,8.188217,None,13.86|2010-09-20T11:50:00Z,53.516566,8.188217,None,13.79|2010-09-20T12:00:00Z,53.516566,8.188217,None,13.52|2010-09-20T12:10:00Z,53.516566,8.188217,None,13.87|2010-09-20T12:20:00Z,53.516566,8.188217,None,13.92|2010-09-20T12:30:00Z,53.516566,8.188217,None,13.97|2010-09-20T12:40:00Z,53.516566,8.188217,None,14.01|2010-09-20T12:50:00Z,53.516566,8.188217,None,14.05|2010-09-20T13:00:00Z,53.516566,8.188217,None,14.08|2010-09-20T13:10:00Z,53.516566,8.188217,None,14.11|2010-09-20T13:20:00Z,53.516566,8.188217,None,14.11|2010-09-20T13:30:00Z,53.516566,8.188217,None,14.1|2010-09-20T13:40:00Z,53.516566,8.188217,None,14.09|2010-09-20T13:50:00Z,53.516566,8.188217,None,14.11|2010-09-20T14:00:00Z,53.516566,8.188217,None,14.12|2010-09-20T14:10:00Z,53.516566,8.188217,None,14.12|2010-09-20T14:20:00Z,53.516566,8.188217,None,14.12|2010-09-20T14:30:00Z,53.516566,8.188217,None,14.12|2010-09-20T14:40:00Z,53.516566,8.188217,None,14.11|2010-09-20T14:50:00Z,53.516566,8.188217,None,14.11|2010-09-20T15:00:00Z,53.516566,8.188217,None,14.11|2010-09-20T15:10:00Z,53.516566,8.188217,None,14.12|2010-09-20T15:20:00Z,53.516566,8.188217,None,14.11|2010-09-20T15:30:00Z,53.516566,8.188217,None,14.11|2010-09-20T15:40:00Z,53.516566,8.188217,None,14.11|2010-09-20T15:50:00Z,53.516566,8.188217,None,14.11|2010-09-20T16:00:00Z,53.516566,8.188217,None,14.12|2010-09-20T16:10:00Z,53.516566,8.188217,None,14.12|2010-09-20T16:20:00Z,53.516566,8.188217,None,14.11|2010-09-20T16:30:00Z,53.516566,8.188217,None,14.09|2010-09-20T16:40:00Z,53.516566,8.188217,None,14.11|2010-09-20T16:50:00Z,53.516566,8.188217,None,14.1|2010-09-20T17:00:00Z,53.516566,8.188217,None,14.05|2010-09-20T17:10:00Z,53.516566,8.188217,None,14.05|2010-09-20T17:20:00Z,53.516566,8.188217,None,14.08|2010-09-20T17:30:00Z,53.516566,8.188217,None,14.08|2010-09-20T17:40:00Z,53.516566,8.188217,None,14.02|2010-09-20T17:50:00Z,53.516566,8.188217,None,14.0|2010-09-20T18:00:00Z,53.516566,8.188217,None,14.04|2010-09-20T18:10:00Z,53.516566,8.188217,None,14.11|2010-09-20T18:20:00Z,53.516566,8.188217,None,14.12|2010-09-20T18:30:00Z,53.516566,8.188217,None,14.02|2010-09-20T18:40:00Z,53.516566,8.188217,None,14.01|2010-09-20T18:50:00Z,53.516566,8.188217,None,13.88|2010-09-20T19:00:00Z,53.516566,8.188217,None,13.82|2010-09-20T19:10:00Z,53.516566,8.188217,None,13.84|2010-09-20T19:20:00Z,53.516566,8.188217,None,13.88|2010-09-20T19:30:00Z,53.516566,8.188217,None,13.87|2010-09-20T19:40:00Z,53.516566,8.188217,None,13.81|2010-09-20T19:50:00Z,53.516566,8.188217,None,13.85|2010-09-20T20:00:00Z,53.516566,8.188217,None,14.02|2010-09-20T20:10:00Z,53.516566,8.188217,None,14.05|2010-09-20T20:20:00Z,53.516566,8.188217,None,14.06|2010-09-20T20:30:00Z,53.516566,8.188217,None,14.17|2010-09-20T20:40:00Z,53.516566,8.188217,None,14.19|2010-09-20T20:50:00Z,53.516566,8.188217,None,14.21|2010-09-20T21:00:00Z,53.516566,8.188217,None,14.24|2010-09-20T21:10:00Z,53.516566,8.188217,None,14.26|2010-09-20T21:20:00Z,53.516566,8.188217,None,14.29|2010-09-20T21:30:00Z,53.516566,8.188217,None,14.32|2010-09-20T21:40:00Z,53.516566,8.188217,None,14.35|2010-09-20T21:50:00Z,53.516566,8.188217,None,14.36|2010-09-20T22:00:00Z,53.516566,8.188217,None,14.37|2010-09-20T22:10:00Z,53.516566,8.188217,None,14.37|2010-09-20T22:20:00Z,53.516566,8.188217,None,14.37|2010-09-20T22:30:00Z,53.516566,8.188217,None,14.37|2010-09-20T22:40:00Z,53.516566,8.188217,None,14.34|2010-09-20T22:50:00Z,53.516566,8.188217,None,14.29|2010-09-20T23:00:00Z,53.516566,8.188217,None,14.23|2010-09-20T23:10:00Z,53.516566,8.188217,None,14.16|2010-09-20T23:20:00Z,53.516566,8.188217,None,14.04|2010-09-20T23:30:00Z,53.516566,8.188217,None,13.83|2010-09-20T23:40:00Z,53.516566,8.188217,None,13.78|2010-09-20T23:50:00Z,53.516566,8.188217,None,13.77|2010-09-21T00:00:00Z,53.516566,8.188217,None,13.88|2010-09-21T00:10:00Z,53.516566,8.188217,None,13.97|2010-09-21T00:20:00Z,53.516566,8.188217,None,13.95|2010-09-21T00:30:00Z,53.516566,8.188217,None,13.93|2010-09-21T00:40:00Z,53.516566,8.188217,None,13.9|2010-09-21T00:50:00Z,53.516566,8.188217,None,13.92|2010-09-21T01:00:00Z,53.516566,8.188217,None,13.96|2010-09-21T01:10:00Z,53.516566,8.188217,None,13.68|2010-09-21T01:20:00Z,53.516566,8.188217,None,13.92|2010-09-21T01:30:00Z,53.516566,8.188217,None,13.95|2010-09-21T01:40:00Z,53.516566,8.188217,None,13.95|2010-09-21T01:50:00Z,53.516566,8.188217,None,13.94|2010-09-21T02:00:00Z,53.516566,8.188217,None,13.92|2010-09-21T02:10:00Z,53.516566,8.188217,None,13.89|2010-09-21T02:20:00Z,53.516566,8.188217,None,13.85|2010-09-21T02:30:00Z,53.516566,8.188217,None,13.84|2010-09-21T02:40:00Z,53.516566,8.188217,None,13.83|2010-09-21T02:50:00Z,53.516566,8.188217,None,13.82|2010-09-21T03:00:00Z,53.516566,8.188217,None,13.81|2010-09-21T03:10:00Z,53.516566,8.188217,None,13.81|2010-09-21T03:20:00Z,53.516566,8.188217,None,13.8|2010-09-21T03:30:00Z,53.516566,8.188217,None,13.8|2010-09-21T03:40:00Z,53.516566,8.188217,None,13.81|2010-09-21T03:50:00Z,53.516566,8.188217,None,13.82|2010-09-21T04:00:00Z,53.516566,8.188217,None,13.84|2010-09-21T04:10:00Z,53.516566,8.188217,None,13.85|2010-09-21T04:20:00Z,53.516566,8.188217,None,13.79|2010-09-21T04:30:00Z,53.516566,8.188217,None,13.7|2010-09-21T04:40:00Z,53.516566,8.188217,None,13.71|2010-09-21T04:50:00Z,53.516566,8.188217,None,13.78|2010-09-21T05:00:00Z,53.516566,8.188217,None,13.82|2010-09-21T05:10:00Z,53.516566,8.188217,None,13.83|2010-09-21T05:20:00Z,53.516566,8.188217,None,13.83|2010-09-21T05:30:00Z,53.516566,8.188217,None,13.8|2010-09-21T05:40:00Z,53.516566,8.188217,None,13.85|2010-09-21T05:50:00Z,53.516566,8.188217,None,13.89|2010-09-21T06:00:00Z,53.516566,8.188217,None,13.91|2010-09-21T06:10:00Z,53.516566,8.188217,None,13.92|2010-09-21T06:20:00Z,53.516566,8.188217,None,13.95|2010-09-21T06:30:00Z,53.516566,8.188217,None,13.92|2010-09-21T06:40:00Z,53.516566,8.188217,None,13.93|2010-09-21T06:50:00Z,53.516566,8.188217,None,13.92|2010-09-21T07:00:00Z,53.516566,8.188217,None,13.89|2010-09-21T07:10:00Z,53.516566,8.188217,None,13.91|2010-09-21T07:20:00Z,53.516566,8.188217,None,13.84|2010-09-21T07:30:00Z,53.516566,8.188217,None,13.89|2010-09-21T07:40:00Z,53.516566,8.188217,None,13.96|2010-09-21T07:50:00Z,53.516566,8.188217,None,13.96|2010-09-21T08:00:00Z,53.516566,8.188217,None,14.02|2010-09-21T08:10:00Z,53.516566,8.188217,None,14.07|2010-09-21T08:20:00Z,53.516566,8.188217,None,14.12|2010-09-21T08:30:00Z,53.516566,8.188217,None,14.13|2010-09-21T08:40:00Z,53.516566,8.188217,None,14.17|2010-09-21T08:50:00Z,53.516566,8.188217,None,14.17|2010-09-21T09:00:00Z,53.516566,8.188217,None,14.19|2010-09-21T09:10:00Z,53.516566,8.188217,None,14.24|2010-09-21T09:20:00Z,53.516566,8.188217,None,14.26|2010-09-21T09:30:00Z,53.516566,8.188217,None,14.28|2010-09-21T09:40:00Z,53.516566,8.188217,None,14.33|2010-09-21T09:50:00Z,53.516566,8.188217,None,14.38|2010-09-21T10:00:00Z,53.516566,8.188217,None,14.42|2010-09-21T10:10:00Z,53.516566,8.188217,None,14.44|2010-09-21T10:20:00Z,53.516566,8.188217,None,14.47|2010-09-21T10:30:00Z,53.516566,8.188217,None,14.49|2010-09-21T10:40:00Z,53.516566,8.188217,None,14.49|2010-09-21T10:50:00Z,53.516566,8.188217,None,14.49|2010-09-21T11:00:00Z,53.516566,8.188217,None,14.48|2010-09-21T11:10:00Z,53.516566,8.188217,None,14.46|2010-09-21T11:20:00Z,53.516566,8.188217,None,14.41|2010-09-21T11:30:00Z,53.516566,8.188217,None,14.39|2010-09-21T11:40:00Z,53.516566,8.188217,None,14.35|2010-09-21T11:50:00Z,53.516566,8.188217,None,14.33|2010-09-21T12:00:00Z,53.516566,8.188217,None,14.23|2010-09-21T12:10:00Z,53.516566,8.188217,None,14.17|2010-09-21T12:20:00Z,53.516566,8.188217,None,14.22|2010-09-21T12:30:00Z,53.516566,8.188217,None,14.17|2010-09-21T12:40:00Z,53.516566,8.188217,None,14.17|2010-09-21T12:50:00Z,53.516566,8.188217,None,14.23|2010-09-21T13:00:00Z,53.516566,8.188217,None,14.25|2010-09-21T13:10:00Z,53.516566,8.188217,None,14.25|2010-09-21T13:20:00Z,53.516566,8.188217,None,14.29|2010-09-21T13:30:00Z,53.516566,8.188217,None,14.31|2010-09-21T13:40:00Z,53.516566,8.188217,None,14.34|2010-09-21T13:50:00Z,53.516566,8.188217,None,14.32|2010-09-21T14:00:00Z,53.516566,8.188217,None,14.34|2010-09-21T14:10:00Z,53.516566,8.188217,None,14.36|2010-09-21T14:20:00Z,53.516566,8.188217,None,14.4|2010-09-21T14:30:00Z,53.516566,8.188217,None,14.42|2010-09-21T14:40:00Z,53.516566,8.188217,None,14.43|2010-09-21T14:50:00Z,53.516566,8.188217,None,14.44|2010-09-21T15:00:00Z,53.516566,8.188217,None,14.46|2010-09-21T15:10:00Z,53.516566,8.188217,None,14.45|2010-09-21T15:20:00Z,53.516566,8.188217,None,14.45|2010-09-21T15:30:00Z,53.516566,8.188217,None,14.44|2010-09-21T15:40:00Z,53.516566,8.188217,None,14.45|2010-09-21T15:50:00Z,53.516566,8.188217,None,14.49|2010-09-21T16:00:00Z,53.516566,8.188217,None,14.56|2010-09-21T16:10:00Z,53.516566,8.188217,None,14.47|2010-09-21T16:20:00Z,53.516566,8.188217,None,14.42|2010-09-21T16:30:00Z,53.516566,8.188217,None,14.36|2010-09-21T16:40:00Z,53.516566,8.188217,None,14.38|2010-09-21T16:50:00Z,53.516566,8.188217,None,14.42|2010-09-21T17:00:00Z,53.516566,8.188217,None,14.43|2010-09-21T17:10:00Z,53.516566,8.188217,None,14.44|2010-09-21T17:20:00Z,53.516566,8.188217,None,14.45|2010-09-21T17:30:00Z,53.516566,8.188217,None,14.45|2010-09-21T17:40:00Z,53.516566,8.188217,None,14.47|2010-09-21T17:50:00Z,53.516566,8.188217,None,14.48|2010-09-21T18:00:00Z,53.516566,8.188217,None,14.45|2010-09-21T18:10:00Z,53.516566,8.188217,None,14.29|2010-09-21T18:20:00Z,53.516566,8.188217,None,14.31|2010-09-21T18:30:00Z,53.516566,8.188217,None,14.35|2010-09-21T18:40:00Z,53.516566,8.188217,None,14.42|2010-09-21T18:50:00Z,53.516566,8.188217,None,14.51|2010-09-21T19:00:00Z,53.516566,8.188217,None,14.55|2010-09-21T19:10:00Z,53.516566,8.188217,None,14.52|2010-09-21T19:20:00Z,53.516566,8.188217,None,14.48|2010-09-21T19:30:00Z,53.516566,8.188217,None,14.36|2010-09-21T19:40:00Z,53.516566,8.188217,None,14.41|2010-09-21T19:50:00Z,53.516566,8.188217,None,14.49|2010-09-21T20:00:00Z,53.516566,8.188217,None,14.45|2010-09-21T20:10:00Z,53.516566,8.188217,None,14.32|2010-09-21T20:20:00Z,53.516566,8.188217,None,14.24|2010-09-21T20:30:00Z,53.516566,8.188217,None,14.27|2010-09-21T20:40:00Z,53.516566,8.188217,None,14.33|2010-09-21T20:50:00Z,53.516566,8.188217,None,14.39|2010-09-21T21:00:00Z,53.516566,8.188217,None,14.43|2010-09-21T21:10:00Z,53.516566,8.188217,None,14.46|2010-09-21T21:20:00Z,53.516566,8.188217,None,14.47|2010-09-21T21:30:00Z,53.516566,8.188217,None,14.46|2010-09-21T21:40:00Z,53.516566,8.188217,None,14.44|2010-09-21T21:50:00Z,53.516566,8.188217,None,14.41|2010-09-21T22:00:00Z,53.516566,8.188217,None,14.39|2010-09-21T22:10:00Z,53.516566,8.188217,None,14.39|2010-09-21T22:20:00Z,53.516566,8.188217,None,14.38|2010-09-21T22:30:00Z,53.516566,8.188217,None,14.35|2010-09-21T22:40:00Z,53.516566,8.188217,None,14.33|2010-09-21T22:50:00Z,53.516566,8.188217,None,14.32|2010-09-21T23:00:00Z,53.516566,8.188217,None,14.32|2010-09-21T23:10:00Z,53.516566,8.188217,None,14.31|2010-09-21T23:20:00Z,53.516566,8.188217,None,14.3|2010-09-21T23:30:00Z,53.516566,8.188217,None,14.3|2010-09-21T23:40:00Z,53.516566,8.188217,None,14.3|2010-09-21T23:50:00Z,53.516566,8.188217,None,14.32|2010-09-22T00:00:00Z,53.516566,8.188217,None,14.33|2010-09-22T00:10:00Z,53.516566,8.188217,None,14.32|2010-09-22T00:20:00Z,53.516566,8.188217,None,14.18|2010-09-22T00:30:00Z,53.516566,8.188217,None,14.22|2010-09-22T00:40:00Z,53.516566,8.188217,None,14.18|2010-09-22T00:50:00Z,53.516566,8.188217,None,14.12|2010-09-22T01:00:00Z,53.516566,8.188217,None,14.1|2010-09-22T01:10:00Z,53.516566,8.188217,None,14.09|2010-09-22T01:20:00Z,53.516566,8.188217,None,14.09|2010-09-22T01:30:00Z,53.516566,8.188217,None,14.1|2010-09-22T01:40:00Z,53.516566,8.188217,None,14.13|2010-09-22T01:50:00Z,53.516566,8.188217,None,14.15|2010-09-22T02:00:00Z,53.516566,8.188217,None,14.23|2010-09-22T02:10:00Z,53.516566,8.188217,None,14.25|2010-09-22T02:20:00Z,53.516566,8.188217,None,14.24|2010-09-22T02:30:00Z,53.516566,8.188217,None,14.23|2010-09-22T02:40:00Z,53.516566,8.188217,None,14.23|2010-09-22T02:50:00Z,53.516566,8.188217,None,14.23|2010-09-22T03:00:00Z,53.516566,8.188217,None,14.22|2010-09-22T03:10:00Z,53.516566,8.188217,None,14.22|2010-09-22T03:20:00Z,53.516566,8.188217,None,14.22|2010-09-22T03:30:00Z,53.516566,8.188217,None,14.21|2010-09-22T03:40:00Z,53.516566,8.188217,None,14.21|2010-09-22T03:50:00Z,53.516566,8.188217,None,14.22|2010-09-22T04:00:00Z,53.516566,8.188217,None,14.14|2010-09-22T04:10:00Z,53.516566,8.188217,None,14.05|2010-09-22T04:20:00Z,53.516566,8.188217,None,14.06|2010-09-22T04:30:00Z,53.516566,8.188217,None,14.07|2010-09-22T04:40:00Z,53.516566,8.188217,None,14.05|2010-09-22T04:50:00Z,53.516566,8.188217,None,14.09|2010-09-22T05:00:00Z,53.516566,8.188217,None,14.19|2010-09-22T05:10:00Z,53.516566,8.188217,None,14.2|2010-09-22T05:20:00Z,53.516566,8.188217,None,14.21|2010-09-22T05:30:00Z,53.516566,8.188217,None,14.22|2010-09-22T05:40:00Z,53.516566,8.188217,None,14.19|2010-09-22T05:50:00Z,53.516566,8.188217,None,14.16|2010-09-22T06:00:00Z,53.516566,8.188217,None,14.14|2010-09-22T06:10:00Z,53.516566,8.188217,None,14.03|2010-09-22T06:20:00Z,53.516566,8.188217,None,13.89|2010-09-22T06:30:00Z,53.516566,8.188217,None,14.05|2010-09-22T06:40:00Z,53.516566,8.188217,None,14.21|2010-09-22T06:50:00Z,53.516566,8.188217,None,14.2|2010-09-22T07:00:00Z,53.516566,8.188217,None,14.21|2010-09-22T07:10:00Z,53.516566,8.188217,None,14.23|2010-09-22T07:20:00Z,53.516566,8.188217,None,14.24|2010-09-22T07:30:00Z,53.516566,8.188217,None,14.25|2010-09-22T07:40:00Z,53.516566,8.188217,None,14.26|2010-09-22T07:50:00Z,53.516566,8.188217,None,14.29|2010-09-22T08:00:00Z,53.516566,8.188217,None,14.3|2010-09-22T08:10:00Z,53.516566,8.188217,None,14.31|2010-09-22T08:20:00Z,53.516566,8.188217,None,14.31|2010-09-22T08:30:00Z,53.516566,8.188217,None,14.3|2010-09-22T08:40:00Z,53.516566,8.188217,None,14.33|2010-09-22T08:50:00Z,53.516566,8.188217,None,14.34|2010-09-22T09:00:00Z,53.516566,8.188217,None,14.34|2010-09-22T09:10:00Z,53.516566,8.188217,None,14.34|2010-09-22T09:20:00Z,53.516566,8.188217,None,14.34|2010-09-22T09:30:00Z,53.516566,8.188217,None,14.33|2010-09-22T09:40:00Z,53.516566,8.188217,None,14.34|2010-09-22T09:50:00Z,53.516566,8.188217,None,14.34|2010-09-22T10:00:00Z,53.516566,8.188217,None,14.34|2010-09-22T10:10:00Z,53.516566,8.188217,None,14.34|2010-09-22T10:20:00Z,53.516566,8.188217,None,14.35|2010-09-22T10:30:00Z,53.516566,8.188217,None,14.38|2010-09-22T10:40:00Z,53.516566,8.188217,None,14.41|2010-09-22T10:50:00Z,53.516566,8.188217,None,14.42|2010-09-22T11:00:00Z,53.516566,8.188217,None,14.44|2010-09-22T11:10:00Z,53.516566,8.188217,None,14.47|2010-09-22T11:20:00Z,53.516566,8.188217,None,14.48|2010-09-22T11:30:00Z,53.516566,8.188217,None,14.49|2010-09-22T11:40:00Z,53.516566,8.188217,None,14.51|2010-09-22T11:50:00Z,53.516566,8.188217,None,14.52|2010-09-22T12:00:00Z,53.516566,8.188217,None,14.49|2010-09-22T12:10:00Z,53.516566,8.188217,None,14.5|2010-09-22T12:20:00Z,53.516566,8.188217,None,14.56|2010-09-22T12:30:00Z,53.516566,8.188217,None,14.66|2010-09-22T12:40:00Z,53.516566,8.188217,None,14.72|2010-09-22T12:50:00Z,53.516566,8.188217,None,14.73|2010-09-22T13:00:00Z,53.516566,8.188217,None,14.72|2010-09-22T13:10:00Z,53.516566,8.188217,None,14.64|2010-09-22T13:20:00Z,53.516566,8.188217,None,14.56|2010-09-22T13:30:00Z,53.516566,8.188217,None,14.52|2010-09-22T13:40:00Z,53.516566,8.188217,None,14.48|2010-09-22T13:50:00Z,53.516566,8.188217,None,14.45|2010-09-22T14:00:00Z,53.516566,8.188217,None,14.47|2010-09-22T14:10:00Z,53.516566,8.188217,None,14.56|2010-09-22T14:20:00Z,53.516566,8.188217,None,14.65|2010-09-22T14:30:00Z,53.516566,8.188217,None,14.72|2010-09-22T14:40:00Z,53.516566,8.188217,None,14.77|2010-09-22T14:50:00Z,53.516566,8.188217,None,14.8|2010-09-22T15:00:00Z,53.516566,8.188217,None,14.81|2010-09-22T15:10:00Z,53.516566,8.188217,None,14.82|2010-09-22T15:20:00Z,53.516566,8.188217,None,14.82|2010-09-22T15:30:00Z,53.516566,8.188217,None,14.83|2010-09-22T15:40:00Z,53.516566,8.188217,None,14.82|2010-09-22T15:50:00Z,53.516566,8.188217,None,14.81|2010-09-22T16:00:00Z,53.516566,8.188217,None,14.79|2010-09-22T16:10:00Z,53.516566,8.188217,None,14.77|2010-09-22T16:20:00Z,53.516566,8.188217,None,14.73|2010-09-22T16:30:00Z,53.516566,8.188217,None,14.87|2010-09-22T16:40:00Z,53.516566,8.188217,None,14.88|2010-09-22T16:50:00Z,53.516566,8.188217,None,14.89|2010-09-22T17:00:00Z,53.516566,8.188217,None,14.9|2010-09-22T17:10:00Z,53.516566,8.188217,None,14.79|2010-09-22T17:20:00Z,53.516566,8.188217,None,14.71|2010-09-22T17:30:00Z,53.516566,8.188217,None,14.69|2010-09-22T17:40:00Z,53.516566,8.188217,None,14.68|2010-09-22T17:50:00Z,53.516566,8.188217,None,14.67|2010-09-22T18:00:00Z,53.516566,8.188217,None,14.7|2010-09-22T18:10:00Z,53.516566,8.188217,None,14.76|2010-09-22T18:20:00Z,53.516566,8.188217,None,14.78|2010-09-22T18:30:00Z,53.516566,8.188217,None,14.75|2010-09-22T18:40:00Z,53.516566,8.188217,None,14.77|2010-09-22T18:50:00Z,53.516566,8.188217,None,14.66|2010-09-22T19:00:00Z,53.516566,8.188217,None,14.88|2010-09-22T19:10:00Z,53.516566,8.188217,None,14.97|2010-09-22T19:20:00Z,53.516566,8.188217,None,14.94|2010-09-22T19:30:00Z,53.516566,8.188217,None,14.98|2010-09-22T19:40:00Z,53.516566,8.188217,None,15.01|2010-09-22T19:50:00Z,53.516566,8.188217,None,14.92|2010-09-22T20:00:00Z,53.516566,8.188217,None,14.77|2010-09-22T20:10:00Z,53.516566,8.188217,None,14.74|2010-09-22T20:20:00Z,53.516566,8.188217,None,14.75|2010-09-22T20:30:00Z,53.516566,8.188217,None,14.73|2010-09-22T20:40:00Z,53.516566,8.188217,None,14.72|2010-09-22T20:50:00Z,53.516566,8.188217,None,14.72|2010-09-22T21:00:00Z,53.516566,8.188217,None,14.71|2010-09-22T21:10:00Z,53.516566,8.188217,None,14.69|2010-09-22T21:20:00Z,53.516566,8.188217,None,14.69|2010-09-22T21:30:00Z,53.516566,8.188217,None,14.7|2010-09-22T21:40:00Z,53.516566,8.188217,None,14.72|2010-09-22T21:50:00Z,53.516566,8.188217,None,14.71|2010-09-22T22:00:00Z,53.516566,8.188217,None,14.66|2010-09-22T22:10:00Z,53.516566,8.188217,None,14.64|2010-09-22T22:20:00Z,53.516566,8.188217,None,14.63|2010-09-22T22:30:00Z,53.516566,8.188217,None,14.62|2010-09-22T22:40:00Z,53.516566,8.188217,None,14.62|2010-09-22T22:50:00Z,53.516566,8.188217,None,14.6|2010-09-22T23:00:00Z,53.516566,8.188217,None,14.6|2010-09-22T23:10:00Z,53.516566,8.188217,None,14.59|2010-09-22T23:20:00Z,53.516566,8.188217,None,14.59|2010-09-22T23:30:00Z,53.516566,8.188217,None,14.58|2010-09-22T23:40:00Z,53.516566,8.188217,None,14.58|2010-09-22T23:50:00Z,53.516566,8.188217,None,14.58|2010-09-23T00:00:00Z,53.516566,8.188217,None,14.57|2010-09-23T00:10:00Z,53.516566,8.188217,None,14.57|2010-09-23T00:20:00Z,53.516566,8.188217,None,14.57|2010-09-23T00:30:00Z,53.516566,8.188217,None,14.58|2010-09-23T00:40:00Z,53.516566,8.188217,None,14.6|2010-09-23T00:50:00Z,53.516566,8.188217,None,14.63|2010-09-23T01:00:00Z,53.516566,8.188217,None,14.63|2010-09-23T01:10:00Z,53.516566,8.188217,None,14.67|2010-09-23T01:20:00Z,53.516566,8.188217,None,14.7|2010-09-23T01:30:00Z,53.516566,8.188217,None,14.67|2010-09-23T01:40:00Z,53.516566,8.188217,None,14.63|2010-09-23T01:50:00Z,53.516566,8.188217,None,14.61|2010-09-23T02:00:00Z,53.516566,8.188217,None,14.6|2010-09-23T02:10:00Z,53.516566,8.188217,None,14.63|2010-09-23T02:20:00Z,53.516566,8.188217,None,14.68|2010-09-23T02:30:00Z,53.516566,8.188217,None,14.73|2010-09-23T02:40:00Z,53.516566,8.188217,None,14.79|2010-09-23T02:50:00Z,53.516566,8.188217,None,14.79|2010-09-23T03:00:00Z,53.516566,8.188217,None,14.78|2010-09-23T03:10:00Z,53.516566,8.188217,None,14.75|2010-09-23T03:20:00Z,53.516566,8.188217,None,14.73|2010-09-23T03:30:00Z,53.516566,8.188217,None,14.71|2010-09-23T03:40:00Z,53.516566,8.188217,None,14.7|2010-09-23T03:50:00Z,53.516566,8.188217,None,14.69|2010-09-23T04:00:00Z,53.516566,8.188217,None,14.71|2010-09-23T04:10:00Z,53.516566,8.188217,None,14.69|2010-09-23T04:20:00Z,53.516566,8.188217,None,14.69|2010-09-23T04:30:00Z,53.516566,8.188217,None,14.69|2010-09-23T04:40:00Z,53.516566,8.188217,None,14.66|2010-09-23T04:50:00Z,53.516566,8.188217,None,14.56|2010-09-23T05:00:00Z,53.516566,8.188217,None,14.62|2010-09-23T05:10:00Z,53.516566,8.188217,None,14.65|2010-09-23T05:20:00Z,53.516566,8.188217,None,14.69|2010-09-23T05:30:00Z,53.516566,8.188217,None,14.67|2010-09-23T05:40:00Z,53.516566,8.188217,None,14.66|2010-09-23T05:50:00Z,53.516566,8.188217,None,14.66|2010-09-23T06:00:00Z,53.516566,8.188217,None,14.67|2010-09-23T06:10:00Z,53.516566,8.188217,None,14.68|2010-09-23T06:20:00Z,53.516566,8.188217,None,14.68|2010-09-23T06:30:00Z,53.516566,8.188217,None,14.68|2010-09-23T06:40:00Z,53.516566,8.188217,None,14.67|2010-09-23T06:50:00Z,53.516566,8.188217,None,14.59|2010-09-23T07:00:00Z,53.516566,8.188217,None,14.66|2010-09-23T07:10:00Z,53.516566,8.188217,None,14.68|2010-09-23T07:20:00Z,53.516566,8.188217,None,14.7|2010-09-23T07:30:00Z,53.516566,8.188217,None,14.73|2010-09-23T07:40:00Z,53.516566,8.188217,None,14.75|2010-09-23T07:50:00Z,53.516566,8.188217,None,14.76|2010-09-23T08:00:00Z,53.516566,8.188217,None,14.76|2010-09-23T08:10:00Z,53.516566,8.188217,None,14.75|2010-09-23T08:20:00Z,53.516566,8.188217,None,14.76|2010-09-23T08:30:00Z,53.516566,8.188217,None,14.75|2010-09-23T08:40:00Z,53.516566,8.188217,None,14.74|2010-09-23T08:50:00Z,53.516566,8.188217,None,14.74|2010-09-23T09:00:00Z,53.516566,8.188217,None,14.72|2010-09-23T09:10:00Z,53.516566,8.188217,None,14.72|2010-09-23T09:20:00Z,53.516566,8.188217,None,14.71|2010-09-23T09:30:00Z,53.516566,8.188217,None,14.7|2010-09-23T09:40:00Z,53.516566,8.188217,None,14.7|2010-09-23T09:50:00Z,53.516566,8.188217,None,14.69|2010-09-23T10:00:00Z,53.516566,8.188217,None,14.69|2010-09-23T10:10:00Z,53.516566,8.188217,None,14.68|2010-09-23T10:20:00Z,53.516566,8.188217,None,14.68|2010-09-23T10:30:00Z,53.516566,8.188217,None,14.66|2010-09-23T10:40:00Z,53.516566,8.188217,None,14.64|2010-09-23T10:50:00Z,53.516566,8.188217,None,14.65|2010-09-23T11:00:00Z,53.516566,8.188217,None,14.64|2010-09-23T11:10:00Z,53.516566,8.188217,None,14.63|2010-09-23T11:20:00Z,53.516566,8.188217,None,14.61|2010-09-23T11:30:00Z,53.516566,8.188217,None,14.62|2010-09-23T11:40:00Z,53.516566,8.188217,None,14.62|2010-09-23T11:50:00Z,53.516566,8.188217,None,14.62|2010-09-23T12:00:00Z,53.516566,8.188217,None,14.63|2010-09-23T12:10:00Z,53.516566,8.188217,None,14.62|2010-09-23T12:20:00Z,53.516566,8.188217,None,14.65|2010-09-23T12:30:00Z,53.516566,8.188217,None,14.68|2010-09-23T12:40:00Z,53.516566,8.188217,None,14.7|2010-09-23T12:50:00Z,53.516566,8.188217,None,14.79|2010-09-23T13:00:00Z,53.516566,8.188217,None,14.91|2010-09-23T13:10:00Z,53.516566,8.188217,None,15.0|2010-09-23T13:20:00Z,53.516566,8.188217,None,15.09|2010-09-23T13:30:00Z,53.516566,8.188217,None,15.17|2010-09-23T13:40:00Z,53.516566,8.188217,None,15.2|2010-09-23T13:50:00Z,53.516566,8.188217,None,15.19|2010-09-23T14:00:00Z,53.516566,8.188217,None,15.21|2010-09-23T14:10:00Z,53.516566,8.188217,None,15.26|2010-09-23T14:20:00Z,53.516566,8.188217,None,15.24|2010-09-23T14:30:00Z,53.516566,8.188217,None,15.19|2010-09-23T14:40:00Z,53.516566,8.188217,None,15.17|2010-09-23T14:50:00Z,53.516566,8.188217,None,15.16|2010-09-23T15:00:00Z,53.516566,8.188217,None,15.18|2010-09-23T15:10:00Z,53.516566,8.188217,None,15.18|2010-09-23T15:20:00Z,53.516566,8.188217,None,15.15|2010-09-23T15:30:00Z,53.516566,8.188217,None,15.13|2010-09-23T15:40:00Z,53.516566,8.188217,None,15.18|2010-09-23T15:50:00Z,53.516566,8.188217,None,15.22|2010-09-23T16:00:00Z,53.516566,8.188217,None,15.23|2010-09-23T16:10:00Z,53.516566,8.188217,None,15.23|2010-09-23T16:20:00Z,53.516566,8.188217,None,15.19|2010-09-23T16:30:00Z,53.516566,8.188217,None,15.13|2010-09-23T16:40:00Z,53.516566,8.188217,None,15.1|2010-09-23T16:50:00Z,53.516566,8.188217,None,15.08|2010-09-23T17:00:00Z,53.516566,8.188217,None,15.07|2010-09-23T17:10:00Z,53.516566,8.188217,None,15.04|2010-09-23T17:20:00Z,53.516566,8.188217,None,15.19|2010-09-23T17:30:00Z,53.516566,8.188217,None,15.25|2010-09-23T17:40:00Z,53.516566,8.188217,None,15.2|2010-09-23T17:50:00Z,53.516566,8.188217,None,15.13|2010-09-23T18:00:00Z,53.516566,8.188217,None,15.06|2010-09-23T18:10:00Z,53.516566,8.188217,None,15.02|2010-09-23T18:20:00Z,53.516566,8.188217,None,15.01|2010-09-23T18:30:00Z,53.516566,8.188217,None,15.0|2010-09-23T18:40:00Z,53.516566,8.188217,None,15.06|2010-09-23T18:50:00Z,53.516566,8.188217,None,15.12|2010-09-23T19:00:00Z,53.516566,8.188217,None,15.16|2010-09-23T19:10:00Z,53.516566,8.188217,None,15.19|2010-09-23T19:20:00Z,53.516566,8.188217,None,15.18|2010-09-23T19:30:00Z,53.516566,8.188217,None,15.14|2010-09-23T19:40:00Z,53.516566,8.188217,None,15.37|2010-09-23T19:50:00Z,53.516566,8.188217,None,15.51|2010-09-23T20:00:00Z,53.516566,8.188217,None,15.43|2010-09-23T20:10:00Z,53.516566,8.188217,None,15.47|2010-09-23T20:20:00Z,53.516566,8.188217,None,15.41|2010-09-23T20:30:00Z,53.516566,8.188217,None,15.35|2010-09-23T20:40:00Z,53.516566,8.188217,None,15.31|2010-09-23T20:50:00Z,53.516566,8.188217,None,15.26|2010-09-23T21:00:00Z,53.516566,8.188217,None,15.22|2010-09-23T21:10:00Z,53.516566,8.188217,None,15.18|2010-09-23T21:20:00Z,53.516566,8.188217,None,15.14|2010-09-23T21:30:00Z,53.516566,8.188217,None,15.12|2010-09-23T21:40:00Z,53.516566,8.188217,None,15.12|2010-09-23T21:50:00Z,53.516566,8.188217,None,15.11|2010-09-23T22:00:00Z,53.516566,8.188217,None,15.09|2010-09-23T22:10:00Z,53.516566,8.188217,None,15.09|2010-09-23T22:20:00Z,53.516566,8.188217,None,15.07|2010-09-23T22:30:00Z,53.516566,8.188217,None,15.01|2010-09-23T22:40:00Z,53.516566,8.188217,None,14.98|2010-09-23T22:50:00Z,53.516566,8.188217,None,14.96|2010-09-23T23:00:00Z,53.516566,8.188217,None,14.96|2010-09-23T23:10:00Z,53.516566,8.188217,None,14.94|2010-09-23T23:20:00Z,53.516566,8.188217,None,14.92|2010-09-23T23:30:00Z,53.516566,8.188217,None,14.91|2010-09-23T23:40:00Z,53.516566,8.188217,None,14.88|2010-09-23T23:50:00Z,53.516566,8.188217,None,14.84|2010-09-24T00:00:00Z,53.516566,8.188217,None,14.8|2010-09-24T00:10:00Z,53.516566,8.188217,None,14.78|2010-09-24T00:20:00Z,53.516566,8.188217,None,14.76|2010-09-24T00:30:00Z,53.516566,8.188217,None,14.76|2010-09-24T00:40:00Z,53.516566,8.188217,None,14.76|2010-09-24T00:50:00Z,53.516566,8.188217,None,14.79|2010-09-24T01:00:00Z,53.516566,8.188217,None,14.86|2010-09-24T01:10:00Z,53.516566,8.188217,None,14.92|2010-09-24T01:20:00Z,53.516566,8.188217,None,14.96|2010-09-24T01:30:00Z,53.516566,8.188217,None,15.05|2010-09-24T01:40:00Z,53.516566,8.188217,None,15.1|2010-09-24T01:50:00Z,53.516566,8.188217,None,15.17|2010-09-24T02:00:00Z,53.516566,8.188217,None,15.17|2010-09-24T02:10:00Z,53.516566,8.188217,None,15.15|2010-09-24T02:20:00Z,53.516566,8.188217,None,15.13|2010-09-24T02:30:00Z,53.516566,8.188217,None,15.09|2010-09-24T02:40:00Z,53.516566,8.188217,None,15.06|2010-09-24T02:50:00Z,53.516566,8.188217,None,15.11|2010-09-24T03:00:00Z,53.516566,8.188217,None,15.2|2010-09-24T03:10:00Z,53.516566,8.188217,None,15.26|2010-09-24T03:20:00Z,53.516566,8.188217,None,15.3|2010-09-24T03:30:00Z,53.516566,8.188217,None,15.29|2010-09-24T03:40:00Z,53.516566,8.188217,None,15.27|2010-09-24T03:50:00Z,53.516566,8.188217,None,15.26|2010-09-24T04:00:00Z,53.516566,8.188217,None,15.25|2010-09-24T04:10:00Z,53.516566,8.188217,None,15.24|2010-09-24T04:20:00Z,53.516566,8.188217,None,15.24|2010-09-24T04:30:00Z,53.516566,8.188217,None,15.25|2010-09-24T04:40:00Z,53.516566,8.188217,None,15.23|2010-09-24T04:50:00Z,53.516566,8.188217,None,15.24|2010-09-24T05:00:00Z,53.516566,8.188217,None,15.18|2010-09-24T05:10:00Z,53.516566,8.188217,None,15.14|2010-09-24T05:20:00Z,53.516566,8.188217,None,15.11|2010-09-24T05:30:00Z,53.516566,8.188217,None,15.13|2010-09-24T05:40:00Z,53.516566,8.188217,None,15.2|2010-09-24T05:50:00Z,53.516566,8.188217,None,15.18|2010-09-24T06:00:00Z,53.516566,8.188217,None,15.14|2010-09-24T06:10:00Z,53.516566,8.188217,None,15.13|2010-09-24T06:20:00Z,53.516566,8.188217,None,15.12|2010-09-24T06:30:00Z,53.516566,8.188217,None,15.11|2010-09-24T06:40:00Z,53.516566,8.188217,None,15.14|2010-09-24T06:50:00Z,53.516566,8.188217,None,15.16|2010-09-24T07:00:00Z,53.516566,8.188217,None,15.14|2010-09-24T07:10:00Z,53.516566,8.188217,None,15.13|2010-09-24T07:20:00Z,53.516566,8.188217,None,15.13|2010-09-24T07:30:00Z,53.516566,8.188217,None,15.15|2010-09-24T07:40:00Z,53.516566,8.188217,None,15.18|2010-09-24T07:50:00Z,53.516566,8.188217,None,15.23|2010-09-24T08:00:00Z,53.516566,8.188217,None,15.25|2010-09-24T08:10:00Z,53.516566,8.188217,None,15.27|2010-09-24T08:20:00Z,53.516566,8.188217,None,15.28|2010-09-24T08:30:00Z,53.516566,8.188217,None,15.26|2010-09-24T08:40:00Z,53.516566,8.188217,None,15.22|2010-09-24T08:50:00Z,53.516566,8.188217,None,15.2|2010-09-24T09:00:00Z,53.516566,8.188217,None,15.15|2010-09-24T09:10:00Z,53.516566,8.188217,None,15.13|2010-09-24T09:20:00Z,53.516566,8.188217,None,15.13|2010-09-24T09:30:00Z,53.516566,8.188217,None,15.12|2010-09-24T09:40:00Z,53.516566,8.188217,None,15.12|2010-09-24T09:50:00Z,53.516566,8.188217,None,15.08|2010-09-24T10:00:00Z,53.516566,8.188217,None,15.06|2010-09-24T10:10:00Z,53.516566,8.188217,None,15.06|2010-09-24T10:20:00Z,53.516566,8.188217,None,15.05|2010-09-24T10:30:00Z,53.516566,8.188217,None,15.03|2010-09-24T10:40:00Z,53.516566,8.188217,None,15.0|2010-09-24T10:50:00Z,53.516566,8.188217,None,14.94|2010-09-24T11:00:00Z,53.516566,8.188217,None,14.92|2010-09-24T11:10:00Z,53.516566,8.188217,None,14.85|2010-09-24T11:20:00Z,53.516566,8.188217,None,14.79|2010-09-24T11:30:00Z,53.516566,8.188217,None,14.77|2010-09-24T11:40:00Z,53.516566,8.188217,None,14.76|2010-09-24T11:50:00Z,53.516566,8.188217,None,14.76|2010-09-24T12:00:00Z,53.516566,8.188217,None,14.76|2010-09-24T12:10:00Z,53.516566,8.188217,None,14.76|2010-09-24T12:20:00Z,53.516566,8.188217,None,14.76|2010-09-24T12:30:00Z,53.516566,8.188217,None,14.76|2010-09-24T12:40:00Z,53.516566,8.188217,None,14.75|2010-09-24T12:50:00Z,53.516566,8.188217,None,14.76|2010-09-24T13:00:00Z,53.516566,8.188217,None,14.79|2010-09-24T13:10:00Z,53.516566,8.188217,None,14.8|2010-09-24T13:20:00Z,53.516566,8.188217,None,14.82|2010-09-24T13:30:00Z,53.516566,8.188217,None,14.92|2010-09-24T13:40:00Z,53.516566,8.188217,None,15.06|2010-09-24T13:50:00Z,53.516566,8.188217,None,15.15|2010-09-24T14:00:00Z,53.516566,8.188217,None,15.21|2010-09-24T14:10:00Z,53.516566,8.188217,None,15.27|2010-09-24T14:20:00Z,53.516566,8.188217,None,15.32|2010-09-24T14:30:00Z,53.516566,8.188217,None,15.36|2010-09-24T14:40:00Z,53.516566,8.188217,None,15.37|2010-09-24T14:50:00Z,53.516566,8.188217,None,15.33|2010-09-24T15:00:00Z,53.516566,8.188217,None,15.3|2010-09-24T15:10:00Z,53.516566,8.188217,None,15.24|2010-09-24T15:20:00Z,53.516566,8.188217,None,15.18|2010-09-24T15:30:00Z,53.516566,8.188217,None,15.14|2010-09-24T15:40:00Z,53.516566,8.188217,None,15.13|2010-09-24T15:50:00Z,53.516566,8.188217,None,15.18|2010-09-24T16:00:00Z,53.516566,8.188217,None,15.21|2010-09-24T16:10:00Z,53.516566,8.188217,None,15.25|2010-09-24T16:20:00Z,53.516566,8.188217,None,15.27|2010-09-24T16:30:00Z,53.516566,8.188217,None,15.3|2010-09-24T16:40:00Z,53.516566,8.188217,None,15.31|2010-09-24T16:50:00Z,53.516566,8.188217,None,15.27|2010-09-24T17:00:00Z,53.516566,8.188217,None,15.27|2010-09-24T17:10:00Z,53.516566,8.188217,None,15.25|2010-09-24T17:20:00Z,53.516566,8.188217,None,15.21|2010-09-24T17:30:00Z,53.516566,8.188217,None,15.2|2010-09-24T17:40:00Z,53.516566,8.188217,None,15.19|2010-09-24T17:50:00Z,53.516566,8.188217,None,15.17|2010-09-24T18:00:00Z,53.516566,8.188217,None,15.15|2010-09-24T18:10:00Z,53.516566,8.188217,None,15.15|2010-09-24T18:20:00Z,53.516566,8.188217,None,15.24|2010-09-24T18:30:00Z,53.516566,8.188217,None,15.31|2010-09-24T18:40:00Z,53.516566,8.188217,None,15.26|2010-09-24T18:50:00Z,53.516566,8.188217,None,15.18|2010-09-24T19:00:00Z,53.516566,8.188217,None,15.16|2010-09-24T19:10:00Z,53.516566,8.188217,None,15.14|2010-09-24T19:20:00Z,53.516566,8.188217,None,15.14|2010-09-24T19:30:00Z,53.516566,8.188217,None,15.15|2010-09-24T19:40:00Z,53.516566,8.188217,None,15.2|2010-09-24T19:50:00Z,53.516566,8.188217,None,15.28|2010-09-24T20:00:00Z,53.516566,8.188217,None,15.28|2010-09-24T20:10:00Z,53.516566,8.188217,None,15.32|2010-09-24T20:20:00Z,53.516566,8.188217,None,15.39|2010-09-24T20:30:00Z,53.516566,8.188217,None,15.42|2010-09-24T20:40:00Z,53.516566,8.188217,None,15.4|2010-09-24T20:50:00Z,53.516566,8.188217,None,15.37|2010-09-24T21:00:00Z,53.516566,8.188217,None,15.31|2010-09-24T21:10:00Z,53.516566,8.188217,None,15.29|2010-09-24T21:20:00Z,53.516566,8.188217,None,15.25|2010-09-24T21:30:00Z,53.516566,8.188217,None,15.22|2010-09-24T21:40:00Z,53.516566,8.188217,None,15.21|2010-09-24T21:50:00Z,53.516566,8.188217,None,15.21|2010-09-24T22:00:00Z,53.516566,8.188217,None,15.19|2010-09-24T22:10:00Z,53.516566,8.188217,None,15.17|2010-09-24T22:20:00Z,53.516566,8.188217,None,15.15|2010-09-24T22:30:00Z,53.516566,8.188217,None,15.11|2010-09-24T22:40:00Z,53.516566,8.188217,None,15.09|2010-09-24T22:50:00Z,53.516566,8.188217,None,15.07|2010-09-24T23:00:00Z,53.516566,8.188217,None,15.05|2010-09-24T23:10:00Z,53.516566,8.188217,None,14.96|2010-09-24T23:20:00Z,53.516566,8.188217,None,14.89|2010-09-24T23:30:00Z,53.516566,8.188217,None,14.85|2010-09-24T23:40:00Z,53.516566,8.188217,None,14.85|2010-09-24T23:50:00Z,53.516566,8.188217,None,14.82|2010-09-25T00:10:00Z,53.516566,8.188217,None,14.79|2010-09-25T00:20:00Z,53.516566,8.188217,None,14.79|2010-09-25T00:30:00Z,53.516566,8.188217,None,14.79|2010-09-25T00:40:00Z,53.516566,8.188217,None,14.8|2010-09-25T00:50:00Z,53.516566,8.188217,None,14.79|2010-09-25T01:00:00Z,53.516566,8.188217,None,14.78|2010-09-25T01:10:00Z,53.516566,8.188217,None,14.78|2010-09-25T01:20:00Z,53.516566,8.188217,None,14.79|2010-09-25T01:30:00Z,53.516566,8.188217,None,14.78|2010-09-25T01:40:00Z,53.516566,8.188217,None,14.8|2010-09-25T01:50:00Z,53.516566,8.188217,None,14.82|2010-09-25T02:00:00Z,53.516566,8.188217,None,14.91|2010-09-25T02:10:00Z,53.516566,8.188217,None,14.99|2010-09-25T02:20:00Z,53.516566,8.188217,None,15.03|2010-09-25T02:30:00Z,53.516566,8.188217,None,15.06|2010-09-25T02:40:00Z,53.516566,8.188217,None,15.08|2010-09-25T02:50:00Z,53.516566,8.188217,None,15.11|2010-09-25T03:00:00Z,53.516566,8.188217,None,15.09|2010-09-25T03:10:00Z,53.516566,8.188217,None,15.07|2010-09-25T03:20:00Z,53.516566,8.188217,None,15.04|2010-09-25T03:30:00Z,53.516566,8.188217,None,15.05|2010-09-25T03:40:00Z,53.516566,8.188217,None,15.05|2010-09-25T03:50:00Z,53.516566,8.188217,None,15.04|2010-09-25T04:00:00Z,53.516566,8.188217,None,15.02|2010-09-25T04:10:00Z,53.516566,8.188217,None,15.02|2010-09-25T04:20:00Z,53.516566,8.188217,None,15.02|2010-09-25T04:30:00Z,53.516566,8.188217,None,15.04|2010-09-25T04:40:00Z,53.516566,8.188217,None,15.04|2010-09-25T04:50:00Z,53.516566,8.188217,None,15.02|2010-09-25T05:00:00Z,53.516566,8.188217,None,14.99|2010-09-25T05:10:00Z,53.516566,8.188217,None,14.97|2010-09-25T05:20:00Z,53.516566,8.188217,None,14.95|2010-09-25T05:30:00Z,53.516566,8.188217,None,14.97|2010-09-25T05:40:00Z,53.516566,8.188217,None,15.01|2010-09-25T05:50:00Z,53.516566,8.188217,None,15.01|2010-09-25T06:00:00Z,53.516566,8.188217,None,15.0|2010-09-25T06:10:00Z,53.516566,8.188217,None,14.99|2010-09-25T06:20:00Z,53.516566,8.188217,None,14.92|2010-09-25T06:30:00Z,53.516566,8.188217,None,14.84|2010-09-25T06:40:00Z,53.516566,8.188217,None,14.8|2010-09-25T06:50:00Z,53.516566,8.188217,None,14.78|2010-09-25T07:00:00Z,53.516566,8.188217,None,14.74|2010-09-25T07:10:00Z,53.516566,8.188217,None,14.76|2010-09-25T07:20:00Z,53.516566,8.188217,None,14.81|2010-09-25T07:30:00Z,53.516566,8.188217,None,14.86|2010-09-25T07:40:00Z,53.516566,8.188217,None,14.85|2010-09-25T07:50:00Z,53.516566,8.188217,None,14.8|2010-09-25T08:00:00Z,53.516566,8.188217,None,14.92|2010-09-25T08:10:00Z,53.516566,8.188217,None,14.95|2010-09-25T08:20:00Z,53.516566,8.188217,None,15.01|2010-09-25T08:30:00Z,53.516566,8.188217,None,15.02|2010-09-25T08:40:00Z,53.516566,8.188217,None,15.01|2010-09-25T08:50:00Z,53.516566,8.188217,None,15.01|2010-09-25T09:00:00Z,53.516566,8.188217,None,15.03|2010-09-25T09:10:00Z,53.516566,8.188217,None,15.05|2010-09-25T09:20:00Z,53.516566,8.188217,None,15.06|2010-09-25T09:30:00Z,53.516566,8.188217,None,15.04|2010-09-25T09:40:00Z,53.516566,8.188217,None,14.86|2010-09-25T09:50:00Z,53.516566,8.188217,None,14.87|2010-09-25T10:00:00Z,53.516566,8.188217,None,11.91|2010-09-25T10:10:00Z,53.516566,8.188217,None,11.96|2010-09-25T10:20:00Z,53.516566,8.188217,None,10.95|2010-09-25T10:30:00Z,53.516566,8.188217,None,10.09|2010-09-25T10:40:00Z,53.516566,8.188217,None,10.16|2010-09-25T10:50:00Z,53.516566,8.188217,None,12.58|2010-09-25T11:00:00Z,53.516566,8.188217,None,13.9|2010-09-25T11:10:00Z,53.516566,8.188217,None,14.25|2010-09-25T11:20:00Z,53.516566,8.188217,None,14.8|2010-09-25T11:30:00Z,53.516566,8.188217,None,14.93|2010-09-25T11:40:00Z,53.516566,8.188217,None,14.89|2010-09-25T11:50:00Z,53.516566,8.188217,None,14.85|2010-09-25T12:00:00Z,53.516566,8.188217,None,14.8|2010-09-25T12:10:00Z,53.516566,8.188217,None,12.14|2010-09-25T12:20:00Z,53.516566,8.188217,None,13.94|2010-09-25T12:30:00Z,53.516566,8.188217,None,14.75|2010-09-25T12:40:00Z,53.516566,8.188217,None,14.75|2010-09-25T12:50:00Z,53.516566,8.188217,None,14.76|2010-09-25T13:00:00Z,53.516566,8.188217,None,14.77|2010-09-25T13:10:00Z,53.516566,8.188217,None,14.78|2010-09-25T13:20:00Z,53.516566,8.188217,None,14.79|2010-09-25T13:30:00Z,53.516566,8.188217,None,14.8|2010-09-25T13:40:00Z,53.516566,8.188217,None,14.81|2010-09-25T13:50:00Z,53.516566,8.188217,None,14.84|2010-09-25T14:00:00Z,53.516566,8.188217,None,14.93|2010-09-25T14:10:00Z,53.516566,8.188217,None,15.04|2010-09-25T14:20:00Z,53.516566,8.188217,None,15.09|2010-09-25T14:30:00Z,53.516566,8.188217,None,15.13|2010-09-25T14:40:00Z,53.516566,8.188217,None,15.13|2010-09-25T14:50:00Z,53.516566,8.188217,None,15.13|2010-09-25T15:00:00Z,53.516566,8.188217,None,15.15|2010-09-25T15:10:00Z,53.516566,8.188217,None,15.12|2010-09-25T15:20:00Z,53.516566,8.188217,None,15.09|2010-09-25T15:30:00Z,53.516566,8.188217,None,15.08|2010-09-25T15:40:00Z,53.516566,8.188217,None,15.06|2010-09-25T15:50:00Z,53.516566,8.188217,None,15.05|2010-09-25T16:00:00Z,53.516566,8.188217,None,15.05|2010-09-25T16:10:00Z,53.516566,8.188217,None,15.08|2010-09-25T16:20:00Z,53.516566,8.188217,None,15.09|2010-09-25T16:30:00Z,53.516566,8.188217,None,15.08|2010-09-25T16:40:00Z,53.516566,8.188217,None,15.07|2010-09-25T16:50:00Z,53.516566,8.188217,None,15.05|2010-09-25T17:00:00Z,53.516566,8.188217,None,15.04|2010-09-25T17:10:00Z,53.516566,8.188217,None,15.02|2010-09-25T17:20:00Z,53.516566,8.188217,None,15.02|2010-09-25T17:30:00Z,53.516566,8.188217,None,15.01|2010-09-25T17:40:00Z,53.516566,8.188217,None,14.99|2010-09-25T17:50:00Z,53.516566,8.188217,None,14.98|2010-09-25T18:00:00Z,53.516566,8.188217,None,14.98|2010-09-25T18:10:00Z,53.516566,8.188217,None,14.99|2010-09-25T18:20:00Z,53.516566,8.188217,None,14.99|2010-09-25T18:30:00Z,53.516566,8.188217,None,14.83|2010-09-25T18:40:00Z,53.516566,8.188217,None,14.81|2010-09-25T18:50:00Z,53.516566,8.188217,None,14.85|2010-09-25T19:00:00Z,53.516566,8.188217,None,14.87|2010-09-25T19:10:00Z,53.516566,8.188217,None,14.87|2010-09-25T19:20:00Z,53.516566,8.188217,None,14.88|2010-09-25T19:30:00Z,53.516566,8.188217,None,14.93|2010-09-25T19:40:00Z,53.516566,8.188217,None,14.95|2010-09-25T19:50:00Z,53.516566,8.188217,None,14.93|2010-09-25T20:00:00Z,53.516566,8.188217,None,14.78|2010-09-25T20:10:00Z,53.516566,8.188217,None,14.77|2010-09-25T20:20:00Z,53.516566,8.188217,None,14.79|2010-09-25T20:30:00Z,53.516566,8.188217,None,14.96|2010-09-25T20:40:00Z,53.516566,8.188217,None,15.03|2010-09-25T20:50:00Z,53.516566,8.188217,None,15.05|2010-09-25T21:00:00Z,53.516566,8.188217,None,15.02|2010-09-25T21:10:00Z,53.516566,8.188217,None,15.03|2010-09-25T21:20:00Z,53.516566,8.188217,None,15.09|2010-09-25T21:30:00Z,53.516566,8.188217,None,15.07|2010-09-25T21:40:00Z,53.516566,8.188217,None,15.06|2010-09-25T21:50:00Z,53.516566,8.188217,None,15.07|2010-09-25T22:00:00Z,53.516566,8.188217,None,14.99|2010-09-25T22:10:00Z,53.516566,8.188217,None,15.0|2010-09-25T22:20:00Z,53.516566,8.188217,None,15.01|2010-09-25T22:30:00Z,53.516566,8.188217,None,15.03|2010-09-25T22:40:00Z,53.516566,8.188217,None,15.05|2010-09-25T22:50:00Z,53.516566,8.188217,None,15.06|2010-09-25T23:00:00Z,53.516566,8.188217,None,15.05|2010-09-25T23:10:00Z,53.516566,8.188217,None,15.04|2010-09-25T23:20:00Z,53.516566,8.188217,None,15.03|2010-09-25T23:30:00Z,53.516566,8.188217,None,15.02|2010-09-25T23:40:00Z,53.516566,8.188217,None,15.01|2010-09-25T23:50:00Z,53.516566,8.188217,None,15.01|2010-09-26T00:00:00Z,53.516566,8.188217,None,15.0|2010-09-26T00:10:00Z,53.516566,8.188217,None,14.97|2010-09-26T00:20:00Z,53.516566,8.188217,None,14.95|2010-09-26T00:30:00Z,53.516566,8.188217,None,14.93|2010-09-26T00:40:00Z,53.516566,8.188217,None,14.92|2010-09-26T00:50:00Z,53.516566,8.188217,None,14.89|2010-09-26T01:00:00Z,53.516566,8.188217,None,14.87|2010-09-26T01:10:00Z,53.516566,8.188217,None,14.85|2010-09-26T01:20:00Z,53.516566,8.188217,None,14.84|2010-09-26T01:30:00Z,53.516566,8.188217,None,14.84|2010-09-26T01:40:00Z,53.516566,8.188217,None,14.85|2010-09-26T01:50:00Z,53.516566,8.188217,None,14.88|2010-09-26T02:00:00Z,53.516566,8.188217,None,14.9|2010-09-26T02:10:00Z,53.516566,8.188217,None,14.93|2010-09-26T02:20:00Z,53.516566,8.188217,None,14.95|2010-09-26T02:30:00Z,53.516566,8.188217,None,14.94|2010-09-26T02:40:00Z,53.516566,8.188217,None,14.92|2010-09-26T02:50:00Z,53.516566,8.188217,None,14.87|2010-09-26T03:00:00Z,53.516566,8.188217,None,14.77|2010-09-26T03:10:00Z,53.516566,8.188217,None,14.67|2010-09-26T03:20:00Z,53.516566,8.188217,None,14.6|2010-09-26T03:30:00Z,53.516566,8.188217,None,14.59|2010-09-26T03:40:00Z,53.516566,8.188217,None,14.6|2010-09-26T03:50:00Z,53.516566,8.188217,None,14.61|2010-09-26T04:00:00Z,53.516566,8.188217,None,14.6|2010-09-26T04:10:00Z,53.516566,8.188217,None,14.66|2010-09-26T04:20:00Z,53.516566,8.188217,None,14.77|2010-09-26T04:30:00Z,53.516566,8.188217,None,14.79|2010-09-26T04:40:00Z,53.516566,8.188217,None,14.77|2010-09-26T04:50:00Z,53.516566,8.188217,None,14.77|2010-09-26T05:00:00Z,53.516566,8.188217,None,14.76|2010-09-26T05:10:00Z,53.516566,8.188217,None,14.76|2010-09-26T05:20:00Z,53.516566,8.188217,None,14.75|2010-09-26T05:30:00Z,53.516566,8.188217,None,14.74|2010-09-26T05:40:00Z,53.516566,8.188217,None,14.73|2010-09-26T05:50:00Z,53.516566,8.188217,None,14.71|2010-09-26T06:00:00Z,53.516566,8.188217,None,14.71|2010-09-26T06:10:00Z,53.516566,8.188217,None,14.73|2010-09-26T06:20:00Z,53.516566,8.188217,None,14.78|2010-09-26T06:30:00Z,53.516566,8.188217,None,14.76|2010-09-26T06:40:00Z,53.516566,8.188217,None,14.64|2010-09-26T06:50:00Z,53.516566,8.188217,None,14.58|2010-09-26T07:00:00Z,53.516566,8.188217,None,14.56|2010-09-26T07:10:00Z,53.516566,8.188217,None,14.55|2010-09-26T07:20:00Z,53.516566,8.188217,None,14.55|2010-09-26T07:30:00Z,53.516566,8.188217,None,14.58|2010-09-26T07:40:00Z,53.516566,8.188217,None,14.68|2010-09-26T07:50:00Z,53.516566,8.188217,None,14.7|2010-09-26T08:00:00Z,53.516566,8.188217,None,14.68|2010-09-26T08:10:00Z,53.516566,8.188217,None,14.67|2010-09-26T08:20:00Z,53.516566,8.188217,None,14.7|2010-09-26T08:30:00Z,53.516566,8.188217,None,14.55|2010-09-26T08:40:00Z,53.516566,8.188217,None,14.61|2010-09-26T08:50:00Z,53.516566,8.188217,None,14.76|2010-09-26T09:00:00Z,53.516566,8.188217,None,14.81|2010-09-26T09:10:00Z,53.516566,8.188217,None,14.8|2010-09-26T09:20:00Z,53.516566,8.188217,None,14.83|2010-09-26T09:30:00Z,53.516566,8.188217,None,14.83|2010-09-26T09:40:00Z,53.516566,8.188217,None,14.84|2010-09-26T09:50:00Z,53.516566,8.188217,None,14.85|2010-09-26T10:00:00Z,53.516566,8.188217,None,14.79|2010-09-26T10:10:00Z,53.516566,8.188217,None,14.75|2010-09-26T10:20:00Z,53.516566,8.188217,None,14.79|2010-09-26T10:30:00Z,53.516566,8.188217,None,14.83|2010-09-26T10:40:00Z,53.516566,8.188217,None,14.84|2010-09-26T10:50:00Z,53.516566,8.188217,None,14.87|2010-09-26T11:00:00Z,53.516566,8.188217,None,14.89|2010-09-26T11:10:00Z,53.516566,8.188217,None,14.9|2010-09-26T11:20:00Z,53.516566,8.188217,None,14.89|2010-09-26T11:30:00Z,53.516566,8.188217,None,14.91|2010-09-26T11:40:00Z,53.516566,8.188217,None,14.91|2010-09-26T11:50:00Z,53.516566,8.188217,None,14.91|2010-09-26T12:00:00Z,53.516566,8.188217,None,14.9|2010-09-26T12:10:00Z,53.516566,8.188217,None,14.89|2010-09-26T12:20:00Z,53.516566,8.188217,None,14.88|2010-09-26T12:30:00Z,53.516566,8.188217,None,14.85|2010-09-26T12:40:00Z,53.516566,8.188217,None,14.82|2010-09-26T12:50:00Z,53.516566,8.188217,None,14.82|2010-09-26T13:00:00Z,53.516566,8.188217,None,14.83|2010-09-26T13:10:00Z,53.516566,8.188217,None,14.83|2010-09-26T13:20:00Z,53.516566,8.188217,None,14.83|2010-09-26T13:30:00Z,53.516566,8.188217,None,14.82|2010-09-26T13:40:00Z,53.516566,8.188217,None,14.82|2010-09-26T13:50:00Z,53.516566,8.188217,None,14.82|2010-09-26T14:00:00Z,53.516566,8.188217,None,14.83|2010-09-26T14:10:00Z,53.516566,8.188217,None,14.82|2010-09-26T14:20:00Z,53.516566,8.188217,None,14.81|2010-09-26T14:30:00Z,53.516566,8.188217,None,14.85|2010-09-26T14:40:00Z,53.516566,8.188217,None,14.86|2010-09-26T14:50:00Z,53.516566,8.188217,None,14.83|2010-09-26T15:00:00Z,53.516566,8.188217,None,14.76|2010-09-26T15:10:00Z,53.516566,8.188217,None,14.74|2010-09-26T15:20:00Z,53.516566,8.188217,None,14.71|2010-09-26T15:30:00Z,53.516566,8.188217,None,14.61|2010-09-26T15:40:00Z,53.516566,8.188217,None,14.61|2010-09-26T15:50:00Z,53.516566,8.188217,None,14.67|2010-09-26T16:00:00Z,53.516566,8.188217,None,14.74|2010-09-26T16:10:00Z,53.516566,8.188217,None,14.78|2010-09-26T16:20:00Z,53.516566,8.188217,None,14.8|2010-09-26T16:30:00Z,53.516566,8.188217,None,14.79|2010-09-26T16:40:00Z,53.516566,8.188217,None,14.74|2010-09-26T16:50:00Z,53.516566,8.188217,None,14.68|2010-09-26T17:00:00Z,53.516566,8.188217,None,14.63|2010-09-26T17:10:00Z,53.516566,8.188217,None,14.62|2010-09-26T17:20:00Z,53.516566,8.188217,None,14.62|2010-09-26T17:30:00Z,53.516566,8.188217,None,14.62|2010-09-26T17:40:00Z,53.516566,8.188217,None,14.62|2010-09-26T17:50:00Z,53.516566,8.188217,None,14.59|2010-09-26T18:00:00Z,53.516566,8.188217,None,14.59|2010-09-26T18:10:00Z,53.516566,8.188217,None,14.57|2010-09-26T18:20:00Z,53.516566,8.188217,None,14.57|2010-09-26T18:30:00Z,53.516566,8.188217,None,14.58|2010-09-26T18:40:00Z,53.516566,8.188217,None,14.59|2010-09-26T18:50:00Z,53.516566,8.188217,None,14.52|2010-09-26T19:00:00Z,53.516566,8.188217,None,14.49|2010-09-26T19:10:00Z,53.516566,8.188217,None,14.47|2010-09-26T19:20:00Z,53.516566,8.188217,None,14.45|2010-09-26T19:30:00Z,53.516566,8.188217,None,14.45|2010-09-26T19:40:00Z,53.516566,8.188217,None,14.46|2010-09-26T19:50:00Z,53.516566,8.188217,None,14.48|2010-09-26T20:00:00Z,53.516566,8.188217,None,14.49|2010-09-26T20:10:00Z,53.516566,8.188217,None,14.49|2010-09-26T20:20:00Z,53.516566,8.188217,None,14.5|2010-09-26T20:30:00Z,53.516566,8.188217,None,14.45|2010-09-26T20:40:00Z,53.516566,8.188217,None,14.45|2010-09-26T20:50:00Z,53.516566,8.188217,None,14.5|2010-09-26T21:00:00Z,53.516566,8.188217,None,14.52|2010-09-26T21:10:00Z,53.516566,8.188217,None,14.55|2010-09-26T21:20:00Z,53.516566,8.188217,None,14.49|2010-09-26T21:30:00Z,53.516566,8.188217,None,14.47|2010-09-26T21:40:00Z,53.516566,8.188217,None,14.51|2010-09-26T21:50:00Z,53.516566,8.188217,None,14.56|2010-09-26T22:00:00Z,53.516566,8.188217,None,14.6|2010-09-26T22:10:00Z,53.516566,8.188217,None,14.57|2010-09-26T22:20:00Z,53.516566,8.188217,None,14.5|2010-09-26T22:30:00Z,53.516566,8.188217,None,14.58|2010-09-26T22:40:00Z,53.516566,8.188217,None,14.66|2010-09-26T22:50:00Z,53.516566,8.188217,None,14.61|2010-09-26T23:00:00Z,53.516566,8.188217,None,14.62|2010-09-26T23:10:00Z,53.516566,8.188217,None,14.66|2010-09-26T23:20:00Z,53.516566,8.188217,None,14.69|2010-09-26T23:30:00Z,53.516566,8.188217,None,14.72|2010-09-26T23:40:00Z,53.516566,8.188217,None,14.72|2010-09-26T23:50:00Z,53.516566,8.188217,None,14.72|2010-09-27T00:00:00Z,53.516566,8.188217,None,14.74|2010-09-27T00:10:00Z,53.516566,8.188217,None,14.74|2010-09-27T00:20:00Z,53.516566,8.188217,None,14.74|2010-09-27T00:30:00Z,53.516566,8.188217,None,14.75|2010-09-27T00:40:00Z,53.516566,8.188217,None,14.76|2010-09-27T00:50:00Z,53.516566,8.188217,None,14.77|2010-09-27T01:00:00Z,53.516566,8.188217,None,14.78|2010-09-27T01:10:00Z,53.516566,8.188217,None,14.78|2010-09-27T01:20:00Z,53.516566,8.188217,None,14.79|2010-09-27T01:30:00Z,53.516566,8.188217,None,14.79|2010-09-27T01:40:00Z,53.516566,8.188217,None,14.78|2010-09-27T01:50:00Z,53.516566,8.188217,None,14.78|2010-09-27T02:00:00Z,53.516566,8.188217,None,14.77|2010-09-27T02:10:00Z,53.516566,8.188217,None,14.77|2010-09-27T02:20:00Z,53.516566,8.188217,None,14.78|2010-09-27T02:30:00Z,53.516566,8.188217,None,14.77|2010-09-27T02:40:00Z,53.516566,8.188217,None,14.74|2010-09-27T02:50:00Z,53.516566,8.188217,None,14.71|2010-09-27T03:00:00Z,53.516566,8.188217,None,14.65|2010-09-27T03:10:00Z,53.516566,8.188217,None,14.59|2010-09-27T03:20:00Z,53.516566,8.188217,None,14.51|2010-09-27T03:30:00Z,53.516566,8.188217,None,14.39|2010-09-27T03:40:00Z,53.516566,8.188217,None,14.33|2010-09-27T03:50:00Z,53.516566,8.188217,None,14.31|2010-09-27T04:00:00Z,53.516566,8.188217,None,14.22|2010-09-27T04:10:00Z,53.516566,8.188217,None,14.22|2010-09-27T04:20:00Z,53.516566,8.188217,None,14.32|2010-09-27T04:30:00Z,53.516566,8.188217,None,14.41|2010-09-27T04:40:00Z,53.516566,8.188217,None,14.44|2010-09-27T04:50:00Z,53.516566,8.188217,None,14.48|2010-09-27T05:00:00Z,53.516566,8.188217,None,14.52|2010-09-27T05:10:00Z,53.516566,8.188217,None,14.55|2010-09-27T05:20:00Z,53.516566,8.188217,None,14.55|2010-09-27T05:30:00Z,53.516566,8.188217,None,14.55|2010-09-27T05:40:00Z,53.516566,8.188217,None,14.54|2010-09-27T05:50:00Z,53.516566,8.188217,None,14.53|2010-09-27T06:00:00Z,53.516566,8.188217,None,14.53|2010-09-27T06:10:00Z,53.516566,8.188217,None,14.52|2010-09-27T06:20:00Z,53.516566,8.188217,None,14.52|2010-09-27T06:30:00Z,53.516566,8.188217,None,14.53|2010-09-27T06:40:00Z,53.516566,8.188217,None,14.48|2010-09-27T06:50:00Z,53.516566,8.188217,None,14.38|2010-09-27T07:00:00Z,53.516566,8.188217,None,14.36|2010-09-27T07:10:00Z,53.516566,8.188217,None,14.35|2010-09-27T07:20:00Z,53.516566,8.188217,None,14.35|2010-09-27T07:30:00Z,53.516566,8.188217,None,14.37|2010-09-27T07:40:00Z,53.516566,8.188217,None,14.44|2010-09-27T07:50:00Z,53.516566,8.188217,None,14.46|2010-09-27T08:00:00Z,53.516566,8.188217,None,14.47|2010-09-27T08:10:00Z,53.516566,8.188217,None,14.47|2010-09-27T08:20:00Z,53.516566,8.188217,None,14.46|2010-09-27T08:30:00Z,53.516566,8.188217,None,14.44|2010-09-27T08:40:00Z,53.516566,8.188217,None,14.4|2010-09-27T08:50:00Z,53.516566,8.188217,None,14.2|2010-09-27T09:00:00Z,53.516566,8.188217,None,14.34|2010-09-27T09:10:00Z,53.516566,8.188217,None,14.48|2010-09-27T09:20:00Z,53.516566,8.188217,None,14.51|2010-09-27T09:30:00Z,53.516566,8.188217,None,14.48|2010-09-27T09:40:00Z,53.516566,8.188217,None,14.48|2010-09-27T09:50:00Z,53.516566,8.188217,None,14.43|2010-09-27T10:00:00Z,53.516566,8.188217,None,14.39|2010-09-27T10:10:00Z,53.516566,8.188217,None,14.4|2010-09-27T10:20:00Z,53.516566,8.188217,None,14.39|2010-09-27T10:30:00Z,53.516566,8.188217,None,14.39|2010-09-27T10:40:00Z,53.516566,8.188217,None,14.45|2010-09-27T10:50:00Z,53.516566,8.188217,None,14.51|2010-09-27T11:00:00Z,53.516566,8.188217,None,14.53|2010-09-27T11:10:00Z,53.516566,8.188217,None,14.57|2010-09-27T11:20:00Z,53.516566,8.188217,None,14.59|2010-09-27T11:30:00Z,53.516566,8.188217,None,14.63|2010-09-27T11:40:00Z,53.516566,8.188217,None,14.65|2010-09-27T11:50:00Z,53.516566,8.188217,None,14.63|2010-09-27T12:00:00Z,53.516566,8.188217,None,14.64|2010-09-27T12:10:00Z,53.516566,8.188217,None,14.66|2010-09-27T12:20:00Z,53.516566,8.188217,None,14.67|2010-09-27T12:30:00Z,53.516566,8.188217,None,14.68|2010-09-27T12:40:00Z,53.516566,8.188217,None,14.69|2010-09-27T12:50:00Z,53.516566,8.188217,None,14.7|2010-09-27T13:00:00Z,53.516566,8.188217,None,14.7|2010-09-27T13:10:00Z,53.516566,8.188217,None,14.71|2010-09-27T13:20:00Z,53.516566,8.188217,None,14.71|2010-09-27T13:30:00Z,53.516566,8.188217,None,14.71|2010-09-27T13:40:00Z,53.516566,8.188217,None,14.71|2010-09-27T13:50:00Z,53.516566,8.188217,None,14.7|2010-09-27T14:00:00Z,53.516566,8.188217,None,14.7|2010-09-27T14:10:00Z,53.516566,8.188217,None,14.7|2010-09-27T14:20:00Z,53.516566,8.188217,None,14.7|2010-09-27T14:30:00Z,53.516566,8.188217,None,14.71|2010-09-27T14:40:00Z,53.516566,8.188217,None,14.71|2010-09-27T14:50:00Z,53.516566,8.188217,None,14.7|2010-09-27T15:00:00Z,53.516566,8.188217,None,14.69|2010-09-27T15:10:00Z,53.516566,8.188217,None,14.66|2010-09-27T15:20:00Z,53.516566,8.188217,None,14.62|2010-09-27T15:30:00Z,53.516566,8.188217,None,14.57|2010-09-27T15:40:00Z,53.516566,8.188217,None,14.46|2010-09-27T15:50:00Z,53.516566,8.188217,None,14.42|2010-09-27T16:00:00Z,53.516566,8.188217,None,14.44|2010-09-27T16:10:00Z,53.516566,8.188217,None,14.42|2010-09-27T16:20:00Z,53.516566,8.188217,None,14.36|2010-09-27T16:30:00Z,53.516566,8.188217,None,14.33|2010-09-27T16:40:00Z,53.516566,8.188217,None,14.35|2010-09-27T16:50:00Z,53.516566,8.188217,None,14.38|2010-09-27T17:00:00Z,53.516566,8.188217,None,14.39|2010-09-27T17:10:00Z,53.516566,8.188217,None,14.37|2010-09-27T17:20:00Z,53.516566,8.188217,None,14.35|2010-09-27T17:30:00Z,53.516566,8.188217,None,14.34|2010-09-27T17:40:00Z,53.516566,8.188217,None,14.33|2010-09-27T17:50:00Z,53.516566,8.188217,None,14.33|2010-09-27T18:00:00Z,53.516566,8.188217,None,14.33|2010-09-27T18:10:00Z,53.516566,8.188217,None,14.32|2010-09-27T18:20:00Z,53.516566,8.188217,None,14.32|2010-09-27T18:30:00Z,53.516566,8.188217,None,14.32|2010-09-27T18:40:00Z,53.516566,8.188217,None,14.35|2010-09-27T18:50:00Z,53.516566,8.188217,None,14.33|2010-09-27T19:00:00Z,53.516566,8.188217,None,14.3|2010-09-27T19:10:00Z,53.516566,8.188217,None,14.27|2010-09-27T19:20:00Z,53.516566,8.188217,None,14.26|2010-09-27T19:30:00Z,53.516566,8.188217,None,14.26|2010-09-27T19:40:00Z,53.516566,8.188217,None,14.28|2010-09-27T19:50:00Z,53.516566,8.188217,None,14.33|2010-09-27T20:00:00Z,53.516566,8.188217,None,14.36|2010-09-27T20:10:00Z,53.516566,8.188217,None,14.35|2010-09-27T20:20:00Z,53.516566,8.188217,None,14.35|2010-09-27T20:30:00Z,53.516566,8.188217,None,14.34|2010-09-27T20:40:00Z,53.516566,8.188217,None,14.31|2010-09-27T20:50:00Z,53.516566,8.188217,None,14.3|2010-09-27T21:00:00Z,53.516566,8.188217,None,14.2|2010-09-27T21:10:00Z,53.516566,8.188217,None,14.22|2010-09-27T21:20:00Z,53.516566,8.188217,None,14.35|2010-09-27T21:30:00Z,53.516566,8.188217,None,14.37|2010-09-27T21:40:00Z,53.516566,8.188217,None,14.37|2010-09-27T21:50:00Z,53.516566,8.188217,None,14.34|2010-09-27T22:00:00Z,53.516566,8.188217,None,14.28|2010-09-27T22:10:00Z,53.516566,8.188217,None,14.28|2010-09-27T22:20:00Z,53.516566,8.188217,None,14.28|2010-09-27T22:30:00Z,53.516566,8.188217,None,14.31|2010-09-27T22:40:00Z,53.516566,8.188217,None,14.32|2010-09-27T22:50:00Z,53.516566,8.188217,None,14.35|2010-09-27T23:00:00Z,53.516566,8.188217,None,14.38|2010-09-27T23:10:00Z,53.516566,8.188217,None,14.39|2010-09-27T23:20:00Z,53.516566,8.188217,None,14.41|2010-09-27T23:30:00Z,53.516566,8.188217,None,14.43|2010-09-27T23:40:00Z,53.516566,8.188217,None,14.46|2010-09-27T23:50:00Z,53.516566,8.188217,None,14.49|2010-09-28T00:00:00Z,53.516566,8.188217,None,14.49|2010-09-28T00:10:00Z,53.516566,8.188217,None,14.5|2010-09-28T00:20:00Z,53.516566,8.188217,None,14.52|2010-09-28T00:30:00Z,53.516566,8.188217,None,14.54|2010-09-28T00:40:00Z,53.516566,8.188217,None,14.54|2010-09-28T00:50:00Z,53.516566,8.188217,None,14.55|2010-09-28T01:00:00Z,53.516566,8.188217,None,14.55|2010-09-28T01:10:00Z,53.516566,8.188217,None,14.57|2010-09-28T01:20:00Z,53.516566,8.188217,None,14.59|2010-09-28T01:30:00Z,53.516566,8.188217,None,14.6|2010-09-28T01:40:00Z,53.516566,8.188217,None,14.61|2010-09-28T01:50:00Z,53.516566,8.188217,None,14.61|2010-09-28T02:00:00Z,53.516566,8.188217,None,14.62|2010-09-28T02:10:00Z,53.516566,8.188217,None,14.62|2010-09-28T02:20:00Z,53.516566,8.188217,None,14.61|2010-09-28T02:30:00Z,53.516566,8.188217,None,14.61|2010-09-28T02:40:00Z,53.516566,8.188217,None,14.6|2010-09-28T02:50:00Z,53.516566,8.188217,None,14.58|2010-09-28T03:00:00Z,53.516566,8.188217,None,14.55|2010-09-28T03:10:00Z,53.516566,8.188217,None,14.5|2010-09-28T03:20:00Z,53.516566,8.188217,None,14.46|2010-09-28T03:30:00Z,53.516566,8.188217,None,14.4|2010-09-28T03:40:00Z,53.516566,8.188217,None,14.32|2010-09-28T03:50:00Z,53.516566,8.188217,None,14.24|2010-09-28T04:00:00Z,53.516566,8.188217,None,14.18|2010-09-28T04:10:00Z,53.516566,8.188217,None,14.18|2010-09-28T04:20:00Z,53.516566,8.188217,None,14.13|2010-09-28T04:30:00Z,53.516566,8.188217,None,14.04|2010-09-28T04:40:00Z,53.516566,8.188217,None,14.03|2010-09-28T04:50:00Z,53.516566,8.188217,None,14.1|2010-09-28T05:00:00Z,53.516566,8.188217,None,14.14|2010-09-28T05:10:00Z,53.516566,8.188217,None,14.16|2010-09-28T05:20:00Z,53.516566,8.188217,None,14.16|2010-09-28T05:30:00Z,53.516566,8.188217,None,14.18|2010-09-28T05:40:00Z,53.516566,8.188217,None,14.2|2010-09-28T05:50:00Z,53.516566,8.188217,None,14.2|2010-09-28T06:00:00Z,53.516566,8.188217,None,14.2|2010-09-28T06:10:00Z,53.516566,8.188217,None,14.19|2010-09-28T06:20:00Z,53.516566,8.188217,None,14.17|2010-09-28T06:30:00Z,53.516566,8.188217,None,14.17|2010-09-28T06:40:00Z,53.516566,8.188217,None,14.17|2010-09-28T06:50:00Z,53.516566,8.188217,None,14.17|2010-09-28T07:00:00Z,53.516566,8.188217,None,14.19|2010-09-28T07:10:00Z,53.516566,8.188217,None,14.21|2010-09-28T07:20:00Z,53.516566,8.188217,None,14.12|2010-09-28T07:30:00Z,53.516566,8.188217,None,14.06|2010-09-28T07:40:00Z,53.516566,8.188217,None,14.05|2010-09-28T07:50:00Z,53.516566,8.188217,None,14.05|2010-09-28T08:00:00Z,53.516566,8.188217,None,14.05|2010-09-28T08:10:00Z,53.516566,8.188217,None,14.16|2010-09-28T08:20:00Z,53.516566,8.188217,None,14.21|2010-09-28T08:30:00Z,53.516566,8.188217,None,14.22|2010-09-28T08:40:00Z,53.516566,8.188217,None,14.23|2010-09-28T08:50:00Z,53.516566,8.188217,None,14.21|2010-09-28T09:00:00Z,53.516566,8.188217,None,14.15|2010-09-28T09:10:00Z,53.516566,8.188217,None,14.09|2010-09-28T09:20:00Z,53.516566,8.188217,None,13.97|2010-09-28T09:30:00Z,53.516566,8.188217,None,14.02|2010-09-28T09:40:00Z,53.516566,8.188217,None,14.18|2010-09-28T09:50:00Z,53.516566,8.188217,None,14.18|2010-09-28T10:00:00Z,53.516566,8.188217,None,14.04|2010-09-28T10:10:00Z,53.516566,8.188217,None,14.02|2010-09-28T10:20:00Z,53.516566,8.188217,None,14.1|2010-09-28T10:30:00Z,53.516566,8.188217,None,14.11|2010-09-28T10:40:00Z,53.516566,8.188217,None,14.11|2010-09-28T10:50:00Z,53.516566,8.188217,None,14.15|2010-09-28T11:00:00Z,53.516566,8.188217,None,14.12|2010-09-28T11:10:00Z,53.516566,8.188217,None,14.15|2010-09-28T11:20:00Z,53.516566,8.188217,None,14.2|2010-09-28T11:30:00Z,53.516566,8.188217,None,14.24|2010-09-28T11:40:00Z,53.516566,8.188217,None,14.23|2010-09-28T11:50:00Z,53.516566,8.188217,None,14.25|2010-09-28T12:00:00Z,53.516566,8.188217,None,14.29|2010-09-28T12:10:00Z,53.516566,8.188217,None,14.32|2010-09-28T12:20:00Z,53.516566,8.188217,None,14.35|2010-09-28T12:30:00Z,53.516566,8.188217,None,14.36|2010-09-28T12:40:00Z,53.516566,8.188217,None,14.36|2010-09-28T12:50:00Z,53.516566,8.188217,None,14.38|2010-09-28T13:00:00Z,53.516566,8.188217,None,14.4|2010-09-28T13:10:00Z,53.516566,8.188217,None,14.4|2010-09-28T13:20:00Z,53.516566,8.188217,None,14.42|2010-09-28T13:30:00Z,53.516566,8.188217,None,14.45|2010-09-28T13:40:00Z,53.516566,8.188217,None,14.47|2010-09-28T13:50:00Z,53.516566,8.188217,None,14.49|2010-09-28T14:00:00Z,53.516566,8.188217,None,14.51|2010-09-28T14:10:00Z,53.516566,8.188217,None,14.52|2010-09-28T14:20:00Z,53.516566,8.188217,None,14.54|2010-09-28T14:30:00Z,53.516566,8.188217,None,14.56|2010-09-28T14:40:00Z,53.516566,8.188217,None,14.56|2010-09-28T14:50:00Z,53.516566,8.188217,None,14.56|2010-09-28T15:00:00Z,53.516566,8.188217,None,14.56|2010-09-28T15:10:00Z,53.516566,8.188217,None,14.53|2010-09-28T15:20:00Z,53.516566,8.188217,None,14.5|2010-09-28T15:30:00Z,53.516566,8.188217,None,14.45|2010-09-28T15:40:00Z,53.516566,8.188217,None,14.41|2010-09-28T15:50:00Z,53.516566,8.188217,None,14.35|2010-09-28T16:00:00Z,53.516566,8.188217,None,14.26|2010-09-28T16:10:00Z,53.516566,8.188217,None,14.19|2010-09-28T16:20:00Z,53.516566,8.188217,None,14.15|2010-09-28T16:30:00Z,53.516566,8.188217,None,14.14|2010-09-28T16:40:00Z,53.516566,8.188217,None,14.1|2010-09-28T16:50:00Z,53.516566,8.188217,None,14.06|2010-09-28T17:00:00Z,53.516566,8.188217,None,14.05|2010-09-28T17:10:00Z,53.516566,8.188217,None,14.07|2010-09-28T17:20:00Z,53.516566,8.188217,None,14.08|2010-09-28T17:30:00Z,53.516566,8.188217,None,14.08|2010-09-28T17:40:00Z,53.516566,8.188217,None,14.07|2010-09-28T17:50:00Z,53.516566,8.188217,None,14.06|2010-09-28T18:00:00Z,53.516566,8.188217,None,14.05|2010-09-28T18:10:00Z,53.516566,8.188217,None,14.03|2010-09-28T18:20:00Z,53.516566,8.188217,None,14.01|2010-09-28T18:30:00Z,53.516566,8.188217,None,14.0|2010-09-28T18:40:00Z,53.516566,8.188217,None,13.99|2010-09-28T18:50:00Z,53.516566,8.188217,None,13.99|2010-09-28T19:00:00Z,53.516566,8.188217,None,14.0|2010-09-28T19:10:00Z,53.516566,8.188217,None,14.0|2010-09-28T19:20:00Z,53.516566,8.188217,None,13.99|2010-09-28T19:30:00Z,53.516566,8.188217,None,13.9|2010-09-28T19:40:00Z,53.516566,8.188217,None,13.84|2010-09-28T19:50:00Z,53.516566,8.188217,None,13.89|2010-09-28T20:00:00Z,53.516566,8.188217,None,13.89|2010-09-28T20:10:00Z,53.516566,8.188217,None,13.9|2010-09-28T20:20:00Z,53.516566,8.188217,None,13.9|2010-09-28T20:30:00Z,53.516566,8.188217,None,13.98|2010-09-28T20:40:00Z,53.516566,8.188217,None,14.02|2010-09-28T20:50:00Z,53.516566,8.188217,None,14.03|2010-09-28T21:00:00Z,53.516566,8.188217,None,14.03|2010-09-28T21:10:00Z,53.516566,8.188217,None,13.99|2010-09-28T21:20:00Z,53.516566,8.188217,None,13.92|2010-09-28T21:30:00Z,53.516566,8.188217,None,13.72|2010-09-28T21:40:00Z,53.516566,8.188217,None,13.75|2010-09-28T21:50:00Z,53.516566,8.188217,None,13.93|2010-09-28T22:00:00Z,53.516566,8.188217,None,14.0|2010-09-28T22:10:00Z,53.516566,8.188217,None,14.02|2010-09-28T22:20:00Z,53.516566,8.188217,None,14.0|2010-09-28T22:30:00Z,53.516566,8.188217,None,13.93|2010-09-28T22:40:00Z,53.516566,8.188217,None,13.93|2010-09-28T22:50:00Z,53.516566,8.188217,None,13.95|2010-09-28T23:00:00Z,53.516566,8.188217,None,13.96|2010-09-28T23:10:00Z,53.516566,8.188217,None,13.99|2010-09-28T23:20:00Z,53.516566,8.188217,None,14.02|2010-09-28T23:30:00Z,53.516566,8.188217,None,14.01|2010-09-28T23:40:00Z,53.516566,8.188217,None,14.05|2010-09-28T23:50:00Z,53.516566,8.188217,None,14.07|2010-09-29T00:00:00Z,53.516566,8.188217,None,14.07|2010-09-29T00:10:00Z,53.516566,8.188217,None,14.12|2010-09-29T00:20:00Z,53.516566,8.188217,None,14.17|2010-09-29T00:30:00Z,53.516566,8.188217,None,14.2|2010-09-29T00:40:00Z,53.516566,8.188217,None,14.23|2010-09-29T00:50:00Z,53.516566,8.188217,None,14.23|2010-09-29T01:00:00Z,53.516566,8.188217,None,14.21|2010-09-29T01:10:00Z,53.516566,8.188217,None,14.22|2010-09-29T01:20:00Z,53.516566,8.188217,None,14.23|2010-09-29T01:30:00Z,53.516566,8.188217,None,14.24|2010-09-29T01:40:00Z,53.516566,8.188217,None,14.25|2010-09-29T01:50:00Z,53.516566,8.188217,None,14.26|2010-09-29T02:00:00Z,53.516566,8.188217,None,14.27|2010-09-29T02:10:00Z,53.516566,8.188217,None,14.29|2010-09-29T02:20:00Z,53.516566,8.188217,None,14.3|2010-09-29T02:30:00Z,53.516566,8.188217,None,14.32|2010-09-29T02:40:00Z,53.516566,8.188217,None,14.32|2010-09-29T02:50:00Z,53.516566,8.188217,None,14.32|2010-09-29T03:00:00Z,53.516566,8.188217,None,14.31|2010-09-29T03:10:00Z,53.516566,8.188217,None,14.28|2010-09-29T03:20:00Z,53.516566,8.188217,None,14.22|2010-09-29T03:30:00Z,53.516566,8.188217,None,14.15|2010-09-29T03:40:00Z,53.516566,8.188217,None,14.09|2010-09-29T03:50:00Z,53.516566,8.188217,None,14.02|2010-09-29T04:00:00Z,53.516566,8.188217,None,13.9|2010-09-29T04:10:00Z,53.516566,8.188217,None,13.77|2010-09-29T04:20:00Z,53.516566,8.188217,None,13.58|2010-09-29T04:30:00Z,53.516566,8.188217,None,13.44|2010-09-29T04:40:00Z,53.516566,8.188217,None,13.39|2010-09-29T04:50:00Z,53.516566,8.188217,None,13.32|2010-09-29T05:00:00Z,53.516566,8.188217,None,13.22|2010-09-29T05:10:00Z,53.516566,8.188217,None,13.23|2010-09-29T05:20:00Z,53.516566,8.188217,None,13.41|2010-09-29T05:30:00Z,53.516566,8.188217,None,13.48|2010-09-29T05:40:00Z,53.516566,8.188217,None,13.52|2010-09-29T05:50:00Z,53.516566,8.188217,None,13.57|2010-09-29T06:00:00Z,53.516566,8.188217,None,13.63|2010-09-29T06:10:00Z,53.516566,8.188217,None,13.69|2010-09-29T06:20:00Z,53.516566,8.188217,None,13.7|2010-09-29T06:30:00Z,53.516566,8.188217,None,13.69|2010-09-29T06:40:00Z,53.516566,8.188217,None,13.68|2010-09-29T06:50:00Z,53.516566,8.188217,None,13.67|2010-09-29T07:00:00Z,53.516566,8.188217,None,13.66|2010-09-29T07:10:00Z,53.516566,8.188217,None,13.67|2010-09-29T07:20:00Z,53.516566,8.188217,None,13.67|2010-09-29T07:30:00Z,53.516566,8.188217,None,13.67|2010-09-29T07:40:00Z,53.516566,8.188217,None,13.65|2010-09-29T07:50:00Z,53.516566,8.188217,None,13.55|2010-09-29T08:00:00Z,53.516566,8.188217,None,13.41|2010-09-29T08:10:00Z,53.516566,8.188217,None,13.36|2010-09-29T08:20:00Z,53.516566,8.188217,None,13.33|2010-09-29T08:30:00Z,53.516566,8.188217,None,13.32|2010-09-29T08:40:00Z,53.516566,8.188217,None,13.31|2010-09-29T08:50:00Z,53.516566,8.188217,None,13.33|2010-09-29T09:00:00Z,53.516566,8.188217,None,13.46|2010-09-29T09:10:00Z,53.516566,8.188217,None,13.54|2010-09-29T09:20:00Z,53.516566,8.188217,None,13.5|2010-09-29T09:30:00Z,53.516566,8.188217,None,13.42|2010-09-29T09:40:00Z,53.516566,8.188217,None,13.4|2010-09-29T09:50:00Z,53.516566,8.188217,None,13.42|2010-09-29T10:00:00Z,53.516566,8.188217,None,13.39|2010-09-29T10:10:00Z,53.516566,8.188217,None,13.43|2010-09-29T10:20:00Z,53.516566,8.188217,None,13.64|2010-09-29T10:30:00Z,53.516566,8.188217,None,13.85|2010-09-29T10:40:00Z,53.516566,8.188217,None,13.83|2010-09-29T10:50:00Z,53.516566,8.188217,None,13.66|2010-09-29T11:00:00Z,53.516566,8.188217,None,13.57|2010-09-29T11:10:00Z,53.516566,8.188217,None,13.68|2010-09-29T11:20:00Z,53.516566,8.188217,None,13.63|2010-09-29T11:30:00Z,53.516566,8.188217,None,13.64|2010-09-29T11:40:00Z,53.516566,8.188217,None,13.64|2010-09-29T11:50:00Z,53.516566,8.188217,None,13.66|2010-09-29T12:00:00Z,53.516566,8.188217,None,13.78|2010-09-29T12:10:00Z,53.516566,8.188217,None,13.82|2010-09-29T12:20:00Z,53.516566,8.188217,None,13.82|2010-09-29T12:30:00Z,53.516566,8.188217,None,13.83|2010-09-29T12:40:00Z,53.516566,8.188217,None,13.89|2010-09-29T12:50:00Z,53.516566,8.188217,None,13.95|2010-09-29T13:00:00Z,53.516566,8.188217,None,13.97|2010-09-29T13:10:00Z,53.516566,8.188217,None,13.99|2010-09-29T13:20:00Z,53.516566,8.188217,None,13.98|2010-09-29T13:30:00Z,53.516566,8.188217,None,13.98|2010-09-29T13:40:00Z,53.516566,8.188217,None,14.01|2010-09-29T13:50:00Z,53.516566,8.188217,None,14.05|2010-09-29T14:00:00Z,53.516566,8.188217,None,14.07|2010-09-29T14:10:00Z,53.516566,8.188217,None,14.06|2010-09-29T14:20:00Z,53.516566,8.188217,None,14.05|2010-09-29T14:30:00Z,53.516566,8.188217,None,14.05|2010-09-29T14:40:00Z,53.516566,8.188217,None,14.07|2010-09-29T14:50:00Z,53.516566,8.188217,None,14.09|2010-09-29T15:00:00Z,53.516566,8.188217,None,14.11|2010-09-29T15:10:00Z,53.516566,8.188217,None,14.12|2010-09-29T15:20:00Z,53.516566,8.188217,None,14.11|2010-09-29T15:30:00Z,53.516566,8.188217,None,14.07|2010-09-29T15:40:00Z,53.516566,8.188217,None,14.06|2010-09-29T15:50:00Z,53.516566,8.188217,None,14.05|2010-09-29T16:00:00Z,53.516566,8.188217,None,13.98|2010-09-29T16:10:00Z,53.516566,8.188217,None,13.95|2010-09-29T16:20:00Z,53.516566,8.188217,None,13.91|2010-09-29T16:30:00Z,53.516566,8.188217,None,13.85|2010-09-29T16:40:00Z,53.516566,8.188217,None,13.93|2010-09-29T16:50:00Z,53.516566,8.188217,None,13.85|2010-09-29T17:00:00Z,53.516566,8.188217,None,13.74|2010-09-29T17:10:00Z,53.516566,8.188217,None,13.72|2010-09-29T17:20:00Z,53.516566,8.188217,None,13.76|2010-09-29T17:30:00Z,53.516566,8.188217,None,13.82|2010-09-29T17:40:00Z,53.516566,8.188217,None,13.87|2010-09-29T17:50:00Z,53.516566,8.188217,None,13.85|2010-09-29T18:00:00Z,53.516566,8.188217,None,13.63|2010-09-29T18:10:00Z,53.516566,8.188217,None,13.55|2010-09-29T18:20:00Z,53.516566,8.188217,None,13.52|2010-09-29T18:30:00Z,53.516566,8.188217,None,13.51|2010-09-29T18:40:00Z,53.516566,8.188217,None,13.5|2010-09-29T18:50:00Z,53.516566,8.188217,None,13.49|2010-09-29T19:00:00Z,53.516566,8.188217,None,13.48|2010-09-29T19:10:00Z,53.516566,8.188217,None,13.48|2010-09-29T19:20:00Z,53.516566,8.188217,None,13.49|2010-09-29T19:30:00Z,53.516566,8.188217,None,13.5|2010-09-29T19:40:00Z,53.516566,8.188217,None,13.5|2010-09-29T19:50:00Z,53.516566,8.188217,None,13.52|2010-09-29T20:00:00Z,53.516566,8.188217,None,13.38|2010-09-29T20:10:00Z,53.516566,8.188217,None,13.6|2010-09-29T20:20:00Z,53.516566,8.188217,None,13.75|2010-09-29T20:30:00Z,53.516566,8.188217,None,13.73|2010-09-29T20:40:00Z,53.516566,8.188217,None,13.76|2010-09-29T20:50:00Z,53.516566,8.188217,None,13.76|2010-09-29T21:00:00Z,53.516566,8.188217,None,13.77|2010-09-29T21:10:00Z,53.516566,8.188217,None,13.78|2010-09-29T21:20:00Z,53.516566,8.188217,None,13.73|2010-09-29T21:30:00Z,53.516566,8.188217,None,13.67|2010-09-29T21:40:00Z,53.516566,8.188217,None,13.63|2010-09-29T21:50:00Z,53.516566,8.188217,None,13.62|2010-09-29T22:00:00Z,53.516566,8.188217,None,13.54|2010-09-29T22:10:00Z,53.516566,8.188217,None,13.32|2010-09-29T22:20:00Z,53.516566,8.188217,None,13.49|2010-09-29T22:30:00Z,53.516566,8.188217,None,13.6|2010-09-29T22:40:00Z,53.516566,8.188217,None,13.7|2010-09-29T22:50:00Z,53.516566,8.188217,None,13.52|2010-09-29T23:00:00Z,53.516566,8.188217,None,13.3|2010-09-29T23:10:00Z,53.516566,8.188217,None,13.33|2010-09-29T23:20:00Z,53.516566,8.188217,None,13.44|2010-09-29T23:30:00Z,53.516566,8.188217,None,13.51|2010-09-29T23:40:00Z,53.516566,8.188217,None,13.48|2010-09-29T23:50:00Z,53.516566,8.188217,None,13.51|2010-09-30T00:00:00Z,53.516566,8.188217,None,13.54|2010-09-30T00:10:00Z,53.516566,8.188217,None,13.58|2010-09-30T00:20:00Z,53.516566,8.188217,None,13.6|2010-09-30T00:30:00Z,53.516566,8.188217,None,13.62|2010-09-30T00:40:00Z,53.516566,8.188217,None,13.63|2010-09-30T00:50:00Z,53.516566,8.188217,None,13.65|2010-09-30T01:00:00Z,53.516566,8.188217,None,13.68|2010-09-30T01:10:00Z,53.516566,8.188217,None,13.75|2010-09-30T01:20:00Z,53.516566,8.188217,None,13.81|2010-09-30T01:30:00Z,53.516566,8.188217,None,13.88|2010-09-30T01:40:00Z,53.516566,8.188217,None,13.9|2010-09-30T01:50:00Z,53.516566,8.188217,None,13.91|2010-09-30T02:00:00Z,53.516566,8.188217,None,13.91|2010-09-30T02:10:00Z,53.516566,8.188217,None,13.91|2010-09-30T02:20:00Z,53.516566,8.188217,None,13.91|2010-09-30T02:30:00Z,53.516566,8.188217,None,13.92|2010-09-30T02:40:00Z,53.516566,8.188217,None,13.92|2010-09-30T02:50:00Z,53.516566,8.188217,None,13.91|2010-09-30T03:00:00Z,53.516566,8.188217,None,13.9|2010-09-30T03:10:00Z,53.516566,8.188217,None,13.89|2010-09-30T03:20:00Z,53.516566,8.188217,None,13.87|2010-09-30T03:30:00Z,53.516566,8.188217,None,13.81|2010-09-30T03:40:00Z,53.516566,8.188217,None,13.75|2010-09-30T03:50:00Z,53.516566,8.188217,None,13.7|2010-09-30T04:00:00Z,53.516566,8.188217,None,13.62|2010-09-30T04:10:00Z,53.516566,8.188217,None,13.41|2010-09-30T04:20:00Z,53.516566,8.188217,None,13.21|2010-09-30T04:30:00Z,53.516566,8.188217,None,13.0|2010-09-30T04:40:00Z,53.516566,8.188217,None,12.98|2010-09-30T04:50:00Z,53.516566,8.188217,None,13.09|2010-09-30T05:00:00Z,53.516566,8.188217,None,13.0|2010-09-30T05:10:00Z,53.516566,8.188217,None,12.87|2010-09-30T05:20:00Z,53.516566,8.188217,None,12.79|2010-09-30T05:30:00Z,53.516566,8.188217,None,12.69|2010-09-30T05:40:00Z,53.516566,8.188217,None,12.59|2010-09-30T05:50:00Z,53.516566,8.188217,None,12.61|2010-09-30T06:00:00Z,53.516566,8.188217,None,12.82|2010-09-30T06:10:00Z,53.516566,8.188217,None,12.91|2010-09-30T06:20:00Z,53.516566,8.188217,None,12.92|2010-09-30T06:30:00Z,53.516566,8.188217,None,12.9|2010-09-30T06:40:00Z,53.516566,8.188217,None,12.89|2010-09-30T06:50:00Z,53.516566,8.188217,None,12.87|2010-09-30T07:00:00Z,53.516566,8.188217,None,12.85|2010-09-30T07:10:00Z,53.516566,8.188217,None,12.85|2010-09-30T07:20:00Z,53.516566,8.188217,None,12.84|2010-09-30T07:30:00Z,53.516566,8.188217,None,12.84|2010-09-30T07:40:00Z,53.516566,8.188217,None,12.82|2010-09-30T07:50:00Z,53.516566,8.188217,None,12.84|2010-09-30T08:00:00Z,53.516566,8.188217,None,12.82|2010-09-30T08:10:00Z,53.516566,8.188217,None,12.84|2010-09-30T08:20:00Z,53.516566,8.188217,None,12.77|2010-09-30T08:30:00Z,53.516566,8.188217,None,12.36|2010-09-30T08:40:00Z,53.516566,8.188217,None,12.4|2010-09-30T08:50:00Z,53.516566,8.188217,None,12.4|2010-09-30T09:00:00Z,53.516566,8.188217,None,12.3|2010-09-30T09:10:00Z,53.516566,8.188217,None,12.28|2010-09-30T09:20:00Z,53.516566,8.188217,None,12.38|2010-09-30T09:30:00Z,53.516566,8.188217,None,12.47|2010-09-30T09:40:00Z,53.516566,8.188217,None,12.6|2010-09-30T09:50:00Z,53.516566,8.188217,None,12.71|2010-09-30T10:00:00Z,53.516566,8.188217,None,12.71|2010-09-30T10:10:00Z,53.516566,8.188217,None,12.69|2010-09-30T10:20:00Z,53.516566,8.188217,None,12.66|2010-09-30T10:30:00Z,53.516566,8.188217,None,12.72|2010-09-30T10:40:00Z,53.516566,8.188217,None,12.77|2010-09-30T10:50:00Z,53.516566,8.188217,None,12.89|2010-09-30T11:00:00Z,53.516566,8.188217,None,13.14|2010-09-30T11:10:00Z,53.516566,8.188217,None,13.39|2010-09-30T11:20:00Z,53.516566,8.188217,None,13.38|2010-09-30T11:30:00Z,53.516566,8.188217,None,13.34|2010-09-30T11:40:00Z,53.516566,8.188217,None,13.31|2010-09-30T11:50:00Z,53.516566,8.188217,None,13.26|2010-09-30T12:00:00Z,53.516566,8.188217,None,13.16|2010-09-30T12:10:00Z,53.516566,8.188217,None,13.16|2010-09-30T12:20:00Z,53.516566,8.188217,None,13.16|2010-09-30T12:30:00Z,53.516566,8.188217,None,13.2|2010-09-30T12:40:00Z,53.516566,8.188217,None,13.27|2010-09-30T12:50:00Z,53.516566,8.188217,None,13.35|2010-09-30T13:00:00Z,53.516566,8.188217,None,13.36|2010-09-30T13:10:00Z,53.516566,8.188217,None,13.37|2010-09-30T13:20:00Z,53.516566,8.188217,None,13.4|2010-09-30T13:30:00Z,53.516566,8.188217,None,13.45|2010-09-30T13:40:00Z,53.516566,8.188217,None,13.54|2010-09-30T13:50:00Z,53.516566,8.188217,None,13.61|2010-09-30T14:00:00Z,53.516566,8.188217,None,13.67|2010-09-30T14:10:00Z,53.516566,8.188217,None,13.7|2010-09-30T14:20:00Z,53.516566,8.188217,None,13.72|2010-09-30T14:30:00Z,53.516566,8.188217,None,13.71|2010-09-30T14:40:00Z,53.516566,8.188217,None,13.7|2010-09-30T14:50:00Z,53.516566,8.188217,None,13.71|2010-09-30T15:00:00Z,53.516566,8.188217,None,13.72|2010-09-30T15:10:00Z,53.516566,8.188217,None,13.73|2010-09-30T15:20:00Z,53.516566,8.188217,None,13.74|2010-09-30T15:30:00Z,53.516566,8.188217,None,13.74|2010-09-30T15:40:00Z,53.516566,8.188217,None,13.73|2010-09-30T15:50:00Z,53.516566,8.188217,None,13.72|2010-09-30T16:00:00Z,53.516566,8.188217,None,13.7|2010-09-30T16:10:00Z,53.516566,8.188217,None,13.65|2010-09-30T16:20:00Z,53.516566,8.188217,None,13.62|2010-09-30T16:30:00Z,53.516566,8.188217,None,13.63|2010-09-30T16:40:00Z,53.516566,8.188217,None,13.5|2010-09-30T16:50:00Z,53.516566,8.188217,None,13.34|2010-09-30T17:00:00Z,53.516566,8.188217,None,13.18|2010-09-30T17:10:00Z,53.516566,8.188217,None,13.09|2010-09-30T17:20:00Z,53.516566,8.188217,None,13.01|2010-09-30T17:30:00Z,53.516566,8.188217,None,12.97|2010-09-30T17:40:00Z,53.516566,8.188217,None,12.95|2010-09-30T17:50:00Z,53.516566,8.188217,None,13.07|2010-09-30T18:00:00Z,53.516566,8.188217,None,13.24|2010-09-30T18:10:00Z,53.516566,8.188217,None,13.42|2010-09-30T18:20:00Z,53.516566,8.188217,None,13.3|2010-09-30T18:30:00Z,53.516566,8.188217,None,13.11|2010-09-30T18:40:00Z,53.516566,8.188217,None,13.05|2010-09-30T18:50:00Z,53.516566,8.188217,None,13.02|2010-09-30T19:00:00Z,53.516566,8.188217,None,12.99|2010-09-30T19:10:00Z,53.516566,8.188217,None,12.97|2010-09-30T19:20:00Z,53.516566,8.188217,None,12.95|2010-09-30T19:30:00Z,53.516566,8.188217,None,12.95|2010-09-30T19:40:00Z,53.516566,8.188217,None,12.94|2010-09-30T19:50:00Z,53.516566,8.188217,None,12.94|2010-09-30T20:00:00Z,53.516566,8.188217,None,12.93|2010-09-30T20:10:00Z,53.516566,8.188217,None,12.93|2010-09-30T20:20:00Z,53.516566,8.188217,None,12.95|2010-09-30T20:30:00Z,53.516566,8.188217,None,13.07|2010-09-30T20:40:00Z,53.516566,8.188217,None,12.83|2010-09-30T20:50:00Z,53.516566,8.188217,None,12.78|2010-09-30T21:00:00Z,53.516566,8.188217,None,12.8|2010-09-30T21:10:00Z,53.516566,8.188217,None,12.79|2010-09-30T21:20:00Z,53.516566,8.188217,None,12.92|2010-09-30T21:30:00Z,53.516566,8.188217,None,12.87|2010-09-30T21:40:00Z,53.516566,8.188217,None,12.89|2010-09-30T21:50:00Z,53.516566,8.188217,None,12.99|2010-09-30T22:00:00Z,53.516566,8.188217,None,12.99|2010-09-30T22:10:00Z,53.516566,8.188217,None,12.9|2010-09-30T22:20:00Z,53.516566,8.188217,None,12.85|2010-09-30T22:30:00Z,53.516566,8.188217,None,12.65|2010-09-30T22:40:00Z,53.516566,8.188217,None,12.52|2010-09-30T22:50:00Z,53.516566,8.188217,None,12.64|2010-09-30T23:00:00Z,53.516566,8.188217,None,12.95|2010-09-30T23:10:00Z,53.516566,8.188217,None,13.11|2010-09-30T23:20:00Z,53.516566,8.188217,None,13.2|2010-09-30T23:30:00Z,53.516566,8.188217,None,13.2|2010-09-30T23:40:00Z,53.516566,8.188217,None,13.19|2010-09-30T23:50:00Z,53.516566,8.188217,None,13.03|2010-10-01T00:00:00Z,53.516566,8.188217,None,12.98|2010-10-01T00:10:00Z,53.516566,8.188217,None,13.05|2010-10-01T00:20:00Z,53.516566,8.188217,None,13.0|2010-10-01T00:30:00Z,53.516566,8.188217,None,12.93|2010-10-01T00:40:00Z,53.516566,8.188217,None,12.98|2010-10-01T00:50:00Z,53.516566,8.188217,None,13.05|2010-10-01T01:00:00Z,53.516566,8.188217,None,13.14|2010-10-01T01:10:00Z,53.516566,8.188217,None,13.15|2010-10-01T01:20:00Z,53.516566,8.188217,None,13.15|2010-10-01T01:30:00Z,53.516566,8.188217,None,13.2|2010-10-01T01:40:00Z,53.516566,8.188217,None,13.32|2010-10-01T01:50:00Z,53.516566,8.188217,None,13.41|2010-10-01T02:00:00Z,53.516566,8.188217,None,13.47|2010-10-01T02:10:00Z,53.516566,8.188217,None,13.52|2010-10-01T02:20:00Z,53.516566,8.188217,None,13.56|2010-10-01T02:30:00Z,53.516566,8.188217,None,13.57|2010-10-01T02:40:00Z,53.516566,8.188217,None,13.59|2010-10-01T02:50:00Z,53.516566,8.188217,None,13.62|2010-10-01T03:00:00Z,53.516566,8.188217,None,13.63|2010-10-01T03:10:00Z,53.516566,8.188217,None,13.65|2010-10-01T03:20:00Z,53.516566,8.188217,None,13.65|2010-10-01T03:30:00Z,53.516566,8.188217,None,13.67|2010-10-01T03:40:00Z,53.516566,8.188217,None,13.67|2010-10-01T03:50:00Z,53.516566,8.188217,None,13.66|2010-10-01T04:00:00Z,53.516566,8.188217,None,13.66|2010-10-01T04:10:00Z,53.516566,8.188217,None,13.63|2010-10-01T04:20:00Z,53.516566,8.188217,None,13.55|2010-10-01T04:30:00Z,53.516566,8.188217,None,13.48|2010-10-01T04:40:00Z,53.516566,8.188217,None,13.4|2010-10-01T04:50:00Z,53.516566,8.188217,None,13.29|2010-10-01T05:00:00Z,53.516566,8.188217,None,13.04|2010-10-01T05:10:00Z,53.516566,8.188217,None,12.78|2010-10-01T05:20:00Z,53.516566,8.188217,None,12.45|2010-10-01T05:30:00Z,53.516566,8.188217,None,12.48|2010-10-01T05:40:00Z,53.516566,8.188217,None,12.45|2010-10-01T05:50:00Z,53.516566,8.188217,None,12.38|2010-10-01T06:00:00Z,53.516566,8.188217,None,12.28|2010-10-01T06:10:00Z,53.516566,8.188217,None,12.18|2010-10-01T06:20:00Z,53.516566,8.188217,None,12.02|2010-10-01T06:30:00Z,53.516566,8.188217,None,11.78|2010-10-01T06:40:00Z,53.516566,8.188217,None,11.91|2010-10-01T06:50:00Z,53.516566,8.188217,None,12.26|2010-10-01T07:00:00Z,53.516566,8.188217,None,12.51|2010-10-01T07:10:00Z,53.516566,8.188217,None,12.66|2010-10-01T07:20:00Z,53.516566,8.188217,None,12.67|2010-10-01T07:30:00Z,53.516566,8.188217,None,12.65|2010-10-01T07:40:00Z,53.516566,8.188217,None,12.62|2010-10-01T07:50:00Z,53.516566,8.188217,None,12.61|2010-10-01T08:00:00Z,53.516566,8.188217,None,12.6|2010-10-01T08:10:00Z,53.516566,8.188217,None,12.61|2010-10-01T08:20:00Z,53.516566,8.188217,None,12.61|2010-10-01T08:30:00Z,53.516566,8.188217,None,12.64|2010-10-01T08:40:00Z,53.516566,8.188217,None,12.66|2010-10-01T08:50:00Z,53.516566,8.188217,None,12.65|2010-10-01T09:00:00Z,53.516566,8.188217,None,12.62|2010-10-01T09:10:00Z,53.516566,8.188217,None,12.61|2010-10-01T09:20:00Z,53.516566,8.188217,None,12.65|2010-10-01T09:30:00Z,53.516566,8.188217,None,12.67|2010-10-01T09:40:00Z,53.516566,8.188217,None,12.59|2010-10-01T09:50:00Z,53.516566,8.188217,None,12.37|2010-10-01T10:00:00Z,53.516566,8.188217,None,12.11|2010-10-01T10:10:00Z,53.516566,8.188217,None,12.13|2010-10-01T10:20:00Z,53.516566,8.188217,None,12.2|2010-10-01T10:30:00Z,53.516566,8.188217,None,12.22|2010-10-01T10:40:00Z,53.516566,8.188217,None,12.28|2010-10-01T10:50:00Z,53.516566,8.188217,None,12.36|2010-10-01T11:00:00Z,53.516566,8.188217,None,12.48|2010-10-01T11:10:00Z,53.516566,8.188217,None,12.46|2010-10-01T11:20:00Z,53.516566,8.188217,None,12.29|2010-10-01T11:30:00Z,53.516566,8.188217,None,12.16|2010-10-01T11:40:00Z,53.516566,8.188217,None,12.68|2010-10-01T11:50:00Z,53.516566,8.188217,None,12.94|2010-10-01T12:00:00Z,53.516566,8.188217,None,12.92|2010-10-01T12:10:00Z,53.516566,8.188217,None,12.94|2010-10-01T12:20:00Z,53.516566,8.188217,None,12.75|2010-10-01T12:30:00Z,53.516566,8.188217,None,12.74|2010-10-01T12:40:00Z,53.516566,8.188217,None,12.64|2010-10-01T12:50:00Z,53.516566,8.188217,None,12.48|2010-10-01T13:00:00Z,53.516566,8.188217,None,12.55|2010-10-01T13:10:00Z,53.516566,8.188217,None,12.59|2010-10-01T13:20:00Z,53.516566,8.188217,None,12.72|2010-10-01T13:30:00Z,53.516566,8.188217,None,12.78|2010-10-01T13:40:00Z,53.516566,8.188217,None,12.86|2010-10-01T13:50:00Z,53.516566,8.188217,None,12.87|2010-10-01T14:00:00Z,53.516566,8.188217,None,12.88|2010-10-01T14:10:00Z,53.516566,8.188217,None,12.88|2010-10-01T14:20:00Z,53.516566,8.188217,None,12.91|2010-10-01T14:30:00Z,53.516566,8.188217,None,13.03|2010-10-01T14:40:00Z,53.516566,8.188217,None,13.1|2010-10-01T14:50:00Z,53.516566,8.188217,None,13.14|2010-10-01T15:00:00Z,53.516566,8.188217,None,13.2|2010-10-01T15:10:00Z,53.516566,8.188217,None,13.24|2010-10-01T15:20:00Z,53.516566,8.188217,None,13.27|2010-10-01T15:30:00Z,53.516566,8.188217,None,13.29|2010-10-01T15:40:00Z,53.516566,8.188217,None,13.3|2010-10-01T15:50:00Z,53.516566,8.188217,None,13.29|2010-10-01T16:00:00Z,53.516566,8.188217,None,13.28|2010-10-01T16:10:00Z,53.516566,8.188217,None,13.27|2010-10-01T16:20:00Z,53.516566,8.188217,None,13.25|2010-10-01T16:30:00Z,53.516566,8.188217,None,13.2|2010-10-01T16:40:00Z,53.516566,8.188217,None,13.11|2010-10-01T16:50:00Z,53.516566,8.188217,None,12.97|2010-10-01T17:00:00Z,53.516566,8.188217,None,12.81|2010-10-01T17:10:00Z,53.516566,8.188217,None,12.58|2010-10-01T17:20:00Z,53.516566,8.188217,None,12.53|2010-10-01T17:30:00Z,53.516566,8.188217,None,12.42|2010-10-01T17:40:00Z,53.516566,8.188217,None,12.42|2010-10-01T17:50:00Z,53.516566,8.188217,None,12.46|2010-10-01T18:00:00Z,53.516566,8.188217,None,12.4|2010-10-01T18:10:00Z,53.516566,8.188217,None,12.35|2010-10-01T18:20:00Z,53.516566,8.188217,None,12.35|2010-10-01T18:30:00Z,53.516566,8.188217,None,12.45|2010-10-01T18:40:00Z,53.516566,8.188217,None,12.48|2010-10-01T18:50:00Z,53.516566,8.188217,None,12.48|2010-10-01T19:00:00Z,53.516566,8.188217,None,12.48|2010-10-01T19:10:00Z,53.516566,8.188217,None,12.48|2010-10-01T19:20:00Z,53.516566,8.188217,None,12.45|2010-10-01T19:30:00Z,53.516566,8.188217,None,12.43|2010-10-01T19:40:00Z,53.516566,8.188217,None,12.42|2010-10-01T19:50:00Z,53.516566,8.188217,None,12.41|2010-10-01T20:00:00Z,53.516566,8.188217,None,12.4|2010-10-01T20:10:00Z,53.516566,8.188217,None,12.39|2010-10-01T20:20:00Z,53.516566,8.188217,None,12.36|2010-10-01T20:30:00Z,53.516566,8.188217,None,12.35|2010-10-01T20:40:00Z,53.516566,8.188217,None,12.36|2010-10-01T20:50:00Z,53.516566,8.188217,None,12.36|2010-10-01T21:00:00Z,53.516566,8.188217,None,12.37|2010-10-01T21:10:00Z,53.516566,8.188217,None,12.37|2010-10-01T21:20:00Z,53.516566,8.188217,None,12.37|2010-10-01T21:30:00Z,53.516566,8.188217,None,12.43|2010-10-01T21:40:00Z,53.516566,8.188217,None,12.43|2010-10-01T21:50:00Z,53.516566,8.188217,None,12.29|2010-10-01T22:00:00Z,53.516566,8.188217,None,12.14|2010-10-01T22:10:00Z,53.516566,8.188217,None,12.22|2010-10-01T22:20:00Z,53.516566,8.188217,None,12.22|2010-10-01T22:30:00Z,53.516566,8.188217,None,12.18|2010-10-01T22:40:00Z,53.516566,8.188217,None,12.18|2010-10-01T22:50:00Z,53.516566,8.188217,None,12.22|2010-10-01T23:00:00Z,53.516566,8.188217,None,12.21|2010-10-01T23:10:00Z,53.516566,8.188217,None,12.18|2010-10-01T23:20:00Z,53.516566,8.188217,None,12.07|2010-10-01T23:30:00Z,53.516566,8.188217,None,11.91|2010-10-01T23:40:00Z,53.516566,8.188217,None,11.86|2010-10-01T23:50:00Z,53.516566,8.188217,None,11.99|2010-10-02T00:00:00Z,53.516566,8.188217,None,12.31|2010-10-02T00:10:00Z,53.516566,8.188217,None,12.59|2010-10-02T00:20:00Z,53.516566,8.188217,None,12.63|2010-10-02T00:30:00Z,53.516566,8.188217,None,12.6|2010-10-02T00:40:00Z,53.516566,8.188217,None,12.62|2010-10-02T00:50:00Z,53.516566,8.188217,None,12.64|2010-10-02T01:00:00Z,53.516566,8.188217,None,12.53|2010-10-02T01:10:00Z,53.516566,8.188217,None,12.36|2010-10-02T01:20:00Z,53.516566,8.188217,None,12.3|2010-10-02T01:30:00Z,53.516566,8.188217,None,12.36|2010-10-02T01:40:00Z,53.516566,8.188217,None,12.31|2010-10-02T01:50:00Z,53.516566,8.188217,None,12.36|2010-10-02T02:00:00Z,53.516566,8.188217,None,12.41|2010-10-02T02:10:00Z,53.516566,8.188217,None,12.48|2010-10-02T02:20:00Z,53.516566,8.188217,None,12.52|2010-10-02T02:30:00Z,53.516566,8.188217,None,12.52|2010-10-02T02:40:00Z,53.516566,8.188217,None,12.53|2010-10-02T02:50:00Z,53.516566,8.188217,None,12.58|2010-10-02T03:00:00Z,53.516566,8.188217,None,12.59|2010-10-02T03:10:00Z,53.516566,8.188217,None,12.59|2010-10-02T03:20:00Z,53.516566,8.188217,None,12.59|2010-10-02T03:30:00Z,53.516566,8.188217,None,12.62|2010-10-02T03:40:00Z,53.516566,8.188217,None,12.68|2010-10-02T03:50:00Z,53.516566,8.188217,None,12.73|2010-10-02T04:00:00Z,53.516566,8.188217,None,12.74|2010-10-02T04:10:00Z,53.516566,8.188217,None,12.73|2010-10-02T04:20:00Z,53.516566,8.188217,None,12.71|2010-10-02T04:30:00Z,53.516566,8.188217,None,12.55|2010-10-02T04:40:00Z,53.516566,8.188217,None,12.11|2010-10-02T04:50:00Z,53.516566,8.188217,None,11.84|2010-10-02T05:00:00Z,53.516566,8.188217,None,11.83|2010-10-02T05:10:00Z,53.516566,8.188217,None,11.81|2010-10-02T05:20:00Z,53.516566,8.188217,None,11.8|2010-10-02T05:30:00Z,53.516566,8.188217,None,11.86|2010-10-02T05:40:00Z,53.516566,8.188217,None,11.73|2010-10-02T05:50:00Z,53.516566,8.188217,None,11.32|2010-10-02T06:00:00Z,53.516566,8.188217,None,11.62|2010-10-02T06:10:00Z,53.516566,8.188217,None,11.56|2010-10-02T06:20:00Z,53.516566,8.188217,None,10.98|2010-10-02T06:30:00Z,53.516566,8.188217,None,10.78|2010-10-02T06:40:00Z,53.516566,8.188217,None,11.12|2010-10-02T06:50:00Z,53.516566,8.188217,None,11.39|2010-10-02T07:00:00Z,53.516566,8.188217,None,11.16|2010-10-02T07:10:00Z,53.516566,8.188217,None,11.19|2010-10-02T07:20:00Z,53.516566,8.188217,None,11.41|2010-10-02T07:30:00Z,53.516566,8.188217,None,11.56|2010-10-02T07:40:00Z,53.516566,8.188217,None,11.72|2010-10-02T07:50:00Z,53.516566,8.188217,None,11.91|2010-10-02T08:00:00Z,53.516566,8.188217,None,12.03|2010-10-02T08:10:00Z,53.516566,8.188217,None,12.04|2010-10-02T08:20:00Z,53.516566,8.188217,None,12.02|2010-10-02T08:30:00Z,53.516566,8.188217,None,12.02|2010-10-02T08:40:00Z,53.516566,8.188217,None,12.05|2010-10-02T08:50:00Z,53.516566,8.188217,None,12.08|2010-10-02T09:00:00Z,53.516566,8.188217,None,12.1|2010-10-02T09:10:00Z,53.516566,8.188217,None,12.1|2010-10-02T09:20:00Z,53.516566,8.188217,None,12.09|2010-10-02T09:30:00Z,53.516566,8.188217,None,12.1|2010-10-02T09:40:00Z,53.516566,8.188217,None,12.1|2010-10-02T09:50:00Z,53.516566,8.188217,None,12.11|2010-10-02T10:00:00Z,53.516566,8.188217,None,12.13|2010-10-02T10:10:00Z,53.516566,8.188217,None,12.11|2010-10-02T10:20:00Z,53.516566,8.188217,None,12.09|2010-10-02T10:30:00Z,53.516566,8.188217,None,12.08|2010-10-02T10:40:00Z,53.516566,8.188217,None,12.06|2010-10-02T10:50:00Z,53.516566,8.188217,None,12.12|2010-10-02T11:00:00Z,53.516566,8.188217,None,12.16|2010-10-02T11:10:00Z,53.516566,8.188217,None,12.14|2010-10-02T11:20:00Z,53.516566,8.188217,None,12.09|2010-10-02T11:30:00Z,53.516566,8.188217,None,11.97|2010-10-02T11:40:00Z,53.516566,8.188217,None,11.9|2010-10-02T11:50:00Z,53.516566,8.188217,None,11.96|2010-10-02T12:00:00Z,53.516566,8.188217,None,12.12|2010-10-02T12:10:00Z,53.516566,8.188217,None,12.12|2010-10-02T12:20:00Z,53.516566,8.188217,None,12.25|2010-10-02T12:30:00Z,53.516566,8.188217,None,12.23|2010-10-02T12:40:00Z,53.516566,8.188217,None,12.02|2010-10-02T12:50:00Z,53.516566,8.188217,None,12.02|2010-10-02T13:00:00Z,53.516566,8.188217,None,12.26|2010-10-02T13:10:00Z,53.516566,8.188217,None,12.52|2010-10-02T13:20:00Z,53.516566,8.188217,None,12.6|2010-10-02T13:30:00Z,53.516566,8.188217,None,12.52|2010-10-02T13:40:00Z,53.516566,8.188217,None,12.37|2010-10-02T13:50:00Z,53.516566,8.188217,None,12.15|2010-10-02T14:00:00Z,53.516566,8.188217,None,12.05|2010-10-02T14:10:00Z,53.516566,8.188217,None,12.12|2010-10-02T14:20:00Z,53.516566,8.188217,None,12.19|2010-10-02T14:30:00Z,53.516566,8.188217,None,12.38|2010-10-02T14:40:00Z,53.516566,8.188217,None,12.52|2010-10-02T14:50:00Z,53.516566,8.188217,None,12.51|2010-10-02T15:00:00Z,53.516566,8.188217,None,12.46|2010-10-02T15:10:00Z,53.516566,8.188217,None,12.45|2010-10-02T15:20:00Z,53.516566,8.188217,None,12.48|2010-10-02T15:30:00Z,53.516566,8.188217,None,12.52|2010-10-02T15:40:00Z,53.516566,8.188217,None,12.56|2010-10-02T15:50:00Z,53.516566,8.188217,None,12.65|2010-10-02T16:00:00Z,53.516566,8.188217,None,12.71|2010-10-02T16:10:00Z,53.516566,8.188217,None,12.8|2010-10-02T16:20:00Z,53.516566,8.188217,None,12.89|2010-10-02T16:30:00Z,53.516566,8.188217,None,12.97|2010-10-02T16:40:00Z,53.516566,8.188217,None,13.03|2010-10-02T16:50:00Z,53.516566,8.188217,None,13.09|2010-10-02T17:00:00Z,53.516566,8.188217,None,13.12|2010-10-02T17:10:00Z,53.516566,8.188217,None,13.13|2010-10-02T17:20:00Z,53.516566,8.188217,None,13.14|2010-10-02T17:30:00Z,53.516566,8.188217,None,13.14|2010-10-02T17:40:00Z,53.516566,8.188217,None,13.15|2010-10-02T17:50:00Z,53.516566,8.188217,None,13.14|2010-10-02T18:00:00Z,53.516566,8.188217,None,13.11|2010-10-02T18:10:00Z,53.516566,8.188217,None,13.08|2010-10-02T18:20:00Z,53.516566,8.188217,None,13.03|2010-10-02T18:30:00Z,53.516566,8.188217,None,12.88|2010-10-02T18:40:00Z,53.516566,8.188217,None,12.68|2010-10-02T18:50:00Z,53.516566,8.188217,None,12.52|2010-10-02T19:00:00Z,53.516566,8.188217,None,12.44|2010-10-02T19:10:00Z,53.516566,8.188217,None,12.43|2010-10-02T19:20:00Z,53.516566,8.188217,None,12.25|2010-10-02T19:30:00Z,53.516566,8.188217,None,12.3|2010-10-02T19:40:00Z,53.516566,8.188217,None,12.27|2010-10-02T19:50:00Z,53.516566,8.188217,None,12.27|2010-10-02T20:00:00Z,53.516566,8.188217,None,12.41|2010-10-02T20:10:00Z,53.516566,8.188217,None,12.54|2010-10-02T20:20:00Z,53.516566,8.188217,None,12.56|2010-10-02T20:30:00Z,53.516566,8.188217,None,12.46|2010-10-02T20:40:00Z,53.516566,8.188217,None,12.3|2010-10-02T20:50:00Z,53.516566,8.188217,None,12.25|2010-10-02T21:00:00Z,53.516566,8.188217,None,12.28|2010-10-02T21:10:00Z,53.516566,8.188217,None,12.33|2010-10-02T21:20:00Z,53.516566,8.188217,None,12.31|2010-10-02T21:30:00Z,53.516566,8.188217,None,12.24|2010-10-02T21:40:00Z,53.516566,8.188217,None,12.2|2010-10-02T21:50:00Z,53.516566,8.188217,None,12.2|2010-10-02T22:00:00Z,53.516566,8.188217,None,12.21|2010-10-02T22:10:00Z,53.516566,8.188217,None,12.21|2010-10-02T22:20:00Z,53.516566,8.188217,None,12.21|2010-10-02T22:30:00Z,53.516566,8.188217,None,12.22|2010-10-02T22:40:00Z,53.516566,8.188217,None,12.27|2010-10-02T22:50:00Z,53.516566,8.188217,None,12.31|2010-10-02T23:00:00Z,53.516566,8.188217,None,12.43|2010-10-02T23:10:00Z,53.516566,8.188217,None,12.52|2010-10-02T23:20:00Z,53.516566,8.188217,None,12.55|2010-10-02T23:30:00Z,53.516566,8.188217,None,12.4|2010-10-02T23:40:00Z,53.516566,8.188217,None,12.24|2010-10-02T23:50:00Z,53.516566,8.188217,None,12.32|2010-10-03T00:00:00Z,53.516566,8.188217,None,12.29|2010-10-03T00:10:00Z,53.516566,8.188217,None,12.33|2010-10-03T00:20:00Z,53.516566,8.188217,None,12.5|2010-10-03T00:30:00Z,53.516566,8.188217,None,12.43|2010-10-03T00:40:00Z,53.516566,8.188217,None,12.34|2010-10-03T00:50:00Z,53.516566,8.188217,None,12.31|2010-10-03T01:00:00Z,53.516566,8.188217,None,12.22|2010-10-03T01:10:00Z,53.516566,8.188217,None,12.34|2010-10-03T01:20:00Z,53.516566,8.188217,None,12.56|2010-10-03T01:30:00Z,53.516566,8.188217,None,12.58|2010-10-03T01:40:00Z,53.516566,8.188217,None,12.57|2010-10-03T01:50:00Z,53.516566,8.188217,None,12.49|2010-10-03T02:00:00Z,53.516566,8.188217,None,12.4|2010-10-03T02:10:00Z,53.516566,8.188217,None,12.4|2010-10-03T02:20:00Z,53.516566,8.188217,None,12.42|2010-10-03T02:30:00Z,53.516566,8.188217,None,12.32|2010-10-03T02:40:00Z,53.516566,8.188217,None,12.25|2010-10-03T02:50:00Z,53.516566,8.188217,None,12.3|2010-10-03T03:00:00Z,53.516566,8.188217,None,12.43|2010-10-03T03:10:00Z,53.516566,8.188217,None,12.5|2010-10-03T03:20:00Z,53.516566,8.188217,None,12.52|2010-10-03T03:30:00Z,53.516566,8.188217,None,12.54|2010-10-03T03:40:00Z,53.516566,8.188217,None,12.5|2010-10-03T03:50:00Z,53.516566,8.188217,None,12.52|2010-10-03T04:00:00Z,53.516566,8.188217,None,12.57|2010-10-03T04:10:00Z,53.516566,8.188217,None,12.67|2010-10-03T04:20:00Z,53.516566,8.188217,None,12.78|2010-10-03T04:30:00Z,53.516566,8.188217,None,12.92|2010-10-03T04:40:00Z,53.516566,8.188217,None,13.01|2010-10-03T04:50:00Z,53.516566,8.188217,None,13.08|2010-10-03T05:00:00Z,53.516566,8.188217,None,13.14|2010-10-03T05:10:00Z,53.516566,8.188217,None,13.17|2010-10-03T05:20:00Z,53.516566,8.188217,None,13.19|2010-10-03T05:30:00Z,53.516566,8.188217,None,13.19|2010-10-03T05:40:00Z,53.516566,8.188217,None,13.19|2010-10-03T05:50:00Z,53.516566,8.188217,None,13.19|2010-10-03T06:00:00Z,53.516566,8.188217,None,13.18|2010-10-03T06:10:00Z,53.516566,8.188217,None,13.17|2010-10-03T06:20:00Z,53.516566,8.188217,None,13.14|2010-10-03T06:30:00Z,53.516566,8.188217,None,13.07|2010-10-03T06:40:00Z,53.516566,8.188217,None,12.85|2010-10-03T06:50:00Z,53.516566,8.188217,None,12.59|2010-10-03T07:00:00Z,53.516566,8.188217,None,12.49|2010-10-03T07:10:00Z,53.516566,8.188217,None,12.41|2010-10-03T07:20:00Z,53.516566,8.188217,None,12.4|2010-10-03T07:30:00Z,53.516566,8.188217,None,12.36|2010-10-03T07:40:00Z,53.516566,8.188217,None,12.37|2010-10-03T07:50:00Z,53.516566,8.188217,None,12.36|2010-10-03T08:00:00Z,53.516566,8.188217,None,12.4|2010-10-03T08:10:00Z,53.516566,8.188217,None,12.51|2010-10-03T08:20:00Z,53.516566,8.188217,None,12.54|2010-10-03T08:30:00Z,53.516566,8.188217,None,12.56|2010-10-03T08:40:00Z,53.516566,8.188217,None,12.55|2010-10-03T08:50:00Z,53.516566,8.188217,None,12.55|2010-10-03T09:00:00Z,53.516566,8.188217,None,12.57|2010-10-03T09:10:00Z,53.516566,8.188217,None,12.57|2010-10-03T09:20:00Z,53.516566,8.188217,None,12.56|2010-10-03T09:30:00Z,53.516566,8.188217,None,12.56|2010-10-03T09:40:00Z,53.516566,8.188217,None,12.58|2010-10-03T09:50:00Z,53.516566,8.188217,None,12.59|2010-10-03T10:00:00Z,53.516566,8.188217,None,12.35|2010-10-03T10:10:00Z,53.516566,8.188217,None,12.62|2010-10-03T10:20:00Z,53.516566,8.188217,None,12.63|2010-10-03T10:30:00Z,53.516566,8.188217,None,13.26|2010-10-03T10:40:00Z,53.516566,8.188217,None,14.28|2010-10-03T10:50:00Z,53.516566,8.188217,None,15.13|2010-10-03T11:00:00Z,53.516566,8.188217,None,15.67|2010-10-03T11:10:00Z,53.516566,8.188217,None,16.16|2010-10-03T11:20:00Z,53.516566,8.188217,None,14.82|2010-10-03T11:30:00Z,53.516566,8.188217,None,13.95|2010-10-03T11:40:00Z,53.516566,8.188217,None,12.72|2010-10-03T11:50:00Z,53.516566,8.188217,None,12.75|2010-10-03T12:00:00Z,53.516566,8.188217,None,12.77|2010-10-03T12:10:00Z,53.516566,8.188217,None,12.79|2010-10-03T12:20:00Z,53.516566,8.188217,None,12.8|2010-10-03T12:30:00Z,53.516566,8.188217,None,12.84|2010-10-03T12:40:00Z,53.516566,8.188217,None,12.83|2010-10-03T12:50:00Z,53.516566,8.188217,None,12.89|2010-10-03T13:00:00Z,53.516566,8.188217,None,12.88|2010-10-03T13:10:00Z,53.516566,8.188217,None,12.85|2010-10-03T13:20:00Z,53.516566,8.188217,None,12.89|2010-10-03T13:30:00Z,53.516566,8.188217,None,13.05|2010-10-03T13:40:00Z,53.516566,8.188217,None,12.78|2010-10-03T13:50:00Z,53.516566,8.188217,None,12.73|2010-10-03T14:00:00Z,53.516566,8.188217,None,12.77|2010-10-03T14:10:00Z,53.516566,8.188217,None,12.78|2010-10-03T14:20:00Z,53.516566,8.188217,None,12.76|2010-10-03T14:30:00Z,53.516566,8.188217,None,12.77|2010-10-03T14:40:00Z,53.516566,8.188217,None,12.76|2010-10-03T14:50:00Z,53.516566,8.188217,None,12.77|2010-10-03T15:00:00Z,53.516566,8.188217,None,12.75|2010-10-03T15:10:00Z,53.516566,8.188217,None,12.66|2010-10-03T15:20:00Z,53.516566,8.188217,None,12.61|2010-10-03T15:30:00Z,53.516566,8.188217,None,12.5|2010-10-03T15:40:00Z,53.516566,8.188217,None,12.36|2010-10-03T15:50:00Z,53.516566,8.188217,None,12.26|2010-10-03T16:00:00Z,53.516566,8.188217,None,12.27|2010-10-03T16:10:00Z,53.516566,8.188217,None,12.33|2010-10-03T16:20:00Z,53.516566,8.188217,None,12.4|2010-10-03T16:30:00Z,53.516566,8.188217,None,12.46|2010-10-03T16:40:00Z,53.516566,8.188217,None,12.51|2010-10-03T16:50:00Z,53.516566,8.188217,None,12.49|2010-10-03T17:00:00Z,53.516566,8.188217,None,12.54|2010-10-03T17:10:00Z,53.516566,8.188217,None,12.65|2010-10-03T17:20:00Z,53.516566,8.188217,None,12.73|2010-10-03T17:30:00Z,53.516566,8.188217,None,12.78|2010-10-03T17:40:00Z,53.516566,8.188217,None,12.81|2010-10-03T17:50:00Z,53.516566,8.188217,None,12.83|2010-10-03T18:00:00Z,53.516566,8.188217,None,12.89|2010-10-03T18:10:00Z,53.516566,8.188217,None,12.97|2010-10-03T18:20:00Z,53.516566,8.188217,None,13.04|2010-10-03T18:30:00Z,53.516566,8.188217,None,13.07|2010-10-03T18:40:00Z,53.516566,8.188217,None,13.1|2010-10-03T18:50:00Z,53.516566,8.188217,None,13.12|2010-10-03T19:00:00Z,53.516566,8.188217,None,13.13|2010-10-03T19:10:00Z,53.516566,8.188217,None,13.14|2010-10-03T19:20:00Z,53.516566,8.188217,None,13.13|2010-10-03T19:30:00Z,53.516566,8.188217,None,13.08|2010-10-03T19:40:00Z,53.516566,8.188217,None,13.0|2010-10-03T19:50:00Z,53.516566,8.188217,None,12.86|2010-10-03T20:00:00Z,53.516566,8.188217,None,12.79|2010-10-03T20:10:00Z,53.516566,8.188217,None,12.62|2010-10-03T20:20:00Z,53.516566,8.188217,None,12.92|2010-10-03T20:30:00Z,53.516566,8.188217,None,13.53|2010-10-03T20:40:00Z,53.516566,8.188217,None,13.42|2010-10-03T20:50:00Z,53.516566,8.188217,None,13.03|2010-10-03T21:00:00Z,53.516566,8.188217,None,13.19|2010-10-03T21:10:00Z,53.516566,8.188217,None,13.45|2010-10-03T21:20:00Z,53.516566,8.188217,None,13.56|2010-10-03T21:30:00Z,53.516566,8.188217,None,13.58|2010-10-03T21:40:00Z,53.516566,8.188217,None,13.56|2010-10-03T21:50:00Z,53.516566,8.188217,None,13.55|2010-10-03T22:00:00Z,53.516566,8.188217,None,13.49|2010-10-03T22:10:00Z,53.516566,8.188217,None,13.48|2010-10-03T22:20:00Z,53.516566,8.188217,None,13.52|2010-10-03T22:30:00Z,53.516566,8.188217,None,13.43|2010-10-03T22:40:00Z,53.516566,8.188217,None,13.29|2010-10-03T22:50:00Z,53.516566,8.188217,None,13.26|2010-10-03T23:00:00Z,53.516566,8.188217,None,13.23|2010-10-03T23:10:00Z,53.516566,8.188217,None,13.21|2010-10-03T23:20:00Z,53.516566,8.188217,None,13.22|2010-10-03T23:30:00Z,53.516566,8.188217,None,13.23|2010-10-03T23:40:00Z,53.516566,8.188217,None,13.21|2010-10-03T23:50:00Z,53.516566,8.188217,None,13.23|2010-10-04T00:00:00Z,53.516566,8.188217,None,13.07|2010-10-04T00:10:00Z,53.516566,8.188217,None,13.01|2010-10-04T00:20:00Z,53.516566,8.188217,None,12.98|2010-10-04T00:30:00Z,53.516566,8.188217,None,13.0|2010-10-04T00:40:00Z,53.516566,8.188217,None,13.0|2010-10-04T00:50:00Z,53.516566,8.188217,None,12.91|2010-10-04T01:00:00Z,53.516566,8.188217,None,12.92|2010-10-04T01:10:00Z,53.516566,8.188217,None,12.97|2010-10-04T01:20:00Z,53.516566,8.188217,None,12.96|2010-10-04T01:30:00Z,53.516566,8.188217,None,12.95|2010-10-04T01:40:00Z,53.516566,8.188217,None,12.97|2010-10-04T01:50:00Z,53.516566,8.188217,None,12.99|2010-10-04T02:00:00Z,53.516566,8.188217,None,13.05|2010-10-04T02:10:00Z,53.516566,8.188217,None,12.93|2010-10-04T02:20:00Z,53.516566,8.188217,None,12.76|2010-10-04T02:30:00Z,53.516566,8.188217,None,12.72|2010-10-04T02:40:00Z,53.516566,8.188217,None,12.68|2010-10-04T02:50:00Z,53.516566,8.188217,None,12.71|2010-10-04T03:00:00Z,53.516566,8.188217,None,12.73|2010-10-04T03:10:00Z,53.516566,8.188217,None,12.75|2010-10-04T03:20:00Z,53.516566,8.188217,None,12.76|2010-10-04T03:30:00Z,53.516566,8.188217,None,12.79|2010-10-04T03:40:00Z,53.516566,8.188217,None,12.8|2010-10-04T03:50:00Z,53.516566,8.188217,None,12.78|2010-10-04T04:00:00Z,53.516566,8.188217,None,12.71|2010-10-04T04:10:00Z,53.516566,8.188217,None,12.71|2010-10-04T04:20:00Z,53.516566,8.188217,None,12.69|2010-10-04T04:30:00Z,53.516566,8.188217,None,12.65|2010-10-04T04:40:00Z,53.516566,8.188217,None,12.64|2010-10-04T04:50:00Z,53.516566,8.188217,None,12.67|2010-10-04T05:00:00Z,53.516566,8.188217,None,12.73|2010-10-04T05:10:00Z,53.516566,8.188217,None,12.79|2010-10-04T05:20:00Z,53.516566,8.188217,None,12.82|2010-10-04T05:30:00Z,53.516566,8.188217,None,12.88|2010-10-04T05:40:00Z,53.516566,8.188217,None,12.97|2010-10-04T05:50:00Z,53.516566,8.188217,None,13.08|2010-10-04T06:00:00Z,53.516566,8.188217,None,13.17|2010-10-04T06:10:00Z,53.516566,8.188217,None,13.21|2010-10-04T06:20:00Z,53.516566,8.188217,None,13.24|2010-10-04T06:30:00Z,53.516566,8.188217,None,13.26|2010-10-04T06:40:00Z,53.516566,8.188217,None,13.28|2010-10-04T06:50:00Z,53.516566,8.188217,None,13.31|2010-10-04T07:00:00Z,53.516566,8.188217,None,13.32|2010-10-04T07:10:00Z,53.516566,8.188217,None,13.33|2010-10-04T07:20:00Z,53.516566,8.188217,None,13.33|2010-10-04T07:30:00Z,53.516566,8.188217,None,13.33|2010-10-04T07:40:00Z,53.516566,8.188217,None,13.32|2010-10-04T07:50:00Z,53.516566,8.188217,None,13.32|2010-10-04T08:00:00Z,53.516566,8.188217,None,13.32|2010-10-04T08:10:00Z,53.516566,8.188217,None,13.28|2010-10-04T08:20:00Z,53.516566,8.188217,None,13.22|2010-10-04T08:30:00Z,53.516566,8.188217,None,13.15|2010-10-04T08:40:00Z,53.516566,8.188217,None,12.99|2010-10-04T08:50:00Z,53.516566,8.188217,None,12.8|2010-10-04T09:00:00Z,53.516566,8.188217,None,12.65|2010-10-04T09:10:00Z,53.516566,8.188217,None,12.76|2010-10-04T09:20:00Z,53.516566,8.188217,None,12.94|2010-10-04T09:30:00Z,53.516566,8.188217,None,12.98|2010-10-04T09:40:00Z,53.516566,8.188217,None,13.13|2010-10-04T09:50:00Z,53.516566,8.188217,None,12.99|2010-10-04T10:00:00Z,53.516566,8.188217,None,12.92|2010-10-04T10:10:00Z,53.516566,8.188217,None,12.84|2010-10-04T10:20:00Z,53.516566,8.188217,None,12.78|2010-10-04T10:30:00Z,53.516566,8.188217,None,12.78|2010-10-04T10:40:00Z,53.516566,8.188217,None,12.84|2010-10-04T10:50:00Z,53.516566,8.188217,None,12.88|2010-10-04T11:00:00Z,53.516566,8.188217,None,12.91|2010-10-04T11:10:00Z,53.516566,8.188217,None,12.94|2010-10-04T11:20:00Z,53.516566,8.188217,None,12.94|2010-10-04T11:30:00Z,53.516566,8.188217,None,12.92|2010-10-04T11:40:00Z,53.516566,8.188217,None,12.92|2010-10-04T11:50:00Z,53.516566,8.188217,None,12.92|2010-10-04T12:00:00Z,53.516566,8.188217,None,12.92|2010-10-04T12:10:00Z,53.516566,8.188217,None,12.92|2010-10-04T12:20:00Z,53.516566,8.188217,None,12.93|2010-10-04T12:30:00Z,53.516566,8.188217,None,12.92|2010-10-04T12:40:00Z,53.516566,8.188217,None,12.93|2010-10-04T12:50:00Z,53.516566,8.188217,None,12.93|2010-10-04T13:00:00Z,53.516566,8.188217,None,12.93|2010-10-04T13:10:00Z,53.516566,8.188217,None,12.94|2010-10-04T13:20:00Z,53.516566,8.188217,None,12.95|2010-10-04T13:30:00Z,53.516566,8.188217,None,12.96|2010-10-04T13:40:00Z,53.516566,8.188217,None,13.0|2010-10-04T13:50:00Z,53.516566,8.188217,None,12.99|2010-10-04T14:00:00Z,53.516566,8.188217,None,12.91|2010-10-04T14:10:00Z,53.516566,8.188217,None,12.86|2010-10-04T14:20:00Z,53.516566,8.188217,None,12.91|2010-10-04T14:30:00Z,53.516566,8.188217,None,12.96|2010-10-04T14:40:00Z,53.516566,8.188217,None,12.97|2010-10-04T14:50:00Z,53.516566,8.188217,None,12.96|2010-10-04T15:00:00Z,53.516566,8.188217,None,12.97|2010-10-04T15:10:00Z,53.516566,8.188217,None,13.06|2010-10-04T15:20:00Z,53.516566,8.188217,None,13.14|2010-10-04T15:30:00Z,53.516566,8.188217,None,13.06|2010-10-04T15:40:00Z,53.516566,8.188217,None,13.01|2010-10-04T15:50:00Z,53.516566,8.188217,None,13.03|2010-10-04T16:00:00Z,53.516566,8.188217,None,13.03|2010-10-04T16:10:00Z,53.516566,8.188217,None,13.06|2010-10-04T16:20:00Z,53.516566,8.188217,None,13.06|2010-10-04T16:30:00Z,53.516566,8.188217,None,13.02|2010-10-04T16:40:00Z,53.516566,8.188217,None,13.0|2010-10-04T16:50:00Z,53.516566,8.188217,None,12.98|2010-10-04T17:00:00Z,53.516566,8.188217,None,12.96|2010-10-04T17:10:00Z,53.516566,8.188217,None,12.92|2010-10-04T17:20:00Z,53.516566,8.188217,None,12.93|2010-10-04T17:30:00Z,53.516566,8.188217,None,12.94|2010-10-04T17:40:00Z,53.516566,8.188217,None,12.94|2010-10-04T17:50:00Z,53.516566,8.188217,None,12.96|2010-10-04T18:00:00Z,53.516566,8.188217,None,12.99|2010-10-04T18:10:00Z,53.516566,8.188217,None,13.01|2010-10-04T18:20:00Z,53.516566,8.188217,None,13.01|2010-10-04T18:30:00Z,53.516566,8.188217,None,13.06|2010-10-04T18:40:00Z,53.516566,8.188217,None,13.14|2010-10-04T18:50:00Z,53.516566,8.188217,None,13.19|2010-10-04T19:00:00Z,53.516566,8.188217,None,13.22|2010-10-04T19:10:00Z,53.516566,8.188217,None,13.22|2010-10-04T19:20:00Z,53.516566,8.188217,None,13.19|2010-10-04T19:30:00Z,53.516566,8.188217,None,13.16|2010-10-04T19:40:00Z,53.516566,8.188217,None,13.16|2010-10-04T19:50:00Z,53.516566,8.188217,None,13.2|2010-10-04T20:00:00Z,53.516566,8.188217,None,13.22|2010-10-04T20:10:00Z,53.516566,8.188217,None,13.23|2010-10-04T20:20:00Z,53.516566,8.188217,None,13.25|2010-10-04T20:30:00Z,53.516566,8.188217,None,13.29|2010-10-04T20:40:00Z,53.516566,8.188217,None,13.31|2010-10-04T20:50:00Z,53.516566,8.188217,None,13.31|2010-10-04T21:00:00Z,53.516566,8.188217,None,13.28|2010-10-04T21:10:00Z,53.516566,8.188217,None,13.22|2010-10-04T21:20:00Z,53.516566,8.188217,None,13.17|2010-10-04T21:30:00Z,53.516566,8.188217,None,13.14|2010-10-04T21:40:00Z,53.516566,8.188217,None,13.13|2010-10-04T21:50:00Z,53.516566,8.188217,None,13.01|2010-10-04T22:00:00Z,53.516566,8.188217,None,12.91|2010-10-04T22:10:00Z,53.516566,8.188217,None,12.91|2010-10-04T22:20:00Z,53.516566,8.188217,None,13.05|2010-10-04T22:30:00Z,53.516566,8.188217,None,13.08|2010-10-04T22:40:00Z,53.516566,8.188217,None,13.09|2010-10-04T22:50:00Z,53.516566,8.188217,None,13.22|2010-10-04T23:00:00Z,53.516566,8.188217,None,13.33|2010-10-04T23:10:00Z,53.516566,8.188217,None,13.37|2010-10-04T23:20:00Z,53.516566,8.188217,None,13.25|2010-10-04T23:30:00Z,53.516566,8.188217,None,13.07|2010-10-04T23:40:00Z,53.516566,8.188217,None,12.97|2010-10-04T23:50:00Z,53.516566,8.188217,None,12.94|2010-10-05T00:00:00Z,53.516566,8.188217,None,12.93|2010-10-05T00:10:00Z,53.516566,8.188217,None,12.91|2010-10-05T00:20:00Z,53.516566,8.188217,None,12.9|2010-10-05T00:30:00Z,53.516566,8.188217,None,12.89|2010-10-05T00:40:00Z,53.516566,8.188217,None,12.88|2010-10-05T00:50:00Z,53.516566,8.188217,None,12.87|2010-10-05T01:00:00Z,53.516566,8.188217,None,12.87|2010-10-05T01:10:00Z,53.516566,8.188217,None,12.87|2010-10-05T01:20:00Z,53.516566,8.188217,None,12.87|2010-10-05T01:30:00Z,53.516566,8.188217,None,12.94|2010-10-05T01:40:00Z,53.516566,8.188217,None,12.95|2010-10-05T01:50:00Z,53.516566,8.188217,None,12.96|2010-10-05T02:00:00Z,53.516566,8.188217,None,12.89|2010-10-05T02:10:00Z,53.516566,8.188217,None,12.81|2010-10-05T02:20:00Z,53.516566,8.188217,None,12.77|2010-10-05T02:30:00Z,53.516566,8.188217,None,12.75|2010-10-05T02:40:00Z,53.516566,8.188217,None,12.79|2010-10-05T02:50:00Z,53.516566,8.188217,None,13.04|2010-10-05T03:00:00Z,53.516566,8.188217,None,13.03|2010-10-05T03:10:00Z,53.516566,8.188217,None,13.05|2010-10-05T03:20:00Z,53.516566,8.188217,None,13.0|2010-10-05T03:30:00Z,53.516566,8.188217,None,12.94|2010-10-05T03:40:00Z,53.516566,8.188217,None,12.76|2010-10-05T03:50:00Z,53.516566,8.188217,None,12.66|2010-10-05T04:00:00Z,53.516566,8.188217,None,12.85|2010-10-05T04:10:00Z,53.516566,8.188217,None,12.97|2010-10-05T04:20:00Z,53.516566,8.188217,None,13.01|2010-10-05T04:30:00Z,53.516566,8.188217,None,13.0|2010-10-05T04:40:00Z,53.516566,8.188217,None,12.99|2010-10-05T04:50:00Z,53.516566,8.188217,None,12.97|2010-10-05T05:00:00Z,53.516566,8.188217,None,12.96|2010-10-05T05:10:00Z,53.516566,8.188217,None,12.98|2010-10-05T05:20:00Z,53.516566,8.188217,None,12.98|2010-10-05T05:30:00Z,53.516566,8.188217,None,13.0|2010-10-05T05:40:00Z,53.516566,8.188217,None,13.02|2010-10-05T05:50:00Z,53.516566,8.188217,None,13.02|2010-10-05T06:00:00Z,53.516566,8.188217,None,13.02|2010-10-05T06:10:00Z,53.516566,8.188217,None,13.06|2010-10-05T06:20:00Z,53.516566,8.188217,None,13.06|2010-10-05T06:30:00Z,53.516566,8.188217,None,13.06|2010-10-05T06:40:00Z,53.516566,8.188217,None,13.07|2010-10-05T06:50:00Z,53.516566,8.188217,None,13.09|2010-10-05T07:00:00Z,53.516566,8.188217,None,13.11|2010-10-05T07:10:00Z,53.516566,8.188217,None,13.15|2010-10-05T07:20:00Z,53.516566,8.188217,None,13.21|2010-10-05T07:30:00Z,53.516566,8.188217,None,13.22|2010-10-05T07:40:00Z,53.516566,8.188217,None,13.18|2010-10-05T07:50:00Z,53.516566,8.188217,None,13.14|2010-10-05T08:00:00Z,53.516566,8.188217,None,13.13|2010-10-05T08:10:00Z,53.516566,8.188217,None,13.16|2010-10-05T08:20:00Z,53.516566,8.188217,None,13.18|2010-10-05T08:30:00Z,53.516566,8.188217,None,13.2|2010-10-05T08:40:00Z,53.516566,8.188217,None,13.21|2010-10-05T08:50:00Z,53.516566,8.188217,None,13.22|2010-10-05T09:00:00Z,53.516566,8.188217,None,13.25|2010-10-05T09:10:00Z,53.516566,8.188217,None,13.27|2010-10-05T09:20:00Z,53.516566,8.188217,None,13.28|2010-10-05T09:30:00Z,53.516566,8.188217,None,13.26|2010-10-05T09:40:00Z,53.516566,8.188217,None,13.24|2010-10-05T09:50:00Z,53.516566,8.188217,None,13.24|2010-10-05T10:00:00Z,53.516566,8.188217,None,13.24|2010-10-05T10:10:00Z,53.516566,8.188217,None,13.28|2010-10-05T10:20:00Z,53.516566,8.188217,None,13.22|2010-10-05T10:30:00Z,53.516566,8.188217,None,13.16|2010-10-05T10:40:00Z,53.516566,8.188217,None,13.09|2010-10-05T10:50:00Z,53.516566,8.188217,None,12.91|2010-10-05T11:00:00Z,53.516566,8.188217,None,12.91|2010-10-05T11:10:00Z,53.516566,8.188217,None,12.82|2010-10-05T11:20:00Z,53.516566,8.188217,None,12.85|2010-10-05T11:30:00Z,53.516566,8.188217,None,12.84|2010-10-05T11:40:00Z,53.516566,8.188217,None,12.86|2010-10-05T11:50:00Z,53.516566,8.188217,None,12.99|2010-10-05T12:00:00Z,53.516566,8.188217,None,13.03|2010-10-05T12:10:00Z,53.516566,8.188217,None,13.06|2010-10-05T12:20:00Z,53.516566,8.188217,None,13.06|2010-10-05T12:30:00Z,53.516566,8.188217,None,13.08|2010-10-05T12:40:00Z,53.516566,8.188217,None,13.1|2010-10-05T12:50:00Z,53.516566,8.188217,None,13.14|2010-10-05T13:00:00Z,53.516566,8.188217,None,13.18|2010-10-05T13:10:00Z,53.516566,8.188217,None,13.2|2010-10-05T13:20:00Z,53.516566,8.188217,None,13.23|2010-10-05T13:30:00Z,53.516566,8.188217,None,13.23|2010-10-05T13:40:00Z,53.516566,8.188217,None,13.23|2010-10-05T13:50:00Z,53.516566,8.188217,None,13.2|2010-10-05T14:00:00Z,53.516566,8.188217,None,13.16|2010-10-05T14:10:00Z,53.516566,8.188217,None,13.15|2010-10-05T14:20:00Z,53.516566,8.188217,None,13.17|2010-10-05T14:30:00Z,53.516566,8.188217,None,13.32|2010-10-05T14:40:00Z,53.516566,8.188217,None,13.37|2010-10-05T14:50:00Z,53.516566,8.188217,None,13.08|2010-10-05T15:00:00Z,53.516566,8.188217,None,12.96|2010-10-05T15:10:00Z,53.516566,8.188217,None,12.95|2010-10-05T15:20:00Z,53.516566,8.188217,None,12.92|2010-10-05T15:30:00Z,53.516566,8.188217,None,13.06|2010-10-05T15:40:00Z,53.516566,8.188217,None,13.14|2010-10-05T15:50:00Z,53.516566,8.188217,None,13.16|2010-10-05T16:00:00Z,53.516566,8.188217,None,13.21|2010-10-05T16:10:00Z,53.516566,8.188217,None,13.27|2010-10-05T16:20:00Z,53.516566,8.188217,None,13.31|2010-10-05T16:30:00Z,53.516566,8.188217,None,13.31|2010-10-05T16:40:00Z,53.516566,8.188217,None,13.29|2010-10-05T16:50:00Z,53.516566,8.188217,None,13.21|2010-10-05T17:00:00Z,53.516566,8.188217,None,13.25|2010-10-05T17:10:00Z,53.516566,8.188217,None,13.26|2010-10-05T17:20:00Z,53.516566,8.188217,None,13.36|2010-10-05T17:30:00Z,53.516566,8.188217,None,13.33|2010-10-05T17:40:00Z,53.516566,8.188217,None,13.32|2010-10-05T17:50:00Z,53.516566,8.188217,None,13.3|2010-10-05T18:00:00Z,53.516566,8.188217,None,13.27|2010-10-05T18:10:00Z,53.516566,8.188217,None,13.23|2010-10-05T18:20:00Z,53.516566,8.188217,None,13.23|2010-10-05T18:30:00Z,53.516566,8.188217,None,13.27|2010-10-05T18:40:00Z,53.516566,8.188217,None,13.27|2010-10-05T18:50:00Z,53.516566,8.188217,None,13.27|2010-10-05T19:00:00Z,53.516566,8.188217,None,13.27|2010-10-05T19:10:00Z,53.516566,8.188217,None,13.27|2010-10-05T19:20:00Z,53.516566,8.188217,None,13.27|2010-10-05T19:30:00Z,53.516566,8.188217,None,13.29|2010-10-05T19:40:00Z,53.516566,8.188217,None,13.29|2010-10-05T19:50:00Z,53.516566,8.188217,None,13.29|2010-10-05T20:00:00Z,53.516566,8.188217,None,13.27|2010-10-05T20:10:00Z,53.516566,8.188217,None,13.26|2010-10-05T20:20:00Z,53.516566,8.188217,None,13.27|2010-10-05T20:30:00Z,53.516566,8.188217,None,13.27|2010-10-05T20:40:00Z,53.516566,8.188217,None,13.28|2010-10-05T20:50:00Z,53.516566,8.188217,None,13.28|2010-10-05T21:00:00Z,53.516566,8.188217,None,13.31|2010-10-05T21:10:00Z,53.516566,8.188217,None,13.37|2010-10-05T21:20:00Z,53.516566,8.188217,None,13.41|2010-10-05T21:30:00Z,53.516566,8.188217,None,13.43|2010-10-05T21:40:00Z,53.516566,8.188217,None,13.45|2010-10-05T21:50:00Z,53.516566,8.188217,None,13.47|2010-10-05T22:00:00Z,53.516566,8.188217,None,13.49|2010-10-05T22:10:00Z,53.516566,8.188217,None,13.5|2010-10-05T22:20:00Z,53.516566,8.188217,None,13.47|2010-10-05T22:30:00Z,53.516566,8.188217,None,13.42|2010-10-05T22:40:00Z,53.516566,8.188217,None,13.37|2010-10-05T22:50:00Z,53.516566,8.188217,None,13.27|2010-10-05T23:00:00Z,53.516566,8.188217,None,13.24|2010-10-05T23:10:00Z,53.516566,8.188217,None,13.25|2010-10-05T23:20:00Z,53.516566,8.188217,None,13.25|2010-10-05T23:30:00Z,53.516566,8.188217,None,13.26|2010-10-05T23:40:00Z,53.516566,8.188217,None,13.31|2010-10-05T23:50:00Z,53.516566,8.188217,None,13.29|2010-10-06T00:00:00Z,53.516566,8.188217,None,13.29|2010-10-06T00:10:00Z,53.516566,8.188217,None,13.37|2010-10-06T00:20:00Z,53.516566,8.188217,None,13.42|2010-10-06T00:30:00Z,53.516566,8.188217,None,13.41|2010-10-06T00:40:00Z,53.516566,8.188217,None,13.36|2010-10-06T00:50:00Z,53.516566,8.188217,None,13.3|2010-10-06T01:00:00Z,53.516566,8.188217,None,13.27|2010-10-06T01:10:00Z,53.516566,8.188217,None,13.24|2010-10-06T01:20:00Z,53.516566,8.188217,None,13.23|2010-10-06T01:30:00Z,53.516566,8.188217,None,13.21|2010-10-06T01:40:00Z,53.516566,8.188217,None,13.2|2010-10-06T01:50:00Z,53.516566,8.188217,None,13.19|2010-10-06T02:00:00Z,53.516566,8.188217,None,13.19|2010-10-06T02:10:00Z,53.516566,8.188217,None,13.19|2010-10-06T02:20:00Z,53.516566,8.188217,None,13.18|2010-10-06T02:30:00Z,53.516566,8.188217,None,13.18|2010-10-06T02:40:00Z,53.516566,8.188217,None,13.19|2010-10-06T02:50:00Z,53.516566,8.188217,None,13.19|2010-10-06T03:00:00Z,53.516566,8.188217,None,13.2|2010-10-06T03:10:00Z,53.516566,8.188217,None,13.24|2010-10-06T03:20:00Z,53.516566,8.188217,None,13.25|2010-10-06T03:30:00Z,53.516566,8.188217,None,13.19|2010-10-06T03:40:00Z,53.516566,8.188217,None,13.12|2010-10-06T03:50:00Z,53.516566,8.188217,None,13.11|2010-10-06T04:00:00Z,53.516566,8.188217,None,13.17|2010-10-06T04:10:00Z,53.516566,8.188217,None,13.24|2010-10-06T04:20:00Z,53.516566,8.188217,None,13.25|2010-10-06T04:30:00Z,53.516566,8.188217,None,13.27|2010-10-06T04:40:00Z,53.516566,8.188217,None,13.28|2010-10-06T04:50:00Z,53.516566,8.188217,None,13.26|2010-10-06T05:00:00Z,53.516566,8.188217,None,13.19|2010-10-06T05:10:00Z,53.516566,8.188217,None,13.17|2010-10-06T05:20:00Z,53.516566,8.188217,None,13.22|2010-10-06T05:30:00Z,53.516566,8.188217,None,13.33|2010-10-06T05:40:00Z,53.516566,8.188217,None,13.32|2010-10-06T05:50:00Z,53.516566,8.188217,None,13.27|2010-10-06T06:00:00Z,53.516566,8.188217,None,13.27|2010-10-06T06:10:00Z,53.516566,8.188217,None,13.27|2010-10-06T06:20:00Z,53.516566,8.188217,None,13.3|2010-10-06T06:30:00Z,53.516566,8.188217,None,13.3|2010-10-06T06:40:00Z,53.516566,8.188217,None,13.32|2010-10-06T06:50:00Z,53.516566,8.188217,None,13.31|2010-10-06T07:00:00Z,53.516566,8.188217,None,13.33|2010-10-06T07:10:00Z,53.516566,8.188217,None,13.37|2010-10-06T07:20:00Z,53.516566,8.188217,None,13.34|2010-10-06T07:30:00Z,53.516566,8.188217,None,13.32|2010-10-06T07:40:00Z,53.516566,8.188217,None,13.32|2010-10-06T07:50:00Z,53.516566,8.188217,None,13.31|2010-10-06T08:00:00Z,53.516566,8.188217,None,13.32|2010-10-06T08:10:00Z,53.516566,8.188217,None,13.31|2010-10-06T08:20:00Z,53.516566,8.188217,None,13.3|2010-10-06T08:30:00Z,53.516566,8.188217,None,13.31|2010-10-06T08:40:00Z,53.516566,8.188217,None,13.34|2010-10-06T08:50:00Z,53.516566,8.188217,None,13.38|2010-10-06T09:00:00Z,53.516566,8.188217,None,13.41|2010-10-06T09:10:00Z,53.516566,8.188217,None,13.46|2010-10-06T09:20:00Z,53.516566,8.188217,None,13.53|2010-10-06T09:30:00Z,53.516566,8.188217,None,13.56|2010-10-06T09:40:00Z,53.516566,8.188217,None,13.58|2010-10-06T09:50:00Z,53.516566,8.188217,None,13.59|2010-10-06T10:00:00Z,53.516566,8.188217,None,13.59|2010-10-06T10:10:00Z,53.516566,8.188217,None,13.6|2010-10-06T10:20:00Z,53.516566,8.188217,None,13.59|2010-10-06T10:30:00Z,53.516566,8.188217,None,13.59|2010-10-06T10:40:00Z,53.516566,8.188217,None,13.61|2010-10-06T10:50:00Z,53.516566,8.188217,None,13.63|2010-10-06T11:00:00Z,53.516566,8.188217,None,13.63|2010-10-06T11:10:00Z,53.516566,8.188217,None,13.56|2010-10-06T11:20:00Z,53.516566,8.188217,None,13.49|2010-10-06T11:30:00Z,53.516566,8.188217,None,13.49|2010-10-06T11:40:00Z,53.516566,8.188217,None,13.52|2010-10-06T11:50:00Z,53.516566,8.188217,None,13.55|2010-10-06T12:00:00Z,53.516566,8.188217,None,13.53|2010-10-06T12:10:00Z,53.516566,8.188217,None,13.5|2010-10-06T12:20:00Z,53.516566,8.188217,None,13.53|2010-10-06T12:30:00Z,53.516566,8.188217,None,13.48|2010-10-06T12:40:00Z,53.516566,8.188217,None,13.4|2010-10-06T12:50:00Z,53.516566,8.188217,None,13.4|2010-10-06T13:00:00Z,53.516566,8.188217,None,13.42|2010-10-06T13:10:00Z,53.516566,8.188217,None,13.45|2010-10-06T13:20:00Z,53.516566,8.188217,None,13.47|2010-10-06T13:30:00Z,53.516566,8.188217,None,13.53|2010-10-06T13:40:00Z,53.516566,8.188217,None,13.58|2010-10-06T13:50:00Z,53.516566,8.188217,None,13.6|2010-10-06T14:00:00Z,53.516566,8.188217,None,13.61|2010-10-06T14:10:00Z,53.516566,8.188217,None,13.62|2010-10-06T14:20:00Z,53.516566,8.188217,None,13.62|2010-10-06T14:30:00Z,53.516566,8.188217,None,13.64|2010-10-06T14:40:00Z,53.516566,8.188217,None,13.64|2010-10-06T14:50:00Z,53.516566,8.188217,None,13.64|2010-10-06T15:00:00Z,53.516566,8.188217,None,13.63|2010-10-06T15:10:00Z,53.516566,8.188217,None,13.62|2010-10-06T15:20:00Z,53.516566,8.188217,None,13.63|2010-10-06T15:30:00Z,53.516566,8.188217,None,13.61|2010-10-06T15:40:00Z,53.516566,8.188217,None,13.59|2010-10-06T15:50:00Z,53.516566,8.188217,None,13.62|2010-10-06T16:00:00Z,53.516566,8.188217,None,13.69|2010-10-06T16:10:00Z,53.516566,8.188217,None,13.67|2010-10-06T16:20:00Z,53.516566,8.188217,None,13.61|2010-10-06T16:30:00Z,53.516566,8.188217,None,13.6|2010-10-06T16:40:00Z,53.516566,8.188217,None,13.59|2010-10-06T16:50:00Z,53.516566,8.188217,None,13.54|2010-10-06T17:00:00Z,53.516566,8.188217,None,13.55|2010-10-06T17:10:00Z,53.516566,8.188217,None,13.58|2010-10-06T17:20:00Z,53.516566,8.188217,None,13.63|2010-10-06T17:30:00Z,53.516566,8.188217,None,13.69|2010-10-06T17:40:00Z,53.516566,8.188217,None,13.65|2010-10-06T17:50:00Z,53.516566,8.188217,None,13.61|2010-10-06T18:00:00Z,53.516566,8.188217,None,13.6|2010-10-06T18:10:00Z,53.516566,8.188217,None,13.7|2010-10-06T18:20:00Z,53.516566,8.188217,None,13.72|2010-10-06T18:30:00Z,53.516566,8.188217,None,13.72|2010-10-06T18:40:00Z,53.516566,8.188217,None,13.74|2010-10-06T18:50:00Z,53.516566,8.188217,None,13.74|2010-10-06T19:00:00Z,53.516566,8.188217,None,13.7|2010-10-06T19:10:00Z,53.516566,8.188217,None,13.67|2010-10-06T19:20:00Z,53.516566,8.188217,None,13.65|2010-10-06T19:30:00Z,53.516566,8.188217,None,13.66|2010-10-06T19:40:00Z,53.516566,8.188217,None,13.67|2010-10-06T19:50:00Z,53.516566,8.188217,None,13.66|2010-10-06T20:00:00Z,53.516566,8.188217,None,13.63|2010-10-06T20:10:00Z,53.516566,8.188217,None,13.61|2010-10-06T20:20:00Z,53.516566,8.188217,None,13.6|2010-10-06T20:30:00Z,53.516566,8.188217,None,13.59|2010-10-06T20:40:00Z,53.516566,8.188217,None,13.57|2010-10-06T20:50:00Z,53.516566,8.188217,None,13.56|2010-10-06T21:00:00Z,53.516566,8.188217,None,13.56|2010-10-06T21:10:00Z,53.516566,8.188217,None,13.55|2010-10-06T21:20:00Z,53.516566,8.188217,None,13.54|2010-10-06T21:30:00Z,53.516566,8.188217,None,13.57|2010-10-06T21:40:00Z,53.516566,8.188217,None,13.61|2010-10-06T21:50:00Z,53.516566,8.188217,None,13.62|2010-10-06T22:00:00Z,53.516566,8.188217,None,13.62|2010-10-06T22:10:00Z,53.516566,8.188217,None,13.63|2010-10-06T22:20:00Z,53.516566,8.188217,None,13.64|2010-10-06T22:30:00Z,53.516566,8.188217,None,13.64|2010-10-06T22:40:00Z,53.516566,8.188217,None,13.65|2010-10-06T22:50:00Z,53.516566,8.188217,None,13.65|2010-10-06T23:00:00Z,53.516566,8.188217,None,13.64|2010-10-06T23:10:00Z,53.516566,8.188217,None,13.64|2010-10-06T23:20:00Z,53.516566,8.188217,None,13.64|2010-10-06T23:30:00Z,53.516566,8.188217,None,13.63|2010-10-06T23:40:00Z,53.516566,8.188217,None,13.62|2010-10-06T23:50:00Z,53.516566,8.188217,None,13.56|2010-10-07T00:00:00Z,53.516566,8.188217,None,13.57|2010-10-07T00:10:00Z,53.516566,8.188217,None,13.61|2010-10-07T00:20:00Z,53.516566,8.188217,None,13.66|2010-10-07T00:30:00Z,53.516566,8.188217,None,13.7|2010-10-07T00:40:00Z,53.516566,8.188217,None,13.71|2010-10-07T00:50:00Z,53.516566,8.188217,None,13.68|2010-10-07T01:00:00Z,53.516566,8.188217,None,13.68|2010-10-07T01:10:00Z,53.516566,8.188217,None,13.71|2010-10-07T01:20:00Z,53.516566,8.188217,None,13.69|2010-10-07T01:30:00Z,53.516566,8.188217,None,13.67|2010-10-07T01:40:00Z,53.516566,8.188217,None,13.64|2010-10-07T01:50:00Z,53.516566,8.188217,None,13.63|2010-10-07T02:00:00Z,53.516566,8.188217,None,13.65|2010-10-07T02:10:00Z,53.516566,8.188217,None,13.65|2010-10-07T02:20:00Z,53.516566,8.188217,None,13.66|2010-10-07T02:30:00Z,53.516566,8.188217,None,13.69|2010-10-07T02:40:00Z,53.516566,8.188217,None,13.69|2010-10-07T02:50:00Z,53.516566,8.188217,None,13.68|2010-10-07T03:00:00Z,53.516566,8.188217,None,13.68|2010-10-07T03:10:00Z,53.516566,8.188217,None,13.67|2010-10-07T03:20:00Z,53.516566,8.188217,None,13.68|2010-10-07T03:30:00Z,53.516566,8.188217,None,13.68|2010-10-07T03:40:00Z,53.516566,8.188217,None,13.67|2010-10-07T03:50:00Z,53.516566,8.188217,None,13.65|2010-10-07T04:00:00Z,53.516566,8.188217,None,13.64|2010-10-07T04:10:00Z,53.516566,8.188217,None,13.63|2010-10-07T04:20:00Z,53.516566,8.188217,None,13.62|2010-10-07T04:30:00Z,53.516566,8.188217,None,13.62|2010-10-07T04:40:00Z,53.516566,8.188217,None,13.61|2010-10-07T04:50:00Z,53.516566,8.188217,None,13.65|2010-10-07T05:00:00Z,53.516566,8.188217,None,13.64|2010-10-07T05:10:00Z,53.516566,8.188217,None,13.64|2010-10-07T05:20:00Z,53.516566,8.188217,None,13.62|2010-10-07T05:30:00Z,53.516566,8.188217,None,13.61|2010-10-07T05:40:00Z,53.516566,8.188217,None,13.62|2010-10-07T05:50:00Z,53.516566,8.188217,None,13.62|2010-10-07T06:00:00Z,53.516566,8.188217,None,13.6|2010-10-07T06:10:00Z,53.516566,8.188217,None,13.61|2010-10-07T06:20:00Z,53.516566,8.188217,None,13.7|2010-10-07T06:30:00Z,53.516566,8.188217,None,13.69|2010-10-07T06:40:00Z,53.516566,8.188217,None,13.77|2010-10-07T06:50:00Z,53.516566,8.188217,None,13.75|2010-10-07T07:00:00Z,53.516566,8.188217,None,13.76|2010-10-07T07:10:00Z,53.516566,8.188217,None,13.76|2010-10-07T07:20:00Z,53.516566,8.188217,None,13.75|2010-10-07T07:30:00Z,53.516566,8.188217,None,13.73|2010-10-07T07:40:00Z,53.516566,8.188217,None,13.76|2010-10-07T07:50:00Z,53.516566,8.188217,None,13.77|2010-10-07T08:00:00Z,53.516566,8.188217,None,13.74|2010-10-07T08:10:00Z,53.516566,8.188217,None,13.71|2010-10-07T08:20:00Z,53.516566,8.188217,None,13.69|2010-10-07T08:30:00Z,53.516566,8.188217,None,13.4|2010-10-07T08:40:00Z,53.516566,8.188217,None,13.66|2010-10-07T08:50:00Z,53.516566,8.188217,None,13.66|2010-10-07T09:00:00Z,53.516566,8.188217,None,13.65|2010-10-07T09:10:00Z,53.516566,8.188217,None,13.64|2010-10-07T09:20:00Z,53.516566,8.188217,None,13.63|2010-10-07T09:30:00Z,53.516566,8.188217,None,13.63|2010-10-07T09:40:00Z,53.516566,8.188217,None,13.63|2010-10-07T09:50:00Z,53.516566,8.188217,None,13.63|2010-10-07T10:00:00Z,53.516566,8.188217,None,13.64|2010-10-07T10:10:00Z,53.516566,8.188217,None,13.64|2010-10-07T10:20:00Z,53.516566,8.188217,None,13.65|2010-10-07T10:30:00Z,53.516566,8.188217,None,13.65|2010-10-07T10:40:00Z,53.516566,8.188217,None,13.65|2010-10-07T10:50:00Z,53.516566,8.188217,None,13.66|2010-10-07T11:00:00Z,53.516566,8.188217,None,13.67|2010-10-07T11:10:00Z,53.516566,8.188217,None,13.68|2010-10-07T11:20:00Z,53.516566,8.188217,None,13.68|2010-10-07T11:30:00Z,53.516566,8.188217,None,13.68|2010-10-07T11:40:00Z,53.516566,8.188217,None,13.7|2010-10-07T11:50:00Z,53.516566,8.188217,None,13.69|2010-10-07T12:00:00Z,53.516566,8.188217,None,13.72|2010-10-07T12:10:00Z,53.516566,8.188217,None,13.77|2010-10-07T12:20:00Z,53.516566,8.188217,None,13.79|2010-10-07T12:30:00Z,53.516566,8.188217,None,13.83|2010-10-07T12:40:00Z,53.516566,8.188217,None,13.89|2010-10-07T12:50:00Z,53.516566,8.188217,None,13.92|2010-10-07T13:00:00Z,53.516566,8.188217,None,13.91|2010-10-07T13:10:00Z,53.516566,8.188217,None,13.87|2010-10-07T13:20:00Z,53.516566,8.188217,None,13.84|2010-10-07T13:30:00Z,53.516566,8.188217,None,13.87|2010-10-07T13:40:00Z,53.516566,8.188217,None,13.93|2010-10-07T13:50:00Z,53.516566,8.188217,None,13.99|2010-10-07T14:00:00Z,53.516566,8.188217,None,14.0|2010-10-07T14:10:00Z,53.516566,8.188217,None,13.7|2010-10-07T14:20:00Z,53.516566,8.188217,None,13.97|2010-10-07T14:30:00Z,53.516566,8.188217,None,14.0|2010-10-07T14:40:00Z,53.516566,8.188217,None,14.03|2010-10-07T14:50:00Z,53.516566,8.188217,None,14.04|2010-10-07T15:00:00Z,53.516566,8.188217,None,14.06|2010-10-07T15:10:00Z,53.516566,8.188217,None,14.06|2010-10-07T15:20:00Z,53.516566,8.188217,None,14.05|2010-10-07T15:30:00Z,53.516566,8.188217,None,14.05|2010-10-07T15:40:00Z,53.516566,8.188217,None,14.06|2010-10-07T15:50:00Z,53.516566,8.188217,None,14.07|2010-10-07T16:00:00Z,53.516566,8.188217,None,14.05|2010-10-07T16:10:00Z,53.516566,8.188217,None,13.99|2010-10-07T16:20:00Z,53.516566,8.188217,None,13.99|2010-10-07T16:30:00Z,53.516566,8.188217,None,14.03|2010-10-07T16:40:00Z,53.516566,8.188217,None,14.03|2010-10-07T16:50:00Z,53.516566,8.188217,None,14.03|2010-10-07T17:00:00Z,53.516566,8.188217,None,14.03|2010-10-07T17:10:00Z,53.516566,8.188217,None,14.04|2010-10-07T17:20:00Z,53.516566,8.188217,None,13.99|2010-10-07T17:30:00Z,53.516566,8.188217,None,13.95|2010-10-07T17:40:00Z,53.516566,8.188217,None,13.93|2010-10-07T17:50:00Z,53.516566,8.188217,None,13.94|2010-10-07T18:00:00Z,53.516566,8.188217,None,13.96|2010-10-07T18:10:00Z,53.516566,8.188217,None,14.02|2010-10-07T18:20:00Z,53.516566,8.188217,None,14.02|2010-10-07T18:30:00Z,53.516566,8.188217,None,14.0|2010-10-07T18:40:00Z,53.516566,8.188217,None,13.93|2010-10-07T18:50:00Z,53.516566,8.188217,None,13.92|2010-10-07T19:00:00Z,53.516566,8.188217,None,13.91|2010-10-07T19:10:00Z,53.516566,8.188217,None,14.01|2010-10-07T19:20:00Z,53.516566,8.188217,None,14.14|2010-10-07T19:30:00Z,53.516566,8.188217,None,14.25|2010-10-07T19:40:00Z,53.516566,8.188217,None,14.28|2010-10-07T19:50:00Z,53.516566,8.188217,None,14.25|2010-10-07T20:00:00Z,53.516566,8.188217,None,14.18|2010-10-07T20:10:00Z,53.516566,8.188217,None,14.13|2010-10-07T20:20:00Z,53.516566,8.188217,None,14.14|2010-10-07T20:30:00Z,53.516566,8.188217,None,14.12|2010-10-07T20:40:00Z,53.516566,8.188217,None,14.05|2010-10-07T20:50:00Z,53.516566,8.188217,None,14.0|2010-10-07T21:00:00Z,53.516566,8.188217,None,13.98|2010-10-07T21:10:00Z,53.516566,8.188217,None,13.97|2010-10-07T21:20:00Z,53.516566,8.188217,None,13.97|2010-10-07T21:30:00Z,53.516566,8.188217,None,13.96|2010-10-07T21:40:00Z,53.516566,8.188217,None,13.95|2010-10-07T21:50:00Z,53.516566,8.188217,None,13.93|2010-10-07T22:00:00Z,53.516566,8.188217,None,13.91|2010-10-07T22:10:00Z,53.516566,8.188217,None,13.89|2010-10-07T22:20:00Z,53.516566,8.188217,None,13.88|2010-10-07T22:30:00Z,53.516566,8.188217,None,13.87|2010-10-07T22:40:00Z,53.516566,8.188217,None,13.87|2010-10-07T22:50:00Z,53.516566,8.188217,None,13.85|2010-10-07T23:00:00Z,53.516566,8.188217,None,13.83|2010-10-07T23:10:00Z,53.516566,8.188217,None,13.82|2010-10-07T23:20:00Z,53.516566,8.188217,None,13.81|2010-10-07T23:30:00Z,53.516566,8.188217,None,13.8|2010-10-07T23:40:00Z,53.516566,8.188217,None,13.79|2010-10-07T23:50:00Z,53.516566,8.188217,None,13.79|2010-10-08T00:00:00Z,53.516566,8.188217,None,13.78|2010-10-08T00:10:00Z,53.516566,8.188217,None,13.78|2010-10-08T00:20:00Z,53.516566,8.188217,None,13.8|2010-10-08T00:30:00Z,53.516566,8.188217,None,13.82|2010-10-08T00:40:00Z,53.516566,8.188217,None,13.85|2010-10-08T00:50:00Z,53.516566,8.188217,None,13.9|2010-10-08T01:00:00Z,53.516566,8.188217,None,13.95|2010-10-08T01:10:00Z,53.516566,8.188217,None,14.03|2010-10-08T01:20:00Z,53.516566,8.188217,None,14.1|2010-10-08T01:30:00Z,53.516566,8.188217,None,14.1|2010-10-08T01:40:00Z,53.516566,8.188217,None,14.07|2010-10-08T01:50:00Z,53.516566,8.188217,None,14.05|2010-10-08T02:00:00Z,53.516566,8.188217,None,14.02|2010-10-08T02:10:00Z,53.516566,8.188217,None,14.02|2010-10-08T02:20:00Z,53.516566,8.188217,None,14.04|2010-10-08T02:30:00Z,53.516566,8.188217,None,14.06|2010-10-08T02:40:00Z,53.516566,8.188217,None,14.08|2010-10-08T02:50:00Z,53.516566,8.188217,None,14.06|2010-10-08T03:00:00Z,53.516566,8.188217,None,14.03|2010-10-08T03:10:00Z,53.516566,8.188217,None,14.03|2010-10-08T03:20:00Z,53.516566,8.188217,None,14.03|2010-10-08T03:30:00Z,53.516566,8.188217,None,14.02|2010-10-08T03:40:00Z,53.516566,8.188217,None,14.03|2010-10-08T03:50:00Z,53.516566,8.188217,None,14.02|2010-10-08T04:00:00Z,53.516566,8.188217,None,14.02|2010-10-08T04:10:00Z,53.516566,8.188217,None,14.02|2010-10-08T04:20:00Z,53.516566,8.188217,None,14.03|2010-10-08T04:30:00Z,53.516566,8.188217,None,14.01|2010-10-08T04:40:00Z,53.516566,8.188217,None,14.02|2010-10-08T04:50:00Z,53.516566,8.188217,None,14.0|2010-10-08T05:00:00Z,53.516566,8.188217,None,14.0|2010-10-08T05:10:00Z,53.516566,8.188217,None,13.99|2010-10-08T05:20:00Z,53.516566,8.188217,None,13.99|2010-10-08T05:30:00Z,53.516566,8.188217,None,13.98|2010-10-08T05:40:00Z,53.516566,8.188217,None,13.97|2010-10-08T05:50:00Z,53.516566,8.188217,None,13.97|2010-10-08T06:00:00Z,53.516566,8.188217,None,13.97|2010-10-08T06:10:00Z,53.516566,8.188217,None,13.97|2010-10-08T06:20:00Z,53.516566,8.188217,None,13.98|2010-10-08T06:30:00Z,53.516566,8.188217,None,13.97|2010-10-08T06:40:00Z,53.516566,8.188217,None,13.97|2010-10-08T06:50:00Z,53.516566,8.188217,None,13.97|2010-10-08T07:00:00Z,53.516566,8.188217,None,13.99|2010-10-08T07:10:00Z,53.516566,8.188217,None,13.98|2010-10-08T07:20:00Z,53.516566,8.188217,None,13.98|2010-10-08T07:30:00Z,53.516566,8.188217,None,14.02|2010-10-08T07:40:00Z,53.516566,8.188217,None,14.06|2010-10-08T07:50:00Z,53.516566,8.188217,None,14.1|2010-10-08T08:00:00Z,53.516566,8.188217,None,14.13|2010-10-08T08:10:00Z,53.516566,8.188217,None,14.12|2010-10-08T08:20:00Z,53.516566,8.188217,None,14.09|2010-10-08T08:30:00Z,53.516566,8.188217,None,14.05|2010-10-08T08:40:00Z,53.516566,8.188217,None,14.06|2010-10-08T08:50:00Z,53.516566,8.188217,None,14.03|2010-10-08T09:00:00Z,53.516566,8.188217,None,14.0|2010-10-08T09:10:00Z,53.516566,8.188217,None,14.01|2010-10-08T09:20:00Z,53.516566,8.188217,None,14.01|2010-10-08T09:30:00Z,53.516566,8.188217,None,14.0|2010-10-08T09:40:00Z,53.516566,8.188217,None,13.98|2010-10-08T09:50:00Z,53.516566,8.188217,None,13.97|2010-10-08T10:00:00Z,53.516566,8.188217,None,13.95|2010-10-08T10:10:00Z,53.516566,8.188217,None,13.95|2010-10-08T10:20:00Z,53.516566,8.188217,None,13.94|2010-10-08T10:30:00Z,53.516566,8.188217,None,13.91|2010-10-08T10:40:00Z,53.516566,8.188217,None,13.89|2010-10-08T10:50:00Z,53.516566,8.188217,None,13.87|2010-10-08T11:00:00Z,53.516566,8.188217,None,13.84|2010-10-08T11:10:00Z,53.516566,8.188217,None,13.82|2010-10-08T11:20:00Z,53.516566,8.188217,None,13.81|2010-10-08T11:30:00Z,53.516566,8.188217,None,13.8|2010-10-08T11:40:00Z,53.516566,8.188217,None,13.8|2010-10-08T11:50:00Z,53.516566,8.188217,None,13.79|2010-10-08T12:00:00Z,53.516566,8.188217,None,13.79|2010-10-08T12:10:00Z,53.516566,8.188217,None,13.79|2010-10-08T12:20:00Z,53.516566,8.188217,None,13.79|2010-10-08T12:30:00Z,53.516566,8.188217,None,13.8|2010-10-08T12:40:00Z,53.516566,8.188217,None,13.83|2010-10-08T12:50:00Z,53.516566,8.188217,None,13.86|2010-10-08T13:00:00Z,53.516566,8.188217,None,13.94|2010-10-08T13:10:00Z,53.516566,8.188217,None,14.02|2010-10-08T13:20:00Z,53.516566,8.188217,None,14.09|2010-10-08T13:30:00Z,53.516566,8.188217,None,14.17|2010-10-08T13:40:00Z,53.516566,8.188217,None,14.24|2010-10-08T13:50:00Z,53.516566,8.188217,None,14.28|2010-10-08T14:00:00Z,53.516566,8.188217,None,14.27|2010-10-08T14:10:00Z,53.516566,8.188217,None,14.23|2010-10-08T14:20:00Z,53.516566,8.188217,None,14.16|2010-10-08T14:30:00Z,53.516566,8.188217,None,14.12|2010-10-08T14:40:00Z,53.516566,8.188217,None,14.13|2010-10-08T14:50:00Z,53.516566,8.188217,None,14.14|2010-10-08T15:00:00Z,53.516566,8.188217,None,14.18|2010-10-08T15:10:00Z,53.516566,8.188217,None,14.25|2010-10-08T15:20:00Z,53.516566,8.188217,None,14.28|2010-10-08T15:30:00Z,53.516566,8.188217,None,14.28|2010-10-08T15:40:00Z,53.516566,8.188217,None,14.28|2010-10-08T15:50:00Z,53.516566,8.188217,None,14.28|2010-10-08T16:00:00Z,53.516566,8.188217,None,14.27|2010-10-08T16:10:00Z,53.516566,8.188217,None,14.27|2010-10-08T16:20:00Z,53.516566,8.188217,None,14.27|2010-10-08T16:30:00Z,53.516566,8.188217,None,14.27|2010-10-08T16:40:00Z,53.516566,8.188217,None,14.26|2010-10-08T16:50:00Z,53.516566,8.188217,None,14.23|2010-10-08T17:00:00Z,53.516566,8.188217,None,14.26|2010-10-08T17:10:00Z,53.516566,8.188217,None,14.22|2010-10-08T17:20:00Z,53.516566,8.188217,None,14.21|2010-10-08T17:30:00Z,53.516566,8.188217,None,14.21|2010-10-08T17:40:00Z,53.516566,8.188217,None,14.2|2010-10-08T17:50:00Z,53.516566,8.188217,None,14.19|2010-10-08T18:00:00Z,53.516566,8.188217,None,14.15|2010-10-08T18:10:00Z,53.516566,8.188217,None,14.13|2010-10-08T18:20:00Z,53.516566,8.188217,None,13.85|2010-10-08T18:30:00Z,53.516566,8.188217,None,14.14|2010-10-08T18:40:00Z,53.516566,8.188217,None,14.14|2010-10-08T18:50:00Z,53.516566,8.188217,None,14.13|2010-10-08T19:00:00Z,53.516566,8.188217,None,14.12|2010-10-08T19:10:00Z,53.516566,8.188217,None,14.11|2010-10-08T19:20:00Z,53.516566,8.188217,None,14.09|2010-10-08T19:30:00Z,53.516566,8.188217,None,14.12|2010-10-08T19:40:00Z,53.516566,8.188217,None,14.11|2010-10-08T19:50:00Z,53.516566,8.188217,None,14.1|2010-10-08T20:00:00Z,53.516566,8.188217,None,14.14|2010-10-08T20:10:00Z,53.516566,8.188217,None,14.23|2010-10-08T20:20:00Z,53.516566,8.188217,None,14.28|2010-10-08T20:30:00Z,53.516566,8.188217,None,14.37|2010-10-08T20:40:00Z,53.516566,8.188217,None,14.37|2010-10-08T20:50:00Z,53.516566,8.188217,None,14.31|2010-10-08T21:00:00Z,53.516566,8.188217,None,14.26|2010-10-08T21:10:00Z,53.516566,8.188217,None,14.23|2010-10-08T21:20:00Z,53.516566,8.188217,None,14.22|2010-10-08T21:30:00Z,53.516566,8.188217,None,14.17|2010-10-08T21:40:00Z,53.516566,8.188217,None,14.15|2010-10-08T21:50:00Z,53.516566,8.188217,None,14.16|2010-10-08T22:00:00Z,53.516566,8.188217,None,14.16|2010-10-08T22:10:00Z,53.516566,8.188217,None,14.15|2010-10-08T22:20:00Z,53.516566,8.188217,None,14.13|2010-10-08T22:30:00Z,53.516566,8.188217,None,14.12|2010-10-08T22:40:00Z,53.516566,8.188217,None,14.11|2010-10-08T22:50:00Z,53.516566,8.188217,None,14.09|2010-10-08T23:00:00Z,53.516566,8.188217,None,14.07|2010-10-08T23:10:00Z,53.516566,8.188217,None,14.06|2010-10-08T23:20:00Z,53.516566,8.188217,None,14.05|2010-10-08T23:30:00Z,53.516566,8.188217,None,14.04|2010-10-08T23:40:00Z,53.516566,8.188217,None,14.03|2010-10-08T23:50:00Z,53.516566,8.188217,None,14.0|2010-10-09T00:00:00Z,53.516566,8.188217,None,13.98|2010-10-09T00:10:00Z,53.516566,8.188217,None,13.96|2010-10-09T00:20:00Z,53.516566,8.188217,None,13.94|2010-10-09T00:30:00Z,53.516566,8.188217,None,13.92|2010-10-09T00:40:00Z,53.516566,8.188217,None,13.9|2010-10-09T00:50:00Z,53.516566,8.188217,None,13.89|2010-10-09T01:00:00Z,53.516566,8.188217,None,13.9|2010-10-09T01:10:00Z,53.516566,8.188217,None,13.91|2010-10-09T01:20:00Z,53.516566,8.188217,None,13.93|2010-10-09T01:30:00Z,53.516566,8.188217,None,13.92|2010-10-09T01:40:00Z,53.516566,8.188217,None,13.93|2010-10-09T01:50:00Z,53.516566,8.188217,None,13.92|2010-10-09T02:00:00Z,53.516566,8.188217,None,13.91|2010-10-09T02:10:00Z,53.516566,8.188217,None,13.91|2010-10-09T02:20:00Z,53.516566,8.188217,None,13.85|2010-10-09T02:30:00Z,53.516566,8.188217,None,13.85|2010-10-09T02:40:00Z,53.516566,8.188217,None,13.88|2010-10-09T02:50:00Z,53.516566,8.188217,None,13.87|2010-10-09T03:00:00Z,53.516566,8.188217,None,13.83|2010-10-09T03:10:00Z,53.516566,8.188217,None,13.68|2010-10-09T03:20:00Z,53.516566,8.188217,None,13.5|2010-10-09T03:30:00Z,53.516566,8.188217,None,13.53|2010-10-09T03:40:00Z,53.516566,8.188217,None,13.62|2010-10-09T03:50:00Z,53.516566,8.188217,None,13.64|2010-10-09T04:00:00Z,53.516566,8.188217,None,13.61|2010-10-09T04:10:00Z,53.516566,8.188217,None,13.63|2010-10-09T04:20:00Z,53.516566,8.188217,None,13.59|2010-10-09T04:30:00Z,53.516566,8.188217,None,13.57|2010-10-09T04:40:00Z,53.516566,8.188217,None,13.51|2010-10-09T04:50:00Z,53.516566,8.188217,None,13.49|2010-10-09T05:00:00Z,53.516566,8.188217,None,13.52|2010-10-09T05:10:00Z,53.516566,8.188217,None,13.69|2010-10-09T05:20:00Z,53.516566,8.188217,None,13.52|2010-10-09T05:30:00Z,53.516566,8.188217,None,13.31|2010-10-09T05:40:00Z,53.516566,8.188217,None,13.28|2010-10-09T05:50:00Z,53.516566,8.188217,None,13.29|2010-10-09T06:00:00Z,53.516566,8.188217,None,13.3|2010-10-09T06:10:00Z,53.516566,8.188217,None,13.34|2010-10-09T06:20:00Z,53.516566,8.188217,None,13.34|2010-10-09T06:30:00Z,53.516566,8.188217,None,13.44|2010-10-09T06:40:00Z,53.516566,8.188217,None,13.42|2010-10-09T06:50:00Z,53.516566,8.188217,None,13.46|2010-10-09T07:00:00Z,53.516566,8.188217,None,13.51|2010-10-09T07:10:00Z,53.516566,8.188217,None,13.46|2010-10-09T07:20:00Z,53.516566,8.188217,None,13.35|2010-10-09T07:30:00Z,53.516566,8.188217,None,13.25|2010-10-09T07:40:00Z,53.516566,8.188217,None,13.21|2010-10-09T07:50:00Z,53.516566,8.188217,None,13.24|2010-10-09T08:00:00Z,53.516566,8.188217,None,13.52|2010-10-09T08:10:00Z,53.516566,8.188217,None,13.69|2010-10-09T08:20:00Z,53.516566,8.188217,None,13.83|2010-10-09T08:30:00Z,53.516566,8.188217,None,13.82|2010-10-09T08:40:00Z,53.516566,8.188217,None,13.81|2010-10-09T08:50:00Z,53.516566,8.188217,None,13.83|2010-10-09T09:00:00Z,53.516566,8.188217,None,13.84|2010-10-09T09:10:00Z,53.516566,8.188217,None,13.83|2010-10-09T09:20:00Z,53.516566,8.188217,None,13.77|2010-10-09T09:30:00Z,53.516566,8.188217,None,13.84|2010-10-09T09:40:00Z,53.516566,8.188217,None,13.88|2010-10-09T09:50:00Z,53.516566,8.188217,None,13.9|2010-10-09T10:00:00Z,53.516566,8.188217,None,13.91|2010-10-09T10:10:00Z,53.516566,8.188217,None,13.92|2010-10-09T10:20:00Z,53.516566,8.188217,None,13.92|2010-10-09T10:30:00Z,53.516566,8.188217,None,13.91|2010-10-09T10:40:00Z,53.516566,8.188217,None,13.9|2010-10-09T10:50:00Z,53.516566,8.188217,None,13.88|2010-10-09T11:00:00Z,53.516566,8.188217,None,13.88|2010-10-09T11:10:00Z,53.516566,8.188217,None,13.89|2010-10-09T11:20:00Z,53.516566,8.188217,None,13.91|2010-10-09T11:30:00Z,53.516566,8.188217,None,13.9|2010-10-09T11:40:00Z,53.516566,8.188217,None,13.88|2010-10-09T11:50:00Z,53.516566,8.188217,None,13.87|2010-10-09T12:00:00Z,53.516566,8.188217,None,13.84|2010-10-09T12:10:00Z,53.516566,8.188217,None,13.82|2010-10-09T12:20:00Z,53.516566,8.188217,None,13.82|2010-10-09T12:30:00Z,53.516566,8.188217,None,13.81|2010-10-09T12:40:00Z,53.516566,8.188217,None,13.82|2010-10-09T12:50:00Z,53.516566,8.188217,None,13.81|2010-10-09T13:00:00Z,53.516566,8.188217,None,13.81|2010-10-09T13:10:00Z,53.516566,8.188217,None,13.82|2010-10-09T13:20:00Z,53.516566,8.188217,None,13.84|2010-10-09T13:30:00Z,53.516566,8.188217,None,13.88|2010-10-09T13:40:00Z,53.516566,8.188217,None,13.94|2010-10-09T13:50:00Z,53.516566,8.188217,None,14.02|2010-10-09T14:00:00Z,53.516566,8.188217,None,14.06|2010-10-09T14:10:00Z,53.516566,8.188217,None,14.11|2010-10-09T14:20:00Z,53.516566,8.188217,None,14.12|2010-10-09T14:30:00Z,53.516566,8.188217,None,14.05|2010-10-09T14:40:00Z,53.516566,8.188217,None,13.99|2010-10-09T14:50:00Z,53.516566,8.188217,None,13.93|2010-10-09T15:00:00Z,53.516566,8.188217,None,13.95|2010-10-09T15:10:00Z,53.516566,8.188217,None,13.98|2010-10-09T15:20:00Z,53.516566,8.188217,None,13.96|2010-10-09T15:30:00Z,53.516566,8.188217,None,13.87|2010-10-09T15:40:00Z,53.516566,8.188217,None,13.78|2010-10-09T15:50:00Z,53.516566,8.188217,None,13.8|2010-10-09T16:00:00Z,53.516566,8.188217,None,13.81|2010-10-09T16:10:00Z,53.516566,8.188217,None,13.77|2010-10-09T16:20:00Z,53.516566,8.188217,None,13.71|2010-10-09T16:30:00Z,53.516566,8.188217,None,13.69|2010-10-09T16:40:00Z,53.516566,8.188217,None,13.67|2010-10-09T16:50:00Z,53.516566,8.188217,None,13.67|2010-10-09T17:00:00Z,53.516566,8.188217,None,13.68|2010-10-09T17:10:00Z,53.516566,8.188217,None,13.72|2010-10-09T17:20:00Z,53.516566,8.188217,None,13.75|2010-10-09T17:30:00Z,53.516566,8.188217,None,13.77|2010-10-09T17:40:00Z,53.516566,8.188217,None,13.75|2010-10-09T17:50:00Z,53.516566,8.188217,None,13.6|2010-10-09T18:00:00Z,53.516566,8.188217,None,13.57|2010-10-09T18:10:00Z,53.516566,8.188217,None,13.53|2010-10-09T18:20:00Z,53.516566,8.188217,None,13.53|2010-10-09T18:30:00Z,53.516566,8.188217,None,13.62|2010-10-09T18:40:00Z,53.516566,8.188217,None,13.62|2010-10-09T18:50:00Z,53.516566,8.188217,None,13.63|2010-10-09T19:00:00Z,53.516566,8.188217,None,13.69|2010-10-09T19:10:00Z,53.516566,8.188217,None,13.67|2010-10-09T19:20:00Z,53.516566,8.188217,None,13.67|2010-10-09T19:30:00Z,53.516566,8.188217,None,13.67|2010-10-09T19:40:00Z,53.516566,8.188217,None,13.63|2010-10-09T19:50:00Z,53.516566,8.188217,None,13.56|2010-10-09T20:00:00Z,53.516566,8.188217,None,13.59|2010-10-09T20:10:00Z,53.516566,8.188217,None,13.53|2010-10-09T20:20:00Z,53.516566,8.188217,None,13.53|2010-10-09T20:30:00Z,53.516566,8.188217,None,13.7|2010-10-09T20:40:00Z,53.516566,8.188217,None,13.84|2010-10-09T20:50:00Z,53.516566,8.188217,None,13.87|2010-10-09T21:00:00Z,53.516566,8.188217,None,13.89|2010-10-09T21:10:00Z,53.516566,8.188217,None,13.9|2010-10-09T21:20:00Z,53.516566,8.188217,None,13.9|2010-10-09T21:30:00Z,53.516566,8.188217,None,13.89|2010-10-09T21:40:00Z,53.516566,8.188217,None,13.86|2010-10-09T21:50:00Z,53.516566,8.188217,None,13.87|2010-10-09T22:00:00Z,53.516566,8.188217,None,13.89|2010-10-09T22:10:00Z,53.516566,8.188217,None,13.86|2010-10-09T22:20:00Z,53.516566,8.188217,None,13.84|2010-10-09T22:30:00Z,53.516566,8.188217,None,13.88|2010-10-09T22:40:00Z,53.516566,8.188217,None,13.92|2010-10-09T22:50:00Z,53.516566,8.188217,None,13.94|2010-10-09T23:00:00Z,53.516566,8.188217,None,13.94|2010-10-09T23:10:00Z,53.516566,8.188217,None,13.94|2010-10-09T23:20:00Z,53.516566,8.188217,None,13.93|2010-10-09T23:30:00Z,53.516566,8.188217,None,13.92|2010-10-09T23:40:00Z,53.516566,8.188217,None,13.91|2010-10-09T23:50:00Z,53.516566,8.188217,None,13.9|2010-10-10T00:00:00Z,53.516566,8.188217,None,13.9|2010-10-10T00:10:00Z,53.516566,8.188217,None,13.9|2010-10-10T00:20:00Z,53.516566,8.188217,None,13.89|2010-10-10T00:30:00Z,53.516566,8.188217,None,13.88|2010-10-10T00:40:00Z,53.516566,8.188217,None,13.85|2010-10-10T00:50:00Z,53.516566,8.188217,None,13.81|2010-10-10T01:00:00Z,53.516566,8.188217,None,13.8|2010-10-10T01:10:00Z,53.516566,8.188217,None,13.79|2010-10-10T01:20:00Z,53.516566,8.188217,None,13.79|2010-10-10T01:30:00Z,53.516566,8.188217,None,13.78|2010-10-10T01:40:00Z,53.516566,8.188217,None,13.75|2010-10-10T01:50:00Z,53.516566,8.188217,None,13.73|2010-10-10T02:00:00Z,53.516566,8.188217,None,13.73|2010-10-10T02:10:00Z,53.516566,8.188217,None,13.7|2010-10-10T02:20:00Z,53.516566,8.188217,None,13.4|2010-10-10T02:30:00Z,53.516566,8.188217,None,13.61|2010-10-10T02:40:00Z,53.516566,8.188217,None,13.49|2010-10-10T02:50:00Z,53.516566,8.188217,None,13.33|2010-10-10T03:00:00Z,53.516566,8.188217,None,13.23|2010-10-10T03:10:00Z,53.516566,8.188217,None,13.29|2010-10-10T03:20:00Z,53.516566,8.188217,None,13.34|2010-10-10T03:30:00Z,53.516566,8.188217,None,13.28|2010-10-10T03:40:00Z,53.516566,8.188217,None,13.14|2010-10-10T03:50:00Z,53.516566,8.188217,None,13.0|2010-10-10T04:00:00Z,53.516566,8.188217,None,13.0|2010-10-10T04:10:00Z,53.516566,8.188217,None,13.11|2010-10-10T04:20:00Z,53.516566,8.188217,None,13.15|2010-10-10T04:30:00Z,53.516566,8.188217,None,13.13|2010-10-10T04:40:00Z,53.516566,8.188217,None,13.1|2010-10-10T04:50:00Z,53.516566,8.188217,None,13.06|2010-10-10T05:00:00Z,53.516566,8.188217,None,13.03|2010-10-10T05:10:00Z,53.516566,8.188217,None,13.02|2010-10-10T05:20:00Z,53.516566,8.188217,None,13.01|2010-10-10T05:30:00Z,53.516566,8.188217,None,12.99|2010-10-10T05:40:00Z,53.516566,8.188217,None,13.03|2010-10-10T05:50:00Z,53.516566,8.188217,None,13.06|2010-10-10T06:00:00Z,53.516566,8.188217,None,12.96|2010-10-10T06:10:00Z,53.516566,8.188217,None,12.84|2010-10-10T06:20:00Z,53.516566,8.188217,None,12.52|2010-10-10T06:30:00Z,53.516566,8.188217,None,12.42|2010-10-10T06:40:00Z,53.516566,8.188217,None,12.4|2010-10-10T06:50:00Z,53.516566,8.188217,None,12.49|2010-10-10T07:00:00Z,53.516566,8.188217,None,12.58|2010-10-10T07:10:00Z,53.516566,8.188217,None,12.75|2010-10-10T07:20:00Z,53.516566,8.188217,None,12.87|2010-10-10T07:30:00Z,53.516566,8.188217,None,12.7|2010-10-10T07:40:00Z,53.516566,8.188217,None,12.67|2010-10-10T07:50:00Z,53.516566,8.188217,None,12.59|2010-10-10T08:00:00Z,53.516566,8.188217,None,12.54|2010-10-10T08:10:00Z,53.516566,8.188217,None,12.61|2010-10-10T08:20:00Z,53.516566,8.188217,None,12.62|2010-10-10T08:30:00Z,53.516566,8.188217,None,12.6|2010-10-10T08:40:00Z,53.516566,8.188217,None,12.87|2010-10-10T08:50:00Z,53.516566,8.188217,None,13.2|2010-10-10T09:00:00Z,53.516566,8.188217,None,13.43|2010-10-10T09:10:00Z,53.516566,8.188217,None,13.43|2010-10-10T09:20:00Z,53.516566,8.188217,None,13.35|2010-10-10T09:30:00Z,53.516566,8.188217,None,13.3|2010-10-10T09:40:00Z,53.516566,8.188217,None,13.29|2010-10-10T09:50:00Z,53.516566,8.188217,None,13.19|2010-10-10T10:00:00Z,53.516566,8.188217,None,13.28|2010-10-10T10:10:00Z,53.516566,8.188217,None,13.39|2010-10-10T10:20:00Z,53.516566,8.188217,None,13.45|2010-10-10T10:30:00Z,53.516566,8.188217,None,13.47|2010-10-10T10:40:00Z,53.516566,8.188217,None,13.48|2010-10-10T10:50:00Z,53.516566,8.188217,None,13.52|2010-10-10T11:00:00Z,53.516566,8.188217,None,13.54|2010-10-10T11:10:00Z,53.516566,8.188217,None,13.57|2010-10-10T11:20:00Z,53.516566,8.188217,None,13.59|2010-10-10T11:30:00Z,53.516566,8.188217,None,13.61|2010-10-10T11:40:00Z,53.516566,8.188217,None,13.64|2010-10-10T11:50:00Z,53.516566,8.188217,None,13.65|2010-10-10T12:00:00Z,53.516566,8.188217,None,13.65|2010-10-10T12:10:00Z,53.516566,8.188217,None,13.67|2010-10-10T12:20:00Z,53.516566,8.188217,None,13.69|2010-10-10T12:30:00Z,53.516566,8.188217,None,13.7|2010-10-10T12:40:00Z,53.516566,8.188217,None,13.7|2010-10-10T12:50:00Z,53.516566,8.188217,None,13.7|2010-10-10T13:00:00Z,53.516566,8.188217,None,13.7|2010-10-10T13:10:00Z,53.516566,8.188217,None,13.71|2010-10-10T13:20:00Z,53.516566,8.188217,None,13.7|2010-10-10T13:30:00Z,53.516566,8.188217,None,13.71|2010-10-10T13:40:00Z,53.516566,8.188217,None,13.7|2010-10-10T13:50:00Z,53.516566,8.188217,None,13.71|2010-10-10T14:00:00Z,53.516566,8.188217,None,13.72|2010-10-10T14:10:00Z,53.516566,8.188217,None,13.74|2010-10-10T14:20:00Z,53.516566,8.188217,None,13.76|2010-10-10T14:30:00Z,53.516566,8.188217,None,13.76|2010-10-10T14:40:00Z,53.516566,8.188217,None,13.74|2010-10-10T14:50:00Z,53.516566,8.188217,None,13.71|2010-10-10T15:00:00Z,53.516566,8.188217,None,13.65|2010-10-10T15:10:00Z,53.516566,8.188217,None,13.55|2010-10-10T15:20:00Z,53.516566,8.188217,None,13.47|2010-10-10T15:30:00Z,53.516566,8.188217,None,13.42|2010-10-10T15:40:00Z,53.516566,8.188217,None,13.32|2010-10-10T15:50:00Z,53.516566,8.188217,None,13.25|2010-10-10T16:00:00Z,53.516566,8.188217,None,13.21|2010-10-10T16:10:00Z,53.516566,8.188217,None,13.18|2010-10-10T16:20:00Z,53.516566,8.188217,None,13.15|2010-10-10T16:30:00Z,53.516566,8.188217,None,13.18|2010-10-10T16:40:00Z,53.516566,8.188217,None,13.24|2010-10-10T16:50:00Z,53.516566,8.188217,None,13.26|2010-10-10T17:00:00Z,53.516566,8.188217,None,13.24|2010-10-10T17:10:00Z,53.516566,8.188217,None,13.21|2010-10-10T17:20:00Z,53.516566,8.188217,None,13.18|2010-10-10T17:30:00Z,53.516566,8.188217,None,13.16|2010-10-10T17:40:00Z,53.516566,8.188217,None,13.16|2010-10-10T17:50:00Z,53.516566,8.188217,None,13.17|2010-10-10T18:00:00Z,53.516566,8.188217,None,13.17|2010-10-10T18:10:00Z,53.516566,8.188217,None,13.18|2010-10-10T18:20:00Z,53.516566,8.188217,None,13.17|2010-10-10T18:30:00Z,53.516566,8.188217,None,13.09|2010-10-10T18:40:00Z,53.516566,8.188217,None,13.02|2010-10-10T18:50:00Z,53.516566,8.188217,None,12.94|2010-10-10T19:00:00Z,53.516566,8.188217,None,12.95|2010-10-10T19:10:00Z,53.516566,8.188217,None,12.97|2010-10-10T19:20:00Z,53.516566,8.188217,None,13.05|2010-10-10T19:30:00Z,53.516566,8.188217,None,13.06|2010-10-10T19:40:00Z,53.516566,8.188217,None,13.08|2010-10-10T19:50:00Z,53.516566,8.188217,None,13.09|2010-10-10T20:00:00Z,53.516566,8.188217,None,13.0|2010-10-10T20:10:00Z,53.516566,8.188217,None,12.97|2010-10-10T20:20:00Z,53.516566,8.188217,None,12.95|2010-10-10T20:30:00Z,53.516566,8.188217,None,12.93|2010-10-10T20:40:00Z,53.516566,8.188217,None,12.88|2010-10-10T20:50:00Z,53.516566,8.188217,None,12.85|2010-10-10T21:00:00Z,53.516566,8.188217,None,12.95|2010-10-10T21:10:00Z,53.516566,8.188217,None,13.1|2010-10-10T21:20:00Z,53.516566,8.188217,None,13.32|2010-10-10T21:30:00Z,53.516566,8.188217,None,13.36|2010-10-10T21:40:00Z,53.516566,8.188217,None,13.27|2010-10-10T21:50:00Z,53.516566,8.188217,None,13.21|2010-10-10T22:00:00Z,53.516566,8.188217,None,13.19|2010-10-10T22:10:00Z,53.516566,8.188217,None,13.17|2010-10-10T22:20:00Z,53.516566,8.188217,None,13.21|2010-10-10T22:30:00Z,53.516566,8.188217,None,13.28|2010-10-10T22:40:00Z,53.516566,8.188217,None,13.31|2010-10-10T22:50:00Z,53.516566,8.188217,None,13.32|2010-10-10T23:00:00Z,53.516566,8.188217,None,13.33|2010-10-10T23:10:00Z,53.516566,8.188217,None,13.35|2010-10-10T23:20:00Z,53.516566,8.188217,None,13.39|2010-10-10T23:30:00Z,53.516566,8.188217,None,13.44|2010-10-10T23:40:00Z,53.516566,8.188217,None,13.5|2010-10-10T23:50:00Z,53.516566,8.188217,None,13.53|2010-10-11T00:00:00Z,53.516566,8.188217,None,13.53|2010-10-11T00:10:00Z,53.516566,8.188217,None,13.53|2010-10-11T00:20:00Z,53.516566,8.188217,None,13.54|2010-10-11T00:30:00Z,53.516566,8.188217,None,13.56|2010-10-11T00:40:00Z,53.516566,8.188217,None,13.58|2010-10-11T00:50:00Z,53.516566,8.188217,None,13.59|2010-10-11T01:00:00Z,53.516566,8.188217,None,13.59|2010-10-11T01:10:00Z,53.516566,8.188217,None,13.6|2010-10-11T01:20:00Z,53.516566,8.188217,None,13.62|2010-10-11T01:30:00Z,53.516566,8.188217,None,13.62|2010-10-11T01:40:00Z,53.516566,8.188217,None,13.63|2010-10-11T01:50:00Z,53.516566,8.188217,None,13.63|2010-10-11T02:00:00Z,53.516566,8.188217,None,13.62|2010-10-11T02:10:00Z,53.516566,8.188217,None,13.61|2010-10-11T02:20:00Z,53.516566,8.188217,None,13.6|2010-10-11T02:30:00Z,53.516566,8.188217,None,13.58|2010-10-11T02:40:00Z,53.516566,8.188217,None,13.54|2010-10-11T02:50:00Z,53.516566,8.188217,None,13.47|2010-10-11T03:00:00Z,53.516566,8.188217,None,13.38|2010-10-11T03:10:00Z,53.516566,8.188217,None,13.26|2010-10-11T03:20:00Z,53.516566,8.188217,None,13.1|2010-10-11T03:30:00Z,53.516566,8.188217,None,12.91|2010-10-11T03:40:00Z,53.516566,8.188217,None,12.67|2010-10-11T03:50:00Z,53.516566,8.188217,None,12.59|2010-10-11T04:00:00Z,53.516566,8.188217,None,12.56|2010-10-11T04:10:00Z,53.516566,8.188217,None,12.45|2010-10-11T04:20:00Z,53.516566,8.188217,None,12.4|2010-10-11T04:30:00Z,53.516566,8.188217,None,12.53|2010-10-11T04:40:00Z,53.516566,8.188217,None,12.73|2010-10-11T04:50:00Z,53.516566,8.188217,None,12.84|2010-10-11T05:00:00Z,53.516566,8.188217,None,12.89|2010-10-11T05:10:00Z,53.516566,8.188217,None,12.88|2010-10-11T05:20:00Z,53.516566,8.188217,None,12.85|2010-10-11T05:30:00Z,53.516566,8.188217,None,12.83|2010-10-11T05:40:00Z,53.516566,8.188217,None,12.82|2010-10-11T05:50:00Z,53.516566,8.188217,None,12.8|2010-10-11T06:00:00Z,53.516566,8.188217,None,12.79|2010-10-11T06:10:00Z,53.516566,8.188217,None,12.78|2010-10-11T06:20:00Z,53.516566,8.188217,None,12.78|2010-10-11T06:30:00Z,53.516566,8.188217,None,12.79|2010-10-11T06:40:00Z,53.516566,8.188217,None,12.79|2010-10-11T06:50:00Z,53.516566,8.188217,None,12.74|2010-10-11T07:00:00Z,53.516566,8.188217,None,12.59|2010-10-11T07:10:00Z,53.516566,8.188217,None,12.38|2010-10-11T07:20:00Z,53.516566,8.188217,None,12.36|2010-10-11T07:30:00Z,53.516566,8.188217,None,12.34|2010-10-11T07:40:00Z,53.516566,8.188217,None,12.33|2010-10-11T07:50:00Z,53.516566,8.188217,None,12.36|2010-10-11T08:00:00Z,53.516566,8.188217,None,12.4|2010-10-11T08:10:00Z,53.516566,8.188217,None,12.42|2010-10-11T08:20:00Z,53.516566,8.188217,None,12.45|2010-10-11T08:30:00Z,53.516566,8.188217,None,12.47|2010-10-11T08:40:00Z,53.516566,8.188217,None,12.48|2010-10-11T08:50:00Z,53.516566,8.188217,None,12.54|2010-10-11T09:00:00Z,53.516566,8.188217,None,12.49|2010-10-11T09:10:00Z,53.516566,8.188217,None,12.26|2010-10-11T09:20:00Z,53.516566,8.188217,None,12.46|2010-10-11T09:30:00Z,53.516566,8.188217,None,12.67|2010-10-11T09:40:00Z,53.516566,8.188217,None,12.97|2010-10-11T09:50:00Z,53.516566,8.188217,None,12.97|2010-10-11T10:00:00Z,53.516566,8.188217,None,12.93|2010-10-11T10:10:00Z,53.516566,8.188217,None,12.87|2010-10-11T10:20:00Z,53.516566,8.188217,None,12.88|2010-10-11T10:30:00Z,53.516566,8.188217,None,12.79|2010-10-11T10:40:00Z,53.516566,8.188217,None,12.75|2010-10-11T10:50:00Z,53.516566,8.188217,None,12.84|2010-10-11T11:00:00Z,53.516566,8.188217,None,12.95|2010-10-11T11:10:00Z,53.516566,8.188217,None,13.01|2010-10-11T11:20:00Z,53.516566,8.188217,None,13.01|2010-10-11T11:30:00Z,53.516566,8.188217,None,13.03|2010-10-11T11:40:00Z,53.516566,8.188217,None,13.07|2010-10-11T11:50:00Z,53.516566,8.188217,None,13.1|2010-10-11T12:00:00Z,53.516566,8.188217,None,13.12|2010-10-11T12:10:00Z,53.516566,8.188217,None,13.15|2010-10-11T12:20:00Z,53.516566,8.188217,None,13.18|2010-10-11T12:30:00Z,53.516566,8.188217,None,13.22|2010-10-11T12:40:00Z,53.516566,8.188217,None,13.26|2010-10-11T12:50:00Z,53.516566,8.188217,None,13.3|2010-10-11T13:00:00Z,53.516566,8.188217,None,13.32|2010-10-11T13:10:00Z,53.516566,8.188217,None,13.34|2010-10-11T13:20:00Z,53.516566,8.188217,None,13.36|2010-10-11T13:30:00Z,53.516566,8.188217,None,13.41|2010-10-11T13:40:00Z,53.516566,8.188217,None,13.48|2010-10-11T13:50:00Z,53.516566,8.188217,None,13.55|2010-10-11T14:00:00Z,53.516566,8.188217,None,13.57|2010-10-11T14:10:00Z,53.516566,8.188217,None,13.56|2010-10-11T14:20:00Z,53.516566,8.188217,None,13.56|2010-10-11T14:30:00Z,53.516566,8.188217,None,13.56|2010-10-11T14:40:00Z,53.516566,8.188217,None,13.57|2010-10-11T14:50:00Z,53.516566,8.188217,None,13.58|2010-10-11T15:00:00Z,53.516566,8.188217,None,13.54|2010-10-11T15:10:00Z,53.516566,8.188217,None,13.42|2010-10-11T15:20:00Z,53.516566,8.188217,None,13.33|2010-10-11T15:30:00Z,53.516566,8.188217,None,13.25|2010-10-11T15:40:00Z,53.516566,8.188217,None,13.18|2010-10-11T15:50:00Z,53.516566,8.188217,None,13.09|2010-10-11T16:00:00Z,53.516566,8.188217,None,12.98|2010-10-11T16:10:00Z,53.516566,8.188217,None,12.92|2010-10-11T16:20:00Z,53.516566,8.188217,None,12.85|2010-10-11T16:30:00Z,53.516566,8.188217,None,12.81|2010-10-11T16:40:00Z,53.516566,8.188217,None,12.76|2010-10-11T16:50:00Z,53.516566,8.188217,None,12.73|2010-10-11T17:00:00Z,53.516566,8.188217,None,12.75|2010-10-11T17:10:00Z,53.516566,8.188217,None,12.8|2010-10-11T17:20:00Z,53.516566,8.188217,None,12.81|2010-10-11T17:30:00Z,53.516566,8.188217,None,12.79|2010-10-11T17:40:00Z,53.516566,8.188217,None,12.78|2010-10-11T17:50:00Z,53.516566,8.188217,None,12.77|2010-10-11T18:00:00Z,53.516566,8.188217,None,12.75|2010-10-11T18:10:00Z,53.516566,8.188217,None,12.74|2010-10-11T18:20:00Z,53.516566,8.188217,None,12.73|2010-10-11T18:30:00Z,53.516566,8.188217,None,12.74|2010-10-11T18:40:00Z,53.516566,8.188217,None,12.75|2010-10-11T18:50:00Z,53.516566,8.188217,None,12.76|2010-10-11T19:00:00Z,53.516566,8.188217,None,12.76|2010-10-11T19:10:00Z,53.516566,8.188217,None,12.72|2010-10-11T19:20:00Z,53.516566,8.188217,None,12.66|2010-10-11T19:30:00Z,53.516566,8.188217,None,12.6|2010-10-11T19:40:00Z,53.516566,8.188217,None,12.62|2010-10-11T19:50:00Z,53.516566,8.188217,None,12.6|2010-10-11T20:00:00Z,53.516566,8.188217,None,12.6|2010-10-11T20:10:00Z,53.516566,8.188217,None,12.63|2010-10-11T20:20:00Z,53.516566,8.188217,None,12.65|2010-10-11T20:30:00Z,53.516566,8.188217,None,12.64|2010-10-11T20:40:00Z,53.516566,8.188217,None,12.63|2010-10-11T20:50:00Z,53.516566,8.188217,None,12.61|2010-10-11T21:00:00Z,53.516566,8.188217,None,12.59|2010-10-11T21:10:00Z,53.516566,8.188217,None,12.58|2010-10-11T21:20:00Z,53.516566,8.188217,None,12.54|2010-10-11T21:30:00Z,53.516566,8.188217,None,12.44|2010-10-11T21:40:00Z,53.516566,8.188217,None,12.55|2010-10-11T21:50:00Z,53.516566,8.188217,None,12.69|2010-10-11T22:00:00Z,53.516566,8.188217,None,12.87|2010-10-11T22:10:00Z,53.516566,8.188217,None,12.88|2010-10-11T22:20:00Z,53.516566,8.188217,None,12.82|2010-10-11T22:30:00Z,53.516566,8.188217,None,12.68|2010-10-11T22:40:00Z,53.516566,8.188217,None,12.72|2010-10-11T22:50:00Z,53.516566,8.188217,None,12.65|2010-10-11T23:00:00Z,53.516566,8.188217,None,12.83|2010-10-11T23:10:00Z,53.516566,8.188217,None,13.01|2010-10-11T23:20:00Z,53.516566,8.188217,None,12.86|2010-10-11T23:30:00Z,53.516566,8.188217,None,12.89|2010-10-11T23:40:00Z,53.516566,8.188217,None,12.88|2010-10-11T23:50:00Z,53.516566,8.188217,None,12.87|2010-10-12T00:00:00Z,53.516566,8.188217,None,12.85|2010-10-12T00:10:00Z,53.516566,8.188217,None,12.86|2010-10-12T00:20:00Z,53.516566,8.188217,None,12.91|2010-10-12T00:30:00Z,53.516566,8.188217,None,12.99|2010-10-12T00:40:00Z,53.516566,8.188217,None,13.03|2010-10-12T00:50:00Z,53.516566,8.188217,None,13.07|2010-10-12T01:00:00Z,53.516566,8.188217,None,13.13|2010-10-12T01:10:00Z,53.516566,8.188217,None,13.17|2010-10-12T01:20:00Z,53.516566,8.188217,None,13.2|2010-10-12T01:30:00Z,53.516566,8.188217,None,13.2|2010-10-12T01:40:00Z,53.516566,8.188217,None,13.2|2010-10-12T01:50:00Z,53.516566,8.188217,None,13.23|2010-10-12T02:00:00Z,53.516566,8.188217,None,13.28|2010-10-12T02:10:00Z,53.516566,8.188217,None,13.32|2010-10-12T02:20:00Z,53.516566,8.188217,None,13.36|2010-10-12T02:30:00Z,53.516566,8.188217,None,13.39|2010-10-12T02:40:00Z,53.516566,8.188217,None,13.4|2010-10-12T02:50:00Z,53.516566,8.188217,None,13.4|2010-10-12T03:00:00Z,53.516566,8.188217,None,13.38|2010-10-12T03:10:00Z,53.516566,8.188217,None,13.29|2010-10-12T03:20:00Z,53.516566,8.188217,None,13.19|2010-10-12T03:30:00Z,53.516566,8.188217,None,13.1|2010-10-12T03:40:00Z,53.516566,8.188217,None,12.89|2010-10-12T03:50:00Z,53.516566,8.188217,None,12.72|2010-10-12T04:00:00Z,53.516566,8.188217,None,12.67|2010-10-12T04:10:00Z,53.516566,8.188217,None,12.46|2010-10-12T04:20:00Z,53.516566,8.188217,None,12.31|2010-10-12T04:30:00Z,53.516566,8.188217,None,12.23|2010-10-12T04:40:00Z,53.516566,8.188217,None,12.17|2010-10-12T04:50:00Z,53.516566,8.188217,None,12.15|2010-10-12T05:00:00Z,53.516566,8.188217,None,12.13|2010-10-12T05:10:00Z,53.516566,8.188217,None,12.24|2010-10-12T05:20:00Z,53.516566,8.188217,None,12.35|2010-10-12T05:30:00Z,53.516566,8.188217,None,12.42|2010-10-12T05:40:00Z,53.516566,8.188217,None,12.45|2010-10-12T05:50:00Z,53.516566,8.188217,None,12.5|2010-10-12T06:00:00Z,53.516566,8.188217,None,12.56|2010-10-12T06:10:00Z,53.516566,8.188217,None,12.57|2010-10-12T06:20:00Z,53.516566,8.188217,None,12.56|2010-10-12T06:30:00Z,53.516566,8.188217,None,12.55|2010-10-12T06:40:00Z,53.516566,8.188217,None,12.54|2010-10-12T06:50:00Z,53.516566,8.188217,None,12.54|2010-10-12T07:00:00Z,53.516566,8.188217,None,12.51|2010-10-12T07:10:00Z,53.516566,8.188217,None,12.48|2010-10-12T07:20:00Z,53.516566,8.188217,None,12.46|2010-10-12T07:30:00Z,53.516566,8.188217,None,12.45|2010-10-12T07:40:00Z,53.516566,8.188217,None,12.5|2010-10-12T07:50:00Z,53.516566,8.188217,None,12.45|2010-10-12T08:00:00Z,53.516566,8.188217,None,12.3|2010-10-12T08:10:00Z,53.516566,8.188217,None,12.24|2010-10-12T08:20:00Z,53.516566,8.188217,None,12.23|2010-10-12T08:30:00Z,53.516566,8.188217,None,12.24|2010-10-12T08:40:00Z,53.516566,8.188217,None,12.22|2010-10-12T08:50:00Z,53.516566,8.188217,None,12.25|2010-10-12T09:00:00Z,53.516566,8.188217,None,12.37|2010-10-12T09:10:00Z,53.516566,8.188217,None,11.02|2010-10-12T09:20:00Z,53.516566,8.188217,None,6.902|2010-10-12T09:30:00Z,53.516566,8.188217,None,7.17|2010-10-12T09:40:00Z,53.516566,8.188217,None,7.41|2010-10-12T09:50:00Z,53.516566,8.188217,None,7.7|2010-10-12T10:00:00Z,53.516566,8.188217,None,8.15|2010-10-12T10:10:00Z,53.516566,8.188217,None,10.99|2010-10-12T10:20:00Z,53.516566,8.188217,None,8.05|2010-10-12T10:30:00Z,53.516566,8.188217,None,8.49|2010-10-12T10:40:00Z,53.516566,8.188217,None,9.49|2010-10-12T10:50:00Z,53.516566,8.188217,None,-3.679|2010-10-12T11:00:00Z,53.516566,8.188217,None,-4.614|2010-10-12T12:00:00Z,53.516566,8.188217,None,12.78|2010-10-12T12:10:00Z,53.516566,8.188217,None,12.77|2010-10-12T12:20:00Z,53.516566,8.188217,None,12.79|2010-10-12T12:30:00Z,53.516566,8.188217,None,12.79|2010-10-12T12:40:00Z,53.516566,8.188217,None,12.81|2010-10-12T12:50:00Z,53.516566,8.188217,None,12.84|2010-10-12T13:00:00Z,53.516566,8.188217,None,12.87|2010-10-12T13:10:00Z,53.516566,8.188217,None,12.88|2010-10-12T13:20:00Z,53.516566,8.188217,None,12.9|2010-10-12T13:30:00Z,53.516566,8.188217,None,12.94|2010-10-12T13:40:00Z,53.516566,8.188217,None,13.0|2010-10-12T13:50:00Z,53.516566,8.188217,None,13.05|2010-10-12T14:00:00Z,53.516566,8.188217,None,13.1|2010-10-12T14:10:00Z,53.516566,8.188217,None,13.15|2010-10-12T14:20:00Z,53.516566,8.188217,None,13.18|2010-10-12T14:30:00Z,53.516566,8.188217,None,13.24|2010-10-12T14:40:00Z,53.516566,8.188217,None,13.29|2010-10-12T14:50:00Z,53.516566,8.188217,None,13.33|2010-10-12T15:00:00Z,53.516566,8.188217,None,13.35|2010-10-12T15:10:00Z,53.516566,8.188217,None,13.36|2010-10-12T15:20:00Z,53.516566,8.188217,None,13.35|2010-10-12T15:30:00Z,53.516566,8.188217,None,13.28|2010-10-12T15:40:00Z,53.516566,8.188217,None,13.19|2010-10-12T15:50:00Z,53.516566,8.188217,None,13.09|2010-10-12T16:00:00Z,53.516566,8.188217,None,12.96|2010-10-12T16:10:00Z,53.516566,8.188217,None,12.87|2010-10-12T16:20:00Z,53.516566,8.188217,None,12.83|2010-10-12T16:30:00Z,53.516566,8.188217,None,12.83|2010-10-12T16:40:00Z,53.516566,8.188217,None,12.88|2010-10-12T16:50:00Z,53.516566,8.188217,None,12.84|2010-10-12T17:00:00Z,53.516566,8.188217,None,12.81|2010-10-12T17:10:00Z,53.516566,8.188217,None,12.84|2010-10-12T17:20:00Z,53.516566,8.188217,None,12.83|2010-10-12T17:30:00Z,53.516566,8.188217,None,12.81|2010-10-12T17:40:00Z,53.516566,8.188217,None,12.77|2010-10-12T17:50:00Z,53.516566,8.188217,None,12.7|2010-10-12T18:00:00Z,53.516566,8.188217,None,12.6|2010-10-12T18:10:00Z,53.516566,8.188217,None,12.56|2010-10-12T18:20:00Z,53.516566,8.188217,None,12.55|2010-10-12T18:30:00Z,53.516566,8.188217,None,12.55|2010-10-12T18:40:00Z,53.516566,8.188217,None,12.54|2010-10-12T18:50:00Z,53.516566,8.188217,None,12.54|2010-10-12T19:00:00Z,53.516566,8.188217,None,12.53|2010-10-12T19:10:00Z,53.516566,8.188217,None,12.53|2010-10-12T19:20:00Z,53.516566,8.188217,None,12.53|2010-10-12T19:30:00Z,53.516566,8.188217,None,12.54|2010-10-12T19:40:00Z,53.516566,8.188217,None,12.55|2010-10-12T19:50:00Z,53.516566,8.188217,None,12.55|2010-10-12T20:00:00Z,53.516566,8.188217,None,12.51|2010-10-12T20:10:00Z,53.516566,8.188217,None,12.47|2010-10-12T20:20:00Z,53.516566,8.188217,None,12.52|2010-10-12T20:30:00Z,53.516566,8.188217,None,12.57|2010-10-12T20:40:00Z,53.516566,8.188217,None,12.57|2010-10-12T20:50:00Z,53.516566,8.188217,None,12.58|2010-10-12T21:00:00Z,53.516566,8.188217,None,12.61|2010-10-12T21:10:00Z,53.516566,8.188217,None,12.6|2010-10-12T21:20:00Z,53.516566,8.188217,None,12.56|2010-10-12T21:30:00Z,53.516566,8.188217,None,12.53|2010-10-12T21:40:00Z,53.516566,8.188217,None,12.5|2010-10-12T21:50:00Z,53.516566,8.188217,None,12.44|2010-10-12T22:00:00Z,53.516566,8.188217,None,12.45|2010-10-12T22:10:00Z,53.516566,8.188217,None,12.49|2010-10-12T22:20:00Z,53.516566,8.188217,None,12.5|2010-10-12T22:30:00Z,53.516566,8.188217,None,12.61|2010-10-12T22:40:00Z,53.516566,8.188217,None,12.66|2010-10-12T22:50:00Z,53.516566,8.188217,None,12.6|2010-10-12T23:00:00Z,53.516566,8.188217,None,12.65|2010-10-12T23:10:00Z,53.516566,8.188217,None,12.57|2010-10-12T23:20:00Z,53.516566,8.188217,None,12.51|2010-10-12T23:30:00Z,53.516566,8.188217,None,12.5|2010-10-12T23:40:00Z,53.516566,8.188217,None,12.7|2010-10-12T23:50:00Z,53.516566,8.188217,None,12.72|2010-10-13T00:00:00Z,53.516566,8.188217,None,12.66|2010-10-13T00:10:00Z,53.516566,8.188217,None,12.71|2010-10-13T00:20:00Z,53.516566,8.188217,None,12.71|2010-10-13T00:30:00Z,53.516566,8.188217,None,12.7|2010-10-13T00:40:00Z,53.516566,8.188217,None,12.74|2010-10-13T00:50:00Z,53.516566,8.188217,None,12.79|2010-10-13T01:00:00Z,53.516566,8.188217,None,12.82|2010-10-13T01:10:00Z,53.516566,8.188217,None,12.86|2010-10-13T01:20:00Z,53.516566,8.188217,None,12.87|2010-10-13T01:30:00Z,53.516566,8.188217,None,12.88|2010-10-13T01:40:00Z,53.516566,8.188217,None,12.9|2010-10-13T01:50:00Z,53.516566,8.188217,None,12.94|2010-10-13T02:00:00Z,53.516566,8.188217,None,12.97|2010-10-13T02:10:00Z,53.516566,8.188217,None,12.99|2010-10-13T02:20:00Z,53.516566,8.188217,None,13.01|2010-10-13T02:30:00Z,53.516566,8.188217,None,13.05|2010-10-13T02:40:00Z,53.516566,8.188217,None,13.06|2010-10-13T02:50:00Z,53.516566,8.188217,None,13.08|2010-10-13T03:00:00Z,53.516566,8.188217,None,13.1|2010-10-13T03:10:00Z,53.516566,8.188217,None,13.11|2010-10-13T03:20:00Z,53.516566,8.188217,None,13.1|2010-10-13T03:30:00Z,53.516566,8.188217,None,13.07|2010-10-13T03:40:00Z,53.516566,8.188217,None,13.01|2010-10-13T03:50:00Z,53.516566,8.188217,None,12.95|2010-10-13T04:00:00Z,53.516566,8.188217,None,12.87|2010-10-13T04:10:00Z,53.516566,8.188217,None,12.75|2010-10-13T04:20:00Z,53.516566,8.188217,None,12.62|2010-10-13T04:30:00Z,53.516566,8.188217,None,12.56|2010-10-13T04:40:00Z,53.516566,8.188217,None,12.41|2010-10-13T04:50:00Z,53.516566,8.188217,None,12.31|2010-10-13T05:00:00Z,53.516566,8.188217,None,12.22|2010-10-13T05:10:00Z,53.516566,8.188217,None,12.21|2010-10-13T05:20:00Z,53.516566,8.188217,None,12.19|2010-10-13T05:30:00Z,53.516566,8.188217,None,12.27|2010-10-13T05:40:00Z,53.516566,8.188217,None,12.35|2010-10-13T05:50:00Z,53.516566,8.188217,None,12.41|2010-10-13T06:00:00Z,53.516566,8.188217,None,12.37|2010-10-13T06:10:00Z,53.516566,8.188217,None,12.37|2010-10-13T06:20:00Z,53.516566,8.188217,None,12.38|2010-10-13T06:30:00Z,53.516566,8.188217,None,12.42|2010-10-13T06:40:00Z,53.516566,8.188217,None,12.45|2010-10-13T06:50:00Z,53.516566,8.188217,None,12.46|2010-10-13T07:00:00Z,53.516566,8.188217,None,12.46|2010-10-13T07:10:00Z,53.516566,8.188217,None,12.45|2010-10-13T07:20:00Z,53.516566,8.188217,None,12.45|2010-10-13T07:30:00Z,53.516566,8.188217,None,12.44|2010-10-13T07:40:00Z,53.516566,8.188217,None,12.18|2010-10-13T07:50:00Z,53.516566,8.188217,None,12.42|2010-10-13T08:00:00Z,53.516566,8.188217,None,12.42|2010-10-13T08:10:00Z,53.516566,8.188217,None,12.41|2010-10-13T08:20:00Z,53.516566,8.188217,None,12.44|2010-10-13T08:30:00Z,53.516566,8.188217,None,12.41|2010-10-13T08:40:00Z,53.516566,8.188217,None,12.33|2010-10-13T08:50:00Z,53.516566,8.188217,None,12.32|2010-10-13T09:00:00Z,53.516566,8.188217,None,12.32|2010-10-13T09:10:00Z,53.516566,8.188217,None,12.32|2010-10-13T09:20:00Z,53.516566,8.188217,None,12.33|2010-10-13T09:30:00Z,53.516566,8.188217,None,12.36|2010-10-13T09:40:00Z,53.516566,8.188217,None,12.36|2010-10-13T09:50:00Z,53.516566,8.188217,None,12.37|2010-10-13T10:00:00Z,53.516566,8.188217,None,12.39|2010-10-13T10:10:00Z,53.516566,8.188217,None,12.42|2010-10-13T10:20:00Z,53.516566,8.188217,None,12.49|2010-10-13T10:30:00Z,53.516566,8.188217,None,12.38|2010-10-13T10:40:00Z,53.516566,8.188217,None,12.27|2010-10-13T10:50:00Z,53.516566,8.188217,None,12.34|2010-10-13T11:00:00Z,53.516566,8.188217,None,12.5|2010-10-13T11:10:00Z,53.516566,8.188217,None,12.61|2010-10-13T11:20:00Z,53.516566,8.188217,None,12.56|2010-10-13T11:30:00Z,53.516566,8.188217,None,12.44|2010-10-13T11:40:00Z,53.516566,8.188217,None,12.42|2010-10-13T11:50:00Z,53.516566,8.188217,None,12.42|2010-10-13T12:00:00Z,53.516566,8.188217,None,12.41|2010-10-13T12:10:00Z,53.516566,8.188217,None,12.51|2010-10-13T12:20:00Z,53.516566,8.188217,None,12.59|2010-10-13T12:30:00Z,53.516566,8.188217,None,12.57|2010-10-13T12:40:00Z,53.516566,8.188217,None,12.61|2010-10-13T12:50:00Z,53.516566,8.188217,None,12.62|2010-10-13T13:00:00Z,53.516566,8.188217,None,12.63|2010-10-13T13:10:00Z,53.516566,8.188217,None,12.66|2010-10-13T13:20:00Z,53.516566,8.188217,None,12.69|2010-10-13T13:30:00Z,53.516566,8.188217,None,12.71|2010-10-13T13:40:00Z,53.516566,8.188217,None,12.74|2010-10-13T13:50:00Z,53.516566,8.188217,None,12.76|2010-10-13T14:00:00Z,53.516566,8.188217,None,12.79|2010-10-13T14:10:00Z,53.516566,8.188217,None,12.82|2010-10-13T14:20:00Z,53.516566,8.188217,None,12.84|2010-10-13T14:30:00Z,53.516566,8.188217,None,12.87|2010-10-13T14:40:00Z,53.516566,8.188217,None,12.89|2010-10-13T14:50:00Z,53.516566,8.188217,None,12.9|2010-10-13T15:00:00Z,53.516566,8.188217,None,12.91|2010-10-13T15:10:00Z,53.516566,8.188217,None,12.91|2010-10-13T15:20:00Z,53.516566,8.188217,None,12.93|2010-10-13T15:30:00Z,53.516566,8.188217,None,12.93|2010-10-13T15:40:00Z,53.516566,8.188217,None,12.93|2010-10-13T15:50:00Z,53.516566,8.188217,None,12.93|2010-10-13T16:00:00Z,53.516566,8.188217,None,12.92|2010-10-13T16:10:00Z,53.516566,8.188217,None,12.91|2010-10-13T16:20:00Z,53.516566,8.188217,None,12.89|2010-10-13T16:30:00Z,53.516566,8.188217,None,12.79|2010-10-13T16:40:00Z,53.516566,8.188217,None,12.72|2010-10-13T16:50:00Z,53.516566,8.188217,None,12.7|2010-10-13T17:00:00Z,53.516566,8.188217,None,12.71|2010-10-13T17:10:00Z,53.516566,8.188217,None,12.84|2010-10-13T17:20:00Z,53.516566,8.188217,None,12.9|2010-10-13T17:30:00Z,53.516566,8.188217,None,12.9|2010-10-13T17:40:00Z,53.516566,8.188217,None,12.94|2010-10-13T17:50:00Z,53.516566,8.188217,None,12.87|2010-10-13T18:00:00Z,53.516566,8.188217,None,12.74|2010-10-13T18:10:00Z,53.516566,8.188217,None,12.7|2010-10-13T18:20:00Z,53.516566,8.188217,None,12.63|2010-10-13T18:30:00Z,53.516566,8.188217,None,12.56|2010-10-13T18:40:00Z,53.516566,8.188217,None,12.51|2010-10-13T18:50:00Z,53.516566,8.188217,None,12.45|2010-10-13T19:00:00Z,53.516566,8.188217,None,12.42|2010-10-13T19:10:00Z,53.516566,8.188217,None,12.41|2010-10-13T19:20:00Z,53.516566,8.188217,None,12.4|2010-10-13T19:30:00Z,53.516566,8.188217,None,12.39|2010-10-13T19:40:00Z,53.516566,8.188217,None,12.39|2010-10-13T19:50:00Z,53.516566,8.188217,None,12.39|2010-10-13T20:00:00Z,53.516566,8.188217,None,12.4|2010-10-13T20:10:00Z,53.516566,8.188217,None,12.4|2010-10-13T20:20:00Z,53.516566,8.188217,None,12.42|2010-10-13T20:30:00Z,53.516566,8.188217,None,12.38|2010-10-13T20:40:00Z,53.516566,8.188217,None,12.31|2010-10-13T20:50:00Z,53.516566,8.188217,None,12.41|2010-10-13T21:00:00Z,53.516566,8.188217,None,12.51|2010-10-13T21:10:00Z,53.516566,8.188217,None,12.5|2010-10-13T21:20:00Z,53.516566,8.188217,None,12.52|2010-10-13T21:30:00Z,53.516566,8.188217,None,12.56|2010-10-13T21:40:00Z,53.516566,8.188217,None,12.54|2010-10-13T21:50:00Z,53.516566,8.188217,None,12.51|2010-10-13T22:00:00Z,53.516566,8.188217,None,12.5|2010-10-13T22:10:00Z,53.516566,8.188217,None,12.42|2010-10-13T22:20:00Z,53.516566,8.188217,None,12.33|2010-10-13T22:30:00Z,53.516566,8.188217,None,12.35|2010-10-13T22:40:00Z,53.516566,8.188217,None,12.44|2010-10-13T22:50:00Z,53.516566,8.188217,None,12.46|2010-10-13T23:00:00Z,53.516566,8.188217,None,12.56|2010-10-13T23:10:00Z,53.516566,8.188217,None,12.58|2010-10-13T23:20:00Z,53.516566,8.188217,None,12.47|2010-10-13T23:30:00Z,53.516566,8.188217,None,12.41|2010-10-13T23:40:00Z,53.516566,8.188217,None,12.4|2010-10-13T23:50:00Z,53.516566,8.188217,None,12.49|2010-10-14T00:00:00Z,53.516566,8.188217,None,12.45|2010-10-14T00:10:00Z,53.516566,8.188217,None,12.51|2010-10-14T00:20:00Z,53.516566,8.188217,None,12.55|2010-10-14T00:30:00Z,53.516566,8.188217,None,12.6|2010-10-14T00:40:00Z,53.516566,8.188217,None,12.61|2010-10-14T00:50:00Z,53.516566,8.188217,None,12.63|2010-10-14T01:00:00Z,53.516566,8.188217,None,12.63|2010-10-14T01:10:00Z,53.516566,8.188217,None,12.63|2010-10-14T01:20:00Z,53.516566,8.188217,None,12.63|2010-10-14T01:30:00Z,53.516566,8.188217,None,12.67|2010-10-14T01:40:00Z,53.516566,8.188217,None,12.71|2010-10-14T01:50:00Z,53.516566,8.188217,None,12.74|2010-10-14T02:00:00Z,53.516566,8.188217,None,12.76|2010-10-14T02:10:00Z,53.516566,8.188217,None,12.79|2010-10-14T02:20:00Z,53.516566,8.188217,None,12.81|2010-10-14T02:30:00Z,53.516566,8.188217,None,12.83|2010-10-14T02:40:00Z,53.516566,8.188217,None,12.58|2010-10-14T02:50:00Z,53.516566,8.188217,None,12.87|2010-10-14T03:00:00Z,53.516566,8.188217,None,12.91|2010-10-14T03:10:00Z,53.516566,8.188217,None,12.96|2010-10-14T03:20:00Z,53.516566,8.188217,None,13.02|2010-10-14T03:30:00Z,53.516566,8.188217,None,13.07|2010-10-14T03:40:00Z,53.516566,8.188217,None,13.1|2010-10-14T03:50:00Z,53.516566,8.188217,None,13.1|2010-10-14T04:00:00Z,53.516566,8.188217,None,13.1|2010-10-14T04:10:00Z,53.516566,8.188217,None,13.09|2010-10-14T04:20:00Z,53.516566,8.188217,None,13.08|2010-10-14T04:30:00Z,53.516566,8.188217,None,12.99|2010-10-14T04:40:00Z,53.516566,8.188217,None,12.86|2010-10-14T04:50:00Z,53.516566,8.188217,None,12.77|2010-10-14T05:00:00Z,53.516566,8.188217,None,12.65|2010-10-14T05:10:00Z,53.516566,8.188217,None,12.53|2010-10-14T05:20:00Z,53.516566,8.188217,None,12.46|2010-10-14T05:30:00Z,53.516566,8.188217,None,12.32|2010-10-14T05:40:00Z,53.516566,8.188217,None,12.28|2010-10-14T05:50:00Z,53.516566,8.188217,None,12.25|2010-10-14T06:00:00Z,53.516566,8.188217,None,12.23|2010-10-14T06:10:00Z,53.516566,8.188217,None,12.25|2010-10-14T06:20:00Z,53.516566,8.188217,None,12.35|2010-10-14T06:30:00Z,53.516566,8.188217,None,12.45|2010-10-14T06:40:00Z,53.516566,8.188217,None,12.47|2010-10-14T06:50:00Z,53.516566,8.188217,None,12.41|2010-10-14T07:00:00Z,53.516566,8.188217,None,12.35|2010-10-14T07:10:00Z,53.516566,8.188217,None,12.29|2010-10-14T07:20:00Z,53.516566,8.188217,None,12.31|2010-10-14T07:30:00Z,53.516566,8.188217,None,12.34|2010-10-14T07:40:00Z,53.516566,8.188217,None,12.37|2010-10-14T07:50:00Z,53.516566,8.188217,None,12.38|2010-10-14T08:00:00Z,53.516566,8.188217,None,12.39|2010-10-14T08:10:00Z,53.516566,8.188217,None,12.39|2010-10-14T08:20:00Z,53.516566,8.188217,None,12.39|2010-10-14T08:30:00Z,53.516566,8.188217,None,12.37|2010-10-14T08:40:00Z,53.516566,8.188217,None,12.35|2010-10-14T08:50:00Z,53.516566,8.188217,None,12.33|2010-10-14T09:00:00Z,53.516566,8.188217,None,12.32|2010-10-14T09:10:00Z,53.516566,8.188217,None,12.37|2010-10-14T09:20:00Z,53.516566,8.188217,None,12.43|2010-10-14T09:30:00Z,53.516566,8.188217,None,12.39|2010-10-14T09:40:00Z,53.516566,8.188217,None,12.3|2010-10-14T09:50:00Z,53.516566,8.188217,None,12.27|2010-10-14T10:00:00Z,53.516566,8.188217,None,12.25|2010-10-14T10:10:00Z,53.516566,8.188217,None,12.28|2010-10-14T10:20:00Z,53.516566,8.188217,None,12.3|2010-10-14T10:30:00Z,53.516566,8.188217,None,12.3|2010-10-14T10:40:00Z,53.516566,8.188217,None,12.32|2010-10-14T10:50:00Z,53.516566,8.188217,None,12.36|2010-10-14T11:00:00Z,53.516566,8.188217,None,12.09|2010-10-14T11:10:00Z,53.516566,8.188217,None,12.08|2010-10-14T11:20:00Z,53.516566,8.188217,None,12.27|2010-10-14T11:30:00Z,53.516566,8.188217,None,12.36|2010-10-14T11:40:00Z,53.516566,8.188217,None,12.37|2010-10-14T11:50:00Z,53.516566,8.188217,None,12.33|2010-10-14T12:00:00Z,53.516566,8.188217,None,12.33|2010-10-14T12:10:00Z,53.516566,8.188217,None,12.31|2010-10-14T12:20:00Z,53.516566,8.188217,None,12.31|2010-10-14T12:30:00Z,53.516566,8.188217,None,12.32|2010-10-14T12:40:00Z,53.516566,8.188217,None,12.33|2010-10-14T12:50:00Z,53.516566,8.188217,None,12.39|2010-10-14T13:00:00Z,53.516566,8.188217,None,12.38|2010-10-14T13:10:00Z,53.516566,8.188217,None,12.39|2010-10-14T13:20:00Z,53.516566,8.188217,None,12.46|2010-10-14T13:30:00Z,53.516566,8.188217,None,12.54|2010-10-14T13:40:00Z,53.516566,8.188217,None,12.58|2010-10-14T13:50:00Z,53.516566,8.188217,None,12.59|2010-10-14T14:00:00Z,53.516566,8.188217,None,12.57|2010-10-14T14:10:00Z,53.516566,8.188217,None,12.55|2010-10-14T14:20:00Z,53.516566,8.188217,None,12.54|2010-10-14T14:30:00Z,53.516566,8.188217,None,12.57|2010-10-14T14:40:00Z,53.516566,8.188217,None,12.6|2010-10-14T14:50:00Z,53.516566,8.188217,None,12.63|2010-10-14T15:00:00Z,53.516566,8.188217,None,12.64|2010-10-14T15:10:00Z,53.516566,8.188217,None,12.66|2010-10-14T15:20:00Z,53.516566,8.188217,None,12.68|2010-10-14T15:30:00Z,53.516566,8.188217,None,12.69|2010-10-14T15:40:00Z,53.516566,8.188217,None,12.72|2010-10-14T15:50:00Z,53.516566,8.188217,None,12.73|2010-10-14T16:00:00Z,53.516566,8.188217,None,12.74|2010-10-14T16:10:00Z,53.516566,8.188217,None,12.74|2010-10-14T16:20:00Z,53.516566,8.188217,None,12.73|2010-10-14T16:30:00Z,53.516566,8.188217,None,12.73|2010-10-14T16:40:00Z,53.516566,8.188217,None,12.72|2010-10-14T16:50:00Z,53.516566,8.188217,None,12.69|2010-10-14T17:00:00Z,53.516566,8.188217,None,12.64|2010-10-14T17:10:00Z,53.516566,8.188217,None,12.59|2010-10-14T17:20:00Z,53.516566,8.188217,None,12.54|2010-10-14T17:30:00Z,53.516566,8.188217,None,12.56|2010-10-14T17:40:00Z,53.516566,8.188217,None,12.2|2010-10-14T17:50:00Z,53.516566,8.188217,None,12.37|2010-10-14T18:00:00Z,53.516566,8.188217,None,12.32|2010-10-14T18:10:00Z,53.516566,8.188217,None,12.32|2010-10-14T18:20:00Z,53.516566,8.188217,None,12.33|2010-10-14T18:30:00Z,53.516566,8.188217,None,12.34|2010-10-14T18:40:00Z,53.516566,8.188217,None,12.32|2010-10-14T18:50:00Z,53.516566,8.188217,None,12.3|2010-10-14T19:00:00Z,53.516566,8.188217,None,12.28|2010-10-14T19:10:00Z,53.516566,8.188217,None,12.25|2010-10-14T19:20:00Z,53.516566,8.188217,None,12.25|2010-10-14T19:30:00Z,53.516566,8.188217,None,12.28|2010-10-14T19:40:00Z,53.516566,8.188217,None,12.32|2010-10-14T19:50:00Z,53.516566,8.188217,None,12.33|2010-10-14T20:00:00Z,53.516566,8.188217,None,12.34|2010-10-14T20:10:00Z,53.516566,8.188217,None,12.33|2010-10-14T20:20:00Z,53.516566,8.188217,None,12.32|2010-10-14T20:30:00Z,53.516566,8.188217,None,12.32|2010-10-14T20:40:00Z,53.516566,8.188217,None,12.31|2010-10-14T20:50:00Z,53.516566,8.188217,None,12.31|2010-10-14T21:00:00Z,53.516566,8.188217,None,12.3|2010-10-14T21:10:00Z,53.516566,8.188217,None,12.28|2010-10-14T21:20:00Z,53.516566,8.188217,None,12.25|2010-10-14T21:30:00Z,53.516566,8.188217,None,12.18|2010-10-14T21:40:00Z,53.516566,8.188217,None,12.19|2010-10-14T21:50:00Z,53.516566,8.188217,None,12.21|2010-10-14T22:00:00Z,53.516566,8.188217,None,12.25|2010-10-14T22:10:00Z,53.516566,8.188217,None,12.27|2010-10-14T22:20:00Z,53.516566,8.188217,None,12.24|2010-10-14T22:30:00Z,53.516566,8.188217,None,12.19|2010-10-14T22:40:00Z,53.516566,8.188217,None,12.16|2010-10-14T22:50:00Z,53.516566,8.188217,None,12.21|2010-10-14T23:00:00Z,53.516566,8.188217,None,12.19|2010-10-14T23:10:00Z,53.516566,8.188217,None,12.17|2010-10-14T23:20:00Z,53.516566,8.188217,None,12.19|2010-10-14T23:30:00Z,53.516566,8.188217,None,12.23|2010-10-14T23:40:00Z,53.516566,8.188217,None,12.27|2010-10-14T23:50:00Z,53.516566,8.188217,None,12.32|2010-10-15T00:00:00Z,53.516566,8.188217,None,12.31|2010-10-15T00:10:00Z,53.516566,8.188217,None,12.29|2010-10-15T00:20:00Z,53.516566,8.188217,None,12.25|2010-10-15T00:30:00Z,53.516566,8.188217,None,12.27|2010-10-15T00:40:00Z,53.516566,8.188217,None,12.27|2010-10-15T00:50:00Z,53.516566,8.188217,None,12.33|2010-10-15T01:00:00Z,53.516566,8.188217,None,12.43|2010-10-15T01:10:00Z,53.516566,8.188217,None,12.48|2010-10-15T01:20:00Z,53.516566,8.188217,None,12.51|2010-10-15T01:30:00Z,53.516566,8.188217,None,12.44|2010-10-15T01:40:00Z,53.516566,8.188217,None,12.45|2010-10-15T01:50:00Z,53.516566,8.188217,None,12.48|2010-10-15T02:00:00Z,53.516566,8.188217,None,12.49|2010-10-15T02:10:00Z,53.516566,8.188217,None,12.49|2010-10-15T02:20:00Z,53.516566,8.188217,None,12.49|2010-10-15T02:30:00Z,53.516566,8.188217,None,12.52|2010-10-15T02:40:00Z,53.516566,8.188217,None,12.55|2010-10-15T02:50:00Z,53.516566,8.188217,None,12.57|2010-10-15T03:00:00Z,53.516566,8.188217,None,12.58|2010-10-15T03:10:00Z,53.516566,8.188217,None,12.6|2010-10-15T03:20:00Z,53.516566,8.188217,None,12.62|2010-10-15T03:30:00Z,53.516566,8.188217,None,12.63|2010-10-15T03:40:00Z,53.516566,8.188217,None,12.63|2010-10-15T03:50:00Z,53.516566,8.188217,None,12.64|2010-10-15T04:00:00Z,53.516566,8.188217,None,12.65|2010-10-15T04:10:00Z,53.516566,8.188217,None,12.65|2010-10-15T04:20:00Z,53.516566,8.188217,None,12.65|2010-10-15T04:30:00Z,53.516566,8.188217,None,12.65|2010-10-15T04:40:00Z,53.516566,8.188217,None,12.64|2010-10-15T04:50:00Z,53.516566,8.188217,None,12.63|2010-10-15T05:00:00Z,53.516566,8.188217,None,12.61|2010-10-15T05:10:00Z,53.516566,8.188217,None,12.55|2010-10-15T05:20:00Z,53.516566,8.188217,None,12.48|2010-10-15T05:30:00Z,53.516566,8.188217,None,12.44|2010-10-15T05:40:00Z,53.516566,8.188217,None,12.41|2010-10-15T05:50:00Z,53.516566,8.188217,None,12.35|2010-10-15T06:00:00Z,53.516566,8.188217,None,12.21|2010-10-15T06:10:00Z,53.516566,8.188217,None,12.16|2010-10-15T06:20:00Z,53.516566,8.188217,None,12.1|2010-10-15T06:30:00Z,53.516566,8.188217,None,12.09|2010-10-15T06:40:00Z,53.516566,8.188217,None,12.11|2010-10-15T06:50:00Z,53.516566,8.188217,None,12.17|2010-10-15T07:00:00Z,53.516566,8.188217,None,12.19|2010-10-15T07:10:00Z,53.516566,8.188217,None,12.18|2010-10-15T07:20:00Z,53.516566,8.188217,None,12.19|2010-10-15T07:30:00Z,53.516566,8.188217,None,12.21|2010-10-15T07:40:00Z,53.516566,8.188217,None,12.28|2010-10-15T07:50:00Z,53.516566,8.188217,None,12.35|2010-10-15T08:00:00Z,53.516566,8.188217,None,12.37|2010-10-15T08:10:00Z,53.516566,8.188217,None,12.39|2010-10-15T08:20:00Z,53.516566,8.188217,None,12.39|2010-10-15T08:30:00Z,53.516566,8.188217,None,12.39|2010-10-15T08:40:00Z,53.516566,8.188217,None,12.39|2010-10-15T08:50:00Z,53.516566,8.188217,None,12.38|2010-10-15T09:00:00Z,53.516566,8.188217,None,12.38|2010-10-15T09:10:00Z,53.516566,8.188217,None,12.37|2010-10-15T09:20:00Z,53.516566,8.188217,None,12.37|2010-10-15T09:30:00Z,53.516566,8.188217,None,12.37|2010-10-15T09:40:00Z,53.516566,8.188217,None,12.38|2010-10-15T09:50:00Z,53.516566,8.188217,None,12.36|2010-10-15T10:00:00Z,53.516566,8.188217,None,12.32|2010-10-15T10:10:00Z,53.516566,8.188217,None,12.3|2010-10-15T10:20:00Z,53.516566,8.188217,None,12.26|2010-10-15T10:30:00Z,53.516566,8.188217,None,12.24|2010-10-15T10:40:00Z,53.516566,8.188217,None,12.23|2010-10-15T10:50:00Z,53.516566,8.188217,None,12.22|2010-10-15T11:00:00Z,53.516566,8.188217,None,12.22|2010-10-15T11:10:00Z,53.516566,8.188217,None,12.22|2010-10-15T11:20:00Z,53.516566,8.188217,None,12.27|2010-10-15T11:30:00Z,53.516566,8.188217,None,12.25|2010-10-15T11:40:00Z,53.516566,8.188217,None,12.2|2010-10-15T11:50:00Z,53.516566,8.188217,None,12.2|2010-10-15T12:00:00Z,53.516566,8.188217,None,12.19|2010-10-15T12:10:00Z,53.516566,8.188217,None,12.2|2010-10-15T12:20:00Z,53.516566,8.188217,None,12.22|2010-10-15T12:30:00Z,53.516566,8.188217,None,12.23|2010-10-15T12:40:00Z,53.516566,8.188217,None,11.99|2010-10-15T12:50:00Z,53.516566,8.188217,None,12.23|2010-10-15T13:00:00Z,53.516566,8.188217,None,12.21|2010-10-15T13:10:00Z,53.516566,8.188217,None,12.2|2010-10-15T13:20:00Z,53.516566,8.188217,None,12.24|2010-10-15T13:30:00Z,53.516566,8.188217,None,12.25|2010-10-15T13:40:00Z,53.516566,8.188217,None,12.29|2010-10-15T13:50:00Z,53.516566,8.188217,None,12.33|2010-10-15T14:00:00Z,53.516566,8.188217,None,12.38|2010-10-15T14:10:00Z,53.516566,8.188217,None,12.42|2010-10-15T14:20:00Z,53.516566,8.188217,None,12.45|2010-10-15T14:30:00Z,53.516566,8.188217,None,12.46|2010-10-15T14:40:00Z,53.516566,8.188217,None,12.45|2010-10-15T14:50:00Z,53.516566,8.188217,None,12.44|2010-10-15T15:00:00Z,53.516566,8.188217,None,12.42|2010-10-15T15:10:00Z,53.516566,8.188217,None,12.42|2010-10-15T15:20:00Z,53.516566,8.188217,None,12.42|2010-10-15T15:30:00Z,53.516566,8.188217,None,12.43|2010-10-15T15:40:00Z,53.516566,8.188217,None,12.46|2010-10-15T15:50:00Z,53.516566,8.188217,None,12.49|2010-10-15T16:00:00Z,53.516566,8.188217,None,12.5|2010-10-15T16:10:00Z,53.516566,8.188217,None,12.52|2010-10-15T16:20:00Z,53.516566,8.188217,None,12.52|2010-10-15T16:30:00Z,53.516566,8.188217,None,12.53|2010-10-15T16:40:00Z,53.516566,8.188217,None,12.53|2010-10-15T16:50:00Z,53.516566,8.188217,None,12.53|2010-10-15T17:00:00Z,53.516566,8.188217,None,12.53|2010-10-15T17:10:00Z,53.516566,8.188217,None,12.52|2010-10-15T17:20:00Z,53.516566,8.188217,None,12.5|2010-10-15T17:30:00Z,53.516566,8.188217,None,12.21|2010-10-15T17:40:00Z,53.516566,8.188217,None,12.43|2010-10-15T17:50:00Z,53.516566,8.188217,None,12.38|2010-10-15T18:00:00Z,53.516566,8.188217,None,12.36|2010-10-15T18:10:00Z,53.516566,8.188217,None,12.38|2010-10-15T18:20:00Z,53.516566,8.188217,None,12.37|2010-10-15T18:30:00Z,53.516566,8.188217,None,12.3|2010-10-15T18:40:00Z,53.516566,8.188217,None,12.21|2010-10-15T18:50:00Z,53.516566,8.188217,None,12.18|2010-10-15T19:00:00Z,53.516566,8.188217,None,12.17|2010-10-15T19:10:00Z,53.516566,8.188217,None,12.17|2010-10-15T19:20:00Z,53.516566,8.188217,None,12.17|2010-10-15T19:30:00Z,53.516566,8.188217,None,12.22|2010-10-15T19:40:00Z,53.516566,8.188217,None,12.28|2010-10-15T19:50:00Z,53.516566,8.188217,None,12.3|2010-10-15T20:00:00Z,53.516566,8.188217,None,12.31|2010-10-15T20:10:00Z,53.516566,8.188217,None,12.31|2010-10-15T20:20:00Z,53.516566,8.188217,None,12.3|2010-10-15T20:30:00Z,53.516566,8.188217,None,12.29|2010-10-15T20:40:00Z,53.516566,8.188217,None,12.27|2010-10-15T20:50:00Z,53.516566,8.188217,None,12.24|2010-10-15T21:00:00Z,53.516566,8.188217,None,12.21|2010-10-15T21:10:00Z,53.516566,8.188217,None,12.2|2010-10-15T21:20:00Z,53.516566,8.188217,None,12.19|2010-10-15T21:30:00Z,53.516566,8.188217,None,12.18|2010-10-15T21:40:00Z,53.516566,8.188217,None,12.19|2010-10-15T21:50:00Z,53.516566,8.188217,None,12.2|2010-10-15T22:00:00Z,53.516566,8.188217,None,12.21|2010-10-15T22:10:00Z,53.516566,8.188217,None,12.19|2010-10-15T22:20:00Z,53.516566,8.188217,None,12.12|2010-10-15T22:30:00Z,53.516566,8.188217,None,12.05|2010-10-15T22:40:00Z,53.516566,8.188217,None,12.04|2010-10-15T22:50:00Z,53.516566,8.188217,None,12.07|2010-10-15T23:00:00Z,53.516566,8.188217,None,12.07|2010-10-15T23:10:00Z,53.516566,8.188217,None,12.09|2010-10-15T23:20:00Z,53.516566,8.188217,None,12.1|2010-10-15T23:30:00Z,53.516566,8.188217,None,12.06|2010-10-15T23:40:00Z,53.516566,8.188217,None,12.13|2010-10-15T23:50:00Z,53.516566,8.188217,None,12.11|2010-10-16T00:00:00Z,53.516566,8.188217,None,12.06|2010-10-16T00:10:00Z,53.516566,8.188217,None,12.11|2010-10-16T00:20:00Z,53.516566,8.188217,None,12.13|2010-10-16T00:30:00Z,53.516566,8.188217,None,12.13|2010-10-16T00:40:00Z,53.516566,8.188217,None,12.17|2010-10-16T00:50:00Z,53.516566,8.188217,None,12.14|2010-10-16T01:00:00Z,53.516566,8.188217,None,12.11|2010-10-16T01:10:00Z,53.516566,8.188217,None,12.13|2010-10-16T01:20:00Z,53.516566,8.188217,None,12.16|2010-10-16T01:30:00Z,53.516566,8.188217,None,12.2|2010-10-16T01:40:00Z,53.516566,8.188217,None,12.21|2010-10-16T01:50:00Z,53.516566,8.188217,None,12.27|2010-10-16T02:00:00Z,53.516566,8.188217,None,12.3|2010-10-16T02:10:00Z,53.516566,8.188217,None,12.25|2010-10-16T02:20:00Z,53.516566,8.188217,None,12.28|2010-10-16T02:30:00Z,53.516566,8.188217,None,12.36|2010-10-16T02:40:00Z,53.516566,8.188217,None,12.36|2010-10-16T02:50:00Z,53.516566,8.188217,None,12.35|2010-10-16T03:00:00Z,53.516566,8.188217,None,12.36|2010-10-16T03:10:00Z,53.516566,8.188217,None,12.37|2010-10-16T03:20:00Z,53.516566,8.188217,None,12.36|2010-10-16T03:30:00Z,53.516566,8.188217,None,12.36|2010-10-16T03:40:00Z,53.516566,8.188217,None,12.39|2010-10-16T03:50:00Z,53.516566,8.188217,None,12.41|2010-10-16T04:00:00Z,53.516566,8.188217,None,12.42|2010-10-16T04:10:00Z,53.516566,8.188217,None,12.42|2010-10-16T04:20:00Z,53.516566,8.188217,None,12.44|2010-10-16T04:30:00Z,53.516566,8.188217,None,12.46|2010-10-16T04:40:00Z,53.516566,8.188217,None,12.47|2010-10-16T04:50:00Z,53.516566,8.188217,None,12.48|2010-10-16T05:00:00Z,53.516566,8.188217,None,12.48|2010-10-16T05:10:00Z,53.516566,8.188217,None,12.47|2010-10-16T05:20:00Z,53.516566,8.188217,None,12.46|2010-10-16T05:30:00Z,53.516566,8.188217,None,12.42|2010-10-16T05:40:00Z,53.516566,8.188217,None,12.36|2010-10-16T05:50:00Z,53.516566,8.188217,None,12.3|2010-10-16T06:00:00Z,53.516566,8.188217,None,12.25|2010-10-16T06:10:00Z,53.516566,8.188217,None,12.19|2010-10-16T06:20:00Z,53.516566,8.188217,None,12.15|2010-10-16T06:30:00Z,53.516566,8.188217,None,12.12|2010-10-16T06:40:00Z,53.516566,8.188217,None,12.05|2010-10-16T06:50:00Z,53.516566,8.188217,None,11.99|2010-10-16T07:00:00Z,53.516566,8.188217,None,11.9|2010-10-16T07:10:00Z,53.516566,8.188217,None,11.82|2010-10-16T07:20:00Z,53.516566,8.188217,None,11.73|2010-10-16T07:30:00Z,53.516566,8.188217,None,11.63|2010-10-16T07:40:00Z,53.516566,8.188217,None,11.65|2010-10-16T07:50:00Z,53.516566,8.188217,None,11.67|2010-10-16T08:00:00Z,53.516566,8.188217,None,11.63|2010-10-16T08:10:00Z,53.516566,8.188217,None,11.62|2010-10-16T08:20:00Z,53.516566,8.188217,None,11.7|2010-10-16T08:30:00Z,53.516566,8.188217,None,11.76|2010-10-16T08:40:00Z,53.516566,8.188217,None,11.77|2010-10-16T08:50:00Z,53.516566,8.188217,None,11.78|2010-10-16T09:00:00Z,53.516566,8.188217,None,11.78|2010-10-16T09:10:00Z,53.516566,8.188217,None,11.76|2010-10-16T09:20:00Z,53.516566,8.188217,None,11.76|2010-10-16T09:30:00Z,53.516566,8.188217,None,11.76|2010-10-16T09:40:00Z,53.516566,8.188217,None,11.75|2010-10-16T09:50:00Z,53.516566,8.188217,None,11.72|2010-10-16T10:00:00Z,53.516566,8.188217,None,11.71|2010-10-16T10:10:00Z,53.516566,8.188217,None,11.73|2010-10-16T10:20:00Z,53.516566,8.188217,None,11.7|2010-10-16T10:30:00Z,53.516566,8.188217,None,11.66|2010-10-16T10:40:00Z,53.516566,8.188217,None,11.6|2010-10-16T10:50:00Z,53.516566,8.188217,None,11.62|2010-10-16T11:00:00Z,53.516566,8.188217,None,11.64|2010-10-16T11:10:00Z,53.516566,8.188217,None,11.69|2010-10-16T11:20:00Z,53.516566,8.188217,None,11.73|2010-10-16T11:30:00Z,53.516566,8.188217,None,11.71|2010-10-16T11:40:00Z,53.516566,8.188217,None,11.7|2010-10-16T11:50:00Z,53.516566,8.188217,None,11.71|2010-10-16T12:00:00Z,53.516566,8.188217,None,11.68|2010-10-16T12:10:00Z,53.516566,8.188217,None,11.7|2010-10-16T12:20:00Z,53.516566,8.188217,None,11.76|2010-10-16T12:30:00Z,53.516566,8.188217,None,11.84|2010-10-16T12:40:00Z,53.516566,8.188217,None,11.84|2010-10-16T12:50:00Z,53.516566,8.188217,None,11.89|2010-10-16T13:00:00Z,53.516566,8.188217,None,11.83|2010-10-16T13:10:00Z,53.516566,8.188217,None,11.81|2010-10-16T13:20:00Z,53.516566,8.188217,None,11.81|2010-10-16T13:30:00Z,53.516566,8.188217,None,11.83|2010-10-16T13:40:00Z,53.516566,8.188217,None,11.82|2010-10-16T13:50:00Z,53.516566,8.188217,None,11.82|2010-10-16T14:00:00Z,53.516566,8.188217,None,11.8|2010-10-16T14:10:00Z,53.516566,8.188217,None,11.77|2010-10-16T14:20:00Z,53.516566,8.188217,None,11.79|2010-10-16T14:30:00Z,53.516566,8.188217,None,11.84|2010-10-16T14:40:00Z,53.516566,8.188217,None,11.84|2010-10-16T14:50:00Z,53.516566,8.188217,None,11.85|2010-10-16T15:00:00Z,53.516566,8.188217,None,11.89|2010-10-16T15:10:00Z,53.516566,8.188217,None,11.9|2010-10-16T15:20:00Z,53.516566,8.188217,None,11.7|2010-10-16T15:30:00Z,53.516566,8.188217,None,11.79|2010-10-16T15:40:00Z,53.516566,8.188217,None,11.71|2010-10-16T15:50:00Z,53.516566,8.188217,None,11.76|2010-10-16T16:00:00Z,53.516566,8.188217,None,11.85|2010-10-16T16:10:00Z,53.516566,8.188217,None,11.91|2010-10-16T16:20:00Z,53.516566,8.188217,None,11.93|2010-10-16T16:30:00Z,53.516566,8.188217,None,11.95|2010-10-16T16:40:00Z,53.516566,8.188217,None,11.98|2010-10-16T16:50:00Z,53.516566,8.188217,None,12.0|2010-10-16T17:00:00Z,53.516566,8.188217,None,12.01|2010-10-16T17:10:00Z,53.516566,8.188217,None,12.01|2010-10-16T17:20:00Z,53.516566,8.188217,None,12.02|2010-10-16T17:30:00Z,53.516566,8.188217,None,12.02|2010-10-16T17:40:00Z,53.516566,8.188217,None,12.03|2010-10-16T17:50:00Z,53.516566,8.188217,None,12.03|2010-10-16T18:00:00Z,53.516566,8.188217,None,12.03|2010-10-16T18:10:00Z,53.516566,8.188217,None,12.04|2010-10-16T18:20:00Z,53.516566,8.188217,None,11.77|2010-10-16T18:30:00Z,53.516566,8.188217,None,12.0|2010-10-16T18:40:00Z,53.516566,8.188217,None,11.94|2010-10-16T18:50:00Z,53.516566,8.188217,None,11.86|2010-10-16T19:00:00Z,53.516566,8.188217,None,11.8|2010-10-16T19:10:00Z,53.516566,8.188217,None,11.75|2010-10-16T19:20:00Z,53.516566,8.188217,None,11.69|2010-10-16T19:30:00Z,53.516566,8.188217,None,11.59|2010-10-16T19:40:00Z,53.516566,8.188217,None,11.36|2010-10-16T19:50:00Z,53.516566,8.188217,None,11.3|2010-10-16T20:00:00Z,53.516566,8.188217,None,11.31|2010-10-16T20:10:00Z,53.516566,8.188217,None,11.27|2010-10-16T20:20:00Z,53.516566,8.188217,None,11.29|2010-10-16T20:30:00Z,53.516566,8.188217,None,11.24|2010-10-16T20:40:00Z,53.516566,8.188217,None,11.13|2010-10-16T20:50:00Z,53.516566,8.188217,None,11.08|2010-10-16T21:00:00Z,53.516566,8.188217,None,11.05|2010-10-16T21:10:00Z,53.516566,8.188217,None,11.05|2010-10-16T21:20:00Z,53.516566,8.188217,None,11.05|2010-10-16T21:30:00Z,53.516566,8.188217,None,11.04|2010-10-16T21:40:00Z,53.516566,8.188217,None,11.02|2010-10-16T21:50:00Z,53.516566,8.188217,None,11.03|2010-10-16T22:00:00Z,53.516566,8.188217,None,11.01|2010-10-16T22:10:00Z,53.516566,8.188217,None,11.01|2010-10-16T22:20:00Z,53.516566,8.188217,None,11.01|2010-10-16T22:30:00Z,53.516566,8.188217,None,10.87|2010-10-16T22:40:00Z,53.516566,8.188217,None,10.79|2010-10-16T22:50:00Z,53.516566,8.188217,None,10.74|2010-10-16T23:00:00Z,53.516566,8.188217,None,10.73|2010-10-16T23:10:00Z,53.516566,8.188217,None,10.85|2010-10-16T23:20:00Z,53.516566,8.188217,None,10.99|2010-10-16T23:30:00Z,53.516566,8.188217,None,11.06|2010-10-16T23:40:00Z,53.516566,8.188217,None,11.09|2010-10-16T23:50:00Z,53.516566,8.188217,None,11.05|2010-10-17T00:00:00Z,53.516566,8.188217,None,10.87|2010-10-17T00:10:00Z,53.516566,8.188217,None,10.78|2010-10-17T00:20:00Z,53.516566,8.188217,None,10.9|2010-10-17T00:30:00Z,53.516566,8.188217,None,11.09|2010-10-17T00:40:00Z,53.516566,8.188217,None,11.22|2010-10-17T00:50:00Z,53.516566,8.188217,None,11.18|2010-10-17T01:00:00Z,53.516566,8.188217,None,11.22|2010-10-17T01:10:00Z,53.516566,8.188217,None,11.28|2010-10-17T01:20:00Z,53.516566,8.188217,None,11.29|2010-10-17T01:30:00Z,53.516566,8.188217,None,11.27|2010-10-17T01:40:00Z,53.516566,8.188217,None,11.27|2010-10-17T01:50:00Z,53.516566,8.188217,None,11.28|2010-10-17T02:00:00Z,53.516566,8.188217,None,11.3|2010-10-17T02:10:00Z,53.516566,8.188217,None,11.31|2010-10-17T02:20:00Z,53.516566,8.188217,None,11.31|2010-10-17T02:30:00Z,53.516566,8.188217,None,11.32|2010-10-17T02:40:00Z,53.516566,8.188217,None,11.32|2010-10-17T02:50:00Z,53.516566,8.188217,None,11.28|2010-10-17T03:00:00Z,53.516566,8.188217,None,11.27|2010-10-17T03:10:00Z,53.516566,8.188217,None,11.32|2010-10-17T03:20:00Z,53.516566,8.188217,None,11.31|2010-10-17T03:30:00Z,53.516566,8.188217,None,11.27|2010-10-17T03:40:00Z,53.516566,8.188217,None,11.24|2010-10-17T03:50:00Z,53.516566,8.188217,None,11.24|2010-10-17T04:00:00Z,53.516566,8.188217,None,11.28|2010-10-17T04:10:00Z,53.516566,8.188217,None,11.31|2010-10-17T04:20:00Z,53.516566,8.188217,None,11.34|2010-10-17T04:30:00Z,53.516566,8.188217,None,11.4|2010-10-17T04:40:00Z,53.516566,8.188217,None,11.48|2010-10-17T04:50:00Z,53.516566,8.188217,None,11.54|2010-10-17T05:00:00Z,53.516566,8.188217,None,11.56|2010-10-17T05:10:00Z,53.516566,8.188217,None,11.57|2010-10-17T05:20:00Z,53.516566,8.188217,None,11.59|2010-10-17T05:30:00Z,53.516566,8.188217,None,11.62|2010-10-17T05:40:00Z,53.516566,8.188217,None,11.64|2010-10-17T05:50:00Z,53.516566,8.188217,None,11.69|2010-10-17T06:00:00Z,53.516566,8.188217,None,11.71|2010-10-17T06:10:00Z,53.516566,8.188217,None,11.72|2010-10-17T06:20:00Z,53.516566,8.188217,None,11.74|2010-10-17T06:30:00Z,53.516566,8.188217,None,11.75|2010-10-17T06:40:00Z,53.516566,8.188217,None,11.75|2010-10-17T06:50:00Z,53.516566,8.188217,None,11.74|2010-10-17T07:00:00Z,53.516566,8.188217,None,11.74|2010-10-17T07:10:00Z,53.516566,8.188217,None,11.73|2010-10-17T07:20:00Z,53.516566,8.188217,None,11.7|2010-10-17T07:30:00Z,53.516566,8.188217,None,11.65|2010-10-17T07:40:00Z,53.516566,8.188217,None,11.59|2010-10-17T07:50:00Z,53.516566,8.188217,None,11.51|2010-10-17T08:00:00Z,53.516566,8.188217,None,11.42|2010-10-17T08:10:00Z,53.516566,8.188217,None,11.33|2010-10-17T08:20:00Z,53.516566,8.188217,None,11.14|2010-10-17T08:30:00Z,53.516566,8.188217,None,11.01|2010-10-17T08:40:00Z,53.516566,8.188217,None,10.99|2010-10-17T08:50:00Z,53.516566,8.188217,None,10.97|2010-10-17T09:00:00Z,53.516566,8.188217,None,10.93|2010-10-17T09:10:00Z,53.516566,8.188217,None,10.83|2010-10-17T09:20:00Z,53.516566,8.188217,None,10.68|2010-10-17T09:30:00Z,53.516566,8.188217,None,10.58|2010-10-17T09:40:00Z,53.516566,8.188217,None,10.52|2010-10-17T09:50:00Z,53.516566,8.188217,None,10.4|2010-10-17T10:00:00Z,53.516566,8.188217,None,10.26|2010-10-17T10:10:00Z,53.516566,8.188217,None,10.19|2010-10-17T10:20:00Z,53.516566,8.188217,None,10.19|2010-10-17T10:30:00Z,53.516566,8.188217,None,10.2|2010-10-17T10:40:00Z,53.516566,8.188217,None,10.22|2010-10-17T10:50:00Z,53.516566,8.188217,None,10.23|2010-10-17T11:00:00Z,53.516566,8.188217,None,10.22|2010-10-17T11:10:00Z,53.516566,8.188217,None,10.2|2010-10-17T11:20:00Z,53.516566,8.188217,None,10.15|2010-10-17T11:30:00Z,53.516566,8.188217,None,10.09|2010-10-17T11:40:00Z,53.516566,8.188217,None,10.09|2010-10-17T11:50:00Z,53.516566,8.188217,None,10.33|2010-10-17T12:00:00Z,53.516566,8.188217,None,10.63|2010-10-17T12:10:00Z,53.516566,8.188217,None,10.47|2010-10-17T12:20:00Z,53.516566,8.188217,None,10.18|2010-10-17T12:30:00Z,53.516566,8.188217,None,10.34|2010-10-17T12:40:00Z,53.516566,8.188217,None,10.36|2010-10-17T12:50:00Z,53.516566,8.188217,None,10.24|2010-10-17T13:00:00Z,53.516566,8.188217,None,10.5|2010-10-17T13:10:00Z,53.516566,8.188217,None,10.68|2010-10-17T13:20:00Z,53.516566,8.188217,None,10.78|2010-10-17T13:30:00Z,53.516566,8.188217,None,10.67|2010-10-17T13:40:00Z,53.516566,8.188217,None,10.62|2010-10-17T13:50:00Z,53.516566,8.188217,None,10.76|2010-10-17T14:00:00Z,53.516566,8.188217,None,11.03|2010-10-17T14:10:00Z,53.516566,8.188217,None,11.12|2010-10-17T14:20:00Z,53.516566,8.188217,None,11.02|2010-10-17T14:30:00Z,53.516566,8.188217,None,11.14|2010-10-17T14:40:00Z,53.516566,8.188217,None,11.16|2010-10-17T14:50:00Z,53.516566,8.188217,None,11.15|2010-10-17T15:00:00Z,53.516566,8.188217,None,11.13|2010-10-17T15:10:00Z,53.516566,8.188217,None,11.11|2010-10-17T15:20:00Z,53.516566,8.188217,None,11.08|2010-10-17T15:30:00Z,53.516566,8.188217,None,11.07|2010-10-17T15:40:00Z,53.516566,8.188217,None,11.06|2010-10-17T15:50:00Z,53.516566,8.188217,None,10.97|2010-10-17T16:00:00Z,53.516566,8.188217,None,10.96|2010-10-17T16:10:00Z,53.516566,8.188217,None,11.06|2010-10-17T16:20:00Z,53.516566,8.188217,None,11.1|2010-10-17T16:30:00Z,53.516566,8.188217,None,11.11|2010-10-17T16:40:00Z,53.516566,8.188217,None,11.14|2010-10-17T16:50:00Z,53.516566,8.188217,None,11.21|2010-10-17T17:00:00Z,53.516566,8.188217,None,11.21|2010-10-17T17:10:00Z,53.516566,8.188217,None,11.24|2010-10-17T17:20:00Z,53.516566,8.188217,None,11.28|2010-10-17T17:30:00Z,53.516566,8.188217,None,11.31|2010-10-17T17:40:00Z,53.516566,8.188217,None,11.34|2010-10-17T17:50:00Z,53.516566,8.188217,None,11.34|2010-10-17T18:00:00Z,53.516566,8.188217,None,11.33|2010-10-17T18:10:00Z,53.516566,8.188217,None,11.31|2010-10-17T18:20:00Z,53.516566,8.188217,None,11.3|2010-10-17T18:30:00Z,53.516566,8.188217,None,11.31|2010-10-17T18:40:00Z,53.516566,8.188217,None,11.32|2010-10-17T18:50:00Z,53.516566,8.188217,None,11.35|2010-10-17T19:00:00Z,53.516566,8.188217,None,11.37|2010-10-17T19:10:00Z,53.516566,8.188217,None,11.39|2010-10-17T19:20:00Z,53.516566,8.188217,None,11.4|2010-10-17T19:30:00Z,53.516566,8.188217,None,11.41|2010-10-17T19:40:00Z,53.516566,8.188217,None,11.41|2010-10-17T19:50:00Z,53.516566,8.188217,None,11.41|2010-10-17T20:00:00Z,53.516566,8.188217,None,11.41|2010-10-17T20:10:00Z,53.516566,8.188217,None,11.38|2010-10-17T20:20:00Z,53.516566,8.188217,None,11.35|2010-10-17T20:30:00Z,53.516566,8.188217,None,11.24|2010-10-17T20:40:00Z,53.516566,8.188217,None,11.19|2010-10-17T20:50:00Z,53.516566,8.188217,None,11.16|2010-10-17T21:00:00Z,53.516566,8.188217,None,11.01|2010-10-17T21:10:00Z,53.516566,8.188217,None,10.81|2010-10-17T21:20:00Z,53.516566,8.188217,None,10.77|2010-10-17T21:30:00Z,53.516566,8.188217,None,10.82|2010-10-17T21:40:00Z,53.516566,8.188217,None,10.76|2010-10-17T21:50:00Z,53.516566,8.188217,None,10.74|2010-10-17T22:00:00Z,53.516566,8.188217,None,10.72|2010-10-17T22:10:00Z,53.516566,8.188217,None,10.72|2010-10-17T22:20:00Z,53.516566,8.188217,None,10.73|2010-10-17T22:30:00Z,53.516566,8.188217,None,10.73|2010-10-17T22:40:00Z,53.516566,8.188217,None,10.72|2010-10-17T22:50:00Z,53.516566,8.188217,None,10.71|2010-10-17T23:00:00Z,53.516566,8.188217,None,10.71|2010-10-17T23:10:00Z,53.516566,8.188217,None,10.72|2010-10-17T23:20:00Z,53.516566,8.188217,None,10.72|2010-10-17T23:30:00Z,53.516566,8.188217,None,10.7|2010-10-17T23:40:00Z,53.516566,8.188217,None,10.68|2010-10-17T23:50:00Z,53.516566,8.188217,None,10.66|2010-10-18T00:00:00Z,53.516566,8.188217,None,10.65|2010-10-18T00:10:00Z,53.516566,8.188217,None,10.66|2010-10-18T00:20:00Z,53.516566,8.188217,None,10.65|2010-10-18T00:30:00Z,53.516566,8.188217,None,10.63|2010-10-18T00:40:00Z,53.516566,8.188217,None,10.61|2010-10-18T00:50:00Z,53.516566,8.188217,None,10.59|2010-10-18T01:00:00Z,53.516566,8.188217,None,10.43|2010-10-18T01:10:00Z,53.516566,8.188217,None,10.27|2010-10-18T01:20:00Z,53.516566,8.188217,None,10.4|2010-10-18T01:30:00Z,53.516566,8.188217,None,10.57|2010-10-18T01:40:00Z,53.516566,8.188217,None,10.58|2010-10-18T01:50:00Z,53.516566,8.188217,None,10.63|2010-10-18T02:00:00Z,53.516566,8.188217,None,10.64|2010-10-18T02:10:00Z,53.516566,8.188217,None,10.54|2010-10-18T02:20:00Z,53.516566,8.188217,None,10.4|2010-10-18T02:30:00Z,53.516566,8.188217,None,10.36|2010-10-18T02:40:00Z,53.516566,8.188217,None,10.65|2010-10-18T02:50:00Z,53.516566,8.188217,None,10.7|2010-10-18T03:00:00Z,53.516566,8.188217,None,10.71|2010-10-18T03:10:00Z,53.516566,8.188217,None,10.77|2010-10-18T03:20:00Z,53.516566,8.188217,None,10.67|2010-10-18T03:30:00Z,53.516566,8.188217,None,10.61|2010-10-18T03:40:00Z,53.516566,8.188217,None,10.62|2010-10-18T03:50:00Z,53.516566,8.188217,None,10.63|2010-10-18T04:00:00Z,53.516566,8.188217,None,10.65|2010-10-18T04:10:00Z,53.516566,8.188217,None,10.73|2010-10-18T04:20:00Z,53.516566,8.188217,None,10.87|2010-10-18T04:30:00Z,53.516566,8.188217,None,10.98|2010-10-18T04:40:00Z,53.516566,8.188217,None,11.06|2010-10-18T04:50:00Z,53.516566,8.188217,None,11.04|2010-10-18T05:00:00Z,53.516566,8.188217,None,11.02|2010-10-18T05:10:00Z,53.516566,8.188217,None,11.03|2010-10-18T05:20:00Z,53.516566,8.188217,None,11.06|2010-10-18T05:30:00Z,53.516566,8.188217,None,11.08|2010-10-18T05:40:00Z,53.516566,8.188217,None,11.12|2010-10-18T05:50:00Z,53.516566,8.188217,None,11.16|2010-10-18T06:00:00Z,53.516566,8.188217,None,11.18|2010-10-18T06:10:00Z,53.516566,8.188217,None,11.19|2010-10-18T06:20:00Z,53.516566,8.188217,None,11.18|2010-10-18T06:30:00Z,53.516566,8.188217,None,11.2|2010-10-18T06:40:00Z,53.516566,8.188217,None,11.21|2010-10-18T06:50:00Z,53.516566,8.188217,None,11.22|2010-10-18T07:00:00Z,53.516566,8.188217,None,11.23|2010-10-18T07:10:00Z,53.516566,8.188217,None,11.25|2010-10-18T07:20:00Z,53.516566,8.188217,None,11.28|2010-10-18T07:30:00Z,53.516566,8.188217,None,11.3|2010-10-18T07:40:00Z,53.516566,8.188217,None,11.33|2010-10-18T07:50:00Z,53.516566,8.188217,None,11.36|2010-10-18T08:00:00Z,53.516566,8.188217,None,11.4|2010-10-18T08:10:00Z,53.516566,8.188217,None,11.42|2010-10-18T08:20:00Z,53.516566,8.188217,None,11.43|2010-10-18T08:30:00Z,53.516566,8.188217,None,11.41|2010-10-18T08:40:00Z,53.516566,8.188217,None,11.3|2010-10-18T08:50:00Z,53.516566,8.188217,None,11.23|2010-10-18T09:00:00Z,53.516566,8.188217,None,11.17|2010-10-18T09:10:00Z,53.516566,8.188217,None,11.13|2010-10-18T09:20:00Z,53.516566,8.188217,None,11.09|2010-10-18T09:30:00Z,53.516566,8.188217,None,11.02|2010-10-18T09:40:00Z,53.516566,8.188217,None,10.87|2010-10-18T09:50:00Z,53.516566,8.188217,None,10.79|2010-10-18T10:00:00Z,53.516566,8.188217,None,10.56|2010-10-18T10:10:00Z,53.516566,8.188217,None,10.38|2010-10-18T10:20:00Z,53.516566,8.188217,None,10.33|2010-10-18T10:30:00Z,53.516566,8.188217,None,10.41|2010-10-18T10:40:00Z,53.516566,8.188217,None,10.57|2010-10-18T10:50:00Z,53.516566,8.188217,None,10.7|2010-10-18T11:00:00Z,53.516566,8.188217,None,10.76|2010-10-18T11:10:00Z,53.516566,8.188217,None,10.79|2010-10-18T11:20:00Z,53.516566,8.188217,None,10.8|2010-10-18T11:30:00Z,53.516566,8.188217,None,10.84|2010-10-18T11:40:00Z,53.516566,8.188217,None,10.87|2010-10-18T11:50:00Z,53.516566,8.188217,None,10.9|2010-10-18T12:00:00Z,53.516566,8.188217,None,10.93|2010-10-18T12:10:00Z,53.516566,8.188217,None,10.95|2010-10-18T12:20:00Z,53.516566,8.188217,None,10.96|2010-10-18T12:30:00Z,53.516566,8.188217,None,10.93|2010-10-18T12:40:00Z,53.516566,8.188217,None,10.9|2010-10-18T12:50:00Z,53.516566,8.188217,None,10.88|2010-10-18T13:00:00Z,53.516566,8.188217,None,10.87|2010-10-18T13:10:00Z,53.516566,8.188217,None,10.86|2010-10-18T13:20:00Z,53.516566,8.188217,None,10.86|2010-10-18T13:30:00Z,53.516566,8.188217,None,10.85|2010-10-18T13:40:00Z,53.516566,8.188217,None,10.84|2010-10-18T13:50:00Z,53.516566,8.188217,None,10.84|2010-10-18T14:00:00Z,53.516566,8.188217,None,10.84|2010-10-18T14:10:00Z,53.516566,8.188217,None,10.87|2010-10-18T14:20:00Z,53.516566,8.188217,None,10.89|2010-10-18T14:30:00Z,53.516566,8.188217,None,10.88|2010-10-18T14:40:00Z,53.516566,8.188217,None,10.88|2010-10-18T14:50:00Z,53.516566,8.188217,None,10.86|2010-10-18T15:00:00Z,53.516566,8.188217,None,10.76|2010-10-18T15:10:00Z,53.516566,8.188217,None,10.73|2010-10-18T15:20:00Z,53.516566,8.188217,None,10.71|2010-10-18T15:30:00Z,53.516566,8.188217,None,10.77|2010-10-18T15:40:00Z,53.516566,8.188217,None,10.84|2010-10-18T15:50:00Z,53.516566,8.188217,None,10.85|2010-10-18T16:00:00Z,53.516566,8.188217,None,10.66|2010-10-18T16:10:00Z,53.516566,8.188217,None,10.61|2010-10-18T16:20:00Z,53.516566,8.188217,None,10.71|2010-10-18T16:30:00Z,53.516566,8.188217,None,10.76|2010-10-18T16:40:00Z,53.516566,8.188217,None,10.77|2010-10-18T16:50:00Z,53.516566,8.188217,None,10.75|2010-10-18T17:00:00Z,53.516566,8.188217,None,10.77|2010-10-18T17:10:00Z,53.516566,8.188217,None,10.75|2010-10-18T17:20:00Z,53.516566,8.188217,None,10.79|2010-10-18T17:30:00Z,53.516566,8.188217,None,10.94|2010-10-18T17:40:00Z,53.516566,8.188217,None,10.98|2010-10-18T17:50:00Z,53.516566,8.188217,None,10.92|2010-10-18T18:00:00Z,53.516566,8.188217,None,10.88|2010-10-18T18:10:00Z,53.516566,8.188217,None,10.89|2010-10-18T18:20:00Z,53.516566,8.188217,None,10.91|2010-10-18T18:30:00Z,53.516566,8.188217,None,10.95|2010-10-18T18:40:00Z,53.516566,8.188217,None,10.99|2010-10-18T18:50:00Z,53.516566,8.188217,None,11.02|2010-10-18T19:00:00Z,53.516566,8.188217,None,11.04|2010-10-18T19:10:00Z,53.516566,8.188217,None,11.05|2010-10-18T19:20:00Z,53.516566,8.188217,None,11.06|2010-10-18T19:30:00Z,53.516566,8.188217,None,11.06|2010-10-18T19:40:00Z,53.516566,8.188217,None,11.07|2010-10-18T19:50:00Z,53.516566,8.188217,None,11.07|2010-10-18T20:00:00Z,53.516566,8.188217,None,11.08|2010-10-18T20:10:00Z,53.516566,8.188217,None,11.1|2010-10-18T20:20:00Z,53.516566,8.188217,None,11.12|2010-10-18T20:30:00Z,53.516566,8.188217,None,11.13|2010-10-18T20:40:00Z,53.516566,8.188217,None,11.13|2010-10-18T20:50:00Z,53.516566,8.188217,None,11.14|2010-10-18T21:00:00Z,53.516566,8.188217,None,11.13|2010-10-18T21:10:00Z,53.516566,8.188217,None,11.12|2010-10-18T21:20:00Z,53.516566,8.188217,None,11.09|2010-10-18T21:30:00Z,53.516566,8.188217,None,11.01|2010-10-18T21:40:00Z,53.516566,8.188217,None,10.96|2010-10-18T21:50:00Z,53.516566,8.188217,None,10.93|2010-10-18T22:00:00Z,53.516566,8.188217,None,10.89|2010-10-18T22:10:00Z,53.516566,8.188217,None,10.8|2010-10-18T22:20:00Z,53.516566,8.188217,None,10.72|2010-10-18T22:30:00Z,53.516566,8.188217,None,10.61|2010-10-18T22:40:00Z,53.516566,8.188217,None,10.5|2010-10-18T22:50:00Z,53.516566,8.188217,None,10.51|2010-10-18T23:00:00Z,53.516566,8.188217,None,10.53|2010-10-18T23:10:00Z,53.516566,8.188217,None,10.58|2010-10-18T23:20:00Z,53.516566,8.188217,None,10.72|2010-10-18T23:30:00Z,53.516566,8.188217,None,10.8|2010-10-18T23:40:00Z,53.516566,8.188217,None,10.83|2010-10-18T23:50:00Z,53.516566,8.188217,None,10.86|2010-10-19T00:10:00Z,53.516566,8.188217,None,10.89|2010-10-19T00:20:00Z,53.516566,8.188217,None,10.89|2010-10-19T00:30:00Z,53.516566,8.188217,None,10.88|2010-10-19T00:40:00Z,53.516566,8.188217,None,10.86|2010-10-19T00:50:00Z,53.516566,8.188217,None,10.85|2010-10-19T01:00:00Z,53.516566,8.188217,None,10.84|2010-10-19T01:10:00Z,53.516566,8.188217,None,10.83|2010-10-19T01:20:00Z,53.516566,8.188217,None,10.82|2010-10-19T01:30:00Z,53.516566,8.188217,None,10.81|2010-10-19T01:40:00Z,53.516566,8.188217,None,10.8|2010-10-19T01:50:00Z,53.516566,8.188217,None,10.8|2010-10-19T02:00:00Z,53.516566,8.188217,None,10.79|2010-10-19T02:10:00Z,53.516566,8.188217,None,10.79|2010-10-19T02:20:00Z,53.516566,8.188217,None,10.78|2010-10-19T02:30:00Z,53.516566,8.188217,None,10.78|2010-10-19T02:40:00Z,53.516566,8.188217,None,10.77|2010-10-19T02:50:00Z,53.516566,8.188217,None,10.77|2010-10-19T03:00:00Z,53.516566,8.188217,None,10.78|2010-10-19T03:10:00Z,53.516566,8.188217,None,10.79|2010-10-19T03:20:00Z,53.516566,8.188217,None,10.81|2010-10-19T03:30:00Z,53.516566,8.188217,None,10.85|2010-10-19T03:40:00Z,53.516566,8.188217,None,10.89|2010-10-19T03:50:00Z,53.516566,8.188217,None,10.9|2010-10-19T04:00:00Z,53.516566,8.188217,None,10.89|2010-10-19T04:10:00Z,53.516566,8.188217,None,10.86|2010-10-19T04:20:00Z,53.516566,8.188217,None,10.83|2010-10-19T04:30:00Z,53.516566,8.188217,None,10.82|2010-10-19T04:40:00Z,53.516566,8.188217,None,10.81|2010-10-19T04:50:00Z,53.516566,8.188217,None,10.79|2010-10-19T05:00:00Z,53.516566,8.188217,None,10.79|2010-10-19T05:10:00Z,53.516566,8.188217,None,10.79|2010-10-19T05:20:00Z,53.516566,8.188217,None,10.78|2010-10-19T05:30:00Z,53.516566,8.188217,None,10.74|2010-10-19T05:40:00Z,53.516566,8.188217,None,10.75|2010-10-19T05:50:00Z,53.516566,8.188217,None,10.77|2010-10-19T06:00:00Z,53.516566,8.188217,None,10.78|2010-10-19T06:10:00Z,53.516566,8.188217,None,10.83|2010-10-19T06:20:00Z,53.516566,8.188217,None,10.78|2010-10-19T06:30:00Z,53.516566,8.188217,None,10.73|2010-10-19T06:40:00Z,53.516566,8.188217,None,10.73|2010-10-19T06:50:00Z,53.516566,8.188217,None,10.74|2010-10-19T07:00:00Z,53.516566,8.188217,None,10.78|2010-10-19T07:10:00Z,53.516566,8.188217,None,10.84|2010-10-19T07:20:00Z,53.516566,8.188217,None,10.88|2010-10-19T07:30:00Z,53.516566,8.188217,None,10.92|2010-10-19T07:40:00Z,53.516566,8.188217,None,10.93|2010-10-19T07:50:00Z,53.516566,8.188217,None,10.95|2010-10-19T08:00:00Z,53.516566,8.188217,None,10.97|2010-10-19T08:10:00Z,53.516566,8.188217,None,11.01|2010-10-19T08:20:00Z,53.516566,8.188217,None,11.09|2010-10-19T08:30:00Z,53.516566,8.188217,None,11.17|2010-10-19T08:40:00Z,53.516566,8.188217,None,11.23|2010-10-19T08:50:00Z,53.516566,8.188217,None,11.27|2010-10-19T09:00:00Z,53.516566,8.188217,None,11.3|2010-10-19T09:10:00Z,53.516566,8.188217,None,11.32|2010-10-19T09:20:00Z,53.516566,8.188217,None,11.33|2010-10-19T09:30:00Z,53.516566,8.188217,None,11.34|2010-10-19T09:40:00Z,53.516566,8.188217,None,11.35|2010-10-19T09:50:00Z,53.516566,8.188217,None,11.34|2010-10-19T10:00:00Z,53.516566,8.188217,None,11.33|2010-10-19T10:10:00Z,53.516566,8.188217,None,11.26|2010-10-19T10:20:00Z,53.516566,8.188217,None,11.1|2010-10-19T10:30:00Z,53.516566,8.188217,None,11.0|2010-10-19T10:40:00Z,53.516566,8.188217,None,10.96|2010-10-19T10:50:00Z,53.516566,8.188217,None,10.86|2010-10-19T11:00:00Z,53.516566,8.188217,None,10.79|2010-10-19T11:10:00Z,53.516566,8.188217,None,10.78|2010-10-19T11:20:00Z,53.516566,8.188217,None,10.81|2010-10-19T11:30:00Z,53.516566,8.188217,None,10.77|2010-10-19T11:40:00Z,53.516566,8.188217,None,10.78|2010-10-19T11:50:00Z,53.516566,8.188217,None,10.82|2010-10-19T12:00:00Z,53.516566,8.188217,None,10.84|2010-10-19T12:10:00Z,53.516566,8.188217,None,10.87|2010-10-19T12:20:00Z,53.516566,8.188217,None,10.92|2010-10-19T12:30:00Z,53.516566,8.188217,None,10.96|2010-10-19T12:40:00Z,53.516566,8.188217,None,10.99|2010-10-19T12:50:00Z,53.516566,8.188217,None,10.99|2010-10-19T13:00:00Z,53.516566,8.188217,None,10.98|2010-10-19T13:10:00Z,53.516566,8.188217,None,10.97|2010-10-19T13:20:00Z,53.516566,8.188217,None,10.94|2010-10-19T13:30:00Z,53.516566,8.188217,None,10.92|2010-10-19T13:40:00Z,53.516566,8.188217,None,10.92|2010-10-19T13:50:00Z,53.516566,8.188217,None,10.94|2010-10-19T14:00:00Z,53.516566,8.188217,None,10.96|2010-10-19T14:10:00Z,53.516566,8.188217,None,10.97|2010-10-19T14:20:00Z,53.516566,8.188217,None,10.98|2010-10-19T14:30:00Z,53.516566,8.188217,None,10.98|2010-10-19T14:40:00Z,53.516566,8.188217,None,10.98|2010-10-19T14:50:00Z,53.516566,8.188217,None,10.99|2010-10-19T15:00:00Z,53.516566,8.188217,None,-2326.0|2010-10-19T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-10-19T15:20:00Z,53.516566,8.188217,None,-3377.0|2010-10-19T15:30:00Z,53.516566,8.188217,None,10.96|2010-10-19T15:40:00Z,53.516566,8.188217,None,10.97|2010-10-19T15:50:00Z,53.516566,8.188217,None,10.97|2010-10-19T16:00:00Z,53.516566,8.188217,None,10.97|2010-10-19T16:10:00Z,53.516566,8.188217,None,10.96|2010-10-19T16:20:00Z,53.516566,8.188217,None,10.94|2010-10-19T16:30:00Z,53.516566,8.188217,None,10.94|2010-10-19T16:40:00Z,53.516566,8.188217,None,10.92|2010-10-19T16:50:00Z,53.516566,8.188217,None,10.82|2010-10-19T17:00:00Z,53.516566,8.188217,None,10.66|2010-10-19T17:10:00Z,53.516566,8.188217,None,10.76|2010-10-19T17:20:00Z,53.516566,8.188217,None,10.78|2010-10-19T17:30:00Z,53.516566,8.188217,None,10.8|2010-10-19T17:40:00Z,53.516566,8.188217,None,10.85|2010-10-19T17:50:00Z,53.516566,8.188217,None,10.81|2010-10-19T18:00:00Z,53.516566,8.188217,None,10.78|2010-10-19T18:10:00Z,53.516566,8.188217,None,10.71|2010-10-19T18:20:00Z,53.516566,8.188217,None,10.75|2010-10-19T18:30:00Z,53.516566,8.188217,None,10.91|2010-10-19T18:40:00Z,53.516566,8.188217,None,10.85|2010-10-19T18:50:00Z,53.516566,8.188217,None,10.75|2010-10-19T19:00:00Z,53.516566,8.188217,None,10.72|2010-10-19T19:10:00Z,53.516566,8.188217,None,10.74|2010-10-19T19:20:00Z,53.516566,8.188217,None,10.79|2010-10-19T19:30:00Z,53.516566,8.188217,None,10.84|2010-10-19T19:40:00Z,53.516566,8.188217,None,10.89|2010-10-19T19:50:00Z,53.516566,8.188217,None,10.91|2010-10-19T20:00:00Z,53.516566,8.188217,None,10.92|2010-10-19T20:10:00Z,53.516566,8.188217,None,10.94|2010-10-19T20:20:00Z,53.516566,8.188217,None,10.98|2010-10-19T20:30:00Z,53.516566,8.188217,None,11.04|2010-10-19T20:40:00Z,53.516566,8.188217,None,11.08|2010-10-19T20:50:00Z,53.516566,8.188217,None,11.12|2010-10-19T21:00:00Z,53.516566,8.188217,None,11.14|2010-10-19T21:10:00Z,53.516566,8.188217,None,11.16|2010-10-19T21:20:00Z,53.516566,8.188217,None,11.17|2010-10-19T21:30:00Z,53.516566,8.188217,None,11.18|2010-10-19T21:40:00Z,53.516566,8.188217,None,11.19|2010-10-19T21:50:00Z,53.516566,8.188217,None,11.19|2010-10-19T22:00:00Z,53.516566,8.188217,None,11.19|2010-10-19T22:10:00Z,53.516566,8.188217,None,11.18|2010-10-19T22:20:00Z,53.516566,8.188217,None,11.16|2010-10-19T22:30:00Z,53.516566,8.188217,None,11.12|2010-10-19T22:40:00Z,53.516566,8.188217,None,11.07|2010-10-19T22:50:00Z,53.516566,8.188217,None,10.99|2010-10-19T23:00:00Z,53.516566,8.188217,None,10.89|2010-10-19T23:10:00Z,53.516566,8.188217,None,10.84|2010-10-19T23:20:00Z,53.516566,8.188217,None,10.75|2010-10-19T23:30:00Z,53.516566,8.188217,None,10.66|2010-10-19T23:40:00Z,53.516566,8.188217,None,10.71|2010-10-19T23:50:00Z,53.516566,8.188217,None,10.65|2010-10-20T00:00:00Z,53.516566,8.188217,None,10.61|2010-10-20T00:10:00Z,53.516566,8.188217,None,10.66|2010-10-20T00:20:00Z,53.516566,8.188217,None,10.69|2010-10-20T00:30:00Z,53.516566,8.188217,None,10.71|2010-10-20T00:40:00Z,53.516566,8.188217,None,10.72|2010-10-20T00:50:00Z,53.516566,8.188217,None,10.71|2010-10-20T01:00:00Z,53.516566,8.188217,None,10.69|2010-10-20T01:10:00Z,53.516566,8.188217,None,10.66|2010-10-20T01:20:00Z,53.516566,8.188217,None,10.63|2010-10-20T01:30:00Z,53.516566,8.188217,None,10.62|2010-10-20T01:40:00Z,53.516566,8.188217,None,10.6|2010-10-20T01:50:00Z,53.516566,8.188217,None,10.59|2010-10-20T02:00:00Z,53.516566,8.188217,None,10.57|2010-10-20T02:10:00Z,53.516566,8.188217,None,10.55|2010-10-20T02:20:00Z,53.516566,8.188217,None,10.55|2010-10-20T02:30:00Z,53.516566,8.188217,None,10.53|2010-10-20T02:40:00Z,53.516566,8.188217,None,10.53|2010-10-20T02:50:00Z,53.516566,8.188217,None,10.54|2010-10-20T03:00:00Z,53.516566,8.188217,None,10.55|2010-10-20T03:10:00Z,53.516566,8.188217,None,10.55|2010-10-20T03:20:00Z,53.516566,8.188217,None,10.55|2010-10-20T03:30:00Z,53.516566,8.188217,None,10.51|2010-10-20T03:40:00Z,53.516566,8.188217,None,10.48|2010-10-20T03:50:00Z,53.516566,8.188217,None,10.52|2010-10-20T04:00:00Z,53.516566,8.188217,None,10.54|2010-10-20T04:10:00Z,53.516566,8.188217,None,10.56|2010-10-20T04:20:00Z,53.516566,8.188217,None,10.55|2010-10-20T04:30:00Z,53.516566,8.188217,None,10.51|2010-10-20T04:40:00Z,53.516566,8.188217,None,10.46|2010-10-20T04:50:00Z,53.516566,8.188217,None,10.48|2010-10-20T05:00:00Z,53.516566,8.188217,None,10.5|2010-10-20T05:10:00Z,53.516566,8.188217,None,10.5|2010-10-20T05:20:00Z,53.516566,8.188217,None,10.5|2010-10-20T05:30:00Z,53.516566,8.188217,None,10.52|2010-10-20T05:40:00Z,53.516566,8.188217,None,10.44|2010-10-20T05:50:00Z,53.516566,8.188217,None,10.39|2010-10-20T06:00:00Z,53.516566,8.188217,None,10.41|2010-10-20T06:10:00Z,53.516566,8.188217,None,10.42|2010-10-20T06:20:00Z,53.516566,8.188217,None,10.41|2010-10-20T06:30:00Z,53.516566,8.188217,None,10.51|2010-10-20T06:40:00Z,53.516566,8.188217,None,10.46|2010-10-20T06:50:00Z,53.516566,8.188217,None,10.49|2010-10-20T07:00:00Z,53.516566,8.188217,None,10.54|2010-10-20T07:10:00Z,53.516566,8.188217,None,10.57|2010-10-20T07:20:00Z,53.516566,8.188217,None,10.52|2010-10-20T07:30:00Z,53.516566,8.188217,None,10.55|2010-10-20T07:40:00Z,53.516566,8.188217,None,10.63|2010-10-20T07:50:00Z,53.516566,8.188217,None,10.7|2010-10-20T08:00:00Z,53.516566,8.188217,None,10.79|2010-10-20T08:10:00Z,53.516566,8.188217,None,10.85|2010-10-20T08:20:00Z,53.516566,8.188217,None,10.89|2010-10-20T08:30:00Z,53.516566,8.188217,None,10.93|2010-10-20T08:40:00Z,53.516566,8.188217,None,10.95|2010-10-20T08:50:00Z,53.516566,8.188217,None,10.99|2010-10-20T09:00:00Z,53.516566,8.188217,None,11.04|2010-10-20T09:10:00Z,53.516566,8.188217,None,11.1|2010-10-20T09:20:00Z,53.516566,8.188217,None,11.17|2010-10-20T09:30:00Z,53.516566,8.188217,None,11.26|2010-10-20T09:40:00Z,53.516566,8.188217,None,11.41|2010-10-20T09:50:00Z,53.516566,8.188217,None,11.5|2010-10-20T10:00:00Z,53.516566,8.188217,None,11.5|2010-10-20T10:10:00Z,53.516566,8.188217,None,11.5|2010-10-20T10:20:00Z,53.516566,8.188217,None,11.52|2010-10-20T10:30:00Z,53.516566,8.188217,None,11.52|2010-10-20T10:40:00Z,53.516566,8.188217,None,11.5|2010-10-20T10:50:00Z,53.516566,8.188217,None,11.46|2010-10-20T11:00:00Z,53.516566,8.188217,None,11.36|2010-10-20T11:10:00Z,53.516566,8.188217,None,11.13|2010-10-20T11:20:00Z,53.516566,8.188217,None,11.0|2010-10-20T11:30:00Z,53.516566,8.188217,None,10.92|2010-10-20T11:40:00Z,53.516566,8.188217,None,10.87|2010-10-20T11:50:00Z,53.516566,8.188217,None,10.77|2010-10-20T12:00:00Z,53.516566,8.188217,None,10.61|2010-10-20T12:10:00Z,53.516566,8.188217,None,10.52|2010-10-20T12:20:00Z,53.516566,8.188217,None,10.45|2010-10-20T12:30:00Z,53.516566,8.188217,None,10.4|2010-10-20T12:40:00Z,53.516566,8.188217,None,10.35|2010-10-20T12:50:00Z,53.516566,8.188217,None,10.37|2010-10-20T13:00:00Z,53.516566,8.188217,None,10.39|2010-10-20T13:10:00Z,53.516566,8.188217,None,10.4|2010-10-20T13:20:00Z,53.516566,8.188217,None,10.42|2010-10-20T13:30:00Z,53.516566,8.188217,None,10.44|2010-10-20T13:40:00Z,53.516566,8.188217,None,10.45|2010-10-20T13:50:00Z,53.516566,8.188217,None,10.43|2010-10-20T14:00:00Z,53.516566,8.188217,None,10.42|2010-10-20T14:10:00Z,53.516566,8.188217,None,10.43|2010-10-20T14:20:00Z,53.516566,8.188217,None,10.43|2010-10-20T14:30:00Z,53.516566,8.188217,None,10.44|2010-10-20T14:40:00Z,53.516566,8.188217,None,10.43|2010-10-20T14:50:00Z,53.516566,8.188217,None,10.42|2010-10-20T15:00:00Z,53.516566,8.188217,None,10.41|2010-10-20T15:10:00Z,53.516566,8.188217,None,10.4|2010-10-20T15:20:00Z,53.516566,8.188217,None,10.39|2010-10-20T15:30:00Z,53.516566,8.188217,None,10.37|2010-10-20T15:40:00Z,53.516566,8.188217,None,10.34|2010-10-20T15:50:00Z,53.516566,8.188217,None,10.33|2010-10-20T16:00:00Z,53.516566,8.188217,None,10.28|2010-10-20T16:10:00Z,53.516566,8.188217,None,10.2|2010-10-20T16:20:00Z,53.516566,8.188217,None,10.17|2010-10-20T16:30:00Z,53.516566,8.188217,None,10.19|2010-10-20T16:40:00Z,53.516566,8.188217,None,10.13|2010-10-20T16:50:00Z,53.516566,8.188217,None,10.07|2010-10-20T17:00:00Z,53.516566,8.188217,None,10.08|2010-10-20T17:10:00Z,53.516566,8.188217,None,10.08|2010-10-20T17:20:00Z,53.516566,8.188217,None,10.08|2010-10-20T17:30:00Z,53.516566,8.188217,None,10.12|2010-10-20T17:40:00Z,53.516566,8.188217,None,10.24|2010-10-20T17:50:00Z,53.516566,8.188217,None,10.29|2010-10-20T18:00:00Z,53.516566,8.188217,None,10.31|2010-10-20T18:10:00Z,53.516566,8.188217,None,10.29|2010-10-20T18:20:00Z,53.516566,8.188217,None,10.27|2010-10-20T18:30:00Z,53.516566,8.188217,None,10.23|2010-10-20T18:40:00Z,53.516566,8.188217,None,10.17|2010-10-20T18:50:00Z,53.516566,8.188217,None,10.12|2010-10-20T19:00:00Z,53.516566,8.188217,None,10.12|2010-10-20T19:10:00Z,53.516566,8.188217,None,10.18|2010-10-20T19:20:00Z,53.516566,8.188217,None,10.1|2010-10-20T19:30:00Z,53.516566,8.188217,None,10.27|2010-10-20T19:40:00Z,53.516566,8.188217,None,10.46|2010-10-20T19:50:00Z,53.516566,8.188217,None,10.46|2010-10-20T20:00:00Z,53.516566,8.188217,None,10.39|2010-10-20T20:10:00Z,53.516566,8.188217,None,10.42|2010-10-20T20:20:00Z,53.516566,8.188217,None,10.46|2010-10-20T20:30:00Z,53.516566,8.188217,None,10.48|2010-10-20T20:40:00Z,53.516566,8.188217,None,10.5|2010-10-20T20:50:00Z,53.516566,8.188217,None,10.52|2010-10-20T21:00:00Z,53.516566,8.188217,None,10.54|2010-10-20T21:10:00Z,53.516566,8.188217,None,10.56|2010-10-20T21:20:00Z,53.516566,8.188217,None,10.59|2010-10-20T21:30:00Z,53.516566,8.188217,None,10.66|2010-10-20T21:40:00Z,53.516566,8.188217,None,10.7|2010-10-20T21:50:00Z,53.516566,8.188217,None,10.71|2010-10-20T22:00:00Z,53.516566,8.188217,None,10.71|2010-10-20T22:10:00Z,53.516566,8.188217,None,10.71|2010-10-20T22:20:00Z,53.516566,8.188217,None,10.72|2010-10-20T22:30:00Z,53.516566,8.188217,None,10.73|2010-10-20T22:40:00Z,53.516566,8.188217,None,10.71|2010-10-20T22:50:00Z,53.516566,8.188217,None,10.61|2010-10-20T23:00:00Z,53.516566,8.188217,None,10.52|2010-10-20T23:10:00Z,53.516566,8.188217,None,10.42|2010-10-20T23:20:00Z,53.516566,8.188217,None,10.33|2010-10-20T23:30:00Z,53.516566,8.188217,None,10.25|2010-10-20T23:40:00Z,53.516566,8.188217,None,10.17|2010-10-20T23:50:00Z,53.516566,8.188217,None,9.95|2010-10-21T00:00:00Z,53.516566,8.188217,None,9.85|2010-10-21T00:10:00Z,53.516566,8.188217,None,9.85|2010-10-21T00:20:00Z,53.516566,8.188217,None,9.86|2010-10-21T00:30:00Z,53.516566,8.188217,None,9.85|2010-10-21T00:40:00Z,53.516566,8.188217,None,9.81|2010-10-21T00:50:00Z,53.516566,8.188217,None,9.81|2010-10-21T01:00:00Z,53.516566,8.188217,None,9.92|2010-10-21T01:10:00Z,53.516566,8.188217,None,10.03|2010-10-21T01:20:00Z,53.516566,8.188217,None,10.08|2010-10-21T01:30:00Z,53.516566,8.188217,None,10.09|2010-10-21T01:40:00Z,53.516566,8.188217,None,10.08|2010-10-21T01:50:00Z,53.516566,8.188217,None,10.06|2010-10-21T02:00:00Z,53.516566,8.188217,None,9.99|2010-10-21T02:10:00Z,53.516566,8.188217,None,9.85|2010-10-21T02:20:00Z,53.516566,8.188217,None,9.75|2010-10-21T02:30:00Z,53.516566,8.188217,None,9.68|2010-10-21T02:40:00Z,53.516566,8.188217,None,9.64|2010-10-21T02:50:00Z,53.516566,8.188217,None,9.61|2010-10-21T03:00:00Z,53.516566,8.188217,None,9.56|2010-10-21T03:10:00Z,53.516566,8.188217,None,9.54|2010-10-21T03:20:00Z,53.516566,8.188217,None,9.56|2010-10-21T03:30:00Z,53.516566,8.188217,None,9.58|2010-10-21T03:40:00Z,53.516566,8.188217,None,9.59|2010-10-21T03:50:00Z,53.516566,8.188217,None,9.61|2010-10-21T04:00:00Z,53.516566,8.188217,None,9.65|2010-10-21T04:10:00Z,53.516566,8.188217,None,9.69|2010-10-21T04:20:00Z,53.516566,8.188217,None,9.73|2010-10-21T04:30:00Z,53.516566,8.188217,None,9.68|2010-10-21T04:40:00Z,53.516566,8.188217,None,9.6|2010-10-21T04:50:00Z,53.516566,8.188217,None,9.5|2010-10-21T05:00:00Z,53.516566,8.188217,None,9.36|2010-10-21T05:10:00Z,53.516566,8.188217,None,9.33|2010-10-21T05:20:00Z,53.516566,8.188217,None,9.32|2010-10-21T05:30:00Z,53.516566,8.188217,None,9.24|2010-10-21T05:40:00Z,53.516566,8.188217,None,9.57|2010-10-21T05:50:00Z,53.516566,8.188217,None,9.63|2010-10-21T06:00:00Z,53.516566,8.188217,None,9.58|2010-10-21T06:10:00Z,53.516566,8.188217,None,9.64|2010-10-21T06:20:00Z,53.516566,8.188217,None,9.76|2010-10-21T06:30:00Z,53.516566,8.188217,None,9.66|2010-10-21T06:40:00Z,53.516566,8.188217,None,9.67|2010-10-21T06:50:00Z,53.516566,8.188217,None,9.83|2010-10-21T07:00:00Z,53.516566,8.188217,None,9.86|2010-10-21T07:10:00Z,53.516566,8.188217,None,10.01|2010-10-21T07:20:00Z,53.516566,8.188217,None,10.08|2010-10-21T07:30:00Z,53.516566,8.188217,None,10.08|2010-10-21T07:40:00Z,53.516566,8.188217,None,9.96|2010-10-21T07:50:00Z,53.516566,8.188217,None,9.96|2010-10-21T08:00:00Z,53.516566,8.188217,None,10.08|2010-10-21T08:10:00Z,53.516566,8.188217,None,10.14|2010-10-21T08:20:00Z,53.516566,8.188217,None,10.15|2010-10-21T08:30:00Z,53.516566,8.188217,None,10.16|2010-10-21T08:40:00Z,53.516566,8.188217,None,10.19|2010-10-21T08:50:00Z,53.516566,8.188217,None,10.21|2010-10-21T09:00:00Z,53.516566,8.188217,None,10.24|2010-10-21T09:10:00Z,53.516566,8.188217,None,10.29|2010-10-21T09:20:00Z,53.516566,8.188217,None,10.34|2010-10-21T09:30:00Z,53.516566,8.188217,None,10.39|2010-10-21T09:40:00Z,53.516566,8.188217,None,10.42|2010-10-21T09:50:00Z,53.516566,8.188217,None,10.44|2010-10-21T10:00:00Z,53.516566,8.188217,None,10.46|2010-10-21T10:10:00Z,53.516566,8.188217,None,10.49|2010-10-21T10:20:00Z,53.516566,8.188217,None,10.52|2010-10-21T10:30:00Z,53.516566,8.188217,None,10.55|2010-10-21T10:40:00Z,53.516566,8.188217,None,10.58|2010-10-21T10:50:00Z,53.516566,8.188217,None,10.58|2010-10-21T11:00:00Z,53.516566,8.188217,None,10.57|2010-10-21T11:10:00Z,53.516566,8.188217,None,10.55|2010-10-21T11:20:00Z,53.516566,8.188217,None,10.49|2010-10-21T11:30:00Z,53.516566,8.188217,None,10.43|2010-10-21T11:40:00Z,53.516566,8.188217,None,10.32|2010-10-21T11:50:00Z,53.516566,8.188217,None,10.23|2010-10-21T12:00:00Z,53.516566,8.188217,None,10.16|2010-10-21T12:10:00Z,53.516566,8.188217,None,10.07|2010-10-21T12:20:00Z,53.516566,8.188217,None,9.96|2010-10-21T12:30:00Z,53.516566,8.188217,None,9.96|2010-10-21T12:40:00Z,53.516566,8.188217,None,9.85|2010-10-21T12:50:00Z,53.516566,8.188217,None,9.95|2010-10-21T13:00:00Z,53.516566,8.188217,None,10.08|2010-10-21T13:10:00Z,53.516566,8.188217,None,10.09|2010-10-21T13:20:00Z,53.516566,8.188217,None,9.98|2010-10-21T13:30:00Z,53.516566,8.188217,None,9.76|2010-10-21T13:40:00Z,53.516566,8.188217,None,9.88|2010-10-21T13:50:00Z,53.516566,8.188217,None,10.02|2010-10-21T14:00:00Z,53.516566,8.188217,None,9.98|2010-10-21T14:10:00Z,53.516566,8.188217,None,9.99|2010-10-21T14:20:00Z,53.516566,8.188217,None,10.09|2010-10-21T14:30:00Z,53.516566,8.188217,None,10.13|2010-10-21T14:40:00Z,53.516566,8.188217,None,10.14|2010-10-21T14:50:00Z,53.516566,8.188217,None,10.15|2010-10-21T15:00:00Z,53.516566,8.188217,None,10.15|2010-10-21T15:10:00Z,53.516566,8.188217,None,10.15|2010-10-21T15:20:00Z,53.516566,8.188217,None,10.14|2010-10-21T15:30:00Z,53.516566,8.188217,None,10.13|2010-10-21T15:40:00Z,53.516566,8.188217,None,10.13|2010-10-21T15:50:00Z,53.516566,8.188217,None,10.11|2010-10-21T16:00:00Z,53.516566,8.188217,None,10.08|2010-10-21T16:10:00Z,53.516566,8.188217,None,10.05|2010-10-21T16:20:00Z,53.516566,8.188217,None,10.02|2010-10-21T16:30:00Z,53.516566,8.188217,None,9.97|2010-10-21T16:40:00Z,53.516566,8.188217,None,9.98|2010-10-21T16:50:00Z,53.516566,8.188217,None,10.0|2010-10-21T17:00:00Z,53.516566,8.188217,None,10.02|2010-10-21T17:10:00Z,53.516566,8.188217,None,10.01|2010-10-21T17:20:00Z,53.516566,8.188217,None,9.98|2010-10-21T17:30:00Z,53.516566,8.188217,None,9.9|2010-10-21T17:40:00Z,53.516566,8.188217,None,9.75|2010-10-21T17:50:00Z,53.516566,8.188217,None,9.5|2010-10-21T18:00:00Z,53.516566,8.188217,None,9.58|2010-10-21T18:10:00Z,53.516566,8.188217,None,9.89|2010-10-21T18:20:00Z,53.516566,8.188217,None,9.89|2010-10-21T18:30:00Z,53.516566,8.188217,None,9.81|2010-10-21T18:40:00Z,53.516566,8.188217,None,9.89|2010-10-21T18:50:00Z,53.516566,8.188217,None,9.86|2010-10-21T19:00:00Z,53.516566,8.188217,None,9.71|2010-10-21T19:10:00Z,53.516566,8.188217,None,9.7|2010-10-21T19:20:00Z,53.516566,8.188217,None,9.84|2010-10-21T19:30:00Z,53.516566,8.188217,None,9.73|2010-10-21T19:40:00Z,53.516566,8.188217,None,9.79|2010-10-21T19:50:00Z,53.516566,8.188217,None,9.87|2010-10-21T20:00:00Z,53.516566,8.188217,None,9.87|2010-10-21T20:10:00Z,53.516566,8.188217,None,9.86|2010-10-21T20:20:00Z,53.516566,8.188217,None,9.91|2010-10-21T20:30:00Z,53.516566,8.188217,None,9.97|2010-10-21T20:40:00Z,53.516566,8.188217,None,10.03|2010-10-21T20:50:00Z,53.516566,8.188217,None,10.08|2010-10-21T21:00:00Z,53.516566,8.188217,None,10.12|2010-10-21T21:10:00Z,53.516566,8.188217,None,10.14|2010-10-21T21:20:00Z,53.516566,8.188217,None,10.14|2010-10-21T21:30:00Z,53.516566,8.188217,None,10.14|2010-10-21T21:40:00Z,53.516566,8.188217,None,10.13|2010-10-21T21:50:00Z,53.516566,8.188217,None,10.14|2010-10-21T22:00:00Z,53.516566,8.188217,None,10.17|2010-10-21T22:10:00Z,53.516566,8.188217,None,10.18|2010-10-21T22:20:00Z,53.516566,8.188217,None,10.2|2010-10-21T22:30:00Z,53.516566,8.188217,None,10.21|2010-10-21T22:40:00Z,53.516566,8.188217,None,10.22|2010-10-21T22:50:00Z,53.516566,8.188217,None,10.23|2010-10-21T23:00:00Z,53.516566,8.188217,None,10.24|2010-10-21T23:10:00Z,53.516566,8.188217,None,10.26|2010-10-21T23:20:00Z,53.516566,8.188217,None,10.27|2010-10-21T23:30:00Z,53.516566,8.188217,None,10.28|2010-10-21T23:40:00Z,53.516566,8.188217,None,10.29|2010-10-21T23:50:00Z,53.516566,8.188217,None,10.3|2010-10-22T00:00:00Z,53.516566,8.188217,None,10.3|2010-10-22T00:10:00Z,53.516566,8.188217,None,10.29|2010-10-22T00:20:00Z,53.516566,8.188217,None,10.27|2010-10-22T00:30:00Z,53.516566,8.188217,None,10.23|2010-10-22T00:40:00Z,53.516566,8.188217,None,10.19|2010-10-22T00:50:00Z,53.516566,8.188217,None,10.14|2010-10-22T01:00:00Z,53.516566,8.188217,None,10.09|2010-10-22T01:10:00Z,53.516566,8.188217,None,10.0|2010-10-22T01:20:00Z,53.516566,8.188217,None,9.95|2010-10-22T01:30:00Z,53.516566,8.188217,None,10.01|2010-10-22T01:40:00Z,53.516566,8.188217,None,10.08|2010-10-22T01:50:00Z,53.516566,8.188217,None,10.14|2010-10-22T02:00:00Z,53.516566,8.188217,None,10.14|2010-10-22T02:10:00Z,53.516566,8.188217,None,10.11|2010-10-22T02:20:00Z,53.516566,8.188217,None,10.06|2010-10-22T02:30:00Z,53.516566,8.188217,None,10.04|2010-10-22T02:40:00Z,53.516566,8.188217,None,10.01|2010-10-22T02:50:00Z,53.516566,8.188217,None,9.99|2010-10-22T03:00:00Z,53.516566,8.188217,None,9.98|2010-10-22T03:10:00Z,53.516566,8.188217,None,9.95|2010-10-22T03:20:00Z,53.516566,8.188217,None,9.93|2010-10-22T03:30:00Z,53.516566,8.188217,None,9.91|2010-10-22T03:40:00Z,53.516566,8.188217,None,9.9|2010-10-22T03:50:00Z,53.516566,8.188217,None,9.87|2010-10-22T04:00:00Z,53.516566,8.188217,None,9.84|2010-10-22T04:10:00Z,53.516566,8.188217,None,9.83|2010-10-22T04:20:00Z,53.516566,8.188217,None,9.84|2010-10-22T04:30:00Z,53.516566,8.188217,None,9.82|2010-10-22T04:40:00Z,53.516566,8.188217,None,9.81|2010-10-22T04:50:00Z,53.516566,8.188217,None,9.81|2010-10-22T05:00:00Z,53.516566,8.188217,None,9.81|2010-10-22T05:10:00Z,53.516566,8.188217,None,9.81|2010-10-22T05:20:00Z,53.516566,8.188217,None,9.81|2010-10-22T05:30:00Z,53.516566,8.188217,None,9.8|2010-10-22T05:40:00Z,53.516566,8.188217,None,9.78|2010-10-22T05:50:00Z,53.516566,8.188217,None,9.81|2010-10-22T06:00:00Z,53.516566,8.188217,None,9.85|2010-10-22T06:10:00Z,53.516566,8.188217,None,9.85|2010-10-22T06:20:00Z,53.516566,8.188217,None,9.79|2010-10-22T06:30:00Z,53.516566,8.188217,None,9.76|2010-10-22T06:40:00Z,53.516566,8.188217,None,9.69|2010-10-22T06:50:00Z,53.516566,8.188217,None,9.64|2010-10-22T07:00:00Z,53.516566,8.188217,None,9.63|2010-10-22T07:10:00Z,53.516566,8.188217,None,9.61|2010-10-22T07:20:00Z,53.516566,8.188217,None,9.63|2010-10-22T07:30:00Z,53.516566,8.188217,None,9.66|2010-10-22T07:40:00Z,53.516566,8.188217,None,9.34|2010-10-22T07:50:00Z,53.516566,8.188217,None,9.4|2010-10-22T08:00:00Z,53.516566,8.188217,None,9.71|2010-10-22T08:10:00Z,53.516566,8.188217,None,9.74|2010-10-22T08:20:00Z,53.516566,8.188217,None,9.53|2010-10-22T08:30:00Z,53.516566,8.188217,None,9.5|2010-10-22T08:40:00Z,53.516566,8.188217,None,9.61|2010-10-22T08:50:00Z,53.516566,8.188217,None,9.67|2010-10-22T09:00:00Z,53.516566,8.188217,None,9.68|2010-10-22T09:10:00Z,53.516566,8.188217,None,9.77|2010-10-22T09:20:00Z,53.516566,8.188217,None,9.84|2010-10-22T09:30:00Z,53.516566,8.188217,None,9.91|2010-10-22T09:40:00Z,53.516566,8.188217,None,9.99|2010-10-22T09:50:00Z,53.516566,8.188217,None,10.07|2010-10-22T10:00:00Z,53.516566,8.188217,None,10.11|2010-10-22T10:10:00Z,53.516566,8.188217,None,10.15|2010-10-22T10:20:00Z,53.516566,8.188217,None,10.18|2010-10-22T10:30:00Z,53.516566,8.188217,None,10.21|2010-10-22T10:40:00Z,53.516566,8.188217,None,10.26|2010-10-22T10:50:00Z,53.516566,8.188217,None,10.31|2010-10-22T11:00:00Z,53.516566,8.188217,None,10.33|2010-10-22T11:10:00Z,53.516566,8.188217,None,10.35|2010-10-22T11:20:00Z,53.516566,8.188217,None,10.35|2010-10-22T11:30:00Z,53.516566,8.188217,None,10.35|2010-10-22T11:40:00Z,53.516566,8.188217,None,10.3|2010-10-22T11:50:00Z,53.516566,8.188217,None,10.24|2010-10-22T12:00:00Z,53.516566,8.188217,None,10.16|2010-10-22T12:10:00Z,53.516566,8.188217,None,9.99|2010-10-22T12:20:00Z,53.516566,8.188217,None,9.84|2010-10-22T12:30:00Z,53.516566,8.188217,None,9.68|2010-10-22T12:40:00Z,53.516566,8.188217,None,9.66|2010-10-22T12:50:00Z,53.516566,8.188217,None,9.63|2010-10-22T13:00:00Z,53.516566,8.188217,None,9.68|2010-10-22T13:10:00Z,53.516566,8.188217,None,9.72|2010-10-22T13:20:00Z,53.516566,8.188217,None,9.77|2010-10-22T13:30:00Z,53.516566,8.188217,None,9.77|2010-10-22T13:40:00Z,53.516566,8.188217,None,9.72|2010-10-22T13:50:00Z,53.516566,8.188217,None,9.71|2010-10-22T14:00:00Z,53.516566,8.188217,None,9.7|2010-10-22T14:10:00Z,53.516566,8.188217,None,9.69|2010-10-22T14:20:00Z,53.516566,8.188217,None,9.69|2010-10-22T14:30:00Z,53.516566,8.188217,None,9.72|2010-10-22T14:40:00Z,53.516566,8.188217,None,9.71|2010-10-22T14:50:00Z,53.516566,8.188217,None,9.7|2010-10-22T15:00:00Z,53.516566,8.188217,None,9.71|2010-10-22T15:10:00Z,53.516566,8.188217,None,9.72|2010-10-22T15:20:00Z,53.516566,8.188217,None,9.73|2010-10-22T15:30:00Z,53.516566,8.188217,None,9.74|2010-10-22T15:40:00Z,53.516566,8.188217,None,9.74|2010-10-22T15:50:00Z,53.516566,8.188217,None,9.75|2010-10-22T16:00:00Z,53.516566,8.188217,None,9.76|2010-10-22T16:10:00Z,53.516566,8.188217,None,9.75|2010-10-22T16:20:00Z,53.516566,8.188217,None,9.73|2010-10-22T16:30:00Z,53.516566,8.188217,None,9.71|2010-10-22T16:40:00Z,53.516566,8.188217,None,9.7|2010-10-22T16:50:00Z,53.516566,8.188217,None,9.7|2010-10-22T17:00:00Z,53.516566,8.188217,None,9.71|2010-10-22T17:10:00Z,53.516566,8.188217,None,9.73|2010-10-22T17:20:00Z,53.516566,8.188217,None,9.74|2010-10-22T17:30:00Z,53.516566,8.188217,None,9.63|2010-10-22T17:40:00Z,53.516566,8.188217,None,9.56|2010-10-22T17:50:00Z,53.516566,8.188217,None,9.53|2010-10-22T18:00:00Z,53.516566,8.188217,None,9.57|2010-10-22T18:10:00Z,53.516566,8.188217,None,9.53|2010-10-22T18:20:00Z,53.516566,8.188217,None,9.6|2010-10-22T18:30:00Z,53.516566,8.188217,None,9.63|2010-10-22T18:40:00Z,53.516566,8.188217,None,9.55|2010-10-22T18:50:00Z,53.516566,8.188217,None,9.51|2010-10-22T19:00:00Z,53.516566,8.188217,None,9.54|2010-10-22T19:10:00Z,53.516566,8.188217,None,9.63|2010-10-22T19:20:00Z,53.516566,8.188217,None,9.66|2010-10-22T19:30:00Z,53.516566,8.188217,None,9.71|2010-10-22T19:40:00Z,53.516566,8.188217,None,9.6|2010-10-22T19:50:00Z,53.516566,8.188217,None,9.53|2010-10-22T20:00:00Z,53.516566,8.188217,None,9.49|2010-10-22T20:10:00Z,53.516566,8.188217,None,9.41|2010-10-22T20:20:00Z,53.516566,8.188217,None,9.3|2010-10-22T20:30:00Z,53.516566,8.188217,None,9.29|2010-10-22T20:40:00Z,53.516566,8.188217,None,9.46|2010-10-22T20:50:00Z,53.516566,8.188217,None,9.58|2010-10-22T21:00:00Z,53.516566,8.188217,None,9.45|2010-10-22T21:10:00Z,53.516566,8.188217,None,9.76|2010-10-22T21:20:00Z,53.516566,8.188217,None,9.81|2010-10-22T21:30:00Z,53.516566,8.188217,None,9.81|2010-10-22T21:40:00Z,53.516566,8.188217,None,9.81|2010-10-22T21:50:00Z,53.516566,8.188217,None,9.84|2010-10-22T22:00:00Z,53.516566,8.188217,None,9.88|2010-10-22T22:10:00Z,53.516566,8.188217,None,9.9|2010-10-22T22:20:00Z,53.516566,8.188217,None,9.92|2010-10-22T22:30:00Z,53.516566,8.188217,None,9.93|2010-10-22T22:40:00Z,53.516566,8.188217,None,9.94|2010-10-22T22:50:00Z,53.516566,8.188217,None,9.96|2010-10-22T23:00:00Z,53.516566,8.188217,None,9.98|2010-10-22T23:10:00Z,53.516566,8.188217,None,10.0|2010-10-22T23:20:00Z,53.516566,8.188217,None,10.03|2010-10-22T23:30:00Z,53.516566,8.188217,None,10.04|2010-10-22T23:40:00Z,53.516566,8.188217,None,10.05|2010-10-22T23:50:00Z,53.516566,8.188217,None,10.0|2010-10-23T00:00:00Z,53.516566,8.188217,None,9.91|2010-10-23T00:10:00Z,53.516566,8.188217,None,9.86|2010-10-23T00:20:00Z,53.516566,8.188217,None,9.79|2010-10-23T00:30:00Z,53.516566,8.188217,None,9.7|2010-10-23T00:40:00Z,53.516566,8.188217,None,9.55|2010-10-23T00:50:00Z,53.516566,8.188217,None,9.31|2010-10-23T01:00:00Z,53.516566,8.188217,None,9.36|2010-10-23T01:10:00Z,53.516566,8.188217,None,9.47|2010-10-23T01:20:00Z,53.516566,8.188217,None,9.42|2010-10-23T01:30:00Z,53.516566,8.188217,None,9.4|2010-10-23T01:40:00Z,53.516566,8.188217,None,9.38|2010-10-23T01:50:00Z,53.516566,8.188217,None,9.39|2010-10-23T02:00:00Z,53.516566,8.188217,None,9.41|2010-10-23T02:10:00Z,53.516566,8.188217,None,9.45|2010-10-23T02:20:00Z,53.516566,8.188217,None,9.43|2010-10-23T02:30:00Z,53.516566,8.188217,None,9.35|2010-10-23T02:40:00Z,53.516566,8.188217,None,9.33|2010-10-23T02:50:00Z,53.516566,8.188217,None,9.37|2010-10-23T03:00:00Z,53.516566,8.188217,None,9.43|2010-10-23T03:10:00Z,53.516566,8.188217,None,9.48|2010-10-23T03:20:00Z,53.516566,8.188217,None,9.51|2010-10-23T03:30:00Z,53.516566,8.188217,None,9.53|2010-10-23T03:40:00Z,53.516566,8.188217,None,9.53|2010-10-23T03:50:00Z,53.516566,8.188217,None,9.49|2010-10-23T04:00:00Z,53.516566,8.188217,None,9.47|2010-10-23T04:10:00Z,53.516566,8.188217,None,9.45|2010-10-23T04:20:00Z,53.516566,8.188217,None,9.43|2010-10-23T04:30:00Z,53.516566,8.188217,None,9.43|2010-10-23T04:40:00Z,53.516566,8.188217,None,9.41|2010-10-23T04:50:00Z,53.516566,8.188217,None,9.38|2010-10-23T05:00:00Z,53.516566,8.188217,None,9.36|2010-10-23T05:10:00Z,53.516566,8.188217,None,9.37|2010-10-23T05:20:00Z,53.516566,8.188217,None,9.41|2010-10-23T05:30:00Z,53.516566,8.188217,None,9.43|2010-10-23T05:40:00Z,53.516566,8.188217,None,9.42|2010-10-23T05:50:00Z,53.516566,8.188217,None,9.4|2010-10-23T06:00:00Z,53.516566,8.188217,None,9.36|2010-10-23T06:10:00Z,53.516566,8.188217,None,9.28|2010-10-23T06:20:00Z,53.516566,8.188217,None,9.27|2010-10-23T06:30:00Z,53.516566,8.188217,None,9.27|2010-10-23T06:40:00Z,53.516566,8.188217,None,9.23|2010-10-23T06:50:00Z,53.516566,8.188217,None,9.22|2010-10-23T07:00:00Z,53.516566,8.188217,None,9.25|2010-10-23T07:10:00Z,53.516566,8.188217,None,9.17|2010-10-23T07:20:00Z,53.516566,8.188217,None,9.22|2010-10-23T07:30:00Z,53.516566,8.188217,None,9.29|2010-10-23T07:40:00Z,53.516566,8.188217,None,9.32|2010-10-23T07:50:00Z,53.516566,8.188217,None,9.41|2010-10-23T08:00:00Z,53.516566,8.188217,None,9.46|2010-10-23T08:10:00Z,53.516566,8.188217,None,9.47|2010-10-23T08:20:00Z,53.516566,8.188217,None,9.27|2010-10-23T08:30:00Z,53.516566,8.188217,None,9.38|2010-10-23T08:40:00Z,53.516566,8.188217,None,9.46|2010-10-23T08:50:00Z,53.516566,8.188217,None,9.34|2010-10-23T09:00:00Z,53.516566,8.188217,None,9.27|2010-10-23T09:10:00Z,53.516566,8.188217,None,9.39|2010-10-23T09:20:00Z,53.516566,8.188217,None,9.49|2010-10-23T09:30:00Z,53.516566,8.188217,None,9.53|2010-10-23T09:40:00Z,53.516566,8.188217,None,9.37|2010-10-23T09:50:00Z,53.516566,8.188217,None,9.68|2010-10-23T10:00:00Z,53.516566,8.188217,None,9.65|2010-10-23T10:10:00Z,53.516566,8.188217,None,9.63|2010-10-23T10:20:00Z,53.516566,8.188217,None,9.68|2010-10-23T10:30:00Z,53.516566,8.188217,None,9.74|2010-10-23T10:40:00Z,53.516566,8.188217,None,9.81|2010-10-23T10:50:00Z,53.516566,8.188217,None,9.9|2010-10-23T11:00:00Z,53.516566,8.188217,None,9.75|2010-10-23T11:10:00Z,53.516566,8.188217,None,10.02|2010-10-23T11:20:00Z,53.516566,8.188217,None,10.05|2010-10-23T11:30:00Z,53.516566,8.188217,None,10.09|2010-10-23T11:40:00Z,53.516566,8.188217,None,10.13|2010-10-23T11:50:00Z,53.516566,8.188217,None,10.17|2010-10-23T12:00:00Z,53.516566,8.188217,None,10.18|2010-10-23T12:10:00Z,53.516566,8.188217,None,10.15|2010-10-23T12:20:00Z,53.516566,8.188217,None,10.04|2010-10-23T12:30:00Z,53.516566,8.188217,None,9.97|2010-10-23T12:40:00Z,53.516566,8.188217,None,9.83|2010-10-23T12:50:00Z,53.516566,8.188217,None,9.66|2010-10-23T13:00:00Z,53.516566,8.188217,None,9.57|2010-10-23T13:10:00Z,53.516566,8.188217,None,9.38|2010-10-23T13:20:00Z,53.516566,8.188217,None,9.29|2010-10-23T13:30:00Z,53.516566,8.188217,None,8.97|2010-10-23T13:40:00Z,53.516566,8.188217,None,8.65|2010-10-23T13:50:00Z,53.516566,8.188217,None,8.77|2010-10-23T14:00:00Z,53.516566,8.188217,None,9.14|2010-10-23T14:10:00Z,53.516566,8.188217,None,9.2|2010-10-23T14:20:00Z,53.516566,8.188217,None,9.23|2010-10-23T14:30:00Z,53.516566,8.188217,None,9.28|2010-10-23T14:40:00Z,53.516566,8.188217,None,9.3|2010-10-23T14:50:00Z,53.516566,8.188217,None,9.25|2010-10-23T15:00:00Z,53.516566,8.188217,None,9.25|2010-10-23T15:10:00Z,53.516566,8.188217,None,9.3|2010-10-23T15:20:00Z,53.516566,8.188217,None,9.32|2010-10-23T15:30:00Z,53.516566,8.188217,None,9.3|2010-10-23T15:40:00Z,53.516566,8.188217,None,9.29|2010-10-23T15:50:00Z,53.516566,8.188217,None,9.29|2010-10-23T16:00:00Z,53.516566,8.188217,None,9.29|2010-10-23T16:10:00Z,53.516566,8.188217,None,9.29|2010-10-23T16:20:00Z,53.516566,8.188217,None,9.3|2010-10-23T16:30:00Z,53.516566,8.188217,None,9.3|2010-10-23T16:40:00Z,53.516566,8.188217,None,9.29|2010-10-23T16:50:00Z,53.516566,8.188217,None,9.29|2010-10-23T17:00:00Z,53.516566,8.188217,None,9.28|2010-10-23T17:10:00Z,53.516566,8.188217,None,9.28|2010-10-23T17:20:00Z,53.516566,8.188217,None,9.26|2010-10-23T17:30:00Z,53.516566,8.188217,None,9.25|2010-10-23T17:40:00Z,53.516566,8.188217,None,9.25|2010-10-23T17:50:00Z,53.516566,8.188217,None,9.24|2010-10-23T18:00:00Z,53.516566,8.188217,None,9.22|2010-10-23T18:10:00Z,53.516566,8.188217,None,9.21|2010-10-23T18:20:00Z,53.516566,8.188217,None,9.2|2010-10-23T18:30:00Z,53.516566,8.188217,None,9.21|2010-10-23T18:40:00Z,53.516566,8.188217,None,9.2|2010-10-23T18:50:00Z,53.516566,8.188217,None,9.17|2010-10-23T19:00:00Z,53.516566,8.188217,None,9.17|2010-10-23T19:10:00Z,53.516566,8.188217,None,9.18|2010-10-23T19:20:00Z,53.516566,8.188217,None,9.14|2010-10-23T19:30:00Z,53.516566,8.188217,None,9.08|2010-10-23T19:40:00Z,53.516566,8.188217,None,9.02|2010-10-23T19:50:00Z,53.516566,8.188217,None,9.21|2010-10-23T20:00:00Z,53.516566,8.188217,None,9.27|2010-10-23T20:10:00Z,53.516566,8.188217,None,9.31|2010-10-23T20:20:00Z,53.516566,8.188217,None,9.3|2010-10-23T20:30:00Z,53.516566,8.188217,None,9.23|2010-10-23T20:40:00Z,53.516566,8.188217,None,9.31|2010-10-23T20:50:00Z,53.516566,8.188217,None,9.27|2010-10-23T21:00:00Z,53.516566,8.188217,None,9.19|2010-10-23T21:10:00Z,53.516566,8.188217,None,9.21|2010-10-23T21:20:00Z,53.516566,8.188217,None,9.28|2010-10-23T21:30:00Z,53.516566,8.188217,None,9.31|2010-10-23T21:40:00Z,53.516566,8.188217,None,9.33|2010-10-23T21:50:00Z,53.516566,8.188217,None,9.4|2010-10-23T22:00:00Z,53.516566,8.188217,None,9.53|2010-10-23T22:10:00Z,53.516566,8.188217,None,9.58|2010-10-23T22:20:00Z,53.516566,8.188217,None,9.59|2010-10-23T22:30:00Z,53.516566,8.188217,None,9.6|2010-10-23T22:40:00Z,53.516566,8.188217,None,9.62|2010-10-23T22:50:00Z,53.516566,8.188217,None,9.63|2010-10-23T23:00:00Z,53.516566,8.188217,None,9.65|2010-10-23T23:10:00Z,53.516566,8.188217,None,9.69|2010-10-23T23:20:00Z,53.516566,8.188217,None,9.77|2010-10-23T23:30:00Z,53.516566,8.188217,None,9.88|2010-10-23T23:40:00Z,53.516566,8.188217,None,9.98|2010-10-23T23:50:00Z,53.516566,8.188217,None,10.01|2010-10-24T00:00:00Z,53.516566,8.188217,None,10.02|2010-10-24T00:10:00Z,53.516566,8.188217,None,10.01|2010-10-24T00:20:00Z,53.516566,8.188217,None,10.01|2010-10-24T00:30:00Z,53.516566,8.188217,None,9.98|2010-10-24T00:40:00Z,53.516566,8.188217,None,9.8|2010-10-24T00:50:00Z,53.516566,8.188217,None,9.6|2010-10-24T01:00:00Z,53.516566,8.188217,None,9.55|2010-10-24T01:10:00Z,53.516566,8.188217,None,9.5|2010-10-24T01:20:00Z,53.516566,8.188217,None,9.27|2010-10-24T01:30:00Z,53.516566,8.188217,None,9.0|2010-10-24T01:40:00Z,53.516566,8.188217,None,8.84|2010-10-24T01:50:00Z,53.516566,8.188217,None,8.99|2010-10-24T02:00:00Z,53.516566,8.188217,None,9.01|2010-10-24T02:10:00Z,53.516566,8.188217,None,9.08|2010-10-24T02:20:00Z,53.516566,8.188217,None,9.37|2010-10-24T02:30:00Z,53.516566,8.188217,None,9.5|2010-10-24T02:40:00Z,53.516566,8.188217,None,9.57|2010-10-24T02:50:00Z,53.516566,8.188217,None,9.61|2010-10-24T03:00:00Z,53.516566,8.188217,None,9.62|2010-10-24T03:10:00Z,53.516566,8.188217,None,9.63|2010-10-24T03:20:00Z,53.516566,8.188217,None,9.63|2010-10-24T03:30:00Z,53.516566,8.188217,None,9.62|2010-10-24T03:40:00Z,53.516566,8.188217,None,9.6|2010-10-24T03:50:00Z,53.516566,8.188217,None,9.58|2010-10-24T04:00:00Z,53.516566,8.188217,None,9.59|2010-10-24T04:10:00Z,53.516566,8.188217,None,9.58|2010-10-24T04:20:00Z,53.516566,8.188217,None,9.58|2010-10-24T04:30:00Z,53.516566,8.188217,None,9.57|2010-10-24T04:40:00Z,53.516566,8.188217,None,9.59|2010-10-24T04:50:00Z,53.516566,8.188217,None,9.59|2010-10-24T05:00:00Z,53.516566,8.188217,None,9.6|2010-10-24T05:10:00Z,53.516566,8.188217,None,9.61|2010-10-24T05:20:00Z,53.516566,8.188217,None,9.4|2010-10-24T05:30:00Z,53.516566,8.188217,None,9.61|2010-10-24T05:40:00Z,53.516566,8.188217,None,9.6|2010-10-24T05:50:00Z,53.516566,8.188217,None,9.59|2010-10-24T06:00:00Z,53.516566,8.188217,None,9.54|2010-10-24T06:10:00Z,53.516566,8.188217,None,9.48|2010-10-24T06:20:00Z,53.516566,8.188217,None,9.47|2010-10-24T06:30:00Z,53.516566,8.188217,None,9.46|2010-10-24T06:40:00Z,53.516566,8.188217,None,9.46|2010-10-24T06:50:00Z,53.516566,8.188217,None,9.42|2010-10-24T07:00:00Z,53.516566,8.188217,None,9.36|2010-10-24T07:10:00Z,53.516566,8.188217,None,9.37|2010-10-24T07:20:00Z,53.516566,8.188217,None,9.39|2010-10-24T07:30:00Z,53.516566,8.188217,None,9.41|2010-10-24T07:40:00Z,53.516566,8.188217,None,9.5|2010-10-24T07:50:00Z,53.516566,8.188217,None,9.44|2010-10-24T08:00:00Z,53.516566,8.188217,None,9.47|2010-10-24T08:10:00Z,53.516566,8.188217,None,9.49|2010-10-24T08:20:00Z,53.516566,8.188217,None,9.43|2010-10-24T08:30:00Z,53.516566,8.188217,None,9.32|2010-10-24T08:40:00Z,53.516566,8.188217,None,9.16|2010-10-24T08:50:00Z,53.516566,8.188217,None,9.08|2010-10-24T09:00:00Z,53.516566,8.188217,None,9.22|2010-10-24T09:10:00Z,53.516566,8.188217,None,9.38|2010-10-24T09:20:00Z,53.516566,8.188217,None,9.36|2010-10-24T09:30:00Z,53.516566,8.188217,None,9.39|2010-10-24T09:40:00Z,53.516566,8.188217,None,9.45|2010-10-24T09:50:00Z,53.516566,8.188217,None,9.54|2010-10-24T10:00:00Z,53.516566,8.188217,None,9.59|2010-10-24T10:10:00Z,53.516566,8.188217,None,9.68|2010-10-24T10:20:00Z,53.516566,8.188217,None,9.84|2010-10-24T10:30:00Z,53.516566,8.188217,None,9.91|2010-10-24T10:40:00Z,53.516566,8.188217,None,9.96|2010-10-24T10:50:00Z,53.516566,8.188217,None,10.0|2010-10-24T11:00:00Z,53.516566,8.188217,None,10.07|2010-10-24T11:10:00Z,53.516566,8.188217,None,10.13|2010-10-24T11:20:00Z,53.516566,8.188217,None,10.14|2010-10-24T11:30:00Z,53.516566,8.188217,None,10.14|2010-10-24T11:40:00Z,53.516566,8.188217,None,10.13|2010-10-24T11:50:00Z,53.516566,8.188217,None,10.13|2010-10-24T12:00:00Z,53.516566,8.188217,None,10.13|2010-10-24T12:10:00Z,53.516566,8.188217,None,10.12|2010-10-24T12:20:00Z,53.516566,8.188217,None,10.12|2010-10-24T12:30:00Z,53.516566,8.188217,None,10.12|2010-10-24T12:40:00Z,53.516566,8.188217,None,10.12|2010-10-24T12:50:00Z,53.516566,8.188217,None,10.11|2010-10-24T13:00:00Z,53.516566,8.188217,None,10.1|2010-10-24T13:10:00Z,53.516566,8.188217,None,10.12|2010-10-24T13:20:00Z,53.516566,8.188217,None,10.13|2010-10-24T13:30:00Z,53.516566,8.188217,None,10.07|2010-10-24T13:40:00Z,53.516566,8.188217,None,9.95|2010-10-24T13:50:00Z,53.516566,8.188217,None,9.9|2010-10-24T14:00:00Z,53.516566,8.188217,None,9.85|2010-10-24T14:10:00Z,53.516566,8.188217,None,9.73|2010-10-24T14:20:00Z,53.516566,8.188217,None,9.58|2010-10-24T14:30:00Z,53.516566,8.188217,None,9.46|2010-10-24T14:40:00Z,53.516566,8.188217,None,9.42|2010-10-24T14:50:00Z,53.516566,8.188217,None,9.4|2010-10-24T15:00:00Z,53.516566,8.188217,None,9.46|2010-10-24T15:10:00Z,53.516566,8.188217,None,9.41|2010-10-24T15:20:00Z,53.516566,8.188217,None,9.38|2010-10-24T15:30:00Z,53.516566,8.188217,None,9.42|2010-10-24T15:40:00Z,53.516566,8.188217,None,9.47|2010-10-24T15:50:00Z,53.516566,8.188217,None,9.51|2010-10-24T16:00:00Z,53.516566,8.188217,None,9.55|2010-10-24T16:10:00Z,53.516566,8.188217,None,9.32|2010-10-24T16:20:00Z,53.516566,8.188217,None,9.51|2010-10-24T16:30:00Z,53.516566,8.188217,None,9.48|2010-10-24T16:40:00Z,53.516566,8.188217,None,9.46|2010-10-24T16:50:00Z,53.516566,8.188217,None,9.45|2010-10-24T17:00:00Z,53.516566,8.188217,None,9.44|2010-10-24T17:10:00Z,53.516566,8.188217,None,9.43|2010-10-24T17:20:00Z,53.516566,8.188217,None,9.42|2010-10-24T17:30:00Z,53.516566,8.188217,None,9.41|2010-10-24T17:40:00Z,53.516566,8.188217,None,9.42|2010-10-24T17:50:00Z,53.516566,8.188217,None,9.42|2010-10-24T18:00:00Z,53.516566,8.188217,None,9.4|2010-10-24T18:10:00Z,53.516566,8.188217,None,9.39|2010-10-24T18:20:00Z,53.516566,8.188217,None,9.37|2010-10-24T18:30:00Z,53.516566,8.188217,None,9.32|2010-10-24T18:40:00Z,53.516566,8.188217,None,9.28|2010-10-24T18:50:00Z,53.516566,8.188217,None,9.26|2010-10-24T19:00:00Z,53.516566,8.188217,None,9.26|2010-10-24T19:10:00Z,53.516566,8.188217,None,9.3|2010-10-24T19:20:00Z,53.516566,8.188217,None,9.11|2010-10-24T19:30:00Z,53.516566,8.188217,None,9.28|2010-10-24T19:40:00Z,53.516566,8.188217,None,9.2|2010-10-24T19:50:00Z,53.516566,8.188217,None,9.22|2010-10-24T20:00:00Z,53.516566,8.188217,None,9.23|2010-10-24T20:10:00Z,53.516566,8.188217,None,9.23|2010-10-24T20:20:00Z,53.516566,8.188217,None,9.25|2010-10-24T20:30:00Z,53.516566,8.188217,None,9.2|2010-10-24T20:40:00Z,53.516566,8.188217,None,9.16|2010-10-24T20:50:00Z,53.516566,8.188217,None,9.18|2010-10-24T21:00:00Z,53.516566,8.188217,None,9.23|2010-10-24T21:10:00Z,53.516566,8.188217,None,9.16|2010-10-24T21:20:00Z,53.516566,8.188217,None,9.23|2010-10-24T21:30:00Z,53.516566,8.188217,None,9.14|2010-10-24T21:40:00Z,53.516566,8.188217,None,9.06|2010-10-24T21:50:00Z,53.516566,8.188217,None,9.23|2010-10-24T22:00:00Z,53.516566,8.188217,None,9.3|2010-10-24T22:10:00Z,53.516566,8.188217,None,9.36|2010-10-24T22:20:00Z,53.516566,8.188217,None,9.4|2010-10-24T22:30:00Z,53.516566,8.188217,None,9.41|2010-10-24T22:40:00Z,53.516566,8.188217,None,9.4|2010-10-24T22:50:00Z,53.516566,8.188217,None,9.39|2010-10-24T23:00:00Z,53.516566,8.188217,None,9.43|2010-10-24T23:10:00Z,53.516566,8.188217,None,9.42|2010-10-24T23:20:00Z,53.516566,8.188217,None,9.44|2010-10-24T23:30:00Z,53.516566,8.188217,None,9.48|2010-10-24T23:40:00Z,53.516566,8.188217,None,9.54|2010-10-24T23:50:00Z,53.516566,8.188217,None,9.59|2010-10-25T00:00:00Z,53.516566,8.188217,None,9.63|2010-10-25T00:10:00Z,53.516566,8.188217,None,9.65|2010-10-25T00:20:00Z,53.516566,8.188217,None,9.72|2010-10-25T00:30:00Z,53.516566,8.188217,None,9.8|2010-10-25T00:40:00Z,53.516566,8.188217,None,9.83|2010-10-25T00:50:00Z,53.516566,8.188217,None,9.83|2010-10-25T01:00:00Z,53.516566,8.188217,None,9.81|2010-10-25T01:10:00Z,53.516566,8.188217,None,9.74|2010-10-25T01:20:00Z,53.516566,8.188217,None,9.69|2010-10-25T01:30:00Z,53.516566,8.188217,None,9.62|2010-10-25T01:40:00Z,53.516566,8.188217,None,9.51|2010-10-25T01:50:00Z,53.516566,8.188217,None,9.39|2010-10-25T02:00:00Z,53.516566,8.188217,None,9.29|2010-10-25T02:10:00Z,53.516566,8.188217,None,9.26|2010-10-25T02:20:00Z,53.516566,8.188217,None,9.1|2010-10-25T02:30:00Z,53.516566,8.188217,None,9.0|2010-10-25T02:40:00Z,53.516566,8.188217,None,8.98|2010-10-25T02:50:00Z,53.516566,8.188217,None,9.03|2010-10-25T03:00:00Z,53.516566,8.188217,None,9.07|2010-10-25T03:10:00Z,53.516566,8.188217,None,9.12|2010-10-25T03:20:00Z,53.516566,8.188217,None,9.16|2010-10-25T03:30:00Z,53.516566,8.188217,None,9.14|2010-10-25T03:40:00Z,53.516566,8.188217,None,9.08|2010-10-25T03:50:00Z,53.516566,8.188217,None,9.0|2010-10-25T04:00:00Z,53.516566,8.188217,None,9.01|2010-10-25T04:10:00Z,53.516566,8.188217,None,9.03|2010-10-25T04:20:00Z,53.516566,8.188217,None,8.96|2010-10-25T04:30:00Z,53.516566,8.188217,None,8.95|2010-10-25T04:40:00Z,53.516566,8.188217,None,8.98|2010-10-25T04:50:00Z,53.516566,8.188217,None,9.0|2010-10-25T05:00:00Z,53.516566,8.188217,None,9.02|2010-10-25T05:10:00Z,53.516566,8.188217,None,9.01|2010-10-25T05:20:00Z,53.516566,8.188217,None,8.99|2010-10-25T05:30:00Z,53.516566,8.188217,None,8.95|2010-10-25T05:40:00Z,53.516566,8.188217,None,8.94|2010-10-25T05:50:00Z,53.516566,8.188217,None,8.96|2010-10-25T06:00:00Z,53.516566,8.188217,None,8.99|2010-10-25T06:10:00Z,53.516566,8.188217,None,8.97|2010-10-25T06:20:00Z,53.516566,8.188217,None,8.99|2010-10-25T06:30:00Z,53.516566,8.188217,None,8.81|2010-10-25T06:40:00Z,53.516566,8.188217,None,8.72|2010-10-25T06:50:00Z,53.516566,8.188217,None,8.74|2010-10-25T07:00:00Z,53.516566,8.188217,None,8.85|2010-10-25T07:10:00Z,53.516566,8.188217,None,8.88|2010-10-25T07:20:00Z,53.516566,8.188217,None,8.86|2010-10-25T07:30:00Z,53.516566,8.188217,None,8.86|2010-10-25T07:40:00Z,53.516566,8.188217,None,8.88|2010-10-25T07:50:00Z,53.516566,8.188217,None,8.82|2010-10-25T08:00:00Z,53.516566,8.188217,None,8.77|2010-10-25T08:10:00Z,53.516566,8.188217,None,8.97|2010-10-25T08:20:00Z,53.516566,8.188217,None,9.05|2010-10-25T08:30:00Z,53.516566,8.188217,None,9.05|2010-10-25T08:40:00Z,53.516566,8.188217,None,8.9|2010-10-25T08:50:00Z,53.516566,8.188217,None,8.84|2010-10-25T09:00:00Z,53.516566,8.188217,None,8.87|2010-10-25T09:10:00Z,53.516566,8.188217,None,8.9|2010-10-25T09:20:00Z,53.516566,8.188217,None,8.95|2010-10-25T09:30:00Z,53.516566,8.188217,None,8.97|2010-10-25T09:40:00Z,53.516566,8.188217,None,8.83|2010-10-25T09:50:00Z,53.516566,8.188217,None,8.79|2010-10-25T10:00:00Z,53.516566,8.188217,None,8.87|2010-10-25T10:10:00Z,53.516566,8.188217,None,8.94|2010-10-25T10:20:00Z,53.516566,8.188217,None,9.03|2010-10-25T10:30:00Z,53.516566,8.188217,None,9.09|2010-10-25T10:40:00Z,53.516566,8.188217,None,9.14|2010-10-25T10:50:00Z,53.516566,8.188217,None,9.17|2010-10-25T11:00:00Z,53.516566,8.188217,None,9.17|2010-10-25T11:10:00Z,53.516566,8.188217,None,9.18|2010-10-25T11:20:00Z,53.516566,8.188217,None,9.2|2010-10-25T11:30:00Z,53.516566,8.188217,None,9.22|2010-10-25T11:40:00Z,53.516566,8.188217,None,9.24|2010-10-25T11:50:00Z,53.516566,8.188217,None,9.28|2010-10-25T12:00:00Z,53.516566,8.188217,None,9.34|2010-10-25T12:10:00Z,53.516566,8.188217,None,9.4|2010-10-25T12:20:00Z,53.516566,8.188217,None,9.44|2010-10-25T12:30:00Z,53.516566,8.188217,None,9.46|2010-10-25T12:40:00Z,53.516566,8.188217,None,9.49|2010-10-25T12:50:00Z,53.516566,8.188217,None,9.54|2010-10-25T13:00:00Z,53.516566,8.188217,None,9.57|2010-10-25T13:10:00Z,53.516566,8.188217,None,9.58|2010-10-25T13:20:00Z,53.516566,8.188217,None,9.56|2010-10-25T13:30:00Z,53.516566,8.188217,None,9.55|2010-10-25T13:40:00Z,53.516566,8.188217,None,9.58|2010-10-25T13:50:00Z,53.516566,8.188217,None,9.52|2010-10-25T14:00:00Z,53.516566,8.188217,None,9.47|2010-10-25T14:10:00Z,53.516566,8.188217,None,9.38|2010-10-25T14:20:00Z,53.516566,8.188217,None,9.29|2010-10-25T14:30:00Z,53.516566,8.188217,None,9.25|2010-10-25T14:40:00Z,53.516566,8.188217,None,9.12|2010-10-25T14:50:00Z,53.516566,8.188217,None,9.06|2010-10-25T15:00:00Z,53.516566,8.188217,None,9.05|2010-10-25T15:10:00Z,53.516566,8.188217,None,9.04|2010-10-25T15:20:00Z,53.516566,8.188217,None,9.05|2010-10-25T15:30:00Z,53.516566,8.188217,None,9.04|2010-10-25T15:40:00Z,53.516566,8.188217,None,9.03|2010-10-25T15:50:00Z,53.516566,8.188217,None,9.0|2010-10-25T16:00:00Z,53.516566,8.188217,None,8.96|2010-10-25T16:10:00Z,53.516566,8.188217,None,8.75|2010-10-25T16:20:00Z,53.516566,8.188217,None,8.98|2010-10-25T16:30:00Z,53.516566,8.188217,None,9.01|2010-10-25T16:40:00Z,53.516566,8.188217,None,9.04|2010-10-25T16:50:00Z,53.516566,8.188217,None,9.06|2010-10-25T17:00:00Z,53.516566,8.188217,None,9.08|2010-10-25T17:10:00Z,53.516566,8.188217,None,9.07|2010-10-25T17:20:00Z,53.516566,8.188217,None,8.85|2010-10-25T17:30:00Z,53.516566,8.188217,None,9.03|2010-10-25T17:40:00Z,53.516566,8.188217,None,9.02|2010-10-25T17:50:00Z,53.516566,8.188217,None,9.01|2010-10-25T18:00:00Z,53.516566,8.188217,None,9.0|2010-10-25T18:10:00Z,53.516566,8.188217,None,9.0|2010-10-25T18:20:00Z,53.516566,8.188217,None,8.99|2010-10-25T18:30:00Z,53.516566,8.188217,None,8.97|2010-10-25T18:40:00Z,53.516566,8.188217,None,8.92|2010-10-25T18:50:00Z,53.516566,8.188217,None,8.87|2010-10-25T19:00:00Z,53.516566,8.188217,None,8.84|2010-10-25T19:10:00Z,53.516566,8.188217,None,8.83|2010-10-25T19:20:00Z,53.516566,8.188217,None,8.83|2010-10-25T19:30:00Z,53.516566,8.188217,None,8.84|2010-10-25T19:40:00Z,53.516566,8.188217,None,8.84|2010-10-25T19:50:00Z,53.516566,8.188217,None,8.84|2010-10-25T20:00:00Z,53.516566,8.188217,None,8.83|2010-10-25T20:10:00Z,53.516566,8.188217,None,8.79|2010-10-25T20:20:00Z,53.516566,8.188217,None,8.76|2010-10-25T20:30:00Z,53.516566,8.188217,None,8.82|2010-10-25T20:40:00Z,53.516566,8.188217,None,8.84|2010-10-25T20:50:00Z,53.516566,8.188217,None,8.96|2010-10-25T21:00:00Z,53.516566,8.188217,None,9.01|2010-10-25T21:10:00Z,53.516566,8.188217,None,8.98|2010-10-25T21:20:00Z,53.516566,8.188217,None,8.96|2010-10-25T21:30:00Z,53.516566,8.188217,None,8.97|2010-10-25T21:40:00Z,53.516566,8.188217,None,8.92|2010-10-25T21:50:00Z,53.516566,8.188217,None,9.05|2010-10-25T22:00:00Z,53.516566,8.188217,None,9.06|2010-10-25T22:10:00Z,53.516566,8.188217,None,8.98|2010-10-25T22:20:00Z,53.516566,8.188217,None,8.98|2010-10-25T22:30:00Z,53.516566,8.188217,None,9.02|2010-10-25T22:40:00Z,53.516566,8.188217,None,9.07|2010-10-25T22:50:00Z,53.516566,8.188217,None,9.1|2010-10-25T23:00:00Z,53.516566,8.188217,None,9.1|2010-10-25T23:10:00Z,53.516566,8.188217,None,9.12|2010-10-25T23:20:00Z,53.516566,8.188217,None,9.16|2010-10-25T23:30:00Z,53.516566,8.188217,None,9.23|2010-10-25T23:40:00Z,53.516566,8.188217,None,9.25|2010-10-25T23:50:00Z,53.516566,8.188217,None,9.27|2010-10-26T00:00:00Z,53.516566,8.188217,None,9.29|2010-10-26T00:10:00Z,53.516566,8.188217,None,9.3|2010-10-26T00:20:00Z,53.516566,8.188217,None,9.33|2010-10-26T00:30:00Z,53.516566,8.188217,None,9.35|2010-10-26T00:40:00Z,53.516566,8.188217,None,9.36|2010-10-26T00:50:00Z,53.516566,8.188217,None,9.38|2010-10-26T01:00:00Z,53.516566,8.188217,None,9.39|2010-10-26T01:10:00Z,53.516566,8.188217,None,9.4|2010-10-26T01:20:00Z,53.516566,8.188217,None,9.42|2010-10-26T01:30:00Z,53.516566,8.188217,None,9.41|2010-10-26T01:40:00Z,53.516566,8.188217,None,9.39|2010-10-26T01:50:00Z,53.516566,8.188217,None,9.35|2010-10-26T02:00:00Z,53.516566,8.188217,None,9.32|2010-10-26T02:10:00Z,53.516566,8.188217,None,9.27|2010-10-26T02:20:00Z,53.516566,8.188217,None,9.22|2010-10-26T02:30:00Z,53.516566,8.188217,None,9.15|2010-10-26T02:40:00Z,53.516566,8.188217,None,9.07|2010-10-26T02:50:00Z,53.516566,8.188217,None,9.0|2010-10-26T03:00:00Z,53.516566,8.188217,None,8.94|2010-10-26T03:10:00Z,53.516566,8.188217,None,8.9|2010-10-26T03:20:00Z,53.516566,8.188217,None,8.9|2010-10-26T03:30:00Z,53.516566,8.188217,None,8.91|2010-10-26T03:40:00Z,53.516566,8.188217,None,8.89|2010-10-26T03:50:00Z,53.516566,8.188217,None,8.83|2010-10-26T04:00:00Z,53.516566,8.188217,None,8.85|2010-10-26T04:10:00Z,53.516566,8.188217,None,8.88|2010-10-26T04:20:00Z,53.516566,8.188217,None,8.9|2010-10-26T04:30:00Z,53.516566,8.188217,None,8.9|2010-10-26T04:40:00Z,53.516566,8.188217,None,8.91|2010-10-26T04:50:00Z,53.516566,8.188217,None,8.93|2010-10-26T05:00:00Z,53.516566,8.188217,None,8.94|2010-10-26T05:10:00Z,53.516566,8.188217,None,8.94|2010-10-26T05:20:00Z,53.516566,8.188217,None,8.93|2010-10-26T05:30:00Z,53.516566,8.188217,None,8.93|2010-10-26T05:40:00Z,53.516566,8.188217,None,8.93|2010-10-26T05:50:00Z,53.516566,8.188217,None,8.94|2010-10-26T06:00:00Z,53.516566,8.188217,None,8.94|2010-10-26T06:10:00Z,53.516566,8.188217,None,8.95|2010-10-26T06:20:00Z,53.516566,8.188217,None,8.98|2010-10-26T06:30:00Z,53.516566,8.188217,None,8.97|2010-10-26T06:40:00Z,53.516566,8.188217,None,8.85|2010-10-26T06:50:00Z,53.516566,8.188217,None,8.79|2010-10-26T07:00:00Z,53.516566,8.188217,None,8.71|2010-10-26T07:10:00Z,53.516566,8.188217,None,8.72|2010-10-26T07:20:00Z,53.516566,8.188217,None,8.81|2010-10-26T07:30:00Z,53.516566,8.188217,None,8.92|2010-10-26T07:40:00Z,53.516566,8.188217,None,8.95|2010-10-26T07:50:00Z,53.516566,8.188217,None,8.88|2010-10-26T08:00:00Z,53.516566,8.188217,None,8.82|2010-10-26T08:10:00Z,53.516566,8.188217,None,8.82|2010-10-26T08:20:00Z,53.516566,8.188217,None,8.88|2010-10-26T08:30:00Z,53.516566,8.188217,None,8.76|2010-10-26T08:40:00Z,53.516566,8.188217,None,8.69|2010-10-26T08:50:00Z,53.516566,8.188217,None,8.67|2010-10-26T09:00:00Z,53.516566,8.188217,None,8.76|2010-10-26T09:10:00Z,53.516566,8.188217,None,8.82|2010-10-26T09:20:00Z,53.516566,8.188217,None,8.8|2010-10-26T09:30:00Z,53.516566,8.188217,None,8.79|2010-10-26T09:40:00Z,53.516566,8.188217,None,8.82|2010-10-26T09:50:00Z,53.516566,8.188217,None,8.84|2010-10-26T10:00:00Z,53.516566,8.188217,None,8.85|2010-10-26T10:10:00Z,53.516566,8.188217,None,8.88|2010-10-26T10:20:00Z,53.516566,8.188217,None,8.96|2010-10-26T10:30:00Z,53.516566,8.188217,None,8.86|2010-10-26T10:40:00Z,53.516566,8.188217,None,8.92|2010-10-26T10:50:00Z,53.516566,8.188217,None,8.99|2010-10-26T11:00:00Z,53.516566,8.188217,None,9.02|2010-10-26T11:10:00Z,53.516566,8.188217,None,9.04|2010-10-26T11:20:00Z,53.516566,8.188217,None,9.05|2010-10-26T11:30:00Z,53.516566,8.188217,None,9.04|2010-10-26T11:40:00Z,53.516566,8.188217,None,9.02|2010-10-26T11:50:00Z,53.516566,8.188217,None,9.02|2010-10-26T12:00:00Z,53.516566,8.188217,None,9.05|2010-10-26T12:10:00Z,53.516566,8.188217,None,9.07|2010-10-26T12:20:00Z,53.516566,8.188217,None,9.08|2010-10-26T12:30:00Z,53.516566,8.188217,None,9.1|2010-10-26T12:40:00Z,53.516566,8.188217,None,9.13|2010-10-26T12:50:00Z,53.516566,8.188217,None,9.17|2010-10-26T13:00:00Z,53.516566,8.188217,None,9.21|2010-10-26T13:10:00Z,53.516566,8.188217,None,9.23|2010-10-26T13:20:00Z,53.516566,8.188217,None,9.25|2010-10-26T13:30:00Z,53.516566,8.188217,None,9.24|2010-10-26T13:40:00Z,53.516566,8.188217,None,9.25|2010-10-26T13:50:00Z,53.516566,8.188217,None,9.25|2010-10-26T14:00:00Z,53.516566,8.188217,None,9.24|2010-10-26T14:10:00Z,53.516566,8.188217,None,9.21|2010-10-26T14:20:00Z,53.516566,8.188217,None,9.17|2010-10-26T14:30:00Z,53.516566,8.188217,None,9.1|2010-10-26T14:40:00Z,53.516566,8.188217,None,9.05|2010-10-26T14:50:00Z,53.516566,8.188217,None,9.05|2010-10-26T15:00:00Z,53.516566,8.188217,None,8.99|2010-10-26T15:10:00Z,53.516566,8.188217,None,8.96|2010-10-26T15:20:00Z,53.516566,8.188217,None,8.97|2010-10-26T15:30:00Z,53.516566,8.188217,None,8.97|2010-10-26T15:40:00Z,53.516566,8.188217,None,8.95|2010-10-26T15:50:00Z,53.516566,8.188217,None,8.93|2010-10-26T16:00:00Z,53.516566,8.188217,None,8.88|2010-10-26T16:10:00Z,53.516566,8.188217,None,8.87|2010-10-26T16:20:00Z,53.516566,8.188217,None,8.89|2010-10-26T16:30:00Z,53.516566,8.188217,None,8.92|2010-10-26T16:40:00Z,53.516566,8.188217,None,8.93|2010-10-26T16:50:00Z,53.516566,8.188217,None,8.95|2010-10-26T17:00:00Z,53.516566,8.188217,None,8.95|2010-10-26T17:10:00Z,53.516566,8.188217,None,8.96|2010-10-26T17:20:00Z,53.516566,8.188217,None,8.96|2010-10-26T17:30:00Z,53.516566,8.188217,None,8.96|2010-10-26T17:40:00Z,53.516566,8.188217,None,8.95|2010-10-26T17:50:00Z,53.516566,8.188217,None,8.95|2010-10-26T18:00:00Z,53.516566,8.188217,None,8.95|2010-10-26T18:10:00Z,53.516566,8.188217,None,8.98|2010-10-26T18:20:00Z,53.516566,8.188217,None,9.01|2010-10-26T18:30:00Z,53.516566,8.188217,None,9.0|2010-10-26T18:40:00Z,53.516566,8.188217,None,8.92|2010-10-26T18:50:00Z,53.516566,8.188217,None,8.85|2010-10-26T19:00:00Z,53.516566,8.188217,None,8.9|2010-10-26T19:10:00Z,53.516566,8.188217,None,9.01|2010-10-26T19:20:00Z,53.516566,8.188217,None,9.0|2010-10-26T19:30:00Z,53.516566,8.188217,None,8.97|2010-10-26T19:40:00Z,53.516566,8.188217,None,8.98|2010-10-26T19:50:00Z,53.516566,8.188217,None,8.92|2010-10-26T20:00:00Z,53.516566,8.188217,None,8.9|2010-10-26T20:10:00Z,53.516566,8.188217,None,8.86|2010-10-26T20:20:00Z,53.516566,8.188217,None,8.84|2010-10-26T20:30:00Z,53.516566,8.188217,None,8.78|2010-10-26T20:40:00Z,53.516566,8.188217,None,8.84|2010-10-26T20:50:00Z,53.516566,8.188217,None,8.85|2010-10-26T21:00:00Z,53.516566,8.188217,None,8.85|2010-10-26T21:10:00Z,53.516566,8.188217,None,8.83|2010-10-26T21:20:00Z,53.516566,8.188217,None,8.87|2010-10-26T21:30:00Z,53.516566,8.188217,None,8.87|2010-10-26T21:40:00Z,53.516566,8.188217,None,8.87|2010-10-26T21:50:00Z,53.516566,8.188217,None,8.91|2010-10-26T22:00:00Z,53.516566,8.188217,None,8.89|2010-10-26T22:10:00Z,53.516566,8.188217,None,8.87|2010-10-26T22:20:00Z,53.516566,8.188217,None,8.86|2010-10-26T22:30:00Z,53.516566,8.188217,None,8.84|2010-10-26T22:40:00Z,53.516566,8.188217,None,8.9|2010-10-26T22:50:00Z,53.516566,8.188217,None,8.9|2010-10-26T23:00:00Z,53.516566,8.188217,None,8.92|2010-10-26T23:10:00Z,53.516566,8.188217,None,8.96|2010-10-26T23:20:00Z,53.516566,8.188217,None,8.99|2010-10-26T23:30:00Z,53.516566,8.188217,None,9.01|2010-10-26T23:40:00Z,53.516566,8.188217,None,9.01|2010-10-26T23:50:00Z,53.516566,8.188217,None,9.01|2010-10-27T00:00:00Z,53.516566,8.188217,None,9.03|2010-10-27T00:10:00Z,53.516566,8.188217,None,9.03|2010-10-27T00:20:00Z,53.516566,8.188217,None,9.03|2010-10-27T00:30:00Z,53.516566,8.188217,None,9.05|2010-10-27T00:40:00Z,53.516566,8.188217,None,9.06|2010-10-27T00:50:00Z,53.516566,8.188217,None,9.05|2010-10-27T01:00:00Z,53.516566,8.188217,None,9.05|2010-10-27T01:10:00Z,53.516566,8.188217,None,9.07|2010-10-27T01:20:00Z,53.516566,8.188217,None,9.08|2010-10-27T01:30:00Z,53.516566,8.188217,None,9.09|2010-10-27T01:40:00Z,53.516566,8.188217,None,9.08|2010-10-27T01:50:00Z,53.516566,8.188217,None,9.08|2010-10-27T02:00:00Z,53.516566,8.188217,None,9.07|2010-10-27T02:10:00Z,53.516566,8.188217,None,9.04|2010-10-27T02:20:00Z,53.516566,8.188217,None,8.99|2010-10-27T02:30:00Z,53.516566,8.188217,None,8.96|2010-10-27T02:40:00Z,53.516566,8.188217,None,8.9|2010-10-27T02:50:00Z,53.516566,8.188217,None,8.79|2010-10-27T03:00:00Z,53.516566,8.188217,None,8.6|2010-10-27T03:10:00Z,53.516566,8.188217,None,8.54|2010-10-27T03:20:00Z,53.516566,8.188217,None,8.54|2010-10-27T03:30:00Z,53.516566,8.188217,None,8.58|2010-10-27T03:40:00Z,53.516566,8.188217,None,8.7|2010-10-27T03:50:00Z,53.516566,8.188217,None,8.76|2010-10-27T04:00:00Z,53.516566,8.188217,None,8.77|2010-10-27T04:10:00Z,53.516566,8.188217,None,8.77|2010-10-27T04:20:00Z,53.516566,8.188217,None,8.77|2010-10-27T04:30:00Z,53.516566,8.188217,None,8.79|2010-10-27T04:40:00Z,53.516566,8.188217,None,8.81|2010-10-27T04:50:00Z,53.516566,8.188217,None,8.84|2010-10-27T05:00:00Z,53.516566,8.188217,None,8.86|2010-10-27T05:10:00Z,53.516566,8.188217,None,8.86|2010-10-27T05:20:00Z,53.516566,8.188217,None,8.86|2010-10-27T05:30:00Z,53.516566,8.188217,None,8.85|2010-10-27T05:40:00Z,53.516566,8.188217,None,8.84|2010-10-27T05:50:00Z,53.516566,8.188217,None,8.83|2010-10-27T06:00:00Z,53.516566,8.188217,None,8.81|2010-10-27T06:10:00Z,53.516566,8.188217,None,8.78|2010-10-27T06:20:00Z,53.516566,8.188217,None,8.76|2010-10-27T06:30:00Z,53.516566,8.188217,None,8.76|2010-10-27T06:40:00Z,53.516566,8.188217,None,8.75|2010-10-27T06:50:00Z,53.516566,8.188217,None,8.74|2010-10-27T07:00:00Z,53.516566,8.188217,None,8.73|2010-10-27T07:10:00Z,53.516566,8.188217,None,8.73|2010-10-27T07:20:00Z,53.516566,8.188217,None,8.76|2010-10-27T07:30:00Z,53.516566,8.188217,None,8.78|2010-10-27T07:40:00Z,53.516566,8.188217,None,8.59|2010-10-27T07:50:00Z,53.516566,8.188217,None,8.8|2010-10-27T08:00:00Z,53.516566,8.188217,None,8.77|2010-10-27T08:10:00Z,53.516566,8.188217,None,8.71|2010-10-27T08:20:00Z,53.516566,8.188217,None,8.69|2010-10-27T08:30:00Z,53.516566,8.188217,None,8.69|2010-10-27T08:40:00Z,53.516566,8.188217,None,8.69|2010-10-27T08:50:00Z,53.516566,8.188217,None,8.69|2010-10-27T09:00:00Z,53.516566,8.188217,None,8.65|2010-10-27T09:10:00Z,53.516566,8.188217,None,8.63|2010-10-27T09:20:00Z,53.516566,8.188217,None,8.62|2010-10-27T09:30:00Z,53.516566,8.188217,None,8.59|2010-10-27T09:40:00Z,53.516566,8.188217,None,8.66|2010-10-27T09:50:00Z,53.516566,8.188217,None,8.7|2010-10-27T10:00:00Z,53.516566,8.188217,None,8.74|2010-10-27T10:10:00Z,53.516566,8.188217,None,8.76|2010-10-27T10:20:00Z,53.516566,8.188217,None,8.76|2010-10-27T10:30:00Z,53.516566,8.188217,None,8.79|2010-10-27T10:40:00Z,53.516566,8.188217,None,8.85|2010-10-27T10:50:00Z,53.516566,8.188217,None,8.85|2010-10-27T11:00:00Z,53.516566,8.188217,None,8.8|2010-10-27T11:10:00Z,53.516566,8.188217,None,8.74|2010-10-27T11:20:00Z,53.516566,8.188217,None,8.72|2010-10-27T11:30:00Z,53.516566,8.188217,None,8.75|2010-10-27T11:40:00Z,53.516566,8.188217,None,8.79|2010-10-27T11:50:00Z,53.516566,8.188217,None,8.81|2010-10-27T12:00:00Z,53.516566,8.188217,None,8.86|2010-10-27T12:10:00Z,53.516566,8.188217,None,8.9|2010-10-27T12:20:00Z,53.516566,8.188217,None,8.92|2010-10-27T12:30:00Z,53.516566,8.188217,None,8.94|2010-10-27T12:40:00Z,53.516566,8.188217,None,8.96|2010-10-27T12:50:00Z,53.516566,8.188217,None,9.0|2010-10-27T13:00:00Z,53.516566,8.188217,None,9.05|2010-10-27T13:10:00Z,53.516566,8.188217,None,9.06|2010-10-27T13:20:00Z,53.516566,8.188217,None,9.07|2010-10-27T13:30:00Z,53.516566,8.188217,None,9.09|2010-10-27T13:40:00Z,53.516566,8.188217,None,9.11|2010-10-27T13:50:00Z,53.516566,8.188217,None,9.12|2010-10-27T14:00:00Z,53.516566,8.188217,None,9.13|2010-10-27T14:10:00Z,53.516566,8.188217,None,9.13|2010-10-27T14:20:00Z,53.516566,8.188217,None,9.13|2010-10-27T14:30:00Z,53.516566,8.188217,None,9.12|2010-10-27T14:40:00Z,53.516566,8.188217,None,9.1|2010-10-27T14:50:00Z,53.516566,8.188217,None,9.07|2010-10-27T15:00:00Z,53.516566,8.188217,None,9.02|2010-10-27T15:10:00Z,53.516566,8.188217,None,8.94|2010-10-27T15:20:00Z,53.516566,8.188217,None,8.85|2010-10-27T15:30:00Z,53.516566,8.188217,None,8.79|2010-10-27T15:40:00Z,53.516566,8.188217,None,8.75|2010-10-27T15:50:00Z,53.516566,8.188217,None,8.63|2010-10-27T16:00:00Z,53.516566,8.188217,None,8.64|2010-10-27T16:10:00Z,53.516566,8.188217,None,8.65|2010-10-27T16:20:00Z,53.516566,8.188217,None,8.71|2010-10-27T16:30:00Z,53.516566,8.188217,None,8.77|2010-10-27T16:40:00Z,53.516566,8.188217,None,8.79|2010-10-27T16:50:00Z,53.516566,8.188217,None,8.8|2010-10-27T17:00:00Z,53.516566,8.188217,None,8.81|2010-10-27T17:10:00Z,53.516566,8.188217,None,8.82|2010-10-27T17:20:00Z,53.516566,8.188217,None,8.84|2010-10-27T17:30:00Z,53.516566,8.188217,None,8.85|2010-10-27T17:40:00Z,53.516566,8.188217,None,8.86|2010-10-27T17:50:00Z,53.516566,8.188217,None,8.88|2010-10-27T18:00:00Z,53.516566,8.188217,None,8.89|2010-10-27T18:10:00Z,53.516566,8.188217,None,8.89|2010-10-27T18:20:00Z,53.516566,8.188217,None,8.89|2010-10-27T18:30:00Z,53.516566,8.188217,None,8.88|2010-10-27T18:40:00Z,53.516566,8.188217,None,8.87|2010-10-27T18:50:00Z,53.516566,8.188217,None,8.86|2010-10-27T19:00:00Z,53.516566,8.188217,None,8.85|2010-10-27T19:10:00Z,53.516566,8.188217,None,8.86|2010-10-27T19:20:00Z,53.516566,8.188217,None,8.85|2010-10-27T19:30:00Z,53.516566,8.188217,None,8.85|2010-10-27T19:40:00Z,53.516566,8.188217,None,8.86|2010-10-27T19:50:00Z,53.516566,8.188217,None,8.88|2010-10-27T20:00:00Z,53.516566,8.188217,None,8.89|2010-10-27T20:10:00Z,53.516566,8.188217,None,8.89|2010-10-27T20:20:00Z,53.516566,8.188217,None,8.87|2010-10-27T20:30:00Z,53.516566,8.188217,None,8.86|2010-10-27T20:40:00Z,53.516566,8.188217,None,8.86|2010-10-27T20:50:00Z,53.516566,8.188217,None,8.84|2010-10-27T21:00:00Z,53.516566,8.188217,None,8.84|2010-10-27T21:10:00Z,53.516566,8.188217,None,8.85|2010-10-27T21:20:00Z,53.516566,8.188217,None,8.81|2010-10-27T21:30:00Z,53.516566,8.188217,None,8.73|2010-10-27T21:40:00Z,53.516566,8.188217,None,8.79|2010-10-27T21:50:00Z,53.516566,8.188217,None,8.82|2010-10-27T22:00:00Z,53.516566,8.188217,None,8.85|2010-10-27T22:10:00Z,53.516566,8.188217,None,8.83|2010-10-27T22:20:00Z,53.516566,8.188217,None,8.85|2010-10-27T22:30:00Z,53.516566,8.188217,None,8.84|2010-10-27T22:40:00Z,53.516566,8.188217,None,8.83|2010-10-27T22:50:00Z,53.516566,8.188217,None,8.84|2010-10-27T23:00:00Z,53.516566,8.188217,None,8.82|2010-10-27T23:10:00Z,53.516566,8.188217,None,8.87|2010-10-27T23:20:00Z,53.516566,8.188217,None,8.9|2010-10-27T23:30:00Z,53.516566,8.188217,None,8.89|2010-10-27T23:40:00Z,53.516566,8.188217,None,8.89|2010-10-27T23:50:00Z,53.516566,8.188217,None,8.9|2010-10-28T00:00:00Z,53.516566,8.188217,None,8.91|2010-10-28T00:10:00Z,53.516566,8.188217,None,8.93|2010-10-28T00:20:00Z,53.516566,8.188217,None,8.93|2010-10-28T00:30:00Z,53.516566,8.188217,None,8.95|2010-10-28T00:40:00Z,53.516566,8.188217,None,8.96|2010-10-28T00:50:00Z,53.516566,8.188217,None,8.96|2010-10-28T01:00:00Z,53.516566,8.188217,None,9.01|2010-10-28T01:10:00Z,53.516566,8.188217,None,9.09|2010-10-28T01:20:00Z,53.516566,8.188217,None,9.12|2010-10-28T01:30:00Z,53.516566,8.188217,None,9.18|2010-10-28T01:40:00Z,53.516566,8.188217,None,9.26|2010-10-28T01:50:00Z,53.516566,8.188217,None,9.26|2010-10-28T02:00:00Z,53.516566,8.188217,None,9.24|2010-10-28T02:10:00Z,53.516566,8.188217,None,9.24|2010-10-28T02:20:00Z,53.516566,8.188217,None,9.24|2010-10-28T02:30:00Z,53.516566,8.188217,None,9.24|2010-10-28T02:40:00Z,53.516566,8.188217,None,9.22|2010-10-28T02:50:00Z,53.516566,8.188217,None,9.21|2010-10-28T03:00:00Z,53.516566,8.188217,None,9.21|2010-10-28T03:10:00Z,53.516566,8.188217,None,9.07|2010-10-28T03:20:00Z,53.516566,8.188217,None,8.93|2010-10-28T03:30:00Z,53.516566,8.188217,None,8.89|2010-10-28T03:40:00Z,53.516566,8.188217,None,8.86|2010-10-28T03:50:00Z,53.516566,8.188217,None,8.83|2010-10-28T04:00:00Z,53.516566,8.188217,None,8.82|2010-10-28T04:10:00Z,53.516566,8.188217,None,8.84|2010-10-28T04:20:00Z,53.516566,8.188217,None,8.83|2010-10-28T04:30:00Z,53.516566,8.188217,None,8.83|2010-10-28T04:40:00Z,53.516566,8.188217,None,8.84|2010-10-28T04:50:00Z,53.516566,8.188217,None,8.84|2010-10-28T05:00:00Z,53.516566,8.188217,None,8.88|2010-10-28T05:10:00Z,53.516566,8.188217,None,8.92|2010-10-28T05:20:00Z,53.516566,8.188217,None,8.95|2010-10-28T05:30:00Z,53.516566,8.188217,None,8.94|2010-10-28T05:40:00Z,53.516566,8.188217,None,8.93|2010-10-28T05:50:00Z,53.516566,8.188217,None,8.92|2010-10-28T06:00:00Z,53.516566,8.188217,None,8.9|2010-10-28T06:10:00Z,53.516566,8.188217,None,8.88|2010-10-28T06:20:00Z,53.516566,8.188217,None,8.86|2010-10-28T06:30:00Z,53.516566,8.188217,None,8.85|2010-10-28T06:40:00Z,53.516566,8.188217,None,8.85|2010-10-28T06:50:00Z,53.516566,8.188217,None,8.86|2010-10-28T07:00:00Z,53.516566,8.188217,None,8.86|2010-10-28T07:10:00Z,53.516566,8.188217,None,8.86|2010-10-28T07:20:00Z,53.516566,8.188217,None,8.86|2010-10-28T07:30:00Z,53.516566,8.188217,None,8.85|2010-10-28T07:40:00Z,53.516566,8.188217,None,8.86|2010-10-28T07:50:00Z,53.516566,8.188217,None,8.87|2010-10-28T08:00:00Z,53.516566,8.188217,None,8.88|2010-10-28T08:10:00Z,53.516566,8.188217,None,8.89|2010-10-28T08:20:00Z,53.516566,8.188217,None,8.89|2010-10-28T08:30:00Z,53.516566,8.188217,None,8.87|2010-10-28T08:40:00Z,53.516566,8.188217,None,8.86|2010-10-28T08:50:00Z,53.516566,8.188217,None,8.85|2010-10-28T09:00:00Z,53.516566,8.188217,None,8.86|2010-10-28T09:10:00Z,53.516566,8.188217,None,8.86|2010-10-28T09:20:00Z,53.516566,8.188217,None,8.88|2010-10-28T09:30:00Z,53.516566,8.188217,None,8.87|2010-10-28T09:40:00Z,53.516566,8.188217,None,8.86|2010-10-28T09:50:00Z,53.516566,8.188217,None,8.86|2010-10-28T10:00:00Z,53.516566,8.188217,None,8.88|2010-10-28T10:10:00Z,53.516566,8.188217,None,8.89|2010-10-28T10:20:00Z,53.516566,8.188217,None,8.9|2010-10-28T10:30:00Z,53.516566,8.188217,None,8.91|2010-10-28T10:40:00Z,53.516566,8.188217,None,8.92|2010-10-28T10:50:00Z,53.516566,8.188217,None,8.89|2010-10-28T11:00:00Z,53.516566,8.188217,None,8.91|2010-10-28T11:10:00Z,53.516566,8.188217,None,8.93|2010-10-28T11:20:00Z,53.516566,8.188217,None,8.92|2010-10-28T11:30:00Z,53.516566,8.188217,None,8.9|2010-10-28T11:40:00Z,53.516566,8.188217,None,8.89|2010-10-28T11:50:00Z,53.516566,8.188217,None,8.87|2010-10-28T12:00:00Z,53.516566,8.188217,None,8.91|2010-10-28T12:10:00Z,53.516566,8.188217,None,8.95|2010-10-28T12:20:00Z,53.516566,8.188217,None,8.96|2010-10-28T12:30:00Z,53.516566,8.188217,None,8.97|2010-10-28T12:40:00Z,53.516566,8.188217,None,8.99|2010-10-28T12:50:00Z,53.516566,8.188217,None,8.98|2010-10-28T13:00:00Z,53.516566,8.188217,None,8.97|2010-10-28T13:10:00Z,53.516566,8.188217,None,8.97|2010-10-28T13:20:00Z,53.516566,8.188217,None,8.98|2010-10-28T13:30:00Z,53.516566,8.188217,None,8.99|2010-10-28T13:40:00Z,53.516566,8.188217,None,9.0|2010-10-28T13:50:00Z,53.516566,8.188217,None,9.03|2010-10-28T14:00:00Z,53.516566,8.188217,None,9.11|2010-10-28T14:10:00Z,53.516566,8.188217,None,9.13|2010-10-28T14:20:00Z,53.516566,8.188217,None,9.14|2010-10-28T14:30:00Z,53.516566,8.188217,None,9.14|2010-10-28T14:40:00Z,53.516566,8.188217,None,9.17|2010-10-28T14:50:00Z,53.516566,8.188217,None,9.19|2010-10-28T15:00:00Z,53.516566,8.188217,None,9.19|2010-10-28T15:10:00Z,53.516566,8.188217,None,9.17|2010-10-28T15:20:00Z,53.516566,8.188217,None,9.13|2010-10-28T15:30:00Z,53.516566,8.188217,None,9.07|2010-10-28T15:40:00Z,53.516566,8.188217,None,9.01|2010-10-28T15:50:00Z,53.516566,8.188217,None,9.02|2010-10-28T16:00:00Z,53.516566,8.188217,None,9.01|2010-10-28T16:10:00Z,53.516566,8.188217,None,9.01|2010-10-28T16:20:00Z,53.516566,8.188217,None,9.04|2010-10-28T16:30:00Z,53.516566,8.188217,None,9.03|2010-10-28T16:40:00Z,53.516566,8.188217,None,9.04|2010-10-28T16:50:00Z,53.516566,8.188217,None,9.05|2010-10-28T17:00:00Z,53.516566,8.188217,None,9.02|2010-10-28T17:10:00Z,53.516566,8.188217,None,8.99|2010-10-28T17:20:00Z,53.516566,8.188217,None,9.0|2010-10-28T17:30:00Z,53.516566,8.188217,None,9.03|2010-10-28T17:40:00Z,53.516566,8.188217,None,9.04|2010-10-28T17:50:00Z,53.516566,8.188217,None,9.05|2010-10-28T18:00:00Z,53.516566,8.188217,None,9.04|2010-10-28T18:10:00Z,53.516566,8.188217,None,9.02|2010-10-28T18:20:00Z,53.516566,8.188217,None,9.01|2010-10-28T18:30:00Z,53.516566,8.188217,None,9.01|2010-10-28T18:40:00Z,53.516566,8.188217,None,9.0|2010-10-28T18:50:00Z,53.516566,8.188217,None,8.99|2010-10-28T19:00:00Z,53.516566,8.188217,None,8.98|2010-10-28T19:10:00Z,53.516566,8.188217,None,8.98|2010-10-28T19:20:00Z,53.516566,8.188217,None,8.98|2010-10-28T19:30:00Z,53.516566,8.188217,None,9.0|2010-10-28T19:40:00Z,53.516566,8.188217,None,9.01|2010-10-28T19:50:00Z,53.516566,8.188217,None,9.01|2010-10-28T20:00:00Z,53.516566,8.188217,None,9.01|2010-10-28T20:10:00Z,53.516566,8.188217,None,9.04|2010-10-28T20:20:00Z,53.516566,8.188217,None,9.06|2010-10-28T20:30:00Z,53.516566,8.188217,None,9.06|2010-10-28T20:40:00Z,53.516566,8.188217,None,9.0|2010-10-28T20:50:00Z,53.516566,8.188217,None,8.99|2010-10-28T21:00:00Z,53.516566,8.188217,None,8.99|2010-10-28T21:10:00Z,53.516566,8.188217,None,9.02|2010-10-28T21:20:00Z,53.516566,8.188217,None,9.04|2010-10-28T21:30:00Z,53.516566,8.188217,None,9.04|2010-10-28T21:40:00Z,53.516566,8.188217,None,9.04|2010-10-28T21:50:00Z,53.516566,8.188217,None,9.03|2010-10-28T22:00:00Z,53.516566,8.188217,None,9.03|2010-10-28T22:10:00Z,53.516566,8.188217,None,9.05|2010-10-28T22:20:00Z,53.516566,8.188217,None,9.09|2010-10-28T22:30:00Z,53.516566,8.188217,None,9.11|2010-10-28T22:40:00Z,53.516566,8.188217,None,9.11|2010-10-28T22:50:00Z,53.516566,8.188217,None,9.12|2010-10-28T23:00:00Z,53.516566,8.188217,None,9.1|2010-10-28T23:10:00Z,53.516566,8.188217,None,9.13|2010-10-28T23:20:00Z,53.516566,8.188217,None,9.09|2010-10-28T23:30:00Z,53.516566,8.188217,None,9.0|2010-10-28T23:40:00Z,53.516566,8.188217,None,9.02|2010-10-28T23:50:00Z,53.516566,8.188217,None,9.04|2010-10-29T00:00:00Z,53.516566,8.188217,None,9.04|2010-10-29T00:10:00Z,53.516566,8.188217,None,9.06|2010-10-29T00:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T00:30:00Z,53.516566,8.188217,None,9.06|2010-10-29T00:40:00Z,53.516566,8.188217,None,9.06|2010-10-29T00:50:00Z,53.516566,8.188217,None,9.07|2010-10-29T01:00:00Z,53.516566,8.188217,None,9.08|2010-10-29T01:10:00Z,53.516566,8.188217,None,9.09|2010-10-29T01:20:00Z,53.516566,8.188217,None,9.09|2010-10-29T01:30:00Z,53.516566,8.188217,None,9.09|2010-10-29T01:40:00Z,53.516566,8.188217,None,9.1|2010-10-29T01:50:00Z,53.516566,8.188217,None,9.12|2010-10-29T02:00:00Z,53.516566,8.188217,None,9.15|2010-10-29T02:10:00Z,53.516566,8.188217,None,9.17|2010-10-29T02:20:00Z,53.516566,8.188217,None,9.18|2010-10-29T02:30:00Z,53.516566,8.188217,None,9.22|2010-10-29T02:40:00Z,53.516566,8.188217,None,9.24|2010-10-29T02:50:00Z,53.516566,8.188217,None,9.26|2010-10-29T03:00:00Z,53.516566,8.188217,None,9.26|2010-10-29T03:10:00Z,53.516566,8.188217,None,9.26|2010-10-29T03:20:00Z,53.516566,8.188217,None,9.23|2010-10-29T03:30:00Z,53.516566,8.188217,None,9.18|2010-10-29T03:40:00Z,53.516566,8.188217,None,9.17|2010-10-29T03:50:00Z,53.516566,8.188217,None,9.14|2010-10-29T04:00:00Z,53.516566,8.188217,None,9.09|2010-10-29T04:10:00Z,53.516566,8.188217,None,9.08|2010-10-29T04:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T04:30:00Z,53.516566,8.188217,None,9.09|2010-10-29T04:40:00Z,53.516566,8.188217,None,9.14|2010-10-29T04:50:00Z,53.516566,8.188217,None,9.15|2010-10-29T05:00:00Z,53.516566,8.188217,None,9.14|2010-10-29T05:10:00Z,53.516566,8.188217,None,9.13|2010-10-29T05:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T05:30:00Z,53.516566,8.188217,None,9.04|2010-10-29T05:40:00Z,53.516566,8.188217,None,9.04|2010-10-29T05:50:00Z,53.516566,8.188217,None,9.04|2010-10-29T06:00:00Z,53.516566,8.188217,None,9.04|2010-10-29T06:10:00Z,53.516566,8.188217,None,9.04|2010-10-29T06:20:00Z,53.516566,8.188217,None,9.03|2010-10-29T06:30:00Z,53.516566,8.188217,None,9.02|2010-10-29T06:40:00Z,53.516566,8.188217,None,9.01|2010-10-29T06:50:00Z,53.516566,8.188217,None,9.01|2010-10-29T07:00:00Z,53.516566,8.188217,None,9.01|2010-10-29T07:10:00Z,53.516566,8.188217,None,9.0|2010-10-29T07:20:00Z,53.516566,8.188217,None,8.99|2010-10-29T07:30:00Z,53.516566,8.188217,None,8.99|2010-10-29T07:40:00Z,53.516566,8.188217,None,9.0|2010-10-29T07:50:00Z,53.516566,8.188217,None,9.03|2010-10-29T08:00:00Z,53.516566,8.188217,None,9.06|2010-10-29T08:10:00Z,53.516566,8.188217,None,9.06|2010-10-29T08:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T08:30:00Z,53.516566,8.188217,None,9.07|2010-10-29T08:40:00Z,53.516566,8.188217,None,9.07|2010-10-29T08:50:00Z,53.516566,8.188217,None,9.0|2010-10-29T09:00:00Z,53.516566,8.188217,None,8.98|2010-10-29T09:10:00Z,53.516566,8.188217,None,9.02|2010-10-29T09:20:00Z,53.516566,8.188217,None,9.08|2010-10-29T09:30:00Z,53.516566,8.188217,None,9.1|2010-10-29T09:40:00Z,53.516566,8.188217,None,9.1|2010-10-29T09:50:00Z,53.516566,8.188217,None,9.11|2010-10-29T10:00:00Z,53.516566,8.188217,None,9.11|2010-10-29T10:10:00Z,53.516566,8.188217,None,9.09|2010-10-29T10:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T10:30:00Z,53.516566,8.188217,None,9.07|2010-10-29T10:40:00Z,53.516566,8.188217,None,9.12|2010-10-29T10:50:00Z,53.516566,8.188217,None,9.14|2010-10-29T11:00:00Z,53.516566,8.188217,None,9.15|2010-10-29T11:10:00Z,53.516566,8.188217,None,9.15|2010-10-29T11:20:00Z,53.516566,8.188217,None,9.15|2010-10-29T11:30:00Z,53.516566,8.188217,None,9.13|2010-10-29T11:40:00Z,53.516566,8.188217,None,9.11|2010-10-29T11:50:00Z,53.516566,8.188217,None,9.09|2010-10-29T12:00:00Z,53.516566,8.188217,None,9.1|2010-10-29T12:10:00Z,53.516566,8.188217,None,9.1|2010-10-29T12:20:00Z,53.516566,8.188217,None,9.07|2010-10-29T12:30:00Z,53.516566,8.188217,None,9.07|2010-10-29T12:40:00Z,53.516566,8.188217,None,9.08|2010-10-29T12:50:00Z,53.516566,8.188217,None,9.1|2010-10-29T13:00:00Z,53.516566,8.188217,None,9.1|2010-10-29T13:10:00Z,53.516566,8.188217,None,9.09|2010-10-29T13:20:00Z,53.516566,8.188217,None,9.1|2010-10-29T13:30:00Z,53.516566,8.188217,None,9.1|2010-10-29T13:40:00Z,53.516566,8.188217,None,9.1|2010-10-29T13:50:00Z,53.516566,8.188217,None,9.11|2010-10-29T14:00:00Z,53.516566,8.188217,None,9.12|2010-10-29T14:10:00Z,53.516566,8.188217,None,9.12|2010-10-29T14:20:00Z,53.516566,8.188217,None,9.13|2010-10-29T14:30:00Z,53.516566,8.188217,None,9.14|2010-10-29T14:40:00Z,53.516566,8.188217,None,9.15|2010-10-29T14:50:00Z,53.516566,8.188217,None,9.15|2010-10-29T15:00:00Z,53.516566,8.188217,None,9.16|2010-10-29T15:10:00Z,53.516566,8.188217,None,9.16|2010-10-29T15:20:00Z,53.516566,8.188217,None,9.15|2010-10-29T15:30:00Z,53.516566,8.188217,None,9.15|2010-10-29T15:40:00Z,53.516566,8.188217,None,9.13|2010-10-29T15:50:00Z,53.516566,8.188217,None,9.11|2010-10-29T16:00:00Z,53.516566,8.188217,None,9.12|2010-10-29T16:10:00Z,53.516566,8.188217,None,9.14|2010-10-29T16:20:00Z,53.516566,8.188217,None,9.27|2010-10-29T16:30:00Z,53.516566,8.188217,None,9.37|2010-10-29T16:40:00Z,53.516566,8.188217,None,9.26|2010-10-29T16:50:00Z,53.516566,8.188217,None,9.3|2010-10-29T17:00:00Z,53.516566,8.188217,None,9.21|2010-10-29T17:10:00Z,53.516566,8.188217,None,9.19|2010-10-29T17:20:00Z,53.516566,8.188217,None,9.16|2010-10-29T17:30:00Z,53.516566,8.188217,None,9.15|2010-10-29T17:40:00Z,53.516566,8.188217,None,9.15|2010-10-29T17:50:00Z,53.516566,8.188217,None,9.13|2010-10-29T18:00:00Z,53.516566,8.188217,None,9.1|2010-10-29T18:10:00Z,53.516566,8.188217,None,9.09|2010-10-29T18:20:00Z,53.516566,8.188217,None,9.09|2010-10-29T18:30:00Z,53.516566,8.188217,None,9.09|2010-10-29T18:40:00Z,53.516566,8.188217,None,9.09|2010-10-29T18:50:00Z,53.516566,8.188217,None,9.1|2010-10-29T19:00:00Z,53.516566,8.188217,None,9.11|2010-10-29T19:10:00Z,53.516566,8.188217,None,9.11|2010-10-29T19:20:00Z,53.516566,8.188217,None,9.12|2010-10-29T19:30:00Z,53.516566,8.188217,None,9.12|2010-10-29T19:40:00Z,53.516566,8.188217,None,9.13|2010-10-29T19:50:00Z,53.516566,8.188217,None,9.12|2010-10-29T20:00:00Z,53.516566,8.188217,None,9.13|2010-10-29T20:10:00Z,53.516566,8.188217,None,9.13|2010-10-29T20:20:00Z,53.516566,8.188217,None,9.12|2010-10-29T20:30:00Z,53.516566,8.188217,None,9.11|2010-10-29T20:40:00Z,53.516566,8.188217,None,9.11|2010-10-29T20:50:00Z,53.516566,8.188217,None,9.12|2010-10-29T21:00:00Z,53.516566,8.188217,None,9.13|2010-10-29T21:10:00Z,53.516566,8.188217,None,9.13|2010-10-29T21:20:00Z,53.516566,8.188217,None,9.14|2010-10-29T21:30:00Z,53.516566,8.188217,None,9.16|2010-10-29T21:40:00Z,53.516566,8.188217,None,9.16|2010-10-29T21:50:00Z,53.516566,8.188217,None,9.16|2010-10-29T22:00:00Z,53.516566,8.188217,None,9.17|2010-10-29T22:10:00Z,53.516566,8.188217,None,9.16|2010-10-29T22:20:00Z,53.516566,8.188217,None,9.15|2010-10-29T22:30:00Z,53.516566,8.188217,None,9.15|2010-10-29T22:40:00Z,53.516566,8.188217,None,9.13|2010-10-29T22:50:00Z,53.516566,8.188217,None,9.17|2010-10-29T23:00:00Z,53.516566,8.188217,None,9.14|2010-10-29T23:10:00Z,53.516566,8.188217,None,9.14|2010-10-29T23:20:00Z,53.516566,8.188217,None,9.14|2010-10-29T23:30:00Z,53.516566,8.188217,None,9.16|2010-10-29T23:40:00Z,53.516566,8.188217,None,9.15|2010-10-29T23:50:00Z,53.516566,8.188217,None,9.13|2010-10-30T00:00:00Z,53.516566,8.188217,None,9.12|2010-10-30T00:10:00Z,53.516566,8.188217,None,9.12|2010-10-30T00:20:00Z,53.516566,8.188217,None,9.11|2010-10-30T00:30:00Z,53.516566,8.188217,None,9.11|2010-10-30T00:40:00Z,53.516566,8.188217,None,9.07|2010-10-30T00:50:00Z,53.516566,8.188217,None,9.07|2010-10-30T01:00:00Z,53.516566,8.188217,None,9.11|2010-10-30T01:10:00Z,53.516566,8.188217,None,9.13|2010-10-30T01:20:00Z,53.516566,8.188217,None,9.14|2010-10-30T01:30:00Z,53.516566,8.188217,None,9.15|2010-10-30T01:40:00Z,53.516566,8.188217,None,9.17|2010-10-30T01:50:00Z,53.516566,8.188217,None,9.19|2010-10-30T02:00:00Z,53.516566,8.188217,None,9.19|2010-10-30T02:10:00Z,53.516566,8.188217,None,9.19|2010-10-30T02:20:00Z,53.516566,8.188217,None,9.18|2010-10-30T02:30:00Z,53.516566,8.188217,None,9.18|2010-10-30T02:40:00Z,53.516566,8.188217,None,9.19|2010-10-30T02:50:00Z,53.516566,8.188217,None,9.2|2010-10-30T03:00:00Z,53.516566,8.188217,None,9.21|2010-10-30T03:10:00Z,53.516566,8.188217,None,9.22|2010-10-30T03:20:00Z,53.516566,8.188217,None,9.22|2010-10-30T03:30:00Z,53.516566,8.188217,None,9.24|2010-10-30T03:40:00Z,53.516566,8.188217,None,9.25|2010-10-30T03:50:00Z,53.516566,8.188217,None,9.26|2010-10-30T04:00:00Z,53.516566,8.188217,None,9.26|2010-10-30T04:10:00Z,53.516566,8.188217,None,9.26|2010-10-30T04:20:00Z,53.516566,8.188217,None,9.24|2010-10-30T04:30:00Z,53.516566,8.188217,None,9.21|2010-10-30T04:40:00Z,53.516566,8.188217,None,9.18|2010-10-30T04:50:00Z,53.516566,8.188217,None,9.16|2010-10-30T05:00:00Z,53.516566,8.188217,None,9.14|2010-10-30T05:10:00Z,53.516566,8.188217,None,9.08|2010-10-30T05:20:00Z,53.516566,8.188217,None,9.03|2010-10-30T05:30:00Z,53.516566,8.188217,None,9.01|2010-10-30T05:40:00Z,53.516566,8.188217,None,9.0|2010-10-30T05:50:00Z,53.516566,8.188217,None,9.0|2010-10-30T06:00:00Z,53.516566,8.188217,None,8.97|2010-10-30T06:10:00Z,53.516566,8.188217,None,8.93|2010-10-30T06:20:00Z,53.516566,8.188217,None,8.92|2010-10-30T06:30:00Z,53.516566,8.188217,None,8.93|2010-10-30T06:40:00Z,53.516566,8.188217,None,9.0|2010-10-30T06:50:00Z,53.516566,8.188217,None,9.04|2010-10-30T07:00:00Z,53.516566,8.188217,None,9.06|2010-10-30T07:10:00Z,53.516566,8.188217,None,9.06|2010-10-30T07:20:00Z,53.516566,8.188217,None,9.06|2010-10-30T07:30:00Z,53.516566,8.188217,None,9.07|2010-10-30T07:40:00Z,53.516566,8.188217,None,9.08|2010-10-30T07:50:00Z,53.516566,8.188217,None,9.08|2010-10-30T08:00:00Z,53.516566,8.188217,None,9.09|2010-10-30T08:10:00Z,53.516566,8.188217,None,9.08|2010-10-30T08:20:00Z,53.516566,8.188217,None,9.08|2010-10-30T08:30:00Z,53.516566,8.188217,None,9.08|2010-10-30T08:40:00Z,53.516566,8.188217,None,9.08|2010-10-30T08:50:00Z,53.516566,8.188217,None,9.09|2010-10-30T09:00:00Z,53.516566,8.188217,None,9.11|2010-10-30T09:10:00Z,53.516566,8.188217,None,9.11|2010-10-30T09:20:00Z,53.516566,8.188217,None,9.13|2010-10-30T09:30:00Z,53.516566,8.188217,None,9.15|2010-10-30T09:40:00Z,53.516566,8.188217,None,9.16|2010-10-30T09:50:00Z,53.516566,8.188217,None,9.16|2010-10-30T10:00:00Z,53.516566,8.188217,None,9.18|2010-10-30T10:10:00Z,53.516566,8.188217,None,8.92|2010-10-30T10:20:00Z,53.516566,8.188217,None,9.14|2010-10-30T10:30:00Z,53.516566,8.188217,None,9.16|2010-10-30T10:40:00Z,53.516566,8.188217,None,9.16|2010-10-30T10:50:00Z,53.516566,8.188217,None,9.16|2010-10-30T11:00:00Z,53.516566,8.188217,None,9.17|2010-10-30T11:10:00Z,53.516566,8.188217,None,9.1|2010-10-30T11:20:00Z,53.516566,8.188217,None,9.08|2010-10-30T11:30:00Z,53.516566,8.188217,None,9.11|2010-10-30T11:40:00Z,53.516566,8.188217,None,9.13|2010-10-30T11:50:00Z,53.516566,8.188217,None,9.15|2010-10-30T12:00:00Z,53.516566,8.188217,None,9.13|2010-10-30T12:10:00Z,53.516566,8.188217,None,9.13|2010-10-30T12:20:00Z,53.516566,8.188217,None,9.11|2010-10-30T12:30:00Z,53.516566,8.188217,None,9.12|2010-10-30T12:40:00Z,53.516566,8.188217,None,9.15|2010-10-30T12:50:00Z,53.516566,8.188217,None,9.12|2010-10-30T13:00:00Z,53.516566,8.188217,None,9.11|2010-10-30T13:10:00Z,53.516566,8.188217,None,9.1|2010-10-30T13:20:00Z,53.516566,8.188217,None,9.09|2010-10-30T13:30:00Z,53.516566,8.188217,None,9.09|2010-10-30T13:40:00Z,53.516566,8.188217,None,9.1|2010-10-30T13:50:00Z,53.516566,8.188217,None,9.11|2010-10-30T14:00:00Z,53.516566,8.188217,None,9.12|2010-10-30T14:10:00Z,53.516566,8.188217,None,9.15|2010-10-30T14:20:00Z,53.516566,8.188217,None,9.16|2010-10-30T14:30:00Z,53.516566,8.188217,None,9.16|2010-10-30T14:40:00Z,53.516566,8.188217,None,9.16|2010-10-30T14:50:00Z,53.516566,8.188217,None,9.16|2010-10-30T15:00:00Z,53.516566,8.188217,None,9.17|2010-10-30T15:10:00Z,53.516566,8.188217,None,9.18|2010-10-30T15:20:00Z,53.516566,8.188217,None,9.18|2010-10-30T15:30:00Z,53.516566,8.188217,None,9.18|2010-10-30T15:40:00Z,53.516566,8.188217,None,9.19|2010-10-30T15:50:00Z,53.516566,8.188217,None,9.2|2010-10-30T16:00:00Z,53.516566,8.188217,None,9.21|2010-10-30T16:10:00Z,53.516566,8.188217,None,9.22|2010-10-30T16:20:00Z,53.516566,8.188217,None,9.23|2010-10-30T16:30:00Z,53.516566,8.188217,None,9.22|2010-10-30T16:40:00Z,53.516566,8.188217,None,9.21|2010-10-30T16:50:00Z,53.516566,8.188217,None,9.19|2010-10-30T17:00:00Z,53.516566,8.188217,None,9.19|2010-10-30T17:10:00Z,53.516566,8.188217,None,9.18|2010-10-30T17:20:00Z,53.516566,8.188217,None,9.16|2010-10-30T17:30:00Z,53.516566,8.188217,None,9.14|2010-10-30T17:40:00Z,53.516566,8.188217,None,9.16|2010-10-30T17:50:00Z,53.516566,8.188217,None,9.19|2010-10-30T18:00:00Z,53.516566,8.188217,None,9.3|2010-10-30T18:10:00Z,53.516566,8.188217,None,9.39|2010-10-30T18:20:00Z,53.516566,8.188217,None,9.41|2010-10-30T18:30:00Z,53.516566,8.188217,None,9.42|2010-10-30T18:40:00Z,53.516566,8.188217,None,9.4|2010-10-30T18:50:00Z,53.516566,8.188217,None,9.31|2010-10-30T19:00:00Z,53.516566,8.188217,None,9.24|2010-10-30T19:10:00Z,53.516566,8.188217,None,9.18|2010-10-30T19:20:00Z,53.516566,8.188217,None,9.15|2010-10-30T19:30:00Z,53.516566,8.188217,None,9.15|2010-10-30T19:40:00Z,53.516566,8.188217,None,9.15|2010-10-30T19:50:00Z,53.516566,8.188217,None,9.15|2010-10-30T20:00:00Z,53.516566,8.188217,None,9.15|2010-10-30T20:10:00Z,53.516566,8.188217,None,9.14|2010-10-30T20:20:00Z,53.516566,8.188217,None,9.14|2010-10-30T20:30:00Z,53.516566,8.188217,None,9.14|2010-10-30T20:40:00Z,53.516566,8.188217,None,9.14|2010-10-30T20:50:00Z,53.516566,8.188217,None,9.15|2010-10-30T21:00:00Z,53.516566,8.188217,None,9.15|2010-10-30T21:10:00Z,53.516566,8.188217,None,9.15|2010-10-30T21:20:00Z,53.516566,8.188217,None,9.14|2010-10-30T21:30:00Z,53.516566,8.188217,None,9.15|2010-10-30T21:40:00Z,53.516566,8.188217,None,9.16|2010-10-30T21:50:00Z,53.516566,8.188217,None,9.18|2010-10-30T22:00:00Z,53.516566,8.188217,None,9.19|2010-10-30T22:10:00Z,53.516566,8.188217,None,9.17|2010-10-30T22:20:00Z,53.516566,8.188217,None,9.19|2010-10-30T22:30:00Z,53.516566,8.188217,None,9.23|2010-10-30T22:40:00Z,53.516566,8.188217,None,9.22|2010-10-30T22:50:00Z,53.516566,8.188217,None,9.22|2010-10-30T23:00:00Z,53.516566,8.188217,None,9.23|2010-10-30T23:10:00Z,53.516566,8.188217,None,9.24|2010-10-30T23:20:00Z,53.516566,8.188217,None,9.22|2010-10-30T23:30:00Z,53.516566,8.188217,None,9.23|2010-10-30T23:40:00Z,53.516566,8.188217,None,9.23|2010-10-30T23:50:00Z,53.516566,8.188217,None,9.23|2010-10-31T00:00:00Z,53.516566,8.188217,None,9.24|2010-10-31T00:10:00Z,53.516566,8.188217,None,9.25|2010-10-31T00:20:00Z,53.516566,8.188217,None,9.24|2010-10-31T00:30:00Z,53.516566,8.188217,None,9.24|2010-10-31T00:40:00Z,53.516566,8.188217,None,9.23|2010-10-31T00:50:00Z,53.516566,8.188217,None,9.21|2010-10-31T01:00:00Z,53.516566,8.188217,None,9.19|2010-10-31T01:10:00Z,53.516566,8.188217,None,9.21|2010-10-31T01:20:00Z,53.516566,8.188217,None,9.2|2010-10-31T01:30:00Z,53.516566,8.188217,None,9.15|2010-10-31T01:40:00Z,53.516566,8.188217,None,9.13|2010-10-31T01:50:00Z,53.516566,8.188217,None,9.15|2010-10-31T02:00:00Z,53.516566,8.188217,None,9.18|2010-10-31T02:10:00Z,53.516566,8.188217,None,9.19|2010-10-31T02:20:00Z,53.516566,8.188217,None,9.19|2010-10-31T02:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T02:40:00Z,53.516566,8.188217,None,9.24|2010-10-31T02:50:00Z,53.516566,8.188217,None,9.26|2010-10-31T03:00:00Z,53.516566,8.188217,None,9.27|2010-10-31T03:10:00Z,53.516566,8.188217,None,9.27|2010-10-31T03:20:00Z,53.516566,8.188217,None,9.27|2010-10-31T03:30:00Z,53.516566,8.188217,None,9.27|2010-10-31T03:40:00Z,53.516566,8.188217,None,9.27|2010-10-31T03:50:00Z,53.516566,8.188217,None,9.27|2010-10-31T04:00:00Z,53.516566,8.188217,None,9.27|2010-10-31T04:10:00Z,53.516566,8.188217,None,9.28|2010-10-31T04:20:00Z,53.516566,8.188217,None,9.29|2010-10-31T04:30:00Z,53.516566,8.188217,None,9.29|2010-10-31T04:40:00Z,53.516566,8.188217,None,9.3|2010-10-31T04:50:00Z,53.516566,8.188217,None,9.29|2010-10-31T05:00:00Z,53.516566,8.188217,None,9.28|2010-10-31T05:10:00Z,53.516566,8.188217,None,9.27|2010-10-31T05:20:00Z,53.516566,8.188217,None,9.25|2010-10-31T05:30:00Z,53.516566,8.188217,None,9.25|2010-10-31T05:40:00Z,53.516566,8.188217,None,9.21|2010-10-31T05:50:00Z,53.516566,8.188217,None,9.16|2010-10-31T06:00:00Z,53.516566,8.188217,None,9.17|2010-10-31T06:10:00Z,53.516566,8.188217,None,9.21|2010-10-31T06:20:00Z,53.516566,8.188217,None,9.21|2010-10-31T06:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T06:40:00Z,53.516566,8.188217,None,9.22|2010-10-31T06:50:00Z,53.516566,8.188217,None,9.21|2010-10-31T07:00:00Z,53.516566,8.188217,None,9.21|2010-10-31T07:10:00Z,53.516566,8.188217,None,9.2|2010-10-31T07:20:00Z,53.516566,8.188217,None,9.18|2010-10-31T07:30:00Z,53.516566,8.188217,None,9.16|2010-10-31T07:40:00Z,53.516566,8.188217,None,9.18|2010-10-31T07:50:00Z,53.516566,8.188217,None,9.19|2010-10-31T08:00:00Z,53.516566,8.188217,None,9.18|2010-10-31T08:10:00Z,53.516566,8.188217,None,9.18|2010-10-31T08:20:00Z,53.516566,8.188217,None,9.18|2010-10-31T08:30:00Z,53.516566,8.188217,None,9.18|2010-10-31T08:40:00Z,53.516566,8.188217,None,9.18|2010-10-31T08:50:00Z,53.516566,8.188217,None,9.18|2010-10-31T09:00:00Z,53.516566,8.188217,None,9.19|2010-10-31T09:10:00Z,53.516566,8.188217,None,9.19|2010-10-31T09:20:00Z,53.516566,8.188217,None,9.18|2010-10-31T09:30:00Z,53.516566,8.188217,None,9.18|2010-10-31T09:40:00Z,53.516566,8.188217,None,9.19|2010-10-31T09:50:00Z,53.516566,8.188217,None,9.2|2010-10-31T10:00:00Z,53.516566,8.188217,None,9.03|2010-10-31T10:10:00Z,53.516566,8.188217,None,8.98|2010-10-31T10:20:00Z,53.516566,8.188217,None,9.09|2010-10-31T10:30:00Z,53.516566,8.188217,None,9.1|2010-10-31T10:40:00Z,53.516566,8.188217,None,9.3|2010-10-31T10:50:00Z,53.516566,8.188217,None,9.31|2010-10-31T11:00:00Z,53.516566,8.188217,None,9.25|2010-10-31T11:10:00Z,53.516566,8.188217,None,9.46|2010-10-31T11:20:00Z,53.516566,8.188217,None,9.38|2010-10-31T11:30:00Z,53.516566,8.188217,None,9.4|2010-10-31T11:40:00Z,53.516566,8.188217,None,9.45|2010-10-31T11:50:00Z,53.516566,8.188217,None,9.25|2010-10-31T12:00:00Z,53.516566,8.188217,None,9.26|2010-10-31T12:10:00Z,53.516566,8.188217,None,9.23|2010-10-31T12:20:00Z,53.516566,8.188217,None,9.24|2010-10-31T12:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T12:40:00Z,53.516566,8.188217,None,9.22|2010-10-31T12:50:00Z,53.516566,8.188217,None,9.22|2010-10-31T13:00:00Z,53.516566,8.188217,None,9.22|2010-10-31T13:10:00Z,53.516566,8.188217,None,9.21|2010-10-31T13:20:00Z,53.516566,8.188217,None,9.19|2010-10-31T13:30:00Z,53.516566,8.188217,None,9.18|2010-10-31T13:40:00Z,53.516566,8.188217,None,9.18|2010-10-31T13:50:00Z,53.516566,8.188217,None,9.19|2010-10-31T14:00:00Z,53.516566,8.188217,None,9.19|2010-10-31T14:10:00Z,53.516566,8.188217,None,9.2|2010-10-31T14:20:00Z,53.516566,8.188217,None,9.2|2010-10-31T14:30:00Z,53.516566,8.188217,None,9.21|2010-10-31T14:40:00Z,53.516566,8.188217,None,9.22|2010-10-31T14:50:00Z,53.516566,8.188217,None,9.21|2010-10-31T15:00:00Z,53.516566,8.188217,None,9.2|2010-10-31T15:10:00Z,53.516566,8.188217,None,9.2|2010-10-31T15:20:00Z,53.516566,8.188217,None,9.21|2010-10-31T15:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T15:40:00Z,53.516566,8.188217,None,9.23|2010-10-31T15:50:00Z,53.516566,8.188217,None,9.25|2010-10-31T16:00:00Z,53.516566,8.188217,None,9.26|2010-10-31T16:10:00Z,53.516566,8.188217,None,9.28|2010-10-31T16:20:00Z,53.516566,8.188217,None,9.3|2010-10-31T16:30:00Z,53.516566,8.188217,None,9.31|2010-10-31T16:40:00Z,53.516566,8.188217,None,9.32|2010-10-31T16:50:00Z,53.516566,8.188217,None,9.32|2010-10-31T17:00:00Z,53.516566,8.188217,None,9.32|2010-10-31T17:10:00Z,53.516566,8.188217,None,9.33|2010-10-31T17:20:00Z,53.516566,8.188217,None,9.33|2010-10-31T17:30:00Z,53.516566,8.188217,None,9.33|2010-10-31T17:40:00Z,53.516566,8.188217,None,9.32|2010-10-31T17:50:00Z,53.516566,8.188217,None,9.32|2010-10-31T18:00:00Z,53.516566,8.188217,None,9.32|2010-10-31T18:10:00Z,53.516566,8.188217,None,9.32|2010-10-31T18:20:00Z,53.516566,8.188217,None,9.31|2010-10-31T18:30:00Z,53.516566,8.188217,None,9.29|2010-10-31T18:40:00Z,53.516566,8.188217,None,9.26|2010-10-31T18:50:00Z,53.516566,8.188217,None,9.22|2010-10-31T19:00:00Z,53.516566,8.188217,None,9.21|2010-10-31T19:10:00Z,53.516566,8.188217,None,9.21|2010-10-31T19:20:00Z,53.516566,8.188217,None,9.26|2010-10-31T19:30:00Z,53.516566,8.188217,None,9.4|2010-10-31T19:40:00Z,53.516566,8.188217,None,9.43|2010-10-31T19:50:00Z,53.516566,8.188217,None,9.38|2010-10-31T20:00:00Z,53.516566,8.188217,None,9.32|2010-10-31T20:10:00Z,53.516566,8.188217,None,9.27|2010-10-31T20:20:00Z,53.516566,8.188217,None,9.26|2010-10-31T20:30:00Z,53.516566,8.188217,None,9.25|2010-10-31T20:40:00Z,53.516566,8.188217,None,9.24|2010-10-31T20:50:00Z,53.516566,8.188217,None,9.23|2010-10-31T21:00:00Z,53.516566,8.188217,None,9.22|2010-10-31T21:10:00Z,53.516566,8.188217,None,9.22|2010-10-31T21:20:00Z,53.516566,8.188217,None,9.22|2010-10-31T21:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T21:40:00Z,53.516566,8.188217,None,9.21|2010-10-31T21:50:00Z,53.516566,8.188217,None,9.21|2010-10-31T22:00:00Z,53.516566,8.188217,None,9.21|2010-10-31T22:10:00Z,53.516566,8.188217,None,9.21|2010-10-31T22:20:00Z,53.516566,8.188217,None,9.21|2010-10-31T22:30:00Z,53.516566,8.188217,None,9.22|2010-10-31T22:40:00Z,53.516566,8.188217,None,9.23|2010-10-31T22:50:00Z,53.516566,8.188217,None,9.23|2010-10-31T23:00:00Z,53.516566,8.188217,None,9.23|2010-10-31T23:10:00Z,53.516566,8.188217,None,9.23|2010-10-31T23:20:00Z,53.516566,8.188217,None,9.24|2010-10-31T23:30:00Z,53.516566,8.188217,None,9.26|2010-10-31T23:40:00Z,53.516566,8.188217,None,9.28|2010-10-31T23:50:00Z,53.516566,8.188217,None,9.27|2010-11-01T00:00:00Z,53.516566,8.188217,None,9.26|2010-11-01T00:10:00Z,53.516566,8.188217,None,9.27|2010-11-01T00:20:00Z,53.516566,8.188217,None,9.27|2010-11-01T00:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T00:40:00Z,53.516566,8.188217,None,9.24|2010-11-01T00:50:00Z,53.516566,8.188217,None,9.25|2010-11-01T01:00:00Z,53.516566,8.188217,None,9.25|2010-11-01T01:10:00Z,53.516566,8.188217,None,9.22|2010-11-01T01:20:00Z,53.516566,8.188217,None,9.21|2010-11-01T01:30:00Z,53.516566,8.188217,None,9.21|2010-11-01T01:40:00Z,53.516566,8.188217,None,9.23|2010-11-01T01:50:00Z,53.516566,8.188217,None,9.24|2010-11-01T02:00:00Z,53.516566,8.188217,None,9.23|2010-11-01T02:10:00Z,53.516566,8.188217,None,9.22|2010-11-01T02:20:00Z,53.516566,8.188217,None,9.23|2010-11-01T02:30:00Z,53.516566,8.188217,None,9.24|2010-11-01T02:40:00Z,53.516566,8.188217,None,9.22|2010-11-01T02:50:00Z,53.516566,8.188217,None,9.23|2010-11-01T03:00:00Z,53.516566,8.188217,None,9.22|2010-11-01T03:10:00Z,53.516566,8.188217,None,9.22|2010-11-01T03:20:00Z,53.516566,8.188217,None,9.22|2010-11-01T03:30:00Z,53.516566,8.188217,None,9.22|2010-11-01T03:40:00Z,53.516566,8.188217,None,9.23|2010-11-01T03:50:00Z,53.516566,8.188217,None,9.25|2010-11-01T04:00:00Z,53.516566,8.188217,None,9.27|2010-11-01T04:10:00Z,53.516566,8.188217,None,9.29|2010-11-01T04:20:00Z,53.516566,8.188217,None,9.32|2010-11-01T04:30:00Z,53.516566,8.188217,None,9.34|2010-11-01T04:40:00Z,53.516566,8.188217,None,9.35|2010-11-01T04:50:00Z,53.516566,8.188217,None,9.34|2010-11-01T05:00:00Z,53.516566,8.188217,None,9.32|2010-11-01T05:10:00Z,53.516566,8.188217,None,9.32|2010-11-01T05:20:00Z,53.516566,8.188217,None,9.32|2010-11-01T05:30:00Z,53.516566,8.188217,None,9.33|2010-11-01T05:40:00Z,53.516566,8.188217,None,9.34|2010-11-01T05:50:00Z,53.516566,8.188217,None,9.36|2010-11-01T06:00:00Z,53.516566,8.188217,None,9.37|2010-11-01T06:10:00Z,53.516566,8.188217,None,9.37|2010-11-01T06:20:00Z,53.516566,8.188217,None,9.37|2010-11-01T06:30:00Z,53.516566,8.188217,None,9.37|2010-11-01T06:40:00Z,53.516566,8.188217,None,9.34|2010-11-01T06:50:00Z,53.516566,8.188217,None,9.32|2010-11-01T07:00:00Z,53.516566,8.188217,None,9.32|2010-11-01T07:10:00Z,53.516566,8.188217,None,9.3|2010-11-01T07:20:00Z,53.516566,8.188217,None,9.24|2010-11-01T07:30:00Z,53.516566,8.188217,None,9.22|2010-11-01T07:40:00Z,53.516566,8.188217,None,9.22|2010-11-01T07:50:00Z,53.516566,8.188217,None,9.22|2010-11-01T08:00:00Z,53.516566,8.188217,None,9.21|2010-11-01T08:10:00Z,53.516566,8.188217,None,9.21|2010-11-01T08:20:00Z,53.516566,8.188217,None,9.22|2010-11-01T08:30:00Z,53.516566,8.188217,None,9.22|2010-11-01T08:40:00Z,53.516566,8.188217,None,9.22|2010-11-01T08:50:00Z,53.516566,8.188217,None,9.23|2010-11-01T09:00:00Z,53.516566,8.188217,None,9.24|2010-11-01T09:10:00Z,53.516566,8.188217,None,9.24|2010-11-01T09:20:00Z,53.516566,8.188217,None,9.24|2010-11-01T09:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T09:40:00Z,53.516566,8.188217,None,9.25|2010-11-01T09:50:00Z,53.516566,8.188217,None,9.25|2010-11-01T10:00:00Z,53.516566,8.188217,None,9.25|2010-11-01T10:10:00Z,53.516566,8.188217,None,9.26|2010-11-01T10:20:00Z,53.516566,8.188217,None,9.26|2010-11-01T10:30:00Z,53.516566,8.188217,None,9.26|2010-11-01T10:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T10:50:00Z,53.516566,8.188217,None,9.26|2010-11-01T11:00:00Z,53.516566,8.188217,None,9.26|2010-11-01T11:10:00Z,53.516566,8.188217,None,9.25|2010-11-01T11:20:00Z,53.516566,8.188217,None,9.27|2010-11-01T11:30:00Z,53.516566,8.188217,None,9.28|2010-11-01T11:40:00Z,53.516566,8.188217,None,9.28|2010-11-01T11:50:00Z,53.516566,8.188217,None,9.28|2010-11-01T12:00:00Z,53.516566,8.188217,None,9.28|2010-11-01T12:10:00Z,53.516566,8.188217,None,9.28|2010-11-01T12:20:00Z,53.516566,8.188217,None,9.29|2010-11-01T12:30:00Z,53.516566,8.188217,None,9.28|2010-11-01T12:40:00Z,53.516566,8.188217,None,9.29|2010-11-01T12:50:00Z,53.516566,8.188217,None,9.29|2010-11-01T13:00:00Z,53.516566,8.188217,None,9.29|2010-11-01T13:10:00Z,53.516566,8.188217,None,9.28|2010-11-01T13:20:00Z,53.516566,8.188217,None,9.26|2010-11-01T13:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T13:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T13:50:00Z,53.516566,8.188217,None,9.27|2010-11-01T14:00:00Z,53.516566,8.188217,None,9.26|2010-11-01T14:10:00Z,53.516566,8.188217,None,9.26|2010-11-01T14:20:00Z,53.516566,8.188217,None,9.27|2010-11-01T14:30:00Z,53.516566,8.188217,None,9.27|2010-11-01T14:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T14:50:00Z,53.516566,8.188217,None,9.26|2010-11-01T15:00:00Z,53.516566,8.188217,None,9.25|2010-11-01T15:10:00Z,53.516566,8.188217,None,9.25|2010-11-01T15:20:00Z,53.516566,8.188217,None,9.25|2010-11-01T15:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T15:40:00Z,53.516566,8.188217,None,9.25|2010-11-01T15:50:00Z,53.516566,8.188217,None,9.26|2010-11-01T16:00:00Z,53.516566,8.188217,None,9.26|2010-11-01T16:10:00Z,53.516566,8.188217,None,9.26|2010-11-01T16:20:00Z,53.516566,8.188217,None,9.27|2010-11-01T16:30:00Z,53.516566,8.188217,None,9.27|2010-11-01T16:40:00Z,53.516566,8.188217,None,9.27|2010-11-01T16:50:00Z,53.516566,8.188217,None,9.27|2010-11-01T17:00:00Z,53.516566,8.188217,None,9.28|2010-11-01T17:10:00Z,53.516566,8.188217,None,9.28|2010-11-01T17:20:00Z,53.516566,8.188217,None,9.28|2010-11-01T17:30:00Z,53.516566,8.188217,None,9.29|2010-11-01T17:40:00Z,53.516566,8.188217,None,9.32|2010-11-01T17:50:00Z,53.516566,8.188217,None,9.34|2010-11-01T18:00:00Z,53.516566,8.188217,None,9.37|2010-11-01T18:10:00Z,53.516566,8.188217,None,9.39|2010-11-01T18:20:00Z,53.516566,8.188217,None,9.39|2010-11-01T18:30:00Z,53.516566,8.188217,None,9.39|2010-11-01T18:40:00Z,53.516566,8.188217,None,9.38|2010-11-01T18:50:00Z,53.516566,8.188217,None,9.37|2010-11-01T19:00:00Z,53.516566,8.188217,None,9.37|2010-11-01T19:10:00Z,53.516566,8.188217,None,9.37|2010-11-01T19:20:00Z,53.516566,8.188217,None,9.36|2010-11-01T19:30:00Z,53.516566,8.188217,None,9.36|2010-11-01T19:40:00Z,53.516566,8.188217,None,9.37|2010-11-01T19:50:00Z,53.516566,8.188217,None,9.35|2010-11-01T20:00:00Z,53.516566,8.188217,None,9.29|2010-11-01T20:10:00Z,53.516566,8.188217,None,9.27|2010-11-01T20:20:00Z,53.516566,8.188217,None,9.25|2010-11-01T20:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T20:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T20:50:00Z,53.516566,8.188217,None,9.29|2010-11-01T21:00:00Z,53.516566,8.188217,None,9.29|2010-11-01T21:10:00Z,53.516566,8.188217,None,9.29|2010-11-01T21:20:00Z,53.516566,8.188217,None,9.27|2010-11-01T21:30:00Z,53.516566,8.188217,None,9.26|2010-11-01T21:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T21:50:00Z,53.516566,8.188217,None,9.25|2010-11-01T22:00:00Z,53.516566,8.188217,None,9.25|2010-11-01T22:10:00Z,53.516566,8.188217,None,9.25|2010-11-01T22:20:00Z,53.516566,8.188217,None,9.26|2010-11-01T22:30:00Z,53.516566,8.188217,None,9.26|2010-11-01T22:40:00Z,53.516566,8.188217,None,9.25|2010-11-01T22:50:00Z,53.516566,8.188217,None,9.26|2010-11-01T23:00:00Z,53.516566,8.188217,None,9.26|2010-11-01T23:10:00Z,53.516566,8.188217,None,9.25|2010-11-01T23:20:00Z,53.516566,8.188217,None,9.25|2010-11-01T23:30:00Z,53.516566,8.188217,None,9.25|2010-11-01T23:40:00Z,53.516566,8.188217,None,9.26|2010-11-01T23:50:00Z,53.516566,8.188217,None,9.26|2010-11-02T00:00:00Z,53.516566,8.188217,None,9.26|2010-11-02T00:10:00Z,53.516566,8.188217,None,9.27|2010-11-02T00:20:00Z,53.516566,8.188217,None,9.26|2010-11-02T00:30:00Z,53.516566,8.188217,None,9.27|2010-11-02T00:40:00Z,53.516566,8.188217,None,9.27|2010-11-02T00:50:00Z,53.516566,8.188217,None,9.26|2010-11-02T01:00:00Z,53.516566,8.188217,None,9.26|2010-11-02T01:10:00Z,53.516566,8.188217,None,9.28|2010-11-02T01:20:00Z,53.516566,8.188217,None,9.29|2010-11-02T01:30:00Z,53.516566,8.188217,None,9.29|2010-11-02T01:40:00Z,53.516566,8.188217,None,9.29|2010-11-02T01:50:00Z,53.516566,8.188217,None,9.28|2010-11-02T02:00:00Z,53.516566,8.188217,None,9.27|2010-11-02T02:10:00Z,53.516566,8.188217,None,9.25|2010-11-02T02:20:00Z,53.516566,8.188217,None,9.26|2010-11-02T02:30:00Z,53.516566,8.188217,None,9.27|2010-11-02T02:40:00Z,53.516566,8.188217,None,9.28|2010-11-02T02:50:00Z,53.516566,8.188217,None,9.28|2010-11-02T03:00:00Z,53.516566,8.188217,None,9.27|2010-11-02T03:10:00Z,53.516566,8.188217,None,9.27|2010-11-02T03:20:00Z,53.516566,8.188217,None,9.28|2010-11-02T03:30:00Z,53.516566,8.188217,None,9.29|2010-11-02T03:40:00Z,53.516566,8.188217,None,9.29|2010-11-02T03:50:00Z,53.516566,8.188217,None,9.29|2010-11-02T04:00:00Z,53.516566,8.188217,None,9.26|2010-11-02T04:10:00Z,53.516566,8.188217,None,9.28|2010-11-02T04:20:00Z,53.516566,8.188217,None,9.3|2010-11-02T04:30:00Z,53.516566,8.188217,None,9.29|2010-11-02T04:40:00Z,53.516566,8.188217,None,9.29|2010-11-02T04:50:00Z,53.516566,8.188217,None,9.29|2010-11-02T05:00:00Z,53.516566,8.188217,None,9.29|2010-11-02T05:10:00Z,53.516566,8.188217,None,9.29|2010-11-02T05:20:00Z,53.516566,8.188217,None,9.29|2010-11-02T05:30:00Z,53.516566,8.188217,None,9.29|2010-11-02T05:40:00Z,53.516566,8.188217,None,9.31|2010-11-02T05:50:00Z,53.516566,8.188217,None,9.32|2010-11-02T06:00:00Z,53.516566,8.188217,None,9.32|2010-11-02T06:10:00Z,53.516566,8.188217,None,9.33|2010-11-02T06:20:00Z,53.516566,8.188217,None,9.34|2010-11-02T06:30:00Z,53.516566,8.188217,None,9.35|2010-11-02T06:40:00Z,53.516566,8.188217,None,9.35|2010-11-02T06:50:00Z,53.516566,8.188217,None,9.35|2010-11-02T07:00:00Z,53.516566,8.188217,None,9.35|2010-11-02T07:10:00Z,53.516566,8.188217,None,9.35|2010-11-02T07:20:00Z,53.516566,8.188217,None,9.34|2010-11-02T07:30:00Z,53.516566,8.188217,None,9.33|2010-11-02T07:40:00Z,53.516566,8.188217,None,9.34|2010-11-02T07:50:00Z,53.516566,8.188217,None,9.33|2010-11-02T08:00:00Z,53.516566,8.188217,None,9.31|2010-11-02T08:10:00Z,53.516566,8.188217,None,9.28|2010-11-02T08:20:00Z,53.516566,8.188217,None,9.25|2010-11-02T08:30:00Z,53.516566,8.188217,None,9.23|2010-11-02T08:40:00Z,53.516566,8.188217,None,9.22|2010-11-02T08:50:00Z,53.516566,8.188217,None,9.22|2010-11-02T09:00:00Z,53.516566,8.188217,None,9.19|2010-11-02T09:10:00Z,53.516566,8.188217,None,9.16|2010-11-02T09:20:00Z,53.516566,8.188217,None,9.17|2010-11-02T09:30:00Z,53.516566,8.188217,None,9.19|2010-11-02T09:40:00Z,53.516566,8.188217,None,9.22|2010-11-02T09:50:00Z,53.516566,8.188217,None,9.25|2010-11-02T10:00:00Z,53.516566,8.188217,None,9.27|2010-11-02T10:10:00Z,53.516566,8.188217,None,9.29|2010-11-02T10:20:00Z,53.516566,8.188217,None,9.29|2010-11-02T10:30:00Z,53.516566,8.188217,None,9.3|2010-11-02T10:40:00Z,53.516566,8.188217,None,9.3|2010-11-02T10:50:00Z,53.516566,8.188217,None,9.31|2010-11-02T11:00:00Z,53.516566,8.188217,None,9.31|2010-11-02T11:10:00Z,53.516566,8.188217,None,9.3|2010-11-02T11:20:00Z,53.516566,8.188217,None,9.31|2010-11-02T11:30:00Z,53.516566,8.188217,None,9.31|2010-11-02T11:40:00Z,53.516566,8.188217,None,9.3|2010-11-02T11:50:00Z,53.516566,8.188217,None,9.3|2010-11-02T12:00:00Z,53.516566,8.188217,None,9.3|2010-11-02T12:10:00Z,53.516566,8.188217,None,9.31|2010-11-02T12:20:00Z,53.516566,8.188217,None,9.31|2010-11-02T12:30:00Z,53.516566,8.188217,None,9.31|2010-11-02T12:40:00Z,53.516566,8.188217,None,9.31|2010-11-02T12:50:00Z,53.516566,8.188217,None,9.32|2010-11-02T13:00:00Z,53.516566,8.188217,None,9.32|2010-11-02T13:10:00Z,53.516566,8.188217,None,9.32|2010-11-02T13:20:00Z,53.516566,8.188217,None,9.33|2010-11-02T13:30:00Z,53.516566,8.188217,None,9.33|2010-11-02T13:40:00Z,53.516566,8.188217,None,9.33|2010-11-02T13:50:00Z,53.516566,8.188217,None,9.33|2010-11-02T14:00:00Z,53.516566,8.188217,None,9.33|2010-11-02T14:10:00Z,53.516566,8.188217,None,9.33|2010-11-02T14:20:00Z,53.516566,8.188217,None,9.32|2010-11-02T14:30:00Z,53.516566,8.188217,None,9.31|2010-11-02T14:40:00Z,53.516566,8.188217,None,9.31|2010-11-02T14:50:00Z,53.516566,8.188217,None,9.32|2010-11-02T15:00:00Z,53.516566,8.188217,None,9.32|2010-11-02T15:10:00Z,53.516566,8.188217,None,9.31|2010-11-02T15:20:00Z,53.516566,8.188217,None,9.29|2010-11-02T15:30:00Z,53.516566,8.188217,None,9.28|2010-11-02T15:40:00Z,53.516566,8.188217,None,9.3|2010-11-02T15:50:00Z,53.516566,8.188217,None,9.3|2010-11-02T16:00:00Z,53.516566,8.188217,None,9.3|2010-11-02T16:10:00Z,53.516566,8.188217,None,9.3|2010-11-02T16:20:00Z,53.516566,8.188217,None,9.28|2010-11-02T16:30:00Z,53.516566,8.188217,None,9.27|2010-11-02T16:40:00Z,53.516566,8.188217,None,9.28|2010-11-02T16:50:00Z,53.516566,8.188217,None,9.31|2010-11-02T17:00:00Z,53.516566,8.188217,None,9.32|2010-11-02T17:10:00Z,53.516566,8.188217,None,9.3|2010-11-02T17:20:00Z,53.516566,8.188217,None,9.29|2010-11-02T17:30:00Z,53.516566,8.188217,None,9.29|2010-11-02T17:40:00Z,53.516566,8.188217,None,9.3|2010-11-02T17:50:00Z,53.516566,8.188217,None,9.31|2010-11-02T18:00:00Z,53.516566,8.188217,None,9.31|2010-11-02T18:10:00Z,53.516566,8.188217,None,9.32|2010-11-02T18:20:00Z,53.516566,8.188217,None,9.32|2010-11-02T18:30:00Z,53.516566,8.188217,None,9.32|2010-11-02T18:40:00Z,53.516566,8.188217,None,9.33|2010-11-02T18:50:00Z,53.516566,8.188217,None,9.33|2010-11-02T19:00:00Z,53.516566,8.188217,None,9.33|2010-11-02T19:10:00Z,53.516566,8.188217,None,9.34|2010-11-02T19:20:00Z,53.516566,8.188217,None,9.34|2010-11-02T19:30:00Z,53.516566,8.188217,None,9.35|2010-11-02T19:40:00Z,53.516566,8.188217,None,9.35|2010-11-02T19:50:00Z,53.516566,8.188217,None,9.35|2010-11-02T20:00:00Z,53.516566,8.188217,None,9.35|2010-11-02T20:10:00Z,53.516566,8.188217,None,9.36|2010-11-02T20:20:00Z,53.516566,8.188217,None,9.36|2010-11-02T20:30:00Z,53.516566,8.188217,None,9.36|2010-11-02T20:40:00Z,53.516566,8.188217,None,9.36|2010-11-02T20:50:00Z,53.516566,8.188217,None,9.36|2010-11-02T21:00:00Z,53.516566,8.188217,None,9.36|2010-11-02T21:10:00Z,53.516566,8.188217,None,9.35|2010-11-02T21:20:00Z,53.516566,8.188217,None,9.34|2010-11-02T21:30:00Z,53.516566,8.188217,None,9.34|2010-11-02T21:40:00Z,53.516566,8.188217,None,9.34|2010-11-02T21:50:00Z,53.516566,8.188217,None,9.34|2010-11-02T22:00:00Z,53.516566,8.188217,None,9.34|2010-11-02T22:10:00Z,53.516566,8.188217,None,9.34|2010-11-02T22:20:00Z,53.516566,8.188217,None,9.35|2010-11-02T22:30:00Z,53.516566,8.188217,None,9.37|2010-11-02T22:40:00Z,53.516566,8.188217,None,9.38|2010-11-02T22:50:00Z,53.516566,8.188217,None,9.36|2010-11-02T23:00:00Z,53.516566,8.188217,None,9.34|2010-11-02T23:10:00Z,53.516566,8.188217,None,9.34|2010-11-02T23:20:00Z,53.516566,8.188217,None,9.34|2010-11-02T23:30:00Z,53.516566,8.188217,None,9.35|2010-11-02T23:40:00Z,53.516566,8.188217,None,9.36|2010-11-02T23:50:00Z,53.516566,8.188217,None,9.36|2010-11-03T00:00:00Z,53.516566,8.188217,None,9.36|2010-11-03T00:10:00Z,53.516566,8.188217,None,9.36|2010-11-03T00:20:00Z,53.516566,8.188217,None,9.37|2010-11-03T00:30:00Z,53.516566,8.188217,None,9.37|2010-11-03T00:40:00Z,53.516566,8.188217,None,9.37|2010-11-03T00:50:00Z,53.516566,8.188217,None,9.37|2010-11-03T01:00:00Z,53.516566,8.188217,None,9.38|2010-11-03T01:10:00Z,53.516566,8.188217,None,9.38|2010-11-03T01:20:00Z,53.516566,8.188217,None,9.38|2010-11-03T01:30:00Z,53.516566,8.188217,None,9.38|2010-11-03T01:40:00Z,53.516566,8.188217,None,9.39|2010-11-03T01:50:00Z,53.516566,8.188217,None,9.39|2010-11-03T02:00:00Z,53.516566,8.188217,None,9.4|2010-11-03T02:10:00Z,53.516566,8.188217,None,9.4|2010-11-03T02:20:00Z,53.516566,8.188217,None,9.4|2010-11-03T02:30:00Z,53.516566,8.188217,None,9.4|2010-11-03T02:40:00Z,53.516566,8.188217,None,9.4|2010-11-03T02:50:00Z,53.516566,8.188217,None,9.4|2010-11-03T03:00:00Z,53.516566,8.188217,None,9.39|2010-11-03T03:10:00Z,53.516566,8.188217,None,9.39|2010-11-03T03:20:00Z,53.516566,8.188217,None,9.39|2010-11-03T03:30:00Z,53.516566,8.188217,None,9.39|2010-11-03T03:40:00Z,53.516566,8.188217,None,9.39|2010-11-03T03:50:00Z,53.516566,8.188217,None,9.42|2010-11-03T04:00:00Z,53.516566,8.188217,None,9.43|2010-11-03T04:10:00Z,53.516566,8.188217,None,9.43|2010-11-03T04:20:00Z,53.516566,8.188217,None,9.44|2010-11-03T04:30:00Z,53.516566,8.188217,None,9.41|2010-11-03T04:40:00Z,53.516566,8.188217,None,9.39|2010-11-03T04:50:00Z,53.516566,8.188217,None,9.41|2010-11-03T05:00:00Z,53.516566,8.188217,None,9.39|2010-11-03T05:10:00Z,53.516566,8.188217,None,9.38|2010-11-03T05:20:00Z,53.516566,8.188217,None,9.37|2010-11-03T05:30:00Z,53.516566,8.188217,None,9.37|2010-11-03T05:40:00Z,53.516566,8.188217,None,9.37|2010-11-03T05:50:00Z,53.516566,8.188217,None,9.38|2010-11-03T06:00:00Z,53.516566,8.188217,None,9.39|2010-11-03T06:10:00Z,53.516566,8.188217,None,9.41|2010-11-03T06:20:00Z,53.516566,8.188217,None,9.42|2010-11-03T06:30:00Z,53.516566,8.188217,None,9.44|2010-11-03T06:40:00Z,53.516566,8.188217,None,9.44|2010-11-03T06:50:00Z,53.516566,8.188217,None,9.46|2010-11-03T07:00:00Z,53.516566,8.188217,None,9.51|2010-11-03T07:10:00Z,53.516566,8.188217,None,9.54|2010-11-03T07:20:00Z,53.516566,8.188217,None,9.58|2010-11-03T07:30:00Z,53.516566,8.188217,None,9.6|2010-11-03T07:40:00Z,53.516566,8.188217,None,9.61|2010-11-03T07:50:00Z,53.516566,8.188217,None,9.62|2010-11-03T08:00:00Z,53.516566,8.188217,None,9.63|2010-11-03T08:10:00Z,53.516566,8.188217,None,9.65|2010-11-03T08:20:00Z,53.516566,8.188217,None,9.65|2010-11-03T08:30:00Z,53.516566,8.188217,None,9.64|2010-11-03T08:40:00Z,53.516566,8.188217,None,9.63|2010-11-03T08:50:00Z,53.516566,8.188217,None,9.6|2010-11-03T09:00:00Z,53.516566,8.188217,None,9.59|2010-11-03T09:10:00Z,53.516566,8.188217,None,9.56|2010-11-03T09:20:00Z,53.516566,8.188217,None,9.5|2010-11-03T09:30:00Z,53.516566,8.188217,None,9.48|2010-11-03T09:40:00Z,53.516566,8.188217,None,9.46|2010-11-03T09:50:00Z,53.516566,8.188217,None,9.45|2010-11-03T10:00:00Z,53.516566,8.188217,None,9.45|2010-11-03T10:10:00Z,53.516566,8.188217,None,9.46|2010-11-03T10:20:00Z,53.516566,8.188217,None,9.48|2010-11-03T10:30:00Z,53.516566,8.188217,None,9.49|2010-11-03T10:40:00Z,53.516566,8.188217,None,9.51|2010-11-03T10:50:00Z,53.516566,8.188217,None,9.51|2010-11-03T11:00:00Z,53.516566,8.188217,None,9.5|2010-11-03T11:10:00Z,53.516566,8.188217,None,9.5|2010-11-03T11:20:00Z,53.516566,8.188217,None,9.51|2010-11-03T11:30:00Z,53.516566,8.188217,None,9.53|2010-11-03T11:40:00Z,53.516566,8.188217,None,9.55|2010-11-03T11:50:00Z,53.516566,8.188217,None,9.56|2010-11-03T12:00:00Z,53.516566,8.188217,None,9.56|2010-11-03T12:10:00Z,53.516566,8.188217,None,9.57|2010-11-03T12:20:00Z,53.516566,8.188217,None,9.58|2010-11-03T12:30:00Z,53.516566,8.188217,None,9.59|2010-11-03T12:40:00Z,53.516566,8.188217,None,9.58|2010-11-03T12:50:00Z,53.516566,8.188217,None,9.58|2010-11-03T13:00:00Z,53.516566,8.188217,None,9.58|2010-11-03T13:10:00Z,53.516566,8.188217,None,9.59|2010-11-03T13:20:00Z,53.516566,8.188217,None,9.59|2010-11-03T13:30:00Z,53.516566,8.188217,None,9.59|2010-11-03T13:40:00Z,53.516566,8.188217,None,9.6|2010-11-03T13:50:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:00:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:10:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:20:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:30:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:40:00Z,53.516566,8.188217,None,9.6|2010-11-03T14:50:00Z,53.516566,8.188217,None,9.6|2010-11-03T15:00:00Z,53.516566,8.188217,None,9.6|2010-11-03T15:10:00Z,53.516566,8.188217,None,9.6|2010-11-03T15:20:00Z,53.516566,8.188217,None,9.6|2010-11-03T15:30:00Z,53.516566,8.188217,None,9.62|2010-11-03T15:40:00Z,53.516566,8.188217,None,9.63|2010-11-03T15:50:00Z,53.516566,8.188217,None,9.66|2010-11-03T16:00:00Z,53.516566,8.188217,None,9.66|2010-11-03T16:10:00Z,53.516566,8.188217,None,9.66|2010-11-03T16:20:00Z,53.516566,8.188217,None,9.63|2010-11-03T16:30:00Z,53.516566,8.188217,None,9.64|2010-11-03T16:40:00Z,53.516566,8.188217,None,9.65|2010-11-03T16:50:00Z,53.516566,8.188217,None,9.66|2010-11-03T17:00:00Z,53.516566,8.188217,None,9.65|2010-11-03T17:10:00Z,53.516566,8.188217,None,9.64|2010-11-03T17:20:00Z,53.516566,8.188217,None,9.63|2010-11-03T17:30:00Z,53.516566,8.188217,None,9.62|2010-11-03T17:40:00Z,53.516566,8.188217,None,9.62|2010-11-03T17:50:00Z,53.516566,8.188217,None,9.61|2010-11-03T18:00:00Z,53.516566,8.188217,None,9.6|2010-11-03T18:10:00Z,53.516566,8.188217,None,9.59|2010-11-03T18:20:00Z,53.516566,8.188217,None,9.59|2010-11-03T18:30:00Z,53.516566,8.188217,None,9.57|2010-11-03T18:40:00Z,53.516566,8.188217,None,9.55|2010-11-03T18:50:00Z,53.516566,8.188217,None,9.55|2010-11-03T19:00:00Z,53.516566,8.188217,None,9.56|2010-11-03T19:10:00Z,53.516566,8.188217,None,9.57|2010-11-03T19:20:00Z,53.516566,8.188217,None,9.58|2010-11-03T19:30:00Z,53.516566,8.188217,None,9.58|2010-11-03T19:40:00Z,53.516566,8.188217,None,9.57|2010-11-03T19:50:00Z,53.516566,8.188217,None,9.55|2010-11-03T20:00:00Z,53.516566,8.188217,None,9.55|2010-11-03T20:10:00Z,53.516566,8.188217,None,9.55|2010-11-03T20:20:00Z,53.516566,8.188217,None,9.55|2010-11-03T20:30:00Z,53.516566,8.188217,None,9.55|2010-11-03T20:40:00Z,53.516566,8.188217,None,9.56|2010-11-03T20:50:00Z,53.516566,8.188217,None,9.58|2010-11-03T21:00:00Z,53.516566,8.188217,None,9.59|2010-11-03T21:10:00Z,53.516566,8.188217,None,9.59|2010-11-03T21:20:00Z,53.516566,8.188217,None,9.6|2010-11-03T21:30:00Z,53.516566,8.188217,None,9.6|2010-11-03T21:40:00Z,53.516566,8.188217,None,9.57|2010-11-03T21:50:00Z,53.516566,8.188217,None,9.54|2010-11-03T22:00:00Z,53.516566,8.188217,None,9.54|2010-11-03T22:10:00Z,53.516566,8.188217,None,9.55|2010-11-03T22:20:00Z,53.516566,8.188217,None,9.54|2010-11-03T22:30:00Z,53.516566,8.188217,None,9.52|2010-11-03T22:40:00Z,53.516566,8.188217,None,9.55|2010-11-03T22:50:00Z,53.516566,8.188217,None,9.56|2010-11-03T23:00:00Z,53.516566,8.188217,None,9.56|2010-11-03T23:10:00Z,53.516566,8.188217,None,9.56|2010-11-03T23:20:00Z,53.516566,8.188217,None,9.57|2010-11-03T23:30:00Z,53.516566,8.188217,None,9.58|2010-11-03T23:40:00Z,53.516566,8.188217,None,9.58|2010-11-03T23:50:00Z,53.516566,8.188217,None,9.57|2010-11-04T00:00:00Z,53.516566,8.188217,None,9.57|2010-11-04T00:10:00Z,53.516566,8.188217,None,9.56|2010-11-04T00:20:00Z,53.516566,8.188217,None,9.56|2010-11-04T00:30:00Z,53.516566,8.188217,None,9.56|2010-11-04T00:40:00Z,53.516566,8.188217,None,9.58|2010-11-04T00:50:00Z,53.516566,8.188217,None,9.59|2010-11-04T01:00:00Z,53.516566,8.188217,None,9.58|2010-11-04T01:10:00Z,53.516566,8.188217,None,9.58|2010-11-04T01:20:00Z,53.516566,8.188217,None,9.58|2010-11-04T01:30:00Z,53.516566,8.188217,None,9.58|2010-11-04T01:40:00Z,53.516566,8.188217,None,9.58|2010-11-04T01:50:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:00:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:10:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:20:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:30:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:40:00Z,53.516566,8.188217,None,9.58|2010-11-04T02:50:00Z,53.516566,8.188217,None,9.58|2010-11-04T03:00:00Z,53.516566,8.188217,None,9.58|2010-11-04T03:10:00Z,53.516566,8.188217,None,9.58|2010-11-04T03:20:00Z,53.516566,8.188217,None,9.59|2010-11-04T03:30:00Z,53.516566,8.188217,None,9.59|2010-11-04T03:40:00Z,53.516566,8.188217,None,9.59|2010-11-04T03:50:00Z,53.516566,8.188217,None,9.59|2010-11-04T04:00:00Z,53.516566,8.188217,None,9.6|2010-11-04T04:10:00Z,53.516566,8.188217,None,9.6|2010-11-04T04:20:00Z,53.516566,8.188217,None,9.59|2010-11-04T04:30:00Z,53.516566,8.188217,None,9.6|2010-11-04T04:40:00Z,53.516566,8.188217,None,9.59|2010-11-04T04:50:00Z,53.516566,8.188217,None,9.59|2010-11-04T05:00:00Z,53.516566,8.188217,None,9.6|2010-11-04T05:10:00Z,53.516566,8.188217,None,9.62|2010-11-04T05:20:00Z,53.516566,8.188217,None,9.61|2010-11-04T05:30:00Z,53.516566,8.188217,None,9.61|2010-11-04T05:40:00Z,53.516566,8.188217,None,9.62|2010-11-04T05:50:00Z,53.516566,8.188217,None,9.62|2010-11-04T06:00:00Z,53.516566,8.188217,None,9.62|2010-11-04T06:10:00Z,53.516566,8.188217,None,9.61|2010-11-04T06:20:00Z,53.516566,8.188217,None,9.61|2010-11-04T06:30:00Z,53.516566,8.188217,None,9.6|2010-11-04T06:40:00Z,53.516566,8.188217,None,9.6|2010-11-04T06:50:00Z,53.516566,8.188217,None,9.6|2010-11-04T07:00:00Z,53.516566,8.188217,None,9.59|2010-11-04T07:10:00Z,53.516566,8.188217,None,9.59|2010-11-04T07:20:00Z,53.516566,8.188217,None,9.59|2010-11-04T07:30:00Z,53.516566,8.188217,None,9.59|2010-11-04T07:40:00Z,53.516566,8.188217,None,9.59|2010-11-04T07:50:00Z,53.516566,8.188217,None,9.58|2010-11-04T08:00:00Z,53.516566,8.188217,None,9.58|2010-11-04T08:10:00Z,53.516566,8.188217,None,9.58|2010-11-04T08:20:00Z,53.516566,8.188217,None,9.59|2010-11-04T08:30:00Z,53.516566,8.188217,None,9.59|2010-11-04T08:40:00Z,53.516566,8.188217,None,9.59|2010-11-04T08:50:00Z,53.516566,8.188217,None,9.6|2010-11-04T09:00:00Z,53.516566,8.188217,None,9.61|2010-11-04T09:10:00Z,53.516566,8.188217,None,9.64|2010-11-04T09:20:00Z,53.516566,8.188217,None,9.65|2010-11-04T09:30:00Z,53.516566,8.188217,None,9.65|2010-11-04T09:40:00Z,53.516566,8.188217,None,9.65|2010-11-04T09:50:00Z,53.516566,8.188217,None,9.65|2010-11-04T10:00:00Z,53.516566,8.188217,None,9.66|2010-11-04T10:10:00Z,53.516566,8.188217,None,9.67|2010-11-04T10:20:00Z,53.516566,8.188217,None,9.66|2010-11-04T10:30:00Z,53.516566,8.188217,None,9.66|2010-11-04T10:40:00Z,53.516566,8.188217,None,9.67|2010-11-04T10:50:00Z,53.516566,8.188217,None,9.68|2010-11-04T11:00:00Z,53.516566,8.188217,None,9.7|2010-11-04T11:10:00Z,53.516566,8.188217,None,9.72|2010-11-04T11:20:00Z,53.516566,8.188217,None,9.74|2010-11-04T11:30:00Z,53.516566,8.188217,None,9.77|2010-11-04T11:40:00Z,53.516566,8.188217,None,9.79|2010-11-04T11:50:00Z,53.516566,8.188217,None,9.79|2010-11-04T12:00:00Z,53.516566,8.188217,None,9.78|2010-11-04T12:10:00Z,53.516566,8.188217,None,9.77|2010-11-04T12:20:00Z,53.516566,8.188217,None,9.78|2010-11-04T12:30:00Z,53.516566,8.188217,None,9.78|2010-11-04T12:40:00Z,53.516566,8.188217,None,9.76|2010-11-04T12:50:00Z,53.516566,8.188217,None,9.77|2010-11-04T13:00:00Z,53.516566,8.188217,None,9.78|2010-11-04T13:10:00Z,53.516566,8.188217,None,9.78|2010-11-04T13:20:00Z,53.516566,8.188217,None,9.79|2010-11-04T13:30:00Z,53.516566,8.188217,None,9.81|2010-11-04T13:40:00Z,53.516566,8.188217,None,9.82|2010-11-04T13:50:00Z,53.516566,8.188217,None,9.83|2010-11-04T14:00:00Z,53.516566,8.188217,None,9.85|2010-11-04T14:10:00Z,53.516566,8.188217,None,9.86|2010-11-04T14:20:00Z,53.516566,8.188217,None,9.87|2010-11-04T14:30:00Z,53.516566,8.188217,None,9.89|2010-11-04T14:40:00Z,53.516566,8.188217,None,9.89|2010-11-04T14:50:00Z,53.516566,8.188217,None,9.9|2010-11-04T15:00:00Z,53.516566,8.188217,None,9.92|2010-11-04T15:10:00Z,53.516566,8.188217,None,9.92|2010-11-04T15:20:00Z,53.516566,8.188217,None,9.93|2010-11-04T15:30:00Z,53.516566,8.188217,None,9.93|2010-11-04T16:10:00Z,53.516566,8.188217,None,9.96|2010-11-04T16:20:00Z,53.516566,8.188217,None,9.97|2010-11-04T16:30:00Z,53.516566,8.188217,None,9.98|2010-11-04T16:40:00Z,53.516566,8.188217,None,10.0|2010-11-04T16:50:00Z,53.516566,8.188217,None,9.99|2010-11-04T17:00:00Z,53.516566,8.188217,None,9.99|2010-11-04T17:10:00Z,53.516566,8.188217,None,10.06|2010-11-04T17:20:00Z,53.516566,8.188217,None,10.06|2010-11-04T17:30:00Z,53.516566,8.188217,None,10.04|2010-11-04T17:40:00Z,53.516566,8.188217,None,10.01|2010-11-04T17:50:00Z,53.516566,8.188217,None,10.04|2010-11-04T18:00:00Z,53.516566,8.188217,None,10.03|2010-11-04T18:10:00Z,53.516566,8.188217,None,10.01|2010-11-04T18:20:00Z,53.516566,8.188217,None,9.99|2010-11-04T18:30:00Z,53.516566,8.188217,None,10.01|2010-11-04T18:40:00Z,53.516566,8.188217,None,10.01|2010-11-04T18:50:00Z,53.516566,8.188217,None,10.03|2010-11-04T19:00:00Z,53.516566,8.188217,None,9.98|2010-11-04T19:10:00Z,53.516566,8.188217,None,9.95|2010-11-04T19:20:00Z,53.516566,8.188217,None,9.92|2010-11-04T19:30:00Z,53.516566,8.188217,None,9.91|2010-11-04T19:40:00Z,53.516566,8.188217,None,9.9|2010-11-04T19:50:00Z,53.516566,8.188217,None,9.89|2010-11-04T20:00:00Z,53.516566,8.188217,None,9.88|2010-11-04T20:10:00Z,53.516566,8.188217,None,9.86|2010-11-04T20:20:00Z,53.516566,8.188217,None,9.85|2010-11-04T20:30:00Z,53.516566,8.188217,None,9.84|2010-11-04T20:40:00Z,53.516566,8.188217,None,9.84|2010-11-04T20:50:00Z,53.516566,8.188217,None,9.84|2010-11-04T21:00:00Z,53.516566,8.188217,None,9.83|2010-11-04T21:10:00Z,53.516566,8.188217,None,9.82|2010-11-04T21:20:00Z,53.516566,8.188217,None,9.82|2010-11-04T21:30:00Z,53.516566,8.188217,None,9.81|2010-11-04T21:40:00Z,53.516566,8.188217,None,9.81|2010-11-04T21:50:00Z,53.516566,8.188217,None,9.8|2010-11-04T22:00:00Z,53.516566,8.188217,None,9.8|2010-11-04T22:10:00Z,53.516566,8.188217,None,9.8|2010-11-04T22:20:00Z,53.516566,8.188217,None,9.81|2010-11-04T22:30:00Z,53.516566,8.188217,None,9.81|2010-11-04T22:40:00Z,53.516566,8.188217,None,9.82|2010-11-04T22:50:00Z,53.516566,8.188217,None,9.83|2010-11-04T23:00:00Z,53.516566,8.188217,None,9.86|2010-11-04T23:10:00Z,53.516566,8.188217,None,9.89|2010-11-04T23:20:00Z,53.516566,8.188217,None,9.92|2010-11-04T23:30:00Z,53.516566,8.188217,None,9.96|2010-11-04T23:40:00Z,53.516566,8.188217,None,10.0|2010-11-04T23:50:00Z,53.516566,8.188217,None,10.02|2010-11-05T00:00:00Z,53.516566,8.188217,None,10.06|2010-11-05T00:10:00Z,53.516566,8.188217,None,10.11|2010-11-05T00:20:00Z,53.516566,8.188217,None,10.13|2010-11-05T00:30:00Z,53.516566,8.188217,None,10.04|2010-11-05T00:40:00Z,53.516566,8.188217,None,9.97|2010-11-05T00:50:00Z,53.516566,8.188217,None,9.96|2010-11-05T01:00:00Z,53.516566,8.188217,None,9.96|2010-11-05T01:10:00Z,53.516566,8.188217,None,9.97|2010-11-05T01:20:00Z,53.516566,8.188217,None,9.97|2010-11-05T01:30:00Z,53.516566,8.188217,None,9.99|2010-11-05T01:40:00Z,53.516566,8.188217,None,10.01|2010-11-05T01:50:00Z,53.516566,8.188217,None,10.02|2010-11-05T02:00:00Z,53.516566,8.188217,None,10.04|2010-11-05T02:10:00Z,53.516566,8.188217,None,10.05|2010-11-05T02:20:00Z,53.516566,8.188217,None,10.07|2010-11-05T02:30:00Z,53.516566,8.188217,None,10.08|2010-11-05T02:40:00Z,53.516566,8.188217,None,10.09|2010-11-05T02:50:00Z,53.516566,8.188217,None,10.11|2010-11-05T03:00:00Z,53.516566,8.188217,None,10.11|2010-11-05T03:10:00Z,53.516566,8.188217,None,10.1|2010-11-05T03:20:00Z,53.516566,8.188217,None,10.12|2010-11-05T03:30:00Z,53.516566,8.188217,None,10.14|2010-11-05T03:40:00Z,53.516566,8.188217,None,10.15|2010-11-05T03:50:00Z,53.516566,8.188217,None,10.15|2010-11-05T04:00:00Z,53.516566,8.188217,None,10.21|2010-11-05T04:10:00Z,53.516566,8.188217,None,10.28|2010-11-05T04:20:00Z,53.516566,8.188217,None,10.3|2010-11-05T04:30:00Z,53.516566,8.188217,None,10.31|2010-11-05T04:40:00Z,53.516566,8.188217,None,10.31|2010-11-05T04:50:00Z,53.516566,8.188217,None,10.31|2010-11-05T05:00:00Z,53.516566,8.188217,None,10.36|2010-11-05T05:10:00Z,53.516566,8.188217,None,10.36|2010-11-05T05:20:00Z,53.516566,8.188217,None,10.38|2010-11-05T05:30:00Z,53.516566,8.188217,None,10.36|2010-11-05T05:40:00Z,53.516566,8.188217,None,10.39|2010-11-05T05:50:00Z,53.516566,8.188217,None,10.42|2010-11-05T06:00:00Z,53.516566,8.188217,None,10.36|2010-11-05T06:10:00Z,53.516566,8.188217,None,10.3|2010-11-05T06:20:00Z,53.516566,8.188217,None,10.32|2010-11-05T06:30:00Z,53.516566,8.188217,None,10.3|2010-11-05T06:40:00Z,53.516566,8.188217,None,10.31|2010-11-05T06:50:00Z,53.516566,8.188217,None,10.28|2010-11-05T07:00:00Z,53.516566,8.188217,None,10.32|2010-11-05T07:10:00Z,53.516566,8.188217,None,10.37|2010-11-05T07:20:00Z,53.516566,8.188217,None,10.37|2010-11-05T07:30:00Z,53.516566,8.188217,None,10.3|2010-11-05T07:40:00Z,53.516566,8.188217,None,10.28|2010-11-05T07:50:00Z,53.516566,8.188217,None,10.22|2010-11-05T08:00:00Z,53.516566,8.188217,None,10.18|2010-11-05T08:10:00Z,53.516566,8.188217,None,10.17|2010-11-05T08:20:00Z,53.516566,8.188217,None,10.15|2010-11-05T08:30:00Z,53.516566,8.188217,None,10.13|2010-11-05T08:40:00Z,53.516566,8.188217,None,10.08|2010-11-05T08:50:00Z,53.516566,8.188217,None,10.07|2010-11-05T09:00:00Z,53.516566,8.188217,None,10.07|2010-11-05T09:10:00Z,53.516566,8.188217,None,10.03|2010-11-05T09:20:00Z,53.516566,8.188217,None,10.0|2010-11-05T09:30:00Z,53.516566,8.188217,None,9.99|2010-11-05T09:40:00Z,53.516566,8.188217,None,9.98|2010-11-05T09:50:00Z,53.516566,8.188217,None,9.97|2010-11-05T10:00:00Z,53.516566,8.188217,None,9.98|2010-11-05T10:10:00Z,53.516566,8.188217,None,9.98|2010-11-05T10:20:00Z,53.516566,8.188217,None,9.99|2010-11-05T10:30:00Z,53.516566,8.188217,None,9.99|2010-11-05T10:40:00Z,53.516566,8.188217,None,10.0|2010-11-05T10:50:00Z,53.516566,8.188217,None,10.0|2010-11-05T11:00:00Z,53.516566,8.188217,None,10.01|2010-11-05T11:10:00Z,53.516566,8.188217,None,10.02|2010-11-05T11:20:00Z,53.516566,8.188217,None,10.03|2010-11-05T11:30:00Z,53.516566,8.188217,None,10.04|2010-11-05T11:40:00Z,53.516566,8.188217,None,10.05|2010-11-05T11:50:00Z,53.516566,8.188217,None,10.07|2010-11-05T12:00:00Z,53.516566,8.188217,None,10.13|2010-11-05T12:10:00Z,53.516566,8.188217,None,10.22|2010-11-05T12:20:00Z,53.516566,8.188217,None,10.34|2010-11-05T12:30:00Z,53.516566,8.188217,None,10.45|2010-11-05T12:40:00Z,53.516566,8.188217,None,10.5|2010-11-05T12:50:00Z,53.516566,8.188217,None,10.49|2010-11-05T13:00:00Z,53.516566,8.188217,None,10.4|2010-11-05T13:10:00Z,53.516566,8.188217,None,10.31|2010-11-05T13:20:00Z,53.516566,8.188217,None,10.23|2010-11-05T13:30:00Z,53.516566,8.188217,None,10.2|2010-11-05T13:40:00Z,53.516566,8.188217,None,10.2|2010-11-05T13:50:00Z,53.516566,8.188217,None,10.21|2010-11-05T14:00:00Z,53.516566,8.188217,None,10.25|2010-11-05T14:10:00Z,53.516566,8.188217,None,10.27|2010-11-05T14:20:00Z,53.516566,8.188217,None,10.27|2010-11-05T14:30:00Z,53.516566,8.188217,None,10.28|2010-11-05T14:40:00Z,53.516566,8.188217,None,10.29|2010-11-05T14:50:00Z,53.516566,8.188217,None,10.3|2010-11-05T15:00:00Z,53.516566,8.188217,None,10.31|2010-11-05T15:10:00Z,53.516566,8.188217,None,10.32|2010-11-05T15:20:00Z,53.516566,8.188217,None,10.32|2010-11-05T15:30:00Z,53.516566,8.188217,None,10.32|2010-11-05T15:40:00Z,53.516566,8.188217,None,10.31|2010-11-05T15:50:00Z,53.516566,8.188217,None,10.32|2010-11-05T16:00:00Z,53.516566,8.188217,None,10.31|2010-11-05T16:10:00Z,53.516566,8.188217,None,10.3|2010-11-05T16:20:00Z,53.516566,8.188217,None,10.3|2010-11-05T16:30:00Z,53.516566,8.188217,None,10.28|2010-11-05T16:40:00Z,53.516566,8.188217,None,10.26|2010-11-05T16:50:00Z,53.516566,8.188217,None,10.26|2010-11-05T17:00:00Z,53.516566,8.188217,None,10.36|2010-11-05T17:10:00Z,53.516566,8.188217,None,10.42|2010-11-05T17:20:00Z,53.516566,8.188217,None,10.43|2010-11-05T17:30:00Z,53.516566,8.188217,None,10.42|2010-11-05T17:40:00Z,53.516566,8.188217,None,10.39|2010-11-05T17:50:00Z,53.516566,8.188217,None,10.37|2010-11-05T18:00:00Z,53.516566,8.188217,None,10.38|2010-11-05T18:10:00Z,53.516566,8.188217,None,10.56|2010-11-05T18:20:00Z,53.516566,8.188217,None,10.68|2010-11-05T18:30:00Z,53.516566,8.188217,None,10.66|2010-11-05T18:40:00Z,53.516566,8.188217,None,10.65|2010-11-05T18:50:00Z,53.516566,8.188217,None,10.68|2010-11-05T19:00:00Z,53.516566,8.188217,None,10.64|2010-11-05T19:10:00Z,53.516566,8.188217,None,10.62|2010-11-05T19:20:00Z,53.516566,8.188217,None,10.67|2010-11-05T19:30:00Z,53.516566,8.188217,None,10.67|2010-11-05T19:40:00Z,53.516566,8.188217,None,10.63|2010-11-05T19:50:00Z,53.516566,8.188217,None,10.58|2010-11-05T20:00:00Z,53.516566,8.188217,None,10.52|2010-11-05T20:10:00Z,53.516566,8.188217,None,10.45|2010-11-05T20:20:00Z,53.516566,8.188217,None,10.42|2010-11-05T20:30:00Z,53.516566,8.188217,None,10.4|2010-11-05T20:40:00Z,53.516566,8.188217,None,10.38|2010-11-05T20:50:00Z,53.516566,8.188217,None,10.35|2010-11-05T21:00:00Z,53.516566,8.188217,None,10.35|2010-11-05T21:10:00Z,53.516566,8.188217,None,10.36|2010-11-05T21:20:00Z,53.516566,8.188217,None,10.31|2010-11-05T21:30:00Z,53.516566,8.188217,None,10.28|2010-11-05T21:40:00Z,53.516566,8.188217,None,10.28|2010-11-05T21:50:00Z,53.516566,8.188217,None,10.24|2010-11-05T22:00:00Z,53.516566,8.188217,None,10.2|2010-11-05T22:10:00Z,53.516566,8.188217,None,10.16|2010-11-05T22:20:00Z,53.516566,8.188217,None,10.15|2010-11-05T22:30:00Z,53.516566,8.188217,None,10.15|2010-11-05T22:40:00Z,53.516566,8.188217,None,10.14|2010-11-05T22:50:00Z,53.516566,8.188217,None,10.13|2010-11-05T23:00:00Z,53.516566,8.188217,None,10.13|2010-11-05T23:10:00Z,53.516566,8.188217,None,10.14|2010-11-05T23:20:00Z,53.516566,8.188217,None,10.13|2010-11-05T23:30:00Z,53.516566,8.188217,None,10.13|2010-11-05T23:40:00Z,53.516566,8.188217,None,10.14|2010-11-05T23:50:00Z,53.516566,8.188217,None,10.14|2010-11-06T00:00:00Z,53.516566,8.188217,None,10.14|2010-11-06T00:10:00Z,53.516566,8.188217,None,10.17|2010-11-06T00:20:00Z,53.516566,8.188217,None,10.25|2010-11-06T00:30:00Z,53.516566,8.188217,None,10.3|2010-11-06T00:40:00Z,53.516566,8.188217,None,10.36|2010-11-06T00:50:00Z,53.516566,8.188217,None,10.49|2010-11-06T01:00:00Z,53.516566,8.188217,None,10.52|2010-11-06T01:10:00Z,53.516566,8.188217,None,10.48|2010-11-06T01:20:00Z,53.516566,8.188217,None,10.45|2010-11-06T01:30:00Z,53.516566,8.188217,None,10.42|2010-11-06T01:40:00Z,53.516566,8.188217,None,10.39|2010-11-06T01:50:00Z,53.516566,8.188217,None,10.39|2010-11-06T02:00:00Z,53.516566,8.188217,None,10.41|2010-11-06T02:10:00Z,53.516566,8.188217,None,10.4|2010-11-06T02:20:00Z,53.516566,8.188217,None,10.39|2010-11-06T02:30:00Z,53.516566,8.188217,None,10.4|2010-11-06T02:40:00Z,53.516566,8.188217,None,10.4|2010-11-06T02:50:00Z,53.516566,8.188217,None,10.39|2010-11-06T03:00:00Z,53.516566,8.188217,None,10.39|2010-11-06T03:10:00Z,53.516566,8.188217,None,10.39|2010-11-06T03:20:00Z,53.516566,8.188217,None,10.4|2010-11-06T03:30:00Z,53.516566,8.188217,None,10.4|2010-11-06T03:40:00Z,53.516566,8.188217,None,10.39|2010-11-06T03:50:00Z,53.516566,8.188217,None,10.39|2010-11-06T04:00:00Z,53.516566,8.188217,None,10.39|2010-11-06T04:10:00Z,53.516566,8.188217,None,10.38|2010-11-06T04:20:00Z,53.516566,8.188217,None,10.41|2010-11-06T04:30:00Z,53.516566,8.188217,None,10.41|2010-11-06T04:40:00Z,53.516566,8.188217,None,10.42|2010-11-06T04:50:00Z,53.516566,8.188217,None,10.42|2010-11-06T05:00:00Z,53.516566,8.188217,None,10.41|2010-11-06T05:10:00Z,53.516566,8.188217,None,10.41|2010-11-06T05:20:00Z,53.516566,8.188217,None,10.42|2010-11-06T05:30:00Z,53.516566,8.188217,None,10.43|2010-11-06T05:40:00Z,53.516566,8.188217,None,10.42|2010-11-06T05:50:00Z,53.516566,8.188217,None,10.41|2010-11-06T06:00:00Z,53.516566,8.188217,None,10.45|2010-11-06T06:10:00Z,53.516566,8.188217,None,10.48|2010-11-06T06:20:00Z,53.516566,8.188217,None,10.48|2010-11-06T06:30:00Z,53.516566,8.188217,None,10.53|2010-11-06T06:40:00Z,53.516566,8.188217,None,10.52|2010-11-06T06:50:00Z,53.516566,8.188217,None,10.54|2010-11-06T07:00:00Z,53.516566,8.188217,None,10.5|2010-11-06T07:10:00Z,53.516566,8.188217,None,10.5|2010-11-06T07:20:00Z,53.516566,8.188217,None,10.49|2010-11-06T07:30:00Z,53.516566,8.188217,None,10.48|2010-11-06T07:40:00Z,53.516566,8.188217,None,10.48|2010-11-06T07:50:00Z,53.516566,8.188217,None,10.51|2010-11-06T08:00:00Z,53.516566,8.188217,None,10.48|2010-11-06T08:10:00Z,53.516566,8.188217,None,10.47|2010-11-06T08:20:00Z,53.516566,8.188217,None,10.49|2010-11-06T08:30:00Z,53.516566,8.188217,None,10.46|2010-11-06T08:40:00Z,53.516566,8.188217,None,10.44|2010-11-06T08:50:00Z,53.516566,8.188217,None,10.44|2010-11-06T09:00:00Z,53.516566,8.188217,None,10.42|2010-11-06T09:10:00Z,53.516566,8.188217,None,10.39|2010-11-06T09:20:00Z,53.516566,8.188217,None,10.36|2010-11-06T09:30:00Z,53.516566,8.188217,None,10.36|2010-11-06T09:40:00Z,53.516566,8.188217,None,10.35|2010-11-06T09:50:00Z,53.516566,8.188217,None,10.36|2010-11-06T10:00:00Z,53.516566,8.188217,None,10.35|2010-11-06T10:10:00Z,53.516566,8.188217,None,10.32|2010-11-06T10:20:00Z,53.516566,8.188217,None,10.26|2010-11-06T10:30:00Z,53.516566,8.188217,None,10.23|2010-11-06T10:40:00Z,53.516566,8.188217,None,10.23|2010-11-06T10:50:00Z,53.516566,8.188217,None,10.21|2010-11-06T11:00:00Z,53.516566,8.188217,None,10.2|2010-11-06T11:10:00Z,53.516566,8.188217,None,10.21|2010-11-06T11:20:00Z,53.516566,8.188217,None,10.21|2010-11-06T11:30:00Z,53.516566,8.188217,None,10.21|2010-11-06T11:40:00Z,53.516566,8.188217,None,10.21|2010-11-06T11:50:00Z,53.516566,8.188217,None,10.22|2010-11-06T12:00:00Z,53.516566,8.188217,None,10.23|2010-11-06T12:10:00Z,53.516566,8.188217,None,10.24|2010-11-06T12:20:00Z,53.516566,8.188217,None,10.25|2010-11-06T12:30:00Z,53.516566,8.188217,None,10.26|2010-11-06T12:40:00Z,53.516566,8.188217,None,10.32|2010-11-06T12:50:00Z,53.516566,8.188217,None,10.39|2010-11-06T13:00:00Z,53.516566,8.188217,None,10.44|2010-11-06T13:10:00Z,53.516566,8.188217,None,10.49|2010-11-06T13:20:00Z,53.516566,8.188217,None,10.52|2010-11-06T13:30:00Z,53.516566,8.188217,None,10.52|2010-11-06T13:40:00Z,53.516566,8.188217,None,10.5|2010-11-06T13:50:00Z,53.516566,8.188217,None,10.47|2010-11-06T14:00:00Z,53.516566,8.188217,None,10.46|2010-11-06T14:10:00Z,53.516566,8.188217,None,10.45|2010-11-06T14:20:00Z,53.516566,8.188217,None,10.43|2010-11-06T14:30:00Z,53.516566,8.188217,None,10.43|2010-11-06T14:40:00Z,53.516566,8.188217,None,10.42|2010-11-06T14:50:00Z,53.516566,8.188217,None,10.42|2010-11-06T15:00:00Z,53.516566,8.188217,None,10.41|2010-11-06T15:10:00Z,53.516566,8.188217,None,10.41|2010-11-06T15:20:00Z,53.516566,8.188217,None,10.41|2010-11-06T15:30:00Z,53.516566,8.188217,None,10.4|2010-11-06T15:40:00Z,53.516566,8.188217,None,10.4|2010-11-06T15:50:00Z,53.516566,8.188217,None,10.41|2010-11-06T16:00:00Z,53.516566,8.188217,None,10.4|2010-11-06T16:10:00Z,53.516566,8.188217,None,10.4|2010-11-06T16:20:00Z,53.516566,8.188217,None,10.4|2010-11-06T16:30:00Z,53.516566,8.188217,None,10.41|2010-11-06T16:40:00Z,53.516566,8.188217,None,10.4|2010-11-06T16:50:00Z,53.516566,8.188217,None,10.4|2010-11-06T17:00:00Z,53.516566,8.188217,None,10.39|2010-11-06T17:10:00Z,53.516566,8.188217,None,10.39|2010-11-06T17:20:00Z,53.516566,8.188217,None,10.38|2010-11-06T17:30:00Z,53.516566,8.188217,None,10.37|2010-11-06T17:40:00Z,53.516566,8.188217,None,10.31|2010-11-06T17:50:00Z,53.516566,8.188217,None,10.32|2010-11-06T18:00:00Z,53.516566,8.188217,None,10.33|2010-11-06T18:10:00Z,53.516566,8.188217,None,10.33|2010-11-06T18:20:00Z,53.516566,8.188217,None,10.34|2010-11-06T18:30:00Z,53.516566,8.188217,None,10.37|2010-11-06T18:40:00Z,53.516566,8.188217,None,10.38|2010-11-06T18:50:00Z,53.516566,8.188217,None,10.38|2010-11-06T19:00:00Z,53.516566,8.188217,None,10.37|2010-11-06T19:10:00Z,53.516566,8.188217,None,10.3|2010-11-06T19:20:00Z,53.516566,8.188217,None,10.39|2010-11-06T19:30:00Z,53.516566,8.188217,None,10.39|2010-11-06T19:40:00Z,53.516566,8.188217,None,10.41|2010-11-06T19:50:00Z,53.516566,8.188217,None,10.45|2010-11-06T20:00:00Z,53.516566,8.188217,None,10.46|2010-11-06T20:10:00Z,53.516566,8.188217,None,10.48|2010-11-06T20:20:00Z,53.516566,8.188217,None,10.49|2010-11-06T20:30:00Z,53.516566,8.188217,None,10.49|2010-11-06T20:40:00Z,53.516566,8.188217,None,10.47|2010-11-06T20:50:00Z,53.516566,8.188217,None,10.45|2010-11-06T21:00:00Z,53.516566,8.188217,None,10.44|2010-11-06T21:10:00Z,53.516566,8.188217,None,10.44|2010-11-06T21:20:00Z,53.516566,8.188217,None,10.43|2010-11-06T21:30:00Z,53.516566,8.188217,None,10.43|2010-11-06T21:40:00Z,53.516566,8.188217,None,10.42|2010-11-06T21:50:00Z,53.516566,8.188217,None,10.41|2010-11-06T22:00:00Z,53.516566,8.188217,None,10.38|2010-11-06T22:10:00Z,53.516566,8.188217,None,10.36|2010-11-06T22:20:00Z,53.516566,8.188217,None,10.37|2010-11-06T22:30:00Z,53.516566,8.188217,None,10.37|2010-11-06T22:40:00Z,53.516566,8.188217,None,10.36|2010-11-06T22:50:00Z,53.516566,8.188217,None,10.35|2010-11-06T23:00:00Z,53.516566,8.188217,None,10.3|2010-11-06T23:10:00Z,53.516566,8.188217,None,10.27|2010-11-06T23:20:00Z,53.516566,8.188217,None,10.27|2010-11-06T23:30:00Z,53.516566,8.188217,None,10.27|2010-11-06T23:40:00Z,53.516566,8.188217,None,10.27|2010-11-06T23:50:00Z,53.516566,8.188217,None,10.28|2010-11-07T00:00:00Z,53.516566,8.188217,None,10.28|2010-11-07T00:10:00Z,53.516566,8.188217,None,10.29|2010-11-07T00:20:00Z,53.516566,8.188217,None,10.29|2010-11-07T00:30:00Z,53.516566,8.188217,None,10.26|2010-11-07T00:40:00Z,53.516566,8.188217,None,10.25|2010-11-07T00:50:00Z,53.516566,8.188217,None,10.24|2010-11-07T01:00:00Z,53.516566,8.188217,None,10.28|2010-11-07T01:10:00Z,53.516566,8.188217,None,10.29|2010-11-07T01:20:00Z,53.516566,8.188217,None,10.27|2010-11-07T01:30:00Z,53.516566,8.188217,None,10.2|2010-11-07T01:40:00Z,53.516566,8.188217,None,10.11|2010-11-07T01:50:00Z,53.516566,8.188217,None,10.09|2010-11-07T02:00:00Z,53.516566,8.188217,None,10.15|2010-11-07T02:10:00Z,53.516566,8.188217,None,10.19|2010-11-07T02:20:00Z,53.516566,8.188217,None,10.2|2010-11-07T02:30:00Z,53.516566,8.188217,None,10.21|2010-11-07T02:40:00Z,53.516566,8.188217,None,10.23|2010-11-07T02:50:00Z,53.516566,8.188217,None,10.22|2010-11-07T03:00:00Z,53.516566,8.188217,None,10.21|2010-11-07T03:10:00Z,53.516566,8.188217,None,10.22|2010-11-07T03:20:00Z,53.516566,8.188217,None,10.23|2010-11-07T03:30:00Z,53.516566,8.188217,None,10.22|2010-11-07T03:40:00Z,53.516566,8.188217,None,10.22|2010-11-07T03:50:00Z,53.516566,8.188217,None,10.22|2010-11-07T04:00:00Z,53.516566,8.188217,None,10.22|2010-11-07T04:10:00Z,53.516566,8.188217,None,10.22|2010-11-07T04:20:00Z,53.516566,8.188217,None,10.22|2010-11-07T04:30:00Z,53.516566,8.188217,None,10.23|2010-11-07T04:40:00Z,53.516566,8.188217,None,10.22|2010-11-07T04:50:00Z,53.516566,8.188217,None,10.23|2010-11-07T05:00:00Z,53.516566,8.188217,None,10.23|2010-11-07T05:10:00Z,53.516566,8.188217,None,10.21|2010-11-07T05:20:00Z,53.516566,8.188217,None,10.23|2010-11-07T05:30:00Z,53.516566,8.188217,None,10.24|2010-11-07T05:40:00Z,53.516566,8.188217,None,10.24|2010-11-07T05:50:00Z,53.516566,8.188217,None,10.24|2010-11-07T06:00:00Z,53.516566,8.188217,None,10.24|2010-11-07T06:10:00Z,53.516566,8.188217,None,10.22|2010-11-07T06:20:00Z,53.516566,8.188217,None,10.15|2010-11-07T06:30:00Z,53.516566,8.188217,None,10.15|2010-11-07T06:40:00Z,53.516566,8.188217,None,10.1|2010-11-07T06:50:00Z,53.516566,8.188217,None,10.07|2010-11-07T07:00:00Z,53.516566,8.188217,None,10.04|2010-11-07T07:10:00Z,53.516566,8.188217,None,10.01|2010-11-07T07:20:00Z,53.516566,8.188217,None,10.07|2010-11-07T07:30:00Z,53.516566,8.188217,None,10.02|2010-11-07T07:40:00Z,53.516566,8.188217,None,9.95|2010-11-07T07:50:00Z,53.516566,8.188217,None,9.91|2010-11-07T08:00:00Z,53.516566,8.188217,None,9.96|2010-11-07T08:10:00Z,53.516566,8.188217,None,10.01|2010-11-07T08:20:00Z,53.516566,8.188217,None,10.06|2010-11-07T08:30:00Z,53.516566,8.188217,None,10.13|2010-11-07T08:40:00Z,53.516566,8.188217,None,10.18|2010-11-07T08:50:00Z,53.516566,8.188217,None,10.18|2010-11-07T09:00:00Z,53.516566,8.188217,None,10.14|2010-11-07T09:10:00Z,53.516566,8.188217,None,10.12|2010-11-07T09:20:00Z,53.516566,8.188217,None,10.19|2010-11-07T09:30:00Z,53.516566,8.188217,None,10.21|2010-11-07T09:40:00Z,53.516566,8.188217,None,10.22|2010-11-07T09:50:00Z,53.516566,8.188217,None,10.22|2010-11-07T10:00:00Z,53.516566,8.188217,None,10.21|2010-11-07T10:10:00Z,53.516566,8.188217,None,10.19|2010-11-07T10:20:00Z,53.516566,8.188217,None,10.15|2010-11-07T10:30:00Z,53.516566,8.188217,None,10.13|2010-11-07T10:40:00Z,53.516566,8.188217,None,10.15|2010-11-07T10:50:00Z,53.516566,8.188217,None,10.17|2010-11-07T11:00:00Z,53.516566,8.188217,None,10.2|2010-11-07T11:10:00Z,53.516566,8.188217,None,10.23|2010-11-07T11:20:00Z,53.516566,8.188217,None,10.24|2010-11-07T11:30:00Z,53.516566,8.188217,None,10.24|2010-11-07T11:40:00Z,53.516566,8.188217,None,10.24|2010-11-07T11:50:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:00:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:10:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:20:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:30:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:40:00Z,53.516566,8.188217,None,10.23|2010-11-07T12:50:00Z,53.516566,8.188217,None,10.23|2010-11-07T13:00:00Z,53.516566,8.188217,None,10.23|2010-11-07T13:10:00Z,53.516566,8.188217,None,10.23|2010-11-07T13:20:00Z,53.516566,8.188217,None,10.25|2010-11-07T13:30:00Z,53.516566,8.188217,None,10.21|2010-11-07T13:40:00Z,53.516566,8.188217,None,10.17|2010-11-07T13:50:00Z,53.516566,8.188217,None,10.17|2010-11-07T14:00:00Z,53.516566,8.188217,None,10.07|2010-11-07T14:10:00Z,53.516566,8.188217,None,10.0|2010-11-07T14:20:00Z,53.516566,8.188217,None,9.99|2010-11-07T14:30:00Z,53.516566,8.188217,None,10.0|2010-11-07T14:40:00Z,53.516566,8.188217,None,10.01|2010-11-07T14:50:00Z,53.516566,8.188217,None,10.02|2010-11-07T15:00:00Z,53.516566,8.188217,None,10.01|2010-11-07T15:10:00Z,53.516566,8.188217,None,10.0|2010-11-07T15:20:00Z,53.516566,8.188217,None,9.98|2010-11-07T15:30:00Z,53.516566,8.188217,None,9.97|2010-11-07T15:40:00Z,53.516566,8.188217,None,9.96|2010-11-07T15:50:00Z,53.516566,8.188217,None,9.94|2010-11-07T16:00:00Z,53.516566,8.188217,None,9.92|2010-11-07T16:10:00Z,53.516566,8.188217,None,9.93|2010-11-07T16:20:00Z,53.516566,8.188217,None,9.96|2010-11-07T16:30:00Z,53.516566,8.188217,None,9.95|2010-11-07T16:40:00Z,53.516566,8.188217,None,10.04|2010-11-07T16:50:00Z,53.516566,8.188217,None,10.02|2010-11-07T17:00:00Z,53.516566,8.188217,None,10.05|2010-11-07T17:10:00Z,53.516566,8.188217,None,10.08|2010-11-07T17:20:00Z,53.516566,8.188217,None,10.04|2010-11-07T17:30:00Z,53.516566,8.188217,None,9.98|2010-11-07T17:40:00Z,53.516566,8.188217,None,10.03|2010-11-07T17:50:00Z,53.516566,8.188217,None,10.06|2010-11-07T18:00:00Z,53.516566,8.188217,None,10.04|2010-11-07T18:10:00Z,53.516566,8.188217,None,10.04|2010-11-07T18:20:00Z,53.516566,8.188217,None,10.04|2010-11-07T18:30:00Z,53.516566,8.188217,None,10.04|2010-11-07T18:40:00Z,53.516566,8.188217,None,9.97|2010-11-07T18:50:00Z,53.516566,8.188217,None,9.82|2010-11-07T19:00:00Z,53.516566,8.188217,None,9.87|2010-11-07T19:10:00Z,53.516566,8.188217,None,9.77|2010-11-07T19:20:00Z,53.516566,8.188217,None,9.84|2010-11-07T19:30:00Z,53.516566,8.188217,None,9.68|2010-11-07T19:40:00Z,53.516566,8.188217,None,9.73|2010-11-07T19:50:00Z,53.516566,8.188217,None,9.76|2010-11-07T20:00:00Z,53.516566,8.188217,None,9.73|2010-11-07T20:10:00Z,53.516566,8.188217,None,9.7|2010-11-07T20:20:00Z,53.516566,8.188217,None,9.83|2010-11-07T20:30:00Z,53.516566,8.188217,None,9.88|2010-11-07T20:40:00Z,53.516566,8.188217,None,9.87|2010-11-07T20:50:00Z,53.516566,8.188217,None,9.94|2010-11-07T21:00:00Z,53.516566,8.188217,None,10.02|2010-11-07T21:10:00Z,53.516566,8.188217,None,10.0|2010-11-07T21:20:00Z,53.516566,8.188217,None,9.98|2010-11-07T21:30:00Z,53.516566,8.188217,None,9.99|2010-11-07T21:40:00Z,53.516566,8.188217,None,10.04|2010-11-07T21:50:00Z,53.516566,8.188217,None,10.05|2010-11-07T22:00:00Z,53.516566,8.188217,None,10.07|2010-11-07T22:10:00Z,53.516566,8.188217,None,10.08|2010-11-07T22:20:00Z,53.516566,8.188217,None,10.08|2010-11-07T22:30:00Z,53.516566,8.188217,None,10.05|2010-11-07T22:40:00Z,53.516566,8.188217,None,10.0|2010-11-07T22:50:00Z,53.516566,8.188217,None,9.98|2010-11-07T23:00:00Z,53.516566,8.188217,None,9.98|2010-11-07T23:10:00Z,53.516566,8.188217,None,10.0|2010-11-07T23:20:00Z,53.516566,8.188217,None,10.02|2010-11-07T23:30:00Z,53.516566,8.188217,None,10.07|2010-11-07T23:40:00Z,53.516566,8.188217,None,10.11|2010-11-07T23:50:00Z,53.516566,8.188217,None,10.12|2010-11-08T00:00:00Z,53.516566,8.188217,None,10.12|2010-11-08T00:10:00Z,53.516566,8.188217,None,10.12|2010-11-08T00:20:00Z,53.516566,8.188217,None,10.13|2010-11-08T00:30:00Z,53.516566,8.188217,None,10.13|2010-11-08T00:40:00Z,53.516566,8.188217,None,10.13|2010-11-08T00:50:00Z,53.516566,8.188217,None,10.13|2010-11-08T01:00:00Z,53.516566,8.188217,None,10.12|2010-11-08T01:10:00Z,53.516566,8.188217,None,10.08|2010-11-08T01:20:00Z,53.516566,8.188217,None,10.02|2010-11-08T01:30:00Z,53.516566,8.188217,None,9.97|2010-11-08T01:40:00Z,53.516566,8.188217,None,9.84|2010-11-08T01:50:00Z,53.516566,8.188217,None,9.77|2010-11-08T02:00:00Z,53.516566,8.188217,None,9.53|2010-11-08T02:10:00Z,53.516566,8.188217,None,9.53|2010-11-08T02:20:00Z,53.516566,8.188217,None,9.2|2010-11-08T02:30:00Z,53.516566,8.188217,None,9.07|2010-11-08T02:40:00Z,53.516566,8.188217,None,9.03|2010-11-08T02:50:00Z,53.516566,8.188217,None,9.14|2010-11-08T03:00:00Z,53.516566,8.188217,None,9.22|2010-11-08T03:10:00Z,53.516566,8.188217,None,9.27|2010-11-08T03:20:00Z,53.516566,8.188217,None,9.27|2010-11-08T03:30:00Z,53.516566,8.188217,None,9.27|2010-11-08T03:40:00Z,53.516566,8.188217,None,9.32|2010-11-08T03:50:00Z,53.516566,8.188217,None,9.3|2010-11-08T04:00:00Z,53.516566,8.188217,None,9.23|2010-11-08T04:10:00Z,53.516566,8.188217,None,9.22|2010-11-08T04:20:00Z,53.516566,8.188217,None,9.21|2010-11-08T04:30:00Z,53.516566,8.188217,None,9.32|2010-11-08T04:40:00Z,53.516566,8.188217,None,9.58|2010-11-08T04:50:00Z,53.516566,8.188217,None,9.53|2010-11-08T05:00:00Z,53.516566,8.188217,None,9.56|2010-11-08T05:10:00Z,53.516566,8.188217,None,9.45|2010-11-08T05:20:00Z,53.516566,8.188217,None,9.46|2010-11-08T05:30:00Z,53.516566,8.188217,None,9.42|2010-11-08T05:40:00Z,53.516566,8.188217,None,9.33|2010-11-08T05:50:00Z,53.516566,8.188217,None,9.36|2010-11-08T06:00:00Z,53.516566,8.188217,None,9.32|2010-11-08T06:10:00Z,53.516566,8.188217,None,9.36|2010-11-08T06:20:00Z,53.516566,8.188217,None,9.46|2010-11-08T06:30:00Z,53.516566,8.188217,None,9.51|2010-11-08T06:40:00Z,53.516566,8.188217,None,9.55|2010-11-08T06:50:00Z,53.516566,8.188217,None,9.55|2010-11-08T07:00:00Z,53.516566,8.188217,None,9.52|2010-11-08T07:10:00Z,53.516566,8.188217,None,9.5|2010-11-08T07:20:00Z,53.516566,8.188217,None,9.5|2010-11-08T07:30:00Z,53.516566,8.188217,None,9.4|2010-11-08T07:40:00Z,53.516566,8.188217,None,9.24|2010-11-08T07:50:00Z,53.516566,8.188217,None,9.22|2010-11-08T08:00:00Z,53.516566,8.188217,None,9.32|2010-11-08T08:10:00Z,53.516566,8.188217,None,9.08|2010-11-08T08:20:00Z,53.516566,8.188217,None,9.11|2010-11-08T08:30:00Z,53.516566,8.188217,None,9.06|2010-11-08T08:40:00Z,53.516566,8.188217,None,8.97|2010-11-08T08:50:00Z,53.516566,8.188217,None,8.89|2010-11-08T09:00:00Z,53.516566,8.188217,None,8.91|2010-11-08T09:10:00Z,53.516566,8.188217,None,9.0|2010-11-08T09:20:00Z,53.516566,8.188217,None,9.09|2010-11-08T09:30:00Z,53.516566,8.188217,None,9.31|2010-11-08T09:40:00Z,53.516566,8.188217,None,9.35|2010-11-08T09:50:00Z,53.516566,8.188217,None,9.3|2010-11-08T10:00:00Z,53.516566,8.188217,None,9.28|2010-11-08T10:10:00Z,53.516566,8.188217,None,9.29|2010-11-08T10:20:00Z,53.516566,8.188217,None,9.38|2010-11-08T10:30:00Z,53.516566,8.188217,None,9.43|2010-11-08T10:40:00Z,53.516566,8.188217,None,9.43|2010-11-08T10:50:00Z,53.516566,8.188217,None,9.46|2010-11-08T11:00:00Z,53.516566,8.188217,None,9.5|2010-11-08T11:10:00Z,53.516566,8.188217,None,9.52|2010-11-08T11:20:00Z,53.516566,8.188217,None,9.51|2010-11-08T11:30:00Z,53.516566,8.188217,None,9.52|2010-11-08T11:40:00Z,53.516566,8.188217,None,9.53|2010-11-08T11:50:00Z,53.516566,8.188217,None,9.52|2010-11-08T12:00:00Z,53.516566,8.188217,None,9.5|2010-11-08T12:10:00Z,53.516566,8.188217,None,9.49|2010-11-08T12:20:00Z,53.516566,8.188217,None,9.49|2010-11-08T12:30:00Z,53.516566,8.188217,None,9.5|2010-11-08T12:40:00Z,53.516566,8.188217,None,9.5|2010-11-08T12:50:00Z,53.516566,8.188217,None,9.51|2010-11-08T13:00:00Z,53.516566,8.188217,None,9.51|2010-11-08T13:10:00Z,53.516566,8.188217,None,9.49|2010-11-08T13:20:00Z,53.516566,8.188217,None,9.46|2010-11-08T13:30:00Z,53.516566,8.188217,None,9.41|2010-11-08T13:40:00Z,53.516566,8.188217,None,9.41|2010-11-08T13:50:00Z,53.516566,8.188217,None,9.36|2010-11-08T14:00:00Z,53.516566,8.188217,None,9.07|2010-11-08T14:10:00Z,53.516566,8.188217,None,8.57|2010-11-08T14:20:00Z,53.516566,8.188217,None,8.44|2010-11-08T14:30:00Z,53.516566,8.188217,None,8.44|2010-11-08T14:40:00Z,53.516566,8.188217,None,8.48|2010-11-08T14:50:00Z,53.516566,8.188217,None,8.5|2010-11-08T15:00:00Z,53.516566,8.188217,None,8.39|2010-11-08T15:10:00Z,53.516566,8.188217,None,8.39|2010-11-08T15:20:00Z,53.516566,8.188217,None,8.46|2010-11-08T15:30:00Z,53.516566,8.188217,None,8.33|2010-11-08T15:40:00Z,53.516566,8.188217,None,7.99|2010-11-08T15:50:00Z,53.516566,8.188217,None,7.6|2010-11-08T16:00:00Z,53.516566,8.188217,None,7.62|2010-11-08T16:10:00Z,53.516566,8.188217,None,8.0|2010-11-08T16:20:00Z,53.516566,8.188217,None,8.52|2010-11-08T16:30:00Z,53.516566,8.188217,None,8.83|2010-11-08T16:40:00Z,53.516566,8.188217,None,9.05|2010-11-08T16:50:00Z,53.516566,8.188217,None,9.15|2010-11-08T17:00:00Z,53.516566,8.188217,None,8.93|2010-11-08T17:10:00Z,53.516566,8.188217,None,8.81|2010-11-08T17:20:00Z,53.516566,8.188217,None,8.8|2010-11-08T17:30:00Z,53.516566,8.188217,None,8.88|2010-11-08T17:40:00Z,53.516566,8.188217,None,8.77|2010-11-08T17:50:00Z,53.516566,8.188217,None,8.61|2010-11-08T18:00:00Z,53.516566,8.188217,None,8.72|2010-11-08T18:10:00Z,53.516566,8.188217,None,8.64|2010-11-08T18:20:00Z,53.516566,8.188217,None,8.63|2010-11-08T18:30:00Z,53.516566,8.188217,None,8.65|2010-11-08T18:40:00Z,53.516566,8.188217,None,8.63|2010-11-08T18:50:00Z,53.516566,8.188217,None,8.61|2010-11-08T19:00:00Z,53.516566,8.188217,None,8.63|2010-11-08T19:10:00Z,53.516566,8.188217,None,8.68|2010-11-08T19:20:00Z,53.516566,8.188217,None,8.79|2010-11-08T19:30:00Z,53.516566,8.188217,None,8.84|2010-11-08T19:40:00Z,53.516566,8.188217,None,8.83|2010-11-08T19:50:00Z,53.516566,8.188217,None,8.81|2010-11-08T20:00:00Z,53.516566,8.188217,None,8.79|2010-11-08T20:10:00Z,53.516566,8.188217,None,8.65|2010-11-08T20:20:00Z,53.516566,8.188217,None,8.5|2010-11-08T20:30:00Z,53.516566,8.188217,None,8.57|2010-11-08T20:40:00Z,53.516566,8.188217,None,8.71|2010-11-08T20:50:00Z,53.516566,8.188217,None,8.85|2010-11-08T21:00:00Z,53.516566,8.188217,None,8.94|2010-11-08T21:10:00Z,53.516566,8.188217,None,8.91|2010-11-08T21:20:00Z,53.516566,8.188217,None,8.66|2010-11-08T21:30:00Z,53.516566,8.188217,None,8.69|2010-11-08T21:40:00Z,53.516566,8.188217,None,8.82|2010-11-08T21:50:00Z,53.516566,8.188217,None,8.9|2010-11-08T22:00:00Z,53.516566,8.188217,None,8.8|2010-11-08T22:10:00Z,53.516566,8.188217,None,8.75|2010-11-08T22:20:00Z,53.516566,8.188217,None,8.84|2010-11-08T22:30:00Z,53.516566,8.188217,None,8.97|2010-11-08T22:40:00Z,53.516566,8.188217,None,9.05|2010-11-08T22:50:00Z,53.516566,8.188217,None,9.08|2010-11-08T23:00:00Z,53.516566,8.188217,None,8.9|2010-11-08T23:10:00Z,53.516566,8.188217,None,9.12|2010-11-08T23:20:00Z,53.516566,8.188217,None,9.11|2010-11-08T23:30:00Z,53.516566,8.188217,None,9.1|2010-11-08T23:40:00Z,53.516566,8.188217,None,9.13|2010-11-08T23:50:00Z,53.516566,8.188217,None,9.16|2010-11-09T00:00:00Z,53.516566,8.188217,None,9.2|2010-11-09T00:10:00Z,53.516566,8.188217,None,9.22|2010-11-09T00:20:00Z,53.516566,8.188217,None,9.2|2010-11-09T00:30:00Z,53.516566,8.188217,None,9.16|2010-11-09T00:40:00Z,53.516566,8.188217,None,9.14|2010-11-09T00:50:00Z,53.516566,8.188217,None,9.15|2010-11-09T01:00:00Z,53.516566,8.188217,None,9.19|2010-11-09T01:10:00Z,53.516566,8.188217,None,9.25|2010-11-09T01:20:00Z,53.516566,8.188217,None,9.3|2010-11-09T01:30:00Z,53.516566,8.188217,None,9.34|2010-11-09T01:40:00Z,53.516566,8.188217,None,9.35|2010-11-09T01:50:00Z,53.516566,8.188217,None,9.33|2010-11-09T02:00:00Z,53.516566,8.188217,None,9.21|2010-11-09T02:10:00Z,53.516566,8.188217,None,9.06|2010-11-09T02:20:00Z,53.516566,8.188217,None,8.98|2010-11-09T02:30:00Z,53.516566,8.188217,None,8.75|2010-11-09T02:40:00Z,53.516566,8.188217,None,8.61|2010-11-09T02:50:00Z,53.516566,8.188217,None,8.1|2010-11-09T03:00:00Z,53.516566,8.188217,None,7.9|2010-11-09T03:10:00Z,53.516566,8.188217,None,7.87|2010-11-09T03:20:00Z,53.516566,8.188217,None,7.87|2010-11-09T03:30:00Z,53.516566,8.188217,None,7.81|2010-11-09T03:40:00Z,53.516566,8.188217,None,7.74|2010-11-09T03:50:00Z,53.516566,8.188217,None,7.58|2010-11-09T04:00:00Z,53.516566,8.188217,None,7.48|2010-11-09T04:10:00Z,53.516566,8.188217,None,7.53|2010-11-09T04:20:00Z,53.516566,8.188217,None,7.61|2010-11-09T04:30:00Z,53.516566,8.188217,None,7.7|2010-11-09T04:40:00Z,53.516566,8.188217,None,7.74|2010-11-09T04:50:00Z,53.516566,8.188217,None,7.77|2010-11-09T05:00:00Z,53.516566,8.188217,None,8.04|2010-11-09T05:10:00Z,53.516566,8.188217,None,8.54|2010-11-09T05:20:00Z,53.516566,8.188217,None,8.56|2010-11-09T05:30:00Z,53.516566,8.188217,None,8.47|2010-11-09T05:40:00Z,53.516566,8.188217,None,8.34|2010-11-09T05:50:00Z,53.516566,8.188217,None,8.33|2010-11-09T06:00:00Z,53.516566,8.188217,None,8.21|2010-11-09T06:10:00Z,53.516566,8.188217,None,7.98|2010-11-09T06:20:00Z,53.516566,8.188217,None,8.29|2010-11-09T06:30:00Z,53.516566,8.188217,None,8.39|2010-11-09T06:40:00Z,53.516566,8.188217,None,8.43|2010-11-09T06:50:00Z,53.516566,8.188217,None,8.46|2010-11-09T07:00:00Z,53.516566,8.188217,None,8.5|2010-11-09T07:10:00Z,53.516566,8.188217,None,8.48|2010-11-09T07:20:00Z,53.516566,8.188217,None,8.51|2010-11-09T07:30:00Z,53.516566,8.188217,None,8.52|2010-11-09T07:40:00Z,53.516566,8.188217,None,8.53|2010-11-09T07:50:00Z,53.516566,8.188217,None,8.53|2010-11-09T08:00:00Z,53.516566,8.188217,None,8.49|2010-11-09T08:10:00Z,53.516566,8.188217,None,8.4|2010-11-09T08:20:00Z,53.516566,8.188217,None,8.41|2010-11-09T08:30:00Z,53.516566,8.188217,None,8.39|2010-11-09T08:40:00Z,53.516566,8.188217,None,8.17|2010-11-09T08:50:00Z,53.516566,8.188217,None,8.32|2010-11-09T09:00:00Z,53.516566,8.188217,None,8.29|2010-11-09T09:10:00Z,53.516566,8.188217,None,8.08|2010-11-09T09:20:00Z,53.516566,8.188217,None,7.92|2010-11-09T09:30:00Z,53.516566,8.188217,None,7.91|2010-11-09T09:40:00Z,53.516566,8.188217,None,7.98|2010-11-09T09:50:00Z,53.516566,8.188217,None,8.14|2010-11-09T10:00:00Z,53.516566,8.188217,None,8.31|2010-11-09T10:10:00Z,53.516566,8.188217,None,8.43|2010-11-09T10:20:00Z,53.516566,8.188217,None,8.42|2010-11-09T10:30:00Z,53.516566,8.188217,None,8.43|2010-11-09T10:40:00Z,53.516566,8.188217,None,8.51|2010-11-09T10:50:00Z,53.516566,8.188217,None,8.62|2010-11-09T11:00:00Z,53.516566,8.188217,None,8.67|2010-11-09T11:10:00Z,53.516566,8.188217,None,8.69|2010-11-09T11:20:00Z,53.516566,8.188217,None,8.73|2010-11-09T11:30:00Z,53.516566,8.188217,None,8.74|2010-11-09T11:40:00Z,53.516566,8.188217,None,8.77|2010-11-09T11:50:00Z,53.516566,8.188217,None,8.81|2010-11-09T12:00:00Z,53.516566,8.188217,None,8.82|2010-11-09T12:10:00Z,53.516566,8.188217,None,8.79|2010-11-09T12:20:00Z,53.516566,8.188217,None,8.78|2010-11-09T12:30:00Z,53.516566,8.188217,None,8.77|2010-11-09T12:40:00Z,53.516566,8.188217,None,8.79|2010-11-09T12:50:00Z,53.516566,8.188217,None,8.83|2010-11-09T13:00:00Z,53.516566,8.188217,None,8.85|2010-11-09T13:10:00Z,53.516566,8.188217,None,8.9|2010-11-09T13:20:00Z,53.516566,8.188217,None,8.97|2010-11-09T13:30:00Z,53.516566,8.188217,None,9.03|2010-11-09T13:40:00Z,53.516566,8.188217,None,9.06|2010-11-09T13:50:00Z,53.516566,8.188217,None,9.07|2010-11-09T14:00:00Z,53.516566,8.188217,None,9.09|2010-11-09T14:10:00Z,53.516566,8.188217,None,9.1|2010-11-09T14:20:00Z,53.516566,8.188217,None,9.08|2010-11-09T14:30:00Z,53.516566,8.188217,None,9.01|2010-11-09T14:40:00Z,53.516566,8.188217,None,8.87|2010-11-09T14:50:00Z,53.516566,8.188217,None,8.75|2010-11-09T15:00:00Z,53.516566,8.188217,None,8.7|2010-11-09T15:10:00Z,53.516566,8.188217,None,8.55|2010-11-09T15:20:00Z,53.516566,8.188217,None,8.25|2010-11-09T15:30:00Z,53.516566,8.188217,None,8.02|2010-11-09T15:40:00Z,53.516566,8.188217,None,7.82|2010-11-09T15:50:00Z,53.516566,8.188217,None,7.78|2010-11-09T16:00:00Z,53.516566,8.188217,None,7.77|2010-11-09T16:10:00Z,53.516566,8.188217,None,7.78|2010-11-09T16:20:00Z,53.516566,8.188217,None,7.85|2010-11-09T16:30:00Z,53.516566,8.188217,None,7.95|2010-11-09T16:40:00Z,53.516566,8.188217,None,7.95|2010-11-09T16:50:00Z,53.516566,8.188217,None,7.97|2010-11-09T17:00:00Z,53.516566,8.188217,None,8.02|2010-11-09T17:10:00Z,53.516566,8.188217,None,8.06|2010-11-09T17:20:00Z,53.516566,8.188217,None,8.09|2010-11-09T17:30:00Z,53.516566,8.188217,None,8.11|2010-11-09T17:40:00Z,53.516566,8.188217,None,8.26|2010-11-09T17:50:00Z,53.516566,8.188217,None,8.33|2010-11-09T18:00:00Z,53.516566,8.188217,None,8.3|2010-11-09T18:10:00Z,53.516566,8.188217,None,8.38|2010-11-09T18:20:00Z,53.516566,8.188217,None,8.46|2010-11-09T18:30:00Z,53.516566,8.188217,None,8.45|2010-11-09T18:40:00Z,53.516566,8.188217,None,8.44|2010-11-09T18:50:00Z,53.516566,8.188217,None,8.34|2010-11-09T19:00:00Z,53.516566,8.188217,None,8.27|2010-11-09T19:10:00Z,53.516566,8.188217,None,8.46|2010-11-09T19:20:00Z,53.516566,8.188217,None,8.47|2010-11-09T19:30:00Z,53.516566,8.188217,None,8.48|2010-11-09T19:40:00Z,53.516566,8.188217,None,8.48|2010-11-09T19:50:00Z,53.516566,8.188217,None,8.47|2010-11-09T20:00:00Z,53.516566,8.188217,None,8.51|2010-11-09T20:10:00Z,53.516566,8.188217,None,8.48|2010-11-09T20:20:00Z,53.516566,8.188217,None,8.33|2010-11-09T20:30:00Z,53.516566,8.188217,None,8.27|2010-11-09T20:40:00Z,53.516566,8.188217,None,8.16|2010-11-09T20:50:00Z,53.516566,8.188217,None,8.0|2010-11-09T21:00:00Z,53.516566,8.188217,None,8.01|2010-11-09T21:10:00Z,53.516566,8.188217,None,8.07|2010-11-09T21:20:00Z,53.516566,8.188217,None,8.03|2010-11-09T21:30:00Z,53.516566,8.188217,None,8.11|2010-11-09T21:40:00Z,53.516566,8.188217,None,7.91|2010-11-09T21:50:00Z,53.516566,8.188217,None,8.0|2010-11-09T22:00:00Z,53.516566,8.188217,None,8.08|2010-11-09T22:10:00Z,53.516566,8.188217,None,8.16|2010-11-09T22:20:00Z,53.516566,8.188217,None,8.26|2010-11-09T22:30:00Z,53.516566,8.188217,None,8.3|2010-11-09T22:40:00Z,53.516566,8.188217,None,8.38|2010-11-09T22:50:00Z,53.516566,8.188217,None,8.4|2010-11-09T23:00:00Z,53.516566,8.188217,None,8.47|2010-11-09T23:10:00Z,53.516566,8.188217,None,8.52|2010-11-09T23:20:00Z,53.516566,8.188217,None,8.55|2010-11-09T23:30:00Z,53.516566,8.188217,None,8.6|2010-11-09T23:40:00Z,53.516566,8.188217,None,8.67|2010-11-09T23:50:00Z,53.516566,8.188217,None,8.66|2010-11-10T00:00:00Z,53.516566,8.188217,None,8.61|2010-11-10T00:10:00Z,53.516566,8.188217,None,8.6|2010-11-10T00:20:00Z,53.516566,8.188217,None,8.68|2010-11-10T00:30:00Z,53.516566,8.188217,None,8.83|2010-11-10T00:40:00Z,53.516566,8.188217,None,9.08|2010-11-10T00:50:00Z,53.516566,8.188217,None,9.2|2010-11-10T01:00:00Z,53.516566,8.188217,None,9.28|2010-11-10T01:10:00Z,53.516566,8.188217,None,9.32|2010-11-10T01:20:00Z,53.516566,8.188217,None,9.35|2010-11-10T01:30:00Z,53.516566,8.188217,None,9.38|2010-11-10T01:40:00Z,53.516566,8.188217,None,9.39|2010-11-10T01:50:00Z,53.516566,8.188217,None,9.4|2010-11-10T02:00:00Z,53.516566,8.188217,None,9.37|2010-11-10T02:10:00Z,53.516566,8.188217,None,9.34|2010-11-10T02:20:00Z,53.516566,8.188217,None,9.34|2010-11-10T02:30:00Z,53.516566,8.188217,None,9.35|2010-11-10T02:40:00Z,53.516566,8.188217,None,9.33|2010-11-10T02:50:00Z,53.516566,8.188217,None,9.29|2010-11-10T03:00:00Z,53.516566,8.188217,None,9.11|2010-11-10T03:10:00Z,53.516566,8.188217,None,8.69|2010-11-10T03:20:00Z,53.516566,8.188217,None,8.51|2010-11-10T03:30:00Z,53.516566,8.188217,None,8.27|2010-11-10T03:40:00Z,53.516566,8.188217,None,8.09|2010-11-10T03:50:00Z,53.516566,8.188217,None,8.01|2010-11-10T04:00:00Z,53.516566,8.188217,None,8.06|2010-11-10T04:10:00Z,53.516566,8.188217,None,8.2|2010-11-10T04:20:00Z,53.516566,8.188217,None,8.29|2010-11-10T04:30:00Z,53.516566,8.188217,None,8.27|2010-11-10T04:40:00Z,53.516566,8.188217,None,8.26|2010-11-10T04:50:00Z,53.516566,8.188217,None,8.25|2010-11-10T05:00:00Z,53.516566,8.188217,None,8.25|2010-11-10T05:10:00Z,53.516566,8.188217,None,8.27|2010-11-10T05:20:00Z,53.516566,8.188217,None,8.22|2010-11-10T05:30:00Z,53.516566,8.188217,None,8.17|2010-11-10T05:40:00Z,53.516566,8.188217,None,8.16|2010-11-10T05:50:00Z,53.516566,8.188217,None,8.15|2010-11-10T06:00:00Z,53.516566,8.188217,None,8.15|2010-11-10T06:10:00Z,53.516566,8.188217,None,8.2|2010-11-10T06:20:00Z,53.516566,8.188217,None,8.18|2010-11-10T06:30:00Z,53.516566,8.188217,None,8.27|2010-11-10T06:40:00Z,53.516566,8.188217,None,8.32|2010-11-10T06:50:00Z,53.516566,8.188217,None,8.41|2010-11-10T07:00:00Z,53.516566,8.188217,None,8.34|2010-11-10T07:10:00Z,53.516566,8.188217,None,8.03|2010-11-10T07:20:00Z,53.516566,8.188217,None,8.07|2010-11-10T07:30:00Z,53.516566,8.188217,None,8.3|2010-11-10T07:40:00Z,53.516566,8.188217,None,8.37|2010-11-10T07:50:00Z,53.516566,8.188217,None,8.41|2010-11-10T08:00:00Z,53.516566,8.188217,None,8.46|2010-11-10T08:10:00Z,53.516566,8.188217,None,8.49|2010-11-10T08:20:00Z,53.516566,8.188217,None,8.34|2010-11-10T08:30:00Z,53.516566,8.188217,None,8.21|2010-11-10T08:40:00Z,53.516566,8.188217,None,8.13|2010-11-10T08:50:00Z,53.516566,8.188217,None,8.17|2010-11-10T09:00:00Z,53.516566,8.188217,None,8.27|2010-11-10T09:10:00Z,53.516566,8.188217,None,8.06|2010-11-10T09:20:00Z,53.516566,8.188217,None,8.03|2010-11-10T09:30:00Z,53.516566,8.188217,None,8.12|2010-11-10T09:40:00Z,53.516566,8.188217,None,8.24|2010-11-10T09:50:00Z,53.516566,8.188217,None,7.99|2010-11-10T10:00:00Z,53.516566,8.188217,None,7.78|2010-11-10T10:10:00Z,53.516566,8.188217,None,7.81|2010-11-10T10:20:00Z,53.516566,8.188217,None,7.9|2010-11-10T10:30:00Z,53.516566,8.188217,None,7.95|2010-11-10T10:40:00Z,53.516566,8.188217,None,7.97|2010-11-10T10:50:00Z,53.516566,8.188217,None,8.04|2010-11-10T11:00:00Z,53.516566,8.188217,None,8.13|2010-11-10T11:10:00Z,53.516566,8.188217,None,8.21|2010-11-10T11:20:00Z,53.516566,8.188217,None,8.26|2010-11-10T11:30:00Z,53.516566,8.188217,None,8.25|2010-11-10T11:40:00Z,53.516566,8.188217,None,8.26|2010-11-10T11:50:00Z,53.516566,8.188217,None,8.32|2010-11-10T12:00:00Z,53.516566,8.188217,None,8.37|2010-11-10T12:10:00Z,53.516566,8.188217,None,8.43|2010-11-10T12:20:00Z,53.516566,8.188217,None,8.48|2010-11-10T12:30:00Z,53.516566,8.188217,None,8.53|2010-11-10T12:40:00Z,53.516566,8.188217,None,8.58|2010-11-10T12:50:00Z,53.516566,8.188217,None,8.61|2010-11-10T13:00:00Z,53.516566,8.188217,None,8.61|2010-11-10T13:10:00Z,53.516566,8.188217,None,8.61|2010-11-10T13:20:00Z,53.516566,8.188217,None,8.63|2010-11-10T13:30:00Z,53.516566,8.188217,None,8.64|2010-11-10T13:40:00Z,53.516566,8.188217,None,8.66|2010-11-10T13:50:00Z,53.516566,8.188217,None,8.66|2010-11-10T14:00:00Z,53.516566,8.188217,None,8.71|2010-11-10T14:10:00Z,53.516566,8.188217,None,8.87|2010-11-10T14:20:00Z,53.516566,8.188217,None,9.01|2010-11-10T14:30:00Z,53.516566,8.188217,None,9.04|2010-11-10T14:40:00Z,53.516566,8.188217,None,9.06|2010-11-10T14:50:00Z,53.516566,8.188217,None,9.06|2010-11-10T15:00:00Z,53.516566,8.188217,None,9.04|2010-11-10T15:10:00Z,53.516566,8.188217,None,8.95|2010-11-10T15:20:00Z,53.516566,8.188217,None,8.75|2010-11-10T15:30:00Z,53.516566,8.188217,None,8.6|2010-11-10T15:40:00Z,53.516566,8.188217,None,8.55|2010-11-10T15:50:00Z,53.516566,8.188217,None,8.47|2010-11-10T16:00:00Z,53.516566,8.188217,None,8.31|2010-11-10T16:10:00Z,53.516566,8.188217,None,8.13|2010-11-10T16:20:00Z,53.516566,8.188217,None,7.97|2010-11-10T16:30:00Z,53.516566,8.188217,None,7.9|2010-11-10T16:40:00Z,53.516566,8.188217,None,7.91|2010-11-10T16:50:00Z,53.516566,8.188217,None,7.99|2010-11-10T17:00:00Z,53.516566,8.188217,None,8.09|2010-11-10T17:10:00Z,53.516566,8.188217,None,8.16|2010-11-10T17:20:00Z,53.516566,8.188217,None,8.16|2010-11-10T17:30:00Z,53.516566,8.188217,None,8.12|2010-11-10T17:40:00Z,53.516566,8.188217,None,8.11|2010-11-10T17:50:00Z,53.516566,8.188217,None,8.09|2010-11-10T18:00:00Z,53.516566,8.188217,None,8.06|2010-11-10T18:10:00Z,53.516566,8.188217,None,8.04|2010-11-10T18:20:00Z,53.516566,8.188217,None,8.03|2010-11-10T18:30:00Z,53.516566,8.188217,None,8.03|2010-11-10T18:40:00Z,53.516566,8.188217,None,8.02|2010-11-10T18:50:00Z,53.516566,8.188217,None,8.05|2010-11-10T19:00:00Z,53.516566,8.188217,None,8.14|2010-11-10T19:10:00Z,53.516566,8.188217,None,8.19|2010-11-10T19:20:00Z,53.516566,8.188217,None,8.24|2010-11-10T19:30:00Z,53.516566,8.188217,None,8.32|2010-11-10T19:40:00Z,53.516566,8.188217,None,8.16|2010-11-10T19:50:00Z,53.516566,8.188217,None,7.86|2010-11-10T20:00:00Z,53.516566,8.188217,None,7.86|2010-11-10T20:10:00Z,53.516566,8.188217,None,8.18|2010-11-10T20:20:00Z,53.516566,8.188217,None,8.33|2010-11-10T20:30:00Z,53.516566,8.188217,None,8.38|2010-11-10T20:40:00Z,53.516566,8.188217,None,8.4|2010-11-10T20:50:00Z,53.516566,8.188217,None,8.25|2010-11-10T21:00:00Z,53.516566,8.188217,None,8.05|2010-11-10T21:10:00Z,53.516566,8.188217,None,7.91|2010-11-10T21:20:00Z,53.516566,8.188217,None,7.77|2010-11-10T21:30:00Z,53.516566,8.188217,None,7.97|2010-11-10T21:40:00Z,53.516566,8.188217,None,7.95|2010-11-10T21:50:00Z,53.516566,8.188217,None,7.93|2010-11-10T22:00:00Z,53.516566,8.188217,None,7.76|2010-11-10T22:10:00Z,53.516566,8.188217,None,7.77|2010-11-10T22:20:00Z,53.516566,8.188217,None,7.81|2010-11-10T22:30:00Z,53.516566,8.188217,None,7.86|2010-11-10T22:40:00Z,53.516566,8.188217,None,7.92|2010-11-10T22:50:00Z,53.516566,8.188217,None,7.95|2010-11-10T23:00:00Z,53.516566,8.188217,None,8.02|2010-11-10T23:10:00Z,53.516566,8.188217,None,8.08|2010-11-10T23:20:00Z,53.516566,8.188217,None,8.16|2010-11-10T23:30:00Z,53.516566,8.188217,None,8.25|2010-11-10T23:40:00Z,53.516566,8.188217,None,8.25|2010-11-10T23:50:00Z,53.516566,8.188217,None,8.28|2010-11-11T00:00:00Z,53.516566,8.188217,None,8.32|2010-11-11T00:10:00Z,53.516566,8.188217,None,8.36|2010-11-11T00:20:00Z,53.516566,8.188217,None,8.37|2010-11-11T00:30:00Z,53.516566,8.188217,None,8.39|2010-11-11T00:40:00Z,53.516566,8.188217,None,8.44|2010-11-11T00:50:00Z,53.516566,8.188217,None,8.44|2010-11-11T01:00:00Z,53.516566,8.188217,None,8.47|2010-11-11T01:10:00Z,53.516566,8.188217,None,8.5|2010-11-11T01:20:00Z,53.516566,8.188217,None,8.58|2010-11-11T01:30:00Z,53.516566,8.188217,None,8.74|2010-11-11T01:40:00Z,53.516566,8.188217,None,8.79|2010-11-11T01:50:00Z,53.516566,8.188217,None,8.83|2010-11-11T02:00:00Z,53.516566,8.188217,None,8.9|2010-11-11T02:10:00Z,53.516566,8.188217,None,8.93|2010-11-11T02:20:00Z,53.516566,8.188217,None,9.01|2010-11-11T02:30:00Z,53.516566,8.188217,None,9.04|2010-11-11T02:40:00Z,53.516566,8.188217,None,9.03|2010-11-11T02:50:00Z,53.516566,8.188217,None,9.03|2010-11-11T03:00:00Z,53.516566,8.188217,None,9.0|2010-11-11T03:10:00Z,53.516566,8.188217,None,8.92|2010-11-11T03:20:00Z,53.516566,8.188217,None,8.78|2010-11-11T03:30:00Z,53.516566,8.188217,None,8.67|2010-11-11T03:40:00Z,53.516566,8.188217,None,8.43|2010-11-11T03:50:00Z,53.516566,8.188217,None,8.27|2010-11-11T04:00:00Z,53.516566,8.188217,None,8.13|2010-11-11T04:10:00Z,53.516566,8.188217,None,7.95|2010-11-11T04:20:00Z,53.516566,8.188217,None,7.82|2010-11-11T04:30:00Z,53.516566,8.188217,None,7.74|2010-11-11T04:40:00Z,53.516566,8.188217,None,7.67|2010-11-11T04:50:00Z,53.516566,8.188217,None,7.7|2010-11-11T05:00:00Z,53.516566,8.188217,None,7.91|2010-11-11T05:10:00Z,53.516566,8.188217,None,8.14|2010-11-11T05:20:00Z,53.516566,8.188217,None,8.24|2010-11-11T05:30:00Z,53.516566,8.188217,None,8.28|2010-11-11T05:40:00Z,53.516566,8.188217,None,8.28|2010-11-11T05:50:00Z,53.516566,8.188217,None,8.28|2010-11-11T06:00:00Z,53.516566,8.188217,None,8.25|2010-11-11T06:10:00Z,53.516566,8.188217,None,8.21|2010-11-11T06:20:00Z,53.516566,8.188217,None,8.15|2010-11-11T06:30:00Z,53.516566,8.188217,None,8.1|2010-11-11T06:40:00Z,53.516566,8.188217,None,8.07|2010-11-11T06:50:00Z,53.516566,8.188217,None,8.04|2010-11-11T07:00:00Z,53.516566,8.188217,None,8.03|2010-11-11T07:10:00Z,53.516566,8.188217,None,8.0|2010-11-11T07:20:00Z,53.516566,8.188217,None,7.97|2010-11-11T07:30:00Z,53.516566,8.188217,None,7.95|2010-11-11T07:40:00Z,53.516566,8.188217,None,7.95|2010-11-11T07:50:00Z,53.516566,8.188217,None,7.98|2010-11-11T08:00:00Z,53.516566,8.188217,None,7.98|2010-11-11T08:10:00Z,53.516566,8.188217,None,7.99|2010-11-11T08:20:00Z,53.516566,8.188217,None,8.0|2010-11-11T08:30:00Z,53.516566,8.188217,None,8.0|2010-11-11T08:40:00Z,53.516566,8.188217,None,7.99|2010-11-11T08:50:00Z,53.516566,8.188217,None,7.96|2010-11-11T09:00:00Z,53.516566,8.188217,None,7.92|2010-11-11T09:10:00Z,53.516566,8.188217,None,7.86|2010-11-11T09:20:00Z,53.516566,8.188217,None,7.91|2010-11-11T09:30:00Z,53.516566,8.188217,None,7.87|2010-11-11T09:40:00Z,53.516566,8.188217,None,7.73|2010-11-11T09:50:00Z,53.516566,8.188217,None,7.64|2010-11-11T10:00:00Z,53.516566,8.188217,None,7.59|2010-11-11T10:10:00Z,53.516566,8.188217,None,7.35|2010-11-11T10:20:00Z,53.516566,8.188217,None,7.33|2010-11-11T10:30:00Z,53.516566,8.188217,None,7.32|2010-11-11T10:40:00Z,53.516566,8.188217,None,7.41|2010-11-11T10:50:00Z,53.516566,8.188217,None,7.48|2010-11-11T11:00:00Z,53.516566,8.188217,None,7.53|2010-11-11T11:10:00Z,53.516566,8.188217,None,7.63|2010-11-11T11:20:00Z,53.516566,8.188217,None,7.62|2010-11-11T11:30:00Z,53.516566,8.188217,None,7.61|2010-11-11T11:40:00Z,53.516566,8.188217,None,7.7|2010-11-11T11:50:00Z,53.516566,8.188217,None,7.81|2010-11-11T12:00:00Z,53.516566,8.188217,None,7.83|2010-11-11T12:10:00Z,53.516566,8.188217,None,7.77|2010-11-11T12:20:00Z,53.516566,8.188217,None,7.77|2010-11-11T12:30:00Z,53.516566,8.188217,None,7.82|2010-11-11T12:40:00Z,53.516566,8.188217,None,7.9|2010-11-11T12:50:00Z,53.516566,8.188217,None,7.96|2010-11-11T13:00:00Z,53.516566,8.188217,None,7.99|2010-11-11T13:10:00Z,53.516566,8.188217,None,8.0|2010-11-11T13:20:00Z,53.516566,8.188217,None,8.0|2010-11-11T13:30:00Z,53.516566,8.188217,None,8.0|2010-11-11T13:40:00Z,53.516566,8.188217,None,8.0|2010-11-11T13:50:00Z,53.516566,8.188217,None,8.0|2010-11-11T14:00:00Z,53.516566,8.188217,None,8.0|2010-11-11T14:10:00Z,53.516566,8.188217,None,8.0|2010-11-11T14:20:00Z,53.516566,8.188217,None,8.0|2010-11-11T14:30:00Z,53.516566,8.188217,None,7.94|2010-11-11T14:40:00Z,53.516566,8.188217,None,7.73|2010-11-11T14:50:00Z,53.516566,8.188217,None,7.58|2010-11-11T15:00:00Z,53.516566,8.188217,None,7.43|2010-11-11T15:10:00Z,53.516566,8.188217,None,7.35|2010-11-11T15:20:00Z,53.516566,8.188217,None,7.26|2010-11-11T15:30:00Z,53.516566,8.188217,None,7.32|2010-11-11T15:40:00Z,53.516566,8.188217,None,7.46|2010-11-11T15:50:00Z,53.516566,8.188217,None,7.52|2010-11-11T16:00:00Z,53.516566,8.188217,None,7.63|2010-11-11T16:10:00Z,53.516566,8.188217,None,7.72|2010-11-11T16:20:00Z,53.516566,8.188217,None,7.82|2010-11-11T16:30:00Z,53.516566,8.188217,None,7.86|2010-11-11T16:40:00Z,53.516566,8.188217,None,7.86|2010-11-11T16:50:00Z,53.516566,8.188217,None,7.87|2010-11-11T17:00:00Z,53.516566,8.188217,None,7.87|2010-11-11T17:10:00Z,53.516566,8.188217,None,7.86|2010-11-11T17:20:00Z,53.516566,8.188217,None,7.86|2010-11-11T17:30:00Z,53.516566,8.188217,None,7.85|2010-11-11T17:40:00Z,53.516566,8.188217,None,7.81|2010-11-11T17:50:00Z,53.516566,8.188217,None,7.77|2010-11-11T18:00:00Z,53.516566,8.188217,None,7.74|2010-11-11T18:10:00Z,53.516566,8.188217,None,7.72|2010-11-11T18:20:00Z,53.516566,8.188217,None,7.72|2010-11-11T18:30:00Z,53.516566,8.188217,None,7.74|2010-11-11T18:40:00Z,53.516566,8.188217,None,7.73|2010-11-11T18:50:00Z,53.516566,8.188217,None,7.7|2010-11-11T19:00:00Z,53.516566,8.188217,None,7.7|2010-11-11T19:10:00Z,53.516566,8.188217,None,7.69|2010-11-11T19:20:00Z,53.516566,8.188217,None,7.68|2010-11-11T19:30:00Z,53.516566,8.188217,None,7.68|2010-11-11T19:40:00Z,53.516566,8.188217,None,7.69|2010-11-11T19:50:00Z,53.516566,8.188217,None,7.7|2010-11-11T20:00:00Z,53.516566,8.188217,None,7.7|2010-11-11T20:10:00Z,53.516566,8.188217,None,7.7|2010-11-11T20:20:00Z,53.516566,8.188217,None,7.69|2010-11-11T20:30:00Z,53.516566,8.188217,None,7.67|2010-11-11T20:40:00Z,53.516566,8.188217,None,7.66|2010-11-11T20:50:00Z,53.516566,8.188217,None,7.63|2010-11-11T21:00:00Z,53.516566,8.188217,None,7.62|2010-11-11T21:10:00Z,53.516566,8.188217,None,7.62|2010-11-11T21:20:00Z,53.516566,8.188217,None,7.61|2010-11-11T21:30:00Z,53.516566,8.188217,None,7.59|2010-11-11T21:40:00Z,53.516566,8.188217,None,7.28|2010-11-11T21:50:00Z,53.516566,8.188217,None,7.3|2010-11-11T22:00:00Z,53.516566,8.188217,None,7.39|2010-11-11T22:10:00Z,53.516566,8.188217,None,7.49|2010-11-11T22:20:00Z,53.516566,8.188217,None,7.53|2010-11-11T22:30:00Z,53.516566,8.188217,None,7.54|2010-11-11T22:40:00Z,53.516566,8.188217,None,7.53|2010-11-11T22:50:00Z,53.516566,8.188217,None,7.53|2010-11-11T23:00:00Z,53.516566,8.188217,None,7.52|2010-11-11T23:10:00Z,53.516566,8.188217,None,7.51|2010-11-11T23:20:00Z,53.516566,8.188217,None,7.41|2010-11-11T23:30:00Z,53.516566,8.188217,None,7.29|2010-11-11T23:40:00Z,53.516566,8.188217,None,7.29|2010-11-11T23:50:00Z,53.516566,8.188217,None,7.32|2010-11-12T00:00:00Z,53.516566,8.188217,None,7.43|2010-11-12T00:10:00Z,53.516566,8.188217,None,7.56|2010-11-12T00:20:00Z,53.516566,8.188217,None,7.6|2010-11-12T00:30:00Z,53.516566,8.188217,None,7.59|2010-11-12T00:40:00Z,53.516566,8.188217,None,7.62|2010-11-12T00:50:00Z,53.516566,8.188217,None,7.65|2010-11-12T01:00:00Z,53.516566,8.188217,None,7.64|2010-11-12T01:10:00Z,53.516566,8.188217,None,7.67|2010-11-12T01:20:00Z,53.516566,8.188217,None,7.74|2010-11-12T01:30:00Z,53.516566,8.188217,None,7.82|2010-11-12T01:40:00Z,53.516566,8.188217,None,7.84|2010-11-12T01:50:00Z,53.516566,8.188217,None,7.88|2010-11-12T02:00:00Z,53.516566,8.188217,None,8.07|2010-11-12T02:10:00Z,53.516566,8.188217,None,8.18|2010-11-12T02:20:00Z,53.516566,8.188217,None,8.3|2010-11-12T02:30:00Z,53.516566,8.188217,None,8.38|2010-11-12T02:40:00Z,53.516566,8.188217,None,8.38|2010-11-12T02:50:00Z,53.516566,8.188217,None,8.37|2010-11-12T03:00:00Z,53.516566,8.188217,None,8.37|2010-11-12T03:10:00Z,53.516566,8.188217,None,8.38|2010-11-12T03:20:00Z,53.516566,8.188217,None,8.35|2010-11-12T03:30:00Z,53.516566,8.188217,None,8.45|2010-11-12T03:40:00Z,53.516566,8.188217,None,8.48|2010-11-12T03:50:00Z,53.516566,8.188217,None,8.51|2010-11-12T04:00:00Z,53.516566,8.188217,None,8.52|2010-11-12T04:10:00Z,53.516566,8.188217,None,8.52|2010-11-12T04:20:00Z,53.516566,8.188217,None,8.51|2010-11-12T04:30:00Z,53.516566,8.188217,None,8.49|2010-11-12T04:40:00Z,53.516566,8.188217,None,8.48|2010-11-12T04:50:00Z,53.516566,8.188217,None,8.44|2010-11-12T05:00:00Z,53.516566,8.188217,None,8.33|2010-11-12T05:10:00Z,53.516566,8.188217,None,8.32|2010-11-12T05:20:00Z,53.516566,8.188217,None,8.16|2010-11-12T05:30:00Z,53.516566,8.188217,None,7.85|2010-11-12T05:40:00Z,53.516566,8.188217,None,7.69|2010-11-12T05:50:00Z,53.516566,8.188217,None,7.63|2010-11-12T06:00:00Z,53.516566,8.188217,None,7.7|2010-11-12T06:10:00Z,53.516566,8.188217,None,8.16|2010-11-12T06:20:00Z,53.516566,8.188217,None,8.38|2010-11-12T06:30:00Z,53.516566,8.188217,None,8.44|2010-11-12T06:40:00Z,53.516566,8.188217,None,8.47|2010-11-12T06:50:00Z,53.516566,8.188217,None,8.48|2010-11-12T07:00:00Z,53.516566,8.188217,None,8.5|2010-11-12T07:10:00Z,53.516566,8.188217,None,8.5|2010-11-12T07:20:00Z,53.516566,8.188217,None,8.52|2010-11-12T07:30:00Z,53.516566,8.188217,None,8.53|2010-11-12T07:40:00Z,53.516566,8.188217,None,8.42|2010-11-12T07:50:00Z,53.516566,8.188217,None,8.39|2010-11-12T08:00:00Z,53.516566,8.188217,None,8.49|2010-11-12T08:10:00Z,53.516566,8.188217,None,8.35|2010-11-12T08:20:00Z,53.516566,8.188217,None,8.26|2010-11-12T08:30:00Z,53.516566,8.188217,None,8.23|2010-11-12T08:40:00Z,53.516566,8.188217,None,8.24|2010-11-12T08:50:00Z,53.516566,8.188217,None,8.26|2010-11-12T09:00:00Z,53.516566,8.188217,None,8.27|2010-11-12T09:10:00Z,53.516566,8.188217,None,8.24|2010-11-12T09:20:00Z,53.516566,8.188217,None,8.22|2010-11-12T09:30:00Z,53.516566,8.188217,None,8.2|2010-11-12T09:40:00Z,53.516566,8.188217,None,8.17|2010-11-12T09:50:00Z,53.516566,8.188217,None,8.14|2010-11-12T10:00:00Z,53.516566,8.188217,None,8.17|2010-11-12T10:10:00Z,53.516566,8.188217,None,8.41|2010-11-12T10:20:00Z,53.516566,8.188217,None,8.43|2010-11-12T10:30:00Z,53.516566,8.188217,None,8.21|2010-11-12T10:40:00Z,53.516566,8.188217,None,7.96|2010-11-12T10:50:00Z,53.516566,8.188217,None,8.08|2010-11-12T11:00:00Z,53.516566,8.188217,None,7.94|2010-11-12T11:10:00Z,53.516566,8.188217,None,7.75|2010-11-12T11:20:00Z,53.516566,8.188217,None,7.65|2010-11-12T11:30:00Z,53.516566,8.188217,None,7.48|2010-11-12T11:40:00Z,53.516566,8.188217,None,7.28|2010-11-12T11:50:00Z,53.516566,8.188217,None,7.19|2010-11-12T12:00:00Z,53.516566,8.188217,None,7.23|2010-11-12T12:10:00Z,53.516566,8.188217,None,7.42|2010-11-12T12:20:00Z,53.516566,8.188217,None,7.56|2010-11-12T12:30:00Z,53.516566,8.188217,None,7.63|2010-11-12T12:40:00Z,53.516566,8.188217,None,7.79|2010-11-12T12:50:00Z,53.516566,8.188217,None,7.93|2010-11-12T13:00:00Z,53.516566,8.188217,None,8.05|2010-11-12T13:10:00Z,53.516566,8.188217,None,8.16|2010-11-12T13:20:00Z,53.516566,8.188217,None,8.32|2010-11-12T13:30:00Z,53.516566,8.188217,None,8.43|2010-11-12T13:40:00Z,53.516566,8.188217,None,8.49|2010-11-12T13:50:00Z,53.516566,8.188217,None,8.53|2010-11-12T14:00:00Z,53.516566,8.188217,None,8.55|2010-11-12T14:10:00Z,53.516566,8.188217,None,8.54|2010-11-12T14:20:00Z,53.516566,8.188217,None,8.54|2010-11-12T14:30:00Z,53.516566,8.188217,None,8.71|2010-11-12T14:40:00Z,53.516566,8.188217,None,8.71|2010-11-12T14:50:00Z,53.516566,8.188217,None,8.75|2010-11-12T15:00:00Z,53.516566,8.188217,None,8.75|2010-11-12T15:10:00Z,53.516566,8.188217,None,8.74|2010-11-12T15:20:00Z,53.516566,8.188217,None,8.72|2010-11-12T15:30:00Z,53.516566,8.188217,None,8.7|2010-11-12T15:40:00Z,53.516566,8.188217,None,8.7|2010-11-12T15:50:00Z,53.516566,8.188217,None,8.72|2010-11-12T16:00:00Z,53.516566,8.188217,None,8.73|2010-11-12T16:10:00Z,53.516566,8.188217,None,8.71|2010-11-12T16:20:00Z,53.516566,8.188217,None,8.65|2010-11-12T16:30:00Z,53.516566,8.188217,None,8.52|2010-11-12T16:40:00Z,53.516566,8.188217,None,8.48|2010-11-12T16:50:00Z,53.516566,8.188217,None,8.37|2010-11-12T17:00:00Z,53.516566,8.188217,None,8.14|2010-11-12T17:10:00Z,53.516566,8.188217,None,7.94|2010-11-12T17:20:00Z,53.516566,8.188217,None,7.97|2010-11-12T17:30:00Z,53.516566,8.188217,None,7.91|2010-11-12T17:40:00Z,53.516566,8.188217,None,7.87|2010-11-12T17:50:00Z,53.516566,8.188217,None,8.11|2010-11-12T18:00:00Z,53.516566,8.188217,None,8.39|2010-11-12T18:10:00Z,53.516566,8.188217,None,8.52|2010-11-12T18:20:00Z,53.516566,8.188217,None,8.57|2010-11-12T18:30:00Z,53.516566,8.188217,None,8.55|2010-11-12T18:40:00Z,53.516566,8.188217,None,8.5|2010-11-12T18:50:00Z,53.516566,8.188217,None,8.5|2010-11-12T19:00:00Z,53.516566,8.188217,None,8.58|2010-11-12T19:10:00Z,53.516566,8.188217,None,8.58|2010-11-12T19:20:00Z,53.516566,8.188217,None,8.51|2010-11-12T19:30:00Z,53.516566,8.188217,None,8.44|2010-11-12T19:40:00Z,53.516566,8.188217,None,8.42|2010-11-12T19:50:00Z,53.516566,8.188217,None,8.37|2010-11-12T20:00:00Z,53.516566,8.188217,None,8.35|2010-11-12T20:10:00Z,53.516566,8.188217,None,8.37|2010-11-12T20:20:00Z,53.516566,8.188217,None,8.35|2010-11-12T20:30:00Z,53.516566,8.188217,None,8.31|2010-11-12T20:40:00Z,53.516566,8.188217,None,8.27|2010-11-12T20:50:00Z,53.516566,8.188217,None,8.23|2010-11-12T21:00:00Z,53.516566,8.188217,None,8.2|2010-11-12T21:10:00Z,53.516566,8.188217,None,8.18|2010-11-12T21:20:00Z,53.516566,8.188217,None,8.17|2010-11-12T21:30:00Z,53.516566,8.188217,None,8.15|2010-11-12T21:40:00Z,53.516566,8.188217,None,8.14|2010-11-12T21:50:00Z,53.516566,8.188217,None,8.13|2010-11-12T22:00:00Z,53.516566,8.188217,None,8.14|2010-11-12T22:10:00Z,53.516566,8.188217,None,8.16|2010-11-12T22:20:00Z,53.516566,8.188217,None,8.17|2010-11-12T22:30:00Z,53.516566,8.188217,None,8.17|2010-11-12T22:40:00Z,53.516566,8.188217,None,8.16|2010-11-12T22:50:00Z,53.516566,8.188217,None,8.13|2010-11-12T23:00:00Z,53.516566,8.188217,None,8.07|2010-11-12T23:10:00Z,53.516566,8.188217,None,8.07|2010-11-12T23:20:00Z,53.516566,8.188217,None,8.09|2010-11-12T23:30:00Z,53.516566,8.188217,None,8.21|2010-11-12T23:40:00Z,53.516566,8.188217,None,8.13|2010-11-12T23:50:00Z,53.516566,8.188217,None,8.14|2010-11-13T00:00:00Z,53.516566,8.188217,None,8.21|2010-11-13T00:10:00Z,53.516566,8.188217,None,8.24|2010-11-13T00:20:00Z,53.516566,8.188217,None,8.18|2010-11-13T00:30:00Z,53.516566,8.188217,None,8.19|2010-11-13T00:40:00Z,53.516566,8.188217,None,8.17|2010-11-13T00:50:00Z,53.516566,8.188217,None,8.15|2010-11-13T01:00:00Z,53.516566,8.188217,None,8.19|2010-11-13T01:10:00Z,53.516566,8.188217,None,8.2|2010-11-13T01:20:00Z,53.516566,8.188217,None,8.22|2010-11-13T01:30:00Z,53.516566,8.188217,None,8.32|2010-11-13T01:40:00Z,53.516566,8.188217,None,8.37|2010-11-13T01:50:00Z,53.516566,8.188217,None,8.38|2010-11-13T02:00:00Z,53.516566,8.188217,None,8.32|2010-11-13T02:10:00Z,53.516566,8.188217,None,8.44|2010-11-13T02:20:00Z,53.516566,8.188217,None,8.51|2010-11-13T02:30:00Z,53.516566,8.188217,None,8.51|2010-11-13T02:40:00Z,53.516566,8.188217,None,8.48|2010-11-13T02:50:00Z,53.516566,8.188217,None,8.46|2010-11-13T03:00:00Z,53.516566,8.188217,None,8.43|2010-11-13T03:10:00Z,53.516566,8.188217,None,8.42|2010-11-13T03:20:00Z,53.516566,8.188217,None,8.4|2010-11-13T03:30:00Z,53.516566,8.188217,None,8.39|2010-11-13T03:40:00Z,53.516566,8.188217,None,8.38|2010-11-13T03:50:00Z,53.516566,8.188217,None,8.38|2010-11-13T04:00:00Z,53.516566,8.188217,None,8.42|2010-11-13T04:10:00Z,53.516566,8.188217,None,8.48|2010-11-13T04:20:00Z,53.516566,8.188217,None,8.44|2010-11-13T04:30:00Z,53.516566,8.188217,None,8.33|2010-11-13T04:40:00Z,53.516566,8.188217,None,8.2|2010-11-13T04:50:00Z,53.516566,8.188217,None,8.09|2010-11-13T05:00:00Z,53.516566,8.188217,None,8.09|2010-11-13T05:10:00Z,53.516566,8.188217,None,8.11|2010-11-13T05:20:00Z,53.516566,8.188217,None,8.15|2010-11-13T05:30:00Z,53.516566,8.188217,None,8.1|2010-11-13T05:40:00Z,53.516566,8.188217,None,8.11|2010-11-13T05:50:00Z,53.516566,8.188217,None,8.14|2010-11-13T06:00:00Z,53.516566,8.188217,None,8.25|2010-11-13T06:10:00Z,53.516566,8.188217,None,8.14|2010-11-13T06:20:00Z,53.516566,8.188217,None,8.38|2010-11-13T06:30:00Z,53.516566,8.188217,None,8.41|2010-11-13T06:40:00Z,53.516566,8.188217,None,8.42|2010-11-13T06:50:00Z,53.516566,8.188217,None,8.42|2010-11-13T07:00:00Z,53.516566,8.188217,None,8.4|2010-11-13T07:10:00Z,53.516566,8.188217,None,8.37|2010-11-13T07:20:00Z,53.516566,8.188217,None,8.35|2010-11-13T07:30:00Z,53.516566,8.188217,None,8.35|2010-11-13T07:40:00Z,53.516566,8.188217,None,8.34|2010-11-13T07:50:00Z,53.516566,8.188217,None,8.33|2010-11-13T08:00:00Z,53.516566,8.188217,None,8.3|2010-11-13T08:10:00Z,53.516566,8.188217,None,8.22|2010-11-13T08:20:00Z,53.516566,8.188217,None,8.13|2010-11-13T08:30:00Z,53.516566,8.188217,None,8.11|2010-11-13T08:40:00Z,53.516566,8.188217,None,8.12|2010-11-13T08:50:00Z,53.516566,8.188217,None,8.15|2010-11-13T09:00:00Z,53.516566,8.188217,None,8.18|2010-11-13T09:10:00Z,53.516566,8.188217,None,8.2|2010-11-13T09:20:00Z,53.516566,8.188217,None,8.23|2010-11-13T09:30:00Z,53.516566,8.188217,None,8.26|2010-11-13T09:40:00Z,53.516566,8.188217,None,8.28|2010-11-13T09:50:00Z,53.516566,8.188217,None,8.28|2010-11-13T10:00:00Z,53.516566,8.188217,None,8.29|2010-11-13T10:10:00Z,53.516566,8.188217,None,8.29|2010-11-13T10:20:00Z,53.516566,8.188217,None,8.28|2010-11-13T10:30:00Z,53.516566,8.188217,None,8.19|2010-11-13T10:40:00Z,53.516566,8.188217,None,8.14|2010-11-13T10:50:00Z,53.516566,8.188217,None,8.19|2010-11-13T11:00:00Z,53.516566,8.188217,None,8.18|2010-11-13T11:10:00Z,53.516566,8.188217,None,8.09|2010-11-13T11:20:00Z,53.516566,8.188217,None,8.11|2010-11-13T11:30:00Z,53.516566,8.188217,None,8.07|2010-11-13T11:40:00Z,53.516566,8.188217,None,8.11|2010-11-13T11:50:00Z,53.516566,8.188217,None,8.15|2010-11-13T12:00:00Z,53.516566,8.188217,None,8.07|2010-11-13T12:10:00Z,53.516566,8.188217,None,8.05|2010-11-13T12:20:00Z,53.516566,8.188217,None,8.15|2010-11-13T12:30:00Z,53.516566,8.188217,None,8.17|2010-11-13T12:40:00Z,53.516566,8.188217,None,8.15|2010-11-13T12:50:00Z,53.516566,8.188217,None,8.02|2010-11-13T13:00:00Z,53.516566,8.188217,None,8.04|2010-11-13T13:10:00Z,53.516566,8.188217,None,8.05|2010-11-13T13:20:00Z,53.516566,8.188217,None,8.03|2010-11-13T13:30:00Z,53.516566,8.188217,None,8.0|2010-11-13T13:40:00Z,53.516566,8.188217,None,8.01|2010-11-13T13:50:00Z,53.516566,8.188217,None,8.02|2010-11-13T14:00:00Z,53.516566,8.188217,None,8.05|2010-11-13T14:10:00Z,53.516566,8.188217,None,8.12|2010-11-13T14:20:00Z,53.516566,8.188217,None,8.13|2010-11-13T14:30:00Z,53.516566,8.188217,None,8.1|2010-11-13T14:40:00Z,53.516566,8.188217,None,8.07|2010-11-13T14:50:00Z,53.516566,8.188217,None,8.12|2010-11-13T15:00:00Z,53.516566,8.188217,None,8.17|2010-11-13T15:10:00Z,53.516566,8.188217,None,8.23|2010-11-13T15:20:00Z,53.516566,8.188217,None,8.26|2010-11-13T15:30:00Z,53.516566,8.188217,None,8.28|2010-11-13T15:40:00Z,53.516566,8.188217,None,8.29|2010-11-13T15:50:00Z,53.516566,8.188217,None,8.29|2010-11-13T16:00:00Z,53.516566,8.188217,None,8.29|2010-11-13T16:10:00Z,53.516566,8.188217,None,8.29|2010-11-13T16:20:00Z,53.516566,8.188217,None,8.28|2010-11-13T16:30:00Z,53.516566,8.188217,None,8.27|2010-11-13T16:40:00Z,53.516566,8.188217,None,8.23|2010-11-13T16:50:00Z,53.516566,8.188217,None,8.18|2010-11-13T17:00:00Z,53.516566,8.188217,None,8.09|2010-11-13T17:10:00Z,53.516566,8.188217,None,8.02|2010-11-13T17:20:00Z,53.516566,8.188217,None,8.0|2010-11-13T17:30:00Z,53.516566,8.188217,None,8.04|2010-11-13T17:40:00Z,53.516566,8.188217,None,8.09|2010-11-13T17:50:00Z,53.516566,8.188217,None,8.15|2010-11-13T18:00:00Z,53.516566,8.188217,None,8.11|2010-11-13T18:10:00Z,53.516566,8.188217,None,8.07|2010-11-13T18:20:00Z,53.516566,8.188217,None,8.06|2010-11-13T18:30:00Z,53.516566,8.188217,None,8.08|2010-11-13T18:40:00Z,53.516566,8.188217,None,8.09|2010-11-13T18:50:00Z,53.516566,8.188217,None,8.1|2010-11-13T19:00:00Z,53.516566,8.188217,None,8.1|2010-11-13T19:10:00Z,53.516566,8.188217,None,8.1|2010-11-13T19:20:00Z,53.516566,8.188217,None,8.11|2010-11-13T19:30:00Z,53.516566,8.188217,None,8.12|2010-11-13T19:40:00Z,53.516566,8.188217,None,8.13|2010-11-13T19:50:00Z,53.516566,8.188217,None,8.12|2010-11-13T20:00:00Z,53.516566,8.188217,None,8.12|2010-11-13T20:10:00Z,53.516566,8.188217,None,8.11|2010-11-13T20:20:00Z,53.516566,8.188217,None,8.1|2010-11-13T20:30:00Z,53.516566,8.188217,None,8.1|2010-11-13T20:40:00Z,53.516566,8.188217,None,8.1|2010-11-13T20:50:00Z,53.516566,8.188217,None,8.09|2010-11-13T21:00:00Z,53.516566,8.188217,None,8.1|2010-11-13T21:10:00Z,53.516566,8.188217,None,8.09|2010-11-13T21:20:00Z,53.516566,8.188217,None,8.09|2010-11-13T21:30:00Z,53.516566,8.188217,None,8.09|2010-11-13T21:40:00Z,53.516566,8.188217,None,8.09|2010-11-13T21:50:00Z,53.516566,8.188217,None,8.08|2010-11-13T22:00:00Z,53.516566,8.188217,None,8.06|2010-11-13T22:10:00Z,53.516566,8.188217,None,8.05|2010-11-13T22:20:00Z,53.516566,8.188217,None,8.03|2010-11-13T22:30:00Z,53.516566,8.188217,None,8.04|2010-11-13T22:40:00Z,53.516566,8.188217,None,8.05|2010-11-13T22:50:00Z,53.516566,8.188217,None,8.06|2010-11-13T23:00:00Z,53.516566,8.188217,None,8.06|2010-11-13T23:10:00Z,53.516566,8.188217,None,8.06|2010-11-13T23:20:00Z,53.516566,8.188217,None,8.03|2010-11-13T23:30:00Z,53.516566,8.188217,None,8.02|2010-11-13T23:40:00Z,53.516566,8.188217,None,8.09|2010-11-13T23:50:00Z,53.516566,8.188217,None,8.1|2010-11-14T00:00:00Z,53.516566,8.188217,None,8.11|2010-11-14T00:10:00Z,53.516566,8.188217,None,8.08|2010-11-14T00:20:00Z,53.516566,8.188217,None,8.08|2010-11-14T00:30:00Z,53.516566,8.188217,None,8.05|2010-11-14T00:40:00Z,53.516566,8.188217,None,8.06|2010-11-14T00:50:00Z,53.516566,8.188217,None,8.06|2010-11-14T01:00:00Z,53.516566,8.188217,None,8.04|2010-11-14T01:10:00Z,53.516566,8.188217,None,8.04|2010-11-14T01:20:00Z,53.516566,8.188217,None,8.04|2010-11-14T01:30:00Z,53.516566,8.188217,None,8.04|2010-11-14T01:40:00Z,53.516566,8.188217,None,8.05|2010-11-14T01:50:00Z,53.516566,8.188217,None,8.09|2010-11-14T02:00:00Z,53.516566,8.188217,None,8.09|2010-11-14T02:10:00Z,53.516566,8.188217,None,8.07|2010-11-14T02:20:00Z,53.516566,8.188217,None,8.1|2010-11-14T02:30:00Z,53.516566,8.188217,None,8.14|2010-11-14T02:40:00Z,53.516566,8.188217,None,8.15|2010-11-14T02:50:00Z,53.516566,8.188217,None,8.21|2010-11-14T03:00:00Z,53.516566,8.188217,None,8.24|2010-11-14T03:10:00Z,53.516566,8.188217,None,8.29|2010-11-14T03:20:00Z,53.516566,8.188217,None,8.32|2010-11-14T03:30:00Z,53.516566,8.188217,None,8.32|2010-11-14T03:40:00Z,53.516566,8.188217,None,8.29|2010-11-14T03:50:00Z,53.516566,8.188217,None,8.27|2010-11-14T04:00:00Z,53.516566,8.188217,None,8.3|2010-11-14T04:10:00Z,53.516566,8.188217,None,8.33|2010-11-14T04:20:00Z,53.516566,8.188217,None,8.35|2010-11-14T04:30:00Z,53.516566,8.188217,None,8.35|2010-11-14T04:40:00Z,53.516566,8.188217,None,8.35|2010-11-14T04:50:00Z,53.516566,8.188217,None,8.35|2010-11-14T05:00:00Z,53.516566,8.188217,None,8.33|2010-11-14T05:10:00Z,53.516566,8.188217,None,8.32|2010-11-14T05:20:00Z,53.516566,8.188217,None,8.3|2010-11-14T05:30:00Z,53.516566,8.188217,None,8.29|2010-11-14T05:40:00Z,53.516566,8.188217,None,8.2|2010-11-14T05:50:00Z,53.516566,8.188217,None,8.07|2010-11-14T06:00:00Z,53.516566,8.188217,None,8.04|2010-11-14T06:10:00Z,53.516566,8.188217,None,8.02|2010-11-14T06:20:00Z,53.516566,8.188217,None,7.96|2010-11-14T06:30:00Z,53.516566,8.188217,None,7.93|2010-11-14T06:40:00Z,53.516566,8.188217,None,7.95|2010-11-14T06:50:00Z,53.516566,8.188217,None,7.98|2010-11-14T07:00:00Z,53.516566,8.188217,None,8.04|2010-11-14T07:10:00Z,53.516566,8.188217,None,8.08|2010-11-14T07:20:00Z,53.516566,8.188217,None,8.1|2010-11-14T07:30:00Z,53.516566,8.188217,None,8.1|2010-11-14T07:40:00Z,53.516566,8.188217,None,8.11|2010-11-14T07:50:00Z,53.516566,8.188217,None,8.11|2010-11-14T08:00:00Z,53.516566,8.188217,None,8.11|2010-11-14T08:10:00Z,53.516566,8.188217,None,8.12|2010-11-14T08:20:00Z,53.516566,8.188217,None,8.15|2010-11-14T08:30:00Z,53.516566,8.188217,None,8.16|2010-11-14T08:40:00Z,53.516566,8.188217,None,8.17|2010-11-14T08:50:00Z,53.516566,8.188217,None,8.18|2010-11-14T09:00:00Z,53.516566,8.188217,None,8.17|2010-11-14T09:10:00Z,53.516566,8.188217,None,8.17|2010-11-14T09:20:00Z,53.516566,8.188217,None,8.17|2010-11-14T09:30:00Z,53.516566,8.188217,None,8.18|2010-11-14T09:40:00Z,53.516566,8.188217,None,8.18|2010-11-14T09:50:00Z,53.516566,8.188217,None,8.19|2010-11-14T10:00:00Z,53.516566,8.188217,None,8.2|2010-11-14T10:10:00Z,53.516566,8.188217,None,8.22|2010-11-14T10:20:00Z,53.516566,8.188217,None,8.23|2010-11-14T10:30:00Z,53.516566,8.188217,None,8.22|2010-11-14T10:40:00Z,53.516566,8.188217,None,8.22|2010-11-14T10:50:00Z,53.516566,8.188217,None,8.25|2010-11-14T11:00:00Z,53.516566,8.188217,None,8.23|2010-11-14T11:10:00Z,53.516566,8.188217,None,8.2|2010-11-14T11:20:00Z,53.516566,8.188217,None,8.16|2010-11-14T11:30:00Z,53.516566,8.188217,None,8.19|2010-11-14T11:40:00Z,53.516566,8.188217,None,8.19|2010-11-14T11:50:00Z,53.516566,8.188217,None,8.07|2010-11-14T12:00:00Z,53.516566,8.188217,None,8.07|2010-11-14T12:10:00Z,53.516566,8.188217,None,8.06|2010-11-14T12:20:00Z,53.516566,8.188217,None,8.07|2010-11-14T12:30:00Z,53.516566,8.188217,None,8.09|2010-11-14T12:40:00Z,53.516566,8.188217,None,8.11|2010-11-14T12:50:00Z,53.516566,8.188217,None,8.11|2010-11-14T13:00:00Z,53.516566,8.188217,None,8.06|2010-11-14T13:10:00Z,53.516566,8.188217,None,8.04|2010-11-14T13:20:00Z,53.516566,8.188217,None,8.12|2010-11-14T13:30:00Z,53.516566,8.188217,None,8.09|2010-11-14T13:40:00Z,53.516566,8.188217,None,8.0|2010-11-14T13:50:00Z,53.516566,8.188217,None,8.0|2010-11-14T14:00:00Z,53.516566,8.188217,None,7.98|2010-11-14T14:10:00Z,53.516566,8.188217,None,7.96|2010-11-14T14:20:00Z,53.516566,8.188217,None,7.98|2010-11-14T14:30:00Z,53.516566,8.188217,None,8.01|2010-11-14T14:40:00Z,53.516566,8.188217,None,8.02|2010-11-14T14:50:00Z,53.516566,8.188217,None,8.03|2010-11-14T15:00:00Z,53.516566,8.188217,None,8.1|2010-11-14T15:10:00Z,53.516566,8.188217,None,8.15|2010-11-14T15:20:00Z,53.516566,8.188217,None,8.16|2010-11-14T15:30:00Z,53.516566,8.188217,None,8.16|2010-11-14T15:40:00Z,53.516566,8.188217,None,8.21|2010-11-14T15:50:00Z,53.516566,8.188217,None,8.24|2010-11-14T16:00:00Z,53.516566,8.188217,None,8.26|2010-11-14T16:10:00Z,53.516566,8.188217,None,8.27|2010-11-14T16:20:00Z,53.516566,8.188217,None,8.27|2010-11-14T16:30:00Z,53.516566,8.188217,None,8.26|2010-11-14T16:40:00Z,53.516566,8.188217,None,8.24|2010-11-14T16:50:00Z,53.516566,8.188217,None,8.23|2010-11-14T17:00:00Z,53.516566,8.188217,None,8.23|2010-11-14T17:10:00Z,53.516566,8.188217,None,8.23|2010-11-14T17:20:00Z,53.516566,8.188217,None,8.22|2010-11-14T17:30:00Z,53.516566,8.188217,None,8.22|2010-11-14T17:40:00Z,53.516566,8.188217,None,8.24|2010-11-14T17:50:00Z,53.516566,8.188217,None,8.26|2010-11-14T18:00:00Z,53.516566,8.188217,None,8.26|2010-11-14T18:10:00Z,53.516566,8.188217,None,8.19|2010-11-14T18:20:00Z,53.516566,8.188217,None,8.08|2010-11-14T18:30:00Z,53.516566,8.188217,None,8.04|2010-11-14T18:40:00Z,53.516566,8.188217,None,8.06|2010-11-14T18:50:00Z,53.516566,8.188217,None,8.13|2010-11-14T19:00:00Z,53.516566,8.188217,None,8.17|2010-11-14T19:10:00Z,53.516566,8.188217,None,8.19|2010-11-14T19:20:00Z,53.516566,8.188217,None,8.22|2010-11-14T19:30:00Z,53.516566,8.188217,None,8.19|2010-11-14T19:40:00Z,53.516566,8.188217,None,8.14|2010-11-14T19:50:00Z,53.516566,8.188217,None,8.11|2010-11-14T20:00:00Z,53.516566,8.188217,None,8.12|2010-11-14T20:10:00Z,53.516566,8.188217,None,8.13|2010-11-14T20:20:00Z,53.516566,8.188217,None,8.14|2010-11-14T20:30:00Z,53.516566,8.188217,None,8.14|2010-11-14T20:40:00Z,53.516566,8.188217,None,8.16|2010-11-14T20:50:00Z,53.516566,8.188217,None,8.16|2010-11-14T21:00:00Z,53.516566,8.188217,None,8.16|2010-11-14T21:10:00Z,53.516566,8.188217,None,8.15|2010-11-14T21:20:00Z,53.516566,8.188217,None,8.17|2010-11-14T21:30:00Z,53.516566,8.188217,None,8.18|2010-11-14T21:40:00Z,53.516566,8.188217,None,8.18|2010-11-14T21:50:00Z,53.516566,8.188217,None,8.17|2010-11-14T22:00:00Z,53.516566,8.188217,None,8.18|2010-11-14T22:10:00Z,53.516566,8.188217,None,8.19|2010-11-14T22:20:00Z,53.516566,8.188217,None,8.21|2010-11-14T22:30:00Z,53.516566,8.188217,None,8.19|2010-11-14T22:40:00Z,53.516566,8.188217,None,8.19|2010-11-14T22:50:00Z,53.516566,8.188217,None,8.2|2010-11-14T23:00:00Z,53.516566,8.188217,None,8.21|2010-11-14T23:10:00Z,53.516566,8.188217,None,8.21|2010-11-14T23:20:00Z,53.516566,8.188217,None,8.21|2010-11-14T23:30:00Z,53.516566,8.188217,None,8.21|2010-11-14T23:40:00Z,53.516566,8.188217,None,8.21|2010-11-14T23:50:00Z,53.516566,8.188217,None,8.22|2010-11-15T00:00:00Z,53.516566,8.188217,None,8.23|2010-11-15T00:10:00Z,53.516566,8.188217,None,8.27|2010-11-15T00:20:00Z,53.516566,8.188217,None,8.28|2010-11-15T00:30:00Z,53.516566,8.188217,None,8.33|2010-11-15T00:40:00Z,53.516566,8.188217,None,8.32|2010-11-15T00:50:00Z,53.516566,8.188217,None,8.3|2010-11-15T01:00:00Z,53.516566,8.188217,None,8.29|2010-11-15T01:10:00Z,53.516566,8.188217,None,8.27|2010-11-15T01:20:00Z,53.516566,8.188217,None,8.21|2010-11-15T01:30:00Z,53.516566,8.188217,None,8.21|2010-11-15T01:40:00Z,53.516566,8.188217,None,8.23|2010-11-15T01:50:00Z,53.516566,8.188217,None,8.16|2010-11-15T02:00:00Z,53.516566,8.188217,None,8.11|2010-11-15T02:10:00Z,53.516566,8.188217,None,8.16|2010-11-15T02:20:00Z,53.516566,8.188217,None,8.14|2010-11-15T02:30:00Z,53.516566,8.188217,None,8.1|2010-11-15T02:40:00Z,53.516566,8.188217,None,8.1|2010-11-15T02:50:00Z,53.516566,8.188217,None,8.11|2010-11-15T03:00:00Z,53.516566,8.188217,None,8.14|2010-11-15T03:10:00Z,53.516566,8.188217,None,8.18|2010-11-15T03:20:00Z,53.516566,8.188217,None,8.2|2010-11-15T03:30:00Z,53.516566,8.188217,None,8.22|2010-11-15T03:40:00Z,53.516566,8.188217,None,8.25|2010-11-15T03:50:00Z,53.516566,8.188217,None,8.26|2010-11-15T04:00:00Z,53.516566,8.188217,None,8.25|2010-11-15T04:10:00Z,53.516566,8.188217,None,8.26|2010-11-15T04:20:00Z,53.516566,8.188217,None,8.3|2010-11-15T04:30:00Z,53.516566,8.188217,None,8.34|2010-11-15T04:40:00Z,53.516566,8.188217,None,8.37|2010-11-15T04:50:00Z,53.516566,8.188217,None,8.39|2010-11-15T05:00:00Z,53.516566,8.188217,None,8.4|2010-11-15T05:10:00Z,53.516566,8.188217,None,8.44|2010-11-15T05:20:00Z,53.516566,8.188217,None,8.45|2010-11-15T05:30:00Z,53.516566,8.188217,None,8.44|2010-11-15T05:40:00Z,53.516566,8.188217,None,8.44|2010-11-15T05:50:00Z,53.516566,8.188217,None,8.41|2010-11-15T06:00:00Z,53.516566,8.188217,None,8.35|2010-11-15T06:10:00Z,53.516566,8.188217,None,8.3|2010-11-15T06:20:00Z,53.516566,8.188217,None,8.22|2010-11-15T06:30:00Z,53.516566,8.188217,None,8.16|2010-11-15T06:40:00Z,53.516566,8.188217,None,8.08|2010-11-15T06:50:00Z,53.516566,8.188217,None,8.02|2010-11-15T07:00:00Z,53.516566,8.188217,None,8.03|2010-11-15T07:10:00Z,53.516566,8.188217,None,8.06|2010-11-15T07:20:00Z,53.516566,8.188217,None,8.08|2010-11-15T07:30:00Z,53.516566,8.188217,None,8.1|2010-11-15T07:40:00Z,53.516566,8.188217,None,8.11|2010-11-15T07:50:00Z,53.516566,8.188217,None,8.11|2010-11-15T08:00:00Z,53.516566,8.188217,None,8.11|2010-11-15T08:10:00Z,53.516566,8.188217,None,8.1|2010-11-15T08:20:00Z,53.516566,8.188217,None,8.09|2010-11-15T08:30:00Z,53.516566,8.188217,None,8.08|2010-11-15T08:40:00Z,53.516566,8.188217,None,8.08|2010-11-15T08:50:00Z,53.516566,8.188217,None,8.08|2010-11-15T09:00:00Z,53.516566,8.188217,None,8.1|2010-11-15T09:10:00Z,53.516566,8.188217,None,8.1|2010-11-15T09:20:00Z,53.516566,8.188217,None,8.12|2010-11-15T09:30:00Z,53.516566,8.188217,None,8.14|2010-11-15T09:40:00Z,53.516566,8.188217,None,8.15|2010-11-15T09:50:00Z,53.516566,8.188217,None,8.15|2010-11-15T10:00:00Z,53.516566,8.188217,None,8.14|2010-11-15T10:10:00Z,53.516566,8.188217,None,8.13|2010-11-15T10:20:00Z,53.516566,8.188217,None,8.13|2010-11-15T10:30:00Z,53.516566,8.188217,None,8.14|2010-11-15T10:40:00Z,53.516566,8.188217,None,8.15|2010-11-15T10:50:00Z,53.516566,8.188217,None,8.15|2010-11-15T11:00:00Z,53.516566,8.188217,None,8.15|2010-11-15T11:10:00Z,53.516566,8.188217,None,8.16|2010-11-15T11:20:00Z,53.516566,8.188217,None,8.16|2010-11-15T11:30:00Z,53.516566,8.188217,None,8.17|2010-11-15T11:40:00Z,53.516566,8.188217,None,8.18|2010-11-15T11:50:00Z,53.516566,8.188217,None,8.18|2010-11-15T12:00:00Z,53.516566,8.188217,None,8.19|2010-11-15T12:10:00Z,53.516566,8.188217,None,8.19|2010-11-15T12:20:00Z,53.516566,8.188217,None,8.19|2010-11-15T12:30:00Z,53.516566,8.188217,None,8.19|2010-11-15T12:40:00Z,53.516566,8.188217,None,8.18|2010-11-15T12:50:00Z,53.516566,8.188217,None,8.17|2010-11-15T13:00:00Z,53.516566,8.188217,None,8.18|2010-11-15T13:10:00Z,53.516566,8.188217,None,8.2|2010-11-15T13:20:00Z,53.516566,8.188217,None,8.23|2010-11-15T13:30:00Z,53.516566,8.188217,None,8.21|2010-11-15T13:40:00Z,53.516566,8.188217,None,8.19|2010-11-15T13:50:00Z,53.516566,8.188217,None,8.18|2010-11-15T14:00:00Z,53.516566,8.188217,None,8.18|2010-11-15T14:10:00Z,53.516566,8.188217,None,8.18|2010-11-15T14:20:00Z,53.516566,8.188217,None,8.17|2010-11-15T14:30:00Z,53.516566,8.188217,None,8.16|2010-11-15T14:40:00Z,53.516566,8.188217,None,8.15|2010-11-15T14:50:00Z,53.516566,8.188217,None,8.15|2010-11-15T15:00:00Z,53.516566,8.188217,None,8.16|2010-11-15T15:10:00Z,53.516566,8.188217,None,8.15|2010-11-15T15:20:00Z,53.516566,8.188217,None,8.15|2010-11-15T15:30:00Z,53.516566,8.188217,None,8.15|2010-11-15T15:40:00Z,53.516566,8.188217,None,8.14|2010-11-15T15:50:00Z,53.516566,8.188217,None,8.13|2010-11-15T16:00:00Z,53.516566,8.188217,None,8.12|2010-11-15T16:10:00Z,53.516566,8.188217,None,8.12|2010-11-15T16:20:00Z,53.516566,8.188217,None,8.12|2010-11-15T16:30:00Z,53.516566,8.188217,None,8.13|2010-11-15T16:40:00Z,53.516566,8.188217,None,8.13|2010-11-15T16:50:00Z,53.516566,8.188217,None,8.13|2010-11-15T17:00:00Z,53.516566,8.188217,None,8.13|2010-11-15T17:10:00Z,53.516566,8.188217,None,8.13|2010-11-15T17:20:00Z,53.516566,8.188217,None,8.13|2010-11-15T17:30:00Z,53.516566,8.188217,None,8.13|2010-11-15T17:40:00Z,53.516566,8.188217,None,8.14|2010-11-15T17:50:00Z,53.516566,8.188217,None,8.15|2010-11-15T18:00:00Z,53.516566,8.188217,None,8.16|2010-11-15T18:10:00Z,53.516566,8.188217,None,8.17|2010-11-15T18:20:00Z,53.516566,8.188217,None,8.18|2010-11-15T18:30:00Z,53.516566,8.188217,None,8.18|2010-11-15T18:40:00Z,53.516566,8.188217,None,8.18|2010-11-15T18:50:00Z,53.516566,8.188217,None,8.17|2010-11-15T19:00:00Z,53.516566,8.188217,None,8.14|2010-11-15T19:10:00Z,53.516566,8.188217,None,8.11|2010-11-15T19:20:00Z,53.516566,8.188217,None,8.07|2010-11-15T19:30:00Z,53.516566,8.188217,None,8.05|2010-11-15T19:40:00Z,53.516566,8.188217,None,8.03|2010-11-15T19:50:00Z,53.516566,8.188217,None,8.05|2010-11-15T20:00:00Z,53.516566,8.188217,None,8.06|2010-11-15T20:10:00Z,53.516566,8.188217,None,8.09|2010-11-15T20:20:00Z,53.516566,8.188217,None,8.12|2010-11-15T20:30:00Z,53.516566,8.188217,None,8.15|2010-11-15T20:40:00Z,53.516566,8.188217,None,8.15|2010-11-15T20:50:00Z,53.516566,8.188217,None,8.13|2010-11-15T21:00:00Z,53.516566,8.188217,None,8.08|2010-11-15T21:10:00Z,53.516566,8.188217,None,8.05|2010-11-15T21:20:00Z,53.516566,8.188217,None,8.05|2010-11-15T21:30:00Z,53.516566,8.188217,None,8.06|2010-11-15T21:40:00Z,53.516566,8.188217,None,8.03|2010-11-15T21:50:00Z,53.516566,8.188217,None,8.02|2010-11-15T22:00:00Z,53.516566,8.188217,None,8.03|2010-11-15T22:10:00Z,53.516566,8.188217,None,8.05|2010-11-15T22:20:00Z,53.516566,8.188217,None,8.08|2010-11-15T22:30:00Z,53.516566,8.188217,None,8.07|2010-11-15T22:40:00Z,53.516566,8.188217,None,8.14|2010-11-15T22:50:00Z,53.516566,8.188217,None,8.11|2010-11-15T23:00:00Z,53.516566,8.188217,None,8.02|2010-11-15T23:10:00Z,53.516566,8.188217,None,8.02|2010-11-15T23:20:00Z,53.516566,8.188217,None,8.02|2010-11-15T23:30:00Z,53.516566,8.188217,None,8.05|2010-11-15T23:40:00Z,53.516566,8.188217,None,8.05|2010-11-15T23:50:00Z,53.516566,8.188217,None,8.03|2010-11-16T00:00:00Z,53.516566,8.188217,None,7.97|2010-11-16T00:10:00Z,53.516566,8.188217,None,7.94|2010-11-16T00:20:00Z,53.516566,8.188217,None,7.92|2010-11-16T00:30:00Z,53.516566,8.188217,None,7.91|2010-11-16T00:40:00Z,53.516566,8.188217,None,7.91|2010-11-16T00:50:00Z,53.516566,8.188217,None,7.91|2010-11-16T01:00:00Z,53.516566,8.188217,None,7.96|2010-11-16T01:10:00Z,53.516566,8.188217,None,8.0|2010-11-16T01:20:00Z,53.516566,8.188217,None,8.0|2010-11-16T01:30:00Z,53.516566,8.188217,None,7.98|2010-11-16T01:40:00Z,53.516566,8.188217,None,7.99|2010-11-16T01:50:00Z,53.516566,8.188217,None,8.01|2010-11-16T02:00:00Z,53.516566,8.188217,None,8.04|2010-11-16T02:10:00Z,53.516566,8.188217,None,7.98|2010-11-16T02:20:00Z,53.516566,8.188217,None,7.96|2010-11-16T02:30:00Z,53.516566,8.188217,None,7.94|2010-11-16T02:40:00Z,53.516566,8.188217,None,7.95|2010-11-16T02:50:00Z,53.516566,8.188217,None,7.99|2010-11-16T03:00:00Z,53.516566,8.188217,None,8.02|2010-11-16T03:10:00Z,53.516566,8.188217,None,8.0|2010-11-16T03:20:00Z,53.516566,8.188217,None,8.02|2010-11-16T03:30:00Z,53.516566,8.188217,None,8.05|2010-11-16T03:40:00Z,53.516566,8.188217,None,8.08|2010-11-16T03:50:00Z,53.516566,8.188217,None,8.07|2010-11-16T04:00:00Z,53.516566,8.188217,None,8.05|2010-11-16T04:10:00Z,53.516566,8.188217,None,8.06|2010-11-16T04:20:00Z,53.516566,8.188217,None,8.07|2010-11-16T04:30:00Z,53.516566,8.188217,None,8.05|2010-11-16T04:40:00Z,53.516566,8.188217,None,8.03|2010-11-16T04:50:00Z,53.516566,8.188217,None,8.06|2010-11-16T05:00:00Z,53.516566,8.188217,None,8.08|2010-11-16T05:10:00Z,53.516566,8.188217,None,8.09|2010-11-16T05:20:00Z,53.516566,8.188217,None,8.1|2010-11-16T05:30:00Z,53.516566,8.188217,None,8.1|2010-11-16T05:40:00Z,53.516566,8.188217,None,8.12|2010-11-16T05:50:00Z,53.516566,8.188217,None,8.17|2010-11-16T06:00:00Z,53.516566,8.188217,None,8.21|2010-11-16T06:10:00Z,53.516566,8.188217,None,8.24|2010-11-16T06:20:00Z,53.516566,8.188217,None,8.25|2010-11-16T06:30:00Z,53.516566,8.188217,None,8.25|2010-11-16T06:40:00Z,53.516566,8.188217,None,8.24|2010-11-16T06:50:00Z,53.516566,8.188217,None,8.24|2010-11-16T07:00:00Z,53.516566,8.188217,None,8.24|2010-11-16T07:10:00Z,53.516566,8.188217,None,8.23|2010-11-16T07:20:00Z,53.516566,8.188217,None,8.22|2010-11-16T07:30:00Z,53.516566,8.188217,None,8.21|2010-11-16T07:40:00Z,53.516566,8.188217,None,8.14|2010-11-16T07:50:00Z,53.516566,8.188217,None,8.02|2010-11-16T08:00:00Z,53.516566,8.188217,None,7.95|2010-11-16T08:10:00Z,53.516566,8.188217,None,7.93|2010-11-16T08:20:00Z,53.516566,8.188217,None,7.9|2010-11-16T08:30:00Z,53.516566,8.188217,None,7.89|2010-11-16T08:40:00Z,53.516566,8.188217,None,7.89|2010-11-16T08:50:00Z,53.516566,8.188217,None,7.93|2010-11-16T09:00:00Z,53.516566,8.188217,None,7.93|2010-11-16T09:10:00Z,53.516566,8.188217,None,7.93|2010-11-16T09:20:00Z,53.516566,8.188217,None,7.96|2010-11-16T09:30:00Z,53.516566,8.188217,None,7.96|2010-11-16T09:40:00Z,53.516566,8.188217,None,7.96|2010-11-16T09:50:00Z,53.516566,8.188217,None,7.95|2010-11-16T10:00:00Z,53.516566,8.188217,None,7.97|2010-11-16T10:10:00Z,53.516566,8.188217,None,8.01|2010-11-16T10:20:00Z,53.516566,8.188217,None,8.02|2010-11-16T10:30:00Z,53.516566,8.188217,None,8.02|2010-11-16T10:40:00Z,53.516566,8.188217,None,8.03|2010-11-16T10:50:00Z,53.516566,8.188217,None,8.03|2010-11-16T11:00:00Z,53.516566,8.188217,None,8.03|2010-11-16T11:10:00Z,53.516566,8.188217,None,8.03|2010-11-16T11:20:00Z,53.516566,8.188217,None,8.07|2010-11-16T11:30:00Z,53.516566,8.188217,None,8.09|2010-11-16T11:40:00Z,53.516566,8.188217,None,8.07|2010-11-16T11:50:00Z,53.516566,8.188217,None,8.08|2010-11-16T12:00:00Z,53.516566,8.188217,None,8.07|2010-11-16T12:10:00Z,53.516566,8.188217,None,8.08|2010-11-16T12:20:00Z,53.516566,8.188217,None,8.1|2010-11-16T12:30:00Z,53.516566,8.188217,None,8.09|2010-11-16T12:40:00Z,53.516566,8.188217,None,8.09|2010-11-16T12:50:00Z,53.516566,8.188217,None,8.09|2010-11-16T13:00:00Z,53.516566,8.188217,None,8.09|2010-11-16T13:10:00Z,53.516566,8.188217,None,8.09|2010-11-16T13:20:00Z,53.516566,8.188217,None,8.09|2010-11-16T13:30:00Z,53.516566,8.188217,None,8.08|2010-11-16T13:40:00Z,53.516566,8.188217,None,8.08|2010-11-16T13:50:00Z,53.516566,8.188217,None,8.07|2010-11-16T14:00:00Z,53.516566,8.188217,None,8.07|2010-11-16T14:10:00Z,53.516566,8.188217,None,8.06|2010-11-16T14:20:00Z,53.516566,8.188217,None,7.98|2010-11-16T14:30:00Z,53.516566,8.188217,None,7.92|2010-11-16T14:40:00Z,53.516566,8.188217,None,7.92|2010-11-16T14:50:00Z,53.516566,8.188217,None,7.91|2010-11-16T15:00:00Z,53.516566,8.188217,None,7.82|2010-11-16T15:10:00Z,53.516566,8.188217,None,7.78|2010-11-16T15:20:00Z,53.516566,8.188217,None,7.79|2010-11-16T15:30:00Z,53.516566,8.188217,None,7.8|2010-11-16T15:40:00Z,53.516566,8.188217,None,7.82|2010-11-16T15:50:00Z,53.516566,8.188217,None,7.83|2010-11-16T16:00:00Z,53.516566,8.188217,None,7.85|2010-11-16T16:10:00Z,53.516566,8.188217,None,7.92|2010-11-16T16:20:00Z,53.516566,8.188217,None,7.94|2010-11-16T16:30:00Z,53.516566,8.188217,None,7.93|2010-11-16T16:40:00Z,53.516566,8.188217,None,7.92|2010-11-16T16:50:00Z,53.516566,8.188217,None,7.9|2010-11-16T17:00:00Z,53.516566,8.188217,None,7.91|2010-11-16T17:10:00Z,53.516566,8.188217,None,7.92|2010-11-16T17:20:00Z,53.516566,8.188217,None,7.93|2010-11-16T17:30:00Z,53.516566,8.188217,None,7.94|2010-11-16T17:40:00Z,53.516566,8.188217,None,7.96|2010-11-16T17:50:00Z,53.516566,8.188217,None,7.97|2010-11-16T18:00:00Z,53.516566,8.188217,None,7.99|2010-11-16T18:10:00Z,53.516566,8.188217,None,8.0|2010-11-16T18:20:00Z,53.516566,8.188217,None,8.02|2010-11-16T18:30:00Z,53.516566,8.188217,None,8.03|2010-11-16T18:40:00Z,53.516566,8.188217,None,8.03|2010-11-16T18:50:00Z,53.516566,8.188217,None,8.06|2010-11-16T19:00:00Z,53.516566,8.188217,None,8.07|2010-11-16T19:10:00Z,53.516566,8.188217,None,8.1|2010-11-16T19:20:00Z,53.516566,8.188217,None,8.11|2010-11-16T19:30:00Z,53.516566,8.188217,None,8.12|2010-11-16T19:40:00Z,53.516566,8.188217,None,8.12|2010-11-16T19:50:00Z,53.516566,8.188217,None,8.11|2010-11-16T20:00:00Z,53.516566,8.188217,None,8.11|2010-11-16T20:10:00Z,53.516566,8.188217,None,8.1|2010-11-16T20:20:00Z,53.516566,8.188217,None,8.06|2010-11-16T20:30:00Z,53.516566,8.188217,None,8.03|2010-11-16T20:40:00Z,53.516566,8.188217,None,7.97|2010-11-16T20:50:00Z,53.516566,8.188217,None,7.89|2010-11-16T21:00:00Z,53.516566,8.188217,None,7.81|2010-11-16T21:10:00Z,53.516566,8.188217,None,7.79|2010-11-16T21:20:00Z,53.516566,8.188217,None,7.78|2010-11-16T21:30:00Z,53.516566,8.188217,None,7.79|2010-11-16T21:40:00Z,53.516566,8.188217,None,7.82|2010-11-16T21:50:00Z,53.516566,8.188217,None,7.82|2010-11-16T22:00:00Z,53.516566,8.188217,None,7.82|2010-11-16T22:10:00Z,53.516566,8.188217,None,7.83|2010-11-16T22:20:00Z,53.516566,8.188217,None,7.83|2010-11-16T22:30:00Z,53.516566,8.188217,None,7.81|2010-11-16T22:40:00Z,53.516566,8.188217,None,7.81|2010-11-16T22:50:00Z,53.516566,8.188217,None,7.82|2010-11-16T23:00:00Z,53.516566,8.188217,None,7.81|2010-11-16T23:10:00Z,53.516566,8.188217,None,7.8|2010-11-16T23:20:00Z,53.516566,8.188217,None,7.79|2010-11-16T23:30:00Z,53.516566,8.188217,None,7.78|2010-11-16T23:40:00Z,53.516566,8.188217,None,7.8|2010-11-16T23:50:00Z,53.516566,8.188217,None,7.86|2010-11-17T00:00:00Z,53.516566,8.188217,None,7.87|2010-11-17T00:10:00Z,53.516566,8.188217,None,7.88|2010-11-17T00:20:00Z,53.516566,8.188217,None,7.87|2010-11-17T00:30:00Z,53.516566,8.188217,None,7.92|2010-11-17T00:40:00Z,53.516566,8.188217,None,7.96|2010-11-17T00:50:00Z,53.516566,8.188217,None,7.96|2010-11-17T01:00:00Z,53.516566,8.188217,None,7.94|2010-11-17T01:10:00Z,53.516566,8.188217,None,7.89|2010-11-17T01:20:00Z,53.516566,8.188217,None,7.9|2010-11-17T01:30:00Z,53.516566,8.188217,None,7.96|2010-11-17T01:40:00Z,53.516566,8.188217,None,7.93|2010-11-17T01:50:00Z,53.516566,8.188217,None,7.93|2010-11-17T02:00:00Z,53.516566,8.188217,None,7.92|2010-11-17T02:10:00Z,53.516566,8.188217,None,7.91|2010-11-17T02:20:00Z,53.516566,8.188217,None,7.91|2010-11-17T02:30:00Z,53.516566,8.188217,None,7.88|2010-11-17T02:40:00Z,53.516566,8.188217,None,7.77|2010-11-17T02:50:00Z,53.516566,8.188217,None,7.64|2010-11-17T03:00:00Z,53.516566,8.188217,None,7.75|2010-11-17T03:10:00Z,53.516566,8.188217,None,7.8|2010-11-17T03:20:00Z,53.516566,8.188217,None,7.78|2010-11-17T03:30:00Z,53.516566,8.188217,None,7.78|2010-11-17T03:40:00Z,53.516566,8.188217,None,7.78|2010-11-17T03:50:00Z,53.516566,8.188217,None,7.71|2010-11-17T04:00:00Z,53.516566,8.188217,None,7.71|2010-11-17T04:10:00Z,53.516566,8.188217,None,7.74|2010-11-17T04:20:00Z,53.516566,8.188217,None,7.74|2010-11-17T04:30:00Z,53.516566,8.188217,None,7.78|2010-11-17T04:40:00Z,53.516566,8.188217,None,7.82|2010-11-17T04:50:00Z,53.516566,8.188217,None,7.86|2010-11-17T05:00:00Z,53.516566,8.188217,None,7.82|2010-11-17T05:10:00Z,53.516566,8.188217,None,7.8|2010-11-17T05:20:00Z,53.516566,8.188217,None,7.83|2010-11-17T05:30:00Z,53.516566,8.188217,None,7.88|2010-11-17T05:40:00Z,53.516566,8.188217,None,7.92|2010-11-17T05:50:00Z,53.516566,8.188217,None,7.93|2010-11-17T06:00:00Z,53.516566,8.188217,None,7.93|2010-11-17T06:10:00Z,53.516566,8.188217,None,7.94|2010-11-17T06:20:00Z,53.516566,8.188217,None,7.95|2010-11-17T06:30:00Z,53.516566,8.188217,None,7.98|2010-11-17T06:40:00Z,53.516566,8.188217,None,8.0|2010-11-17T06:50:00Z,53.516566,8.188217,None,8.01|2010-11-17T07:00:00Z,53.516566,8.188217,None,8.04|2010-11-17T07:10:00Z,53.516566,8.188217,None,8.05|2010-11-17T07:20:00Z,53.516566,8.188217,None,8.06|2010-11-17T07:30:00Z,53.516566,8.188217,None,8.06|2010-11-17T07:40:00Z,53.516566,8.188217,None,8.06|2010-11-17T07:50:00Z,53.516566,8.188217,None,8.06|2010-11-17T08:00:00Z,53.516566,8.188217,None,8.05|2010-11-17T08:10:00Z,53.516566,8.188217,None,8.05|2010-11-17T08:20:00Z,53.516566,8.188217,None,8.05|2010-11-17T08:30:00Z,53.516566,8.188217,None,8.04|2010-11-17T08:40:00Z,53.516566,8.188217,None,8.04|2010-11-17T08:50:00Z,53.516566,8.188217,None,8.02|2010-11-17T09:00:00Z,53.516566,8.188217,None,7.98|2010-11-17T09:10:00Z,53.516566,8.188217,None,7.91|2010-11-17T09:20:00Z,53.516566,8.188217,None,7.84|2010-11-17T09:30:00Z,53.516566,8.188217,None,7.76|2010-11-17T09:40:00Z,53.516566,8.188217,None,7.67|2010-11-17T09:50:00Z,53.516566,8.188217,None,7.58|2010-11-17T10:00:00Z,53.516566,8.188217,None,7.43|2010-11-17T10:10:00Z,53.516566,8.188217,None,7.41|2010-11-17T10:20:00Z,53.516566,8.188217,None,7.35|2010-11-17T10:30:00Z,53.516566,8.188217,None,7.27|2010-11-17T10:40:00Z,53.516566,8.188217,None,7.26|2010-11-17T10:50:00Z,53.516566,8.188217,None,7.26|2010-11-17T11:00:00Z,53.516566,8.188217,None,7.29|2010-11-17T11:10:00Z,53.516566,8.188217,None,7.35|2010-11-17T11:20:00Z,53.516566,8.188217,None,7.39|2010-11-17T11:30:00Z,53.516566,8.188217,None,7.42|2010-11-17T11:40:00Z,53.516566,8.188217,None,7.43|2010-11-17T11:50:00Z,53.516566,8.188217,None,7.43|2010-11-17T12:00:00Z,53.516566,8.188217,None,7.43|2010-11-17T12:10:00Z,53.516566,8.188217,None,7.44|2010-11-17T12:20:00Z,53.516566,8.188217,None,7.45|2010-11-17T12:30:00Z,53.516566,8.188217,None,7.52|2010-11-17T12:40:00Z,53.516566,8.188217,None,7.54|2010-11-17T12:50:00Z,53.516566,8.188217,None,7.56|2010-11-17T13:00:00Z,53.516566,8.188217,None,7.56|2010-11-17T13:10:00Z,53.516566,8.188217,None,7.59|2010-11-17T13:20:00Z,53.516566,8.188217,None,7.58|2010-11-17T13:30:00Z,53.516566,8.188217,None,7.5|2010-11-17T13:40:00Z,53.516566,8.188217,None,7.58|2010-11-17T13:50:00Z,53.516566,8.188217,None,7.6|2010-11-17T14:00:00Z,53.516566,8.188217,None,7.6|2010-11-17T14:10:00Z,53.516566,8.188217,None,7.64|2010-11-17T14:20:00Z,53.516566,8.188217,None,7.66|2010-11-17T14:30:00Z,53.516566,8.188217,None,7.66|2010-11-17T14:40:00Z,53.516566,8.188217,None,7.65|2010-11-17T14:50:00Z,53.516566,8.188217,None,7.69|2010-11-17T15:00:00Z,53.516566,8.188217,None,7.66|2010-11-17T15:10:00Z,53.516566,8.188217,None,7.61|2010-11-17T15:20:00Z,53.516566,8.188217,None,7.52|2010-11-17T15:30:00Z,53.516566,8.188217,None,7.39|2010-11-17T15:40:00Z,53.516566,8.188217,None,7.58|2010-11-17T15:50:00Z,53.516566,8.188217,None,7.64|2010-11-17T16:00:00Z,53.516566,8.188217,None,7.64|2010-11-17T16:10:00Z,53.516566,8.188217,None,7.65|2010-11-17T16:20:00Z,53.516566,8.188217,None,7.66|2010-11-17T16:30:00Z,53.516566,8.188217,None,7.57|2010-11-17T16:40:00Z,53.516566,8.188217,None,7.44|2010-11-17T16:50:00Z,53.516566,8.188217,None,7.45|2010-11-17T17:00:00Z,53.516566,8.188217,None,7.5|2010-11-17T17:10:00Z,53.516566,8.188217,None,7.53|2010-11-17T17:20:00Z,53.516566,8.188217,None,7.57|2010-11-17T17:30:00Z,53.516566,8.188217,None,7.64|2010-11-17T17:40:00Z,53.516566,8.188217,None,7.61|2010-11-17T17:50:00Z,53.516566,8.188217,None,7.61|2010-11-17T18:00:00Z,53.516566,8.188217,None,7.62|2010-11-17T18:10:00Z,53.516566,8.188217,None,7.63|2010-11-17T18:20:00Z,53.516566,8.188217,None,7.66|2010-11-17T18:30:00Z,53.516566,8.188217,None,7.68|2010-11-17T18:40:00Z,53.516566,8.188217,None,7.68|2010-11-17T18:50:00Z,53.516566,8.188217,None,7.69|2010-11-17T19:00:00Z,53.516566,8.188217,None,7.7|2010-11-17T19:10:00Z,53.516566,8.188217,None,7.72|2010-11-17T19:20:00Z,53.516566,8.188217,None,7.73|2010-11-17T19:30:00Z,53.516566,8.188217,None,7.76|2010-11-17T19:40:00Z,53.516566,8.188217,None,7.8|2010-11-17T19:50:00Z,53.516566,8.188217,None,7.81|2010-11-17T20:00:00Z,53.516566,8.188217,None,7.83|2010-11-17T20:10:00Z,53.516566,8.188217,None,7.84|2010-11-17T20:20:00Z,53.516566,8.188217,None,7.85|2010-11-17T20:30:00Z,53.516566,8.188217,None,7.84|2010-11-17T20:40:00Z,53.516566,8.188217,None,7.84|2010-11-17T20:50:00Z,53.516566,8.188217,None,7.81|2010-11-17T21:00:00Z,53.516566,8.188217,None,7.77|2010-11-17T21:10:00Z,53.516566,8.188217,None,7.67|2010-11-17T21:20:00Z,53.516566,8.188217,None,7.62|2010-11-17T21:30:00Z,53.516566,8.188217,None,7.56|2010-11-17T21:40:00Z,53.516566,8.188217,None,7.36|2010-11-17T21:50:00Z,53.516566,8.188217,None,7.19|2010-11-17T22:00:00Z,53.516566,8.188217,None,7.06|2010-11-17T22:10:00Z,53.516566,8.188217,None,7.08|2010-11-17T22:20:00Z,53.516566,8.188217,None,7.21|2010-11-17T22:30:00Z,53.516566,8.188217,None,7.31|2010-11-17T22:40:00Z,53.516566,8.188217,None,7.34|2010-11-17T22:50:00Z,53.516566,8.188217,None,7.36|2010-11-17T23:00:00Z,53.516566,8.188217,None,7.35|2010-11-17T23:10:00Z,53.516566,8.188217,None,7.32|2010-11-17T23:20:00Z,53.516566,8.188217,None,7.26|2010-11-17T23:30:00Z,53.516566,8.188217,None,7.22|2010-11-17T23:40:00Z,53.516566,8.188217,None,7.22|2010-11-17T23:50:00Z,53.516566,8.188217,None,7.32|2010-11-18T00:00:00Z,53.516566,8.188217,None,7.38|2010-11-18T00:10:00Z,53.516566,8.188217,None,7.45|2010-11-18T00:20:00Z,53.516566,8.188217,None,7.57|2010-11-18T00:30:00Z,53.516566,8.188217,None,7.58|2010-11-18T00:40:00Z,53.516566,8.188217,None,7.58|2010-11-18T00:50:00Z,53.516566,8.188217,None,7.58|2010-11-18T01:00:00Z,53.516566,8.188217,None,7.54|2010-11-18T01:10:00Z,53.516566,8.188217,None,7.56|2010-11-18T01:20:00Z,53.516566,8.188217,None,7.52|2010-11-18T01:30:00Z,53.516566,8.188217,None,7.56|2010-11-18T01:40:00Z,53.516566,8.188217,None,7.55|2010-11-18T01:50:00Z,53.516566,8.188217,None,7.52|2010-11-18T02:00:00Z,53.516566,8.188217,None,7.5|2010-11-18T02:10:00Z,53.516566,8.188217,None,7.52|2010-11-18T02:20:00Z,53.516566,8.188217,None,7.54|2010-11-18T02:30:00Z,53.516566,8.188217,None,7.55|2010-11-18T02:40:00Z,53.516566,8.188217,None,7.54|2010-11-18T02:50:00Z,53.516566,8.188217,None,7.56|2010-11-18T03:00:00Z,53.516566,8.188217,None,7.57|2010-11-18T03:10:00Z,53.516566,8.188217,None,7.54|2010-11-18T03:20:00Z,53.516566,8.188217,None,7.5|2010-11-18T03:30:00Z,53.516566,8.188217,None,7.4|2010-11-18T03:40:00Z,53.516566,8.188217,None,7.34|2010-11-18T03:50:00Z,53.516566,8.188217,None,7.32|2010-11-18T04:00:00Z,53.516566,8.188217,None,7.34|2010-11-18T04:10:00Z,53.516566,8.188217,None,7.46|2010-11-18T04:20:00Z,53.516566,8.188217,None,7.46|2010-11-18T04:30:00Z,53.516566,8.188217,None,7.45|2010-11-18T04:40:00Z,53.516566,8.188217,None,7.34|2010-11-18T04:50:00Z,53.516566,8.188217,None,7.24|2010-11-18T05:00:00Z,53.516566,8.188217,None,7.26|2010-11-18T05:10:00Z,53.516566,8.188217,None,7.29|2010-11-18T05:20:00Z,53.516566,8.188217,None,7.33|2010-11-18T05:30:00Z,53.516566,8.188217,None,7.42|2010-11-18T05:40:00Z,53.516566,8.188217,None,7.45|2010-11-18T05:50:00Z,53.516566,8.188217,None,7.44|2010-11-18T06:00:00Z,53.516566,8.188217,None,7.43|2010-11-18T06:10:00Z,53.516566,8.188217,None,7.44|2010-11-18T06:20:00Z,53.516566,8.188217,None,7.47|2010-11-18T06:30:00Z,53.516566,8.188217,None,7.51|2010-11-18T06:40:00Z,53.516566,8.188217,None,7.55|2010-11-18T06:50:00Z,53.516566,8.188217,None,7.58|2010-11-18T07:00:00Z,53.516566,8.188217,None,7.61|2010-11-18T07:10:00Z,53.516566,8.188217,None,7.65|2010-11-18T07:20:00Z,53.516566,8.188217,None,7.69|2010-11-18T07:30:00Z,53.516566,8.188217,None,7.72|2010-11-18T07:40:00Z,53.516566,8.188217,None,7.74|2010-11-18T07:50:00Z,53.516566,8.188217,None,7.79|2010-11-18T08:00:00Z,53.516566,8.188217,None,7.81|2010-11-18T08:10:00Z,53.516566,8.188217,None,7.83|2010-11-18T08:20:00Z,53.516566,8.188217,None,7.84|2010-11-18T08:30:00Z,53.516566,8.188217,None,7.84|2010-11-18T08:40:00Z,53.516566,8.188217,None,7.85|2010-11-18T08:50:00Z,53.516566,8.188217,None,7.85|2010-11-18T09:00:00Z,53.516566,8.188217,None,7.85|2010-11-18T09:10:00Z,53.516566,8.188217,None,7.85|2010-11-18T09:20:00Z,53.516566,8.188217,None,7.84|2010-11-18T09:30:00Z,53.516566,8.188217,None,7.83|2010-11-18T09:40:00Z,53.516566,8.188217,None,7.81|2010-11-18T09:50:00Z,53.516566,8.188217,None,7.77|2010-11-18T10:00:00Z,53.516566,8.188217,None,7.71|2010-11-18T10:10:00Z,53.516566,8.188217,None,7.59|2010-11-18T10:20:00Z,53.516566,8.188217,None,7.42|2010-11-18T10:30:00Z,53.516566,8.188217,None,7.24|2010-11-18T10:40:00Z,53.516566,8.188217,None,7.05|2010-11-18T10:50:00Z,53.516566,8.188217,None,6.953|2010-11-18T11:00:00Z,53.516566,8.188217,None,6.92|2010-11-18T11:10:00Z,53.516566,8.188217,None,6.925|2010-11-18T11:20:00Z,53.516566,8.188217,None,6.976|2010-11-18T11:30:00Z,53.516566,8.188217,None,7.01|2010-11-18T11:40:00Z,53.516566,8.188217,None,7.06|2010-11-18T11:50:00Z,53.516566,8.188217,None,7.14|2010-11-18T12:00:00Z,53.516566,8.188217,None,7.14|2010-11-18T12:10:00Z,53.516566,8.188217,None,7.14|2010-11-18T12:20:00Z,53.516566,8.188217,None,7.18|2010-11-18T12:30:00Z,53.516566,8.188217,None,7.2|2010-11-18T12:40:00Z,53.516566,8.188217,None,7.22|2010-11-18T12:50:00Z,53.516566,8.188217,None,7.22|2010-11-18T13:00:00Z,53.516566,8.188217,None,7.25|2010-11-18T13:10:00Z,53.516566,8.188217,None,7.36|2010-11-18T13:20:00Z,53.516566,8.188217,None,7.46|2010-11-18T13:30:00Z,53.516566,8.188217,None,7.55|2010-11-18T13:40:00Z,53.516566,8.188217,None,7.52|2010-11-18T13:50:00Z,53.516566,8.188217,None,7.51|2010-11-18T14:00:00Z,53.516566,8.188217,None,7.47|2010-11-18T14:10:00Z,53.516566,8.188217,None,7.45|2010-11-18T14:20:00Z,53.516566,8.188217,None,7.43|2010-11-18T14:30:00Z,53.516566,8.188217,None,7.44|2010-11-18T14:40:00Z,53.516566,8.188217,None,7.39|2010-11-18T14:50:00Z,53.516566,8.188217,None,7.45|2010-11-18T15:00:00Z,53.516566,8.188217,None,7.5|2010-11-18T15:10:00Z,53.516566,8.188217,None,7.48|2010-11-18T15:20:00Z,53.516566,8.188217,None,7.5|2010-11-18T15:30:00Z,53.516566,8.188217,None,7.5|2010-11-18T15:40:00Z,53.516566,8.188217,None,7.47|2010-11-18T15:50:00Z,53.516566,8.188217,None,7.46|2010-11-18T16:00:00Z,53.516566,8.188217,None,7.44|2010-11-18T16:10:00Z,53.516566,8.188217,None,7.35|2010-11-18T16:20:00Z,53.516566,8.188217,None,7.25|2010-11-18T16:30:00Z,53.516566,8.188217,None,7.25|2010-11-18T16:40:00Z,53.516566,8.188217,None,7.24|2010-11-18T16:50:00Z,53.516566,8.188217,None,7.16|2010-11-18T17:00:00Z,53.516566,8.188217,None,7.09|2010-11-18T17:10:00Z,53.516566,8.188217,None,7.1|2010-11-18T17:20:00Z,53.516566,8.188217,None,7.16|2010-11-18T17:30:00Z,53.516566,8.188217,None,7.18|2010-11-18T17:40:00Z,53.516566,8.188217,None,7.22|2010-11-18T17:50:00Z,53.516566,8.188217,None,7.28|2010-11-18T18:00:00Z,53.516566,8.188217,None,7.3|2010-11-18T18:10:00Z,53.516566,8.188217,None,7.27|2010-11-18T18:20:00Z,53.516566,8.188217,None,7.3|2010-11-18T18:30:00Z,53.516566,8.188217,None,7.31|2010-11-18T18:40:00Z,53.516566,8.188217,None,7.32|2010-11-18T18:50:00Z,53.516566,8.188217,None,7.36|2010-11-18T19:00:00Z,53.516566,8.188217,None,7.4|2010-11-18T19:10:00Z,53.516566,8.188217,None,7.42|2010-11-18T19:20:00Z,53.516566,8.188217,None,7.48|2010-11-18T19:30:00Z,53.516566,8.188217,None,7.52|2010-11-18T19:40:00Z,53.516566,8.188217,None,7.57|2010-11-18T19:50:00Z,53.516566,8.188217,None,7.61|2010-11-18T20:00:00Z,53.516566,8.188217,None,7.63|2010-11-18T20:10:00Z,53.516566,8.188217,None,7.64|2010-11-18T20:20:00Z,53.516566,8.188217,None,7.67|2010-11-18T20:30:00Z,53.516566,8.188217,None,7.7|2010-11-18T20:40:00Z,53.516566,8.188217,None,7.7|2010-11-18T20:50:00Z,53.516566,8.188217,None,7.7|2010-11-18T21:00:00Z,53.516566,8.188217,None,7.7|2010-11-18T21:10:00Z,53.516566,8.188217,None,7.71|2010-11-18T21:20:00Z,53.516566,8.188217,None,7.71|2010-11-18T21:30:00Z,53.516566,8.188217,None,7.7|2010-11-18T21:40:00Z,53.516566,8.188217,None,7.67|2010-11-18T21:50:00Z,53.516566,8.188217,None,7.67|2010-11-18T22:00:00Z,53.516566,8.188217,None,7.67|2010-11-18T22:10:00Z,53.516566,8.188217,None,7.66|2010-11-18T22:20:00Z,53.516566,8.188217,None,7.63|2010-11-18T22:30:00Z,53.516566,8.188217,None,7.52|2010-11-18T22:40:00Z,53.516566,8.188217,None,7.34|2010-11-18T22:50:00Z,53.516566,8.188217,None,7.21|2010-11-18T23:00:00Z,53.516566,8.188217,None,7.12|2010-11-18T23:10:00Z,53.516566,8.188217,None,7.02|2010-11-18T23:20:00Z,53.516566,8.188217,None,6.977|2010-11-18T23:30:00Z,53.516566,8.188217,None,6.967|2010-11-18T23:40:00Z,53.516566,8.188217,None,6.953|2010-11-18T23:50:00Z,53.516566,8.188217,None,7.02|2010-11-19T00:00:00Z,53.516566,8.188217,None,7.14|2010-11-19T00:10:00Z,53.516566,8.188217,None,7.2|2010-11-19T00:20:00Z,53.516566,8.188217,None,7.2|2010-11-19T00:30:00Z,53.516566,8.188217,None,7.21|2010-11-19T00:40:00Z,53.516566,8.188217,None,7.21|2010-11-19T00:50:00Z,53.516566,8.188217,None,7.22|2010-11-19T01:00:00Z,53.516566,8.188217,None,7.22|2010-11-19T01:10:00Z,53.516566,8.188217,None,7.21|2010-11-19T01:20:00Z,53.516566,8.188217,None,7.21|2010-11-19T01:30:00Z,53.516566,8.188217,None,7.22|2010-11-19T01:40:00Z,53.516566,8.188217,None,7.22|2010-11-19T01:50:00Z,53.516566,8.188217,None,7.29|2010-11-19T02:00:00Z,53.516566,8.188217,None,7.46|2010-11-19T02:10:00Z,53.516566,8.188217,None,7.45|2010-11-19T02:20:00Z,53.516566,8.188217,None,7.37|2010-11-19T02:30:00Z,53.516566,8.188217,None,7.36|2010-11-19T02:40:00Z,53.516566,8.188217,None,7.36|2010-11-19T02:50:00Z,53.516566,8.188217,None,7.36|2010-11-19T03:00:00Z,53.516566,8.188217,None,7.34|2010-11-19T03:10:00Z,53.516566,8.188217,None,7.24|2010-11-19T03:20:00Z,53.516566,8.188217,None,7.3|2010-11-19T03:30:00Z,53.516566,8.188217,None,7.27|2010-11-19T03:40:00Z,53.516566,8.188217,None,7.38|2010-11-19T03:50:00Z,53.516566,8.188217,None,7.4|2010-11-19T04:00:00Z,53.516566,8.188217,None,7.41|2010-11-19T04:10:00Z,53.516566,8.188217,None,7.38|2010-11-19T04:20:00Z,53.516566,8.188217,None,7.36|2010-11-19T04:30:00Z,53.516566,8.188217,None,7.29|2010-11-19T04:40:00Z,53.516566,8.188217,None,7.21|2010-11-19T04:50:00Z,53.516566,8.188217,None,7.12|2010-11-19T05:00:00Z,53.516566,8.188217,None,7.01|2010-11-19T05:10:00Z,53.516566,8.188217,None,7.02|2010-11-19T05:20:00Z,53.516566,8.188217,None,7.0|2010-11-19T05:30:00Z,53.516566,8.188217,None,7.02|2010-11-19T05:40:00Z,53.516566,8.188217,None,7.07|2010-11-19T05:50:00Z,53.516566,8.188217,None,7.11|2010-11-19T06:00:00Z,53.516566,8.188217,None,7.14|2010-11-19T06:10:00Z,53.516566,8.188217,None,7.18|2010-11-19T06:20:00Z,53.516566,8.188217,None,7.21|2010-11-19T06:30:00Z,53.516566,8.188217,None,7.22|2010-11-19T06:40:00Z,53.516566,8.188217,None,7.23|2010-11-19T06:50:00Z,53.516566,8.188217,None,7.26|2010-11-19T07:00:00Z,53.516566,8.188217,None,7.32|2010-11-19T07:10:00Z,53.516566,8.188217,None,7.36|2010-11-19T07:20:00Z,53.516566,8.188217,None,7.4|2010-11-19T07:30:00Z,53.516566,8.188217,None,7.43|2010-11-19T07:40:00Z,53.516566,8.188217,None,7.46|2010-11-19T07:50:00Z,53.516566,8.188217,None,7.48|2010-11-19T08:00:00Z,53.516566,8.188217,None,7.52|2010-11-19T08:10:00Z,53.516566,8.188217,None,7.55|2010-11-19T08:20:00Z,53.516566,8.188217,None,7.57|2010-11-19T08:30:00Z,53.516566,8.188217,None,7.6|2010-11-19T08:40:00Z,53.516566,8.188217,None,7.61|2010-11-19T08:50:00Z,53.516566,8.188217,None,7.61|2010-11-19T09:00:00Z,53.516566,8.188217,None,7.61|2010-11-19T09:10:00Z,53.516566,8.188217,None,7.62|2010-11-19T09:20:00Z,53.516566,8.188217,None,7.63|2010-11-19T09:30:00Z,53.516566,8.188217,None,7.63|2010-11-19T09:40:00Z,53.516566,8.188217,None,7.64|2010-11-19T09:50:00Z,53.516566,8.188217,None,7.67|2010-11-19T10:00:00Z,53.516566,8.188217,None,7.69|2010-11-19T10:10:00Z,53.516566,8.188217,None,7.66|2010-11-19T10:20:00Z,53.516566,8.188217,None,7.62|2010-11-19T10:30:00Z,53.516566,8.188217,None,7.6|2010-11-19T10:40:00Z,53.516566,8.188217,None,7.58|2010-11-19T10:50:00Z,53.516566,8.188217,None,7.5|2010-11-19T11:00:00Z,53.516566,8.188217,None,7.39|2010-11-19T11:10:00Z,53.516566,8.188217,None,7.25|2010-11-19T11:20:00Z,53.516566,8.188217,None,7.13|2010-11-19T11:30:00Z,53.516566,8.188217,None,7.04|2010-11-19T11:40:00Z,53.516566,8.188217,None,6.984|2010-11-19T11:50:00Z,53.516566,8.188217,None,6.953|2010-11-19T12:00:00Z,53.516566,8.188217,None,7.01|2010-11-19T12:10:00Z,53.516566,8.188217,None,7.16|2010-11-19T12:20:00Z,53.516566,8.188217,None,7.24|2010-11-19T12:30:00Z,53.516566,8.188217,None,7.31|2010-11-19T12:40:00Z,53.516566,8.188217,None,7.38|2010-11-19T12:50:00Z,53.516566,8.188217,None,7.43|2010-11-19T13:00:00Z,53.516566,8.188217,None,7.46|2010-11-19T13:10:00Z,53.516566,8.188217,None,7.47|2010-11-19T13:20:00Z,53.516566,8.188217,None,7.47|2010-11-19T13:30:00Z,53.516566,8.188217,None,7.45|2010-11-19T13:40:00Z,53.516566,8.188217,None,7.45|2010-11-19T13:50:00Z,53.516566,8.188217,None,7.46|2010-11-19T14:00:00Z,53.516566,8.188217,None,7.47|2010-11-19T14:10:00Z,53.516566,8.188217,None,7.49|2010-11-19T14:20:00Z,53.516566,8.188217,None,7.5|2010-11-19T14:30:00Z,53.516566,8.188217,None,7.52|2010-11-19T14:40:00Z,53.516566,8.188217,None,7.53|2010-11-19T14:50:00Z,53.516566,8.188217,None,7.52|2010-11-19T15:00:00Z,53.516566,8.188217,None,7.51|2010-11-19T15:10:00Z,53.516566,8.188217,None,7.53|2010-11-19T15:20:00Z,53.516566,8.188217,None,7.54|2010-11-19T15:30:00Z,53.516566,8.188217,None,7.52|2010-11-19T15:40:00Z,53.516566,8.188217,None,7.52|2010-11-19T15:50:00Z,53.516566,8.188217,None,7.52|2010-11-19T16:00:00Z,53.516566,8.188217,None,7.5|2010-11-19T16:10:00Z,53.516566,8.188217,None,7.5|2010-11-19T16:20:00Z,53.516566,8.188217,None,7.5|2010-11-19T16:30:00Z,53.516566,8.188217,None,7.5|2010-11-19T16:40:00Z,53.516566,8.188217,None,7.43|2010-11-19T16:50:00Z,53.516566,8.188217,None,7.42|2010-11-19T17:00:00Z,53.516566,8.188217,None,7.15|2010-11-19T17:10:00Z,53.516566,8.188217,None,7.1|2010-11-19T17:20:00Z,53.516566,8.188217,None,7.13|2010-11-19T17:30:00Z,53.516566,8.188217,None,7.09|2010-11-19T17:40:00Z,53.516566,8.188217,None,7.11|2010-11-19T17:50:00Z,53.516566,8.188217,None,7.12|2010-11-19T18:00:00Z,53.516566,8.188217,None,7.15|2010-11-19T18:10:00Z,53.516566,8.188217,None,7.18|2010-11-19T18:20:00Z,53.516566,8.188217,None,7.19|2010-11-19T18:30:00Z,53.516566,8.188217,None,7.2|2010-11-19T18:40:00Z,53.516566,8.188217,None,7.25|2010-11-19T18:50:00Z,53.516566,8.188217,None,7.31|2010-11-19T19:00:00Z,53.516566,8.188217,None,7.27|2010-11-19T19:10:00Z,53.516566,8.188217,None,7.3|2010-11-19T19:20:00Z,53.516566,8.188217,None,7.4|2010-11-19T19:30:00Z,53.516566,8.188217,None,7.41|2010-11-19T19:40:00Z,53.516566,8.188217,None,7.39|2010-11-19T19:50:00Z,53.516566,8.188217,None,7.4|2010-11-19T20:00:00Z,53.516566,8.188217,None,7.43|2010-11-19T20:10:00Z,53.516566,8.188217,None,7.47|2010-11-19T20:20:00Z,53.516566,8.188217,None,7.51|2010-11-19T20:30:00Z,53.516566,8.188217,None,7.56|2010-11-19T20:40:00Z,53.516566,8.188217,None,7.59|2010-11-19T20:50:00Z,53.516566,8.188217,None,7.61|2010-11-19T21:00:00Z,53.516566,8.188217,None,7.61|2010-11-19T21:10:00Z,53.516566,8.188217,None,7.59|2010-11-19T21:20:00Z,53.516566,8.188217,None,7.59|2010-11-19T21:30:00Z,53.516566,8.188217,None,7.6|2010-11-19T21:40:00Z,53.516566,8.188217,None,7.6|2010-11-19T21:50:00Z,53.516566,8.188217,None,7.61|2010-11-19T22:00:00Z,53.516566,8.188217,None,7.62|2010-11-19T22:10:00Z,53.516566,8.188217,None,7.65|2010-11-19T22:20:00Z,53.516566,8.188217,None,7.64|2010-11-19T22:30:00Z,53.516566,8.188217,None,7.63|2010-11-19T22:40:00Z,53.516566,8.188217,None,7.63|2010-11-19T22:50:00Z,53.516566,8.188217,None,7.6|2010-11-19T23:00:00Z,53.516566,8.188217,None,7.58|2010-11-19T23:10:00Z,53.516566,8.188217,None,7.57|2010-11-19T23:20:00Z,53.516566,8.188217,None,7.52|2010-11-19T23:30:00Z,53.516566,8.188217,None,7.37|2010-11-19T23:40:00Z,53.516566,8.188217,None,7.23|2010-11-19T23:50:00Z,53.516566,8.188217,None,7.14|2010-11-20T00:00:00Z,53.516566,8.188217,None,7.06|2010-11-20T00:10:00Z,53.516566,8.188217,None,7.03|2010-11-20T00:20:00Z,53.516566,8.188217,None,7.0|2010-11-20T00:30:00Z,53.516566,8.188217,None,7.02|2010-11-20T00:40:00Z,53.516566,8.188217,None,7.08|2010-11-20T00:50:00Z,53.516566,8.188217,None,7.15|2010-11-20T01:00:00Z,53.516566,8.188217,None,7.2|2010-11-20T01:10:00Z,53.516566,8.188217,None,7.24|2010-11-20T01:20:00Z,53.516566,8.188217,None,7.27|2010-11-20T01:30:00Z,53.516566,8.188217,None,7.3|2010-11-20T01:40:00Z,53.516566,8.188217,None,7.32|2010-11-20T01:50:00Z,53.516566,8.188217,None,7.32|2010-11-20T02:00:00Z,53.516566,8.188217,None,7.31|2010-11-20T02:10:00Z,53.516566,8.188217,None,7.33|2010-11-20T02:20:00Z,53.516566,8.188217,None,7.37|2010-11-20T02:30:00Z,53.516566,8.188217,None,7.41|2010-11-20T02:40:00Z,53.516566,8.188217,None,7.45|2010-11-20T02:50:00Z,53.516566,8.188217,None,7.46|2010-11-20T03:00:00Z,53.516566,8.188217,None,7.43|2010-11-20T03:10:00Z,53.516566,8.188217,None,7.42|2010-11-20T03:20:00Z,53.516566,8.188217,None,7.42|2010-11-20T03:30:00Z,53.516566,8.188217,None,7.41|2010-11-20T03:40:00Z,53.516566,8.188217,None,7.4|2010-11-20T03:50:00Z,53.516566,8.188217,None,7.39|2010-11-20T04:00:00Z,53.516566,8.188217,None,7.39|2010-11-20T04:10:00Z,53.516566,8.188217,None,7.41|2010-11-20T04:20:00Z,53.516566,8.188217,None,7.44|2010-11-20T04:30:00Z,53.516566,8.188217,None,7.45|2010-11-20T04:40:00Z,53.516566,8.188217,None,7.45|2010-11-20T04:50:00Z,53.516566,8.188217,None,7.41|2010-11-20T05:00:00Z,53.516566,8.188217,None,7.38|2010-11-20T05:10:00Z,53.516566,8.188217,None,7.3|2010-11-20T05:20:00Z,53.516566,8.188217,None,7.31|2010-11-20T05:30:00Z,53.516566,8.188217,None,7.09|2010-11-20T05:40:00Z,53.516566,8.188217,None,7.08|2010-11-20T05:50:00Z,53.516566,8.188217,None,7.05|2010-11-20T06:00:00Z,53.516566,8.188217,None,7.03|2010-11-20T06:10:00Z,53.516566,8.188217,None,7.01|2010-11-20T06:20:00Z,53.516566,8.188217,None,6.998|2010-11-20T06:30:00Z,53.516566,8.188217,None,7.05|2010-11-20T06:40:00Z,53.516566,8.188217,None,7.06|2010-11-20T06:50:00Z,53.516566,8.188217,None,7.12|2010-11-20T07:00:00Z,53.516566,8.188217,None,7.18|2010-11-20T07:10:00Z,53.516566,8.188217,None,7.24|2010-11-20T07:20:00Z,53.516566,8.188217,None,7.15|2010-11-20T07:30:00Z,53.516566,8.188217,None,7.14|2010-11-20T07:40:00Z,53.516566,8.188217,None,7.24|2010-11-20T07:50:00Z,53.516566,8.188217,None,7.3|2010-11-20T08:00:00Z,53.516566,8.188217,None,7.32|2010-11-20T08:10:00Z,53.516566,8.188217,None,7.33|2010-11-20T08:20:00Z,53.516566,8.188217,None,7.36|2010-11-20T08:30:00Z,53.516566,8.188217,None,7.41|2010-11-20T08:40:00Z,53.516566,8.188217,None,7.46|2010-11-20T08:50:00Z,53.516566,8.188217,None,7.5|2010-11-20T09:00:00Z,53.516566,8.188217,None,7.53|2010-11-20T09:10:00Z,53.516566,8.188217,None,7.52|2010-11-20T09:20:00Z,53.516566,8.188217,None,7.51|2010-11-20T09:30:00Z,53.516566,8.188217,None,7.52|2010-11-20T09:40:00Z,53.516566,8.188217,None,7.53|2010-11-20T09:50:00Z,53.516566,8.188217,None,7.54|2010-11-20T10:00:00Z,53.516566,8.188217,None,7.63|2010-11-20T10:10:00Z,53.516566,8.188217,None,7.68|2010-11-20T10:20:00Z,53.516566,8.188217,None,7.72|2010-11-20T10:30:00Z,53.516566,8.188217,None,7.73|2010-11-20T10:40:00Z,53.516566,8.188217,None,7.74|2010-11-20T10:50:00Z,53.516566,8.188217,None,7.74|2010-11-20T11:00:00Z,53.516566,8.188217,None,7.74|2010-11-20T11:10:00Z,53.516566,8.188217,None,7.71|2010-11-20T11:20:00Z,53.516566,8.188217,None,7.61|2010-11-20T11:30:00Z,53.516566,8.188217,None,7.52|2010-11-20T11:40:00Z,53.516566,8.188217,None,7.47|2010-11-20T11:50:00Z,53.516566,8.188217,None,7.35|2010-11-20T12:00:00Z,53.516566,8.188217,None,7.2|2010-11-20T12:10:00Z,53.516566,8.188217,None,7.07|2010-11-20T12:20:00Z,53.516566,8.188217,None,6.932|2010-11-20T12:30:00Z,53.516566,8.188217,None,6.872|2010-11-20T12:40:00Z,53.516566,8.188217,None,6.848|2010-11-20T12:50:00Z,53.516566,8.188217,None,6.865|2010-11-20T13:00:00Z,53.516566,8.188217,None,7.0|2010-11-20T13:10:00Z,53.516566,8.188217,None,7.09|2010-11-20T13:20:00Z,53.516566,8.188217,None,7.15|2010-11-20T13:30:00Z,53.516566,8.188217,None,7.19|2010-11-20T13:40:00Z,53.516566,8.188217,None,7.25|2010-11-20T13:50:00Z,53.516566,8.188217,None,7.3|2010-11-20T14:00:00Z,53.516566,8.188217,None,7.32|2010-11-20T14:10:00Z,53.516566,8.188217,None,7.35|2010-11-20T14:20:00Z,53.516566,8.188217,None,7.36|2010-11-20T14:30:00Z,53.516566,8.188217,None,7.35|2010-11-20T14:40:00Z,53.516566,8.188217,None,7.33|2010-11-20T14:50:00Z,53.516566,8.188217,None,7.34|2010-11-20T15:00:00Z,53.516566,8.188217,None,7.35|2010-11-20T15:10:00Z,53.516566,8.188217,None,7.38|2010-11-20T15:20:00Z,53.516566,8.188217,None,7.38|2010-11-20T15:30:00Z,53.516566,8.188217,None,7.39|2010-11-20T15:40:00Z,53.516566,8.188217,None,7.42|2010-11-20T15:50:00Z,53.516566,8.188217,None,7.44|2010-11-20T16:00:00Z,53.516566,8.188217,None,7.46|2010-11-20T16:10:00Z,53.516566,8.188217,None,7.43|2010-11-20T16:20:00Z,53.516566,8.188217,None,7.46|2010-11-20T16:30:00Z,53.516566,8.188217,None,7.46|2010-11-20T16:40:00Z,53.516566,8.188217,None,7.47|2010-11-20T16:50:00Z,53.516566,8.188217,None,7.47|2010-11-20T17:00:00Z,53.516566,8.188217,None,7.45|2010-11-20T17:10:00Z,53.516566,8.188217,None,7.31|2010-11-20T17:20:00Z,53.516566,8.188217,None,7.27|2010-11-20T17:30:00Z,53.516566,8.188217,None,7.21|2010-11-20T17:40:00Z,53.516566,8.188217,None,7.15|2010-11-20T17:50:00Z,53.516566,8.188217,None,7.11|2010-11-20T18:00:00Z,53.516566,8.188217,None,7.14|2010-11-20T18:10:00Z,53.516566,8.188217,None,7.14|2010-11-20T18:20:00Z,53.516566,8.188217,None,7.12|2010-11-20T18:30:00Z,53.516566,8.188217,None,7.07|2010-11-20T18:40:00Z,53.516566,8.188217,None,7.04|2010-11-20T18:50:00Z,53.516566,8.188217,None,7.05|2010-11-20T19:00:00Z,53.516566,8.188217,None,7.07|2010-11-20T19:10:00Z,53.516566,8.188217,None,7.1|2010-11-20T19:20:00Z,53.516566,8.188217,None,7.13|2010-11-20T19:30:00Z,53.516566,8.188217,None,7.15|2010-11-20T19:40:00Z,53.516566,8.188217,None,7.15|2010-11-20T19:50:00Z,53.516566,8.188217,None,7.23|2010-11-20T20:00:00Z,53.516566,8.188217,None,7.25|2010-11-20T20:10:00Z,53.516566,8.188217,None,7.27|2010-11-20T20:20:00Z,53.516566,8.188217,None,7.28|2010-11-20T20:30:00Z,53.516566,8.188217,None,7.31|2010-11-20T20:40:00Z,53.516566,8.188217,None,7.34|2010-11-20T20:50:00Z,53.516566,8.188217,None,7.38|2010-11-20T21:00:00Z,53.516566,8.188217,None,7.42|2010-11-20T21:10:00Z,53.516566,8.188217,None,7.45|2010-11-20T21:20:00Z,53.516566,8.188217,None,7.48|2010-11-20T21:30:00Z,53.516566,8.188217,None,7.5|2010-11-20T21:40:00Z,53.516566,8.188217,None,7.5|2010-11-20T21:50:00Z,53.516566,8.188217,None,7.48|2010-11-20T22:00:00Z,53.516566,8.188217,None,7.45|2010-11-20T22:10:00Z,53.516566,8.188217,None,7.45|2010-11-20T22:20:00Z,53.516566,8.188217,None,7.47|2010-11-20T22:30:00Z,53.516566,8.188217,None,7.52|2010-11-20T22:40:00Z,53.516566,8.188217,None,7.57|2010-11-20T22:50:00Z,53.516566,8.188217,None,7.62|2010-11-20T23:00:00Z,53.516566,8.188217,None,7.63|2010-11-20T23:10:00Z,53.516566,8.188217,None,7.63|2010-11-20T23:20:00Z,53.516566,8.188217,None,7.62|2010-11-20T23:30:00Z,53.516566,8.188217,None,7.62|2010-11-20T23:40:00Z,53.516566,8.188217,None,7.56|2010-11-20T23:50:00Z,53.516566,8.188217,None,7.48|2010-11-21T00:00:00Z,53.516566,8.188217,None,7.44|2010-11-21T00:10:00Z,53.516566,8.188217,None,7.37|2010-11-21T00:20:00Z,53.516566,8.188217,None,7.22|2010-11-21T00:30:00Z,53.516566,8.188217,None,7.13|2010-11-21T00:40:00Z,53.516566,8.188217,None,7.03|2010-11-21T00:50:00Z,53.516566,8.188217,None,6.95|2010-11-21T01:00:00Z,53.516566,8.188217,None,6.928|2010-11-21T01:10:00Z,53.516566,8.188217,None,6.934|2010-11-21T01:20:00Z,53.516566,8.188217,None,6.983|2010-11-21T01:30:00Z,53.516566,8.188217,None,7.04|2010-11-21T01:40:00Z,53.516566,8.188217,None,7.07|2010-11-21T01:50:00Z,53.516566,8.188217,None,7.11|2010-11-21T02:00:00Z,53.516566,8.188217,None,7.16|2010-11-21T02:10:00Z,53.516566,8.188217,None,7.22|2010-11-21T02:20:00Z,53.516566,8.188217,None,7.25|2010-11-21T02:30:00Z,53.516566,8.188217,None,7.25|2010-11-21T02:40:00Z,53.516566,8.188217,None,7.24|2010-11-21T02:50:00Z,53.516566,8.188217,None,7.24|2010-11-21T03:00:00Z,53.516566,8.188217,None,7.26|2010-11-21T03:10:00Z,53.516566,8.188217,None,7.32|2010-11-21T03:20:00Z,53.516566,8.188217,None,7.35|2010-11-21T03:30:00Z,53.516566,8.188217,None,7.32|2010-11-21T03:40:00Z,53.516566,8.188217,None,7.34|2010-11-21T03:50:00Z,53.516566,8.188217,None,7.33|2010-11-21T04:00:00Z,53.516566,8.188217,None,7.34|2010-11-21T04:10:00Z,53.516566,8.188217,None,7.36|2010-11-21T04:20:00Z,53.516566,8.188217,None,7.39|2010-11-21T04:30:00Z,53.516566,8.188217,None,7.39|2010-11-21T04:40:00Z,53.516566,8.188217,None,7.4|2010-11-21T04:50:00Z,53.516566,8.188217,None,7.39|2010-11-21T05:00:00Z,53.516566,8.188217,None,7.35|2010-11-21T05:10:00Z,53.516566,8.188217,None,7.31|2010-11-21T05:20:00Z,53.516566,8.188217,None,7.27|2010-11-21T05:30:00Z,53.516566,8.188217,None,7.28|2010-11-21T05:40:00Z,53.516566,8.188217,None,7.32|2010-11-21T05:50:00Z,53.516566,8.188217,None,7.29|2010-11-21T06:00:00Z,53.516566,8.188217,None,7.25|2010-11-21T06:10:00Z,53.516566,8.188217,None,7.16|2010-11-21T06:20:00Z,53.516566,8.188217,None,7.13|2010-11-21T06:30:00Z,53.516566,8.188217,None,7.13|2010-11-21T06:40:00Z,53.516566,8.188217,None,7.1|2010-11-21T06:50:00Z,53.516566,8.188217,None,7.08|2010-11-21T07:00:00Z,53.516566,8.188217,None,7.06|2010-11-21T07:10:00Z,53.516566,8.188217,None,7.1|2010-11-21T07:20:00Z,53.516566,8.188217,None,7.05|2010-11-21T07:30:00Z,53.516566,8.188217,None,7.04|2010-11-21T07:40:00Z,53.516566,8.188217,None,7.09|2010-11-21T07:50:00Z,53.516566,8.188217,None,7.14|2010-11-21T08:00:00Z,53.516566,8.188217,None,7.13|2010-11-21T08:10:00Z,53.516566,8.188217,None,7.11|2010-11-21T08:20:00Z,53.516566,8.188217,None,7.16|2010-11-21T08:30:00Z,53.516566,8.188217,None,7.22|2010-11-21T08:40:00Z,53.516566,8.188217,None,7.23|2010-11-21T08:50:00Z,53.516566,8.188217,None,7.25|2010-11-21T09:00:00Z,53.516566,8.188217,None,7.29|2010-11-21T09:10:00Z,53.516566,8.188217,None,7.32|2010-11-21T09:20:00Z,53.516566,8.188217,None,7.37|2010-11-21T09:30:00Z,53.516566,8.188217,None,7.42|2010-11-21T09:40:00Z,53.516566,8.188217,None,7.45|2010-11-21T09:50:00Z,53.516566,8.188217,None,7.45|2010-11-21T10:00:00Z,53.516566,8.188217,None,7.45|2010-11-21T10:10:00Z,53.516566,8.188217,None,7.45|2010-11-21T10:20:00Z,53.516566,8.188217,None,7.44|2010-11-21T10:30:00Z,53.516566,8.188217,None,7.46|2010-11-21T10:40:00Z,53.516566,8.188217,None,7.48|2010-11-21T10:50:00Z,53.516566,8.188217,None,7.5|2010-11-21T11:00:00Z,53.516566,8.188217,None,7.53|2010-11-21T11:10:00Z,53.516566,8.188217,None,7.58|2010-11-21T11:20:00Z,53.516566,8.188217,None,7.6|2010-11-21T11:30:00Z,53.516566,8.188217,None,7.61|2010-11-21T11:40:00Z,53.516566,8.188217,None,7.61|2010-11-21T11:50:00Z,53.516566,8.188217,None,7.62|2010-11-21T12:00:00Z,53.516566,8.188217,None,7.59|2010-11-21T12:10:00Z,53.516566,8.188217,None,7.54|2010-11-21T12:20:00Z,53.516566,8.188217,None,7.47|2010-11-21T12:30:00Z,53.516566,8.188217,None,7.44|2010-11-21T12:40:00Z,53.516566,8.188217,None,7.36|2010-11-21T12:50:00Z,53.516566,8.188217,None,7.23|2010-11-21T13:00:00Z,53.516566,8.188217,None,7.1|2010-11-21T13:10:00Z,53.516566,8.188217,None,7.01|2010-11-21T13:20:00Z,53.516566,8.188217,None,6.972|2010-11-21T13:30:00Z,53.516566,8.188217,None,6.937|2010-11-21T13:40:00Z,53.516566,8.188217,None,6.916|2010-11-21T13:50:00Z,53.516566,8.188217,None,6.944|2010-11-21T14:00:00Z,53.516566,8.188217,None,6.983|2010-11-21T14:10:00Z,53.516566,8.188217,None,7.01|2010-11-21T14:20:00Z,53.516566,8.188217,None,7.05|2010-11-21T14:30:00Z,53.516566,8.188217,None,7.08|2010-11-21T14:40:00Z,53.516566,8.188217,None,7.09|2010-11-21T14:50:00Z,53.516566,8.188217,None,7.09|2010-11-21T15:00:00Z,53.516566,8.188217,None,7.1|2010-11-21T15:10:00Z,53.516566,8.188217,None,7.1|2010-11-21T15:20:00Z,53.516566,8.188217,None,7.12|2010-11-21T15:30:00Z,53.516566,8.188217,None,7.13|2010-11-21T15:40:00Z,53.516566,8.188217,None,7.19|2010-11-21T15:50:00Z,53.516566,8.188217,None,7.19|2010-11-21T16:00:00Z,53.516566,8.188217,None,7.19|2010-11-21T16:10:00Z,53.516566,8.188217,None,7.21|2010-11-21T16:20:00Z,53.516566,8.188217,None,7.26|2010-11-21T16:30:00Z,53.516566,8.188217,None,7.19|2010-11-21T16:40:00Z,53.516566,8.188217,None,7.12|2010-11-21T16:50:00Z,53.516566,8.188217,None,7.21|2010-11-21T17:00:00Z,53.516566,8.188217,None,7.23|2010-11-21T17:10:00Z,53.516566,8.188217,None,7.24|2010-11-21T17:20:00Z,53.516566,8.188217,None,7.25|2010-11-21T17:30:00Z,53.516566,8.188217,None,7.24|2010-11-21T17:40:00Z,53.516566,8.188217,None,7.21|2010-11-21T17:50:00Z,53.516566,8.188217,None,7.12|2010-11-21T18:00:00Z,53.516566,8.188217,None,7.11|2010-11-21T18:10:00Z,53.516566,8.188217,None,7.19|2010-11-21T18:20:00Z,53.516566,8.188217,None,7.14|2010-11-21T18:30:00Z,53.516566,8.188217,None,7.05|2010-11-21T18:40:00Z,53.516566,8.188217,None,7.12|2010-11-21T18:50:00Z,53.516566,8.188217,None,7.13|2010-11-21T19:00:00Z,53.516566,8.188217,None,7.13|2010-11-21T19:10:00Z,53.516566,8.188217,None,7.13|2010-11-21T19:20:00Z,53.516566,8.188217,None,7.08|2010-11-21T19:30:00Z,53.516566,8.188217,None,7.04|2010-11-21T19:40:00Z,53.516566,8.188217,None,7.06|2010-11-21T19:50:00Z,53.516566,8.188217,None,7.06|2010-11-21T20:00:00Z,53.516566,8.188217,None,7.05|2010-11-21T20:10:00Z,53.516566,8.188217,None,7.12|2010-11-21T20:20:00Z,53.516566,8.188217,None,7.14|2010-11-21T20:30:00Z,53.516566,8.188217,None,7.15|2010-11-21T20:40:00Z,53.516566,8.188217,None,7.15|2010-11-21T20:50:00Z,53.516566,8.188217,None,7.17|2010-11-21T21:00:00Z,53.516566,8.188217,None,7.19|2010-11-21T21:10:00Z,53.516566,8.188217,None,7.21|2010-11-21T21:20:00Z,53.516566,8.188217,None,7.25|2010-11-21T21:30:00Z,53.516566,8.188217,None,7.32|2010-11-21T21:40:00Z,53.516566,8.188217,None,7.36|2010-11-21T21:50:00Z,53.516566,8.188217,None,7.39|2010-11-21T22:00:00Z,53.516566,8.188217,None,7.41|2010-11-21T22:10:00Z,53.516566,8.188217,None,7.4|2010-11-21T22:20:00Z,53.516566,8.188217,None,7.38|2010-11-21T22:30:00Z,53.516566,8.188217,None,7.39|2010-11-21T22:40:00Z,53.516566,8.188217,None,7.4|2010-11-21T22:50:00Z,53.516566,8.188217,None,7.4|2010-11-21T23:00:00Z,53.516566,8.188217,None,7.42|2010-11-21T23:10:00Z,53.516566,8.188217,None,7.45|2010-11-21T23:20:00Z,53.516566,8.188217,None,7.46|2010-11-21T23:30:00Z,53.516566,8.188217,None,7.47|2010-11-21T23:40:00Z,53.516566,8.188217,None,7.48|2010-11-21T23:50:00Z,53.516566,8.188217,None,7.5|2010-11-22T00:00:00Z,53.516566,8.188217,None,7.51|2010-11-22T00:10:00Z,53.516566,8.188217,None,7.5|2010-11-22T00:20:00Z,53.516566,8.188217,None,7.47|2010-11-22T00:30:00Z,53.516566,8.188217,None,7.42|2010-11-22T00:40:00Z,53.516566,8.188217,None,7.37|2010-11-22T00:50:00Z,53.516566,8.188217,None,7.34|2010-11-22T01:00:00Z,53.516566,8.188217,None,7.23|2010-11-22T01:10:00Z,53.516566,8.188217,None,7.08|2010-11-22T01:20:00Z,53.516566,8.188217,None,6.997|2010-11-22T01:30:00Z,53.516566,8.188217,None,6.918|2010-11-22T01:40:00Z,53.516566,8.188217,None,6.865|2010-11-22T01:50:00Z,53.516566,8.188217,None,6.839|2010-11-22T02:00:00Z,53.516566,8.188217,None,6.838|2010-11-22T02:10:00Z,53.516566,8.188217,None,6.851|2010-11-22T02:20:00Z,53.516566,8.188217,None,6.937|2010-11-22T02:30:00Z,53.516566,8.188217,None,7.02|2010-11-22T02:40:00Z,53.516566,8.188217,None,7.03|2010-11-22T02:50:00Z,53.516566,8.188217,None,7.03|2010-11-22T03:00:00Z,53.516566,8.188217,None,7.02|2010-11-22T03:10:00Z,53.516566,8.188217,None,7.01|2010-11-22T03:20:00Z,53.516566,8.188217,None,6.988|2010-11-22T03:30:00Z,53.516566,8.188217,None,6.993|2010-11-22T03:40:00Z,53.516566,8.188217,None,7.04|2010-11-22T03:50:00Z,53.516566,8.188217,None,7.09|2010-11-22T04:00:00Z,53.516566,8.188217,None,7.09|2010-11-22T04:10:00Z,53.516566,8.188217,None,7.1|2010-11-22T04:20:00Z,53.516566,8.188217,None,7.09|2010-11-22T04:30:00Z,53.516566,8.188217,None,7.09|2010-11-22T04:40:00Z,53.516566,8.188217,None,7.09|2010-11-22T04:50:00Z,53.516566,8.188217,None,6.964|2010-11-22T05:00:00Z,53.516566,8.188217,None,6.932|2010-11-22T05:10:00Z,53.516566,8.188217,None,7.03|2010-11-22T05:20:00Z,53.516566,8.188217,None,7.04|2010-11-22T05:30:00Z,53.516566,8.188217,None,7.08|2010-11-22T05:40:00Z,53.516566,8.188217,None,7.08|2010-11-22T05:50:00Z,53.516566,8.188217,None,7.1|2010-11-22T06:00:00Z,53.516566,8.188217,None,7.09|2010-11-22T06:10:00Z,53.516566,8.188217,None,7.0|2010-11-22T06:20:00Z,53.516566,8.188217,None,6.937|2010-11-22T06:30:00Z,53.516566,8.188217,None,6.939|2010-11-22T06:40:00Z,53.516566,8.188217,None,6.926|2010-11-22T06:50:00Z,53.516566,8.188217,None,6.954|2010-11-22T07:00:00Z,53.516566,8.188217,None,6.884|2010-11-22T07:10:00Z,53.516566,8.188217,None,7.01|2010-11-22T07:20:00Z,53.516566,8.188217,None,6.961|2010-11-22T07:30:00Z,53.516566,8.188217,None,6.904|2010-11-22T07:40:00Z,53.516566,8.188217,None,6.922|2010-11-22T07:50:00Z,53.516566,8.188217,None,6.921|2010-11-22T08:00:00Z,53.516566,8.188217,None,6.894|2010-11-22T08:10:00Z,53.516566,8.188217,None,6.92|2010-11-22T08:20:00Z,53.516566,8.188217,None,6.927|2010-11-22T08:30:00Z,53.516566,8.188217,None,6.967|2010-11-22T08:40:00Z,53.516566,8.188217,None,7.0|2010-11-22T08:50:00Z,53.516566,8.188217,None,7.04|2010-11-22T09:00:00Z,53.516566,8.188217,None,7.04|2010-11-22T09:10:00Z,53.516566,8.188217,None,7.05|2010-11-22T09:20:00Z,53.516566,8.188217,None,7.07|2010-11-22T09:30:00Z,53.516566,8.188217,None,7.11|2010-11-22T09:40:00Z,53.516566,8.188217,None,7.17|2010-11-22T09:50:00Z,53.516566,8.188217,None,7.22|2010-11-22T10:00:00Z,53.516566,8.188217,None,7.27|2010-11-22T10:10:00Z,53.516566,8.188217,None,7.3|2010-11-22T10:20:00Z,53.516566,8.188217,None,7.31|2010-11-22T10:30:00Z,53.516566,8.188217,None,7.33|2010-11-22T10:40:00Z,53.516566,8.188217,None,7.33|2010-11-22T10:50:00Z,53.516566,8.188217,None,7.32|2010-11-22T11:00:00Z,53.516566,8.188217,None,7.32|2010-11-22T11:10:00Z,53.516566,8.188217,None,7.33|2010-11-22T11:20:00Z,53.516566,8.188217,None,7.34|2010-11-22T11:30:00Z,53.516566,8.188217,None,7.36|2010-11-22T11:40:00Z,53.516566,8.188217,None,7.38|2010-11-22T11:50:00Z,53.516566,8.188217,None,7.39|2010-11-22T12:00:00Z,53.516566,8.188217,None,7.4|2010-11-22T12:10:00Z,53.516566,8.188217,None,7.42|2010-11-22T12:20:00Z,53.516566,8.188217,None,7.42|2010-11-22T12:30:00Z,53.516566,8.188217,None,7.41|2010-11-22T12:40:00Z,53.516566,8.188217,None,7.41|2010-11-22T12:50:00Z,53.516566,8.188217,None,7.39|2010-11-22T13:00:00Z,53.516566,8.188217,None,7.36|2010-11-22T13:10:00Z,53.516566,8.188217,None,7.34|2010-11-22T13:20:00Z,53.516566,8.188217,None,7.33|2010-11-22T13:30:00Z,53.516566,8.188217,None,7.26|2010-11-22T13:40:00Z,53.516566,8.188217,None,7.13|2010-11-22T13:50:00Z,53.516566,8.188217,None,6.995|2010-11-22T14:00:00Z,53.516566,8.188217,None,6.901|2010-11-22T14:10:00Z,53.516566,8.188217,None,6.563|2010-11-22T14:20:00Z,53.516566,8.188217,None,-4.359|2010-11-22T14:30:00Z,53.516566,8.188217,None,-4.87|2010-11-22T14:40:00Z,53.516566,8.188217,None,-4.877|2010-11-22T14:50:00Z,53.516566,8.188217,None,-4.896|2010-11-22T15:00:00Z,53.516566,8.188217,None,-4.912|2010-11-22T15:10:00Z,53.516566,8.188217,None,-4.92|2010-11-22T15:20:00Z,53.516566,8.188217,None,-4.933|2010-11-22T15:30:00Z,53.516566,8.188217,None,-4.911|2010-11-22T15:40:00Z,53.516566,8.188217,None,-4.923|2010-11-22T15:50:00Z,53.516566,8.188217,None,-4.926|2010-11-22T16:00:00Z,53.516566,8.188217,None,-4.929|2010-11-22T16:10:00Z,53.516566,8.188217,None,-4.93|2010-11-22T16:20:00Z,53.516566,8.188217,None,-4.93|2010-11-22T16:30:00Z,53.516566,8.188217,None,-4.934|2010-11-22T16:40:00Z,53.516566,8.188217,None,-4.926|2010-11-22T16:50:00Z,53.516566,8.188217,None,-4.932|2010-11-22T17:00:00Z,53.516566,8.188217,None,-4.929|2010-11-22T17:10:00Z,53.516566,8.188217,None,-4.937|2010-11-22T17:20:00Z,53.516566,8.188217,None,-4.929|2010-11-22T17:30:00Z,53.516566,8.188217,None,-4.939|2010-11-22T17:40:00Z,53.516566,8.188217,None,-4.941|2010-11-22T17:50:00Z,53.516566,8.188217,None,-4.94|2010-11-22T18:00:00Z,53.516566,8.188217,None,-4.941|2010-11-22T18:10:00Z,53.516566,8.188217,None,-4.944|2010-11-22T18:20:00Z,53.516566,8.188217,None,-4.95|2010-11-22T18:30:00Z,53.516566,8.188217,None,-4.955|2010-11-22T18:40:00Z,53.516566,8.188217,None,-4.954|2010-11-22T18:50:00Z,53.516566,8.188217,None,-4.946|2010-11-22T19:00:00Z,53.516566,8.188217,None,-4.957|2010-11-22T19:10:00Z,53.516566,8.188217,None,-4.951|2010-11-22T19:20:00Z,53.516566,8.188217,None,-4.95|2010-11-22T19:30:00Z,53.516566,8.188217,None,-4.957|2010-11-22T19:40:00Z,53.516566,8.188217,None,-4.961|2010-11-22T19:50:00Z,53.516566,8.188217,None,-4.959|2010-11-22T20:00:00Z,53.516566,8.188217,None,-4.957|2010-11-22T20:10:00Z,53.516566,8.188217,None,-4.964|2010-11-22T20:20:00Z,53.516566,8.188217,None,-4.967|2010-11-22T20:30:00Z,53.516566,8.188217,None,-4.967|2010-11-22T20:40:00Z,53.516566,8.188217,None,-4.963|2010-11-22T20:50:00Z,53.516566,8.188217,None,-4.968|2010-11-22T21:00:00Z,53.516566,8.188217,None,-4.96|2010-11-22T21:10:00Z,53.516566,8.188217,None,-4.959|2010-11-22T21:20:00Z,53.516566,8.188217,None,-4.966|2010-11-22T21:30:00Z,53.516566,8.188217,None,-4.962|2010-11-22T21:40:00Z,53.516566,8.188217,None,-4.968|2010-11-22T21:50:00Z,53.516566,8.188217,None,-4.973|2010-11-22T22:00:00Z,53.516566,8.188217,None,-4.966|2010-11-22T22:10:00Z,53.516566,8.188217,None,-4.974|2010-11-22T22:20:00Z,53.516566,8.188217,None,-4.97|2010-11-22T22:30:00Z,53.516566,8.188217,None,-4.975|2010-11-22T22:40:00Z,53.516566,8.188217,None,-4.976|2010-11-22T22:50:00Z,53.516566,8.188217,None,-4.974|2010-11-22T23:00:00Z,53.516566,8.188217,None,-4.98|2010-11-22T23:10:00Z,53.516566,8.188217,None,-4.972|2010-11-22T23:20:00Z,53.516566,8.188217,None,-4.976|2010-11-22T23:30:00Z,53.516566,8.188217,None,-4.981|2010-11-22T23:40:00Z,53.516566,8.188217,None,-4.985|2010-11-22T23:50:00Z,53.516566,8.188217,None,-4.985|2010-11-23T00:00:00Z,53.516566,8.188217,None,-4.978|2010-11-23T00:10:00Z,53.516566,8.188217,None,-4.987|2010-11-23T00:20:00Z,53.516566,8.188217,None,-4.986|2010-11-23T00:30:00Z,53.516566,8.188217,None,-4.996|2010-11-23T00:40:00Z,53.516566,8.188217,None,-4.996|2010-11-23T00:50:00Z,53.516566,8.188217,None,-4.989|2010-11-23T01:00:00Z,53.516566,8.188217,None,-4.989|2010-11-23T01:10:00Z,53.516566,8.188217,None,-4.992|2010-11-23T01:20:00Z,53.516566,8.188217,None,-4.986|2010-11-23T01:30:00Z,53.516566,8.188217,None,-4.98|2010-11-23T01:40:00Z,53.516566,8.188217,None,-4.977|2010-11-23T01:50:00Z,53.516566,8.188217,None,-4.975|2010-11-23T02:00:00Z,53.516566,8.188217,None,-4.974|2010-11-23T02:10:00Z,53.516566,8.188217,None,-4.966|2010-11-23T02:20:00Z,53.516566,8.188217,None,-4.969|2010-11-23T02:30:00Z,53.516566,8.188217,None,-4.967|2010-11-23T02:40:00Z,53.516566,8.188217,None,-4.967|2010-11-23T02:50:00Z,53.516566,8.188217,None,-4.963|2010-11-23T03:00:00Z,53.516566,8.188217,None,-4.96|2010-11-23T03:10:00Z,53.516566,8.188217,None,-4.969|2010-11-23T03:20:00Z,53.516566,8.188217,None,-4.967|2010-11-23T03:30:00Z,53.516566,8.188217,None,-4.963|2010-11-23T03:40:00Z,53.516566,8.188217,None,-4.964|2010-11-23T03:50:00Z,53.516566,8.188217,None,-4.968|2010-11-23T04:00:00Z,53.516566,8.188217,None,-4.965|2010-11-23T04:10:00Z,53.516566,8.188217,None,-4.96|2010-11-23T04:20:00Z,53.516566,8.188217,None,-4.967|2010-11-23T04:30:00Z,53.516566,8.188217,None,-4.964|2010-11-23T04:40:00Z,53.516566,8.188217,None,-4.967|2010-11-23T04:50:00Z,53.516566,8.188217,None,-4.965|2010-11-23T05:00:00Z,53.516566,8.188217,None,-4.964|2010-11-23T05:10:00Z,53.516566,8.188217,None,-4.974|2010-11-23T05:20:00Z,53.516566,8.188217,None,-4.974|2010-11-23T05:30:00Z,53.516566,8.188217,None,-4.97|2010-11-23T05:40:00Z,53.516566,8.188217,None,-4.966|2010-11-23T05:50:00Z,53.516566,8.188217,None,-4.962|2010-11-23T06:00:00Z,53.516566,8.188217,None,-4.964|2010-11-23T06:10:00Z,53.516566,8.188217,None,-4.967|2010-11-23T06:20:00Z,53.516566,8.188217,None,-4.964|2010-11-23T06:30:00Z,53.516566,8.188217,None,-4.967|2010-11-23T06:40:00Z,53.516566,8.188217,None,-4.969|2010-11-23T06:50:00Z,53.516566,8.188217,None,-4.969|2010-11-23T07:00:00Z,53.516566,8.188217,None,-4.974|2010-11-23T07:10:00Z,53.516566,8.188217,None,-4.972|2010-11-23T07:20:00Z,53.516566,8.188217,None,-4.977|2010-11-23T07:30:00Z,53.516566,8.188217,None,-4.972|2010-11-23T07:40:00Z,53.516566,8.188217,None,-4.977|2010-11-23T07:50:00Z,53.516566,8.188217,None,-4.995|2010-11-23T08:00:00Z,53.516566,8.188217,None,-4.994|2010-11-23T08:10:00Z,53.516566,8.188217,None,-4.995|2010-11-23T08:20:00Z,53.516566,8.188217,None,-4.995|2010-11-23T08:30:00Z,53.516566,8.188217,None,-5.001|2010-11-23T08:40:00Z,53.516566,8.188217,None,-5.0|2010-11-23T08:50:00Z,53.516566,8.188217,None,-5.0|2010-11-23T09:00:00Z,53.516566,8.188217,None,-5.009|2010-11-23T09:10:00Z,53.516566,8.188217,None,-5.01|2010-11-23T09:20:00Z,53.516566,8.188217,None,-5.011|2010-11-23T09:30:00Z,53.516566,8.188217,None,-5.003|2010-11-23T09:40:00Z,53.516566,8.188217,None,-4.994|2010-11-23T09:50:00Z,53.516566,8.188217,None,-4.999|2010-11-23T10:00:00Z,53.516566,8.188217,None,-4.999|2010-11-23T10:10:00Z,53.516566,8.188217,None,-5.002|2010-11-23T10:20:00Z,53.516566,8.188217,None,-5.011|2010-11-23T10:30:00Z,53.516566,8.188217,None,-5.012|2010-11-23T10:40:00Z,53.516566,8.188217,None,-5.015|2010-11-23T10:50:00Z,53.516566,8.188217,None,-5.022|2010-11-23T11:00:00Z,53.516566,8.188217,None,-5.017|2010-11-23T11:10:00Z,53.516566,8.188217,None,-5.019|2010-11-23T11:20:00Z,53.516566,8.188217,None,-5.03|2010-11-23T11:30:00Z,53.516566,8.188217,None,-5.032|2010-11-23T11:40:00Z,53.516566,8.188217,None,-5.03|2010-11-23T11:50:00Z,53.516566,8.188217,None,-5.034|2010-11-23T12:00:00Z,53.516566,8.188217,None,-5.036|2010-11-23T12:10:00Z,53.516566,8.188217,None,-5.032|2010-11-23T12:20:00Z,53.516566,8.188217,None,-5.033|2010-11-23T12:30:00Z,53.516566,8.188217,None,-5.038|2010-11-23T12:40:00Z,53.516566,8.188217,None,-5.035|2010-11-23T12:50:00Z,53.516566,8.188217,None,-5.039|2010-11-23T13:00:00Z,53.516566,8.188217,None,-5.038|2010-11-23T13:10:00Z,53.516566,8.188217,None,-5.044|2010-11-23T13:20:00Z,53.516566,8.188217,None,-5.04|2010-11-23T13:30:00Z,53.516566,8.188217,None,-5.033|2010-11-23T13:40:00Z,53.516566,8.188217,None,-5.035|2010-11-23T13:50:00Z,53.516566,8.188217,None,-5.032|2010-11-23T14:00:00Z,53.516566,8.188217,None,-5.038|2010-11-23T14:10:00Z,53.516566,8.188217,None,-5.046|2010-11-23T14:20:00Z,53.516566,8.188217,None,-5.045|2010-11-23T14:30:00Z,53.516566,8.188217,None,-5.042|2010-11-23T14:40:00Z,53.516566,8.188217,None,-5.042|2010-11-23T14:50:00Z,53.516566,8.188217,None,-5.028|2010-11-23T15:00:00Z,53.516566,8.188217,None,-5.026|2010-11-23T15:10:00Z,53.516566,8.188217,None,-5.028|2010-11-23T15:20:00Z,53.516566,8.188217,None,-5.033|2010-11-23T15:30:00Z,53.516566,8.188217,None,-5.031|2010-11-23T15:40:00Z,53.516566,8.188217,None,-5.03|2010-11-23T15:50:00Z,53.516566,8.188217,None,-5.029|2010-11-23T16:00:00Z,53.516566,8.188217,None,-5.03|2010-11-23T16:10:00Z,53.516566,8.188217,None,-5.029|2010-11-23T16:20:00Z,53.516566,8.188217,None,-5.028|2010-11-23T16:30:00Z,53.516566,8.188217,None,-5.034|2010-11-23T16:40:00Z,53.516566,8.188217,None,-5.034|2010-11-23T16:50:00Z,53.516566,8.188217,None,-5.036|2010-11-23T17:00:00Z,53.516566,8.188217,None,-5.036|2010-11-23T17:10:00Z,53.516566,8.188217,None,-5.031|2010-11-23T17:20:00Z,53.516566,8.188217,None,-5.032|2010-11-23T17:30:00Z,53.516566,8.188217,None,-5.029|2010-11-23T17:40:00Z,53.516566,8.188217,None,-5.035|2010-11-23T17:50:00Z,53.516566,8.188217,None,-5.037|2010-11-23T18:00:00Z,53.516566,8.188217,None,-5.034|2010-11-23T18:10:00Z,53.516566,8.188217,None,-5.036|2010-11-23T18:20:00Z,53.516566,8.188217,None,-5.033|2010-11-23T18:30:00Z,53.516566,8.188217,None,-5.041|2010-11-23T18:40:00Z,53.516566,8.188217,None,-5.043|2010-11-23T18:50:00Z,53.516566,8.188217,None,-5.04|2010-11-23T19:00:00Z,53.516566,8.188217,None,-5.043|2010-11-23T19:10:00Z,53.516566,8.188217,None,-5.042|2010-11-23T19:20:00Z,53.516566,8.188217,None,-5.043|2010-11-23T19:30:00Z,53.516566,8.188217,None,-5.039|2010-11-23T19:40:00Z,53.516566,8.188217,None,-5.043|2010-11-23T19:50:00Z,53.516566,8.188217,None,-5.033|2010-11-23T20:00:00Z,53.516566,8.188217,None,-5.035|2010-11-23T20:10:00Z,53.516566,8.188217,None,-5.032|2010-11-23T20:20:00Z,53.516566,8.188217,None,-5.036|2010-11-23T20:30:00Z,53.516566,8.188217,None,-5.045|2010-11-23T20:40:00Z,53.516566,8.188217,None,-5.039|2010-11-23T20:50:00Z,53.516566,8.188217,None,-5.039|2010-11-23T21:00:00Z,53.516566,8.188217,None,-5.043|2010-11-23T21:10:00Z,53.516566,8.188217,None,-5.045|2010-11-23T21:20:00Z,53.516566,8.188217,None,-5.043|2010-11-23T21:30:00Z,53.516566,8.188217,None,-5.034|2010-11-23T21:40:00Z,53.516566,8.188217,None,-5.038|2010-11-23T21:50:00Z,53.516566,8.188217,None,-5.032|2010-11-23T22:00:00Z,53.516566,8.188217,None,-5.032|2010-11-23T22:10:00Z,53.516566,8.188217,None,-5.036|2010-11-23T22:20:00Z,53.516566,8.188217,None,-5.035|2010-11-23T22:30:00Z,53.516566,8.188217,None,-5.031|2010-11-23T22:40:00Z,53.516566,8.188217,None,-5.034|2010-11-23T22:50:00Z,53.516566,8.188217,None,-5.033|2010-11-23T23:00:00Z,53.516566,8.188217,None,-5.035|2010-11-23T23:10:00Z,53.516566,8.188217,None,-5.035|2010-11-23T23:20:00Z,53.516566,8.188217,None,-5.031|2010-11-23T23:30:00Z,53.516566,8.188217,None,-5.027|2010-11-23T23:40:00Z,53.516566,8.188217,None,-5.03|2010-11-23T23:50:00Z,53.516566,8.188217,None,-5.03|2010-11-24T00:00:00Z,53.516566,8.188217,None,-5.028|2010-11-24T00:10:00Z,53.516566,8.188217,None,-5.024|2010-11-24T00:20:00Z,53.516566,8.188217,None,-5.032|2010-11-24T00:30:00Z,53.516566,8.188217,None,-5.026|2010-11-24T00:40:00Z,53.516566,8.188217,None,-5.035|2010-11-24T00:50:00Z,53.516566,8.188217,None,-5.031|2010-11-24T01:00:00Z,53.516566,8.188217,None,-5.029|2010-11-24T01:10:00Z,53.516566,8.188217,None,-5.021|2010-11-24T01:20:00Z,53.516566,8.188217,None,-5.022|2010-11-24T01:30:00Z,53.516566,8.188217,None,-5.028|2010-11-24T01:40:00Z,53.516566,8.188217,None,-5.024|2010-11-24T01:50:00Z,53.516566,8.188217,None,-5.023|2010-11-24T02:00:00Z,53.516566,8.188217,None,-5.025|2010-11-24T02:10:00Z,53.516566,8.188217,None,-5.021|2010-11-24T02:20:00Z,53.516566,8.188217,None,-5.019|2010-11-24T02:30:00Z,53.516566,8.188217,None,-5.025|2010-11-24T02:40:00Z,53.516566,8.188217,None,-5.032|2010-11-24T02:50:00Z,53.516566,8.188217,None,-5.034|2010-11-24T03:00:00Z,53.516566,8.188217,None,-5.033|2010-11-24T03:10:00Z,53.516566,8.188217,None,-5.041|2010-11-24T03:20:00Z,53.516566,8.188217,None,-5.044|2010-11-24T03:30:00Z,53.516566,8.188217,None,-5.046|2010-11-24T03:40:00Z,53.516566,8.188217,None,-5.046|2010-11-24T03:50:00Z,53.516566,8.188217,None,-5.045|2010-11-24T04:00:00Z,53.516566,8.188217,None,-5.048|2010-11-24T04:10:00Z,53.516566,8.188217,None,-5.051|2010-11-24T04:20:00Z,53.516566,8.188217,None,-5.055|2010-11-24T04:30:00Z,53.516566,8.188217,None,-5.059|2010-11-24T04:40:00Z,53.516566,8.188217,None,-5.055|2010-11-24T04:50:00Z,53.516566,8.188217,None,-5.056|2010-11-24T05:00:00Z,53.516566,8.188217,None,-5.057|2010-11-24T05:10:00Z,53.516566,8.188217,None,-5.058|2010-11-24T05:20:00Z,53.516566,8.188217,None,-5.061|2010-11-24T05:30:00Z,53.516566,8.188217,None,-5.057|2010-11-24T05:40:00Z,53.516566,8.188217,None,-5.057|2010-11-24T05:50:00Z,53.516566,8.188217,None,-5.058|2010-11-24T06:00:00Z,53.516566,8.188217,None,-5.063|2010-11-24T06:10:00Z,53.516566,8.188217,None,-5.07|2010-11-24T06:20:00Z,53.516566,8.188217,None,-5.066|2010-11-24T06:30:00Z,53.516566,8.188217,None,-5.07|2010-11-24T06:40:00Z,53.516566,8.188217,None,-5.071|2010-11-24T06:50:00Z,53.516566,8.188217,None,-5.068|2010-11-24T07:00:00Z,53.516566,8.188217,None,-5.065|2010-11-24T07:10:00Z,53.516566,8.188217,None,-5.065|2010-11-24T07:20:00Z,53.516566,8.188217,None,-5.086|2010-11-24T07:30:00Z,53.516566,8.188217,None,-5.098|2010-11-24T07:40:00Z,53.516566,8.188217,None,-5.101|2010-11-24T07:50:00Z,53.516566,8.188217,None,-5.104|2010-11-24T08:00:00Z,53.516566,8.188217,None,-5.106|2010-11-24T08:10:00Z,53.516566,8.188217,None,-5.1|2010-11-24T08:20:00Z,53.516566,8.188217,None,-5.089|2010-11-24T08:30:00Z,53.516566,8.188217,None,-5.09|2010-11-24T08:40:00Z,53.516566,8.188217,None,-5.083|2010-11-24T08:50:00Z,53.516566,8.188217,None,-5.086|2010-11-24T09:00:00Z,53.516566,8.188217,None,-5.082|2010-11-24T09:10:00Z,53.516566,8.188217,None,-5.089|2010-11-24T09:20:00Z,53.516566,8.188217,None,-5.084|2010-11-24T09:30:00Z,53.516566,8.188217,None,-5.081|2010-11-24T09:40:00Z,53.516566,8.188217,None,-5.044|2010-11-24T09:50:00Z,53.516566,8.188217,None,-5.052|2010-11-24T10:00:00Z,53.516566,8.188217,None,-5.044|2010-11-24T10:10:00Z,53.516566,8.188217,None,-5.056|2010-11-24T10:20:00Z,53.516566,8.188217,None,-5.057|2010-11-24T10:30:00Z,53.516566,8.188217,None,-5.045|2010-11-24T10:40:00Z,53.516566,8.188217,None,-5.048|2010-11-24T10:50:00Z,53.516566,8.188217,None,-5.037|2010-11-24T11:00:00Z,53.516566,8.188217,None,-5.05|2010-11-24T11:10:00Z,53.516566,8.188217,None,-5.03|2010-11-24T11:20:00Z,53.516566,8.188217,None,-5.028|2010-11-24T11:30:00Z,53.516566,8.188217,None,-5.017|2010-11-24T11:40:00Z,53.516566,8.188217,None,-5.043|2010-11-24T11:50:00Z,53.516566,8.188217,None,-5.031|2010-11-24T12:00:00Z,53.516566,8.188217,None,-5.006|2010-11-24T12:10:00Z,53.516566,8.188217,None,-1.193|2010-11-24T12:20:00Z,53.516566,8.188217,None,3.237|2010-11-24T12:30:00Z,53.516566,8.188217,None,2.953|2010-11-24T12:40:00Z,53.516566,8.188217,None,3.052|2010-11-24T12:50:00Z,53.516566,8.188217,None,3.296|2010-11-24T13:00:00Z,53.516566,8.188217,None,5.415|2010-11-24T13:10:00Z,53.516566,8.188217,None,4.35|2010-11-24T13:20:00Z,53.516566,8.188217,None,5.189|2010-11-24T13:30:00Z,53.516566,8.188217,None,5.997|2010-11-24T13:40:00Z,53.516566,8.188217,None,6.718|2010-11-24T13:50:00Z,53.516566,8.188217,None,6.721|2010-11-24T14:00:00Z,53.516566,8.188217,None,6.723|2010-11-24T14:10:00Z,53.516566,8.188217,None,4.989|2010-11-24T14:20:00Z,53.516566,8.188217,None,5.263|2010-11-24T14:30:00Z,53.516566,8.188217,None,4.94|2010-11-24T14:40:00Z,53.516566,8.188217,None,6.429|2010-11-24T14:50:00Z,53.516566,8.188217,None,6.249|2010-11-24T15:00:00Z,53.516566,8.188217,None,4.063|2010-11-24T15:10:00Z,53.516566,8.188217,None,2.129|2010-11-24T15:20:00Z,53.516566,8.188217,None,2.234|2010-11-24T15:30:00Z,53.516566,8.188217,None,2.326|2010-11-24T15:40:00Z,53.516566,8.188217,None,3.842|2010-11-24T15:50:00Z,53.516566,8.188217,None,5.982|2010-11-24T16:00:00Z,53.516566,8.188217,None,6.16|2010-11-24T16:10:00Z,53.516566,8.188217,None,6.234|2010-11-24T16:20:00Z,53.516566,8.188217,None,6.252|2010-11-24T16:40:00Z,53.516566,8.188217,None,6.265|2010-11-24T16:50:00Z,53.516566,8.188217,None,6.252|2010-11-24T17:00:00Z,53.516566,8.188217,None,6.227|2010-11-24T17:10:00Z,53.516566,8.188217,None,6.211|2010-11-24T17:20:00Z,53.516566,8.188217,None,6.203|2010-11-24T17:30:00Z,53.516566,8.188217,None,6.188|2010-11-24T17:40:00Z,53.516566,8.188217,None,6.193|2010-11-24T17:50:00Z,53.516566,8.188217,None,6.214|2010-11-24T18:00:00Z,53.516566,8.188217,None,6.238|2010-11-24T18:10:00Z,53.516566,8.188217,None,6.258|2010-11-24T18:20:00Z,53.516566,8.188217,None,6.26|2010-11-24T18:30:00Z,53.516566,8.188217,None,6.08|2010-11-24T18:40:00Z,53.516566,8.188217,None,6.044|2010-11-24T18:50:00Z,53.516566,8.188217,None,5.993|2010-11-24T19:00:00Z,53.516566,8.188217,None,5.923|2010-11-24T19:10:00Z,53.516566,8.188217,None,5.91|2010-11-24T19:20:00Z,53.516566,8.188217,None,5.946|2010-11-24T19:30:00Z,53.516566,8.188217,None,5.995|2010-11-24T19:40:00Z,53.516566,8.188217,None,6.114|2010-11-24T19:50:00Z,53.516566,8.188217,None,6.092|2010-11-24T20:00:00Z,53.516566,8.188217,None,5.939|2010-11-24T20:10:00Z,53.516566,8.188217,None,5.753|2010-11-24T20:20:00Z,53.516566,8.188217,None,5.978|2010-11-24T20:30:00Z,53.516566,8.188217,None,5.943|2010-11-24T20:40:00Z,53.516566,8.188217,None,5.72|2010-11-24T20:50:00Z,53.516566,8.188217,None,5.659|2010-11-24T21:00:00Z,53.516566,8.188217,None,5.629|2010-11-24T21:10:00Z,53.516566,8.188217,None,5.638|2010-11-24T21:20:00Z,53.516566,8.188217,None,5.678|2010-11-24T21:30:00Z,53.516566,8.188217,None,5.706|2010-11-24T21:40:00Z,53.516566,8.188217,None,5.715|2010-11-24T21:50:00Z,53.516566,8.188217,None,5.773|2010-11-24T22:00:00Z,53.516566,8.188217,None,5.927|2010-11-24T22:10:00Z,53.516566,8.188217,None,5.986|2010-11-24T22:20:00Z,53.516566,8.188217,None,5.975|2010-11-24T22:30:00Z,53.516566,8.188217,None,6.011|2010-11-24T22:40:00Z,53.516566,8.188217,None,6.038|2010-11-24T22:50:00Z,53.516566,8.188217,None,6.046|2010-11-24T23:00:00Z,53.516566,8.188217,None,6.117|2010-11-24T23:10:00Z,53.516566,8.188217,None,6.234|2010-11-24T23:20:00Z,53.516566,8.188217,None,6.311|2010-11-24T23:30:00Z,53.516566,8.188217,None,6.344|2010-11-24T23:40:00Z,53.516566,8.188217,None,6.34|2010-11-24T23:50:00Z,53.516566,8.188217,None,6.323|2010-11-25T00:00:00Z,53.516566,8.188217,None,6.297|2010-11-25T00:10:00Z,53.516566,8.188217,None,6.289|2010-11-25T00:20:00Z,53.516566,8.188217,None,6.346|2010-11-25T00:30:00Z,53.516566,8.188217,None,6.416|2010-11-25T00:40:00Z,53.516566,8.188217,None,6.472|2010-11-25T00:50:00Z,53.516566,8.188217,None,6.496|2010-11-25T01:00:00Z,53.516566,8.188217,None,6.505|2010-11-25T01:10:00Z,53.516566,8.188217,None,6.532|2010-11-25T01:20:00Z,53.516566,8.188217,None,6.567|2010-11-25T01:30:00Z,53.516566,8.188217,None,6.6|2010-11-25T01:40:00Z,53.516566,8.188217,None,6.613|2010-11-25T01:50:00Z,53.516566,8.188217,None,6.585|2010-11-25T02:00:00Z,53.516566,8.188217,None,6.529|2010-11-25T02:10:00Z,53.516566,8.188217,None,6.474|2010-11-25T02:20:00Z,53.516566,8.188217,None,6.406|2010-11-25T02:30:00Z,53.516566,8.188217,None,6.224|2010-11-25T02:40:00Z,53.516566,8.188217,None,6.106|2010-11-25T02:50:00Z,53.516566,8.188217,None,6.017|2010-11-25T03:00:00Z,53.516566,8.188217,None,5.752|2010-11-25T03:10:00Z,53.516566,8.188217,None,5.533|2010-11-25T03:20:00Z,53.516566,8.188217,None,5.438|2010-11-25T03:30:00Z,53.516566,8.188217,None,5.362|2010-11-25T03:40:00Z,53.516566,8.188217,None,5.383|2010-11-25T03:50:00Z,53.516566,8.188217,None,5.541|2010-11-25T04:00:00Z,53.516566,8.188217,None,5.661|2010-11-25T04:10:00Z,53.516566,8.188217,None,5.724|2010-11-25T04:20:00Z,53.516566,8.188217,None,5.728|2010-11-25T04:30:00Z,53.516566,8.188217,None,5.639|2010-11-25T04:40:00Z,53.516566,8.188217,None,5.584|2010-11-25T04:50:00Z,53.516566,8.188217,None,5.642|2010-11-25T05:00:00Z,53.516566,8.188217,None,5.704|2010-11-25T05:10:00Z,53.516566,8.188217,None,5.736|2010-11-25T05:20:00Z,53.516566,8.188217,None,5.753|2010-11-25T05:30:00Z,53.516566,8.188217,None,5.756|2010-11-25T05:40:00Z,53.516566,8.188217,None,5.754|2010-11-25T05:50:00Z,53.516566,8.188217,None,5.742|2010-11-25T06:00:00Z,53.516566,8.188217,None,5.739|2010-11-25T06:10:00Z,53.516566,8.188217,None,5.735|2010-11-25T06:20:00Z,53.516566,8.188217,None,5.719|2010-11-25T06:30:00Z,53.516566,8.188217,None,5.805|2010-11-25T06:40:00Z,53.516566,8.188217,None,5.926|2010-11-25T06:50:00Z,53.516566,8.188217,None,5.764|2010-11-25T07:00:00Z,53.516566,8.188217,None,5.642|2010-11-25T07:10:00Z,53.516566,8.188217,None,5.52|2010-11-25T07:30:00Z,53.516566,8.188217,None,5.449|2010-11-25T07:40:00Z,53.516566,8.188217,None,5.469|2010-11-25T07:50:00Z,53.516566,8.188217,None,5.802|2010-11-25T08:00:00Z,53.516566,8.188217,None,5.835|2010-11-25T08:10:00Z,53.516566,8.188217,None,5.833|2010-11-25T08:20:00Z,53.516566,8.188217,None,5.846|2010-11-25T08:30:00Z,53.516566,8.188217,None,5.788|2010-11-25T08:40:00Z,53.516566,8.188217,None,5.751|2010-11-25T08:50:00Z,53.516566,8.188217,None,5.603|2010-11-25T09:00:00Z,53.516566,8.188217,None,5.362|2010-11-25T09:10:00Z,53.516566,8.188217,None,5.594|2010-11-25T09:20:00Z,53.516566,8.188217,None,5.706|2010-11-25T09:30:00Z,53.516566,8.188217,None,5.531|2010-11-25T09:40:00Z,53.516566,8.188217,None,5.452|2010-11-25T09:50:00Z,53.516566,8.188217,None,4.707|2010-11-25T10:00:00Z,53.516566,8.188217,None,5.026|2010-11-25T10:40:00Z,53.516566,8.188217,None,0.726|2010-11-25T10:50:00Z,53.516566,8.188217,None,0.834|2010-11-25T11:00:00Z,53.516566,8.188217,None,5.691|2010-11-25T11:10:00Z,53.516566,8.188217,None,5.732|2010-11-25T11:20:00Z,53.516566,8.188217,None,5.793|2010-11-25T11:30:00Z,53.516566,8.188217,None,5.855|2010-11-25T11:40:00Z,53.516566,8.188217,None,5.897|2010-11-25T11:50:00Z,53.516566,8.188217,None,5.965|2010-11-25T12:00:00Z,53.516566,8.188217,None,6.011|2010-11-25T12:10:00Z,53.516566,8.188217,None,6.067|2010-11-25T12:20:00Z,53.516566,8.188217,None,6.169|2010-11-25T12:30:00Z,53.516566,8.188217,None,6.21|2010-11-25T12:40:00Z,53.516566,8.188217,None,6.209|2010-11-25T12:50:00Z,53.516566,8.188217,None,6.19|2010-11-25T13:00:00Z,53.516566,8.188217,None,6.168|2010-11-25T13:10:00Z,53.516566,8.188217,None,6.154|2010-11-25T13:20:00Z,53.516566,8.188217,None,6.167|2010-11-25T13:30:00Z,53.516566,8.188217,None,6.207|2010-11-25T13:40:00Z,53.516566,8.188217,None,6.225|2010-11-25T13:50:00Z,53.516566,8.188217,None,6.233|2010-11-25T14:00:00Z,53.516566,8.188217,None,6.244|2010-11-25T14:10:00Z,53.516566,8.188217,None,6.246|2010-11-25T14:20:00Z,53.516566,8.188217,None,6.237|2010-11-25T14:30:00Z,53.516566,8.188217,None,6.225|2010-11-25T14:40:00Z,53.516566,8.188217,None,6.19|2010-11-25T14:50:00Z,53.516566,8.188217,None,6.137|2010-11-25T15:00:00Z,53.516566,8.188217,None,6.105|2010-11-25T15:10:00Z,53.516566,8.188217,None,6.035|2010-11-25T15:20:00Z,53.516566,8.188217,None,5.833|2010-11-25T15:30:00Z,53.516566,8.188217,None,5.571|2010-11-25T15:40:00Z,53.516566,8.188217,None,5.424|2010-11-25T15:50:00Z,53.516566,8.188217,None,5.345|2010-11-25T16:00:00Z,53.516566,8.188217,None,5.319|2010-11-25T16:10:00Z,53.516566,8.188217,None,5.278|2010-11-25T16:20:00Z,53.516566,8.188217,None,5.318|2010-11-25T16:30:00Z,53.516566,8.188217,None,5.374|2010-11-25T16:40:00Z,53.516566,8.188217,None,5.423|2010-11-25T16:50:00Z,53.516566,8.188217,None,5.425|2010-11-25T17:00:00Z,53.516566,8.188217,None,5.413|2010-11-25T17:10:00Z,53.516566,8.188217,None,5.386|2010-11-25T17:20:00Z,53.516566,8.188217,None,5.367|2010-11-25T17:30:00Z,53.516566,8.188217,None,5.371|2010-11-25T17:40:00Z,53.516566,8.188217,None,5.368|2010-11-25T17:50:00Z,53.516566,8.188217,None,5.37|2010-11-25T18:00:00Z,53.516566,8.188217,None,5.349|2010-11-25T18:10:00Z,53.516566,8.188217,None,5.348|2010-11-25T18:20:00Z,53.516566,8.188217,None,5.346|2010-11-25T18:30:00Z,53.516566,8.188217,None,5.383|2010-11-25T18:40:00Z,53.516566,8.188217,None,5.514|2010-11-25T18:50:00Z,53.516566,8.188217,None,5.574|2010-11-25T19:00:00Z,53.516566,8.188217,None,5.529|2010-11-25T19:10:00Z,53.516566,8.188217,None,5.517|2010-11-25T19:20:00Z,53.516566,8.188217,None,5.147|2010-11-25T19:30:00Z,53.516566,8.188217,None,5.467|2010-11-25T19:40:00Z,53.516566,8.188217,None,5.603|2010-11-25T19:50:00Z,53.516566,8.188217,None,5.623|2010-11-25T20:00:00Z,53.516566,8.188217,None,5.652|2010-11-25T20:10:00Z,53.516566,8.188217,None,5.67|2010-11-25T20:20:00Z,53.516566,8.188217,None,5.649|2010-11-25T20:30:00Z,53.516566,8.188217,None,5.6|2010-11-25T20:40:00Z,53.516566,8.188217,None,5.534|2010-11-25T20:50:00Z,53.516566,8.188217,None,5.462|2010-11-25T21:00:00Z,53.516566,8.188217,None,5.344|2010-11-25T21:10:00Z,53.516566,8.188217,None,5.116|2010-11-25T21:20:00Z,53.516566,8.188217,None,5.354|2010-11-25T21:30:00Z,53.516566,8.188217,None,5.544|2010-11-25T21:40:00Z,53.516566,8.188217,None,5.313|2010-11-25T21:50:00Z,53.516566,8.188217,None,5.081|2010-11-25T22:00:00Z,53.516566,8.188217,None,5.155|2010-11-25T22:10:00Z,53.516566,8.188217,None,5.179|2010-11-25T22:20:00Z,53.516566,8.188217,None,5.245|2010-11-25T22:30:00Z,53.516566,8.188217,None,5.145|2010-11-25T22:40:00Z,53.516566,8.188217,None,5.309|2010-11-25T22:50:00Z,53.516566,8.188217,None,5.365|2010-11-25T23:00:00Z,53.516566,8.188217,None,5.474|2010-11-25T23:10:00Z,53.516566,8.188217,None,5.577|2010-11-25T23:20:00Z,53.516566,8.188217,None,5.576|2010-11-25T23:30:00Z,53.516566,8.188217,None,5.614|2010-11-25T23:40:00Z,53.516566,8.188217,None,5.689|2010-11-25T23:50:00Z,53.516566,8.188217,None,5.765|2010-11-26T00:00:00Z,53.516566,8.188217,None,5.848|2010-11-26T00:10:00Z,53.516566,8.188217,None,5.938|2010-11-26T00:20:00Z,53.516566,8.188217,None,5.957|2010-11-26T00:30:00Z,53.516566,8.188217,None,5.935|2010-11-26T00:40:00Z,53.516566,8.188217,None,5.948|2010-11-26T00:50:00Z,53.516566,8.188217,None,5.964|2010-11-26T01:00:00Z,53.516566,8.188217,None,5.968|2010-11-26T01:10:00Z,53.516566,8.188217,None,6.013|2010-11-26T01:20:00Z,53.516566,8.188217,None,6.071|2010-11-26T01:30:00Z,53.516566,8.188217,None,6.108|2010-11-26T01:40:00Z,53.516566,8.188217,None,6.14|2010-11-26T01:50:00Z,53.516566,8.188217,None,6.202|2010-11-26T02:00:00Z,53.516566,8.188217,None,6.239|2010-11-26T02:10:00Z,53.516566,8.188217,None,6.263|2010-11-26T02:20:00Z,53.516566,8.188217,None,6.284|2010-11-26T02:30:00Z,53.516566,8.188217,None,6.276|2010-11-26T02:40:00Z,53.516566,8.188217,None,6.267|2010-11-26T02:50:00Z,53.516566,8.188217,None,6.233|2010-11-26T03:00:00Z,53.516566,8.188217,None,6.142|2010-11-26T03:10:00Z,53.516566,8.188217,None,6.001|2010-11-26T03:20:00Z,53.516566,8.188217,None,5.886|2010-11-26T03:30:00Z,53.516566,8.188217,None,5.778|2010-11-26T03:40:00Z,53.516566,8.188217,None,5.545|2010-11-26T03:50:00Z,53.516566,8.188217,None,5.252|2010-11-26T04:00:00Z,53.516566,8.188217,None,5.029|2010-11-26T04:10:00Z,53.516566,8.188217,None,4.982|2010-11-26T04:20:00Z,53.516566,8.188217,None,4.94|2010-11-26T04:30:00Z,53.516566,8.188217,None,4.858|2010-11-26T04:40:00Z,53.516566,8.188217,None,4.914|2010-11-26T04:50:00Z,53.516566,8.188217,None,4.993|2010-11-26T05:00:00Z,53.516566,8.188217,None,5.068|2010-11-26T05:10:00Z,53.516566,8.188217,None,5.169|2010-11-26T05:20:00Z,53.516566,8.188217,None,5.207|2010-11-26T05:30:00Z,53.516566,8.188217,None,5.199|2010-11-26T05:40:00Z,53.516566,8.188217,None,5.215|2010-11-26T05:50:00Z,53.516566,8.188217,None,5.265|2010-11-26T06:00:00Z,53.516566,8.188217,None,5.29|2010-11-26T06:10:00Z,53.516566,8.188217,None,5.3|2010-11-26T06:20:00Z,53.516566,8.188217,None,5.296|2010-11-26T06:30:00Z,53.516566,8.188217,None,5.259|2010-11-26T06:40:00Z,53.516566,8.188217,None,5.253|2010-11-26T06:50:00Z,53.516566,8.188217,None,5.246|2010-11-26T07:00:00Z,53.516566,8.188217,None,5.257|2010-11-26T07:10:00Z,53.516566,8.188217,None,5.345|2010-11-26T07:20:00Z,53.516566,8.188217,None,5.352|2010-11-26T07:30:00Z,53.516566,8.188217,None,5.11|2010-11-26T07:40:00Z,53.516566,8.188217,None,5.018|2010-11-26T07:50:00Z,53.516566,8.188217,None,5.0|2010-11-26T08:00:00Z,53.516566,8.188217,None,4.99|2010-11-26T08:10:00Z,53.516566,8.188217,None,5.179|2010-11-26T08:20:00Z,53.516566,8.188217,None,5.45|2010-11-26T08:30:00Z,53.516566,8.188217,None,5.51|2010-11-26T08:40:00Z,53.516566,8.188217,None,5.565|2010-11-26T08:50:00Z,53.516566,8.188217,None,5.541|2010-11-26T09:00:00Z,53.516566,8.188217,None,5.239|2010-11-26T09:10:00Z,53.516566,8.188217,None,5.209|2010-11-26T09:20:00Z,53.516566,8.188217,None,5.363|2010-11-26T09:30:00Z,53.516566,8.188217,None,5.092|2010-11-26T09:40:00Z,53.516566,8.188217,None,5.099|2010-11-26T09:50:00Z,53.516566,8.188217,None,5.269|2010-11-26T10:00:00Z,53.516566,8.188217,None,5.212|2010-11-26T10:10:00Z,53.516566,8.188217,None,5.066|2010-11-26T10:20:00Z,53.516566,8.188217,None,4.774|2010-11-26T10:30:00Z,53.516566,8.188217,None,4.782|2010-11-26T10:40:00Z,53.516566,8.188217,None,4.838|2010-11-26T10:50:00Z,53.516566,8.188217,None,4.884|2010-11-26T11:00:00Z,53.516566,8.188217,None,4.969|2010-11-26T11:10:00Z,53.516566,8.188217,None,5.002|2010-11-26T11:20:00Z,53.516566,8.188217,None,5.016|2010-11-26T11:30:00Z,53.516566,8.188217,None,5.224|2010-11-26T11:40:00Z,53.516566,8.188217,None,5.285|2010-11-26T11:50:00Z,53.516566,8.188217,None,5.352|2010-11-26T12:00:00Z,53.516566,8.188217,None,5.407|2010-11-26T12:10:00Z,53.516566,8.188217,None,5.414|2010-11-26T12:20:00Z,53.516566,8.188217,None,5.403|2010-11-26T12:30:00Z,53.516566,8.188217,None,5.378|2010-11-26T12:40:00Z,53.516566,8.188217,None,5.479|2010-11-26T12:50:00Z,53.516566,8.188217,None,5.563|2010-11-26T13:00:00Z,53.516566,8.188217,None,5.643|2010-11-26T13:10:00Z,53.516566,8.188217,None,5.731|2010-11-26T13:20:00Z,53.516566,8.188217,None,5.772|2010-11-26T13:30:00Z,53.516566,8.188217,None,5.799|2010-11-26T13:40:00Z,53.516566,8.188217,None,5.801|2010-11-26T13:50:00Z,53.516566,8.188217,None,5.786|2010-11-26T14:00:00Z,53.516566,8.188217,None,5.764|2010-11-26T14:10:00Z,53.516566,8.188217,None,5.74|2010-11-26T14:20:00Z,53.516566,8.188217,None,5.731|2010-11-26T14:30:00Z,53.516566,8.188217,None,5.731|2010-11-26T14:40:00Z,53.516566,8.188217,None,5.744|2010-11-26T14:50:00Z,53.516566,8.188217,None,5.755|2010-11-26T15:00:00Z,53.516566,8.188217,None,5.758|2010-11-26T15:10:00Z,53.516566,8.188217,None,5.752|2010-11-26T15:20:00Z,53.516566,8.188217,None,5.713|2010-11-26T15:30:00Z,53.516566,8.188217,None,5.716|2010-11-26T15:40:00Z,53.516566,8.188217,None,5.733|2010-11-26T15:50:00Z,53.516566,8.188217,None,5.586|2010-11-26T16:00:00Z,53.516566,8.188217,None,5.336|2010-11-26T16:10:00Z,53.516566,8.188217,None,5.095|2010-11-26T16:20:00Z,53.516566,8.188217,None,4.918|2010-11-26T16:30:00Z,53.516566,8.188217,None,4.836|2010-11-26T16:40:00Z,53.516566,8.188217,None,4.848|2010-11-26T16:50:00Z,53.516566,8.188217,None,4.861|2010-11-26T17:00:00Z,53.516566,8.188217,None,4.857|2010-11-26T17:10:00Z,53.516566,8.188217,None,4.9|2010-11-26T17:20:00Z,53.516566,8.188217,None,4.944|2010-11-26T17:30:00Z,53.516566,8.188217,None,4.958|2010-11-26T17:40:00Z,53.516566,8.188217,None,4.95|2010-11-26T17:50:00Z,53.516566,8.188217,None,4.983|2010-11-26T18:00:00Z,53.516566,8.188217,None,5.013|2010-11-26T18:10:00Z,53.516566,8.188217,None,5.021|2010-11-26T18:20:00Z,53.516566,8.188217,None,5.007|2010-11-26T18:30:00Z,53.516566,8.188217,None,4.994|2010-11-26T18:40:00Z,53.516566,8.188217,None,4.973|2010-11-26T18:50:00Z,53.516566,8.188217,None,4.979|2010-11-26T19:00:00Z,53.516566,8.188217,None,5.002|2010-11-26T19:10:00Z,53.516566,8.188217,None,5.047|2010-11-26T19:20:00Z,53.516566,8.188217,None,5.115|2010-11-26T19:30:00Z,53.516566,8.188217,None,5.162|2010-11-26T19:40:00Z,53.516566,8.188217,None,5.091|2010-11-26T19:50:00Z,53.516566,8.188217,None,4.814|2010-11-26T20:00:00Z,53.516566,8.188217,None,4.809|2010-11-26T20:10:00Z,53.516566,8.188217,None,4.907|2010-11-26T20:20:00Z,53.516566,8.188217,None,5.113|2010-11-26T20:30:00Z,53.516566,8.188217,None,5.236|2010-11-26T20:40:00Z,53.516566,8.188217,None,5.255|2010-11-26T20:50:00Z,53.516566,8.188217,None,5.258|2010-11-26T21:00:00Z,53.516566,8.188217,None,5.122|2010-11-26T21:10:00Z,53.516566,8.188217,None,5.039|2010-11-26T21:20:00Z,53.516566,8.188217,None,5.013|2010-11-26T21:30:00Z,53.516566,8.188217,None,5.014|2010-11-26T21:40:00Z,53.516566,8.188217,None,4.963|2010-11-26T21:50:00Z,53.516566,8.188217,None,4.722|2010-11-26T22:00:00Z,53.516566,8.188217,None,4.927|2010-11-26T22:10:00Z,53.516566,8.188217,None,5.036|2010-11-26T22:20:00Z,53.516566,8.188217,None,4.794|2010-11-26T22:30:00Z,53.516566,8.188217,None,4.671|2010-11-26T22:40:00Z,53.516566,8.188217,None,4.691|2010-11-26T22:50:00Z,53.516566,8.188217,None,4.737|2010-11-26T23:00:00Z,53.516566,8.188217,None,4.778|2010-11-26T23:10:00Z,53.516566,8.188217,None,4.813|2010-11-26T23:20:00Z,53.516566,8.188217,None,4.784|2010-11-26T23:30:00Z,53.516566,8.188217,None,4.844|2010-11-26T23:40:00Z,53.516566,8.188217,None,4.96|2010-11-26T23:50:00Z,53.516566,8.188217,None,5.007|2010-11-27T00:00:00Z,53.516566,8.188217,None,5.141|2010-11-27T00:10:00Z,53.516566,8.188217,None,5.199|2010-11-27T00:20:00Z,53.516566,8.188217,None,5.227|2010-11-27T00:30:00Z,53.516566,8.188217,None,5.262|2010-11-27T00:40:00Z,53.516566,8.188217,None,5.314|2010-11-27T00:50:00Z,53.516566,8.188217,None,5.444|2010-11-27T01:00:00Z,53.516566,8.188217,None,5.515|2010-11-27T01:10:00Z,53.516566,8.188217,None,5.576|2010-11-27T01:20:00Z,53.516566,8.188217,None,5.612|2010-11-27T01:30:00Z,53.516566,8.188217,None,5.64|2010-11-27T01:40:00Z,53.516566,8.188217,None,5.638|2010-11-27T01:50:00Z,53.516566,8.188217,None,5.603|2010-11-27T02:00:00Z,53.516566,8.188217,None,5.574|2010-11-27T02:10:00Z,53.516566,8.188217,None,5.571|2010-11-27T02:20:00Z,53.516566,8.188217,None,5.557|2010-11-27T02:30:00Z,53.516566,8.188217,None,5.553|2010-11-27T02:40:00Z,53.516566,8.188217,None,5.57|2010-11-27T02:50:00Z,53.516566,8.188217,None,5.598|2010-11-27T03:00:00Z,53.516566,8.188217,None,5.62|2010-11-27T03:10:00Z,53.516566,8.188217,None,5.619|2010-11-27T03:20:00Z,53.516566,8.188217,None,5.56|2010-11-27T03:30:00Z,53.516566,8.188217,None,5.492|2010-11-27T03:40:00Z,53.516566,8.188217,None,5.498|2010-11-27T03:50:00Z,53.516566,8.188217,None,5.484|2010-11-27T04:00:00Z,53.516566,8.188217,None,5.304|2010-11-27T04:10:00Z,53.516566,8.188217,None,5.022|2010-11-27T04:20:00Z,53.516566,8.188217,None,4.721|2010-11-27T04:30:00Z,53.516566,8.188217,None,4.532|2010-11-27T04:40:00Z,53.516566,8.188217,None,4.435|2010-11-27T04:50:00Z,53.516566,8.188217,None,4.357|2010-11-27T05:00:00Z,53.516566,8.188217,None,4.343|2010-11-27T05:10:00Z,53.516566,8.188217,None,4.55|2010-11-27T05:20:00Z,53.516566,8.188217,None,4.631|2010-11-27T05:30:00Z,53.516566,8.188217,None,4.652|2010-11-27T05:40:00Z,53.516566,8.188217,None,4.641|2010-11-27T05:50:00Z,53.516566,8.188217,None,4.677|2010-11-27T06:00:00Z,53.516566,8.188217,None,4.839|2010-11-27T06:10:00Z,53.516566,8.188217,None,4.967|2010-11-27T06:20:00Z,53.516566,8.188217,None,5.013|2010-11-27T06:30:00Z,53.516566,8.188217,None,4.996|2010-11-27T06:40:00Z,53.516566,8.188217,None,4.958|2010-11-27T06:50:00Z,53.516566,8.188217,None,4.927|2010-11-27T07:00:00Z,53.516566,8.188217,None,4.966|2010-11-27T07:10:00Z,53.516566,8.188217,None,4.903|2010-11-27T07:20:00Z,53.516566,8.188217,None,4.96|2010-11-27T07:30:00Z,53.516566,8.188217,None,5.02|2010-11-27T07:40:00Z,53.516566,8.188217,None,5.116|2010-11-27T07:50:00Z,53.516566,8.188217,None,5.081|2010-11-27T08:00:00Z,53.516566,8.188217,None,4.741|2010-11-27T08:10:00Z,53.516566,8.188217,None,4.466|2010-11-27T08:20:00Z,53.516566,8.188217,None,4.751|2010-11-27T08:30:00Z,53.516566,8.188217,None,4.88|2010-11-27T08:40:00Z,53.516566,8.188217,None,4.989|2010-11-27T08:50:00Z,53.516566,8.188217,None,5.106|2010-11-27T09:00:00Z,53.516566,8.188217,None,5.072|2010-11-27T09:10:00Z,53.516566,8.188217,None,5.044|2010-11-27T09:20:00Z,53.516566,8.188217,None,4.979|2010-11-27T09:30:00Z,53.516566,8.188217,None,4.869|2010-11-27T09:40:00Z,53.516566,8.188217,None,4.788|2010-11-27T09:50:00Z,53.516566,8.188217,None,4.855|2010-11-27T10:00:00Z,53.516566,8.188217,None,4.87|2010-11-27T10:10:00Z,53.516566,8.188217,None,4.755|2010-11-27T10:20:00Z,53.516566,8.188217,None,4.352|2010-11-27T10:30:00Z,53.516566,8.188217,None,4.254|2010-11-27T10:40:00Z,53.516566,8.188217,None,4.274|2010-11-27T10:50:00Z,53.516566,8.188217,None,4.248|2010-11-27T11:00:00Z,53.516566,8.188217,None,4.168|2010-11-27T11:10:00Z,53.516566,8.188217,None,4.209|2010-11-27T11:20:00Z,53.516566,8.188217,None,4.288|2010-11-27T11:30:00Z,53.516566,8.188217,None,4.369|2010-11-27T11:40:00Z,53.516566,8.188217,None,4.449|2010-11-27T11:50:00Z,53.516566,8.188217,None,4.547|2010-11-27T12:00:00Z,53.516566,8.188217,None,4.483|2010-11-27T12:10:00Z,53.516566,8.188217,None,4.434|2010-11-27T12:20:00Z,53.516566,8.188217,None,4.48|2010-11-27T12:30:00Z,53.516566,8.188217,None,4.685|2010-11-27T12:40:00Z,53.516566,8.188217,None,4.759|2010-11-27T12:50:00Z,53.516566,8.188217,None,4.811|2010-11-27T13:00:00Z,53.516566,8.188217,None,4.876|2010-11-27T13:10:00Z,53.516566,8.188217,None,4.963|2010-11-27T13:20:00Z,53.516566,8.188217,None,5.014|2010-11-27T13:30:00Z,53.516566,8.188217,None,5.056|2010-11-27T13:40:00Z,53.516566,8.188217,None,5.166|2010-11-27T13:50:00Z,53.516566,8.188217,None,5.281|2010-11-27T14:00:00Z,53.516566,8.188217,None,5.359|2010-11-27T14:10:00Z,53.516566,8.188217,None,5.424|2010-11-27T14:20:00Z,53.516566,8.188217,None,5.458|2010-11-27T14:30:00Z,53.516566,8.188217,None,5.47|2010-11-27T14:40:00Z,53.516566,8.188217,None,5.48|2010-11-27T14:50:00Z,53.516566,8.188217,None,5.489|2010-11-27T15:00:00Z,53.516566,8.188217,None,5.499|2010-11-27T15:10:00Z,53.516566,8.188217,None,5.502|2010-11-27T15:20:00Z,53.516566,8.188217,None,5.5|2010-11-27T15:30:00Z,53.516566,8.188217,None,5.486|2010-11-27T15:40:00Z,53.516566,8.188217,None,5.449|2010-11-27T15:50:00Z,53.516566,8.188217,None,5.389|2010-11-27T16:00:00Z,53.516566,8.188217,None,5.333|2010-11-27T16:10:00Z,53.516566,8.188217,None,5.313|2010-11-27T16:20:00Z,53.516566,8.188217,None,5.244|2010-11-27T16:30:00Z,53.516566,8.188217,None,5.02|2010-11-27T16:40:00Z,53.516566,8.188217,None,4.703|2010-11-27T16:50:00Z,53.516566,8.188217,None,4.266|2010-11-27T17:00:00Z,53.516566,8.188217,None,4.001|2010-11-27T17:10:00Z,53.516566,8.188217,None,3.931|2010-11-27T17:20:00Z,53.516566,8.188217,None,4.033|2010-11-27T17:30:00Z,53.516566,8.188217,None,4.167|2010-11-27T17:40:00Z,53.516566,8.188217,None,4.25|2010-11-27T17:50:00Z,53.516566,8.188217,None,4.211|2010-11-27T18:00:00Z,53.516566,8.188217,None,4.205|2010-11-27T18:10:00Z,53.516566,8.188217,None,4.22|2010-11-27T18:20:00Z,53.516566,8.188217,None,4.226|2010-11-27T18:30:00Z,53.516566,8.188217,None,4.212|2010-11-27T18:40:00Z,53.516566,8.188217,None,4.178|2010-11-27T18:50:00Z,53.516566,8.188217,None,4.129|2010-11-27T19:00:00Z,53.516566,8.188217,None,4.125|2010-11-27T19:10:00Z,53.516566,8.188217,None,4.141|2010-11-27T19:20:00Z,53.516566,8.188217,None,4.407|2010-11-27T19:30:00Z,53.516566,8.188217,None,4.468|2010-11-27T19:40:00Z,53.516566,8.188217,None,4.53|2010-11-27T19:50:00Z,53.516566,8.188217,None,4.53|2010-11-27T20:00:00Z,53.516566,8.188217,None,4.469|2010-11-27T20:10:00Z,53.516566,8.188217,None,4.497|2010-11-27T20:20:00Z,53.516566,8.188217,None,4.522|2010-11-27T20:30:00Z,53.516566,8.188217,None,4.545|2010-11-27T20:40:00Z,53.516566,8.188217,None,4.489|2010-11-27T20:50:00Z,53.516566,8.188217,None,4.575|2010-11-27T21:00:00Z,53.516566,8.188217,None,4.621|2010-11-27T21:10:00Z,53.516566,8.188217,None,4.61|2010-11-27T21:20:00Z,53.516566,8.188217,None,4.628|2010-11-27T21:30:00Z,53.516566,8.188217,None,4.612|2010-11-27T21:40:00Z,53.516566,8.188217,None,4.605|2010-11-27T21:50:00Z,53.516566,8.188217,None,4.541|2010-11-27T22:00:00Z,53.516566,8.188217,None,4.259|2010-11-27T22:10:00Z,53.516566,8.188217,None,3.923|2010-11-27T22:20:00Z,53.516566,8.188217,None,4.086|2010-11-27T22:30:00Z,53.516566,8.188217,None,4.279|2010-11-27T22:40:00Z,53.516566,8.188217,None,4.137|2010-11-27T22:50:00Z,53.516566,8.188217,None,4.3|2010-11-27T23:00:00Z,53.516566,8.188217,None,4.366|2010-11-27T23:10:00Z,53.516566,8.188217,None,4.314|2010-11-27T23:20:00Z,53.516566,8.188217,None,4.278|2010-11-27T23:30:00Z,53.516566,8.188217,None,4.149|2010-11-27T23:40:00Z,53.516566,8.188217,None,4.043|2010-11-27T23:50:00Z,53.516566,8.188217,None,4.105|2010-11-28T00:00:00Z,53.516566,8.188217,None,3.872|2010-11-28T00:10:00Z,53.516566,8.188217,None,4.004|2010-11-28T00:20:00Z,53.516566,8.188217,None,4.17|2010-11-28T00:30:00Z,53.516566,8.188217,None,4.282|2010-11-28T00:40:00Z,53.516566,8.188217,None,4.508|2010-11-28T00:50:00Z,53.516566,8.188217,None,4.677|2010-11-28T01:00:00Z,53.516566,8.188217,None,4.718|2010-11-28T01:10:00Z,53.516566,8.188217,None,4.722|2010-11-28T01:20:00Z,53.516566,8.188217,None,4.734|2010-11-28T01:30:00Z,53.516566,8.188217,None,4.817|2010-11-28T01:40:00Z,53.516566,8.188217,None,4.916|2010-11-28T01:50:00Z,53.516566,8.188217,None,5.004|2010-11-28T02:00:00Z,53.516566,8.188217,None,5.086|2010-11-28T02:10:00Z,53.516566,8.188217,None,5.142|2010-11-28T02:20:00Z,53.516566,8.188217,None,5.17|2010-11-28T02:30:00Z,53.516566,8.188217,None,5.143|2010-11-28T02:40:00Z,53.516566,8.188217,None,5.096|2010-11-28T02:50:00Z,53.516566,8.188217,None,5.064|2010-11-28T03:00:00Z,53.516566,8.188217,None,5.092|2010-11-28T03:10:00Z,53.516566,8.188217,None,5.117|2010-11-28T03:20:00Z,53.516566,8.188217,None,5.138|2010-11-28T03:30:00Z,53.516566,8.188217,None,5.159|2010-11-28T03:40:00Z,53.516566,8.188217,None,5.158|2010-11-28T03:50:00Z,53.516566,8.188217,None,5.159|2010-11-28T04:00:00Z,53.516566,8.188217,None,5.146|2010-11-28T04:10:00Z,53.516566,8.188217,None,5.079|2010-11-28T04:20:00Z,53.516566,8.188217,None,4.994|2010-11-28T04:30:00Z,53.516566,8.188217,None,4.893|2010-11-28T04:40:00Z,53.516566,8.188217,None,4.908|2010-11-28T04:50:00Z,53.516566,8.188217,None,4.723|2010-11-28T05:00:00Z,53.516566,8.188217,None,4.396|2010-11-28T05:10:00Z,53.516566,8.188217,None,3.837|2010-11-28T05:20:00Z,53.516566,8.188217,None,3.471|2010-11-28T05:30:00Z,53.516566,8.188217,None,3.37|2010-11-28T05:40:00Z,53.516566,8.188217,None,3.463|2010-11-28T05:50:00Z,53.516566,8.188217,None,3.405|2010-11-28T06:00:00Z,53.516566,8.188217,None,3.361|2010-11-28T06:10:00Z,53.516566,8.188217,None,3.335|2010-11-28T06:20:00Z,53.516566,8.188217,None,3.354|2010-11-28T06:30:00Z,53.516566,8.188217,None,3.465|2010-11-28T06:40:00Z,53.516566,8.188217,None,3.614|2010-11-28T06:50:00Z,53.516566,8.188217,None,3.669|2010-11-28T07:00:00Z,53.516566,8.188217,None,3.663|2010-11-28T07:10:00Z,53.516566,8.188217,None,3.651|2010-11-28T07:20:00Z,53.516566,8.188217,None,3.648|2010-11-28T07:30:00Z,53.516566,8.188217,None,3.616|2010-11-28T07:40:00Z,53.516566,8.188217,None,3.609|2010-11-28T07:50:00Z,53.516566,8.188217,None,3.599|2010-11-28T08:00:00Z,53.516566,8.188217,None,3.703|2010-11-28T08:10:00Z,53.516566,8.188217,None,3.794|2010-11-28T08:20:00Z,53.516566,8.188217,None,3.911|2010-11-28T08:30:00Z,53.516566,8.188217,None,3.943|2010-11-28T08:40:00Z,53.516566,8.188217,None,3.924|2010-11-28T08:50:00Z,53.516566,8.188217,None,3.579|2010-11-28T09:00:00Z,53.516566,8.188217,None,3.199|2010-11-28T09:10:00Z,53.516566,8.188217,None,3.119|2010-11-28T09:20:00Z,53.516566,8.188217,None,3.272|2010-11-28T09:30:00Z,53.516566,8.188217,None,3.475|2010-11-28T09:40:00Z,53.516566,8.188217,None,3.756|2010-11-28T09:50:00Z,53.516566,8.188217,None,3.871|2010-11-28T10:00:00Z,53.516566,8.188217,None,3.953|2010-11-28T10:10:00Z,53.516566,8.188217,None,3.989|2010-11-28T10:20:00Z,53.516566,8.188217,None,4.03|2010-11-28T10:30:00Z,53.516566,8.188217,None,3.973|2010-11-28T10:40:00Z,53.516566,8.188217,None,3.961|2010-11-28T10:50:00Z,53.516566,8.188217,None,4.011|2010-11-28T11:00:00Z,53.516566,8.188217,None,3.584|2010-11-28T11:10:00Z,53.516566,8.188217,None,3.249|2010-11-28T11:20:00Z,53.516566,8.188217,None,3.876|2010-11-28T11:30:00Z,53.516566,8.188217,None,3.922|2010-11-28T11:40:00Z,53.516566,8.188217,None,3.897|2010-11-28T11:50:00Z,53.516566,8.188217,None,3.642|2010-11-28T12:00:00Z,53.516566,8.188217,None,3.273|2010-11-28T12:10:00Z,53.516566,8.188217,None,3.303|2010-11-28T12:20:00Z,53.516566,8.188217,None,3.474|2010-11-28T12:30:00Z,53.516566,8.188217,None,3.62|2010-11-28T12:40:00Z,53.516566,8.188217,None,3.639|2010-11-28T12:50:00Z,53.516566,8.188217,None,3.451|2010-11-28T13:00:00Z,53.516566,8.188217,None,3.433|2010-11-28T13:10:00Z,53.516566,8.188217,None,3.686|2010-11-28T13:20:00Z,53.516566,8.188217,None,3.795|2010-11-28T13:30:00Z,53.516566,8.188217,None,3.812|2010-11-28T13:40:00Z,53.516566,8.188217,None,3.85|2010-11-28T13:50:00Z,53.516566,8.188217,None,3.954|2010-11-28T14:00:00Z,53.516566,8.188217,None,4.038|2010-11-28T14:10:00Z,53.516566,8.188217,None,4.187|2010-11-28T14:20:00Z,53.516566,8.188217,None,4.279|2010-11-28T14:30:00Z,53.516566,8.188217,None,4.36|2010-11-28T14:40:00Z,53.516566,8.188217,None,4.392|2010-11-28T14:50:00Z,53.516566,8.188217,None,4.5|2010-11-28T15:00:00Z,53.516566,8.188217,None,4.582|2010-11-28T15:10:00Z,53.516566,8.188217,None,4.621|2010-11-28T15:20:00Z,53.516566,8.188217,None,4.671|2010-11-28T15:30:00Z,53.516566,8.188217,None,4.831|2010-11-28T15:40:00Z,53.516566,8.188217,None,4.883|2010-11-28T15:50:00Z,53.516566,8.188217,None,4.879|2010-11-28T16:00:00Z,53.516566,8.188217,None,4.881|2010-11-28T16:10:00Z,53.516566,8.188217,None,4.876|2010-11-28T16:20:00Z,53.516566,8.188217,None,4.856|2010-11-28T16:30:00Z,53.516566,8.188217,None,4.819|2010-11-28T16:40:00Z,53.516566,8.188217,None,4.809|2010-11-28T16:50:00Z,53.516566,8.188217,None,4.818|2010-11-28T17:00:00Z,53.516566,8.188217,None,4.77|2010-11-28T17:10:00Z,53.516566,8.188217,None,4.669|2010-11-28T17:20:00Z,53.516566,8.188217,None,4.535|2010-11-28T17:30:00Z,53.516566,8.188217,None,4.275|2010-11-28T17:40:00Z,53.516566,8.188217,None,3.992|2010-11-28T17:50:00Z,53.516566,8.188217,None,3.628|2010-11-28T18:00:00Z,53.516566,8.188217,None,3.302|2010-11-28T18:10:00Z,53.516566,8.188217,None,3.19|2010-11-28T18:20:00Z,53.516566,8.188217,None,2.903|2010-11-28T18:30:00Z,53.516566,8.188217,None,3.054|2010-11-28T18:40:00Z,53.516566,8.188217,None,3.433|2010-11-28T18:50:00Z,53.516566,8.188217,None,3.402|2010-11-28T19:00:00Z,53.516566,8.188217,None,3.328|2010-11-28T19:10:00Z,53.516566,8.188217,None,3.301|2010-11-28T19:20:00Z,53.516566,8.188217,None,3.328|2010-11-28T19:30:00Z,53.516566,8.188217,None,3.346|2010-11-28T19:40:00Z,53.516566,8.188217,None,3.33|2010-11-28T19:50:00Z,53.516566,8.188217,None,3.305|2010-11-28T20:00:00Z,53.516566,8.188217,None,3.294|2010-11-28T20:10:00Z,53.516566,8.188217,None,3.293|2010-11-28T20:20:00Z,53.516566,8.188217,None,3.298|2010-11-28T20:30:00Z,53.516566,8.188217,None,3.28|2010-11-28T20:40:00Z,53.516566,8.188217,None,3.379|2010-11-28T20:50:00Z,53.516566,8.188217,None,3.461|2010-11-28T21:00:00Z,53.516566,8.188217,None,3.388|2010-11-28T21:10:00Z,53.516566,8.188217,None,3.069|2010-11-28T21:20:00Z,53.516566,8.188217,None,2.827|2010-11-28T21:30:00Z,53.516566,8.188217,None,2.836|2010-11-28T21:40:00Z,53.516566,8.188217,None,2.902|2010-11-28T21:50:00Z,53.516566,8.188217,None,3.148|2010-11-28T22:00:00Z,53.516566,8.188217,None,3.361|2010-11-28T22:10:00Z,53.516566,8.188217,None,3.402|2010-11-28T22:20:00Z,53.516566,8.188217,None,3.401|2010-11-28T22:30:00Z,53.516566,8.188217,None,3.432|2010-11-28T22:40:00Z,53.516566,8.188217,None,3.408|2010-11-28T22:50:00Z,53.516566,8.188217,None,3.396|2010-11-28T23:00:00Z,53.516566,8.188217,None,3.318|2010-11-28T23:10:00Z,53.516566,8.188217,None,2.985|2010-11-28T23:20:00Z,53.516566,8.188217,None,2.831|2010-11-28T23:30:00Z,53.516566,8.188217,None,3.134|2010-11-28T23:40:00Z,53.516566,8.188217,None,3.373|2010-11-28T23:50:00Z,53.516566,8.188217,None,3.511|2010-11-29T00:00:00Z,53.516566,8.188217,None,3.506|2010-11-29T00:10:00Z,53.516566,8.188217,None,3.429|2010-11-29T00:20:00Z,53.516566,8.188217,None,3.286|2010-11-29T00:30:00Z,53.516566,8.188217,None,2.975|2010-11-29T00:40:00Z,53.516566,8.188217,None,3.035|2010-11-29T00:50:00Z,53.516566,8.188217,None,3.111|2010-11-29T01:00:00Z,53.516566,8.188217,None,3.047|2010-11-29T01:10:00Z,53.516566,8.188217,None,3.182|2010-11-29T01:20:00Z,53.516566,8.188217,None,3.421|2010-11-29T01:30:00Z,53.516566,8.188217,None,3.49|2010-11-29T01:40:00Z,53.516566,8.188217,None,3.545|2010-11-29T01:50:00Z,53.516566,8.188217,None,3.774|2010-11-29T02:00:00Z,53.516566,8.188217,None,3.908|2010-11-29T02:10:00Z,53.516566,8.188217,None,3.931|2010-11-29T02:20:00Z,53.516566,8.188217,None,4.032|2010-11-29T02:30:00Z,53.516566,8.188217,None,4.19|2010-11-29T02:40:00Z,53.516566,8.188217,None,4.331|2010-11-29T02:50:00Z,53.516566,8.188217,None,4.441|2010-11-29T03:00:00Z,53.516566,8.188217,None,4.51|2010-11-29T03:10:00Z,53.516566,8.188217,None,4.578|2010-11-29T03:20:00Z,53.516566,8.188217,None,4.616|2010-11-29T03:30:00Z,53.516566,8.188217,None,4.634|2010-11-29T03:40:00Z,53.516566,8.188217,None,4.641|2010-11-29T03:50:00Z,53.516566,8.188217,None,4.627|2010-11-29T04:00:00Z,53.516566,8.188217,None,4.617|2010-11-29T04:10:00Z,53.516566,8.188217,None,4.62|2010-11-29T04:20:00Z,53.516566,8.188217,None,4.651|2010-11-29T04:30:00Z,53.516566,8.188217,None,4.683|2010-11-29T04:40:00Z,53.516566,8.188217,None,4.699|2010-11-29T04:50:00Z,53.516566,8.188217,None,4.683|2010-11-29T05:00:00Z,53.516566,8.188217,None,4.65|2010-11-29T05:10:00Z,53.516566,8.188217,None,4.573|2010-11-29T05:20:00Z,53.516566,8.188217,None,4.521|2010-11-29T05:30:00Z,53.516566,8.188217,None,4.499|2010-11-29T05:40:00Z,53.516566,8.188217,None,4.34|2010-11-29T05:50:00Z,53.516566,8.188217,None,3.892|2010-11-29T06:00:00Z,53.516566,8.188217,None,3.314|2010-11-29T06:10:00Z,53.516566,8.188217,None,2.893|2010-11-29T06:20:00Z,53.516566,8.188217,None,2.917|2010-11-29T06:30:00Z,53.516566,8.188217,None,2.653|2010-11-29T06:40:00Z,53.516566,8.188217,None,2.344|2010-11-29T06:50:00Z,53.516566,8.188217,None,2.301|2010-11-29T07:00:00Z,53.516566,8.188217,None,2.229|2010-11-29T07:10:00Z,53.516566,8.188217,None,2.253|2010-11-29T07:20:00Z,53.516566,8.188217,None,2.389|2010-11-29T07:30:00Z,53.516566,8.188217,None,2.699|2010-11-29T07:40:00Z,53.516566,8.188217,None,2.874|2010-11-29T07:50:00Z,53.516566,8.188217,None,2.874|2010-11-29T08:00:00Z,53.516566,8.188217,None,2.877|2010-11-29T08:10:00Z,53.516566,8.188217,None,2.882|2010-11-29T08:20:00Z,53.516566,8.188217,None,2.882|2010-11-29T08:30:00Z,53.516566,8.188217,None,2.879|2010-11-29T08:40:00Z,53.516566,8.188217,None,2.875|2010-11-29T08:50:00Z,53.516566,8.188217,None,2.9|2010-11-29T09:00:00Z,53.516566,8.188217,None,2.985|2010-11-29T09:10:00Z,53.516566,8.188217,None,3.052|2010-11-29T09:20:00Z,53.516566,8.188217,None,3.184|2010-11-29T09:30:00Z,53.516566,8.188217,None,3.095|2010-11-29T09:40:00Z,53.516566,8.188217,None,2.841|2010-11-29T09:50:00Z,53.516566,8.188217,None,2.539|2010-11-29T10:00:00Z,53.516566,8.188217,None,2.631|2010-11-29T10:10:00Z,53.516566,8.188217,None,2.848|2010-11-29T10:20:00Z,53.516566,8.188217,None,2.831|2010-11-29T10:30:00Z,53.516566,8.188217,None,3.009|2010-11-29T10:40:00Z,53.516566,8.188217,None,3.212|2010-11-29T10:50:00Z,53.516566,8.188217,None,3.173|2010-11-29T11:00:00Z,53.516566,8.188217,None,3.095|2010-11-29T11:10:00Z,53.516566,8.188217,None,3.115|2010-11-29T11:20:00Z,53.516566,8.188217,None,3.114|2010-11-29T11:30:00Z,53.516566,8.188217,None,3.113|2010-11-29T11:40:00Z,53.516566,8.188217,None,3.1|2010-11-29T11:50:00Z,53.516566,8.188217,None,3.055|2010-11-29T12:00:00Z,53.516566,8.188217,None,2.888|2010-11-29T12:10:00Z,53.516566,8.188217,None,2.718|2010-11-29T12:20:00Z,53.516566,8.188217,None,2.796|2010-11-29T12:30:00Z,53.516566,8.188217,None,3.166|2010-11-29T12:40:00Z,53.516566,8.188217,None,3.186|2010-11-29T12:50:00Z,53.516566,8.188217,None,2.867|2010-11-29T13:00:00Z,53.516566,8.188217,None,2.742|2010-11-29T13:10:00Z,53.516566,8.188217,None,2.855|2010-11-29T13:20:00Z,53.516566,8.188217,None,2.686|2010-11-29T13:30:00Z,53.516566,8.188217,None,2.622|2010-11-29T13:40:00Z,53.516566,8.188217,None,2.594|2010-11-29T13:50:00Z,53.516566,8.188217,None,2.594|2010-11-29T14:00:00Z,53.516566,8.188217,None,2.746|2010-11-29T14:10:00Z,53.516566,8.188217,None,2.716|2010-11-29T14:20:00Z,53.516566,8.188217,None,2.932|2010-11-29T14:30:00Z,53.516566,8.188217,None,3.068|2010-11-29T14:40:00Z,53.516566,8.188217,None,3.135|2010-11-29T14:50:00Z,53.516566,8.188217,None,3.275|2010-11-29T15:00:00Z,53.516566,8.188217,None,3.483|2010-11-29T15:10:00Z,53.516566,8.188217,None,3.52|2010-11-29T15:20:00Z,53.516566,8.188217,None,3.615|2010-11-29T15:30:00Z,53.516566,8.188217,None,3.674|2010-11-29T15:40:00Z,53.516566,8.188217,None,3.653|2010-11-29T15:50:00Z,53.516566,8.188217,None,3.685|2010-11-29T16:00:00Z,53.516566,8.188217,None,3.789|2010-11-29T16:10:00Z,53.516566,8.188217,None,3.926|2010-11-29T16:20:00Z,53.516566,8.188217,None,3.991|2010-11-29T16:30:00Z,53.516566,8.188217,None,4.039|2010-11-29T16:40:00Z,53.516566,8.188217,None,4.085|2010-11-29T16:50:00Z,53.516566,8.188217,None,4.103|2010-11-29T17:00:00Z,53.516566,8.188217,None,4.118|2010-11-29T17:10:00Z,53.516566,8.188217,None,4.134|2010-11-29T17:20:00Z,53.516566,8.188217,None,4.124|2010-11-29T17:30:00Z,53.516566,8.188217,None,4.1|2010-11-29T17:40:00Z,53.516566,8.188217,None,4.031|2010-11-29T17:50:00Z,53.516566,8.188217,None,3.949|2010-11-29T18:00:00Z,53.516566,8.188217,None,3.795|2010-11-29T18:10:00Z,53.516566,8.188217,None,3.567|2010-11-29T18:20:00Z,53.516566,8.188217,None,3.42|2010-11-29T18:30:00Z,53.516566,8.188217,None,3.12|2010-11-29T18:40:00Z,53.516566,8.188217,None,2.691|2010-11-29T18:50:00Z,53.516566,8.188217,None,2.536|2010-11-29T19:00:00Z,53.516566,8.188217,None,2.54|2010-11-29T19:10:00Z,53.516566,8.188217,None,2.458|2010-11-29T19:20:00Z,53.516566,8.188217,None,2.382|2010-11-29T19:30:00Z,53.516566,8.188217,None,2.386|2010-11-29T19:40:00Z,53.516566,8.188217,None,2.415|2010-11-29T19:50:00Z,53.516566,8.188217,None,2.406|2010-11-29T20:00:00Z,53.516566,8.188217,None,2.364|2010-11-29T20:10:00Z,53.516566,8.188217,None,2.335|2010-11-29T20:20:00Z,53.516566,8.188217,None,2.346|2010-11-29T20:30:00Z,53.516566,8.188217,None,2.37|2010-11-29T20:40:00Z,53.516566,8.188217,None,2.377|2010-11-29T20:50:00Z,53.516566,8.188217,None,2.369|2010-11-29T21:00:00Z,53.516566,8.188217,None,2.365|2010-11-29T21:10:00Z,53.516566,8.188217,None,2.354|2010-11-29T21:20:00Z,53.516566,8.188217,None,2.336|2010-11-29T21:30:00Z,53.516566,8.188217,None,2.339|2010-11-29T21:40:00Z,53.516566,8.188217,None,2.393|2010-11-29T21:50:00Z,53.516566,8.188217,None,2.35|2010-11-29T22:00:00Z,53.516566,8.188217,None,2.534|2010-11-29T22:10:00Z,53.516566,8.188217,None,2.57|2010-11-29T22:20:00Z,53.516566,8.188217,None,2.567|2010-11-29T22:30:00Z,53.516566,8.188217,None,2.566|2010-11-29T22:40:00Z,53.516566,8.188217,None,2.548|2010-11-29T22:50:00Z,53.516566,8.188217,None,2.448|2010-11-29T23:00:00Z,53.516566,8.188217,None,2.389|2010-11-29T23:10:00Z,53.516566,8.188217,None,2.393|2010-11-29T23:20:00Z,53.516566,8.188217,None,2.375|2010-11-29T23:30:00Z,53.516566,8.188217,None,2.372|2010-11-29T23:40:00Z,53.516566,8.188217,None,2.431|2010-11-29T23:50:00Z,53.516566,8.188217,None,2.605|2010-11-30T00:00:00Z,53.516566,8.188217,None,2.778|2010-11-30T00:10:00Z,53.516566,8.188217,None,2.905|2010-11-30T00:20:00Z,53.516566,8.188217,None,2.876|2010-11-30T00:30:00Z,53.516566,8.188217,None,2.812|2010-11-30T00:40:00Z,53.516566,8.188217,None,2.809|2010-11-30T00:50:00Z,53.516566,8.188217,None,2.804|2010-11-30T01:00:00Z,53.516566,8.188217,None,2.83|2010-11-30T01:10:00Z,53.516566,8.188217,None,2.842|2010-11-30T01:20:00Z,53.516566,8.188217,None,2.806|2010-11-30T01:30:00Z,53.516566,8.188217,None,2.743|2010-11-30T01:40:00Z,53.516566,8.188217,None,2.542|2010-11-30T01:50:00Z,53.516566,8.188217,None,2.477|2010-11-30T02:00:00Z,53.516566,8.188217,None,2.576|2010-11-30T02:10:00Z,53.516566,8.188217,None,2.669|2010-11-30T02:20:00Z,53.516566,8.188217,None,2.968|2010-11-30T02:30:00Z,53.516566,8.188217,None,2.934|2010-11-30T02:40:00Z,53.516566,8.188217,None,2.993|2010-11-30T02:50:00Z,53.516566,8.188217,None,3.17|2010-11-30T03:00:00Z,53.516566,8.188217,None,3.292|2010-11-30T03:10:00Z,53.516566,8.188217,None,3.269|2010-11-30T03:20:00Z,53.516566,8.188217,None,3.258|2010-11-30T03:30:00Z,53.516566,8.188217,None,3.314|2010-11-30T03:40:00Z,53.516566,8.188217,None,3.341|2010-11-30T03:50:00Z,53.516566,8.188217,None,3.364|2010-11-30T04:00:00Z,53.516566,8.188217,None,3.465|2010-11-30T04:10:00Z,53.516566,8.188217,None,3.587|2010-11-30T04:20:00Z,53.516566,8.188217,None,3.682|2010-11-30T04:30:00Z,53.516566,8.188217,None,3.747|2010-11-30T04:40:00Z,53.516566,8.188217,None,3.795|2010-11-30T04:50:00Z,53.516566,8.188217,None,3.844|2010-11-30T05:00:00Z,53.516566,8.188217,None,3.882|2010-11-30T05:10:00Z,53.516566,8.188217,None,3.895|2010-11-30T05:20:00Z,53.516566,8.188217,None,3.899|2010-11-30T05:30:00Z,53.516566,8.188217,None,3.901|2010-11-30T05:40:00Z,53.516566,8.188217,None,3.897|2010-11-30T05:50:00Z,53.516566,8.188217,None,3.885|2010-11-30T06:00:00Z,53.516566,8.188217,None,3.842|2010-11-30T06:10:00Z,53.516566,8.188217,None,3.736|2010-11-30T06:20:00Z,53.516566,8.188217,None,3.531|2010-11-30T06:30:00Z,53.516566,8.188217,None,3.251|2010-11-30T06:40:00Z,53.516566,8.188217,None,3.094|2010-11-30T06:50:00Z,53.516566,8.188217,None,2.965|2010-11-30T07:00:00Z,53.516566,8.188217,None,2.674|2010-11-30T07:10:00Z,53.516566,8.188217,None,2.264|2010-11-30T07:20:00Z,53.516566,8.188217,None,2.11|2010-11-30T07:30:00Z,53.516566,8.188217,None,2.223|2010-11-30T07:40:00Z,53.516566,8.188217,None,2.047|2010-11-30T07:50:00Z,53.516566,8.188217,None,1.673|2010-11-30T08:00:00Z,53.516566,8.188217,None,1.495|2010-11-30T08:10:00Z,53.516566,8.188217,None,1.511|2010-11-30T08:20:00Z,53.516566,8.188217,None,1.565|2010-11-30T08:30:00Z,53.516566,8.188217,None,1.575|2010-11-30T08:40:00Z,53.516566,8.188217,None,1.523|2010-11-30T08:50:00Z,53.516566,8.188217,None,1.518|2010-11-30T09:00:00Z,53.516566,8.188217,None,1.515|2010-11-30T09:10:00Z,53.516566,8.188217,None,1.507|2010-11-30T09:20:00Z,53.516566,8.188217,None,1.534|2010-11-30T09:30:00Z,53.516566,8.188217,None,1.595|2010-11-30T09:40:00Z,53.516566,8.188217,None,1.596|2010-11-30T09:50:00Z,53.516566,8.188217,None,1.616|2010-11-30T10:00:00Z,53.516566,8.188217,None,1.633|2010-11-30T10:10:00Z,53.516566,8.188217,None,1.536|2010-11-30T10:20:00Z,53.516566,8.188217,None,1.478|2010-11-30T10:30:00Z,53.516566,8.188217,None,1.708|2010-11-30T10:40:00Z,53.516566,8.188217,None,1.793|2010-11-30T10:50:00Z,53.516566,8.188217,None,1.797|2010-11-30T11:00:00Z,53.516566,8.188217,None,1.789|2010-11-30T11:10:00Z,53.516566,8.188217,None,1.758|2010-11-30T11:20:00Z,53.516566,8.188217,None,1.748|2010-11-30T11:30:00Z,53.516566,8.188217,None,1.805|2010-11-30T11:40:00Z,53.516566,8.188217,None,1.818|2010-11-30T11:50:00Z,53.516566,8.188217,None,1.728|2010-11-30T12:00:00Z,53.516566,8.188217,None,1.768|2010-11-30T12:10:00Z,53.516566,8.188217,None,1.734|2010-11-30T12:20:00Z,53.516566,8.188217,None,1.652|2010-11-30T12:30:00Z,53.516566,8.188217,None,1.647|2010-11-30T12:40:00Z,53.516566,8.188217,None,1.962|2010-11-30T12:50:00Z,53.516566,8.188217,None,2.281|2010-11-30T13:00:00Z,53.516566,8.188217,None,2.418|2010-11-30T13:10:00Z,53.516566,8.188217,None,2.43|2010-11-30T13:20:00Z,53.516566,8.188217,None,2.385|2010-11-30T13:30:00Z,53.516566,8.188217,None,2.435|2010-11-30T13:40:00Z,53.516566,8.188217,None,2.48|2010-11-30T13:50:00Z,53.516566,8.188217,None,2.546|2010-11-30T14:00:00Z,53.516566,8.188217,None,2.485|2010-11-30T14:10:00Z,53.516566,8.188217,None,2.384|2010-11-30T14:20:00Z,53.516566,8.188217,None,2.339|2010-11-30T14:30:00Z,53.516566,8.188217,None,2.225|2010-11-30T14:40:00Z,53.516566,8.188217,None,1.811|2010-11-30T14:50:00Z,53.516566,8.188217,None,1.744|2010-11-30T15:00:00Z,53.516566,8.188217,None,1.848|2010-11-30T15:10:00Z,53.516566,8.188217,None,2.019|2010-11-30T15:20:00Z,53.516566,8.188217,None,2.347|2010-11-30T15:30:00Z,53.516566,8.188217,None,2.613|2010-11-30T15:40:00Z,53.516566,8.188217,None,2.584|2010-11-30T15:50:00Z,53.516566,8.188217,None,2.591|2010-11-30T16:00:00Z,53.516566,8.188217,None,2.833|2010-11-30T16:10:00Z,53.516566,8.188217,None,2.963|2010-11-30T16:20:00Z,53.516566,8.188217,None,2.94|2010-11-30T16:30:00Z,53.516566,8.188217,None,2.9|2010-11-30T16:40:00Z,53.516566,8.188217,None,2.932|2010-11-30T16:50:00Z,53.516566,8.188217,None,2.976|2010-11-30T17:00:00Z,53.516566,8.188217,None,2.995|2010-11-30T17:10:00Z,53.516566,8.188217,None,3.02|2010-11-30T17:20:00Z,53.516566,8.188217,None,3.086|2010-11-30T17:30:00Z,53.516566,8.188217,None,3.164|2010-11-30T17:40:00Z,53.516566,8.188217,None,3.23|2010-11-30T17:50:00Z,53.516566,8.188217,None,3.274|2010-11-30T18:00:00Z,53.516566,8.188217,None,3.304|2010-11-30T18:10:00Z,53.516566,8.188217,None,3.319|2010-11-30T18:20:00Z,53.516566,8.188217,None,3.33|2010-11-30T18:30:00Z,53.516566,8.188217,None,3.333|2010-11-30T18:40:00Z,53.516566,8.188217,None,3.327|2010-11-30T18:50:00Z,53.516566,8.188217,None,3.295|2010-11-30T19:00:00Z,53.516566,8.188217,None,3.192|2010-11-30T19:10:00Z,53.516566,8.188217,None,3.01|2010-11-30T19:20:00Z,53.516566,8.188217,None,2.819|2010-11-30T19:30:00Z,53.516566,8.188217,None,2.693|2010-11-30T19:40:00Z,53.516566,8.188217,None,2.647|2010-11-30T19:50:00Z,53.516566,8.188217,None,2.106|2010-11-30T20:00:00Z,53.516566,8.188217,None,1.657|2010-11-30T20:10:00Z,53.516566,8.188217,None,1.502|2010-11-30T20:20:00Z,53.516566,8.188217,None,1.193|2010-11-30T20:30:00Z,53.516566,8.188217,None,1.231|2010-11-30T20:40:00Z,53.516566,8.188217,None,1.22|2010-11-30T20:50:00Z,53.516566,8.188217,None,1.197|2010-11-30T21:00:00Z,53.516566,8.188217,None,1.177|2010-11-30T21:10:00Z,53.516566,8.188217,None,1.116|2010-11-30T21:20:00Z,53.516566,8.188217,None,1.068|2010-11-30T21:30:00Z,53.516566,8.188217,None,1.037|2010-11-30T21:40:00Z,53.516566,8.188217,None,1.018|2010-11-30T21:50:00Z,53.516566,8.188217,None,1.01|2010-11-30T22:00:00Z,53.516566,8.188217,None,1.006|2010-11-30T22:10:00Z,53.516566,8.188217,None,1.011|2010-11-30T22:20:00Z,53.516566,8.188217,None,1.111|2010-11-30T22:30:00Z,53.516566,8.188217,None,1.138|2010-11-30T22:40:00Z,53.516566,8.188217,None,1.199|2010-11-30T22:50:00Z,53.516566,8.188217,None,1.194|2010-11-30T23:00:00Z,53.516566,8.188217,None,0.842|2010-11-30T23:10:00Z,53.516566,8.188217,None,1.027|2010-11-30T23:20:00Z,53.516566,8.188217,None,1.526|2010-11-30T23:30:00Z,53.516566,8.188217,None,1.664|2010-11-30T23:40:00Z,53.516566,8.188217,None,1.65|2010-11-30T23:50:00Z,53.516566,8.188217,None,1.598|2010-12-01T00:00:00Z,53.516566,8.188217,None,1.496|2010-12-01T00:10:00Z,53.516566,8.188217,None,1.385|2010-12-01T00:20:00Z,53.516566,8.188217,None,1.234|2010-12-01T00:30:00Z,53.516566,8.188217,None,1.054|2010-12-01T00:40:00Z,53.516566,8.188217,None,1.129|2010-12-01T00:50:00Z,53.516566,8.188217,None,1.387|2010-12-01T01:00:00Z,53.516566,8.188217,None,1.333|2010-12-01T01:10:00Z,53.516566,8.188217,None,1.554|2010-12-01T01:20:00Z,53.516566,8.188217,None,1.662|2010-12-01T01:30:00Z,53.516566,8.188217,None,1.813|2010-12-01T01:40:00Z,53.516566,8.188217,None,2.095|2010-12-01T01:50:00Z,53.516566,8.188217,None,2.236|2010-12-01T02:00:00Z,53.516566,8.188217,None,2.199|2010-12-01T02:10:00Z,53.516566,8.188217,None,2.148|2010-12-01T02:20:00Z,53.516566,8.188217,None,2.075|2010-12-01T02:30:00Z,53.516566,8.188217,None,2.118|2010-12-01T02:40:00Z,53.516566,8.188217,None,2.073|2010-12-01T02:50:00Z,53.516566,8.188217,None,1.841|2010-12-01T03:00:00Z,53.516566,8.188217,None,1.686|2010-12-01T03:10:00Z,53.516566,8.188217,None,1.523|2010-12-01T03:20:00Z,53.516566,8.188217,None,1.521|2010-12-01T03:30:00Z,53.516566,8.188217,None,1.63|2010-12-01T03:40:00Z,53.516566,8.188217,None,1.727|2010-12-01T03:50:00Z,53.516566,8.188217,None,1.997|2010-12-01T04:00:00Z,53.516566,8.188217,None,2.197|2010-12-01T04:10:00Z,53.516566,8.188217,None,2.333|2010-12-01T04:20:00Z,53.516566,8.188217,None,2.308|2010-12-01T04:30:00Z,53.516566,8.188217,None,2.356|2010-12-01T04:40:00Z,53.516566,8.188217,None,2.5|2010-12-01T04:50:00Z,53.516566,8.188217,None,2.575|2010-12-01T05:00:00Z,53.516566,8.188217,None,2.526|2010-12-01T05:10:00Z,53.516566,8.188217,None,2.434|2010-12-01T05:20:00Z,53.516566,8.188217,None,2.43|2010-12-01T05:30:00Z,53.516566,8.188217,None,2.543|2010-12-01T05:40:00Z,53.516566,8.188217,None,2.675|2010-12-01T05:50:00Z,53.516566,8.188217,None,2.734|2010-12-01T06:00:00Z,53.516566,8.188217,None,2.792|2010-12-01T06:10:00Z,53.516566,8.188217,None,2.844|2010-12-01T06:20:00Z,53.516566,8.188217,None,2.875|2010-12-01T06:30:00Z,53.516566,8.188217,None,2.893|2010-12-01T06:40:00Z,53.516566,8.188217,None,2.901|2010-12-01T06:50:00Z,53.516566,8.188217,None,2.876|2010-12-01T07:00:00Z,53.516566,8.188217,None,2.842|2010-12-01T07:10:00Z,53.516566,8.188217,None,2.73|2010-12-01T07:20:00Z,53.516566,8.188217,None,2.559|2010-12-01T07:30:00Z,53.516566,8.188217,None,2.291|2010-12-01T07:40:00Z,53.516566,8.188217,None,2.103|2010-12-01T07:50:00Z,53.516566,8.188217,None,1.851|2010-12-01T08:00:00Z,53.516566,8.188217,None,1.045|2010-12-01T08:10:00Z,53.516566,8.188217,None,0.729|2010-12-01T08:20:00Z,53.516566,8.188217,None,0.992|2010-12-01T08:30:00Z,53.516566,8.188217,None,1.316|2010-12-01T08:40:00Z,53.516566,8.188217,None,1.597|2010-12-01T08:50:00Z,53.516566,8.188217,None,1.503|2010-12-01T09:00:00Z,53.516566,8.188217,None,1.402|2010-12-01T09:10:00Z,53.516566,8.188217,None,1.401|2010-12-01T09:20:00Z,53.516566,8.188217,None,1.42|2010-12-01T09:30:00Z,53.516566,8.188217,None,1.429|2010-12-01T09:40:00Z,53.516566,8.188217,None,1.4|2010-12-01T09:50:00Z,53.516566,8.188217,None,1.365|2010-12-01T10:00:00Z,53.516566,8.188217,None,1.303|2010-12-01T10:10:00Z,53.516566,8.188217,None,1.268|2010-12-01T10:20:00Z,53.516566,8.188217,None,1.232|2010-12-01T10:30:00Z,53.516566,8.188217,None,1.228|2010-12-01T10:40:00Z,53.516566,8.188217,None,1.2|2010-12-01T10:50:00Z,53.516566,8.188217,None,1.126|2010-12-01T11:00:00Z,53.516566,8.188217,None,1.104|2010-12-01T11:10:00Z,53.516566,8.188217,None,1.233|2010-12-01T11:20:00Z,53.516566,8.188217,None,1.504|2010-12-01T11:30:00Z,53.516566,8.188217,None,1.616|2010-12-01T11:40:00Z,53.516566,8.188217,None,1.477|2010-12-01T11:50:00Z,53.516566,8.188217,None,1.363|2010-12-01T12:00:00Z,53.516566,8.188217,None,1.409|2010-12-01T12:10:00Z,53.516566,8.188217,None,1.46|2010-12-01T12:20:00Z,53.516566,8.188217,None,1.24|2010-12-01T12:30:00Z,53.516566,8.188217,None,1.104|2010-12-01T12:40:00Z,53.516566,8.188217,None,1.213|2010-12-01T12:50:00Z,53.516566,8.188217,None,0.982|2010-12-01T13:00:00Z,53.516566,8.188217,None,0.792|2010-12-01T13:10:00Z,53.516566,8.188217,None,0.154|2010-12-01T13:20:00Z,53.516566,8.188217,None,0.275|2010-12-01T13:30:00Z,53.516566,8.188217,None,0.605|2010-12-01T13:40:00Z,53.516566,8.188217,None,1.248|2010-12-01T13:50:00Z,53.516566,8.188217,None,1.568|2010-12-01T14:00:00Z,53.516566,8.188217,None,1.733|2010-12-01T14:10:00Z,53.516566,8.188217,None,1.714|2010-12-01T14:20:00Z,53.516566,8.188217,None,1.687|2010-12-01T14:30:00Z,53.516566,8.188217,None,1.6|2010-12-01T14:40:00Z,53.516566,8.188217,None,1.517|2010-12-01T14:50:00Z,53.516566,8.188217,None,1.402|2010-12-01T15:00:00Z,53.516566,8.188217,None,1.287|2010-12-01T15:10:00Z,53.516566,8.188217,None,1.223|2010-12-01T15:20:00Z,53.516566,8.188217,None,1.165|2010-12-01T15:30:00Z,53.516566,8.188217,None,1.129|2010-12-01T15:40:00Z,53.516566,8.188217,None,1.107|2010-12-01T15:50:00Z,53.516566,8.188217,None,1.105|2010-12-01T16:00:00Z,53.516566,8.188217,None,1.121|2010-12-01T16:10:00Z,53.516566,8.188217,None,1.22|2010-12-01T16:20:00Z,53.516566,8.188217,None,1.251|2010-12-01T16:30:00Z,53.516566,8.188217,None,1.208|2010-12-01T16:40:00Z,53.516566,8.188217,None,1.227|2010-12-01T16:50:00Z,53.516566,8.188217,None,1.166|2010-12-01T17:00:00Z,53.516566,8.188217,None,1.077|2010-12-01T17:10:00Z,53.516566,8.188217,None,1.015|2010-12-01T17:20:00Z,53.516566,8.188217,None,1.118|2010-12-01T17:30:00Z,53.516566,8.188217,None,1.163|2010-12-01T17:40:00Z,53.516566,8.188217,None,1.391|2010-12-01T17:50:00Z,53.516566,8.188217,None,1.396|2010-12-01T18:00:00Z,53.516566,8.188217,None,1.415|2010-12-01T18:10:00Z,53.516566,8.188217,None,1.549|2010-12-01T18:20:00Z,53.516566,8.188217,None,1.693|2010-12-01T18:30:00Z,53.516566,8.188217,None,1.766|2010-12-01T18:40:00Z,53.516566,8.188217,None,1.788|2010-12-01T18:50:00Z,53.516566,8.188217,None,1.79|2010-12-01T19:00:00Z,53.516566,8.188217,None,1.782|2010-12-01T19:10:00Z,53.516566,8.188217,None,1.768|2010-12-01T19:20:00Z,53.516566,8.188217,None,1.761|2010-12-01T19:30:00Z,53.516566,8.188217,None,1.747|2010-12-01T19:40:00Z,53.516566,8.188217,None,1.729|2010-12-01T19:50:00Z,53.516566,8.188217,None,1.705|2010-12-01T20:00:00Z,53.516566,8.188217,None,1.627|2010-12-01T20:10:00Z,53.516566,8.188217,None,1.533|2010-12-01T20:20:00Z,53.516566,8.188217,None,1.443|2010-12-01T20:30:00Z,53.516566,8.188217,None,1.09|2010-12-01T20:40:00Z,53.516566,8.188217,None,0.659|2010-12-01T20:50:00Z,53.516566,8.188217,None,0.887|2010-12-01T21:00:00Z,53.516566,8.188217,None,0.783|2010-12-01T21:10:00Z,53.516566,8.188217,None,0.042|2010-12-01T21:20:00Z,53.516566,8.188217,None,0.584|2010-12-01T21:30:00Z,53.516566,8.188217,None,0.718|2010-12-01T21:40:00Z,53.516566,8.188217,None,0.757|2010-12-01T21:50:00Z,53.516566,8.188217,None,0.806|2010-12-01T22:00:00Z,53.516566,8.188217,None,0.842|2010-12-01T22:10:00Z,53.516566,8.188217,None,0.855|2010-12-01T22:20:00Z,53.516566,8.188217,None,0.878|2010-12-01T22:30:00Z,53.516566,8.188217,None,0.91|2010-12-01T22:40:00Z,53.516566,8.188217,None,0.924|2010-12-01T22:50:00Z,53.516566,8.188217,None,0.912|2010-12-01T23:00:00Z,53.516566,8.188217,None,0.883|2010-12-01T23:10:00Z,53.516566,8.188217,None,0.848|2010-12-01T23:20:00Z,53.516566,8.188217,None,0.808|2010-12-01T23:30:00Z,53.516566,8.188217,None,0.764|2010-12-01T23:40:00Z,53.516566,8.188217,None,0.667|2010-12-01T23:50:00Z,53.516566,8.188217,None,0.962|2010-12-02T00:00:00Z,53.516566,8.188217,None,1.007|2010-12-02T00:10:00Z,53.516566,8.188217,None,1.048|2010-12-02T00:20:00Z,53.516566,8.188217,None,1.061|2010-12-02T00:30:00Z,53.516566,8.188217,None,1.058|2010-12-02T00:40:00Z,53.516566,8.188217,None,1.029|2010-12-02T00:50:00Z,53.516566,8.188217,None,0.807|2010-12-02T01:00:00Z,53.516566,8.188217,None,0.715|2010-12-02T01:10:00Z,53.516566,8.188217,None,0.732|2010-12-02T01:20:00Z,53.516566,8.188217,None,0.776|2010-12-02T01:30:00Z,53.516566,8.188217,None,0.818|2010-12-02T01:40:00Z,53.516566,8.188217,None,0.777|2010-12-02T01:50:00Z,53.516566,8.188217,None,0.531|2010-12-02T02:00:00Z,53.516566,8.188217,None,0.557|2010-12-02T02:10:00Z,53.516566,8.188217,None,0.76|2010-12-02T02:20:00Z,53.516566,8.188217,None,1.037|2010-12-02T02:30:00Z,53.516566,8.188217,None,1.127|2010-12-02T02:40:00Z,53.516566,8.188217,None,1.04|2010-12-02T02:50:00Z,53.516566,8.188217,None,0.8|2010-12-02T03:00:00Z,53.516566,8.188217,None,0.534|2010-12-02T03:10:00Z,53.516566,8.188217,None,0.401|2010-12-02T03:20:00Z,53.516566,8.188217,None,0.338|2010-12-02T03:30:00Z,53.516566,8.188217,None,0.329|2010-12-02T03:40:00Z,53.516566,8.188217,None,0.301|2010-12-02T03:50:00Z,53.516566,8.188217,None,0.283|2010-12-02T04:00:00Z,53.516566,8.188217,None,0.335|2010-12-02T04:10:00Z,53.516566,8.188217,None,0.369|2010-12-02T04:20:00Z,53.516566,8.188217,None,0.519|2010-12-02T04:30:00Z,53.516566,8.188217,None,0.545|2010-12-02T04:40:00Z,53.516566,8.188217,None,0.49|2010-12-02T04:50:00Z,53.516566,8.188217,None,0.495|2010-12-02T05:00:00Z,53.516566,8.188217,None,0.66|2010-12-02T05:10:00Z,53.516566,8.188217,None,0.933|2010-12-02T05:20:00Z,53.516566,8.188217,None,0.892|2010-12-02T05:30:00Z,53.516566,8.188217,None,0.889|2010-12-02T05:40:00Z,53.516566,8.188217,None,1.133|2010-12-02T05:50:00Z,53.516566,8.188217,None,1.225|2010-12-02T06:00:00Z,53.516566,8.188217,None,1.227|2010-12-02T06:10:00Z,53.516566,8.188217,None,1.134|2010-12-02T06:20:00Z,53.516566,8.188217,None,1.032|2010-12-02T06:30:00Z,53.516566,8.188217,None,1.046|2010-12-02T06:40:00Z,53.516566,8.188217,None,1.082|2010-12-02T06:50:00Z,53.516566,8.188217,None,1.123|2010-12-02T07:00:00Z,53.516566,8.188217,None,1.176|2010-12-02T07:10:00Z,53.516566,8.188217,None,1.328|2010-12-02T07:20:00Z,53.516566,8.188217,None,1.53|2010-12-02T07:30:00Z,53.516566,8.188217,None,1.667|2010-12-02T07:40:00Z,53.516566,8.188217,None,1.715|2010-12-02T07:50:00Z,53.516566,8.188217,None,1.797|2010-12-02T08:00:00Z,53.516566,8.188217,None,1.847|2010-12-02T08:10:00Z,53.516566,8.188217,None,1.89|2010-12-02T08:20:00Z,53.516566,8.188217,None,1.914|2010-12-02T08:30:00Z,53.516566,8.188217,None,1.911|2010-12-02T08:40:00Z,53.516566,8.188217,None,1.882|2010-12-02T08:50:00Z,53.516566,8.188217,None,1.78|2010-12-02T09:00:00Z,53.516566,8.188217,None,1.566|2010-12-02T09:10:00Z,53.516566,8.188217,None,1.204|2010-12-02T09:20:00Z,53.516566,8.188217,None,0.974|2010-12-02T09:30:00Z,53.516566,8.188217,None,0.975|2010-12-02T09:40:00Z,53.516566,8.188217,None,0.624|2010-12-02T09:50:00Z,53.516566,8.188217,None,0.187|2010-12-02T10:00:00Z,53.516566,8.188217,None,0.173|2010-12-02T10:10:00Z,53.516566,8.188217,None,0.392|2010-12-02T10:20:00Z,53.516566,8.188217,None,0.261|2010-12-02T10:30:00Z,53.516566,8.188217,None,-0.103|2010-12-02T10:40:00Z,53.516566,8.188217,None,-0.577|2010-12-02T10:50:00Z,53.516566,8.188217,None,-0.721|2010-12-02T11:00:00Z,53.516566,8.188217,None,-0.835|2010-12-02T11:10:00Z,53.516566,8.188217,None,-1.116|2010-12-02T11:20:00Z,53.516566,8.188217,None,-1.283|2010-12-02T11:30:00Z,53.516566,8.188217,None,-1.31|2010-12-02T11:40:00Z,53.516566,8.188217,None,-1.292|2010-12-02T11:50:00Z,53.516566,8.188217,None,-1.276|2010-12-02T12:00:00Z,53.516566,8.188217,None,-1.278|2010-12-02T12:10:00Z,53.516566,8.188217,None,-1.274|2010-12-02T12:20:00Z,53.516566,8.188217,None,-1.282|2010-12-02T12:30:00Z,53.516566,8.188217,None,-1.289|2010-12-02T12:40:00Z,53.516566,8.188217,None,-1.289|2010-12-02T12:50:00Z,53.516566,8.188217,None,-1.274|2010-12-02T13:00:00Z,53.516566,8.188217,None,-1.274|2010-12-02T13:10:00Z,53.516566,8.188217,None,-1.161|2010-12-02T13:20:00Z,53.516566,8.188217,None,-1.073|2010-12-02T13:30:00Z,53.516566,8.188217,None,-0.368|2010-12-02T13:40:00Z,53.516566,8.188217,None,0.22|2010-12-02T13:50:00Z,53.516566,8.188217,None,0.126|2010-12-02T14:00:00Z,53.516566,8.188217,None,-0.435|2010-12-02T14:10:00Z,53.516566,8.188217,None,-0.353|2010-12-02T14:20:00Z,53.516566,8.188217,None,-0.667|2010-12-02T14:30:00Z,53.516566,8.188217,None,-1.082|2010-12-02T14:40:00Z,53.516566,8.188217,None,-1.166|2010-12-02T14:50:00Z,53.516566,8.188217,None,-1.143|2010-12-02T15:00:00Z,53.516566,8.188217,None,-1.214|2010-12-02T15:10:00Z,53.516566,8.188217,None,-1.135|2010-12-02T15:20:00Z,53.516566,8.188217,None,-0.599|2010-12-02T15:30:00Z,53.516566,8.188217,None,0.135|2010-12-02T15:40:00Z,53.516566,8.188217,None,0.903|2010-12-02T15:50:00Z,53.516566,8.188217,None,0.745|2010-12-02T16:00:00Z,53.516566,8.188217,None,0.647|2010-12-02T16:10:00Z,53.516566,8.188217,None,0.587|2010-12-02T16:20:00Z,53.516566,8.188217,None,0.476|2010-12-02T16:30:00Z,53.516566,8.188217,None,0.3|2010-12-02T16:40:00Z,53.516566,8.188217,None,0.111|2010-12-02T16:50:00Z,53.516566,8.188217,None,-0.073|2010-12-02T17:00:00Z,53.516566,8.188217,None,0.026|2010-12-02T17:10:00Z,53.516566,8.188217,None,0.22|2010-12-02T17:20:00Z,53.516566,8.188217,None,0.61|2010-12-02T17:30:00Z,53.516566,8.188217,None,0.834|2010-12-02T17:40:00Z,53.516566,8.188217,None,0.93|2010-12-02T17:50:00Z,53.516566,8.188217,None,0.976|2010-12-02T18:00:00Z,53.516566,8.188217,None,0.996|2010-12-02T18:10:00Z,53.516566,8.188217,None,0.981|2010-12-02T18:20:00Z,53.516566,8.188217,None,0.967|2010-12-02T18:30:00Z,53.516566,8.188217,None,0.954|2010-12-02T18:40:00Z,53.516566,8.188217,None,1.007|2010-12-02T18:50:00Z,53.516566,8.188217,None,1.042|2010-12-02T19:00:00Z,53.516566,8.188217,None,1.066|2010-12-02T19:10:00Z,53.516566,8.188217,None,1.165|2010-12-02T19:20:00Z,53.516566,8.188217,None,1.282|2010-12-02T19:30:00Z,53.516566,8.188217,None,1.382|2010-12-02T19:40:00Z,53.516566,8.188217,None,1.469|2010-12-02T19:50:00Z,53.516566,8.188217,None,1.546|2010-12-02T20:00:00Z,53.516566,8.188217,None,1.61|2010-12-02T20:10:00Z,53.516566,8.188217,None,1.659|2010-12-02T20:20:00Z,53.516566,8.188217,None,1.685|2010-12-02T20:30:00Z,53.516566,8.188217,None,1.697|2010-12-02T20:40:00Z,53.516566,8.188217,None,1.71|2010-12-02T20:50:00Z,53.516566,8.188217,None,1.711|2010-12-02T21:00:00Z,53.516566,8.188217,None,1.706|2010-12-02T21:10:00Z,53.516566,8.188217,None,1.693|2010-12-02T21:20:00Z,53.516566,8.188217,None,1.662|2010-12-02T21:30:00Z,53.516566,8.188217,None,1.61|2010-12-02T21:40:00Z,53.516566,8.188217,None,1.543|2010-12-02T21:50:00Z,53.516566,8.188217,None,1.375|2010-12-02T22:00:00Z,53.516566,8.188217,None,1.115|2010-12-02T22:10:00Z,53.516566,8.188217,None,0.961|2010-12-02T22:20:00Z,53.516566,8.188217,None,0.874|2010-12-02T22:30:00Z,53.516566,8.188217,None,0.48|2010-12-02T22:40:00Z,53.516566,8.188217,None,-0.033|2010-12-02T22:50:00Z,53.516566,8.188217,None,-0.177|2010-12-02T23:00:00Z,53.516566,8.188217,None,0.03|2010-12-02T23:10:00Z,53.516566,8.188217,None,0.13|2010-12-02T23:20:00Z,53.516566,8.188217,None,0.244|2010-12-02T23:30:00Z,53.516566,8.188217,None,0.477|2010-12-02T23:40:00Z,53.516566,8.188217,None,0.624|2010-12-02T23:50:00Z,53.516566,8.188217,None,0.656|2010-12-03T00:00:00Z,53.516566,8.188217,None,0.643|2010-12-03T00:10:00Z,53.516566,8.188217,None,0.626|2010-12-03T00:20:00Z,53.516566,8.188217,None,0.625|2010-12-03T00:30:00Z,53.516566,8.188217,None,0.635|2010-12-03T00:40:00Z,53.516566,8.188217,None,0.64|2010-12-03T00:50:00Z,53.516566,8.188217,None,0.601|2010-12-03T01:00:00Z,53.516566,8.188217,None,0.566|2010-12-03T01:10:00Z,53.516566,8.188217,None,0.586|2010-12-03T01:20:00Z,53.516566,8.188217,None,0.55|2010-12-03T01:30:00Z,53.516566,8.188217,None,0.553|2010-12-03T01:40:00Z,53.516566,8.188217,None,0.54|2010-12-03T01:50:00Z,53.516566,8.188217,None,0.212|2010-12-03T02:00:00Z,53.516566,8.188217,None,-0.106|2010-12-03T02:10:00Z,53.516566,8.188217,None,-0.159|2010-12-03T02:20:00Z,53.516566,8.188217,None,-0.126|2010-12-03T02:30:00Z,53.516566,8.188217,None,-0.11|2010-12-03T02:40:00Z,53.516566,8.188217,None,-0.096|2010-12-03T02:50:00Z,53.516566,8.188217,None,-0.013|2010-12-03T03:00:00Z,53.516566,8.188217,None,0.027|2010-12-03T03:10:00Z,53.516566,8.188217,None,0.014|2010-12-03T03:20:00Z,53.516566,8.188217,None,0.008|2010-12-03T03:30:00Z,53.516566,8.188217,None,-0.03|2010-12-03T03:40:00Z,53.516566,8.188217,None,-0.079|2010-12-03T03:50:00Z,53.516566,8.188217,None,-0.106|2010-12-03T04:00:00Z,53.516566,8.188217,None,-0.41|2010-12-03T04:10:00Z,53.516566,8.188217,None,-0.479|2010-12-03T04:20:00Z,53.516566,8.188217,None,0.026|2010-12-03T04:30:00Z,53.516566,8.188217,None,0.23|2010-12-03T04:40:00Z,53.516566,8.188217,None,0.173|2010-12-03T04:50:00Z,53.516566,8.188217,None,0.069|2010-12-03T05:00:00Z,53.516566,8.188217,None,0.042|2010-12-03T05:10:00Z,53.516566,8.188217,None,0.091|2010-12-03T05:20:00Z,53.516566,8.188217,None,-0.037|2010-12-03T05:30:00Z,53.516566,8.188217,None,0.032|2010-12-03T05:40:00Z,53.516566,8.188217,None,0.342|2010-12-03T05:50:00Z,53.516566,8.188217,None,0.481|2010-12-03T06:00:00Z,53.516566,8.188217,None,0.663|2010-12-03T06:10:00Z,53.516566,8.188217,None,0.744|2010-12-03T06:20:00Z,53.516566,8.188217,None,0.865|2010-12-03T06:30:00Z,53.516566,8.188217,None,0.891|2010-12-03T06:40:00Z,53.516566,8.188217,None,0.82|2010-12-03T06:50:00Z,53.516566,8.188217,None,0.857|2010-12-03T07:00:00Z,53.516566,8.188217,None,0.991|2010-12-03T07:10:00Z,53.516566,8.188217,None,1.07|2010-12-03T07:20:00Z,53.516566,8.188217,None,1.173|2010-12-03T07:30:00Z,53.516566,8.188217,None,1.255|2010-12-03T07:40:00Z,53.516566,8.188217,None,1.294|2010-12-03T07:50:00Z,53.516566,8.188217,None,1.31|2010-12-03T08:00:00Z,53.516566,8.188217,None,1.337|2010-12-03T08:10:00Z,53.516566,8.188217,None,1.385|2010-12-03T08:20:00Z,53.516566,8.188217,None,1.418|2010-12-03T08:30:00Z,53.516566,8.188217,None,1.451|2010-12-03T08:40:00Z,53.516566,8.188217,None,1.527|2010-12-03T08:50:00Z,53.516566,8.188217,None,1.622|2010-12-03T09:00:00Z,53.516566,8.188217,None,1.718|2010-12-03T09:10:00Z,53.516566,8.188217,None,1.757|2010-12-03T09:20:00Z,53.516566,8.188217,None,1.755|2010-12-03T09:30:00Z,53.516566,8.188217,None,1.75|2010-12-03T09:40:00Z,53.516566,8.188217,None,1.722|2010-12-03T09:50:00Z,53.516566,8.188217,None,1.633|2010-12-03T10:00:00Z,53.516566,8.188217,None,1.43|2010-12-03T10:10:00Z,53.516566,8.188217,None,1.301|2010-12-03T10:20:00Z,53.516566,8.188217,None,1.197|2010-12-03T10:30:00Z,53.516566,8.188217,None,0.991|2010-12-03T10:40:00Z,53.516566,8.188217,None,0.784|2010-12-03T10:50:00Z,53.516566,8.188217,None,0.445|2010-12-03T11:00:00Z,53.516566,8.188217,None,-0.014|2010-12-03T11:10:00Z,53.516566,8.188217,None,-0.065|2010-12-03T11:20:00Z,53.516566,8.188217,None,-0.149|2010-12-03T11:30:00Z,53.516566,8.188217,None,-0.239|2010-12-03T11:40:00Z,53.516566,8.188217,None,-0.051|2010-12-03T11:50:00Z,53.516566,8.188217,None,0.227|2010-12-03T12:00:00Z,53.516566,8.188217,None,0.519|2010-12-03T12:10:00Z,53.516566,8.188217,None,0.676|2010-12-03T12:20:00Z,53.516566,8.188217,None,0.766|2010-12-03T12:30:00Z,53.516566,8.188217,None,0.833|2010-12-03T12:40:00Z,53.516566,8.188217,None,0.855|2010-12-03T12:50:00Z,53.516566,8.188217,None,0.84|2010-12-03T13:00:00Z,53.516566,8.188217,None,0.821|2010-12-03T13:10:00Z,53.516566,8.188217,None,0.81|2010-12-03T13:20:00Z,53.516566,8.188217,None,0.797|2010-12-03T13:30:00Z,53.516566,8.188217,None,0.799|2010-12-03T13:40:00Z,53.516566,8.188217,None,0.795|2010-12-03T13:50:00Z,53.516566,8.188217,None,0.793|2010-12-03T14:00:00Z,53.516566,8.188217,None,0.799|2010-12-03T14:10:00Z,53.516566,8.188217,None,0.798|2010-12-03T14:20:00Z,53.516566,8.188217,None,0.781|2010-12-03T14:30:00Z,53.516566,8.188217,None,0.724|2010-12-03T14:40:00Z,53.516566,8.188217,None,0.703|2010-12-03T14:50:00Z,53.516566,8.188217,None,0.619|2010-12-03T15:00:00Z,53.516566,8.188217,None,0.493|2010-12-03T15:10:00Z,53.516566,8.188217,None,0.412|2010-12-03T15:20:00Z,53.516566,8.188217,None,0.224|2010-12-03T15:30:00Z,53.516566,8.188217,None,0.122|2010-12-03T15:40:00Z,53.516566,8.188217,None,0.158|2010-12-03T15:50:00Z,53.516566,8.188217,None,0.252|2010-12-03T16:00:00Z,53.516566,8.188217,None,0.345|2010-12-03T16:10:00Z,53.516566,8.188217,None,0.348|2010-12-03T16:20:00Z,53.516566,8.188217,None,0.221|2010-12-03T16:30:00Z,53.516566,8.188217,None,0.076|2010-12-03T16:40:00Z,53.516566,8.188217,None,-0.206|2010-12-03T16:50:00Z,53.516566,8.188217,None,-0.59|2010-12-03T17:00:00Z,53.516566,8.188217,None,-0.497|2010-12-03T17:10:00Z,53.516566,8.188217,None,-0.434|2010-12-03T17:20:00Z,53.516566,8.188217,None,-0.395|2010-12-03T17:30:00Z,53.516566,8.188217,None,-0.339|2010-12-03T17:40:00Z,53.516566,8.188217,None,-0.39|2010-12-03T17:50:00Z,53.516566,8.188217,None,-0.29|2010-12-03T18:00:00Z,53.516566,8.188217,None,-0.127|2010-12-03T18:10:00Z,53.516566,8.188217,None,-0.018|2010-12-03T18:20:00Z,53.516566,8.188217,None,0.086|2010-12-03T18:30:00Z,53.516566,8.188217,None,0.377|2010-12-03T18:40:00Z,53.516566,8.188217,None,0.453|2010-12-03T18:50:00Z,53.516566,8.188217,None,0.417|2010-12-03T19:00:00Z,53.516566,8.188217,None,0.487|2010-12-03T19:10:00Z,53.516566,8.188217,None,0.68|2010-12-03T19:20:00Z,53.516566,8.188217,None,0.757|2010-12-03T19:30:00Z,53.516566,8.188217,None,0.808|2010-12-03T19:40:00Z,53.516566,8.188217,None,0.78|2010-12-03T19:50:00Z,53.516566,8.188217,None,0.805|2010-12-03T20:00:00Z,53.516566,8.188217,None,0.842|2010-12-03T20:10:00Z,53.516566,8.188217,None,0.861|2010-12-03T20:20:00Z,53.516566,8.188217,None,0.882|2010-12-03T20:30:00Z,53.516566,8.188217,None,0.954|2010-12-03T20:40:00Z,53.516566,8.188217,None,1.101|2010-12-03T20:50:00Z,53.516566,8.188217,None,1.154|2010-12-03T21:00:00Z,53.516566,8.188217,None,1.162|2010-12-03T21:10:00Z,53.516566,8.188217,None,1.157|2010-12-03T21:20:00Z,53.516566,8.188217,None,1.151|2010-12-03T21:30:00Z,53.516566,8.188217,None,1.155|2010-12-03T21:40:00Z,53.516566,8.188217,None,1.158|2010-12-03T21:50:00Z,53.516566,8.188217,None,1.161|2010-12-03T22:00:00Z,53.516566,8.188217,None,1.165|2010-12-03T22:10:00Z,53.516566,8.188217,None,1.14|2010-12-03T22:20:00Z,53.516566,8.188217,None,1.095|2010-12-03T22:30:00Z,53.516566,8.188217,None,1.057|2010-12-03T22:40:00Z,53.516566,8.188217,None,0.947|2010-12-03T22:50:00Z,53.516566,8.188217,None,0.708|2010-12-03T23:00:00Z,53.516566,8.188217,None,0.602|2010-12-03T23:10:00Z,53.516566,8.188217,None,0.47|2010-12-03T23:20:00Z,53.516566,8.188217,None,0.08|2010-12-03T23:30:00Z,53.516566,8.188217,None,-0.519|2010-12-03T23:40:00Z,53.516566,8.188217,None,-0.791|2010-12-03T23:50:00Z,53.516566,8.188217,None,-0.582|2010-12-04T00:00:00Z,53.516566,8.188217,None,-0.695|2010-12-04T00:10:00Z,53.516566,8.188217,None,-0.495|2010-12-04T00:20:00Z,53.516566,8.188217,None,-0.074|2010-12-04T00:30:00Z,53.516566,8.188217,None,0.361|2010-12-04T00:40:00Z,53.516566,8.188217,None,0.535|2010-12-04T00:50:00Z,53.516566,8.188217,None,0.558|2010-12-04T01:00:00Z,53.516566,8.188217,None,0.535|2010-12-04T01:10:00Z,53.516566,8.188217,None,0.563|2010-12-04T01:20:00Z,53.516566,8.188217,None,0.568|2010-12-04T01:30:00Z,53.516566,8.188217,None,0.552|2010-12-04T01:40:00Z,53.516566,8.188217,None,0.514|2010-12-04T01:50:00Z,53.516566,8.188217,None,0.469|2010-12-04T02:00:00Z,53.516566,8.188217,None,0.446|2010-12-04T02:10:00Z,53.516566,8.188217,None,0.416|2010-12-04T02:20:00Z,53.516566,8.188217,None,0.404|2010-12-04T02:30:00Z,53.516566,8.188217,None,0.4|2010-12-04T02:40:00Z,53.516566,8.188217,None,0.424|2010-12-04T02:50:00Z,53.516566,8.188217,None,0.451|2010-12-04T03:00:00Z,53.516566,8.188217,None,0.445|2010-12-04T03:10:00Z,53.516566,8.188217,None,0.411|2010-12-04T03:20:00Z,53.516566,8.188217,None,0.345|2010-12-04T03:30:00Z,53.516566,8.188217,None,0.348|2010-12-04T03:40:00Z,53.516566,8.188217,None,0.354|2010-12-04T03:50:00Z,53.516566,8.188217,None,0.288|2010-12-04T04:00:00Z,53.516566,8.188217,None,0.057|2010-12-04T04:10:00Z,53.516566,8.188217,None,0.004|2010-12-04T04:20:00Z,53.516566,8.188217,None,-0.003|2010-12-04T04:30:00Z,53.516566,8.188217,None,-0.032|2010-12-04T04:40:00Z,53.516566,8.188217,None,-0.078|2010-12-04T04:50:00Z,53.516566,8.188217,None,-0.093|2010-12-04T05:00:00Z,53.516566,8.188217,None,-0.297|2010-12-04T05:10:00Z,53.516566,8.188217,None,-0.632|2010-12-04T05:20:00Z,53.516566,8.188217,None,-0.706|2010-12-04T05:30:00Z,53.516566,8.188217,None,-0.639|2010-12-04T05:40:00Z,53.516566,8.188217,None,-0.65|2010-12-04T05:50:00Z,53.516566,8.188217,None,-0.493|2010-12-04T06:00:00Z,53.516566,8.188217,None,-0.378|2010-12-04T06:10:00Z,53.516566,8.188217,None,-0.308|2010-12-04T06:20:00Z,53.516566,8.188217,None,-0.376|2010-12-04T06:30:00Z,53.516566,8.188217,None,-0.346|2010-12-04T06:40:00Z,53.516566,8.188217,None,-0.407|2010-12-04T06:50:00Z,53.516566,8.188217,None,-0.646|2010-12-04T07:00:00Z,53.516566,8.188217,None,-0.408|2010-12-04T07:10:00Z,53.516566,8.188217,None,0.091|2010-12-04T07:20:00Z,53.516566,8.188217,None,0.199|2010-12-04T07:30:00Z,53.516566,8.188217,None,0.164|2010-12-04T07:40:00Z,53.516566,8.188217,None,0.153|2010-12-04T07:50:00Z,53.516566,8.188217,None,0.098|2010-12-04T08:00:00Z,53.516566,8.188217,None,0.181|2010-12-04T08:10:00Z,53.516566,8.188217,None,0.256|2010-12-04T08:20:00Z,53.516566,8.188217,None,0.459|2010-12-04T08:30:00Z,53.516566,8.188217,None,0.546|2010-12-04T08:40:00Z,53.516566,8.188217,None,0.66|2010-12-04T08:50:00Z,53.516566,8.188217,None,0.794|2010-12-04T09:00:00Z,53.516566,8.188217,None,0.859|2010-12-04T09:10:00Z,53.516566,8.188217,None,0.895|2010-12-04T09:20:00Z,53.516566,8.188217,None,0.91|2010-12-04T09:30:00Z,53.516566,8.188217,None,0.92|2010-12-04T09:40:00Z,53.516566,8.188217,None,0.941|2010-12-04T09:50:00Z,53.516566,8.188217,None,0.966|2010-12-04T10:00:00Z,53.516566,8.188217,None,0.995|2010-12-04T10:10:00Z,53.516566,8.188217,None,1.008|2010-12-04T10:20:00Z,53.516566,8.188217,None,1.007|2010-12-04T10:30:00Z,53.516566,8.188217,None,0.964|2010-12-04T10:40:00Z,53.516566,8.188217,None,0.886|2010-12-04T10:50:00Z,53.516566,8.188217,None,0.83|2010-12-04T11:00:00Z,53.516566,8.188217,None,0.656|2010-12-04T11:10:00Z,53.516566,8.188217,None,0.317|2010-12-04T11:20:00Z,53.516566,8.188217,None,0.059|2010-12-04T11:30:00Z,53.516566,8.188217,None,-0.155|2010-12-04T11:40:00Z,53.516566,8.188217,None,-0.53|2010-12-04T11:50:00Z,53.516566,8.188217,None,-0.777|2010-12-04T12:00:00Z,53.516566,8.188217,None,-0.854|2010-12-04T12:10:00Z,53.516566,8.188217,None,-0.861|2010-12-04T12:20:00Z,53.516566,8.188217,None,-0.872|2010-12-04T12:30:00Z,53.516566,8.188217,None,-0.666|2010-12-04T12:40:00Z,53.516566,8.188217,None,-0.415|2010-12-04T12:50:00Z,53.516566,8.188217,None,-0.33|2010-12-04T13:00:00Z,53.516566,8.188217,None,-0.11|2010-12-04T13:10:00Z,53.516566,8.188217,None,0.041|2010-12-04T13:20:00Z,53.516566,8.188217,None,0.078|2010-12-04T13:30:00Z,53.516566,8.188217,None,0.167|2010-12-04T13:40:00Z,53.516566,8.188217,None,0.275|2010-12-04T13:50:00Z,53.516566,8.188217,None,0.366|2010-12-04T14:00:00Z,53.516566,8.188217,None,0.404|2010-12-04T14:10:00Z,53.516566,8.188217,None,0.401|2010-12-04T14:20:00Z,53.516566,8.188217,None,0.376|2010-12-04T14:30:00Z,53.516566,8.188217,None,0.337|2010-12-04T14:40:00Z,53.516566,8.188217,None,0.315|2010-12-04T14:50:00Z,53.516566,8.188217,None,0.287|2010-12-04T15:00:00Z,53.516566,8.188217,None,0.265|2010-12-04T15:10:00Z,53.516566,8.188217,None,0.238|2010-12-04T15:20:00Z,53.516566,8.188217,None,0.216|2010-12-04T15:30:00Z,53.516566,8.188217,None,0.193|2010-12-04T15:40:00Z,53.516566,8.188217,None,0.205|2010-12-04T15:50:00Z,53.516566,8.188217,None,0.285|2010-12-04T16:00:00Z,53.516566,8.188217,None,0.242|2010-12-04T16:10:00Z,53.516566,8.188217,None,0.26|2010-12-04T16:20:00Z,53.516566,8.188217,None,0.281|2010-12-04T16:30:00Z,53.516566,8.188217,None,0.257|2010-12-04T16:40:00Z,53.516566,8.188217,None,0.156|2010-12-04T16:50:00Z,53.516566,8.188217,None,0.092|2010-12-04T17:00:00Z,53.516566,8.188217,None,0.107|2010-12-04T17:10:00Z,53.516566,8.188217,None,-0.008|2010-12-04T17:20:00Z,53.516566,8.188217,None,-0.117|2010-12-04T17:30:00Z,53.516566,8.188217,None,-0.068|2010-12-04T17:40:00Z,53.516566,8.188217,None,-0.042|2010-12-04T17:50:00Z,53.516566,8.188217,None,-0.039|2010-12-04T18:00:00Z,53.516566,8.188217,None,-0.158|2010-12-04T18:10:00Z,53.516566,8.188217,None,-0.463|2010-12-04T18:20:00Z,53.516566,8.188217,None,-0.399|2010-12-04T18:30:00Z,53.516566,8.188217,None,-0.074|2010-12-04T18:40:00Z,53.516566,8.188217,None,-0.27|2010-12-04T18:50:00Z,53.516566,8.188217,None,-0.383|2010-12-04T19:00:00Z,53.516566,8.188217,None,-0.301|2010-12-04T19:10:00Z,53.516566,8.188217,None,-0.225|2010-12-04T19:20:00Z,53.516566,8.188217,None,-0.12|2010-12-04T19:30:00Z,53.516566,8.188217,None,-0.151|2010-12-04T19:40:00Z,53.516566,8.188217,None,-0.294|2010-12-04T19:50:00Z,53.516566,8.188217,None,-0.446|2010-12-04T20:00:00Z,53.516566,8.188217,None,-0.512|2010-12-04T20:10:00Z,53.516566,8.188217,None,-0.47|2010-12-04T20:20:00Z,53.516566,8.188217,None,-0.313|2010-12-04T20:30:00Z,53.516566,8.188217,None,-0.185|2010-12-04T20:40:00Z,53.516566,8.188217,None,-0.032|2010-12-04T20:50:00Z,53.516566,8.188217,None,0.009|2010-12-04T21:00:00Z,53.516566,8.188217,None,0.042|2010-12-04T21:10:00Z,53.516566,8.188217,None,0.116|2010-12-04T21:20:00Z,53.516566,8.188217,None,0.152|2010-12-04T21:30:00Z,53.516566,8.188217,None,0.213|2010-12-04T21:40:00Z,53.516566,8.188217,None,0.29|2010-12-04T21:50:00Z,53.516566,8.188217,None,0.329|2010-12-04T22:00:00Z,53.516566,8.188217,None,0.378|2010-12-04T22:10:00Z,53.516566,8.188217,None,0.396|2010-12-04T22:20:00Z,53.516566,8.188217,None,0.393|2010-12-04T22:30:00Z,53.516566,8.188217,None,0.386|2010-12-04T22:40:00Z,53.516566,8.188217,None,0.383|2010-12-04T22:50:00Z,53.516566,8.188217,None,0.374|2010-12-04T23:00:00Z,53.516566,8.188217,None,0.361|2010-12-04T23:10:00Z,53.516566,8.188217,None,0.334|2010-12-04T23:20:00Z,53.516566,8.188217,None,0.228|2010-12-04T23:30:00Z,53.516566,8.188217,None,0.056|2010-12-04T23:40:00Z,53.516566,8.188217,None,-0.096|2010-12-04T23:50:00Z,53.516566,8.188217,None,-0.321|2010-12-05T00:00:00Z,53.516566,8.188217,None,-0.336|2010-12-05T00:10:00Z,53.516566,8.188217,None,-0.45|2010-12-05T00:20:00Z,53.516566,8.188217,None,-0.129|2010-12-05T00:30:00Z,53.516566,8.188217,None,-0.042|2010-12-05T00:40:00Z,53.516566,8.188217,None,-0.092|2010-12-05T00:50:00Z,53.516566,8.188217,None,-0.16|2010-12-05T01:00:00Z,53.516566,8.188217,None,-0.24|2010-12-05T01:10:00Z,53.516566,8.188217,None,-0.294|2010-12-05T01:20:00Z,53.516566,8.188217,None,-0.357|2010-12-05T01:30:00Z,53.516566,8.188217,None,-0.352|2010-12-05T01:40:00Z,53.516566,8.188217,None,-0.283|2010-12-05T01:50:00Z,53.516566,8.188217,None,-0.173|2010-12-05T02:00:00Z,53.516566,8.188217,None,-0.072|2010-12-05T02:10:00Z,53.516566,8.188217,None,-0.018|2010-12-05T02:20:00Z,53.516566,8.188217,None,-0.015|2010-12-05T02:30:00Z,53.516566,8.188217,None,-0.029|2010-12-05T02:40:00Z,53.516566,8.188217,None,-0.038|2010-12-05T02:50:00Z,53.516566,8.188217,None,-0.047|2010-12-05T03:00:00Z,53.516566,8.188217,None,-0.061|2010-12-05T03:10:00Z,53.516566,8.188217,None,-0.102|2010-12-05T03:20:00Z,53.516566,8.188217,None,-0.098|2010-12-05T03:30:00Z,53.516566,8.188217,None,-0.067|2010-12-05T03:40:00Z,53.516566,8.188217,None,-0.111|2010-12-05T03:50:00Z,53.516566,8.188217,None,-0.108|2010-12-05T04:00:00Z,53.516566,8.188217,None,-0.121|2010-12-05T04:10:00Z,53.516566,8.188217,None,-0.115|2010-12-05T04:20:00Z,53.516566,8.188217,None,-0.17|2010-12-05T04:30:00Z,53.516566,8.188217,None,-0.223|2010-12-05T04:40:00Z,53.516566,8.188217,None,-0.198|2010-12-05T04:50:00Z,53.516566,8.188217,None,-0.327|2010-12-05T05:00:00Z,53.516566,8.188217,None,-0.434|2010-12-05T05:10:00Z,53.516566,8.188217,None,-0.426|2010-12-05T05:20:00Z,53.516566,8.188217,None,-0.411|2010-12-05T05:30:00Z,53.516566,8.188217,None,-0.396|2010-12-05T05:40:00Z,53.516566,8.188217,None,-0.378|2010-12-05T05:50:00Z,53.516566,8.188217,None,-0.389|2010-12-05T06:00:00Z,53.516566,8.188217,None,-0.411|2010-12-05T06:10:00Z,53.516566,8.188217,None,-0.342|2010-12-05T06:20:00Z,53.516566,8.188217,None,-0.205|2010-12-05T06:30:00Z,53.516566,8.188217,None,-0.187|2010-12-05T06:40:00Z,53.516566,8.188217,None,-0.066|2010-12-05T06:50:00Z,53.516566,8.188217,None,-0.05|2010-12-05T07:00:00Z,53.516566,8.188217,None,-0.352|2010-12-05T07:10:00Z,53.516566,8.188217,None,-0.438|2010-12-05T07:20:00Z,53.516566,8.188217,None,-0.264|2010-12-05T07:30:00Z,53.516566,8.188217,None,-0.248|2010-12-05T07:40:00Z,53.516566,8.188217,None,-0.181|2010-12-05T07:50:00Z,53.516566,8.188217,None,-0.203|2010-12-05T08:00:00Z,53.516566,8.188217,None,-0.38|2010-12-05T08:10:00Z,53.516566,8.188217,None,-0.51|2010-12-05T08:20:00Z,53.516566,8.188217,None,-0.36|2010-12-05T08:30:00Z,53.516566,8.188217,None,-0.071|2010-12-05T08:40:00Z,53.516566,8.188217,None,0.025|2010-12-05T08:50:00Z,53.516566,8.188217,None,0.124|2010-12-05T09:00:00Z,53.516566,8.188217,None,0.249|2010-12-05T09:10:00Z,53.516566,8.188217,None,0.226|2010-12-05T09:20:00Z,53.516566,8.188217,None,0.203|2010-12-05T09:30:00Z,53.516566,8.188217,None,0.272|2010-12-05T09:40:00Z,53.516566,8.188217,None,0.384|2010-12-05T09:50:00Z,53.516566,8.188217,None,0.485|2010-12-05T10:00:00Z,53.516566,8.188217,None,0.612|2010-12-05T10:10:00Z,53.516566,8.188217,None,0.729|2010-12-05T10:20:00Z,53.516566,8.188217,None,0.93|2010-12-05T10:30:00Z,53.516566,8.188217,None,1.113|2010-12-05T10:40:00Z,53.516566,8.188217,None,1.146|2010-12-05T10:50:00Z,53.516566,8.188217,None,1.185|2010-12-05T11:00:00Z,53.516566,8.188217,None,1.22|2010-12-05T11:10:00Z,53.516566,8.188217,None,1.232|2010-12-05T11:20:00Z,53.516566,8.188217,None,1.238|2010-12-05T11:30:00Z,53.516566,8.188217,None,1.243|2010-12-05T11:40:00Z,53.516566,8.188217,None,1.198|2010-12-05T11:50:00Z,53.516566,8.188217,None,1.107|2010-12-05T12:00:00Z,53.516566,8.188217,None,0.963|2010-12-05T12:10:00Z,53.516566,8.188217,None,0.629|2010-12-05T12:20:00Z,53.516566,8.188217,None,0.386|2010-12-05T12:30:00Z,53.516566,8.188217,None,0.222|2010-12-05T12:40:00Z,53.516566,8.188217,None,-0.063|2010-12-05T12:50:00Z,53.516566,8.188217,None,-0.213|2010-12-05T13:00:00Z,53.516566,8.188217,None,-0.14|2010-12-05T13:10:00Z,53.516566,8.188217,None,-0.141|2010-12-05T13:20:00Z,53.516566,8.188217,None,-0.174|2010-12-05T13:30:00Z,53.516566,8.188217,None,-0.229|2010-12-05T13:40:00Z,53.516566,8.188217,None,-0.209|2010-12-05T13:50:00Z,53.516566,8.188217,None,-0.153|2010-12-05T14:00:00Z,53.516566,8.188217,None,-0.114|2010-12-05T14:10:00Z,53.516566,8.188217,None,-0.179|2010-12-05T14:20:00Z,53.516566,8.188217,None,-0.165|2010-12-05T14:30:00Z,53.516566,8.188217,None,-0.115|2010-12-05T14:40:00Z,53.516566,8.188217,None,-0.107|2010-12-05T14:50:00Z,53.516566,8.188217,None,-0.123|2010-12-05T15:00:00Z,53.516566,8.188217,None,-0.145|2010-12-05T15:10:00Z,53.516566,8.188217,None,-0.165|2010-12-05T15:20:00Z,53.516566,8.188217,None,-0.152|2010-12-05T15:30:00Z,53.516566,8.188217,None,-0.101|2010-12-05T15:40:00Z,53.516566,8.188217,None,-0.09|2010-12-05T15:50:00Z,53.516566,8.188217,None,-0.079|2010-12-05T16:00:00Z,53.516566,8.188217,None,-0.045|2010-12-05T16:10:00Z,53.516566,8.188217,None,-0.107|2010-12-05T16:20:00Z,53.516566,8.188217,None,-0.151|2010-12-05T16:30:00Z,53.516566,8.188217,None,-0.143|2010-12-05T16:40:00Z,53.516566,8.188217,None,-0.147|2010-12-05T16:50:00Z,53.516566,8.188217,None,-0.14|2010-12-05T17:00:00Z,53.516566,8.188217,None,-0.07|2010-12-05T17:10:00Z,53.516566,8.188217,None,0.028|2010-12-05T17:20:00Z,53.516566,8.188217,None,-0.022|2010-12-05T17:30:00Z,53.516566,8.188217,None,-0.084|2010-12-05T17:40:00Z,53.516566,8.188217,None,-0.164|2010-12-05T17:50:00Z,53.516566,8.188217,None,-0.151|2010-12-05T18:00:00Z,53.516566,8.188217,None,-0.125|2010-12-05T18:10:00Z,53.516566,8.188217,None,-0.146|2010-12-05T18:20:00Z,53.516566,8.188217,None,-0.151|2010-12-05T18:30:00Z,53.516566,8.188217,None,-0.168|2010-12-05T18:40:00Z,53.516566,8.188217,None,-0.209|2010-12-05T18:50:00Z,53.516566,8.188217,None,-0.252|2010-12-05T19:00:00Z,53.516566,8.188217,None,-0.14|2010-12-05T19:10:00Z,53.516566,8.188217,None,-0.003|2010-12-05T19:20:00Z,53.516566,8.188217,None,-0.131|2010-12-05T19:30:00Z,53.516566,8.188217,None,-0.24|2010-12-05T19:40:00Z,53.516566,8.188217,None,-0.21|2010-12-05T19:50:00Z,53.516566,8.188217,None,-0.074|2010-12-05T20:00:00Z,53.516566,8.188217,None,-0.103|2010-12-05T20:10:00Z,53.516566,8.188217,None,-0.214|2010-12-05T20:20:00Z,53.516566,8.188217,None,-0.22|2010-12-05T20:30:00Z,53.516566,8.188217,None,-0.265|2010-12-05T20:40:00Z,53.516566,8.188217,None,-0.3|2010-12-05T20:50:00Z,53.516566,8.188217,None,-0.134|2010-12-05T21:00:00Z,53.516566,8.188217,None,0.223|2010-12-05T21:10:00Z,53.516566,8.188217,None,0.438|2010-12-05T21:20:00Z,53.516566,8.188217,None,0.624|2010-12-05T21:30:00Z,53.516566,8.188217,None,0.77|2010-12-05T21:40:00Z,53.516566,8.188217,None,0.812|2010-12-05T21:50:00Z,53.516566,8.188217,None,0.706|2010-12-05T22:00:00Z,53.516566,8.188217,None,0.72|2010-12-05T22:10:00Z,53.516566,8.188217,None,0.724|2010-12-05T22:20:00Z,53.516566,8.188217,None,0.739|2010-12-05T22:30:00Z,53.516566,8.188217,None,0.793|2010-12-05T22:40:00Z,53.516566,8.188217,None,0.9|2010-12-05T22:50:00Z,53.516566,8.188217,None,1.28|2010-12-05T23:00:00Z,53.516566,8.188217,None,1.569|2010-12-05T23:10:00Z,53.516566,8.188217,None,1.671|2010-12-05T23:20:00Z,53.516566,8.188217,None,1.726|2010-12-05T23:30:00Z,53.516566,8.188217,None,1.717|2010-12-05T23:40:00Z,53.516566,8.188217,None,1.685|2010-12-05T23:50:00Z,53.516566,8.188217,None,1.659|2010-12-06T00:00:00Z,53.516566,8.188217,None,1.591|2010-12-06T00:10:00Z,53.516566,8.188217,None,1.593|2010-12-06T00:20:00Z,53.516566,8.188217,None,1.429|2010-12-06T00:30:00Z,53.516566,8.188217,None,0.848|2010-12-06T00:40:00Z,53.516566,8.188217,None,0.652|2010-12-06T00:50:00Z,53.516566,8.188217,None,0.478|2010-12-06T01:00:00Z,53.516566,8.188217,None,-0.058|2010-12-06T01:10:00Z,53.516566,8.188217,None,-0.224|2010-12-06T01:20:00Z,53.516566,8.188217,None,-0.211|2010-12-06T01:30:00Z,53.516566,8.188217,None,-0.235|2010-12-06T01:40:00Z,53.516566,8.188217,None,-0.271|2010-12-06T01:50:00Z,53.516566,8.188217,None,-0.293|2010-12-06T02:00:00Z,53.516566,8.188217,None,-0.268|2010-12-06T02:10:00Z,53.516566,8.188217,None,-0.203|2010-12-06T02:20:00Z,53.516566,8.188217,None,-0.15|2010-12-06T02:30:00Z,53.516566,8.188217,None,-0.146|2010-12-06T02:40:00Z,53.516566,8.188217,None,-0.171|2010-12-06T02:50:00Z,53.516566,8.188217,None,-0.174|2010-12-06T03:00:00Z,53.516566,8.188217,None,-0.161|2010-12-06T03:10:00Z,53.516566,8.188217,None,-0.169|2010-12-06T03:20:00Z,53.516566,8.188217,None,-0.201|2010-12-06T03:30:00Z,53.516566,8.188217,None,-0.212|2010-12-06T03:40:00Z,53.516566,8.188217,None,-0.209|2010-12-06T03:50:00Z,53.516566,8.188217,None,-0.205|2010-12-06T04:00:00Z,53.516566,8.188217,None,-0.203|2010-12-06T04:10:00Z,53.516566,8.188217,None,-0.177|2010-12-06T04:20:00Z,53.516566,8.188217,None,-0.214|2010-12-06T04:30:00Z,53.516566,8.188217,None,-0.213|2010-12-06T04:40:00Z,53.516566,8.188217,None,-0.211|2010-12-06T04:50:00Z,53.516566,8.188217,None,-0.198|2010-12-06T05:00:00Z,53.516566,8.188217,None,-0.075|2010-12-06T05:10:00Z,53.516566,8.188217,None,0.101|2010-12-06T05:20:00Z,53.516566,8.188217,None,0.064|2010-12-06T05:30:00Z,53.516566,8.188217,None,-0.072|2010-12-06T05:40:00Z,53.516566,8.188217,None,-0.198|2010-12-06T05:50:00Z,53.516566,8.188217,None,-0.194|2010-12-06T06:00:00Z,53.516566,8.188217,None,-0.171|2010-12-06T06:10:00Z,53.516566,8.188217,None,-0.137|2010-12-06T06:20:00Z,53.516566,8.188217,None,-0.054|2010-12-06T06:30:00Z,53.516566,8.188217,None,-0.056|2010-12-06T06:40:00Z,53.516566,8.188217,None,-0.122|2010-12-06T06:50:00Z,53.516566,8.188217,None,-0.15|2010-12-06T07:00:00Z,53.516566,8.188217,None,-0.161|2010-12-06T07:10:00Z,53.516566,8.188217,None,-0.257|2010-12-06T07:20:00Z,53.516566,8.188217,None,-0.282|2010-12-06T07:30:00Z,53.516566,8.188217,None,-0.195|2010-12-06T07:40:00Z,53.516566,8.188217,None,-0.087|2010-12-06T07:50:00Z,53.516566,8.188217,None,-0.22|2010-12-06T08:00:00Z,53.516566,8.188217,None,-0.249|2010-12-06T08:10:00Z,53.516566,8.188217,None,-0.217|2010-12-06T08:20:00Z,53.516566,8.188217,None,-0.179|2010-12-06T08:30:00Z,53.516566,8.188217,None,-0.161|2010-12-06T08:40:00Z,53.516566,8.188217,None,-0.114|2010-12-06T08:50:00Z,53.516566,8.188217,None,-0.088|2010-12-06T09:00:00Z,53.516566,8.188217,None,-0.006|2010-12-06T09:10:00Z,53.516566,8.188217,None,-0.013|2010-12-06T09:20:00Z,53.516566,8.188217,None,-0.016|2010-12-06T09:30:00Z,53.516566,8.188217,None,0.061|2010-12-06T09:40:00Z,53.516566,8.188217,None,0.178|2010-12-06T09:50:00Z,53.516566,8.188217,None,0.392|2010-12-06T10:00:00Z,53.516566,8.188217,None,0.609|2010-12-06T10:10:00Z,53.516566,8.188217,None,0.76|2010-12-06T10:20:00Z,53.516566,8.188217,None,0.853|2010-12-06T10:30:00Z,53.516566,8.188217,None,0.87|2010-12-06T10:40:00Z,53.516566,8.188217,None,0.761|2010-12-06T10:50:00Z,53.516566,8.188217,None,0.61|2010-12-06T11:00:00Z,53.516566,8.188217,None,0.514|2010-12-06T11:10:00Z,53.516566,8.188217,None,0.588|2010-12-06T11:20:00Z,53.516566,8.188217,None,0.713|2010-12-06T11:30:00Z,53.516566,8.188217,None,0.874|2010-12-06T11:40:00Z,53.516566,8.188217,None,0.949|2010-12-06T11:50:00Z,53.516566,8.188217,None,0.984|2010-12-06T12:00:00Z,53.516566,8.188217,None,1.012|2010-12-06T12:10:00Z,53.516566,8.188217,None,1.036|2010-12-06T12:20:00Z,53.516566,8.188217,None,1.048|2010-12-06T12:30:00Z,53.516566,8.188217,None,1.017|2010-12-06T12:40:00Z,53.516566,8.188217,None,0.935|2010-12-06T12:50:00Z,53.516566,8.188217,None,0.761|2010-12-06T13:00:00Z,53.516566,8.188217,None,0.611|2010-12-06T13:10:00Z,53.516566,8.188217,None,0.681|2010-12-06T13:20:00Z,53.516566,8.188217,None,0.307|2010-12-06T13:30:00Z,53.516566,8.188217,None,0.062|2010-12-06T13:40:00Z,53.516566,8.188217,None,-0.086|2010-12-06T13:50:00Z,53.516566,8.188217,None,-0.08|2010-12-06T14:00:00Z,53.516566,8.188217,None,-0.001|2010-12-06T14:10:00Z,53.516566,8.188217,None,0.06|2010-12-06T14:20:00Z,53.516566,8.188217,None,0.073|2010-12-06T14:30:00Z,53.516566,8.188217,None,0.003|2010-12-06T14:40:00Z,53.516566,8.188217,None,-0.091|2010-12-06T14:50:00Z,53.516566,8.188217,None,-0.1|2010-12-06T15:00:00Z,53.516566,8.188217,None,-0.08|2010-12-06T15:10:00Z,53.516566,8.188217,None,-0.041|2010-12-06T15:20:00Z,53.516566,8.188217,None,0.055|2010-12-06T15:30:00Z,53.516566,8.188217,None,0.169|2010-12-06T15:40:00Z,53.516566,8.188217,None,0.218|2010-12-06T15:50:00Z,53.516566,8.188217,None,0.194|2010-12-06T16:00:00Z,53.516566,8.188217,None,0.134|2010-12-06T16:10:00Z,53.516566,8.188217,None,0.135|2010-12-06T16:20:00Z,53.516566,8.188217,None,0.134|2010-12-06T16:30:00Z,53.516566,8.188217,None,0.18|2010-12-06T16:40:00Z,53.516566,8.188217,None,0.208|2010-12-06T16:50:00Z,53.516566,8.188217,None,0.239|2010-12-06T17:00:00Z,53.516566,8.188217,None,0.38|2010-12-06T17:10:00Z,53.516566,8.188217,None,0.223|2010-12-06T17:20:00Z,53.516566,8.188217,None,0.116|2010-12-06T17:30:00Z,53.516566,8.188217,None,-0.021|2010-12-06T17:40:00Z,53.516566,8.188217,None,-0.058|2010-12-06T17:50:00Z,53.516566,8.188217,None,-0.04|2010-12-06T18:00:00Z,53.516566,8.188217,None,0.258|2010-12-06T18:10:00Z,53.516566,8.188217,None,0.506|2010-12-06T18:20:00Z,53.516566,8.188217,None,0.603|2010-12-06T18:30:00Z,53.516566,8.188217,None,0.651|2010-12-06T18:40:00Z,53.516566,8.188217,None,0.608|2010-12-06T18:50:00Z,53.516566,8.188217,None,0.471|2010-12-06T19:00:00Z,53.516566,8.188217,None,0.287|2010-12-06T19:10:00Z,53.516566,8.188217,None,0.17|2010-12-06T19:20:00Z,53.516566,8.188217,None,0.03|2010-12-06T19:30:00Z,53.516566,8.188217,None,0.079|2010-12-06T19:40:00Z,53.516566,8.188217,None,0.179|2010-12-06T19:50:00Z,53.516566,8.188217,None,0.103|2010-12-06T20:00:00Z,53.516566,8.188217,None,-0.026|2010-12-06T20:10:00Z,53.516566,8.188217,None,-0.065|2010-12-06T20:20:00Z,53.516566,8.188217,None,-0.117|2010-12-06T20:30:00Z,53.516566,8.188217,None,-0.173|2010-12-06T20:40:00Z,53.516566,8.188217,None,-0.069|2010-12-06T20:50:00Z,53.516566,8.188217,None,-0.073|2010-12-06T21:00:00Z,53.516566,8.188217,None,0.01|2010-12-06T21:10:00Z,53.516566,8.188217,None,0.106|2010-12-06T21:20:00Z,53.516566,8.188217,None,0.225|2010-12-06T21:30:00Z,53.516566,8.188217,None,0.246|2010-12-06T21:40:00Z,53.516566,8.188217,None,0.272|2010-12-06T21:50:00Z,53.516566,8.188217,None,0.388|2010-12-06T22:00:00Z,53.516566,8.188217,None,0.584|2010-12-06T22:10:00Z,53.516566,8.188217,None,0.838|2010-12-06T22:20:00Z,53.516566,8.188217,None,0.922|2010-12-06T22:30:00Z,53.516566,8.188217,None,0.865|2010-12-06T22:40:00Z,53.516566,8.188217,None,0.778|2010-12-06T22:50:00Z,53.516566,8.188217,None,0.621|2010-12-06T23:00:00Z,53.516566,8.188217,None,0.562|2010-12-06T23:10:00Z,53.516566,8.188217,None,0.616|2010-12-06T23:20:00Z,53.516566,8.188217,None,0.936|2010-12-06T23:30:00Z,53.516566,8.188217,None,1.449|2010-12-06T23:40:00Z,53.516566,8.188217,None,1.568|2010-12-06T23:50:00Z,53.516566,8.188217,None,1.563|2010-12-07T00:00:00Z,53.516566,8.188217,None,1.716|2010-12-07T00:10:00Z,53.516566,8.188217,None,1.718|2010-12-07T00:20:00Z,53.516566,8.188217,None,1.752|2010-12-07T00:30:00Z,53.516566,8.188217,None,1.789|2010-12-07T00:40:00Z,53.516566,8.188217,None,1.756|2010-12-07T00:50:00Z,53.516566,8.188217,None,1.616|2010-12-07T01:00:00Z,53.516566,8.188217,None,1.458|2010-12-07T01:10:00Z,53.516566,8.188217,None,1.18|2010-12-07T01:20:00Z,53.516566,8.188217,None,0.652|2010-12-07T01:30:00Z,53.516566,8.188217,None,0.62|2010-12-07T01:40:00Z,53.516566,8.188217,None,0.344|2010-12-07T01:50:00Z,53.516566,8.188217,None,0.071|2010-12-07T02:00:00Z,53.516566,8.188217,None,-0.081|2010-12-07T02:10:00Z,53.516566,8.188217,None,-0.019|2010-12-07T02:20:00Z,53.516566,8.188217,None,0.062|2010-12-07T02:30:00Z,53.516566,8.188217,None,0.073|2010-12-07T02:40:00Z,53.516566,8.188217,None,0.106|2010-12-07T02:50:00Z,53.516566,8.188217,None,0.1|2010-12-07T03:00:00Z,53.516566,8.188217,None,0.06|2010-12-07T03:10:00Z,53.516566,8.188217,None,0.011|2010-12-07T03:20:00Z,53.516566,8.188217,None,-0.022|2010-12-07T03:30:00Z,53.516566,8.188217,None,-0.028|2010-12-07T03:40:00Z,53.516566,8.188217,None,0.036|2010-12-07T03:50:00Z,53.516566,8.188217,None,0.112|2010-12-07T04:00:00Z,53.516566,8.188217,None,0.147|2010-12-07T04:10:00Z,53.516566,8.188217,None,0.231|2010-12-07T04:20:00Z,53.516566,8.188217,None,0.291|2010-12-07T04:30:00Z,53.516566,8.188217,None,0.317|2010-12-07T04:40:00Z,53.516566,8.188217,None,0.354|2010-12-07T04:50:00Z,53.516566,8.188217,None,0.32|2010-12-07T05:00:00Z,53.516566,8.188217,None,0.235|2010-12-07T05:10:00Z,53.516566,8.188217,None,0.251|2010-12-07T05:20:00Z,53.516566,8.188217,None,0.434|2010-12-07T05:30:00Z,53.516566,8.188217,None,0.546|2010-12-07T05:40:00Z,53.516566,8.188217,None,0.548|2010-12-07T05:50:00Z,53.516566,8.188217,None,0.285|2010-12-07T06:00:00Z,53.516566,8.188217,None,0.181|2010-12-07T06:10:00Z,53.516566,8.188217,None,0.04|2010-12-07T06:20:00Z,53.516566,8.188217,None,0.043|2010-12-07T06:30:00Z,53.516566,8.188217,None,0.285|2010-12-07T06:40:00Z,53.516566,8.188217,None,0.512|2010-12-07T06:50:00Z,53.516566,8.188217,None,0.618|2010-12-07T07:00:00Z,53.516566,8.188217,None,0.639|2010-12-07T07:10:00Z,53.516566,8.188217,None,0.641|2010-12-07T07:20:00Z,53.516566,8.188217,None,0.605|2010-12-07T07:30:00Z,53.516566,8.188217,None,0.47|2010-12-07T07:40:00Z,53.516566,8.188217,None,0.174|2010-12-07T07:50:00Z,53.516566,8.188217,None,0.04|2010-12-07T08:00:00Z,53.516566,8.188217,None,0.172|2010-12-07T08:10:00Z,53.516566,8.188217,None,0.181|2010-12-07T08:20:00Z,53.516566,8.188217,None,0.145|2010-12-07T08:30:00Z,53.516566,8.188217,None,0.052|2010-12-07T08:40:00Z,53.516566,8.188217,None,0.01|2010-12-07T08:50:00Z,53.516566,8.188217,None,-0.035|2010-12-07T09:00:00Z,53.516566,8.188217,None,0.015|2010-12-07T09:10:00Z,53.516566,8.188217,None,-0.044|2010-12-07T09:20:00Z,53.516566,8.188217,None,0.034|2010-12-07T09:30:00Z,53.516566,8.188217,None,0.154|2010-12-07T09:40:00Z,53.516566,8.188217,None,0.307|2010-12-07T09:50:00Z,53.516566,8.188217,None,0.336|2010-12-07T10:00:00Z,53.516566,8.188217,None,0.33|2010-12-07T10:10:00Z,53.516566,8.188217,None,0.356|2010-12-07T10:20:00Z,53.516566,8.188217,None,0.448|2010-12-07T10:30:00Z,53.516566,8.188217,None,0.566|2010-12-07T10:40:00Z,53.516566,8.188217,None,0.773|2010-12-07T10:50:00Z,53.516566,8.188217,None,0.988|2010-12-07T11:00:00Z,53.516566,8.188217,None,1.1|2010-12-07T11:10:00Z,53.516566,8.188217,None,1.031|2010-12-07T11:20:00Z,53.516566,8.188217,None,0.916|2010-12-07T11:30:00Z,53.516566,8.188217,None,0.819|2010-12-07T11:40:00Z,53.516566,8.188217,None,0.754|2010-12-07T11:50:00Z,53.516566,8.188217,None,0.777|2010-12-07T12:00:00Z,53.516566,8.188217,None,0.874|2010-12-07T12:10:00Z,53.516566,8.188217,None,0.974|2010-12-07T12:20:00Z,53.516566,8.188217,None,1.057|2010-12-07T12:30:00Z,53.516566,8.188217,None,1.155|2010-12-07T12:40:00Z,53.516566,8.188217,None,1.249|2010-12-07T12:50:00Z,53.516566,8.188217,None,1.292|2010-12-07T13:00:00Z,53.516566,8.188217,None,1.315|2010-12-07T13:10:00Z,53.516566,8.188217,None,1.276|2010-12-07T13:20:00Z,53.516566,8.188217,None,1.024|2010-12-07T13:30:00Z,53.516566,8.188217,None,0.833|2010-12-07T13:40:00Z,53.516566,8.188217,None,0.843|2010-12-07T13:50:00Z,53.516566,8.188217,None,0.799|2010-12-07T14:00:00Z,53.516566,8.188217,None,0.517|2010-12-07T14:10:00Z,53.516566,8.188217,None,0.221|2010-12-07T14:20:00Z,53.516566,8.188217,None,0.062|2010-12-07T14:30:00Z,53.516566,8.188217,None,0.091|2010-12-07T14:40:00Z,53.516566,8.188217,None,0.158|2010-12-07T14:50:00Z,53.516566,8.188217,None,0.2|2010-12-07T15:00:00Z,53.516566,8.188217,None,0.22|2010-12-07T15:10:00Z,53.516566,8.188217,None,0.198|2010-12-07T15:20:00Z,53.516566,8.188217,None,0.17|2010-12-07T15:30:00Z,53.516566,8.188217,None,0.133|2010-12-07T15:40:00Z,53.516566,8.188217,None,0.131|2010-12-07T15:50:00Z,53.516566,8.188217,None,0.164|2010-12-07T16:00:00Z,53.516566,8.188217,None,0.227|2010-12-07T16:10:00Z,53.516566,8.188217,None,0.283|2010-12-07T16:20:00Z,53.516566,8.188217,None,0.324|2010-12-07T16:30:00Z,53.516566,8.188217,None,0.381|2010-12-07T16:40:00Z,53.516566,8.188217,None,0.414|2010-12-07T16:50:00Z,53.516566,8.188217,None,0.429|2010-12-07T17:00:00Z,53.516566,8.188217,None,0.413|2010-12-07T17:10:00Z,53.516566,8.188217,None,0.389|2010-12-07T17:20:00Z,53.516566,8.188217,None,0.364|2010-12-07T17:30:00Z,53.516566,8.188217,None,0.406|2010-12-07T17:40:00Z,53.516566,8.188217,None,0.579|2010-12-07T17:50:00Z,53.516566,8.188217,None,0.539|2010-12-07T18:00:00Z,53.516566,8.188217,None,0.352|2010-12-07T18:10:00Z,53.516566,8.188217,None,0.273|2010-12-07T18:20:00Z,53.516566,8.188217,None,0.199|2010-12-07T18:30:00Z,53.516566,8.188217,None,0.225|2010-12-07T18:40:00Z,53.516566,8.188217,None,0.462|2010-12-07T18:50:00Z,53.516566,8.188217,None,0.628|2010-12-07T19:00:00Z,53.516566,8.188217,None,0.722|2010-12-07T19:10:00Z,53.516566,8.188217,None,0.772|2010-12-07T19:20:00Z,53.516566,8.188217,None,0.754|2010-12-07T19:30:00Z,53.516566,8.188217,None,0.745|2010-12-07T19:40:00Z,53.516566,8.188217,None,0.664|2010-12-07T19:50:00Z,53.516566,8.188217,None,0.433|2010-12-07T20:00:00Z,53.516566,8.188217,None,0.209|2010-12-07T20:10:00Z,53.516566,8.188217,None,0.367|2010-12-07T20:20:00Z,53.516566,8.188217,None,0.345|2010-12-07T20:30:00Z,53.516566,8.188217,None,0.345|2010-12-07T20:40:00Z,53.516566,8.188217,None,0.188|2010-12-07T20:50:00Z,53.516566,8.188217,None,0.123|2010-12-07T21:00:00Z,53.516566,8.188217,None,0.101|2010-12-07T21:10:00Z,53.516566,8.188217,None,0.016|2010-12-07T21:20:00Z,53.516566,8.188217,None,-0.007|2010-12-07T21:30:00Z,53.516566,8.188217,None,0.058|2010-12-07T21:40:00Z,53.516566,8.188217,None,0.198|2010-12-07T21:50:00Z,53.516566,8.188217,None,0.396|2010-12-07T22:00:00Z,53.516566,8.188217,None,0.505|2010-12-07T22:10:00Z,53.516566,8.188217,None,0.561|2010-12-07T22:20:00Z,53.516566,8.188217,None,0.54|2010-12-07T22:30:00Z,53.516566,8.188217,None,0.557|2010-12-07T22:40:00Z,53.516566,8.188217,None,0.675|2010-12-07T22:50:00Z,53.516566,8.188217,None,1.019|2010-12-07T23:00:00Z,53.516566,8.188217,None,1.224|2010-12-07T23:10:00Z,53.516566,8.188217,None,1.222|2010-12-07T23:20:00Z,53.516566,8.188217,None,1.113|2010-12-07T23:30:00Z,53.516566,8.188217,None,1.017|2010-12-07T23:40:00Z,53.516566,8.188217,None,0.916|2010-12-07T23:50:00Z,53.516566,8.188217,None,0.884|2010-12-08T00:00:00Z,53.516566,8.188217,None,1.013|2010-12-08T00:10:00Z,53.516566,8.188217,None,1.14|2010-12-08T00:20:00Z,53.516566,8.188217,None,1.461|2010-12-08T00:30:00Z,53.516566,8.188217,None,1.596|2010-12-08T00:40:00Z,53.516566,8.188217,None,1.698|2010-12-08T00:50:00Z,53.516566,8.188217,None,1.811|2010-12-08T01:00:00Z,53.516566,8.188217,None,1.815|2010-12-08T01:10:00Z,53.516566,8.188217,None,1.813|2010-12-08T01:20:00Z,53.516566,8.188217,None,1.803|2010-12-08T01:30:00Z,53.516566,8.188217,None,1.761|2010-12-08T01:40:00Z,53.516566,8.188217,None,1.634|2010-12-08T01:50:00Z,53.516566,8.188217,None,1.533|2010-12-08T02:00:00Z,53.516566,8.188217,None,1.047|2010-12-08T02:10:00Z,53.516566,8.188217,None,0.907|2010-12-08T02:20:00Z,53.516566,8.188217,None,0.79|2010-12-08T02:30:00Z,53.516566,8.188217,None,0.524|2010-12-08T02:40:00Z,53.516566,8.188217,None,0.168|2010-12-08T02:50:00Z,53.516566,8.188217,None,0.058|2010-12-08T03:00:00Z,53.516566,8.188217,None,0.07|2010-12-08T03:10:00Z,53.516566,8.188217,None,0.074|2010-12-08T03:20:00Z,53.516566,8.188217,None,0.102|2010-12-08T03:30:00Z,53.516566,8.188217,None,0.147|2010-12-08T03:40:00Z,53.516566,8.188217,None,0.211|2010-12-08T03:50:00Z,53.516566,8.188217,None,0.277|2010-12-08T04:00:00Z,53.516566,8.188217,None,0.363|2010-12-08T04:10:00Z,53.516566,8.188217,None,0.446|2010-12-08T04:20:00Z,53.516566,8.188217,None,0.503|2010-12-08T04:30:00Z,53.516566,8.188217,None,0.515|2010-12-08T04:40:00Z,53.516566,8.188217,None,0.512|2010-12-08T04:50:00Z,53.516566,8.188217,None,0.54|2010-12-08T05:00:00Z,53.516566,8.188217,None,0.538|2010-12-08T05:10:00Z,53.516566,8.188217,None,0.53|2010-12-08T05:20:00Z,53.516566,8.188217,None,0.516|2010-12-08T05:30:00Z,53.516566,8.188217,None,0.542|2010-12-08T05:40:00Z,53.516566,8.188217,None,0.571|2010-12-08T05:50:00Z,53.516566,8.188217,None,0.639|2010-12-08T06:00:00Z,53.516566,8.188217,None,0.674|2010-12-08T06:10:00Z,53.516566,8.188217,None,0.626|2010-12-08T06:20:00Z,53.516566,8.188217,None,0.356|2010-12-08T06:30:00Z,53.516566,8.188217,None,0.207|2010-12-08T06:40:00Z,53.516566,8.188217,None,0.579|2010-12-08T06:50:00Z,53.516566,8.188217,None,0.694|2010-12-08T07:00:00Z,53.516566,8.188217,None,0.72|2010-12-08T07:10:00Z,53.516566,8.188217,None,0.741|2010-12-08T07:20:00Z,53.516566,8.188217,None,0.75|2010-12-08T07:30:00Z,53.516566,8.188217,None,0.778|2010-12-08T07:40:00Z,53.516566,8.188217,None,0.684|2010-12-08T07:50:00Z,53.516566,8.188217,None,0.564|2010-12-08T08:00:00Z,53.516566,8.188217,None,0.595|2010-12-08T08:10:00Z,53.516566,8.188217,None,0.597|2010-12-08T08:20:00Z,53.516566,8.188217,None,0.655|2010-12-08T08:30:00Z,53.516566,8.188217,None,0.357|2010-12-08T08:40:00Z,53.516566,8.188217,None,0.355|2010-12-08T08:50:00Z,53.516566,8.188217,None,0.325|2010-12-08T09:00:00Z,53.516566,8.188217,None,0.254|2010-12-08T09:10:00Z,53.516566,8.188217,None,0.248|2010-12-08T09:20:00Z,53.516566,8.188217,None,0.123|2010-12-08T09:30:00Z,53.516566,8.188217,None,0.025|2010-12-08T09:40:00Z,53.516566,8.188217,None,0.033|2010-12-08T09:50:00Z,53.516566,8.188217,None,0.015|2010-12-08T10:00:00Z,53.516566,8.188217,None,0.133|2010-12-08T10:10:00Z,53.516566,8.188217,None,0.21|2010-12-08T10:20:00Z,53.516566,8.188217,None,0.249|2010-12-08T10:30:00Z,53.516566,8.188217,None,0.415|2010-12-08T10:40:00Z,53.516566,8.188217,None,0.466|2010-12-08T10:50:00Z,53.516566,8.188217,None,0.489|2010-12-08T11:00:00Z,53.516566,8.188217,None,0.506|2010-12-08T11:10:00Z,53.516566,8.188217,None,0.581|2010-12-08T11:20:00Z,53.516566,8.188217,None,-1.855|2010-12-08T11:30:00Z,53.516566,8.188217,None,-3.154|2010-12-08T11:40:00Z,53.516566,8.188217,None,-2.779|2010-12-08T11:50:00Z,53.516566,8.188217,None,1.223|2010-12-08T12:00:00Z,53.516566,8.188217,None,1.186|2010-12-08T12:10:00Z,53.516566,8.188217,None,1.125|2010-12-08T12:20:00Z,53.516566,8.188217,None,1.076|2010-12-08T12:30:00Z,53.516566,8.188217,None,1.049|2010-12-08T12:40:00Z,53.516566,8.188217,None,1.039|2010-12-08T12:50:00Z,53.516566,8.188217,None,1.025|2010-12-08T13:00:00Z,53.516566,8.188217,None,1.051|2010-12-08T13:10:00Z,53.516566,8.188217,None,1.073|2010-12-08T13:20:00Z,53.516566,8.188217,None,1.104|2010-12-08T13:30:00Z,53.516566,8.188217,None,1.144|2010-12-08T13:40:00Z,53.516566,8.188217,None,1.167|2010-12-08T13:50:00Z,53.516566,8.188217,None,1.165|2010-12-08T14:00:00Z,53.516566,8.188217,None,1.138|2010-12-08T14:10:00Z,53.516566,8.188217,None,1.076|2010-12-08T14:20:00Z,53.516566,8.188217,None,1.013|2010-12-08T14:30:00Z,53.516566,8.188217,None,1.008|2010-12-08T14:40:00Z,53.516566,8.188217,None,1.062|2010-12-08T14:50:00Z,53.516566,8.188217,None,0.847|2010-12-08T15:00:00Z,53.516566,8.188217,None,0.532|2010-12-08T15:10:00Z,53.516566,8.188217,None,0.19|2010-12-08T15:20:00Z,53.516566,8.188217,None,0.083|2010-12-08T15:30:00Z,53.516566,8.188217,None,0.063|2010-12-08T15:40:00Z,53.516566,8.188217,None,0.061|2010-12-08T15:50:00Z,53.516566,8.188217,None,0.081|2010-12-08T16:00:00Z,53.516566,8.188217,None,0.117|2010-12-08T16:10:00Z,53.516566,8.188217,None,0.17|2010-12-08T16:20:00Z,53.516566,8.188217,None,0.241|2010-12-08T16:30:00Z,53.516566,8.188217,None,0.309|2010-12-08T16:40:00Z,53.516566,8.188217,None,0.384|2010-12-08T16:50:00Z,53.516566,8.188217,None,0.432|2010-12-08T17:00:00Z,53.516566,8.188217,None,0.489|2010-12-08T17:10:00Z,53.516566,8.188217,None,0.497|2010-12-08T17:20:00Z,53.516566,8.188217,None,0.481|2010-12-08T17:30:00Z,53.516566,8.188217,None,0.485|2010-12-08T17:40:00Z,53.516566,8.188217,None,0.496|2010-12-08T17:50:00Z,53.516566,8.188217,None,0.466|2010-12-08T18:00:00Z,53.516566,8.188217,None,0.488|2010-12-08T18:10:00Z,53.516566,8.188217,None,0.533|2010-12-08T18:20:00Z,53.516566,8.188217,None,0.618|2010-12-08T18:30:00Z,53.516566,8.188217,None,0.546|2010-12-08T18:40:00Z,53.516566,8.188217,None,0.181|2010-12-08T18:50:00Z,53.516566,8.188217,None,0.097|2010-12-08T19:00:00Z,53.516566,8.188217,None,0.203|2010-12-08T19:10:00Z,53.516566,8.188217,None,0.561|2010-12-08T19:20:00Z,53.516566,8.188217,None,0.681|2010-12-08T19:30:00Z,53.516566,8.188217,None,0.697|2010-12-08T19:40:00Z,53.516566,8.188217,None,0.646|2010-12-08T19:50:00Z,53.516566,8.188217,None,0.481|2010-12-08T20:00:00Z,53.516566,8.188217,None,0.459|2010-12-08T20:10:00Z,53.516566,8.188217,None,0.396|2010-12-08T20:20:00Z,53.516566,8.188217,None,0.244|2010-12-08T20:30:00Z,53.516566,8.188217,None,0.379|2010-12-08T20:40:00Z,53.516566,8.188217,None,0.144|2010-12-08T20:50:00Z,53.516566,8.188217,None,0.214|2010-12-08T21:00:00Z,53.516566,8.188217,None,0.202|2010-12-08T21:10:00Z,53.516566,8.188217,None,0.177|2010-12-08T21:20:00Z,53.516566,8.188217,None,0.084|2010-12-08T21:30:00Z,53.516566,8.188217,None,0.091|2010-12-08T21:40:00Z,53.516566,8.188217,None,0.076|2010-12-08T21:50:00Z,53.516566,8.188217,None,0.085|2010-12-08T22:00:00Z,53.516566,8.188217,None,0.091|2010-12-08T22:10:00Z,53.516566,8.188217,None,0.245|2010-12-08T22:20:00Z,53.516566,8.188217,None,0.344|2010-12-08T22:30:00Z,53.516566,8.188217,None,0.407|2010-12-08T22:40:00Z,53.516566,8.188217,None,0.511|2010-12-08T22:50:00Z,53.516566,8.188217,None,0.574|2010-12-08T23:00:00Z,53.516566,8.188217,None,0.569|2010-12-08T23:10:00Z,53.516566,8.188217,None,0.614|2010-12-08T23:20:00Z,53.516566,8.188217,None,0.736|2010-12-08T23:30:00Z,53.516566,8.188217,None,0.987|2010-12-08T23:40:00Z,53.516566,8.188217,None,1.152|2010-12-08T23:50:00Z,53.516566,8.188217,None,1.167|2010-12-09T00:00:00Z,53.516566,8.188217,None,1.084|2010-12-09T00:10:00Z,53.516566,8.188217,None,1.002|2010-12-09T00:20:00Z,53.516566,8.188217,None,1.019|2010-12-09T00:30:00Z,53.516566,8.188217,None,1.018|2010-12-09T00:40:00Z,53.516566,8.188217,None,1.49|2010-12-09T00:50:00Z,53.516566,8.188217,None,1.662|2010-12-09T01:00:00Z,53.516566,8.188217,None,1.687|2010-12-09T01:10:00Z,53.516566,8.188217,None,1.837|2010-12-09T01:20:00Z,53.516566,8.188217,None,1.916|2010-12-09T01:30:00Z,53.516566,8.188217,None,1.901|2010-12-09T01:40:00Z,53.516566,8.188217,None,1.895|2010-12-09T01:50:00Z,53.516566,8.188217,None,1.908|2010-12-09T02:00:00Z,53.516566,8.188217,None,1.908|2010-12-09T02:10:00Z,53.516566,8.188217,None,1.903|2010-12-09T02:20:00Z,53.516566,8.188217,None,1.881|2010-12-09T02:30:00Z,53.516566,8.188217,None,1.848|2010-12-09T02:40:00Z,53.516566,8.188217,None,1.794|2010-12-09T02:50:00Z,53.516566,8.188217,None,1.554|2010-12-09T03:00:00Z,53.516566,8.188217,None,1.048|2010-12-09T03:10:00Z,53.516566,8.188217,None,0.816|2010-12-09T03:20:00Z,53.516566,8.188217,None,0.546|2010-12-09T03:30:00Z,53.516566,8.188217,None,0.264|2010-12-09T03:40:00Z,53.516566,8.188217,None,0.092|2010-12-09T03:50:00Z,53.516566,8.188217,None,0.038|2010-12-09T04:00:00Z,53.516566,8.188217,None,0.019|2010-12-09T04:10:00Z,53.516566,8.188217,None,0.077|2010-12-09T04:20:00Z,53.516566,8.188217,None,0.213|2010-12-09T04:30:00Z,53.516566,8.188217,None,0.497|2010-12-09T04:40:00Z,53.516566,8.188217,None,0.797|2010-12-09T04:50:00Z,53.516566,8.188217,None,0.857|2010-12-09T05:00:00Z,53.516566,8.188217,None,0.825|2010-12-09T05:10:00Z,53.516566,8.188217,None,0.698|2010-12-09T05:20:00Z,53.516566,8.188217,None,0.512|2010-12-09T05:30:00Z,53.516566,8.188217,None,0.31|2010-12-09T05:40:00Z,53.516566,8.188217,None,0.2|2010-12-09T05:50:00Z,53.516566,8.188217,None,0.272|2010-12-09T06:00:00Z,53.516566,8.188217,None,0.321|2010-12-09T06:10:00Z,53.516566,8.188217,None,0.357|2010-12-09T06:20:00Z,53.516566,8.188217,None,0.445|2010-12-09T06:30:00Z,53.516566,8.188217,None,0.469|2010-12-09T06:40:00Z,53.516566,8.188217,None,0.459|2010-12-09T06:50:00Z,53.516566,8.188217,None,0.415|2010-12-09T07:00:00Z,53.516566,8.188217,None,0.499|2010-12-09T07:10:00Z,53.516566,8.188217,None,0.52|2010-12-09T07:20:00Z,53.516566,8.188217,None,0.429|2010-12-09T07:30:00Z,53.516566,8.188217,None,0.509|2010-12-09T07:40:00Z,53.516566,8.188217,None,0.457|2010-12-09T07:50:00Z,53.516566,8.188217,None,0.258|2010-12-09T08:00:00Z,53.516566,8.188217,None,0.152|2010-12-09T08:10:00Z,53.516566,8.188217,None,0.13|2010-12-09T08:20:00Z,53.516566,8.188217,None,0.204|2010-12-09T08:30:00Z,53.516566,8.188217,None,0.199|2010-12-09T08:40:00Z,53.516566,8.188217,None,0.212|2010-12-09T08:50:00Z,53.516566,8.188217,None,0.174|2010-12-09T09:00:00Z,53.516566,8.188217,None,0.009|2010-12-09T09:10:00Z,53.516566,8.188217,None,0.28|2010-12-09T09:20:00Z,53.516566,8.188217,None,0.224|2010-12-09T09:30:00Z,53.516566,8.188217,None,0.124|2010-12-09T09:40:00Z,53.516566,8.188217,None,0.124|2010-12-09T09:50:00Z,53.516566,8.188217,None,0.187|2010-12-09T10:00:00Z,53.516566,8.188217,None,0.249|2010-12-09T10:10:00Z,53.516566,8.188217,None,0.197|2010-12-09T10:20:00Z,53.516566,8.188217,None,0.274|2010-12-09T10:30:00Z,53.516566,8.188217,None,0.404|2010-12-09T10:40:00Z,53.516566,8.188217,None,0.459|2010-12-09T10:50:00Z,53.516566,8.188217,None,0.546|2010-12-09T11:00:00Z,53.516566,8.188217,None,0.592|2010-12-09T11:10:00Z,53.516566,8.188217,None,0.706|2010-12-09T11:20:00Z,53.516566,8.188217,None,0.768|2010-12-09T11:30:00Z,53.516566,8.188217,None,0.742|2010-12-09T11:40:00Z,53.516566,8.188217,None,0.885|2010-12-09T11:50:00Z,53.516566,8.188217,None,1.042|2010-12-09T12:00:00Z,53.516566,8.188217,None,1.114|2010-12-09T12:10:00Z,53.516566,8.188217,None,1.286|2010-12-09T12:20:00Z,53.516566,8.188217,None,1.408|2010-12-09T12:30:00Z,53.516566,8.188217,None,1.446|2010-12-09T12:40:00Z,53.516566,8.188217,None,1.38|2010-12-09T12:50:00Z,53.516566,8.188217,None,1.451|2010-12-09T13:00:00Z,53.516566,8.188217,None,1.494|2010-12-09T13:10:00Z,53.516566,8.188217,None,1.546|2010-12-09T13:20:00Z,53.516566,8.188217,None,1.559|2010-12-09T13:30:00Z,53.516566,8.188217,None,1.566|2010-12-09T13:40:00Z,53.516566,8.188217,None,1.65|2010-12-09T13:50:00Z,53.516566,8.188217,None,1.831|2010-12-09T14:00:00Z,53.516566,8.188217,None,1.858|2010-12-09T14:10:00Z,53.516566,8.188217,None,1.875|2010-12-09T14:20:00Z,53.516566,8.188217,None,1.826|2010-12-09T14:30:00Z,53.516566,8.188217,None,1.759|2010-12-09T14:40:00Z,53.516566,8.188217,None,1.62|2010-12-09T14:50:00Z,53.516566,8.188217,None,1.498|2010-12-09T15:00:00Z,53.516566,8.188217,None,1.367|2010-12-09T15:10:00Z,53.516566,8.188217,None,1.148|2010-12-09T15:20:00Z,53.516566,8.188217,None,0.85|2010-12-09T15:30:00Z,53.516566,8.188217,None,0.642|2010-12-09T15:40:00Z,53.516566,8.188217,None,0.447|2010-12-09T15:50:00Z,53.516566,8.188217,None,0.318|2010-12-09T16:00:00Z,53.516566,8.188217,None,0.249|2010-12-09T16:10:00Z,53.516566,8.188217,None,0.206|2010-12-09T16:20:00Z,53.516566,8.188217,None,0.257|2010-12-09T16:30:00Z,53.516566,8.188217,None,0.425|2010-12-09T16:40:00Z,53.516566,8.188217,None,0.61|2010-12-09T16:50:00Z,53.516566,8.188217,None,0.829|2010-12-09T17:00:00Z,53.516566,8.188217,None,1.002|2010-12-09T17:10:00Z,53.516566,8.188217,None,1.093|2010-12-09T17:20:00Z,53.516566,8.188217,None,1.153|2010-12-09T17:30:00Z,53.516566,8.188217,None,1.129|2010-12-09T17:40:00Z,53.516566,8.188217,None,1.065|2010-12-09T17:50:00Z,53.516566,8.188217,None,1.095|2010-12-09T18:00:00Z,53.516566,8.188217,None,1.099|2010-12-09T18:10:00Z,53.516566,8.188217,None,1.069|2010-12-09T18:20:00Z,53.516566,8.188217,None,1.033|2010-12-09T18:30:00Z,53.516566,8.188217,None,1.005|2010-12-09T18:40:00Z,53.516566,8.188217,None,0.99|2010-12-09T18:50:00Z,53.516566,8.188217,None,1.049|2010-12-09T19:00:00Z,53.516566,8.188217,None,1.051|2010-12-09T19:10:00Z,53.516566,8.188217,None,1.059|2010-12-09T19:20:00Z,53.516566,8.188217,None,1.106|2010-12-09T19:30:00Z,53.516566,8.188217,None,0.797|2010-12-09T19:40:00Z,53.516566,8.188217,None,0.739|2010-12-09T19:50:00Z,53.516566,8.188217,None,0.659|2010-12-09T20:00:00Z,53.516566,8.188217,None,0.644|2010-12-09T20:10:00Z,53.516566,8.188217,None,0.645|2010-12-09T20:20:00Z,53.516566,8.188217,None,0.667|2010-12-09T20:30:00Z,53.516566,8.188217,None,0.623|2010-12-09T20:40:00Z,53.516566,8.188217,None,0.427|2010-12-09T20:50:00Z,53.516566,8.188217,None,0.35|2010-12-09T21:00:00Z,53.516566,8.188217,None,0.261|2010-12-09T21:10:00Z,53.516566,8.188217,None,0.323|2010-12-09T21:20:00Z,53.516566,8.188217,None,0.288|2010-12-09T21:30:00Z,53.516566,8.188217,None,0.318|2010-12-09T21:40:00Z,53.516566,8.188217,None,0.39|2010-12-09T21:50:00Z,53.516566,8.188217,None,0.279|2010-12-09T22:00:00Z,53.516566,8.188217,None,0.147|2010-12-09T22:10:00Z,53.516566,8.188217,None,0.161|2010-12-09T22:20:00Z,53.516566,8.188217,None,0.273|2010-12-09T22:30:00Z,53.516566,8.188217,None,0.314|2010-12-09T22:40:00Z,53.516566,8.188217,None,0.447|2010-12-09T22:50:00Z,53.516566,8.188217,None,0.614|2010-12-09T23:00:00Z,53.516566,8.188217,None,0.576|2010-12-09T23:10:00Z,53.516566,8.188217,None,0.603|2010-12-09T23:20:00Z,53.516566,8.188217,None,0.791|2010-12-09T23:30:00Z,53.516566,8.188217,None,0.812|2010-12-09T23:40:00Z,53.516566,8.188217,None,0.805|2010-12-09T23:50:00Z,53.516566,8.188217,None,0.914|2010-12-10T00:00:00Z,53.516566,8.188217,None,1.048|2010-12-10T00:10:00Z,53.516566,8.188217,None,1.204|2010-12-10T00:20:00Z,53.516566,8.188217,None,1.335|2010-12-10T00:30:00Z,53.516566,8.188217,None,1.402|2010-12-10T00:40:00Z,53.516566,8.188217,None,1.413|2010-12-10T00:50:00Z,53.516566,8.188217,None,1.385|2010-12-10T01:00:00Z,53.516566,8.188217,None,1.339|2010-12-10T01:10:00Z,53.516566,8.188217,None,1.352|2010-12-10T01:20:00Z,53.516566,8.188217,None,1.366|2010-12-10T01:30:00Z,53.516566,8.188217,None,1.378|2010-12-10T01:40:00Z,53.516566,8.188217,None,1.436|2010-12-10T01:50:00Z,53.516566,8.188217,None,1.465|2010-12-10T02:00:00Z,53.516566,8.188217,None,1.473|2010-12-10T02:10:00Z,53.516566,8.188217,None,1.484|2010-12-10T02:20:00Z,53.516566,8.188217,None,1.486|2010-12-10T02:30:00Z,53.516566,8.188217,None,1.482|2010-12-10T02:40:00Z,53.516566,8.188217,None,1.442|2010-12-10T02:50:00Z,53.516566,8.188217,None,1.354|2010-12-10T03:00:00Z,53.516566,8.188217,None,1.307|2010-12-10T03:10:00Z,53.516566,8.188217,None,1.182|2010-12-10T03:20:00Z,53.516566,8.188217,None,0.907|2010-12-10T03:30:00Z,53.516566,8.188217,None,0.614|2010-12-10T03:40:00Z,53.516566,8.188217,None,0.413|2010-12-10T03:50:00Z,53.516566,8.188217,None,0.223|2010-12-10T04:00:00Z,53.516566,8.188217,None,0.194|2010-12-10T04:10:00Z,53.516566,8.188217,None,0.176|2010-12-10T04:20:00Z,53.516566,8.188217,None,0.177|2010-12-10T04:30:00Z,53.516566,8.188217,None,0.209|2010-12-10T04:40:00Z,53.516566,8.188217,None,0.285|2010-12-10T04:50:00Z,53.516566,8.188217,None,0.394|2010-12-10T05:00:00Z,53.516566,8.188217,None,0.524|2010-12-10T05:10:00Z,53.516566,8.188217,None,0.597|2010-12-10T05:20:00Z,53.516566,8.188217,None,0.682|2010-12-10T05:30:00Z,53.516566,8.188217,None,0.728|2010-12-10T05:40:00Z,53.516566,8.188217,None,0.756|2010-12-10T05:50:00Z,53.516566,8.188217,None,0.808|2010-12-10T06:00:00Z,53.516566,8.188217,None,0.832|2010-12-10T06:10:00Z,53.516566,8.188217,None,0.82|2010-12-10T06:20:00Z,53.516566,8.188217,None,0.795|2010-12-10T06:30:00Z,53.516566,8.188217,None,0.796|2010-12-10T06:40:00Z,53.516566,8.188217,None,0.788|2010-12-10T06:50:00Z,53.516566,8.188217,None,0.789|2010-12-10T07:00:00Z,53.516566,8.188217,None,0.747|2010-12-10T07:10:00Z,53.516566,8.188217,None,0.778|2010-12-10T07:20:00Z,53.516566,8.188217,None,0.872|2010-12-10T07:30:00Z,53.516566,8.188217,None,0.908|2010-12-10T07:40:00Z,53.516566,8.188217,None,0.723|2010-12-10T07:50:00Z,53.516566,8.188217,None,0.564|2010-12-10T08:00:00Z,53.516566,8.188217,None,0.52|2010-12-10T08:10:00Z,53.516566,8.188217,None,0.515|2010-12-10T08:20:00Z,53.516566,8.188217,None,0.526|2010-12-10T08:30:00Z,53.516566,8.188217,None,0.402|2010-12-10T08:40:00Z,53.516566,8.188217,None,0.436|2010-12-10T08:50:00Z,53.516566,8.188217,None,0.445|2010-12-10T09:00:00Z,53.516566,8.188217,None,0.482|2010-12-10T09:10:00Z,53.516566,8.188217,None,0.477|2010-12-10T09:20:00Z,53.516566,8.188217,None,0.411|2010-12-10T09:30:00Z,53.516566,8.188217,None,0.271|2010-12-10T09:40:00Z,53.516566,8.188217,None,0.243|2010-12-10T09:50:00Z,53.516566,8.188217,None,0.21|2010-12-10T10:00:00Z,53.516566,8.188217,None,0.032|2010-12-10T10:10:00Z,53.516566,8.188217,None,-0.081|2010-12-10T10:20:00Z,53.516566,8.188217,None,-0.061|2010-12-10T10:30:00Z,53.516566,8.188217,None,-0.026|2010-12-10T10:40:00Z,53.516566,8.188217,None,0.018|2010-12-10T10:50:00Z,53.516566,8.188217,None,0.067|2010-12-10T11:00:00Z,53.516566,8.188217,None,0.138|2010-12-10T11:10:00Z,53.516566,8.188217,None,0.146|2010-12-10T11:20:00Z,53.516566,8.188217,None,0.259|2010-12-10T11:30:00Z,53.516566,8.188217,None,0.34|2010-12-10T11:40:00Z,53.516566,8.188217,None,0.396|2010-12-10T11:50:00Z,53.516566,8.188217,None,0.466|2010-12-10T12:00:00Z,53.516566,8.188217,None,0.545|2010-12-10T12:10:00Z,53.516566,8.188217,None,0.618|2010-12-10T12:20:00Z,53.516566,8.188217,None,0.646|2010-12-10T12:30:00Z,53.516566,8.188217,None,0.696|2010-12-10T12:40:00Z,53.516566,8.188217,None,0.811|2010-12-10T12:50:00Z,53.516566,8.188217,None,0.965|2010-12-10T13:00:00Z,53.516566,8.188217,None,1.109|2010-12-10T13:10:00Z,53.516566,8.188217,None,1.18|2010-12-10T13:20:00Z,53.516566,8.188217,None,1.205|2010-12-10T13:30:00Z,53.516566,8.188217,None,1.197|2010-12-10T13:40:00Z,53.516566,8.188217,None,1.197|2010-12-10T13:50:00Z,53.516566,8.188217,None,1.181|2010-12-10T14:00:00Z,53.516566,8.188217,None,1.153|2010-12-10T14:10:00Z,53.516566,8.188217,None,1.133|2010-12-10T14:20:00Z,53.516566,8.188217,None,1.119|2010-12-10T14:30:00Z,53.516566,8.188217,None,1.107|2010-12-10T14:40:00Z,53.516566,8.188217,None,1.099|2010-12-10T14:50:00Z,53.516566,8.188217,None,1.09|2010-12-10T15:00:00Z,53.516566,8.188217,None,1.08|2010-12-10T15:10:00Z,53.516566,8.188217,None,1.083|2010-12-10T15:20:00Z,53.516566,8.188217,None,1.117|2010-12-10T15:30:00Z,53.516566,8.188217,None,1.177|2010-12-10T15:40:00Z,53.516566,8.188217,None,1.177|2010-12-10T15:50:00Z,53.516566,8.188217,None,0.937|2010-12-10T16:00:00Z,53.516566,8.188217,None,0.624|2010-12-10T16:10:00Z,53.516566,8.188217,None,0.376|2010-12-10T16:20:00Z,53.516566,8.188217,None,0.272|2010-12-10T16:30:00Z,53.516566,8.188217,None,0.228|2010-12-10T16:40:00Z,53.516566,8.188217,None,0.246|2010-12-10T16:50:00Z,53.516566,8.188217,None,0.233|2010-12-10T17:00:00Z,53.516566,8.188217,None,0.348|2010-12-10T17:10:00Z,53.516566,8.188217,None,0.579|2010-12-10T17:20:00Z,53.516566,8.188217,None,0.671|2010-12-10T17:30:00Z,53.516566,8.188217,None,0.715|2010-12-10T17:40:00Z,53.516566,8.188217,None,0.739|2010-12-10T17:50:00Z,53.516566,8.188217,None,0.74|2010-12-10T18:00:00Z,53.516566,8.188217,None,0.743|2010-12-10T18:10:00Z,53.516566,8.188217,None,0.745|2010-12-10T18:20:00Z,53.516566,8.188217,None,0.735|2010-12-10T18:30:00Z,53.516566,8.188217,None,0.747|2010-12-10T18:40:00Z,53.516566,8.188217,None,0.756|2010-12-10T18:50:00Z,53.516566,8.188217,None,0.956|2010-12-10T19:00:00Z,53.516566,8.188217,None,1.05|2010-12-10T19:10:00Z,53.516566,8.188217,None,1.028|2010-12-10T19:20:00Z,53.516566,8.188217,None,1.022|2010-12-10T19:30:00Z,53.516566,8.188217,None,0.936|2010-12-10T19:40:00Z,53.516566,8.188217,None,0.869|2010-12-10T19:50:00Z,53.516566,8.188217,None,0.838|2010-12-10T20:00:00Z,53.516566,8.188217,None,0.84|2010-12-10T20:10:00Z,53.516566,8.188217,None,0.77|2010-12-10T20:20:00Z,53.516566,8.188217,None,0.72|2010-12-10T20:30:00Z,53.516566,8.188217,None,0.636|2010-12-10T20:40:00Z,53.516566,8.188217,None,0.593|2010-12-10T20:50:00Z,53.516566,8.188217,None,0.576|2010-12-10T21:00:00Z,53.516566,8.188217,None,0.549|2010-12-10T21:10:00Z,53.516566,8.188217,None,0.513|2010-12-10T21:20:00Z,53.516566,8.188217,None,0.516|2010-12-10T21:30:00Z,53.516566,8.188217,None,0.374|2010-12-10T21:40:00Z,53.516566,8.188217,None,0.203|2010-12-10T21:50:00Z,53.516566,8.188217,None,0.175|2010-12-10T22:00:00Z,53.516566,8.188217,None,0.212|2010-12-10T22:10:00Z,53.516566,8.188217,None,0.399|2010-12-10T22:20:00Z,53.516566,8.188217,None,0.403|2010-12-10T22:30:00Z,53.516566,8.188217,None,0.433|2010-12-10T22:40:00Z,53.516566,8.188217,None,0.454|2010-12-10T22:50:00Z,53.516566,8.188217,None,0.512|2010-12-10T23:00:00Z,53.516566,8.188217,None,0.489|2010-12-10T23:10:00Z,53.516566,8.188217,None,0.458|2010-12-10T23:20:00Z,53.516566,8.188217,None,0.473|2010-12-10T23:30:00Z,53.516566,8.188217,None,0.544|2010-12-10T23:40:00Z,53.516566,8.188217,None,0.654|2010-12-10T23:50:00Z,53.516566,8.188217,None,0.734|2010-12-11T00:00:00Z,53.516566,8.188217,None,0.767|2010-12-11T00:10:00Z,53.516566,8.188217,None,0.822|2010-12-11T00:20:00Z,53.516566,8.188217,None,0.923|2010-12-11T00:30:00Z,53.516566,8.188217,None,1.113|2010-12-11T00:40:00Z,53.516566,8.188217,None,1.211|2010-12-11T00:50:00Z,53.516566,8.188217,None,1.173|2010-12-11T01:00:00Z,53.516566,8.188217,None,1.069|2010-12-11T01:10:00Z,53.516566,8.188217,None,1.108|2010-12-11T01:20:00Z,53.516566,8.188217,None,1.344|2010-12-11T01:30:00Z,53.516566,8.188217,None,1.543|2010-12-11T01:40:00Z,53.516566,8.188217,None,1.526|2010-12-11T01:50:00Z,53.516566,8.188217,None,1.589|2010-12-11T02:00:00Z,53.516566,8.188217,None,1.888|2010-12-11T02:10:00Z,53.516566,8.188217,None,1.912|2010-12-11T02:20:00Z,53.516566,8.188217,None,1.893|2010-12-11T02:30:00Z,53.516566,8.188217,None,1.954|2010-12-11T02:40:00Z,53.516566,8.188217,None,2.118|2010-12-11T02:50:00Z,53.516566,8.188217,None,2.173|2010-12-11T03:00:00Z,53.516566,8.188217,None,2.185|2010-12-11T03:10:00Z,53.516566,8.188217,None,2.193|2010-12-11T03:20:00Z,53.516566,8.188217,None,2.19|2010-12-11T03:30:00Z,53.516566,8.188217,None,2.189|2010-12-11T03:40:00Z,53.516566,8.188217,None,2.204|2010-12-11T03:50:00Z,53.516566,8.188217,None,2.14|2010-12-11T04:00:00Z,53.516566,8.188217,None,1.965|2010-12-11T04:10:00Z,53.516566,8.188217,None,1.792|2010-12-11T04:20:00Z,53.516566,8.188217,None,1.538|2010-12-11T04:30:00Z,53.516566,8.188217,None,1.302|2010-12-11T04:40:00Z,53.516566,8.188217,None,1.237|2010-12-11T04:50:00Z,53.516566,8.188217,None,1.015|2010-12-11T05:00:00Z,53.516566,8.188217,None,0.9|2010-12-11T05:10:00Z,53.516566,8.188217,None,1.019|2010-12-11T05:20:00Z,53.516566,8.188217,None,1.176|2010-12-11T05:30:00Z,53.516566,8.188217,None,1.223|2010-12-11T05:40:00Z,53.516566,8.188217,None,1.225|2010-12-11T05:50:00Z,53.516566,8.188217,None,1.223|2010-12-11T06:00:00Z,53.516566,8.188217,None,1.128|2010-12-11T06:10:00Z,53.516566,8.188217,None,0.984|2010-12-11T06:20:00Z,53.516566,8.188217,None,0.946|2010-12-11T06:30:00Z,53.516566,8.188217,None,0.972|2010-12-11T06:40:00Z,53.516566,8.188217,None,1.027|2010-12-11T06:50:00Z,53.516566,8.188217,None,1.076|2010-12-11T07:00:00Z,53.516566,8.188217,None,1.137|2010-12-11T07:10:00Z,53.516566,8.188217,None,1.195|2010-12-11T07:20:00Z,53.516566,8.188217,None,1.238|2010-12-11T07:30:00Z,53.516566,8.188217,None,1.279|2010-12-11T07:40:00Z,53.516566,8.188217,None,1.309|2010-12-11T07:50:00Z,53.516566,8.188217,None,1.327|2010-12-11T08:00:00Z,53.516566,8.188217,None,1.334|2010-12-11T08:10:00Z,53.516566,8.188217,None,1.335|2010-12-11T08:20:00Z,53.516566,8.188217,None,1.328|2010-12-11T08:30:00Z,53.516566,8.188217,None,1.32|2010-12-11T08:40:00Z,53.516566,8.188217,None,1.289|2010-12-11T08:50:00Z,53.516566,8.188217,None,1.226|2010-12-11T09:00:00Z,53.516566,8.188217,None,1.208|2010-12-11T09:10:00Z,53.516566,8.188217,None,1.191|2010-12-11T09:20:00Z,53.516566,8.188217,None,0.966|2010-12-11T09:30:00Z,53.516566,8.188217,None,0.892|2010-12-11T09:40:00Z,53.516566,8.188217,None,0.966|2010-12-11T09:50:00Z,53.516566,8.188217,None,0.984|2010-12-11T10:00:00Z,53.516566,8.188217,None,0.94|2010-12-11T10:10:00Z,53.516566,8.188217,None,0.922|2010-12-11T10:20:00Z,53.516566,8.188217,None,0.911|2010-12-11T10:30:00Z,53.516566,8.188217,None,0.934|2010-12-11T10:40:00Z,53.516566,8.188217,None,1.014|2010-12-11T10:50:00Z,53.516566,8.188217,None,1.053|2010-12-11T11:00:00Z,53.516566,8.188217,None,1.016|2010-12-11T11:10:00Z,53.516566,8.188217,None,0.985|2010-12-11T11:20:00Z,53.516566,8.188217,None,1.023|2010-12-11T11:30:00Z,53.516566,8.188217,None,0.952|2010-12-11T11:40:00Z,53.516566,8.188217,None,0.953|2010-12-11T11:50:00Z,53.516566,8.188217,None,1.018|2010-12-11T12:00:00Z,53.516566,8.188217,None,1.058|2010-12-11T12:10:00Z,53.516566,8.188217,None,1.113|2010-12-11T12:20:00Z,53.516566,8.188217,None,1.203|2010-12-11T12:30:00Z,53.516566,8.188217,None,1.22|2010-12-11T12:40:00Z,53.516566,8.188217,None,1.243|2010-12-11T12:50:00Z,53.516566,8.188217,None,1.406|2010-12-11T13:00:00Z,53.516566,8.188217,None,1.549|2010-12-11T13:10:00Z,53.516566,8.188217,None,1.739|2010-12-11T13:20:00Z,53.516566,8.188217,None,1.879|2010-12-11T13:30:00Z,53.516566,8.188217,None,1.889|2010-12-11T13:40:00Z,53.516566,8.188217,None,1.825|2010-12-11T13:50:00Z,53.516566,8.188217,None,1.718|2010-12-11T14:00:00Z,53.516566,8.188217,None,1.75|2010-12-11T14:10:00Z,53.516566,8.188217,None,1.808|2010-12-11T14:20:00Z,53.516566,8.188217,None,1.876|2010-12-11T14:30:00Z,53.516566,8.188217,None,1.915|2010-12-11T14:40:00Z,53.516566,8.188217,None,1.97|2010-12-11T14:50:00Z,53.516566,8.188217,None,1.97|2010-12-11T15:00:00Z,53.516566,8.188217,None,2.002|2010-12-11T15:10:00Z,53.516566,8.188217,None,2.024|2010-12-11T15:20:00Z,53.516566,8.188217,None,2.036|2010-12-11T15:30:00Z,53.516566,8.188217,None,2.044|2010-12-11T15:40:00Z,53.516566,8.188217,None,2.048|2010-12-11T15:50:00Z,53.516566,8.188217,None,1.992|2010-12-11T16:00:00Z,53.516566,8.188217,None,1.92|2010-12-11T16:10:00Z,53.516566,8.188217,None,1.865|2010-12-11T16:20:00Z,53.516566,8.188217,None,1.84|2010-12-11T16:30:00Z,53.516566,8.188217,None,1.88|2010-12-11T16:40:00Z,53.516566,8.188217,None,1.663|2010-12-11T16:50:00Z,53.516566,8.188217,None,1.52|2010-12-11T17:00:00Z,53.516566,8.188217,None,1.444|2010-12-11T17:10:00Z,53.516566,8.188217,None,1.409|2010-12-11T17:20:00Z,53.516566,8.188217,None,1.351|2010-12-11T17:30:00Z,53.516566,8.188217,None,1.364|2010-12-11T17:40:00Z,53.516566,8.188217,None,1.481|2010-12-11T17:50:00Z,53.516566,8.188217,None,1.575|2010-12-11T18:00:00Z,53.516566,8.188217,None,1.64|2010-12-11T18:10:00Z,53.516566,8.188217,None,1.692|2010-12-11T18:20:00Z,53.516566,8.188217,None,1.7|2010-12-11T18:30:00Z,53.516566,8.188217,None,1.63|2010-12-11T18:40:00Z,53.516566,8.188217,None,1.582|2010-12-11T18:50:00Z,53.516566,8.188217,None,1.572|2010-12-11T19:00:00Z,53.516566,8.188217,None,1.557|2010-12-11T19:10:00Z,53.516566,8.188217,None,1.564|2010-12-11T19:20:00Z,53.516566,8.188217,None,1.553|2010-12-11T19:30:00Z,53.516566,8.188217,None,1.552|2010-12-11T19:40:00Z,53.516566,8.188217,None,1.551|2010-12-11T19:50:00Z,53.516566,8.188217,None,1.545|2010-12-11T20:00:00Z,53.516566,8.188217,None,1.539|2010-12-11T20:10:00Z,53.516566,8.188217,None,1.535|2010-12-11T20:20:00Z,53.516566,8.188217,None,1.539|2010-12-11T20:30:00Z,53.516566,8.188217,None,1.523|2010-12-11T20:40:00Z,53.516566,8.188217,None,1.49|2010-12-11T20:50:00Z,53.516566,8.188217,None,1.447|2010-12-11T21:00:00Z,53.516566,8.188217,None,1.421|2010-12-11T21:10:00Z,53.516566,8.188217,None,1.378|2010-12-11T21:20:00Z,53.516566,8.188217,None,1.358|2010-12-11T21:30:00Z,53.516566,8.188217,None,1.366|2010-12-11T21:40:00Z,53.516566,8.188217,None,1.411|2010-12-11T21:50:00Z,53.516566,8.188217,None,1.434|2010-12-11T22:00:00Z,53.516566,8.188217,None,1.408|2010-12-11T22:10:00Z,53.516566,8.188217,None,1.419|2010-12-11T22:20:00Z,53.516566,8.188217,None,1.425|2010-12-11T22:30:00Z,53.516566,8.188217,None,1.423|2010-12-11T22:40:00Z,53.516566,8.188217,None,1.414|2010-12-11T22:50:00Z,53.516566,8.188217,None,1.397|2010-12-11T23:00:00Z,53.516566,8.188217,None,1.393|2010-12-11T23:10:00Z,53.516566,8.188217,None,1.4|2010-12-11T23:20:00Z,53.516566,8.188217,None,1.392|2010-12-11T23:30:00Z,53.516566,8.188217,None,1.359|2010-12-11T23:40:00Z,53.516566,8.188217,None,1.371|2010-12-11T23:50:00Z,53.516566,8.188217,None,1.412|2010-12-12T00:00:00Z,53.516566,8.188217,None,1.487|2010-12-12T00:10:00Z,53.516566,8.188217,None,1.503|2010-12-12T00:20:00Z,53.516566,8.188217,None,1.52|2010-12-12T00:30:00Z,53.516566,8.188217,None,1.54|2010-12-12T00:40:00Z,53.516566,8.188217,None,1.531|2010-12-12T00:50:00Z,53.516566,8.188217,None,1.642|2010-12-12T01:00:00Z,53.516566,8.188217,None,1.763|2010-12-12T01:10:00Z,53.516566,8.188217,None,1.887|2010-12-12T01:20:00Z,53.516566,8.188217,None,2.086|2010-12-12T01:30:00Z,53.516566,8.188217,None,2.123|2010-12-12T01:40:00Z,53.516566,8.188217,None,2.071|2010-12-12T01:50:00Z,53.516566,8.188217,None,1.97|2010-12-12T02:00:00Z,53.516566,8.188217,None,1.874|2010-12-12T02:10:00Z,53.516566,8.188217,None,1.887|2010-12-12T02:20:00Z,53.516566,8.188217,None,1.91|2010-12-12T02:30:00Z,53.516566,8.188217,None,1.902|2010-12-12T02:40:00Z,53.516566,8.188217,None,1.925|2010-12-12T02:50:00Z,53.516566,8.188217,None,2.031|2010-12-12T03:00:00Z,53.516566,8.188217,None,2.099|2010-12-12T03:10:00Z,53.516566,8.188217,None,2.157|2010-12-12T03:20:00Z,53.516566,8.188217,None,2.186|2010-12-12T03:30:00Z,53.516566,8.188217,None,2.192|2010-12-12T03:40:00Z,53.516566,8.188217,None,2.183|2010-12-12T03:50:00Z,53.516566,8.188217,None,2.146|2010-12-12T04:00:00Z,53.516566,8.188217,None,2.021|2010-12-12T04:10:00Z,53.516566,8.188217,None,1.917|2010-12-12T04:20:00Z,53.516566,8.188217,None,1.917|2010-12-12T04:30:00Z,53.516566,8.188217,None,2.039|2010-12-12T04:40:00Z,53.516566,8.188217,None,1.938|2010-12-12T04:50:00Z,53.516566,8.188217,None,1.665|2010-12-12T05:00:00Z,53.516566,8.188217,None,1.514|2010-12-12T05:10:00Z,53.516566,8.188217,None,1.452|2010-12-12T05:20:00Z,53.516566,8.188217,None,1.45|2010-12-12T05:30:00Z,53.516566,8.188217,None,1.543|2010-12-12T05:40:00Z,53.516566,8.188217,None,1.626|2010-12-12T05:50:00Z,53.516566,8.188217,None,1.692|2010-12-12T06:00:00Z,53.516566,8.188217,None,1.748|2010-12-12T06:10:00Z,53.516566,8.188217,None,1.788|2010-12-12T06:20:00Z,53.516566,8.188217,None,1.804|2010-12-12T06:30:00Z,53.516566,8.188217,None,1.812|2010-12-12T06:40:00Z,53.516566,8.188217,None,1.822|2010-12-12T06:50:00Z,53.516566,8.188217,None,1.83|2010-12-12T07:00:00Z,53.516566,8.188217,None,1.821|2010-12-12T07:10:00Z,53.516566,8.188217,None,1.8|2010-12-12T07:20:00Z,53.516566,8.188217,None,1.771|2010-12-12T07:30:00Z,53.516566,8.188217,None,1.739|2010-12-12T07:40:00Z,53.516566,8.188217,None,1.703|2010-12-12T07:50:00Z,53.516566,8.188217,None,1.672|2010-12-12T08:00:00Z,53.516566,8.188217,None,1.654|2010-12-12T08:10:00Z,53.516566,8.188217,None,1.658|2010-12-12T08:20:00Z,53.516566,8.188217,None,1.666|2010-12-12T08:30:00Z,53.516566,8.188217,None,1.658|2010-12-12T08:40:00Z,53.516566,8.188217,None,1.639|2010-12-12T08:50:00Z,53.516566,8.188217,None,1.579|2010-12-12T09:00:00Z,53.516566,8.188217,None,1.537|2010-12-12T09:10:00Z,53.516566,8.188217,None,1.507|2010-12-12T09:20:00Z,53.516566,8.188217,None,1.537|2010-12-12T09:30:00Z,53.516566,8.188217,None,1.567|2010-12-12T09:40:00Z,53.516566,8.188217,None,1.585|2010-12-12T09:50:00Z,53.516566,8.188217,None,1.572|2010-12-12T10:00:00Z,53.516566,8.188217,None,1.526|2010-12-12T10:10:00Z,53.516566,8.188217,None,1.523|2010-12-12T10:20:00Z,53.516566,8.188217,None,1.497|2010-12-12T10:30:00Z,53.516566,8.188217,None,1.483|2010-12-12T10:40:00Z,53.516566,8.188217,None,1.484|2010-12-12T10:50:00Z,53.516566,8.188217,None,1.484|2010-12-12T11:00:00Z,53.516566,8.188217,None,1.529|2010-12-12T11:10:00Z,53.516566,8.188217,None,1.558|2010-12-12T11:20:00Z,53.516566,8.188217,None,1.625|2010-12-12T11:30:00Z,53.516566,8.188217,None,1.639|2010-12-12T11:40:00Z,53.516566,8.188217,None,1.618|2010-12-12T11:50:00Z,53.516566,8.188217,None,1.591|2010-12-12T12:00:00Z,53.516566,8.188217,None,1.573|2010-12-12T12:10:00Z,53.516566,8.188217,None,1.557|2010-12-12T12:20:00Z,53.516566,8.188217,None,1.542|2010-12-12T12:30:00Z,53.516566,8.188217,None,1.541|2010-12-12T12:40:00Z,53.516566,8.188217,None,1.614|2010-12-12T12:50:00Z,53.516566,8.188217,None,1.647|2010-12-12T13:00:00Z,53.516566,8.188217,None,1.66|2010-12-12T13:10:00Z,53.516566,8.188217,None,1.668|2010-12-12T13:20:00Z,53.516566,8.188217,None,1.666|2010-12-12T13:30:00Z,53.516566,8.188217,None,1.675|2010-12-12T13:40:00Z,53.516566,8.188217,None,1.688|2010-12-12T13:50:00Z,53.516566,8.188217,None,1.688|2010-12-12T14:00:00Z,53.516566,8.188217,None,1.677|2010-12-12T14:10:00Z,53.516566,8.188217,None,1.686|2010-12-12T14:20:00Z,53.516566,8.188217,None,1.732|2010-12-12T14:30:00Z,53.516566,8.188217,None,1.786|2010-12-12T14:40:00Z,53.516566,8.188217,None,1.832|2010-12-12T14:50:00Z,53.516566,8.188217,None,1.872|2010-12-12T15:00:00Z,53.516566,8.188217,None,1.923|2010-12-12T15:10:00Z,53.516566,8.188217,None,1.962|2010-12-12T15:20:00Z,53.516566,8.188217,None,1.979|2010-12-12T15:30:00Z,53.516566,8.188217,None,1.979|2010-12-12T15:40:00Z,53.516566,8.188217,None,1.971|2010-12-12T15:50:00Z,53.516566,8.188217,None,1.952|2010-12-12T16:00:00Z,53.516566,8.188217,None,1.935|2010-12-12T16:10:00Z,53.516566,8.188217,None,1.919|2010-12-12T16:20:00Z,53.516566,8.188217,None,1.936|2010-12-12T16:30:00Z,53.516566,8.188217,None,1.95|2010-12-12T16:40:00Z,53.516566,8.188217,None,1.94|2010-12-12T16:50:00Z,53.516566,8.188217,None,1.896|2010-12-12T17:00:00Z,53.516566,8.188217,None,1.815|2010-12-12T17:10:00Z,53.516566,8.188217,None,1.691|2010-12-12T17:20:00Z,53.516566,8.188217,None,1.568|2010-12-12T17:30:00Z,53.516566,8.188217,None,1.525|2010-12-12T17:40:00Z,53.516566,8.188217,None,1.475|2010-12-12T17:50:00Z,53.516566,8.188217,None,1.411|2010-12-12T18:00:00Z,53.516566,8.188217,None,1.381|2010-12-12T18:10:00Z,53.516566,8.188217,None,1.366|2010-12-12T18:20:00Z,53.516566,8.188217,None,1.339|2010-12-12T18:30:00Z,53.516566,8.188217,None,1.303|2010-12-12T18:40:00Z,53.516566,8.188217,None,1.29|2010-12-12T18:50:00Z,53.516566,8.188217,None,1.291|2010-12-12T19:00:00Z,53.516566,8.188217,None,1.293|2010-12-12T19:10:00Z,53.516566,8.188217,None,1.293|2010-12-12T19:20:00Z,53.516566,8.188217,None,1.299|2010-12-12T19:30:00Z,53.516566,8.188217,None,1.311|2010-12-12T19:40:00Z,53.516566,8.188217,None,1.313|2010-12-12T19:50:00Z,53.516566,8.188217,None,1.31|2010-12-12T20:00:00Z,53.516566,8.188217,None,1.309|2010-12-12T20:10:00Z,53.516566,8.188217,None,1.313|2010-12-12T20:20:00Z,53.516566,8.188217,None,1.335|2010-12-12T20:30:00Z,53.516566,8.188217,None,1.345|2010-12-12T20:40:00Z,53.516566,8.188217,None,1.334|2010-12-12T20:50:00Z,53.516566,8.188217,None,1.223|2010-12-12T21:00:00Z,53.516566,8.188217,None,1.174|2010-12-12T21:10:00Z,53.516566,8.188217,None,1.243|2010-12-12T21:20:00Z,53.516566,8.188217,None,1.268|2010-12-12T21:30:00Z,53.516566,8.188217,None,1.305|2010-12-12T21:40:00Z,53.516566,8.188217,None,1.347|2010-12-12T21:50:00Z,53.516566,8.188217,None,1.379|2010-12-12T22:00:00Z,53.516566,8.188217,None,1.363|2010-12-12T22:10:00Z,53.516566,8.188217,None,1.317|2010-12-12T22:20:00Z,53.516566,8.188217,None,1.212|2010-12-12T22:30:00Z,53.516566,8.188217,None,1.207|2010-12-12T22:40:00Z,53.516566,8.188217,None,1.211|2010-12-12T22:50:00Z,53.516566,8.188217,None,1.128|2010-12-12T23:00:00Z,53.516566,8.188217,None,1.186|2010-12-12T23:10:00Z,53.516566,8.188217,None,1.233|2010-12-12T23:20:00Z,53.516566,8.188217,None,1.231|2010-12-12T23:30:00Z,53.516566,8.188217,None,1.241|2010-12-12T23:40:00Z,53.516566,8.188217,None,1.274|2010-12-12T23:50:00Z,53.516566,8.188217,None,1.298|2010-12-13T00:00:00Z,53.516566,8.188217,None,1.304|2010-12-13T00:10:00Z,53.516566,8.188217,None,1.307|2010-12-13T00:20:00Z,53.516566,8.188217,None,1.34|2010-12-13T00:30:00Z,53.516566,8.188217,None,1.353|2010-12-13T00:40:00Z,53.516566,8.188217,None,1.339|2010-12-13T00:50:00Z,53.516566,8.188217,None,1.352|2010-12-13T01:00:00Z,53.516566,8.188217,None,1.389|2010-12-13T01:10:00Z,53.516566,8.188217,None,1.396|2010-12-13T01:20:00Z,53.516566,8.188217,None,1.419|2010-12-13T01:30:00Z,53.516566,8.188217,None,1.467|2010-12-13T01:40:00Z,53.516566,8.188217,None,1.484|2010-12-13T01:50:00Z,53.516566,8.188217,None,1.499|2010-12-13T02:00:00Z,53.516566,8.188217,None,1.502|2010-12-13T02:10:00Z,53.516566,8.188217,None,1.534|2010-12-13T02:20:00Z,53.516566,8.188217,None,1.596|2010-12-13T02:30:00Z,53.516566,8.188217,None,1.666|2010-12-13T02:40:00Z,53.516566,8.188217,None,1.697|2010-12-13T02:50:00Z,53.516566,8.188217,None,1.714|2010-12-13T03:00:00Z,53.516566,8.188217,None,1.735|2010-12-13T03:10:00Z,53.516566,8.188217,None,1.742|2010-12-13T03:20:00Z,53.516566,8.188217,None,1.732|2010-12-13T03:30:00Z,53.516566,8.188217,None,1.711|2010-12-13T03:40:00Z,53.516566,8.188217,None,1.692|2010-12-13T03:50:00Z,53.516566,8.188217,None,1.683|2010-12-13T04:00:00Z,53.516566,8.188217,None,1.689|2010-12-13T04:10:00Z,53.516566,8.188217,None,1.688|2010-12-13T04:20:00Z,53.516566,8.188217,None,1.692|2010-12-13T04:30:00Z,53.516566,8.188217,None,1.692|2010-12-13T04:40:00Z,53.516566,8.188217,None,1.687|2010-12-13T04:50:00Z,53.516566,8.188217,None,1.667|2010-12-13T05:00:00Z,53.516566,8.188217,None,1.653|2010-12-13T05:10:00Z,53.516566,8.188217,None,1.67|2010-12-13T05:20:00Z,53.516566,8.188217,None,1.606|2010-12-13T05:30:00Z,53.516566,8.188217,None,1.435|2010-12-13T05:40:00Z,53.516566,8.188217,None,1.241|2010-12-13T05:50:00Z,53.516566,8.188217,None,1.116|2010-12-13T06:00:00Z,53.516566,8.188217,None,1.127|2010-12-13T06:10:00Z,53.516566,8.188217,None,1.158|2010-12-13T06:20:00Z,53.516566,8.188217,None,1.166|2010-12-13T06:30:00Z,53.516566,8.188217,None,1.164|2010-12-13T06:40:00Z,53.516566,8.188217,None,1.156|2010-12-13T06:50:00Z,53.516566,8.188217,None,1.152|2010-12-13T07:00:00Z,53.516566,8.188217,None,1.147|2010-12-13T07:10:00Z,53.516566,8.188217,None,1.132|2010-12-13T07:20:00Z,53.516566,8.188217,None,1.163|2010-12-13T07:30:00Z,53.516566,8.188217,None,1.183|2010-12-13T07:40:00Z,53.516566,8.188217,None,1.212|2010-12-13T07:50:00Z,53.516566,8.188217,None,1.225|2010-12-13T08:00:00Z,53.516566,8.188217,None,1.246|2010-12-13T08:10:00Z,53.516566,8.188217,None,1.308|2010-12-13T08:20:00Z,53.516566,8.188217,None,1.339|2010-12-13T08:30:00Z,53.516566,8.188217,None,1.354|2010-12-13T08:40:00Z,53.516566,8.188217,None,1.276|2010-12-13T08:50:00Z,53.516566,8.188217,None,1.15|2010-12-13T09:00:00Z,53.516566,8.188217,None,1.126|2010-12-13T09:10:00Z,53.516566,8.188217,None,1.111|2010-12-13T09:20:00Z,53.516566,8.188217,None,1.095|2010-12-13T09:30:00Z,53.516566,8.188217,None,1.069|2010-12-13T09:40:00Z,53.516566,8.188217,None,1.052|2010-12-13T09:50:00Z,53.516566,8.188217,None,1.046|2010-12-13T10:00:00Z,53.516566,8.188217,None,1.041|2010-12-13T10:10:00Z,53.516566,8.188217,None,1.034|2010-12-13T10:20:00Z,53.516566,8.188217,None,1.058|2010-12-13T10:30:00Z,53.516566,8.188217,None,1.061|2010-12-13T10:40:00Z,53.516566,8.188217,None,1.011|2010-12-13T10:50:00Z,53.516566,8.188217,None,0.819|2010-12-13T11:00:00Z,53.516566,8.188217,None,0.457|2010-12-13T11:10:00Z,53.516566,8.188217,None,0.452|2010-12-13T11:20:00Z,53.516566,8.188217,None,0.519|2010-12-13T11:30:00Z,53.516566,8.188217,None,0.62|2010-12-13T11:40:00Z,53.516566,8.188217,None,0.864|2010-12-13T11:50:00Z,53.516566,8.188217,None,0.836|2010-12-13T12:00:00Z,53.516566,8.188217,None,0.803|2010-12-13T12:10:00Z,53.516566,8.188217,None,0.813|2010-12-13T12:20:00Z,53.516566,8.188217,None,0.833|2010-12-13T12:30:00Z,53.516566,8.188217,None,0.892|2010-12-13T12:40:00Z,53.516566,8.188217,None,0.954|2010-12-13T12:50:00Z,53.516566,8.188217,None,1.006|2010-12-13T13:00:00Z,53.516566,8.188217,None,1.005|2010-12-13T13:10:00Z,53.516566,8.188217,None,1.069|2010-12-13T13:20:00Z,53.516566,8.188217,None,1.158|2010-12-13T13:30:00Z,53.516566,8.188217,None,1.143|2010-12-13T13:40:00Z,53.516566,8.188217,None,1.134|2010-12-13T13:50:00Z,53.516566,8.188217,None,1.177|2010-12-13T14:00:00Z,53.516566,8.188217,None,1.225|2010-12-13T14:10:00Z,53.516566,8.188217,None,1.245|2010-12-13T14:20:00Z,53.516566,8.188217,None,1.265|2010-12-13T14:30:00Z,53.516566,8.188217,None,1.31|2010-12-13T14:40:00Z,53.516566,8.188217,None,1.338|2010-12-13T14:50:00Z,53.516566,8.188217,None,1.366|2010-12-13T15:00:00Z,53.516566,8.188217,None,1.399|2010-12-13T15:10:00Z,53.516566,8.188217,None,1.44|2010-12-13T15:20:00Z,53.516566,8.188217,None,1.513|2010-12-13T15:30:00Z,53.516566,8.188217,None,1.549|2010-12-13T15:40:00Z,53.516566,8.188217,None,1.564|2010-12-13T15:50:00Z,53.516566,8.188217,None,1.584|2010-12-13T16:00:00Z,53.516566,8.188217,None,1.589|2010-12-13T16:10:00Z,53.516566,8.188217,None,1.587|2010-12-13T16:20:00Z,53.516566,8.188217,None,1.585|2010-12-13T16:30:00Z,53.516566,8.188217,None,1.579|2010-12-13T16:40:00Z,53.516566,8.188217,None,1.573|2010-12-13T16:50:00Z,53.516566,8.188217,None,1.535|2010-12-13T17:00:00Z,53.516566,8.188217,None,1.511|2010-12-13T17:10:00Z,53.516566,8.188217,None,1.503|2010-12-13T17:20:00Z,53.516566,8.188217,None,1.473|2010-12-13T17:30:00Z,53.516566,8.188217,None,1.375|2010-12-13T17:40:00Z,53.516566,8.188217,None,1.22|2010-12-13T17:50:00Z,53.516566,8.188217,None,0.998|2010-12-13T18:00:00Z,53.516566,8.188217,None,0.871|2010-12-13T18:10:00Z,53.516566,8.188217,None,0.824|2010-12-13T18:20:00Z,53.516566,8.188217,None,0.8|2010-12-13T18:30:00Z,53.516566,8.188217,None,0.811|2010-12-13T18:40:00Z,53.516566,8.188217,None,0.883|2010-12-13T18:50:00Z,53.516566,8.188217,None,0.916|2010-12-13T19:00:00Z,53.516566,8.188217,None,0.907|2010-12-13T19:10:00Z,53.516566,8.188217,None,0.911|2010-12-13T19:20:00Z,53.516566,8.188217,None,0.933|2010-12-13T19:30:00Z,53.516566,8.188217,None,0.916|2010-12-13T19:40:00Z,53.516566,8.188217,None,0.877|2010-12-13T19:50:00Z,53.516566,8.188217,None,0.847|2010-12-13T20:00:00Z,53.516566,8.188217,None,0.893|2010-12-13T20:10:00Z,53.516566,8.188217,None,0.939|2010-12-13T20:20:00Z,53.516566,8.188217,None,1.006|2010-12-13T20:30:00Z,53.516566,8.188217,None,1.1|2010-12-13T20:40:00Z,53.516566,8.188217,None,1.156|2010-12-13T20:50:00Z,53.516566,8.188217,None,1.082|2010-12-13T21:00:00Z,53.516566,8.188217,None,1.009|2010-12-13T21:10:00Z,53.516566,8.188217,None,0.952|2010-12-13T21:20:00Z,53.516566,8.188217,None,0.875|2010-12-13T21:30:00Z,53.516566,8.188217,None,0.916|2010-12-13T21:40:00Z,53.516566,8.188217,None,0.82|2010-12-13T21:50:00Z,53.516566,8.188217,None,0.808|2010-12-13T22:00:00Z,53.516566,8.188217,None,0.841|2010-12-13T22:10:00Z,53.516566,8.188217,None,0.922|2010-12-13T22:20:00Z,53.516566,8.188217,None,0.913|2010-12-13T22:30:00Z,53.516566,8.188217,None,1.041|2010-12-13T22:40:00Z,53.516566,8.188217,None,1.04|2010-12-13T22:50:00Z,53.516566,8.188217,None,0.961|2010-12-13T23:00:00Z,53.516566,8.188217,None,0.891|2010-12-13T23:10:00Z,53.516566,8.188217,None,0.915|2010-12-13T23:20:00Z,53.516566,8.188217,None,0.885|2010-12-13T23:30:00Z,53.516566,8.188217,None,0.938|2010-12-13T23:40:00Z,53.516566,8.188217,None,0.891|2010-12-13T23:50:00Z,53.516566,8.188217,None,0.696|2010-12-14T00:00:00Z,53.516566,8.188217,None,0.65|2010-12-14T00:10:00Z,53.516566,8.188217,None,0.654|2010-12-14T00:20:00Z,53.516566,8.188217,None,0.684|2010-12-14T00:30:00Z,53.516566,8.188217,None,0.71|2010-12-14T00:40:00Z,53.516566,8.188217,None,0.742|2010-12-14T00:50:00Z,53.516566,8.188217,None,0.768|2010-12-14T01:00:00Z,53.516566,8.188217,None,0.827|2010-12-14T01:10:00Z,53.516566,8.188217,None,0.893|2010-12-14T01:20:00Z,53.516566,8.188217,None,0.912|2010-12-14T01:30:00Z,53.516566,8.188217,None,0.942|2010-12-14T01:40:00Z,53.516566,8.188217,None,1.005|2010-12-14T01:50:00Z,53.516566,8.188217,None,1.077|2010-12-14T02:00:00Z,53.516566,8.188217,None,1.13|2010-12-14T02:10:00Z,53.516566,8.188217,None,1.155|2010-12-14T02:20:00Z,53.516566,8.188217,None,1.171|2010-12-14T02:30:00Z,53.516566,8.188217,None,1.184|2010-12-14T02:40:00Z,53.516566,8.188217,None,1.226|2010-12-14T02:50:00Z,53.516566,8.188217,None,1.25|2010-12-14T03:00:00Z,53.516566,8.188217,None,1.28|2010-12-14T03:10:00Z,53.516566,8.188217,None,1.343|2010-12-14T03:20:00Z,53.516566,8.188217,None,1.417|2010-12-14T03:30:00Z,53.516566,8.188217,None,1.448|2010-12-14T03:40:00Z,53.516566,8.188217,None,1.449|2010-12-14T03:50:00Z,53.516566,8.188217,None,1.444|2010-12-14T04:00:00Z,53.516566,8.188217,None,1.426|2010-12-14T04:10:00Z,53.516566,8.188217,None,1.421|2010-12-14T04:20:00Z,53.516566,8.188217,None,1.429|2010-12-14T04:30:00Z,53.516566,8.188217,None,1.414|2010-12-14T04:40:00Z,53.516566,8.188217,None,1.399|2010-12-14T04:50:00Z,53.516566,8.188217,None,1.407|2010-12-14T05:00:00Z,53.516566,8.188217,None,1.423|2010-12-14T05:10:00Z,53.516566,8.188217,None,1.435|2010-12-14T05:20:00Z,53.516566,8.188217,None,1.431|2010-12-14T05:30:00Z,53.516566,8.188217,None,1.402|2010-12-14T05:40:00Z,53.516566,8.188217,None,1.367|2010-12-14T05:50:00Z,53.516566,8.188217,None,1.375|2010-12-14T06:00:00Z,53.516566,8.188217,None,1.364|2010-12-14T06:10:00Z,53.516566,8.188217,None,1.236|2010-12-14T06:20:00Z,53.516566,8.188217,None,1.066|2010-12-14T06:30:00Z,53.516566,8.188217,None,0.811|2010-12-14T06:40:00Z,53.516566,8.188217,None,0.61|2010-12-14T06:50:00Z,53.516566,8.188217,None,0.499|2010-12-14T07:00:00Z,53.516566,8.188217,None,0.369|2010-12-14T07:10:00Z,53.516566,8.188217,None,0.421|2010-12-14T07:20:00Z,53.516566,8.188217,None,0.519|2010-12-14T07:30:00Z,53.516566,8.188217,None,0.621|2010-12-14T07:40:00Z,53.516566,8.188217,None,0.63|2010-12-14T07:50:00Z,53.516566,8.188217,None,0.617|2010-12-14T08:00:00Z,53.516566,8.188217,None,0.626|2010-12-14T08:10:00Z,53.516566,8.188217,None,0.617|2010-12-14T08:20:00Z,53.516566,8.188217,None,0.635|2010-12-14T08:30:00Z,53.516566,8.188217,None,0.672|2010-12-14T08:40:00Z,53.516566,8.188217,None,0.67|2010-12-14T08:50:00Z,53.516566,8.188217,None,0.664|2010-12-14T09:00:00Z,53.516566,8.188217,None,0.785|2010-12-14T09:10:00Z,53.516566,8.188217,None,0.81|2010-12-14T09:20:00Z,53.516566,8.188217,None,0.863|2010-12-14T09:30:00Z,53.516566,8.188217,None,0.843|2010-12-14T09:40:00Z,53.516566,8.188217,None,0.812|2010-12-14T09:50:00Z,53.516566,8.188217,None,0.831|2010-12-14T10:00:00Z,53.516566,8.188217,None,0.843|2010-12-14T10:10:00Z,53.516566,8.188217,None,0.677|2010-12-14T10:20:00Z,53.516566,8.188217,None,0.592|2010-12-14T10:30:00Z,53.516566,8.188217,None,0.683|2010-12-14T10:40:00Z,53.516566,8.188217,None,0.748|2010-12-14T10:50:00Z,53.516566,8.188217,None,0.772|2010-12-14T11:00:00Z,53.516566,8.188217,None,0.758|2010-12-14T11:10:00Z,53.516566,8.188217,None,0.755|2010-12-14T11:20:00Z,53.516566,8.188217,None,0.728|2010-12-14T11:30:00Z,53.516566,8.188217,None,0.663|2010-12-14T11:40:00Z,53.516566,8.188217,None,0.595|2010-12-14T11:50:00Z,53.516566,8.188217,None,0.523|2010-12-14T12:00:00Z,53.516566,8.188217,None,0.505|2010-12-14T12:10:00Z,53.516566,8.188217,None,0.617|2010-12-14T12:20:00Z,53.516566,8.188217,None,0.604|2010-12-14T12:30:00Z,53.516566,8.188217,None,0.564|2010-12-14T12:40:00Z,53.516566,8.188217,None,0.52|2010-12-14T12:50:00Z,53.516566,8.188217,None,0.491|2010-12-14T13:00:00Z,53.516566,8.188217,None,0.456|2010-12-14T13:10:00Z,53.516566,8.188217,None,0.424|2010-12-14T13:20:00Z,53.516566,8.188217,None,0.452|2010-12-14T13:30:00Z,53.516566,8.188217,None,0.504|2010-12-14T13:40:00Z,53.516566,8.188217,None,0.57|2010-12-14T13:50:00Z,53.516566,8.188217,None,0.593|2010-12-14T14:00:00Z,53.516566,8.188217,None,0.642|2010-12-14T14:10:00Z,53.516566,8.188217,None,0.671|2010-12-14T14:20:00Z,53.516566,8.188217,None,0.75|2010-12-14T14:30:00Z,53.516566,8.188217,None,0.729|2010-12-14T14:40:00Z,53.516566,8.188217,None,0.706|2010-12-14T14:50:00Z,53.516566,8.188217,None,0.788|2010-12-14T15:00:00Z,53.516566,8.188217,None,0.824|2010-12-14T15:10:00Z,53.516566,8.188217,None,0.819|2010-12-14T15:20:00Z,53.516566,8.188217,None,0.832|2010-12-14T15:30:00Z,53.516566,8.188217,None,0.861|2010-12-14T15:40:00Z,53.516566,8.188217,None,0.908|2010-12-14T15:50:00Z,53.516566,8.188217,None,0.941|2010-12-14T16:00:00Z,53.516566,8.188217,None,0.983|2010-12-14T16:10:00Z,53.516566,8.188217,None,1.043|2010-12-14T16:20:00Z,53.516566,8.188217,None,1.078|2010-12-14T16:30:00Z,53.516566,8.188217,None,1.097|2010-12-14T16:40:00Z,53.516566,8.188217,None,1.118|2010-12-14T16:50:00Z,53.516566,8.188217,None,1.13|2010-12-14T17:00:00Z,53.516566,8.188217,None,1.144|2010-12-14T17:10:00Z,53.516566,8.188217,None,1.16|2010-12-14T17:20:00Z,53.516566,8.188217,None,1.158|2010-12-14T17:30:00Z,53.516566,8.188217,None,1.15|2010-12-14T17:40:00Z,53.516566,8.188217,None,1.146|2010-12-14T17:50:00Z,53.516566,8.188217,None,1.138|2010-12-14T18:00:00Z,53.516566,8.188217,None,1.116|2010-12-14T18:10:00Z,53.516566,8.188217,None,1.09|2010-12-14T18:20:00Z,53.516566,8.188217,None,1.056|2010-12-14T18:30:00Z,53.516566,8.188217,None,0.997|2010-12-14T18:40:00Z,53.516566,8.188217,None,0.924|2010-12-14T18:50:00Z,53.516566,8.188217,None,0.771|2010-12-14T19:00:00Z,53.516566,8.188217,None,0.654|2010-12-14T19:10:00Z,53.516566,8.188217,None,0.517|2010-12-14T19:20:00Z,53.516566,8.188217,None,0.4|2010-12-14T19:30:00Z,53.516566,8.188217,None,0.335|2010-12-14T19:40:00Z,53.516566,8.188217,None,0.333|2010-12-14T19:50:00Z,53.516566,8.188217,None,0.307|2010-12-14T20:00:00Z,53.516566,8.188217,None,0.233|2010-12-14T20:10:00Z,53.516566,8.188217,None,0.134|2010-12-14T20:20:00Z,53.516566,8.188217,None,0.079|2010-12-14T20:30:00Z,53.516566,8.188217,None,0.066|2010-12-14T20:40:00Z,53.516566,8.188217,None,0.11|2010-12-14T20:50:00Z,53.516566,8.188217,None,0.239|2010-12-14T21:00:00Z,53.516566,8.188217,None,0.179|2010-12-14T21:10:00Z,53.516566,8.188217,None,0.164|2010-12-14T21:20:00Z,53.516566,8.188217,None,0.26|2010-12-14T21:30:00Z,53.516566,8.188217,None,0.3|2010-12-14T21:40:00Z,53.516566,8.188217,None,0.309|2010-12-14T21:50:00Z,53.516566,8.188217,None,0.176|2010-12-14T22:00:00Z,53.516566,8.188217,None,0.211|2010-12-14T22:10:00Z,53.516566,8.188217,None,0.148|2010-12-14T22:20:00Z,53.516566,8.188217,None,0.152|2010-12-14T22:30:00Z,53.516566,8.188217,None,0.075|2010-12-14T22:40:00Z,53.516566,8.188217,None,0.101|2010-12-14T22:50:00Z,53.516566,8.188217,None,0.258|2010-12-14T23:00:00Z,53.516566,8.188217,None,0.354|2010-12-14T23:10:00Z,53.516566,8.188217,None,0.447|2010-12-14T23:20:00Z,53.516566,8.188217,None,0.421|2010-12-14T23:30:00Z,53.516566,8.188217,None,0.234|2010-12-14T23:40:00Z,53.516566,8.188217,None,0.211|2010-12-14T23:50:00Z,53.516566,8.188217,None,0.125|2010-12-15T00:00:00Z,53.516566,8.188217,None,0.099|2010-12-15T00:10:00Z,53.516566,8.188217,None,0.045|2010-12-15T00:20:00Z,53.516566,8.188217,None,0.224|2010-12-15T00:30:00Z,53.516566,8.188217,None,0.361|2010-12-15T00:40:00Z,53.516566,8.188217,None,0.385|2010-12-15T00:50:00Z,53.516566,8.188217,None,0.383|2010-12-15T01:00:00Z,53.516566,8.188217,None,0.315|2010-12-15T01:10:00Z,53.516566,8.188217,None,0.29|2010-12-15T01:20:00Z,53.516566,8.188217,None,0.253|2010-12-15T01:30:00Z,53.516566,8.188217,None,0.242|2010-12-15T01:40:00Z,53.516566,8.188217,None,0.276|2010-12-15T01:50:00Z,53.516566,8.188217,None,0.297|2010-12-15T02:00:00Z,53.516566,8.188217,None,0.371|2010-12-15T02:10:00Z,53.516566,8.188217,None,0.417|2010-12-15T02:20:00Z,53.516566,8.188217,None,0.435|2010-12-15T02:30:00Z,53.516566,8.188217,None,0.36|2010-12-15T02:40:00Z,53.516566,8.188217,None,0.393|2010-12-15T02:50:00Z,53.516566,8.188217,None,0.559|2010-12-15T03:00:00Z,53.516566,8.188217,None,0.543|2010-12-15T03:10:00Z,53.516566,8.188217,None,0.481|2010-12-15T03:20:00Z,53.516566,8.188217,None,0.511|2010-12-15T03:30:00Z,53.516566,8.188217,None,0.633|2010-12-15T03:40:00Z,53.516566,8.188217,None,0.74|2010-12-15T03:50:00Z,53.516566,8.188217,None,0.809|2010-12-15T04:00:00Z,53.516566,8.188217,None,0.852|2010-12-15T04:10:00Z,53.516566,8.188217,None,0.916|2010-12-15T04:20:00Z,53.516566,8.188217,None,1.008|2010-12-15T04:30:00Z,53.516566,8.188217,None,1.087|2010-12-15T04:40:00Z,53.516566,8.188217,None,1.124|2010-12-15T04:50:00Z,53.516566,8.188217,None,1.137|2010-12-15T05:00:00Z,53.516566,8.188217,None,1.151|2010-12-15T05:10:00Z,53.516566,8.188217,None,1.159|2010-12-15T05:20:00Z,53.516566,8.188217,None,1.161|2010-12-15T05:30:00Z,53.516566,8.188217,None,1.161|2010-12-15T05:40:00Z,53.516566,8.188217,None,1.16|2010-12-15T05:50:00Z,53.516566,8.188217,None,1.161|2010-12-15T06:00:00Z,53.516566,8.188217,None,1.165|2010-12-15T06:10:00Z,53.516566,8.188217,None,1.161|2010-12-15T06:20:00Z,53.516566,8.188217,None,1.141|2010-12-15T06:30:00Z,53.516566,8.188217,None,1.131|2010-12-15T06:40:00Z,53.516566,8.188217,None,1.115|2010-12-15T06:50:00Z,53.516566,8.188217,None,1.067|2010-12-15T07:00:00Z,53.516566,8.188217,None,0.949|2010-12-15T07:10:00Z,53.516566,8.188217,None,0.747|2010-12-15T07:20:00Z,53.516566,8.188217,None,0.534|2010-12-15T07:30:00Z,53.516566,8.188217,None,0.406|2010-12-15T07:40:00Z,53.516566,8.188217,None,0.309|2010-12-15T07:50:00Z,53.516566,8.188217,None,0.226|2010-12-15T08:00:00Z,53.516566,8.188217,None,0.078|2010-12-15T08:10:00Z,53.516566,8.188217,None,-0.069|2010-12-15T08:20:00Z,53.516566,8.188217,None,-0.112|2010-12-15T08:30:00Z,53.516566,8.188217,None,-0.167|2010-12-15T08:40:00Z,53.516566,8.188217,None,-0.272|2010-12-15T08:50:00Z,53.516566,8.188217,None,-0.333|2010-12-15T09:00:00Z,53.516566,8.188217,None,-0.249|2010-12-15T09:10:00Z,53.516566,8.188217,None,-0.159|2010-12-15T09:20:00Z,53.516566,8.188217,None,-0.103|2010-12-15T09:30:00Z,53.516566,8.188217,None,-0.079|2010-12-15T09:40:00Z,53.516566,8.188217,None,-0.083|2010-12-15T09:50:00Z,53.516566,8.188217,None,-0.106|2010-12-15T10:00:00Z,53.516566,8.188217,None,-0.1|2010-12-15T10:10:00Z,53.516566,8.188217,None,-0.085|2010-12-15T10:20:00Z,53.516566,8.188217,None,-0.001|2010-12-15T10:30:00Z,53.516566,8.188217,None,0.08|2010-12-15T10:40:00Z,53.516566,8.188217,None,0.157|2010-12-15T10:50:00Z,53.516566,8.188217,None,0.107|2010-12-15T11:00:00Z,53.516566,8.188217,None,-0.085|2010-12-15T11:10:00Z,53.516566,8.188217,None,0.097|2010-12-15T11:20:00Z,53.516566,8.188217,None,0.102|2010-12-15T11:30:00Z,53.516566,8.188217,None,0.135|2010-12-15T11:40:00Z,53.516566,8.188217,None,0.137|2010-12-15T11:50:00Z,53.516566,8.188217,None,0.36|2010-12-15T12:00:00Z,53.516566,8.188217,None,0.321|2010-12-15T12:10:00Z,53.516566,8.188217,None,0.144|2010-12-15T12:20:00Z,53.516566,8.188217,None,-0.008|2010-12-15T12:30:00Z,53.516566,8.188217,None,-0.155|2010-12-15T12:40:00Z,53.516566,8.188217,None,-0.104|2010-12-15T12:50:00Z,53.516566,8.188217,None,0.067|2010-12-15T13:00:00Z,53.516566,8.188217,None,0.142|2010-12-15T13:10:00Z,53.516566,8.188217,None,0.192|2010-12-15T13:20:00Z,53.516566,8.188217,None,0.229|2010-12-15T13:30:00Z,53.516566,8.188217,None,0.216|2010-12-15T13:40:00Z,53.516566,8.188217,None,0.153|2010-12-15T13:50:00Z,53.516566,8.188217,None,0.142|2010-12-15T14:00:00Z,53.516566,8.188217,None,0.065|2010-12-15T14:10:00Z,53.516566,8.188217,None,0.052|2010-12-15T14:20:00Z,53.516566,8.188217,None,0.081|2010-12-15T14:30:00Z,53.516566,8.188217,None,0.115|2010-12-15T14:40:00Z,53.516566,8.188217,None,0.14|2010-12-15T14:50:00Z,53.516566,8.188217,None,0.154|2010-12-15T15:00:00Z,53.516566,8.188217,None,0.194|2010-12-15T15:10:00Z,53.516566,8.188217,None,0.275|2010-12-15T15:20:00Z,53.516566,8.188217,None,0.289|2010-12-15T15:30:00Z,53.516566,8.188217,None,0.181|2010-12-15T15:40:00Z,53.516566,8.188217,None,0.237|2010-12-15T15:50:00Z,53.516566,8.188217,None,0.319|2010-12-15T16:00:00Z,53.516566,8.188217,None,0.411|2010-12-15T16:10:00Z,53.516566,8.188217,None,0.471|2010-12-15T16:20:00Z,53.516566,8.188217,None,0.514|2010-12-15T16:30:00Z,53.516566,8.188217,None,0.525|2010-12-15T16:40:00Z,53.516566,8.188217,None,0.541|2010-12-15T16:50:00Z,53.516566,8.188217,None,0.562|2010-12-15T17:00:00Z,53.516566,8.188217,None,0.567|2010-12-15T17:10:00Z,53.516566,8.188217,None,0.578|2010-12-15T17:20:00Z,53.516566,8.188217,None,0.628|2010-12-15T17:30:00Z,53.516566,8.188217,None,0.662|2010-12-15T17:40:00Z,53.516566,8.188217,None,0.697|2010-12-15T17:50:00Z,53.516566,8.188217,None,0.724|2010-12-15T18:00:00Z,53.516566,8.188217,None,0.745|2010-12-15T18:10:00Z,53.516566,8.188217,None,0.763|2010-12-15T18:20:00Z,53.516566,8.188217,None,0.773|2010-12-15T18:30:00Z,53.516566,8.188217,None,0.782|2010-12-15T18:40:00Z,53.516566,8.188217,None,0.779|2010-12-15T18:50:00Z,53.516566,8.188217,None,0.77|2010-12-15T19:00:00Z,53.516566,8.188217,None,0.724|2010-12-15T19:10:00Z,53.516566,8.188217,None,0.674|2010-12-15T19:20:00Z,53.516566,8.188217,None,0.582|2010-12-15T19:30:00Z,53.516566,8.188217,None,0.457|2010-12-15T19:40:00Z,53.516566,8.188217,None,0.41|2010-12-15T19:50:00Z,53.516566,8.188217,None,0.176|2010-12-15T20:00:00Z,53.516566,8.188217,None,-0.086|2010-12-15T20:10:00Z,53.516566,8.188217,None,-0.253|2010-12-15T20:20:00Z,53.516566,8.188217,None,-0.346|2010-12-15T20:30:00Z,53.516566,8.188217,None,-0.353|2010-12-15T20:40:00Z,53.516566,8.188217,None,-0.327|2010-12-15T20:50:00Z,53.516566,8.188217,None,-0.277|2010-12-15T21:00:00Z,53.516566,8.188217,None,-0.222|2010-12-15T21:10:00Z,53.516566,8.188217,None,-0.224|2010-12-15T21:20:00Z,53.516566,8.188217,None,-0.217|2010-12-15T21:30:00Z,53.516566,8.188217,None,-0.195|2010-12-15T21:40:00Z,53.516566,8.188217,None,-0.222|2010-12-15T21:50:00Z,53.516566,8.188217,None,-0.229|2010-12-15T22:00:00Z,53.516566,8.188217,None,-0.224|2010-12-15T22:10:00Z,53.516566,8.188217,None,-0.217|2010-12-15T22:20:00Z,53.516566,8.188217,None,-0.191|2010-12-15T22:30:00Z,53.516566,8.188217,None,0.095|2010-12-15T22:40:00Z,53.516566,8.188217,None,0.225|2010-12-15T22:50:00Z,53.516566,8.188217,None,0.008|2010-12-15T23:00:00Z,53.516566,8.188217,None,-0.053|2010-12-15T23:10:00Z,53.516566,8.188217,None,0.074|2010-12-15T23:20:00Z,53.516566,8.188217,None,0.12|2010-12-15T23:30:00Z,53.516566,8.188217,None,0.155|2010-12-15T23:40:00Z,53.516566,8.188217,None,0.219|2010-12-15T23:50:00Z,53.516566,8.188217,None,0.177|2010-12-16T00:00:00Z,53.516566,8.188217,None,-0.138|2010-12-16T00:10:00Z,53.516566,8.188217,None,0.087|2010-12-16T00:20:00Z,53.516566,8.188217,None,0.189|2010-12-16T00:30:00Z,53.516566,8.188217,None,0.203|2010-12-16T00:40:00Z,53.516566,8.188217,None,0.246|2010-12-16T00:50:00Z,53.516566,8.188217,None,0.159|2010-12-16T01:00:00Z,53.516566,8.188217,None,-0.032|2010-12-16T01:10:00Z,53.516566,8.188217,None,-0.115|2010-12-16T01:20:00Z,53.516566,8.188217,None,-0.19|2010-12-16T01:30:00Z,53.516566,8.188217,None,-0.315|2010-12-16T01:40:00Z,53.516566,8.188217,None,-0.243|2010-12-16T01:50:00Z,53.516566,8.188217,None,-0.367|2010-12-16T02:00:00Z,53.516566,8.188217,None,-0.363|2010-12-16T02:10:00Z,53.516566,8.188217,None,-0.349|2010-12-16T02:20:00Z,53.516566,8.188217,None,-0.308|2010-12-16T02:30:00Z,53.516566,8.188217,None,-0.309|2010-12-16T02:40:00Z,53.516566,8.188217,None,-0.261|2010-12-16T02:50:00Z,53.516566,8.188217,None,-0.251|2010-12-16T03:00:00Z,53.516566,8.188217,None,-0.221|2010-12-16T03:10:00Z,53.516566,8.188217,None,-0.077|2010-12-16T03:20:00Z,53.516566,8.188217,None,-0.068|2010-12-16T03:30:00Z,53.516566,8.188217,None,-0.037|2010-12-16T03:40:00Z,53.516566,8.188217,None,-0.044|2010-12-16T03:50:00Z,53.516566,8.188217,None,0.007|2010-12-16T04:00:00Z,53.516566,8.188217,None,0.046|2010-12-16T04:10:00Z,53.516566,8.188217,None,0.159|2010-12-16T04:20:00Z,53.516566,8.188217,None,0.286|2010-12-16T04:30:00Z,53.516566,8.188217,None,0.318|2010-12-16T04:40:00Z,53.516566,8.188217,None,0.324|2010-12-16T04:50:00Z,53.516566,8.188217,None,0.355|2010-12-16T05:00:00Z,53.516566,8.188217,None,0.413|2010-12-16T05:10:00Z,53.516566,8.188217,None,0.468|2010-12-16T05:20:00Z,53.516566,8.188217,None,0.507|2010-12-16T05:30:00Z,53.516566,8.188217,None,0.581|2010-12-16T05:40:00Z,53.516566,8.188217,None,0.634|2010-12-16T05:50:00Z,53.516566,8.188217,None,0.679|2010-12-16T06:00:00Z,53.516566,8.188217,None,0.727|2010-12-16T06:10:00Z,53.516566,8.188217,None,0.761|2010-12-16T06:20:00Z,53.516566,8.188217,None,0.807|2010-12-16T06:30:00Z,53.516566,8.188217,None,0.818|2010-12-16T06:40:00Z,53.516566,8.188217,None,0.827|2010-12-16T06:50:00Z,53.516566,8.188217,None,0.857|2010-12-16T07:00:00Z,53.516566,8.188217,None,0.866|2010-12-16T07:10:00Z,53.516566,8.188217,None,0.875|2010-12-16T07:20:00Z,53.516566,8.188217,None,0.873|2010-12-16T07:30:00Z,53.516566,8.188217,None,0.841|2010-12-16T07:40:00Z,53.516566,8.188217,None,0.752|2010-12-16T07:50:00Z,53.516566,8.188217,None,0.689|2010-12-16T08:00:00Z,53.516566,8.188217,None,0.581|2010-12-16T08:10:00Z,53.516566,8.188217,None,0.386|2010-12-16T08:20:00Z,53.516566,8.188217,None,0.127|2010-12-16T08:30:00Z,53.516566,8.188217,None,-0.164|2010-12-16T08:40:00Z,53.516566,8.188217,None,-0.515|2010-12-16T08:50:00Z,53.516566,8.188217,None,-0.839|2010-12-16T09:00:00Z,53.516566,8.188217,None,-0.807|2010-12-16T09:10:00Z,53.516566,8.188217,None,-0.698|2010-12-16T09:20:00Z,53.516566,8.188217,None,-0.523|2010-12-16T09:30:00Z,53.516566,8.188217,None,-0.246|2010-12-16T09:40:00Z,53.516566,8.188217,None,0.063|2010-12-16T09:50:00Z,53.516566,8.188217,None,0.187|2010-12-16T10:00:00Z,53.516566,8.188217,None,0.255|2010-12-16T10:10:00Z,53.516566,8.188217,None,0.278|2010-12-16T10:20:00Z,53.516566,8.188217,None,0.282|2010-12-16T10:30:00Z,53.516566,8.188217,None,0.322|2010-12-16T10:40:00Z,53.516566,8.188217,None,0.333|2010-12-16T10:50:00Z,53.516566,8.188217,None,0.337|2010-12-16T11:00:00Z,53.516566,8.188217,None,0.366|2010-12-16T11:10:00Z,53.516566,8.188217,None,0.374|2010-12-16T11:20:00Z,53.516566,8.188217,None,0.322|2010-12-16T11:30:00Z,53.516566,8.188217,None,0.114|2010-12-16T11:40:00Z,53.516566,8.188217,None,-2.109|2010-12-16T11:50:00Z,53.516566,8.188217,None,-4.013|2010-12-16T12:00:00Z,53.516566,8.188217,None,-3.972|2010-12-16T12:10:00Z,53.516566,8.188217,None,-3.956|2010-12-16T12:20:00Z,53.516566,8.188217,None,-3.999|2010-12-16T12:30:00Z,53.516566,8.188217,None,-4.112|2010-12-16T12:40:00Z,53.516566,8.188217,None,-4.038|2010-12-16T12:50:00Z,53.516566,8.188217,None,-3.972|2010-12-16T13:00:00Z,53.516566,8.188217,None,-4.151|2010-12-16T13:10:00Z,53.516566,8.188217,None,-4.006|2010-12-16T13:20:00Z,53.516566,8.188217,None,-4.035|2010-12-16T13:30:00Z,53.516566,8.188217,None,-4.021|2010-12-16T13:40:00Z,53.516566,8.188217,None,-4.024|2010-12-16T13:50:00Z,53.516566,8.188217,None,-4.035|2010-12-16T14:00:00Z,53.516566,8.188217,None,-3.846|2010-12-16T14:10:00Z,53.516566,8.188217,None,-4.186|2010-12-16T14:20:00Z,53.516566,8.188217,None,-4.133|2010-12-16T14:30:00Z,53.516566,8.188217,None,-4.189|2010-12-16T14:40:00Z,53.516566,8.188217,None,-4.203|2010-12-16T14:50:00Z,53.516566,8.188217,None,-4.125|2010-12-16T15:00:00Z,53.516566,8.188217,None,-4.257|2010-12-16T15:10:00Z,53.516566,8.188217,None,-4.112|2010-12-16T15:20:00Z,53.516566,8.188217,None,-4.129|2010-12-16T15:30:00Z,53.516566,8.188217,None,-4.247|2010-12-16T15:40:00Z,53.516566,8.188217,None,-3.967|2010-12-16T15:50:00Z,53.516566,8.188217,None,-4.122|2010-12-16T16:00:00Z,53.516566,8.188217,None,-4.165|2010-12-16T16:10:00Z,53.516566,8.188217,None,-4.077|2010-12-16T16:20:00Z,53.516566,8.188217,None,-4.089|2010-12-16T16:30:00Z,53.516566,8.188217,None,-4.09|2010-12-16T16:40:00Z,53.516566,8.188217,None,-4.316|2010-12-16T16:50:00Z,53.516566,8.188217,None,-3.897|2010-12-16T17:00:00Z,53.516566,8.188217,None,-4.215|2010-12-16T17:10:00Z,53.516566,8.188217,None,-4.169|2010-12-16T17:20:00Z,53.516566,8.188217,None,-4.198|2010-12-16T17:30:00Z,53.516566,8.188217,None,-4.045|2010-12-16T17:40:00Z,53.516566,8.188217,None,-4.108|2010-12-16T17:50:00Z,53.516566,8.188217,None,-4.158|2010-12-16T18:00:00Z,53.516566,8.188217,None,-4.143|2010-12-16T18:10:00Z,53.516566,8.188217,None,-4.24|2010-12-16T18:20:00Z,53.516566,8.188217,None,-4.135|2010-12-16T18:30:00Z,53.516566,8.188217,None,-4.282|2010-12-16T18:40:00Z,53.516566,8.188217,None,-4.168|2010-12-16T18:50:00Z,53.516566,8.188217,None,-4.137|2010-12-16T19:00:00Z,53.516566,8.188217,None,-4.354|2010-12-16T19:10:00Z,53.516566,8.188217,None,-4.155|2010-12-16T19:20:00Z,53.516566,8.188217,None,-4.202|2010-12-16T19:30:00Z,53.516566,8.188217,None,-2624.0|2010-12-16T19:40:00Z,53.516566,8.188217,None,-5408.0|2010-12-16T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-16T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T11:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T12:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T13:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T14:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T15:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T16:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T17:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T18:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-17T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T11:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T12:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T13:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T14:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T15:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T16:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T17:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T18:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-18T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T11:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T12:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T13:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T14:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T15:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T16:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T17:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T18:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-19T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T11:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T12:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T13:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T14:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T15:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T16:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T17:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T18:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-20T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T11:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T12:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T13:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T14:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T15:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T16:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T17:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T18:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T19:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T20:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T21:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T22:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-21T23:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T00:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T01:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T02:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T03:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T04:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T05:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T06:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T07:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T08:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T09:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:10:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:20:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:30:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:40:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T10:50:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T11:00:00Z,53.516566,8.188217,None,-6999.0|2010-12-22T11:10:00Z,53.516566,8.188217,None,-6999.0</om:result>
#		</om:Observation>

# TODO create parsing function for om:resultDefinition with exchangeability feature!

jade1.watertemp.result <- sosResult(jade1.watertemp)
summary(jade1.watertemp.result)

cat("Demo finished, try another one!\n")