% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msarima.R
\name{msarima}
\alias{msarima}
\title{Multiple Seasonal ARIMA}
\usage{
msarima(y, orders = list(ar = c(0), i = c(1), ma = c(1)), lags = c(1),
  constant = FALSE, AR = NULL, MA = NULL, initial = c("backcasting",
  "optimal"), ic = c("AICc", "AIC", "BIC", "BICc"), loss = c("MSE", "MAE",
  "HAM", "MSEh", "TMSE", "GTMSE", "MSCE"), h = 10, holdout = FALSE,
  cumulative = FALSE, interval = c("none", "parametric", "likelihood",
  "semiparametric", "nonparametric"), level = 0.95, occurrence = c("none",
  "auto", "fixed", "general", "odds-ratio", "inverse-odds-ratio", "direct"),
  oesmodel = "MNN", bounds = c("admissible", "none"), silent = c("all",
  "graph", "legend", "output", "none"), xreg = NULL, xregDo = c("use",
  "select"), initialX = NULL, updateX = FALSE, persistenceX = NULL,
  transitionX = NULL, ...)
}
\arguments{
\item{y}{Vector or ts object, containing data needed to be forecasted.}

\item{orders}{List of orders, containing vector variables \code{ar},
\code{i} and \code{ma}. Example:
\code{orders=list(ar=c(1,2),i=c(1),ma=c(1,1,1))}. If a variable is not
provided in the list, then it is assumed to be equal to zero. At least one
variable should have the same length as \code{lags}. Another option is to
specify orders as a vector of a form \code{orders=c(p,d,q)}. The non-seasonal
ARIMA(p,d,q) is constructed in this case.}

\item{lags}{Defines lags for the corresponding orders (see examples above).
The length of \code{lags} must correspond to the length of either \code{ar},
\code{i} or \code{ma} in \code{orders} variable. There is no restrictions on
the length of \code{lags} vector. It is recommended to order \code{lags}
ascending.
The orders are set by a user. If you want the automatic order selection,
then use \link[smooth]{auto.ssarima} function instead.}

\item{constant}{If \code{TRUE}, constant term is included in the model. Can
also be a number (constant value).}

\item{AR}{Vector or matrix of AR parameters. The order of parameters should
be lag-wise. This means that first all the AR parameters of the firs lag
should be passed, then for the second etc. AR of another ssarima can be
passed here.}

\item{MA}{Vector or matrix of MA parameters. The order of parameters should
be lag-wise. This means that first all the MA parameters of the firs lag
should be passed, then for the second etc. MA of another ssarima can be
passed here.}

\item{initial}{Can be either character or a vector of initial states. If it
is character, then it can be \code{"optimal"}, meaning that the initial
states are optimised, or \code{"backcasting"}, meaning that the initials are
produced using backcasting procedure.}

\item{ic}{The information criterion used in the model selection procedure.}

\item{loss}{The type of Loss Function used in optimization. \code{loss} can
be: \code{MSE} (Mean Squared Error), \code{MAE} (Mean Absolute Error),
\code{HAM} (Half Absolute Moment), \code{TMSE} - Trace Mean Squared Error,
\code{GTMSE} - Geometric Trace Mean Squared Error, \code{MSEh} - optimisation
using only h-steps ahead error, \code{MSCE} - Mean Squared Cumulative Error.
If \code{loss!="MSE"}, then likelihood and model selection is done based
on equivalent \code{MSE}. Model selection in this cases becomes not optimal.

There are also available analytical approximations for multistep functions:
\code{aMSEh}, \code{aTMSE} and \code{aGTMSE}. These can be useful in cases
of small samples.

Finally, just for fun the absolute and half analogues of multistep estimators
are available: \code{MAEh}, \code{TMAE}, \code{GTMAE}, \code{MACE}, \code{TMAE},
\code{HAMh}, \code{THAM}, \code{GTHAM}, \code{CHAM}.}

\item{h}{Length of forecasting horizon.}

\item{holdout}{If \code{TRUE}, holdout sample of size \code{h} is taken from
the end of the data.}

\item{cumulative}{If \code{TRUE}, then the cumulative forecast and prediction
interval are produced instead of the normal ones. This is useful for
inventory control systems.}

\item{interval}{Type of interval to construct. This can be:

\itemize{
\item \code{"none"}, aka \code{"n"} - do not produce prediction
interval.
\item \code{"parametric"}, \code{"p"} - use state-space structure of ETS. In
case of mixed models this is done using simulations, which may take longer
time than for the pure additive and pure multiplicative models. This type
of interval relies on unbiased estimate of in-sample error variance, which
divides the sume of squared errors by T-k rather than just T.
\item \code{"likelihood"}, \code{"l"} - these are the same as \code{"p"}, but
relies on the biased estimate of variance from the likelihood (division by
T, not by T-k).
\item \code{"semiparametric"}, \code{"sp"} - interval based on covariance
matrix of 1 to h steps ahead errors and assumption of normal / log-normal
distribution (depending on error type).
\item \code{"nonparametric"}, \code{"np"} - interval based on values from a
quantile regression on error matrix (see Taylor and Bunn, 1999). The model
used in this process is e[j] = a j^b, where j=1,..,h.
}
The parameter also accepts \code{TRUE} and \code{FALSE}. The former means that
parametric interval are constructed, while the latter is equivalent to
\code{none}.
If the forecasts of the models were combined, then the interval are combined
quantile-wise (Lichtendahl et al., 2013).}

\item{level}{Confidence level. Defines width of prediction interval.}

\item{occurrence}{The type of model used in probability estimation. Can be
\code{"none"} - none,
\code{"fixed"} - constant probability,
\code{"general"} - the general Beta model with two parameters,
\code{"odds-ratio"} - the Odds-ratio model with b=1 in Beta distribution,
\code{"inverse-odds-ratio"} - the model with a=1 in Beta distribution,
\code{"direct"} - the TSB-like (Teunter et al., 2011) probability update
mechanism a+b=1,
\code{"auto"} - the automatically selected type of occurrence model.}

\item{oesmodel}{The type of ETS model used for the modelling of the time varying
probability. Object of the class "oes" can be provided here, and its parameters
would be used in iETS model.}

\item{bounds}{What type of bounds to use in the model estimation. The first
letter can be used instead of the whole word.}

\item{silent}{If \code{silent="none"}, then nothing is silent, everything is
printed out and drawn. \code{silent="all"} means that nothing is produced or
drawn (except for warnings). In case of \code{silent="graph"}, no graph is
produced. If \code{silent="legend"}, then legend of the graph is skipped.
And finally \code{silent="output"} means that nothing is printed out in the
console, but the graph is produced. \code{silent} also accepts \code{TRUE}
and \code{FALSE}. In this case \code{silent=TRUE} is equivalent to
\code{silent="all"}, while \code{silent=FALSE} is equivalent to
\code{silent="none"}. The parameter also accepts first letter of words ("n",
"a", "g", "l", "o").}

\item{xreg}{The vector (either numeric or time series) or the matrix (or
data.frame) of exogenous variables that should be included in the model. If
matrix included than columns should contain variables and rows - observations.
Note that \code{xreg} should have number of observations equal either to
in-sample or to the whole series. If the number of observations in
\code{xreg} is equal to in-sample, then values for the holdout sample are
produced using \link[smooth]{es} function.}

\item{xregDo}{The variable defines what to do with the provided xreg:
\code{"use"} means that all of the data should be used, while
\code{"select"} means that a selection using \code{ic} should be done.
\code{"combine"} will be available at some point in future...}

\item{initialX}{The vector of initial parameters for exogenous variables.
Ignored if \code{xreg} is NULL.}

\item{updateX}{If \code{TRUE}, transition matrix for exogenous variables is
estimated, introducing non-linear interactions between parameters.
Prerequisite - non-NULL \code{xreg}.}

\item{persistenceX}{The persistence vector \eqn{g_X}, containing smoothing
parameters for exogenous variables. If \code{NULL}, then estimated.
Prerequisite - non-NULL \code{xreg}.}

\item{transitionX}{The transition matrix \eqn{F_x} for exogenous variables. Can
be provided as a vector. Matrix will be formed using the default
\code{matrix(transition,nc,nc)}, where \code{nc} is number of components in
state vector. If \code{NULL}, then estimated. Prerequisite - non-NULL
\code{xreg}.}

\item{...}{Other non-documented parameters.

Parameter \code{model} can accept a previously estimated SARIMA model and
use all its parameters.

\code{FI=TRUE} will make the function produce Fisher Information matrix,
which then can be used to calculated variances of parameters of the model.}
}
\value{
Object of class "smooth" is returned. It contains the list of the
following values:

\itemize{
\item \code{model} - the name of the estimated model.
\item \code{timeElapsed} - time elapsed for the construction of the model.
\item \code{states} - the matrix of the fuzzy components of ssarima, where
\code{rows} correspond to time and \code{cols} to states.
\item \code{transition} - matrix F.
\item \code{persistence} - the persistence vector. This is the place, where
smoothing parameters live.
\item \code{measurement} - measurement vector of the model.
\item \code{AR} - the matrix of coefficients of AR terms.
\item \code{I} - the matrix of coefficients of I terms.
\item \code{MA} - the matrix of coefficients of MA terms.
\item \code{constant} - the value of the constant term.
\item \code{initialType} - Type of the initial values used.
\item \code{initial} - the initial values of the state vector (extracted
from \code{states}).
\item \code{nParam} - table with the number of estimated / provided parameters.
If a previous model was reused, then its initials are reused and the number of
provided parameters will take this into account.
\item \code{fitted} - the fitted values.
\item \code{forecast} - the point forecast.
\item \code{lower} - the lower bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{upper} - the higher bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{residuals} - the residuals of the estimated model.
\item \code{errors} - The matrix of 1 to h steps ahead errors.
\item \code{s2} - variance of the residuals (taking degrees of freedom into
account).
\item \code{interval} - type of interval asked by user.
\item \code{level} - confidence level for interval.
\item \code{cumulative} - whether the produced forecast was cumulative or not.
\item \code{y} - the original data.
\item \code{holdout} - the holdout part of the original data.
\item \code{occurrence} - model of the class "oes" if the occurrence model was estimated.
If the model is non-intermittent, then occurrence is \code{NULL}.
\item \code{xreg} - provided vector or matrix of exogenous variables. If
\code{xregDo="s"}, then this value will contain only selected exogenous
variables.
\item \code{updateX} - boolean,
defining, if the states of exogenous variables were estimated as well.
\item \code{initialX} - initial values for parameters of exogenous
variables.
\item \code{persistenceX} - persistence vector g for exogenous variables.
\item \code{transitionX} - transition matrix F for exogenous variables.
\item \code{ICs} - values of information criteria of the model. Includes
AIC, AICc, BIC and BICc.
\item \code{logLik} - log-likelihood of the function.
\item \code{lossValue} - Cost function value.
\item \code{loss} - Type of loss function used in the estimation.
\item \code{FI} - Fisher Information. Equal to NULL if \code{FI=FALSE}
or when \code{FI} is not provided at all.
\item \code{accuracy} - vector of accuracy measures for the holdout sample.
In case of non-intermittent data includes: MPE, MAPE, SMAPE, MASE, sMAE,
RelMAE, sMSE and Bias coefficient (based on complex numbers). In case of
intermittent data the set of errors will be: sMSE, sPIS, sCE (scaled
cumulative error) and Bias coefficient. This is available only when
\code{holdout=TRUE}.
\item \code{B} - the vector of all the estimated parameters.
}
}
\description{
Function constructs Multiple Seasonal State Space ARIMA, estimating AR, MA
terms and initial states.
}
\details{
The model, implemented in this function differs from the one in
\link[smooth]{ssarima} function (Svetunkov & Boylan, 2019), but it is more
efficient and better fitting the data (which might be a limitation).

The basic ARIMA(p,d,q) used in the function has the following form:

\eqn{(1 - B)^d (1 - a_1 B - a_2 B^2 - ... - a_p B^p) y_[t] = (1 + b_1 B +
b_2 B^2 + ... + b_q B^q) \epsilon_[t] + c}

where \eqn{y_[t]} is the actual values, \eqn{\epsilon_[t]} is the error term,
\eqn{a_i, b_j} are the parameters for AR and MA respectively and \eqn{c} is
the constant. In case of non-zero differences \eqn{c} acts as drift.

This model is then transformed into ARIMA in the Single Source of Error
State space form (based by Snyder, 1985, but in a slightly different
formulation):

\eqn{y_{t} = o_{t} (w' v_{t-l} + x_t a_{t-1} + \epsilon_{t})}

\eqn{v_{t} = F v_{t-l} + g \epsilon_{t}}

\eqn{a_{t} = F_{X} a_{t-1} + g_{X} \epsilon_{t} / x_{t}}

Where \eqn{o_{t}} is the Bernoulli distributed random variable (in case of
normal data equal to 1), \eqn{v_{t}} is the state vector (defined based on
\code{orders}) and \eqn{l} is the vector of \code{lags}, \eqn{x_t} is the
vector of exogenous parameters. \eqn{w} is the \code{measurement} vector,
\eqn{F} is the \code{transition} matrix, \eqn{g} is the \code{persistence}
vector, \eqn{a_t} is the vector of parameters for exogenous variables,
\eqn{F_{X}} is the \code{transitionX} matrix and \eqn{g_{X}} is the
\code{persistenceX} matrix. The main difference from \link[smooth]{ssarima}
function is that this implementation skips zero polynomials, substantially
decreasing the dimension of the transition matrix. As a result, this
function works faster than \link[smooth]{ssarima} on high frequency data,
and it is more accurate.

Due to the flexibility of the model, multiple seasonalities can be used. For
example, something crazy like this can be constructed:
SARIMA(1,1,1)(0,1,1)[24](2,0,1)[24*7](0,0,1)[24*30], but the estimation may
take some time... Still this should be estimated in finite time (not like
with \code{ssarima}).

For some additional details see the vignette: \code{vignette("ssarima","smooth")}
}
\examples{

# The previous one is equivalent to:
ourModel <- msarima(rnorm(118,100,3),orders=c(1,1,1),lags=1,h=18,holdout=TRUE,interval="p")

# Example of SARIMA(2,0,0)(1,0,0)[4]
msarima(rnorm(118,100,3),orders=list(ar=c(2,1)),lags=c(1,4),h=18,holdout=TRUE)

# SARIMA of a peculiar order on AirPassengers data with Fisher Information
ourModel <- msarima(AirPassengers,orders=list(ar=c(1,0,3),i=c(1,0,1),ma=c(0,1,2)),
                    lags=c(1,6,12),h=10,holdout=TRUE,FI=TRUE)

# Construct the 95\% confidence interval for the parameters of the model
ourCoefs <- coef(ourModel)
ourCoefsSD <- sqrt(abs(diag(solve(ourModel$FI))))
# Sort values accordingly
ourCoefs <- ourCoefs[names(ourCoefsSD)]
ourConfInt <- cbind(ourCoefs + qt(0.025,nobs(ourModel)) * ourCoefsSD,
                    ourCoefs + qt(0.975,nobs(ourModel)) * ourCoefsSD)
colnames(ourConfInt) <- c("2.25\%","97.5\%")
ourConfInt

# ARIMA(1,1,1) with Mean Squared Trace Forecast Error
msarima(rnorm(118,100,3),orders=list(ar=1,i=1,ma=1),lags=1,h=18,holdout=TRUE,loss="TMSE")

msarima(rnorm(118,100,3),orders=list(ar=1,i=1,ma=1),lags=1,h=18,holdout=TRUE,loss="aTMSE")

# SARIMA(0,1,1) with exogenous variables with crazy estimation of xreg
ourModel <- msarima(rnorm(118,100,3),orders=list(i=1,ma=1),h=18,holdout=TRUE,
                    xreg=c(1:118),updateX=TRUE)

summary(ourModel)
forecast(ourModel)
plot(forecast(ourModel))

}
\references{
\itemize{
\item Taylor, J.W. and Bunn, D.W. (1999) A Quantile Regression Approach to
Generating Prediction Intervals. Management Science, Vol 45, No 2, pp
225-237.
\item Lichtendahl Kenneth C., Jr., Grushka-Cockayne Yael, Winkler
Robert L., (2013) Is It Better to Average Probabilities or
Quantiles? Management Science 59(7):1594-1611. DOI:
[10.1287/mnsc.1120.1667](https://doi.org/10.1287/mnsc.1120.1667)
}

\itemize{
\item Snyder, R. D., 1985. Recursive Estimation of Dynamic Linear Models.
Journal of the Royal Statistical Society, Series B (Methodological) 47 (2), 272-276.
\item Hyndman, R.J., Koehler, A.B., Ord, J.K., and Snyder, R.D. (2008)
Forecasting with exponential smoothing: the state space approach,
Springer-Verlag. \url{http://dx.doi.org/10.1007/978-3-540-71918-2}.
}

\itemize{
\item Svetunkov, I., & Boylan, J. E. (2019). State-space ARIMA for supply-chain forecasting.
International Journal of Production Research, 0(0), 1–10.
\url{https://doi.org/10.1080/00207543.2019.1600764}
}
}
\seealso{
\code{\link[smooth]{auto.msarima}, \link[smooth]{orders},
\link[smooth]{ssarima}, \link[forecast]{auto.arima}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{smooth}
\keyword{ts}
\keyword{univar}
