% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ces.R
\name{ces}
\alias{ces}
\title{Complex Exponential Smoothing}
\usage{
ces(y, seasonality = c("none", "simple", "partial", "full"),
  initial = c("backcasting", "optimal"), A = NULL, B = NULL,
  ic = c("AICc", "AIC", "BIC", "BICc"), loss = c("MSE", "MAE", "HAM",
  "MSEh", "TMSE", "GTMSE", "MSCE"), h = 10, holdout = FALSE,
  cumulative = FALSE, interval = c("none", "parametric",
  "semiparametric", "nonparametric"), level = 0.95,
  occurrence = c("none", "auto", "fixed", "general", "odds-ratio",
  "inverse-odds-ratio", "direct"), oesmodel = "MNN",
  bounds = c("admissible", "none"), silent = c("all", "graph",
  "legend", "output", "none"), xreg = NULL, xregDo = c("use",
  "select"), initialX = NULL, updateX = FALSE, persistenceX = NULL,
  transitionX = NULL, ...)
}
\arguments{
\item{y}{Vector or ts object, containing data needed to be forecasted.}

\item{seasonality}{The type of seasonality used in CES. Can be: \code{none}
- No seasonality; \code{simple} - Simple seasonality, using lagged CES
(based on \code{t-m} observation, where \code{m} is the seasonality lag);
\code{partial} - Partial seasonality with real seasonal components
(equivalent to additive seasonality); \code{full} - Full seasonality with
complex seasonal components (can do both multiplicative and additive
seasonality, depending on the data). First letter can be used instead of
full words.  Any seasonal CES can only be constructed for time series
vectors.}

\item{initial}{Can be either character or a vector of initial states. If it
is character, then it can be \code{"optimal"}, meaning that the initial
states are optimised, or \code{"backcasting"}, meaning that the initials are
produced using backcasting procedure.}

\item{A}{First complex smoothing parameter. Should be a complex number.

NOTE! CES is very sensitive to A and B values so it is advised either to
leave them alone, or to use values from previously estimated model.}

\item{B}{Second complex smoothing parameter. Can be real if
\code{seasonality="partial"}. In case of \code{seasonality="full"} must be
complex number.}

\item{ic}{The information criterion used in the model selection procedure.}

\item{loss}{The type of Loss Function used in optimization. \code{loss} can
be: \code{MSE} (Mean Squared Error), \code{MAE} (Mean Absolute Error),
\code{HAM} (Half Absolute Moment), \code{TMSE} - Trace Mean Squared Error,
\code{GTMSE} - Geometric Trace Mean Squared Error, \code{MSEh} - optimisation
using only h-steps ahead error, \code{MSCE} - Mean Squared Cumulative Error.
If \code{loss!="MSE"}, then likelihood and model selection is done based
on equivalent \code{MSE}. Model selection in this cases becomes not optimal.

There are also available analytical approximations for multistep functions:
\code{aMSEh}, \code{aTMSE} and \code{aGTMSE}. These can be useful in cases
of small samples.

Finally, just for fun the absolute and half analogues of multistep estimators
are available: \code{MAEh}, \code{TMAE}, \code{GTMAE}, \code{MACE}, \code{TMAE},
\code{HAMh}, \code{THAM}, \code{GTHAM}, \code{CHAM}.}

\item{h}{Length of forecasting horizon.}

\item{holdout}{If \code{TRUE}, holdout sample of size \code{h} is taken from
the end of the data.}

\item{cumulative}{If \code{TRUE}, then the cumulative forecast and prediction
interval are produced instead of the normal ones. This is useful for
inventory control systems.}

\item{interval}{Type of interval to construct. This can be:

\itemize{
\item \code{none}, aka \code{n} - do not produce prediction
interval.
\item \code{parametric}, \code{p} - use state-space structure of ETS. In
case of mixed models this is done using simulations, which may take longer
time than for the pure additive and pure multiplicative models.
\item \code{semiparametric}, \code{sp} - interval based on covariance
matrix of 1 to h steps ahead errors and assumption of normal / log-normal
distribution (depending on error type).
\item \code{nonparametric}, \code{np} - interval based on values from a
quantile regression on error matrix (see Taylor and Bunn, 1999). The model
used in this process is e[j] = a j^b, where j=1,..,h.
}
The parameter also accepts \code{TRUE} and \code{FALSE}. The former means that
parametric interval are constructed, while the latter is equivalent to
\code{none}.
If the forecasts of the models were combined, then the interval are combined
quantile-wise (Lichtendahl et al., 2013).}

\item{level}{Confidence level. Defines width of prediction interval.}

\item{occurrence}{The type of model used in probability estimation. Can be
\code{"none"} - none,
\code{"fixed"} - constant probability,
\code{"general"} - the general Beta model with two parameters,
\code{"odds-ratio"} - the Odds-ratio model with b=1 in Beta distribution,
\code{"inverse-odds-ratio"} - the model with a=1 in Beta distribution,
\code{"direct"} - the TSB-like (Teunter et al., 2011) probability update
mechanism a+b=1,
\code{"auto"} - the automatically selected type of occurrence model.}

\item{oesmodel}{The type of ETS model used for the modelling of the time varying
probability. Object of the class "oes" can be provided here, and its parameters
would be used in iETS model.}

\item{bounds}{What type of bounds to use in the model estimation. The first
letter can be used instead of the whole word.}

\item{silent}{If \code{silent="none"}, then nothing is silent, everything is
printed out and drawn. \code{silent="all"} means that nothing is produced or
drawn (except for warnings). In case of \code{silent="graph"}, no graph is
produced. If \code{silent="legend"}, then legend of the graph is skipped.
And finally \code{silent="output"} means that nothing is printed out in the
console, but the graph is produced. \code{silent} also accepts \code{TRUE}
and \code{FALSE}. In this case \code{silent=TRUE} is equivalent to
\code{silent="all"}, while \code{silent=FALSE} is equivalent to
\code{silent="none"}. The parameter also accepts first letter of words ("n",
"a", "g", "l", "o").}

\item{xreg}{The vector (either numeric or time series) or the matrix (or
data.frame) of exogenous variables that should be included in the model. If
matrix included than columns should contain variables and rows - observations.
Note that \code{xreg} should have number of observations equal either to
in-sample or to the whole series. If the number of observations in
\code{xreg} is equal to in-sample, then values for the holdout sample are
produced using \link[smooth]{es} function.}

\item{xregDo}{The variable defines what to do with the provided xreg:
\code{"use"} means that all of the data should be used, while
\code{"select"} means that a selection using \code{ic} should be done.
\code{"combine"} will be available at some point in future...}

\item{initialX}{The vector of initial parameters for exogenous variables.
Ignored if \code{xreg} is NULL.}

\item{updateX}{If \code{TRUE}, transition matrix for exogenous variables is
estimated, introducing non-linear interactions between parameters.
Prerequisite - non-NULL \code{xreg}.}

\item{persistenceX}{The persistence vector \eqn{g_X}, containing smoothing
parameters for exogenous variables. If \code{NULL}, then estimated.
Prerequisite - non-NULL \code{xreg}.}

\item{transitionX}{The transition matrix \eqn{F_x} for exogenous variables. Can
be provided as a vector. Matrix will be formed using the default
\code{matrix(transition,nc,nc)}, where \code{nc} is number of components in
state vector. If \code{NULL}, then estimated. Prerequisite - non-NULL
\code{xreg}.}

\item{...}{Other non-documented parameters.  For example parameter
\code{model} can accept a previously estimated CES model and use all its
parameters.  \code{FI=TRUE} will make the function produce Fisher
Information matrix, which then can be used to calculated variances of
parameters of the model.}
}
\value{
Object of class "smooth" is returned. It contains the list of the
following values: \itemize{
\item \code{model} - type of constructed model.
\item \code{timeElapsed} - time elapsed for the construction of the model.
\item \code{states} - the matrix of the components of CES. The included
minimum is "level" and "potential". In the case of seasonal model the
seasonal component is also included. In the case of exogenous variables the
estimated coefficients for the exogenous variables are also included.
\item \code{A} - complex smoothing parameter in the form a0 + ia1
\item \code{B} - smoothing parameter for the seasonal component. Can either
be real (if \code{seasonality="P"}) or complex (if \code{seasonality="F"})
in a form b0 + ib1.
\item \code{persistence} - persistence vector. This is the place, where
smoothing parameters live.
\item \code{transition} - transition matrix of the model.
\item \code{measurement} - measurement vector of the model.
\item \code{initialType} - Type of the initial values used.
\item \code{initial} - the initial values of the state vector (non-seasonal).
\item \code{nParam} - table with the number of estimated / provided parameters.
If a previous model was reused, then its initials are reused and the number of
provided parameters will take this into account.
\item \code{fitted} - the fitted values of CES.
\item \code{forecast} - the point forecast of CES.
\item \code{lower} - the lower bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{upper} - the upper bound of prediction interval. When
\code{interval="none"} then NA is returned.
\item \code{residuals} - the residuals of the estimated model.
\item \code{errors} - The matrix of 1 to h steps ahead errors.
\item \code{s2} - variance of the residuals (taking degrees of
freedom into account).
\item \code{interval} - type of interval asked by user.
\item \code{level} - confidence level for interval.
\item \code{cumulative} - whether the produced forecast was cumulative or not.
\item \code{y} - The data provided in the call of the function.
\item \code{holdout} - the holdout part of the original data.
\item \code{occurrence} - model of the class "oes" if the occurrence model was estimated.
If the model is non-intermittent, then occurrence is \code{NULL}.
\item \code{xreg} - provided vector or matrix of exogenous variables. If
\code{xregDo="s"}, then this value will contain only selected exogenous
variables.
\item \code{updateX} - boolean, defining, if the states of
exogenous variables were estimated as well.
\item \code{initialX} - initial values for parameters of exogenous variables.
\item \code{persistenceX} - persistence vector g for exogenous variables.
\item \code{transitionX} - transition matrix F for exogenous variables.
\item \code{ICs} - values of information criteria of the model. Includes
AIC, AICc, BIC and BICc.
\item \code{logLik} - log-likelihood of the function.
\item \code{lossValue} - Cost function value.
\item \code{loss} - Type of loss function used in the estimation.
\item \code{FI} - Fisher Information. Equal to NULL if \code{FI=FALSE}
or when \code{FI} is not provided at all.
\item \code{accuracy} - vector of accuracy measures for the holdout sample. In
case of non-intermittent data includes: MPE, MAPE, SMAPE, MASE, sMAE,
RelMAE, sMSE and Bias coefficient (based on complex numbers). In case of
intermittent data the set of errors will be: sMSE, sPIS, sCE (scaled
cumulative error) and Bias coefficient. This is available only when
\code{holdout=TRUE}.
}
}
\description{
Function estimates CES in state space form with information potential equal
to errors and returns several variables.
}
\details{
The function estimates Complex Exponential Smoothing in the state space 2
described in Svetunkov, Kourentzes (2017) with the information potential
equal to the approximation error.  The estimation of initial states of xt is
done using backcast.

For some more information about the model and its implementation, see the
vignette: \code{vignette("ces","smooth")}
}
\examples{

y <- rnorm(100,10,3)
ces(y,h=20,holdout=TRUE)
ces(y,h=20,holdout=FALSE)

y <- 500 - c(1:100)*0.5 + rnorm(100,10,3)
ces(y,h=20,holdout=TRUE,interval="p",bounds="a")

library("Mcomp")
y <- ts(c(M3$N0740$x,M3$N0740$xx),start=start(M3$N0740$x),frequency=frequency(M3$N0740$x))
ces(y,h=8,holdout=TRUE,seasonality="s",interval="sp",level=0.8)

\dontrun{y <- ts(c(M3$N1683$x,M3$N1683$xx),start=start(M3$N1683$x),frequency=frequency(M3$N1683$x))
ces(y,h=18,holdout=TRUE,seasonality="s",interval="sp")
ces(y,h=18,holdout=TRUE,seasonality="p",interval="np")
ces(y,h=18,holdout=TRUE,seasonality="f",interval="p")}

\dontrun{x <- cbind(c(rep(0,25),1,rep(0,43)),c(rep(0,10),1,rep(0,58)))
ces(ts(c(M3$N1457$x,M3$N1457$xx),frequency=12),h=18,holdout=TRUE,
    interval="np",xreg=x,loss="TMSE")}

# Exogenous variables in CES
\dontrun{x <- cbind(c(rep(0,25),1,rep(0,43)),c(rep(0,10),1,rep(0,58)))
ces(ts(c(M3$N1457$x,M3$N1457$xx),frequency=12),h=18,holdout=TRUE,xreg=x)
ourModel <- ces(ts(c(M3$N1457$x,M3$N1457$xx),frequency=12),h=18,holdout=TRUE,xreg=x,updateX=TRUE)
# This will be the same model as in previous line but estimated on new portion of data
ces(ts(c(M3$N1457$x,M3$N1457$xx),frequency=12),model=ourModel,h=18,holdout=FALSE)}

# Intermittent data example
x <- rpois(100,0.2)
# Best type of the occurrence model based on iETS(Z,Z,N)
ourModel <- ces(x,occurrence="auto")

summary(ourModel)
forecast(ourModel)
plot(forecast(ourModel))

}
\references{
\itemize{
\item Svetunkov, I., Kourentzes, N. (February 2015). Complex exponential
smoothing. Working Paper of Department of Management Science, Lancaster
University 2015:1, 1-31.
\item Svetunkov I., Kourentzes N. (2017) Complex Exponential Smoothing
for Time Series Forecasting. Not yet published.
}
}
\seealso{
\code{\link[forecast]{ets}, \link[forecast]{forecast},
\link[stats]{ts}, \link[smooth]{auto.ces}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{smooth}
\keyword{ts}
\keyword{univar}
