% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smmnonparametric.R
\name{smmnonparametric}
\alias{smmnonparametric}
\title{Non-parametric semi-Markov model specification}
\usage{
smmnonparametric(
  states,
  init,
  ptrans,
  type.sojourn = c("fij", "fi", "fj", "f"),
  distr,
  cens.beg = FALSE,
  cens.end = FALSE
)
}
\arguments{
\item{states}{Vector of state space of length \eqn{s}.}

\item{init}{Vector of initial distribution of length \eqn{s}.}

\item{ptrans}{Matrix of transition probabilities of the embedded Markov
chain \eqn{J=(J_m)_{m}} of dimension \eqn{(s, s)}.}

\item{type.sojourn}{Type of sojourn time (for more explanations, see Details).}

\item{distr}{\itemize{
\item Array of dimension \eqn{(s, s, kmax)} if \code{type.sojourn = "fij"};
\item Matrix of dimension \eqn{(s, kmax)} if \code{type.sojourn = "fi"} or \code{"fj"};
\item Vector of length \eqn{kmax} if the \code{type.sojourn = "f"}.
}
\eqn{kmax} is the maximum length of the sojourn times.}

\item{cens.beg}{Optional. A logical value indicating whether or not
sequences are censored at the beginning.}

\item{cens.end}{Optional. A logical value indicating whether or not
sequences are censored at the end.}
}
\value{
Returns an object of class \code{smm}, \link{smmnonparametric}.
}
\description{
Creates a non-parametric model specification for a semi-Markov model.
}
\details{
This function creates a semi-Markov model object in the
non-parametric case, taking into account the type of sojourn time and the
censoring described in references. The non-parametric specification concerns
sojourn time distributions defined by the user.

The difference between the Markov model and the semi-Markov model concerns
the modeling of the sojourn time. With a Markov chain, the sojourn time
distribution is modeled by a Geometric distribution (in discrete time).
With a semi-Markov chain, the sojourn time can be any arbitrary distribution.

We define :
\itemize{
\item the semi-Markov kernel \eqn{q_{ij}(k) = P( J_{m+1} = j, T_{m+1} - T_{m} = k | J_{m} = i )};
\item the transition matrix \eqn{(p_{trans}(i,j))_{i,j} \in states} of the embedded Markov chain \eqn{J = (J_m)_m}, \eqn{p_{trans}(i,j) = P( J_{m+1} = j | J_m = i )};
\item the initial distribution \eqn{\mu_i = P(J_1 = i) = P(Z_1 = i)}, \eqn{i \in 1, 2, \dots, s};
\item the conditional sojourn time distributions \eqn{(f_{ij}(k))_{i,j} \in states,\ k \in N ,\ f_{ij}(k) = P(T_{m+1} - T_m = k | J_m = i, J_{m+1} = j )},
f is specified by the argument \code{distr} in the non-parametric case.
}

In this package we can choose different types of sojourn time.
Four options are available for the sojourn times:
\itemize{
\item depending on the present state and on the next state (\eqn{f_{ij}});
\item depending only on the present state (\eqn{f_{i}});
\item depending only on the next state (\eqn{f_{j}});
\item depending neither on the current, nor on the next state (\eqn{f}).
}

Let define \eqn{kmax} the maximum length of the sojourn times.
If  \code{type.sojourn = "fij"}, \code{distr} is an array of dimension \eqn{(s, s, kmax)}.
If \code{type.sojourn = "fi"} or \code{"fj"}, \code{distr} must be a matrix of dimension \eqn{(s, kmax)}.
If \code{type.sojourn = "f"}, \code{distr} must be a vector of length \eqn{kmax}.

If the sequence is censored at the beginning and/or at the end, \code{cens.beg}
must be equal to \code{TRUE} and/or \code{cens.end} must be equal to \code{TRUE}.
All the sequences must be censored in the same way.
}
\examples{
states <- c("a", "c", "g", "t")
s <- length(states)

# Creation of the initial distribution
vect.init <- c(1 / 4, 1 / 4, 1 / 4, 1 / 4)

# Creation of transition matrix
pij <- matrix(c(0, 0.2, 0.5, 0.3, 
                0.2, 0, 0.3, 0.5, 
                0.3, 0.5, 0, 0.2, 
                0.4, 0.2, 0.4, 0), 
              ncol = s, byrow = TRUE)

# Creation of a matrix corresponding to the 
# conditional sojourn time distributions
kmax <- 6
nparam.matrix <- matrix(c(0.2, 0.1, 0.3, 0.2, 
                          0.2, 0, 0.4, 0.2, 
                          0.1, 0, 0.2, 0.1, 
                          0.5, 0.3, 0.15, 0.05, 
                          0, 0, 0.3, 0.2, 
                          0.1, 0.2, 0.2, 0), 
                        nrow = s, ncol = kmax, byrow = TRUE)

smm2 <- smmnonparametric(states = states, init = vect.init, ptrans = pij, 
                         type.sojourn = "fj", distr = nparam.matrix)

smm2
}
\references{
V. S. Barbu, N. Limnios. (2008). Semi-Markov Chains and Hidden Semi-Markov
Models Toward Applications - Their Use in Reliability and DNA Analysis.
New York: Lecture Notes in Statistics, vol. 191, Springer.
}
\seealso{
\link{simulate}, \link{fitsmm}, \link{smmparametric}
}
