% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree-sequences.R
\name{ts_load}
\alias{ts_load}
\title{Load a tree sequence file produced by a given model}
\usage{
ts_load(file, model = NULL)
}
\arguments{
\item{file}{A path to the tree-sequence file (either originating from a
slendr model or a standard non-slendr tree sequence).}

\item{model}{Optional \code{slendr_model} object which produced the
tree-sequence \code{file}. Used for adding various annotation data and
metadata to the standard tskit tree-sequence object.}
}
\value{
Tree-sequence object of the class \code{slendr_ts}, which serves as
an interface point for the Python module tskit using slendr functions with
the \code{ts_} prefix.
}
\description{
This function loads a tree sequence file simulated from a given slendr model.
Optionally, the tree sequence can be recapitated and simplified.
}
\details{
The loading, recapitation and simplification is performed using the Python
module pyslim which serves as a link between tree sequences generated by SLiM
and the tskit module for manipulation of tree sequence data. All of these
steps have been modelled after the official pyslim tutorial and documentation
available at: \url{https://tskit.dev/pyslim/docs/latest/tutorial.html}.

The recapitation and simplification steps can also be performed individually
using the functions \code{\link{ts_recapitate}} and
\code{\link{ts_simplify}}.
}
\examples{
\dontshow{check_dependencies(python = TRUE, quit = TRUE) # dependencies must be present
}
init_env()

# load an example model with an already simulated tree sequence
slendr_ts <- system.file("extdata/models/introgression.trees", package = "slendr")
model <- read_model(path = system.file("extdata/models/introgression", package = "slendr"))

# load tree sequence generated by a given model
ts <- ts_load(slendr_ts, model)

# even tree sequences generated by non-slendr models can be
msprime_ts <- system.file("extdata/models/msprime.trees", package = "slendr")
ts <- ts_load(msprime_ts)

# load tree sequence and immediately simplify it only to sampled individuals
# (note that the example tree sequence is already simplified so this operation
# does not do anything in this case)
ts <- ts_load(slendr_ts, model = model) \%>\% ts_simplify(keep_input_roots = TRUE)

# load tree sequence and simplify it to a subset of sampled individuals
ts_small <- ts_simplify(ts, simplify_to = c("CH_1", "NEA_1", "NEA_2",
                                            "AFR_1", "AFR_2", "EUR_1", "EUR_2"))

# load tree sequence, recapitate it and simplify it
ts <- ts_load(slendr_ts, model) \%>\%
  ts_recapitate(recombination_rate = 1e-8, Ne = 10000, random_seed = 42) \%>\%
  ts_simplify()

# load tree sequence, recapitate it, simplify it and overlay neutral mutations
ts <- ts_load(slendr_ts, model) \%>\%
  ts_recapitate(recombination_rate = 1e-8, Ne = 10000, random_seed = 42) \%>\%
  ts_simplify() \%>\%
  ts_mutate(mutation_rate = 1e-8)

ts
}
\seealso{
\code{\link{ts_nodes}} for extracting useful information about
individuals, nodes, coalescent times and geospatial locations of nodes on a
map
}
