#' @title Intraclass-Correlation Coefficient
#' @name icc
#' @description This function calculates the intraclass-correlation
#'    (icc) - sometimes also called \emph{variance partition coefficient}
#'    (vpc) - for random intercepts of mixed effects models. Currently,
#'    \code{\link[lme4]{merMod}}, \code{\link[glmmTMB]{glmmTMB}},
#'    \code{stanreg} and \code{\link[brms]{brmsfit}} objects are supported.
#'
#' @param x Fitted mixed effects model (of class \code{merMod}, \code{glmmTMB},
#'    \code{stanreg} or \code{brmsfit}).
#' @param ... Currently not used.
#' @param re.form Formula containing group-level effects to be considered in
#'   the prediction. If \code{NULL} (default), include all group-level effects.
#'   Else, for instance for nested models, name a specific group-level effect
#'   to calculate the ICC for this group-level. Only applies if \code{ppd = TRUE}.
#' @param typical Character vector, naming the function that will be used as
#'   measure of central tendency for the ICC. The default is "mean". See
#'   \code{typical_value} for options.
#' @param ppd Logical, if \code{TRUE}, variance decomposition is based on the
#'   posterior predictive distribution, which is the correct way for Bayesian
#'   non-Gaussian models.
#'
#' @inheritParams hdi
#'
#' @return A numeric vector with all random intercept intraclass-correlation-coefficients.
#'    Furthermore, between- and within-group variances as well as random-slope
#'    variance are returned as attributes.
#'    \cr \cr
#'    For \code{stanreg} or \code{brmsfit} objects, the HDI for each statistic
#'    is also  included as attribute.
#'
#' @references \itemize{
#'               \item Aguinis H, Gottfredson RK, Culpepper SA. 2013. Best-Practice Recommendations for Estimating Cross-Level Interaction Effects Using Multilevel Modeling. Journal of Management 39(6): 1490–1528 (\doi{10.1177/0149206313478188})
#'               \item Aly SS, Zhao J, Li B, Jiang J. 2014. Reliability of environmental sampling culture results using the negative binomial intraclass correlation coefficient. Springerplus 14(3) (\doi{10.1186/2193-1801-3-40})
#'               \item Goldstein H, Browne W, Rasbash J. 2010. Partitioning Variation in Multilevel Models. Understanding Statistics, 1:4, 223-231 (\doi{10.1207/S15328031US0104_02})
#'               \item Grace-Martion K. The Intraclass Correlation Coefficient in Mixed Models, \href{http://www.theanalysisfactor.com/the-intraclass-correlation-coefficient-in-mixed-models/}{web}
#'               \item Hox J. 2002. Multilevel analysis: techniques and applications. Mahwah, NJ: Erlbaum
#'               \item Rabe-Hesketh S, Skrondal A. 2012. Multilevel and longitudinal modeling using Stata. 3rd ed. College Station, Tex: Stata Press Publication
#'               \item Raudenbush SW, Bryk AS. 2002. Hierarchical linear models: applications and data analysis methods. 2nd ed. Thousand Oaks: Sage Publications
#'               \item Stryhn H, Sanchez J, Morley P, Booker C, Dohoo IR. 2006. Interpretation of variance parameters in multilevel Poisson regression models. Proceedings of the 11th International Symposium on Veterinary Epidemiology and Economics, 2006 Available at \url{http://www.sciquest.org.nz/node/64294}
#'               \item Wu S, Crespi CM, Wong WK. 2012. Comparison of methods for estimating the intraclass correlation coefficient for binary responses in cancer prevention cluster randomized trials. Contempory Clinical Trials 33: 869-880 (\doi{10.1016/j.cct.2012.05.004})
#'             }
#'             Further helpful online-ressources:
#'             \itemize{
#'               \item \href{http://stats.stackexchange.com/questions/18088/intraclass-correlation-icc-for-an-interaction/28100#28100}{CrossValidated (2012) \emph{Intraclass correlation (ICC) for an interaction?}}
#'               \item \href{http://stats.stackexchange.com/questions/113577/interpreting-the-random-effect-in-a-mixed-effect-model/113825#113825}{CrossValidated (2014) \emph{Interpreting the random effect in a mixed-effect model}}
#'               \item \href{http://stats.stackexchange.com/questions/67247/how-to-partition-the-variance-explained-at-group-level-and-individual-level/67356#67356}{CrossValidated (2014) \emph{how to partition the variance explained at group level and individual level}}
#'             }
#'
#'
#' @note Some notes on why the ICC is useful, based on \cite{Grace-Martin}:
#'       \itemize{
#'        \item It can help you determine whether or not a linear mixed model is even necessary. If you find that the correlation is zero, that means the observations within clusters are no more similar than observations from different clusters. Go ahead and use a simpler analysis technique.
#'        \item It can be theoretically meaningful to understand how much of the overall variation in the response is explained simply by clustering. For example, in a repeated measures psychological study you can tell to what extent mood is a trait (varies among people, but not within a person on different occasions) or state (varies little on average among people, but varies a lot across occasions).
#'        \item It can also be meaningful to see how the ICC (as well as the between and within cluster variances) changes as variable are added to the model.
#'       }
#'       In short, the ICC can be interpreted as \dQuote{the proportion of the variance
#'       explained by the grouping structure in the population} \cite{(Hox 2002: 15)}.
#'       \cr \cr
#'       Usually, the ICC is calculated for the null model ("unconditional model").
#'       However, according to \cite{Raudenbush and Bryk (2002)} or
#'       \cite{Rabe-Hesketh and Skrondal (2012)} it is also feasible to compute the ICC
#'       for full models with covariates ("conditional models") and compare how
#'       much a level-2 variable explains the portion of variation in the grouping
#'       structure (random intercept).
#'       \cr \cr
#'       \strong{Caution:} For three-level-models, depending on the nested structure
#'       of the model, the ICC only reports the proportion of variance explained
#'       for each grouping level. However, the proportion of variance for specific
#'       levels related to each other (e.g., similarity of level-1-units within
#'       level-2-units or level-2-units within level-3-units) must be computed
#'       manually. Use \code{\link{get_re_var}} to get the between-group-variances
#'       and residual variance of the model, and calculate the ICC for the various level
#'       correlations.
#'       \cr \cr
#'       For example, for the ICC between level 1 and 2: \cr
#'       \code{sum(get_re_var(fit)) / (sum(get_re_var(fit)) + get_re_var(fit, "sigma_2"))}
#'       \cr \cr
#'       or for the ICC between level 2 and 3: \cr
#'       \code{get_re_var(fit)[2] / sum(get_re_var(fit))}
#'
#' @details The ICC is calculated by dividing the between-group-variance (random
#'   intercept variance) by the total variance (i.e. sum of between-group-variance
#'   and within-group (residual) variance). \cr \cr
#'   The calculation of the ICC for generalized linear mixed models with binary outcome is based on
#'   \cite{Wu et al. (2012)}. For Poisson multilevel models, please refer to \cite{Stryhn et al. (2006)}.
#'   \cite{Aly et al. (2014)} describe computation of ICC for negative binomial models.
#'   \cr \cr
#'   \strong{Caution:} For models with random slopes and random intercepts,
#'   the ICC would differ at each unit of the predictors. Hence, the ICC for these
#'   kind of models cannot be understood simply as proportion of variance
#'   (see \cite{Goldstein et al. 2010}). For convenience reasons, as the
#'   \code{icc()} function also extracts the different random effects
#'   variances, the ICC for random-slope-intercept-models is reported
#'   nonetheless, but it is usually no meaningful summary of the
#'   proportion of variances.
#'   \cr \cr
#'   The random effect variances indicate the between- and within-group
#'   variances as well as random-slope variance and random-slope-intercept
#'   correlation. The components are denoted as following:
#'   \itemize{
#'     \item Within-group (residual) variance: sigma_2
#'     \item Between-group-variance: tau.00 (variation between individual intercepts and average intercept)
#'     \item Random-slope-variance: tau.11 (variation between individual slopes and average slope)
#'     \item Random-Intercept-Slope-covariance: tau.01
#'     \item Random-Intercept-Slope-correlation: rho.01
#'   }
#'   If \code{ppd = TRUE}, \code{icc()} calculates a variance decomposition based on
#'   the posterior predictive distribution. In this case, first, the draws from
#'   the posterior predictive distribution \emph{not conditioned} on group-level
#'   terms (\code{posterior_predict(..., re.form = NA)}) are calculated as well
#'   as draws from this distribution \emph{conditioned} on \emph{all random effects}
#'   (by default, unless specified else in \code{re.form}) are taken. Then, second,
#'   the variances for each of these draws are calculated. The "ICC" is then the
#'   ratio between these two variances. This is the recommended way to
#'   analyse random-effect-variances for non-Gaussian models. It is then possible
#'   to compare variances accross models, also by specifying different group-level
#'   terms via the \code{re.form}-argument.
#'   \cr \cr
#'   Sometimes, when the variance of the posterior predictive distribution is
#'   very large, the variance ratio in the output makes no sense, e.g. because
#'   it is negative. In such cases, it might help to use a more robust measure
#'   to calculate the central tendency of the variances. For example, use
#'   \code{typical = "median"}.
#'
#' @seealso \code{\link{re_var}}
#'
#' @examples
#' library(lme4)
#' fit0 <- lmer(Reaction ~ 1 + (1 | Subject), sleepstudy)
#' icc(fit0)
#'
#' # note: ICC for random-slope-intercept model usually not
#' # meaningful - see 'Note'.
#' fit1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
#' icc(fit1)
#'
#' sleepstudy$mygrp <- sample(1:45, size = 180, replace = TRUE)
#' fit2 <- lmer(Reaction ~ Days + (1 | mygrp) + (1 | Subject), sleepstudy)
#' icc(fit2)
#'
#' icc1 <- icc(fit1)
#' icc2 <- icc(fit2)
#'
#' print(icc1, comp = "var")
#' print(icc2, comp = "var")
#'
#' \dontrun{
#' # compute ICC for Bayesian mixed model, with an ICC for each
#' # sample of the posterior. The print()-method then shows
#' # the median ICC as well as 89% HDI for the ICC.
#' # Change interval with print-method:
#' # print(icc(m, posterior = TRUE), prob = .5)
#'
#' if (requireNamespace("brms", quietly = TRUE)) {
#'   library(dplyr)
#'   sleepstudy$mygrp <- sample(1:5, size = 180, replace = TRUE)
#'   sleepstudy <- sleepstudy %>%
#'     group_by(mygrp) %>%
#'     mutate(mysubgrp = sample(1:30, size = n(), replace = TRUE))
#'   m <- brms::brm(
#'     Reaction ~ Days + (1 | mygrp / mysubgrp) + (1 | Subject),
#'     data = sleepstudy
#'   )
#'
#'   # by default, 89% interval
#'   icc(m)
#'
#'   # show 50% interval
#'   icc(m, prob = .5)
#'
#'   # variances based on posterior predictive distribution
#'   icc(m, ppd = TRUE)
#' }}
#'
#' @importFrom purrr map2
#' @export
icc <- function(x, ...) {
  UseMethod("icc")
}


#' @importFrom lme4 VarCorr fixef getME
#' @importFrom stats formula
#' @importFrom purrr map map_dbl map_lgl
#' @export
icc.merMod <- function(x, ...) {
  # get family
  fitfam <- model_family(x)


  # random effects variances
  # for details on tau and sigma, see
  # Aguinis H, Gottfredson RK, Culpepper SA2013. Best-Practice Recommendations
  # for Estimating Cross-Level Interaction Effects Using Multilevel Modeling.
  # Journal of Management 39(6): 1490–1528. doi:10.1177/0149206313478188.
  reva <- lme4::VarCorr(x)

  # retrieve only intercepts
  vars <- purrr::map(reva, ~ .x[1])

  # random intercept-variances, i.e.
  # between-subject-variance (tau 00)
  tau.00 <- purrr::map_dbl(vars, ~ .x)

  # random slope-variances (tau 11)
  tau.11 <- unlist(lapply(reva, function(x) diag(x)[-1]))

  # get residual standard deviation sigma
  sig <- attr(reva, "sc")

  # set default, if no residual variance is available

  if (is.null(sig)) {
    if (fitfam$is_bin)
      sig <- sqrt((pi ^ 2) / 3)
    else
      sig <- 1
  }


  # residual variances, i.e.
  # within-cluster-variance (sigma^2)

  if (fitfam$is_bin) {
    # for logistic models, we use pi / 3
    resid_var <- (pi ^ 2) / 3
  } else if (inherits(x, "glmerMod") && fitfam$is_negbin) {
    # for negative binomial models, we use 1
    resid_var <- 1
  } else {
    # for linear and poisson models, we have a clear residual variance
    resid_var <- sig ^ 2
  }


  # total variance, sum of random intercept and residual variances
  total_var <- sum(purrr::map_dbl(vars, ~ sum(.x)), resid_var)


  # check whether we have negative binomial

  if (fitfam$is_negbin) {
    # for negative binomial models, we also need the intercept...
    beta <- as.numeric(lme4::fixef(x)["(Intercept)"])
    # ... and the theta value to compute the ICC
    r <- lme4::getME(x, "glmer.nb.theta")

    # make formula more readable

    numerator <- (exp(tau.00) - 1)
    denominator <- ((exp(total_var) - 1) + (exp(total_var) / r) + exp(-beta - (total_var / 2)))

    ri.icc <- numerator / denominator
  } else {
    # random intercept icc
    ri.icc <- tau.00 / total_var
  }


  # get random slope random intercept correlations
  # do we have any rnd slopes?

  has_rnd_slope <- purrr::map_lgl(reva, ~ dim(attr(.x, "correlation"))[1] > 1)
  tau.01 <- rho.01 <- NULL


  # get rnd slopes

  if (any(has_rnd_slope)) {

    rnd_slope <- reva[has_rnd_slope]

    # get slope-intercept-correlations
    rho.01 <- purrr::map_dbl(rnd_slope, ~ attr(.x, "correlation")[1, 2])
    # get standard deviations, multiplied
    std_ <- purrr::map_dbl(rnd_slope, ~ prod(attr(.x, "stddev")))

    # bind to matrix
    tau.01 <- apply(cbind(rho.01, std_), MARGIN = 1, FUN = prod)

    message("Caution! ICC for random-slope-intercept models usually not meaningful. See 'Note' in `?icc`.")
  }

  # name values
  names(ri.icc) <- names(reva)


  if (inherits(x, "glmerMod"))
    mt <- "Generalized linear mixed model"
  else
    mt <- "Linear mixed model"

  # add attributes, for print method
  class(ri.icc) <- c("sj_icc_merMod", class(ri.icc))
  attr(ri.icc, "family") <- fitfam$family
  attr(ri.icc, "link") <- fitfam$link.fun
  attr(ri.icc, "formula") <- stats::formula(x)
  attr(ri.icc, "model") <- mt
  attr(ri.icc, "tau.00") <- tau.00
  attr(ri.icc, "tau.01") <- tau.01
  attr(ri.icc, "rho.01") <- rho.01
  attr(ri.icc, "tau.11") <- tau.11
  attr(ri.icc, "sigma_2") <- resid_var
  attr(ri.icc, "rnd.slope.model") <- any(has_rnd_slope)


  # finally, save name of fitted model object. May be needed for
  # the 'se()' function, which accesses the global environment

  attr(ri.icc, ".obj.name") <- deparse(substitute(x))

  # return results
  ri.icc
}


#' @importFrom lme4 VarCorr fixef getME
#' @importFrom glmmTMB VarCorr fixef getME
#' @importFrom stats family formula
#' @importFrom purrr map map_dbl map_lgl
#' @export
icc.glmmTMB <- function(x, ...) {
  # get family
  fitfam <- model_family(x)


  # random effects variances
  # for details on tau and sigma, see
  # Aguinis H, Gottfredson RK, Culpepper SA2013. Best-Practice Recommendations
  # for Estimating Cross-Level Interaction Effects Using Multilevel Modeling.
  # Journal of Management 39(6): 1490–1528. doi:10.1177/0149206313478188.
  reva <- glmmTMB::VarCorr(x)[[1]]

  # retrieve only intercepts
  vars <- purrr::map(reva, ~ .x[1])

  # random intercept-variances, i.e.
  # between-subject-variance (tau 00)
  tau.00 <- purrr::map_dbl(vars, ~ .x)

  # random slope-variances (tau 11)
  tau.11 <- unlist(lapply(reva, function(x) diag(x)[-1]))

  # get residual standard deviation sigma
  sig <- attr(reva, "sc")


  # set default, if no residual variance is available

  if (is.null(sig)) {
    if (fitfam$is_bin)
      sig <- sqrt((pi ^ 2) / 3)
    else
      sig <- 1
  }


  # residual variances, i.e.
  # within-cluster-variance (sigma^2)

  if (fitfam$is_bin) {
    # for logistic models, we use pi / 3
    resid_var <- (pi ^ 2) / 3
  } else {
    # for linear and poisson models, we have a clear residual variance
    resid_var <- sig ^ 2
  }


  # total variance, sum of random intercept and residual variances
  total_var <- sum(purrr::map_dbl(vars, ~ sum(.x)), resid_var)


  # check whether we have negative binomial

  if (fitfam$is_negbin) {
    # for negative binomial models, we also need the intercept...
    beta <- as.numeric(glmmTMB::fixef(x)[[1]]["(Intercept)"])
    # ... and the theta value to compute the ICC
    r <- sig

    # make formula more readable

    numerator <- (exp(tau.00) - 1)
    denominator <- ((exp(total_var) - 1) + (exp(total_var) / r) + exp(-beta - (total_var / 2)))

    ri.icc <- numerator / denominator
  } else {
    # random intercept icc
    ri.icc <- tau.00 / total_var
  }


  # get random slope random intercept correlations
  # do we have any rnd slopes?

  has_rnd_slope <- purrr::map_lgl(reva, ~ dim(attr(.x, "correlation"))[1] > 1)
  tau.01 <- rho.01 <- NULL


  # get rnd slopes

  if (any(has_rnd_slope)) {

    rnd_slope <- reva[has_rnd_slope]

    # get slope-intercept-correlations
    rho.01 <- purrr::map_dbl(rnd_slope, ~ attr(.x, "correlation")[1, 2])
    # get standard deviations, multiplied
    std_ <- purrr::map_dbl(rnd_slope, ~ prod(attr(.x, "stddev")))

    # bind to matrix
    tau.01 <- apply(cbind(rho.01, std_), MARGIN = 1, FUN = prod)

    message("Caution! ICC for random-slope-intercept models usually not meaningful. See 'Note' in `?icc`.")
  }

  # name values
  names(ri.icc) <- names(reva)


  mt <- "Generalized linear mixed model"

  # add attributes, for print method
  class(ri.icc) <- c("sj_icc_merMod", class(ri.icc))
  attr(ri.icc, "family") <- fitfam$family
  attr(ri.icc, "link") <- fitfam$link.fun
  attr(ri.icc, "formula") <- stats::formula(x)
  attr(ri.icc, "model") <- mt
  attr(ri.icc, "tau.00") <- tau.00
  attr(ri.icc, "tau.01") <- tau.01
  attr(ri.icc, "rho.01") <- rho.01
  attr(ri.icc, "tau.11") <- tau.11
  attr(ri.icc, "sigma_2") <- resid_var
  attr(ri.icc, "rnd.slope.model") <- any(has_rnd_slope)


  # finally, save name of fitted model object. May be needed for
  # the 'se()' function, which accesses the global environment

  attr(ri.icc, ".obj.name") <- deparse(substitute(x))

  # return results
  ri.icc
}


#' @importFrom stats formula
#' @importFrom purrr map map_dbl map_lgl
#' @importFrom sjmisc row_sums is_empty
#' @rdname icc
#' @export
icc.stanreg <- function(x, re.form = NULL, typical = "mean", prob = .89, ppd = FALSE, ...) {

  if (!requireNamespace("rstanarm", quietly = TRUE))
    stop("Please install and load package `rstanarm` first.", call. = F)

  # get family
  fitfam <- model_family(x)
  xdat <- as.data.frame(x)


  if (ppd) {

    ## TODO automatically calculate for multiple levels / nested models

    PPD <- rstanarm::posterior_predict(x, re.form = re.form)
    total_var <- apply(PPD, MARGIN = 1, FUN = stats::var)

    PPD_0 <- rstanarm::posterior_predict(x, re.form = NA)
    tau.00 <- apply(PPD_0, MARGIN = 1, FUN = stats::var)

    ri.icc <- tau.00 / total_var
    resid_var <- total_var - tau.00

    icc_ <- c(
      1 - typical_value(ri.icc, fun = typical),
      typical_value(tau.00, fun = typical),
      typical_value(resid_var, fun = typical),
      typical_value(total_var, fun = typical)
    )

    attr(icc_, "hdi.icc") <- rev(1 - hdi(ri.icc, prob = prob))
    attr(icc_, "hdi.tau.00") <- hdi(tau.00, prob = prob)
    attr(icc_, "hdi.resid") <- hdi(resid_var, prob = prob)
    attr(icc_, "hdi.total") <- hdi(total_var, prob = prob)
    attr(icc_, "re.form") <- re.form
    attr(icc_, "ranef") <- x$ranef$group[1]

    has_rnd_slope <- FALSE
    names(icc_) <- c("icc", "tau.00", "resid.var", "total.var")
    class(icc_) <- c("icc_ppd", class(icc_))

  } else {

    # random intercept-variances, i.e.
    # between-subject-variance (tau 00)
    tau00s <- grep("^Sigma\\[(.*):\\(Intercept\\),\\(Intercept\\)", colnames(xdat))
    tau.00 <- xdat[, tau00s, drop = FALSE]

    names(tau.00) <- gsub(
      "^Sigma\\[(.*):\\(Intercept\\),\\(Intercept\\)\\]",
      "\\1",
      colnames(xdat))[tau00s]


    # random slope-variances (tau 11)
    tau11s <- grep("^Sigma\\[(.*):[^\\(\\)](.*),[^\\(\\)](.*)\\]", colnames(xdat))

    if (!sjmisc::is_empty(tau11s)) {
      tau.11 <- xdat[, tau11s, drop = FALSE]
      names(tau.11) <- gsub(
        "^Sigma\\[(.*):[^\\(\\)](.*),[^\\(\\)](.*)\\]",
        "\\1",
        colnames(xdat))[tau11s]
    } else {
      tau.11 <- NULL
    }


    # get residual standard deviation sigma
    sig <- xdat[["sigma"]]

    # for linear and poisson models, we have a clear residual variance
    resid_var <- sig ^ 2

    # total variance, sum of random intercept and residual variances
    total_var <- sjmisc::row_sums(
      cbind(tau.00, data.frame(resid_var)),
      var = "total_var",
      append = FALSE
    )

    # random intercept icc
    ri.icc <- tau.00 / total_var$total_var


    # get random slope random intercept correlations
    # do we have any rnd slopes?

    has_rnd_slope <- !sjmisc::is_empty(tau11s)
    tau.01 <- rho.01 <- NULL


    # get rnd slopes

    if (any(has_rnd_slope)) {

      # get slope-intercept-covariance
      tau01s <- grep("^Sigma\\[(.*):[^\\(\\)](.*),\\(Intercept\\)\\]", colnames(xdat))
      tau.01 <- xdat[, tau01s, drop = FALSE]

      tau.00.sums <- sjmisc::row_sums(tau.00, var = "t0sums")$t0sums
      tau.11.sums <- sjmisc::row_sums(tau.11, var = "t1sums")$t1sums

      # get slope-intercept-correlations
      rho.01 <- tau.01 / sqrt(tau.00.sums * tau.11.sums)

      message("Caution! ICC for random-slope-intercept models usually not meaningful. See 'Note' in `?icc`.")

    }

    if (inherits(x, "glmerMod"))
      mt <- "Generalized linear mixed model"
    else
      mt <- "Linear mixed model"


    icc_ <- purrr::map_dbl(ri.icc, ~ typical_value(.x, fun = typical))
    attr(icc_, "hdi.icc") <- purrr::map(ri.icc, ~ hdi(.x, prob = prob))

    attr(icc_, "hdi.tau.00") <- purrr::map(tau.00, ~ hdi(.x, prob = prob))
    tau.00 <- purrr::map_dbl(tau.00, ~ typical_value(.x, fun = typical))

    attr(icc_, "hdi.sigma_2") <- hdi(resid_var, prob = prob)
    resid_var <- typical_value(resid_var, fun = typical)

    if (!is.null(tau.11)) {
      attr(icc_, "hdi.tau.11") <- purrr::map(tau.11, ~ hdi(.x, prob = prob))
      tau.11 <- purrr::map_dbl(tau.11, ~ typical_value(.x, fun = typical))
    }

    if (!is.null(rho.01)) {
      attr(icc_, "hdi.rho.01") <- purrr::map(rho.01, ~ hdi(.x, prob = prob))
      rho.01 <- purrr::map_dbl(rho.01, ~ typical_value(.x, fun = typical))
    }

    if (!is.null(tau.01)) {
      attr(icc_, "hdi.tau.01") <- purrr::map(tau.01, ~ hdi(.x, prob = prob))
      tau.01 <- purrr::map_dbl(tau.01, ~ typical_value(.x, fun = typical))
    }

    attr(icc_, "tau.00") <- tau.00
    attr(icc_, "tau.01") <- tau.01
    attr(icc_, "rho.01") <- rho.01
    attr(icc_, "tau.11") <- tau.11
    attr(icc_, "sigma_2") <- resid_var
    attr(icc_, "rnd.slope.model") <- any(has_rnd_slope)
    attr(icc_, "model") <- mt

    class(icc_) <- c("sj_icc_stanreg", class(icc_))
  }

  # add attributes, for print method
  attr(icc_, "family") <- fitfam$family
  attr(icc_, "link") <- fitfam$link.fun
  attr(icc_, "formula") <- stats::formula(x)
  attr(icc_, "prob") <- prob


  # return results
  icc_
}


#' @importFrom purrr map_df map_if map_lgl map_dbl
#' @importFrom dplyr bind_cols
#' @importFrom sjmisc all_na
#' @rdname icc
#' @export
icc.brmsfit <- function(x, re.form = NULL, typical = "mean", prob = .89, ppd = FALSE, ...) {

  if (!requireNamespace("brms", quietly = TRUE))
    stop("Please install and load package `brms` first.", call. = F)

  # get family
  fitfam <- model_family(x)


  if (ppd) {

    ## TODO automatically calculate for multiple levels / nested models

    PPD <- brms::posterior_predict(x, re.form = re.form, summary = FALSE)
    total_var <- apply(PPD, MARGIN = 1, FUN = stats::var)

    PPD_0 <- brms::posterior_predict(x, re.form = NA, summary = FALSE)
    tau.00 <- apply(PPD_0, MARGIN = 1, FUN = stats::var)

    ri.icc <- tau.00 / total_var
    resid_var <- total_var - tau.00

    icc_ <- c(
      1 - typical_value(ri.icc, fun = typical),
      typical_value(tau.00, fun = typical),
      typical_value(resid_var, fun = typical),
      typical_value(total_var, fun = typical)
    )

    attr(icc_, "hdi.icc") <- rev(1 - hdi(ri.icc, prob = prob))
    attr(icc_, "hdi.tau.00") <- hdi(tau.00, prob = prob)
    attr(icc_, "hdi.resid") <- hdi(resid_var, prob = prob)
    attr(icc_, "hdi.total") <- hdi(total_var, prob = prob)
    attr(icc_, "prob") <- prob
    attr(icc_, "re.form") <- re.form
    attr(icc_, "ranef") <- x$ranef$group[1]

    has_rnd_slope <- FALSE
    names(icc_) <- c("icc", "tau.00", "resid.var", "total.var")
    class(icc_) <- c("icc_ppd", class(icc_))

  } else {

    # get random effect variances for each sample of posterior
    reva <- brms::VarCorr(x, summary = FALSE)

    # remove "residual__" element from list
    # and save in separate object
    reva.resid <- reva[names(reva) == "residual__"]
    reva <- reva[!(names(reva) == "residual__")]


    # retrieve only intercepts
    vars <- purrr::map(reva, ~ .x$sd[, 1] ^ 2)

    # random intercept-variances, i.e.
    # between-subject-variance (tau 00)
    tau.00 <- purrr::map(vars, ~ .x)

    # random slope-variances (tau 11)
    tau.11 <- purrr::map(reva, ~ .x$cov[, 2, 2])

    # get residual standard deviation sigma
    sig <- reva.resid[["residual__"]]$sd[, 1]

    # set default, if no residual variance is available
    if (is.null(sig)) {
      if (fitfam$is_bin)
        sig <- sqrt((pi ^ 2) / 3)
      else
        sig <- 1
    }


    # residual variances, i.e.
    # within-cluster-variance (sigma^2)

    resid_var <- sig ^ 2


    # total variance, sum of random intercept and residual variances
    total_var <- apply(as.data.frame(vars), MARGIN = 1, FUN = sum) + resid_var

    # make sure residual variance has same length as other components
    # if not, just repeat the current value to match number of samples
    if (length(resid_var) == 1) resid_var <- rep(resid_var, length(total_var))

    # check whether we have negative binomial

    if (fitfam$is_negbin) {

      # for negative binomial models, we also need the intercept...
      beta <- as.numeric(brms::fixef(x)[[1]])
      # ... and the theta value to compute the ICC
      r <- sig

      # make formula more readable

      numerator <- purrr::map(tau.00, ~ exp(.x) - 1)
      denominator <- ((exp(total_var) - 1) + (exp(total_var) / r) + exp(-beta - (total_var / 2)))

      ri.icc <- purrr::map(numerator, ~ .x / denominator)
    } else {
      # random intercept icc
      ri.icc <- purrr::map(tau.00, ~ .x / total_var)
    }

    tau.11 <- purrr::map_if(tau.11, is.null, ~ rep(NA, length(resid_var)))

    names(ri.icc) <- sprintf("icc_%s", names(ri.icc))
    names(tau.00) <- sprintf("tau.00_%s", names(tau.00))
    names(tau.11) <- sprintf("tau.11_%s", names(tau.11))

    icc_ <- purrr::map_dbl(ri.icc, ~ typical_value(.x, fun = typical))

    attr(icc_, "tau.00") <- purrr::map_dbl(tau.00, ~ typical_value(.x, fun = typical))
    attr(icc_, "hdi.icc") <- purrr::map(ri.icc, ~ hdi(.x, prob = prob))
    attr(icc_, "hdi.tau.00") <- purrr::map(tau.00, ~ hdi(.x, prob = prob))

    attr(icc_, "sigma_2") <- typical_value(resid_var, fun = typical)
    attr(icc_, "hdi.sigma_2") <- hdi(resid_var, prob = prob)

    attr(icc_, "prob") <- prob

    check_tau <- purrr::map_lgl(tau.11, ~ sjmisc::all_na(.x))
    if (any(!check_tau)) {
      tau.11 <- tau.11[!check_tau]
      attr(icc_, "tau.11") <- purrr::map_dbl(tau.11, ~ typical_value(.x, fun = typical))
      attr(icc_, "hdi.tau.11") <- purrr::map(tau.11, ~ hdi(.x, prob = prob))
    }

    has_rnd_slope <- any(isTRUE(purrr::map_lgl(brms::ranef(x), ~ dim(.x)[3] > 1)))

    if (has_rnd_slope)
      message("Caution! ICC for random-slope-intercept models usually not meaningful. See 'Note' in `?icc`.")

    class(icc_) <- c("sj_icc_brms", class(icc_))
  }


  attr(icc_, "family") <- fitfam$family
  attr(icc_, "link") <- fitfam$link.fun
  attr(icc_, "formula") <- stats::formula(x)
  attr(icc_, "model") <- "Bayesian mixed model"
  attr(ri.icc, "rnd.slope.model") <- any(has_rnd_slope)

  # return results
  icc_
}


#' @title Random effect variances
#' @name re_var
#' @description These functions extracts random effect variances as well as
#'                random-intercept-slope-correlation of mixed effects models.
#'                Currently, \code{\link[lme4]{merMod}}, \code{\link[glmmTMB]{glmmTMB}},
#'                \code{stanreg} and \code{\link[brms]{brmsfit}}
#'                objects are supported.
#'
#' @param x Fitted mixed effects model (of class \code{merMod}, \code{glmmTMB},
#'          \code{stanreg} or \code{brmsfit}). \code{get_re_var()} also accepts
#'           an object of class \code{icc.lme4}, as returned by the
#'           \code{\link{icc}} function.
#' @param comp Name of the variance component to be returned. See 'Details'.
#'
#' @return \code{get_re_var()} returns the value of the requested variance component,
#'           \code{re_var()} returns all random effects variances.
#'
#' @references Aguinis H, Gottfredson RK, Culpepper SA. 2013. Best-Practice Recommendations for Estimating Cross-Level Interaction Effects Using Multilevel Modeling. Journal of Management 39(6): 1490–1528 (\doi{10.1177/0149206313478188})
#'
#' @details The random effect variances indicate the between- and within-group
#'         variances as well as random-slope variance and random-slope-intercept
#'         correlation. Use following values for \code{comp} to get the particular
#'         variance component:
#'         \describe{
#'          \item{\code{"sigma_2"}}{Within-group (residual) variance}
#'          \item{\code{"tau.00"}}{Between-group-variance (variation between individual intercepts and average intercept)}
#'          \item{\code{"tau.11"}}{Random-slope-variance (variation between individual slopes and average slope)}
#'          \item{\code{"tau.01"}}{Random-Intercept-Slope-covariance}
#'          \item{\code{"rho.01"}}{Random-Intercept-Slope-correlation}
#'         }
#'         The within-group-variance is affected by factors at level one, i.e.
#'         by the lower-level direct effects. Level two factors (i.e. cross-level
#'         direct effects) affect the between-group-variance. Cross-level
#'         interaction effects are group-level factors that explain the
#'         variance in random slopes (Aguinis et al. 2013).
#'
#' @seealso \code{\link{icc}}
#'
#' @examples
#' library(lme4)
#' fit1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)
#'
#' # all random effect variance components
#' re_var(fit1)
#'
#' # just the rand. slope-intercept covariance
#' get_re_var(fit1, "tau.01")
#'
#' sleepstudy$mygrp <- sample(1:45, size = 180, replace = TRUE)
#' fit2 <- lmer(Reaction ~ Days + (1 | mygrp) + (Days | Subject), sleepstudy)
#' re_var(fit2)
#'
#' @importFrom stats family
#' @importFrom purrr map map2 flatten_dbl flatten_chr
#' @importFrom sjmisc trim
#' @export
re_var <- function(x) {
  # iterate all attributes and return them as vector
  rv <- c("sigma_2", "tau.00", "tau.11", "tau.01", "rho.01")

  # compute icc
  icc_ <- suppressMessages(icc(x))

  rv_ <- purrr::map(rv, ~ attr(icc_, .x, exact = TRUE))
  rn <- purrr::map2(1:length(rv_), rv, ~ sjmisc::trim(paste(names(rv_[[.x]]), .y, sep = "_")))
  rv_ <- purrr::flatten_dbl(rv_)

  names(rv_) <- purrr::flatten_chr(rn)[1:length(rv_)]

  class(rv_) <- c("sj_revar", class(rv_))

  rv_
}


#' @rdname re_var
#' @export
get_re_var <- function(x, comp = c("tau.00", "tau.01", "tau.11", "rho.01", "sigma_2")) {
  # check if we have a valid object
  if (!inherits(x, "icc.lme4") && !is_merMod(x) && !inherits(x, c("glmmTMB", "brmsfit"))) {
    stop("`x` must either be an object returned by the `icc` function, or a merMod-, glmmTMB- or brmsfit-object.", call. = F)
  }

  # check arguments
  comp <- match.arg(comp)

  # do we have a merMod object? If yes, get ICC and var components
  if (is_merMod(x) || inherits(x, c("glmmTMB", "brmsfit"))) x <- suppressMessages(icc(x))

  # return results
  attr(x, comp, exact = TRUE)
}
