%% File Name: tam2mirt.Rd
%% File Version: 0.14
%% File Last Change: 2017-01-18 18:08:42

\name{tam2mirt}
\alias{tam2mirt}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Converting a fitted \code{TAM} Object into a \code{mirt} Object
}
\description{
Converts a fitted \code{TAM} object into a \code{mirt} object.
As a by-product, \code{lavaan} syntax is generated which can
be used with \code{\link{lavaan2mirt}} for re-estimating
the model in the \pkg{mirt} package.
Up to now, only single group models are supported.
There must not exist background covariates (no latent regression
models!).
}

\usage{
tam2mirt(tamobj)
}

%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{tamobj}{
Object of class  \code{\link[TAM:tam.mml]{TAM::tam.mml}}
}
}

%\details{
%%  ~~ If necessary, more details than the description above ~~
%}


\value{
A list with following entries

\item{mirt}{Object generated by \code{mirt} function if
\code{est.mirt=TRUE}}
\item{mirt.model}{Generated \code{mirt} model}
\item{mirt.syntax}{Generated \code{mirt} syntax}
\item{mirt.pars}{Generated parameter specifications
   in \code{mirt}}
\item{lavaan.model}{Used \code{lavaan} model transformed by
\code{lavaanify} function}
\item{dat}{Used dataset. If necessary, only items used in the
model are included in the dataset.}
\item{lavaan.syntax.fixed}{Generated \code{lavaan} syntax
with fixed parameter estimates.}
\item{lavaan.syntax.freed}{Generated \code{lavaan} syntax
with freed parameters for estimation.}
}


%\references{
%% ~put references to the literature/web site here ~
%}

\author{
Alexander Robitzsch
}

%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
See \code{\link{mirt.wrapper}} for convenience wrapper functions
for \code{\link[mirt:mirt]{mirt}} objects.

See \code{\link{lavaan2mirt}} for converting \code{lavaan}
syntax to \code{mirt} syntax.
}


\examples{
\dontrun{	
library(TAM)
library(mirt)

#############################################################################
# EXAMPLE 1: Estimations in TAM for data.read dataset
#############################################################################

data(data.read)
dat <- data.read

#**************************************
#*** Model 1: Rasch model
#**************************************

# estimation in TAM package
mod <- TAM::tam.mml( dat )
summary(mod)
# conversion to mirt
res <- sirt::tam2mirt(mod)
# generated lavaan syntax
cat(res$lavaan.syntax.fixed)
cat(res$lavaan.syntax.freed)
# extract object of class mirt
mres <- res$mirt
# print and parameter values
print(mres)
mirt::mod2values(mres)
# model fit
mirt::M2(mres)
# residual statistics
mirt::residuals(mres , type="Q3")
mirt::residuals(mres , type="LD")
# item fit
mirt::itemfit(mres)
# person fit
mirt::personfit(mres)
# compute several types of factor scores (quite slow)
f1 <- mirt::fscores(mres, method='WLE',response.pattern=dat[1:10,]) 
     # method = MAP and EAP also possible 
# item plot
mirt::itemplot(mres,"A3")    # item A3
mirt::itemplot(mres,4)       # fourth item
# some more plots
plot(mres,type="info")
plot(mres,type="score")
plot(mres,type="trace")
# compare estimates with estimated Rasch model in mirt
mres1 <- mirt::mirt(dat,1,"Rasch" )
print(mres1)
mirt.wrapper.coef(mres1)

#**************************************
#*** Model 2: 2PL model
#**************************************

# estimation in TAM
mod <- TAM::tam.mml.2pl( dat )
summary(mod)
# conversion to mirt
res <- sirt::tam2mirt(mod)
mres <- res$mirt
# lavaan syntax
cat(res$lavaan.syntax.fixed)
cat(res$lavaan.syntax.freed)
# parameter estimates
print(mres)
mod2values(mres)
mres@nest   # number of estimated parameters
# some plots
plot(mres,type="info")
plot(mres,type="score")
plot(mres,type="trace")
# model fit
mirt::M2(mres)
# residual statistics
mirt::residuals(mres , type="Q3")
mirt::residuals(mres , type="LD")
# item fit
mirt::itemfit(mres)

#**************************************
#*** Model 3: 3-dimensional Rasch model
#**************************************

# define Q-matrix
Q <- matrix( 0 , nrow=12 , ncol=3 )
Q[ cbind(1:12 , rep(1:3,each=4) ) ] <- 1
rownames(Q) <- colnames(dat)
colnames(Q) <- c("A","B","C")
# estimation in TAM
mod <- TAM::tam.mml( resp=dat , Q=Q , control=list(snodes=1000,maxiter=30) )
summary(mod)
# mirt conversion
res <- sirt::tam2mirt(mod)
mres <- res$mirt
# mirt syntax
cat(res$mirt.syntax)
  ##   Dim01=1,2,3,4
  ##   Dim02=5,6,7,8
  ##   Dim03=9,10,11,12
  ##   COV = Dim01*Dim01,Dim02*Dim02,Dim03*Dim03,Dim01*Dim02,Dim01*Dim03,Dim02*Dim03
  ##   MEAN = Dim01,Dim02,Dim03
# lavaan syntax
cat(res$lavaan.syntax.freed)
  ##   Dim01 =~ 1*A1+1*A2+1*A3+1*A4
  ##   Dim02 =~ 1*B1+1*B2+1*B3+1*B4
  ##   Dim03 =~ 1*C1+1*C2+1*C3+1*C4
  ##   A1 | t1_1*t1
  ##   A2 | t1_2*t1
  ##   A3 | t1_3*t1
  ##   A4 | t1_4*t1
  ##   B1 | t1_5*t1
  ##   B2 | t1_6*t1
  ##   B3 | t1_7*t1
  ##   B4 | t1_8*t1
  ##   C1 | t1_9*t1
  ##   C2 | t1_10*t1
  ##   C3 | t1_11*t1
  ##   C4 | t1_12*t1
  ##   Dim01 ~ 0*1
  ##   Dim02 ~ 0*1
  ##   Dim03 ~ 0*1
  ##   Dim01 ~~ Cov_11*Dim01
  ##   Dim02 ~~ Cov_22*Dim02
  ##   Dim03 ~~ Cov_33*Dim03
  ##   Dim01 ~~ Cov_12*Dim02
  ##   Dim01 ~~ Cov_13*Dim03
  ##   Dim02 ~~ Cov_23*Dim03 
# model fit
mirt::M2(mres)
# residual statistics
residuals(mres,type="LD")
# item fit
mirt::itemfit(mres)

#**************************************
#*** Model 4: 3-dimensional 2PL model
#**************************************

# estimation in TAM
mod <- TAM::tam.mml.2pl( resp=dat , Q=Q , control=list(snodes=1000,maxiter=30) )
summary(mod)
# mirt conversion
res <- sirt::tam2mirt(mod)
mres <- res$mirt
# generated lavaan syntax
cat(res$lavaan.syntax.fixed)
cat(res$lavaan.syntax.freed)
# write lavaan syntax on disk
  sink( "mod4_lav_freed.txt" , split=TRUE )
cat(res$lavaan.syntax.freed)
  sink()
# some statistics from mirt
print(mres)
summary(mres)
mirt::M2(mres)
mirt::residuals(mres)
mirt::itemfit(mres)

# estimate mirt model by using the generated lavaan syntax with freed parameters
res2 <- sirt::lavaan2mirt( dat , res$lavaan.syntax.freed ,
            technical=list(NCYCLES=3) , verbose=TRUE)  
                 # use only few cycles for illustrational purposes
mirt.wrapper.coef(res2$mirt)
summary(res2$mirt)
print(res2$mirt)

#############################################################################
# EXAMPLE 4: mirt conversions for polytomous dataset data.big5
#############################################################################

data(data.big5)
# select some items
items <- c( grep( "O" , colnames(data.big5) , value=TRUE )[1:6] ,
     grep( "N" , colnames(data.big5) , value=TRUE )[1:4] )
# O3 O8 O13 O18 O23 O28 N1 N6 N11 N16
dat <- data.big5[ , items ]
library(psych)
psych::describe(dat)

library(TAM)
#******************
#*** Model 1: Partial credit model in TAM
mod1 <- TAM::tam.mml( dat[,1:6] )
summary(mod1)
# convert to mirt object
mmod1 <- sirt::tam2mirt( mod1 )
rmod1 <- mmod1$mirt
# coefficients in mirt
coef(rmod1)
mirt.wrapper.coef(rmod1)
# model fit
mirt::M2(rmod1)
# item fit
mirt::itemfit(rmod1)
# plots
plot(rmod1,type="trace")
plot(rmod1, type = "trace", which.items = 1:4 )
mirt::itemplot(rmod1,"O3")

#******************
#*** Model 2: Generalized partial credit model in TAM
mod2 <- TAM::tam.mml.2pl( dat[,1:6] , irtmodel="GPCM" )
summary(mod2)
# convert to mirt object
mmod2 <- sirt::tam2mirt( mod2 )
rmod2 <- mmod2$mirt
# coefficients in mirt
mirt.wrapper.coef(rmod2)
# model fit
mirt::M2(rmod2)
# item fit
mirt::itemfit(rmod2)
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{TAM}
\keyword{mirt}
