
#' @title Insert dependencies to track usage of a Shiny app
#'
#' @description If used in \code{ui} of an application,
#'  this will create new \code{input}s available in the server.
#'  Set \code{dependencies = FALSE} in \code{\link{track_usage}}
#'  server-side to load dependencies only once.
#'
#' @param on_unload Logical, save log when user close the browser window or tab,
#'  if \code{TRUE} it prevent to create \code{shinylogs}
#'  input during normal use of the application, there will
#'  be created only on close, downside is that a popup will appear asking to close the page.
#' @param exclude_input_regex Regular expression to exclude inputs from tracking.
#' @param exclude_input_id Vector of \code{inputId} to exclude from tracking.
#'
#' @note The following \code{input}s will be accessible in the server:
#'
#'   - \strong{.shinylogs_lastInput} : last \code{input} used by the user
#'
#'   - \strong{.shinylogs_input} : all \code{input}s send from the browser to the server
#'
#'   - \strong{.shinylogs_error} : all errors generated by \code{output}s elements
#'
#'   - \strong{.shinylogs_output} : all \code{output}s generated from the server
#'
#'   - \strong{.shinylogs_browserData} : information about the browser where application is displayed.
#'
#' @export
#'
#' @importFrom htmltools attachDependencies tags singleton
#' @importFrom jsonlite toJSON
#' @importFrom bit64 as.integer64
#' @importFrom nanotime nanotime
#' @importFrom digest digest
#'
#' @examples
#' if (interactive()) {
#'
#'   library(shiny)
#'
#'   ui <- fluidPage(
#'
#'     use_tracking(),
#'
#'     splitLayout(
#'       cellArgs = list(style = "height: 250px"),
#'       radioButtons("radio", "Radio:", names(iris)),
#'       checkboxGroupInput("checkbox", "Checkbox:", names(iris)),
#'       selectInput("select", "Select:", names(iris))
#'     ),
#'
#'     verbatimTextOutput("last")
#'   )
#'
#'   server <- function(input, output, session) {
#'
#'     output$last <- renderPrint({
#'       input$.shinylogs_lastInput
#'     })
#'
#'   }
#'
#'   shinyApp(ui, server)
#'
#' }
use_tracking <- function(on_unload = FALSE, exclude_input_regex = NULL, exclude_input_id = NULL) {
  app_name <- basename(getwd())
  timestamp <- Sys.time()
  init_log <- data.frame(
    app = app_name,
    server_connected = get_timestamp(timestamp),
    stringsAsFactors = FALSE
  )
  timestamp <- format(as.integer64(nanotime(timestamp)), scientific = FALSE)
  init_log$sessionid <- digest::digest(timestamp)
  tag_log <- tags$script(
    id = "shinylogs-tracking",
    type = "application/json",
    `data-for` = "shinylogs",
    toJSON(dropNulls(list(
      logsonunload = isTRUE(on_unload),
      exclude_input_regex = exclude_input_regex,
      exclude_input_id = exclude_input_id,
      sessionid = init_log$sessionid
    )), auto_unbox = TRUE, json_verbatim = TRUE)
  )
  attachDependencies(
    x = singleton(tag_log),
    value = list(
      localforage_dependencies(),
      dayjs_dependencies(),
      shinylogs_lf_dependencies()
    )
  )
}



#' @importFrom stats setNames
parse_logInput <- function(x, shinysession, name) {
  lapply(
    X = x,
    FUN = function(x) {
      setNames(x, NULL)
    }
  )
}

#' @importFrom anytime anytime
parse_lastInput <- function(x, shinysession, name) {
  if (!is.null(x)) {
    x$timestamp <- anytime(x$timestamp)
  }
  return(x)
}


#' @title Track usage of a Shiny app
#'
#' @description Used in Shiny \code{server} it will record all inputs and
#'  output changes and errors that occurs through an output.
#'
#' @param storage_mode Storage mode to use : \code{\link{store_json}}, \code{\link{store_rds}},
#'  \code{\link{store_sqlite}} or \code{\link{store_null}}.
#' @param exclude_input_regex Regular expression to exclude inputs from tracking.
#' @param exclude_input_id Vector of \code{inputId} to exclude from tracking.
#' @param on_unload Logical, save log when user close the browser window or tab,
#'  if \code{TRUE} it prevent to create \code{shinylogs}
#'  input during normal use of the application, there will
#'  be created only on close, downside is that a popup will appear asking to close the page.
#' @param exclude_users Character vectors of user for whom it is not necessary to save the log.
#' @param get_user A \code{function} to get user name, it should
#'  return a character and take one argument: the Shiny session.
#' @param dependencies Load dependencies.
#' @param session The shiny session.
#'
#' @export
#'
#' @note The following \code{input}s will be accessible in the server:
#'
#'   - \strong{.shinylogs_lastInput} : last \code{input} used by the user
#'
#'   - \strong{.shinylogs_input} : all \code{input}s send from the browser to the server
#'
#'   - \strong{.shinylogs_error} : all errors generated by \code{output}s elements
#'
#'   - \strong{.shinylogs_output} : all \code{output}s generated from the server
#'
#'   - \strong{.shinylogs_browserData} : information about the browser where application is displayed.
#'
#' @importFrom shiny getDefaultReactiveDomain insertUI onSessionEnded isolate observe
#' @importFrom nanotime nanotime
#' @importFrom bit64 as.integer64
#' @importFrom digest digest
#' @importFrom jsonlite toJSON
#' @importFrom htmltools tags singleton
#'
#' @examples
#'
#' ## Save logs on disk
#'
#' if (interactive()) {
#'
#'   # temporary directory for writing logs
#'   tmp <- tempdir()
#'
#'   # when app stop,
#'   # navigate to the directory containing logs
#'   onStop(function() {
#'     browseURL(url = tmp)
#'   })
#'
#'   # Classic Iris clustering with Shiny
#'   ui <- fluidPage(
#'
#'     headerPanel("Iris k-means clustering"),
#'
#'     sidebarLayout(
#'       sidebarPanel(
#'         selectInput(
#'           inputId = "xcol",
#'           label = "X Variable",
#'           choices = names(iris)
#'         ),
#'         selectInput(
#'           inputId = "ycol",
#'           label = "Y Variable",
#'           choices = names(iris),
#'           selected = names(iris)[[2]]
#'         ),
#'         numericInput(
#'           inputId = "clusters",
#'           label = "Cluster count",
#'           value = 3,
#'           min = 1,
#'           max = 9
#'         )
#'       ),
#'       mainPanel(
#'         plotOutput("plot1")
#'       )
#'     )
#'   )
#'
#'   server <- function(input, output, session) {
#'
#'     # Store JSON with logs in the temp dir
#'     track_usage(
#'       storage_mode = store_json(path = tmp)
#'     )
#'
#'     # classic server logic
#'
#'     selectedData <- reactive({
#'       iris[, c(input$xcol, input$ycol)]
#'     })
#'
#'     clusters <- reactive({
#'       kmeans(selectedData(), input$clusters)
#'     })
#'
#'     output$plot1 <- renderPlot({
#'       palette(c("#E41A1C", "#377EB8", "#4DAF4A", "#984EA3",
#'                 "#FF7F00", "#FFFF33", "#A65628", "#F781BF", "#999999"))
#'
#'       par(mar = c(5.1, 4.1, 0, 1))
#'       plot(selectedData(),
#'            col = clusters()$cluster,
#'            pch = 20, cex = 3)
#'       points(clusters()$centers, pch = 4, cex = 4, lwd = 4)
#'     })
#'
#'   }
#'
#'   shinyApp(ui, server)
#'
#' }
#'
#'
#' ## Special inputs :
#'
#' if (interactive()) {
#'   library(shiny)
#'   library(shinylogs)
#'
#'   ui <- fluidPage(
#'     tags$h2("Record inputs change"),
#'     fluidRow(
#'       column(
#'         width = 3,
#'         selectInput(
#'           inputId = "select",
#'           label = "Select input",
#'           choices = month.name
#'         ),
#'         numericInput(
#'           inputId = "numeric",
#'           label = "Numerci input",
#'           value = 4,
#'           min = 0, max = 20
#'         ),
#'         checkboxGroupInput(
#'           inputId = "checkboxGroup",
#'           label = "Checkbox group input",
#'           choices = LETTERS[1:5]
#'         ),
#'         sliderInput(
#'           inputId = "slider",
#'           label = "Slider input",
#'           min = 0, max = 100, value = 50
#'         )
#'       ),
#'       column(
#'         width = 9,
#'         tags$b("Last input:"),
#'         verbatimTextOutput(outputId = "last_input"),
#'         tags$b("Last input:"),
#'         verbatimTextOutput(outputId = "all_inputs")
#'       )
#'     )
#'   )
#'
#'   server <- function(input, output, session) {
#'
#'     track_usage(
#'       storage_mode = store_null() # dont store on disk
#'     )
#'
#'     output$last_input <- renderPrint({
#'       input$.shinylogs_lastInput # last input triggered
#'     })
#'
#'     output$all_inputs <- renderPrint({
#'       input$.shinylogs_input # all inputs that have changed
#'     })
#'
#'   }
#'
#'   shinyApp(ui, server)
#' }
track_usage <- function(storage_mode,
                        exclude_input_regex = NULL,
                        exclude_input_id = NULL,
                        on_unload = FALSE,
                        exclude_users = NULL,
                        get_user = NULL,
                        dependencies = TRUE,
                        session = getDefaultReactiveDomain()) {

  stopifnot(inherits(storage_mode, "shinylogs.storage_mode"))

  app_name <- basename(getwd())
  if (is.null(get_user))
    get_user <- get_user_
  if (!is.function(get_user))
    stop("get_user must be a function", call. = FALSE)
  user <- get_user(session)
  timestamp <- Sys.time()
  init_log <- data.frame(
    app = app_name,
    user = user,
    server_connected = get_timestamp(timestamp),
    stringsAsFactors = FALSE
  )
  storage_mode$appname <- app_name
  storage_mode$timestamp <- format(as.integer64(nanotime(timestamp)), scientific = FALSE)
  init_log$sessionid <- digest::digest(storage_mode$timestamp)

  if (isTRUE(dependencies)) {
    insertUI(
      selector = "body", where = "afterBegin",
      ui = singleton(tags$script(
        id = "shinylogs-tracking",
        type = "application/json",
        `data-for` = "shinylogs",
        toJSON(dropNulls(list(
          logsonunload = isTRUE(on_unload),
          exclude_input_regex = exclude_input_regex,
          exclude_input_id = exclude_input_id,
          sessionid = init_log$sessionid
        )), auto_unbox = TRUE, json_verbatim = TRUE)
      )),
      immediate = TRUE,
      session = session
    )
    insertUI(
      selector = "body", where = "afterBegin",
      ui = attachDependencies(
        x = tags$div(),
        value = list(
          localforage_dependencies(),
          dayjs_dependencies(),
          shinylogs_lf_dependencies()
        )
      ),
      immediate = FALSE,
      session = session
    )
  }


  if (isTRUE(storage_mode$console)) {
    observe({
      to_console(session$input$.shinylogs_browserData, init_log)
    })
    observe({
      to_console(session$input$.shinylogs_lastInput)
    })
  }


  onSessionEnded(
    fun = function() {
      init_log$server_disconnected <- get_timestamp(Sys.time())
      logs <- c(isolate(session$input$.shinylogs_input),
                isolate(session$input$.shinylogs_error),
                isolate(session$input$.shinylogs_output))
      browser_data <- isolate(session$input$.shinylogs_browserData)
      if (!is.null(browser_data)) {
        browser_data <- as.data.frame(browser_data)
        logs$session <- cbind(init_log, browser_data)
      } else {
        logs$session <- init_log
      }
      if (isTRUE(!user %in% exclude_users)) {
        write_logs(storage_mode, logs)
      }
    },
    session = session
  )
}


get_user_ <- function(session) {
  if (!is.null(session$user))
    return(session$user)
  user <- Sys.getenv("SHINYPROXY_USERNAME")
  if (user != "") {
    return(user)
  } else {
    getOption("shinylogs.default_user", default = Sys.info()[['user']])
  }
}



to_console <- function(obj, ...) {
  if (!is.null(obj)) {
    json <- jsonlite::toJSON(
      x = c(obj, ...),
      pretty = TRUE, auto_unbox = TRUE
    )
    print(json)
  }
}


