% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MutationProfiling.R
\name{calcObservedMutations}
\alias{calcObservedMutations}
\title{Count the number of observed mutations in a sequence.}
\usage{
calcObservedMutations(
  inputSeq,
  germlineSeq,
  regionDefinition = NULL,
  mutationDefinition = NULL,
  ambiguousMode = c("eitherOr", "and"),
  returnRaw = FALSE,
  frequency = FALSE
)
}
\arguments{
\item{inputSeq}{input sequence. IUPAC ambiguous characters for DNA are 
supported.}

\item{germlineSeq}{germline sequence. IUPAC ambiguous characters for DNA 
are supported.}

\item{regionDefinition}{\link{RegionDefinition} object defining the regions
and boundaries of the Ig sequences. Note, only the part of
sequences defined in \code{regionDefinition} are analyzed.
If NULL, mutations are counted for entire sequence.}

\item{mutationDefinition}{\link{MutationDefinition} object defining replacement
and silent mutation criteria. If \code{NULL} then 
replacement and silent are determined by exact 
amino acid identity.}

\item{ambiguousMode}{whether to consider ambiguous characters as 
\code{"either or"} or \code{"and"} when determining and 
counting the type(s) of mutations. Applicable only if
\code{inputSeq} and/or \code{germlineSeq} 
contain(s) ambiguous characters. One of 
\code{c("eitherOr", "and")}. Default is \code{"eitherOr"}.}

\item{returnRaw}{return the positions of point mutations and their 
corresponding mutation types, as opposed to counts of 
mutations across positions. Also returns the number of 
bases used as the denominator when calculating frequency. 
Default is \code{FALSE}.}

\item{frequency}{\code{logical} indicating whether or not to calculate
mutation frequencies. The denominator used is the number 
of bases that are not one of "N", "-", or "." in either 
the input or the germline sequences. If set, this 
overwrites \code{returnRaw}. Default is \code{FALSE}.}
}
\value{
For \code{returnRaw=FALSE}, an \code{array} with the numbers of replacement (R) 
          and silent (S) mutations. 
          
          For \code{returnRaw=TRUE}, a list containing 
          \itemize{
               \item \code{$pos}: A data frame whose columns (\code{position}, \code{r}, 
                     \code{s}, and \code{region}) indicate, respecitively, the nucleotide 
                     position, the number of R mutations at that position, the number of S 
                     mutations at that position, and the region in which that nucleotide
                     is in.
               \item \code{$nonN}: A vector indicating the number of bases in regions 
                     defined by \code{regionDefinition} (excluding non-triplet overhang, 
                     if any) that are not one of "N", "-", or "." in either the 
                     \code{inputSeq} or \code{germlineSeq}.
          }
          
          For \code{frequency=TRUE}, regardless of \code{returnRaw}, an \code{array} 
          with the frequencies of replacement (R) and silent (S) mutations.
}
\description{
\code{calcObservedMutations} determines all the mutations in a given input sequence 
compared to its germline sequence.
}
\details{
\strong{Each mutation is considered independently in the germline context}. For illustration,
consider the case where the germline is \code{TGG} and the observed is \code{TAC}.
When determining the mutation type at position 2, which sees a change from \code{G} to 
\code{A}, we compare the codon \code{TGG} (germline) to \code{TAG} (mutation at position
2 independent of other mutations in the germline context). Similarly, when determining 
the mutation type at position 3, which sees a change from \code{G} to \code{C}, we 
compare the codon \code{TGG} (germline) to \code{TGC} (mutation at position 3 independent 
of other mutations in the germline context).

If specified, only the part of \code{inputSeq} defined in \code{regionDefinition} is 
analyzed. For example, when using the default \link{IMGT_V} definition, then mutations 
in positions beyond 312 will be ignored. Additionally, non-triplet overhang at the 
sequence end is ignored.

Only replacement (R) and silent (S) mutations are included in the results. \strong{Excluded}
are: 
\itemize{
     \item Stop mutations
     
           E.g.: the case where \code{TAGTGG} is observed for the germline \code{TGGTGG}.
           
     \item Mutations occurring in codons where one or both of the observed and the 
           germline involve(s) one or more of "N", "-", or ".".
           
           E.g.: the case where \code{TTG} is observed for the germline being any one of 
           \code{TNG}, \code{.TG}, or \code{-TG}. Similarly, the case where any one of 
           \code{TTN}, \code{TT.}, or \code{TT-} is observed for the germline \code{TTG}.
           
}
In other words, a result that is \code{NA} or zero indicates absence of R and S mutations, 
not necessarily all types of mutations, such as the excluded ones mentioned above.

\code{NA} is also returned if \code{inputSeq} or \code{germlineSeq} is shorter than 3
nucleotides.
}
\section{Ambiguous characters}{

When there are ambiguous characters present, the user could choose how mutations involving
ambiguous characters are counted through \code{ambiguousMode}. The two available modes 
are \code{"eitherOr"} and \code{"and"}. 
\itemize{
     \item With \code{"eitherOr"}, ambiguous characters are each expanded but only 
           1 mutation is recorded. When determining the type of mutation, the 
           priority for different types of mutations, in decreasing order, is as follows:
           no mutation, replacement mutation, silent mutation, and stop mutation. 
           
           When counting the number of non-N, non-dash, and non-dot positions, each
           position is counted only once, regardless of the presence of ambiguous
           characters.
           
           As an example, consider the case where \code{germlineSeq} is \code{"TST"} and
           \code{inputSeq} is \code{"THT"}. Expanding \code{"H"} at position 2 in 
           \code{inputSeq} into \code{"A"}, \code{"C"}, and \code{"T"}, as well as 
           expanding \code{"S"} at position 2 in \code{germlineSeq} into \code{"C"} and 
           \code{"G"}, one gets:
           
           \itemize{
                \item \code{"TCT"} (germline) to \code{"TAT"} (observed): replacement
                \item \code{"TCT"} (germline) to \code{"TCT"} (observed): no mutation
                \item \code{"TCT"} (germline) to \code{"TTT"} (observed): replacement 
                \item \code{"TGT"} (germline) to \code{"TAT"} (observed): replacement 
                \item \code{"TGT"} (germline) to \code{"TCT"} (observed): replacement
                \item \code{"TGT"} (germline) to \code{"TTT"} (observed): replacement
           }
           
           Because "no mutation" takes priority over replacement mutation, the final 
           mutation count returned for this example is \code{NA} (recall that only R and 
           S mutations are returned). The number of non-N, non-dash, and non-dot 
           positions is 3.
           
     \item With \code{"and"}, ambiguous characters are each expanded and mutation(s)
           from all expansions are recorded.
           
           When counting the number of non-N, non-dash, and non-dot positions, if a 
           position contains ambiguous character(s) in \code{inputSeq} and/or 
           \code{germlineSeq}, the count at that position is taken to be the total 
           number of combinations of germline and observed codons after expansion.
           
           Using the same example from above, the final result returned for this example
           is that there are 5 R mutations at position 2. The number of non-N, non-dash,
           and non-dot positions is 8, since there are 6 combinations stemming from 
           position 2 after expanding the germline codon (\code{"TST"}) and the observed 
           codon (\code{"THT"}).
}
}

\examples{
# Use an entry in the example data for input and germline sequence
data(ExampleDb, package="alakazam")
in_seq <- ExampleDb[["sequence_alignment"]][100]
germ_seq <-  ExampleDb[["germline_alignment_d_mask"]][100]

# Identify all mutations in the sequence
ex1_raw <- calcObservedMutations(in_seq, germ_seq, returnRaw=TRUE)
# Count all mutations in the sequence
ex1_count <- calcObservedMutations(in_seq, germ_seq, returnRaw=FALSE)
ex1_freq <- calcObservedMutations(in_seq, germ_seq, returnRaw=FALSE, frequency=TRUE)
# Compare this with ex1_count
table(ex1_raw$pos$region, ex1_raw$pos$r)[, "1"]
table(ex1_raw$pos$region, ex1_raw$pos$s)[, "1"]
# Compare this with ex1_freq
table(ex1_raw$pos$region, ex1_raw$pos$r)[, "1"]/ex1_raw$nonN
table(ex1_raw$pos$region, ex1_raw$pos$s)[, "1"]/ex1_raw$nonN

# Identify only mutations the V segment minus CDR3
ex2_raw <- calcObservedMutations(in_seq, germ_seq, 
                                regionDefinition=IMGT_V, returnRaw=TRUE)
# Count only mutations the V segment minus CDR3
ex2_count <- calcObservedMutations(in_seq, germ_seq, 
                                  regionDefinition=IMGT_V, returnRaw=FALSE)
ex2_freq <- calcObservedMutations(in_seq, germ_seq, 
                                 regionDefinition=IMGT_V, returnRaw=FALSE,
                                 frequency=TRUE)
# Compare this with ex2_count
table(ex2_raw$pos$region, ex2_raw$pos$r)[, "1"]
table(ex2_raw$pos$region, ex2_raw$pos$s)[, "1"]                              
# Compare this with ex2_freq
table(ex2_raw$pos$region, ex2_raw$pos$r)[, "1"]/ex2_raw$nonN     
table(ex2_raw$pos$region, ex2_raw$pos$s)[, "1"]/ex2_raw$nonN                                       

# Identify mutations by change in hydropathy class
ex3_raw <- calcObservedMutations(in_seq, germ_seq, regionDefinition=IMGT_V,
                                mutationDefinition=HYDROPATHY_MUTATIONS, 
                                returnRaw=TRUE)
# Count mutations by change in hydropathy class
ex3_count <- calcObservedMutations(in_seq, germ_seq, regionDefinition=IMGT_V,
                                  mutationDefinition=HYDROPATHY_MUTATIONS, 
                                  returnRaw=FALSE)
ex3_freq <- calcObservedMutations(in_seq, germ_seq, regionDefinition=IMGT_V,
                                 mutationDefinition=HYDROPATHY_MUTATIONS, 
                                 returnRaw=FALSE, frequency=TRUE)
# Compre this with ex3_count
table(ex3_raw$pos$region, ex3_raw$pos$r)[, "1"]
table(ex3_raw$pos$region, ex3_raw$pos$s)[, "1"]
# Compare this with ex3_freq
table(ex3_raw$pos$region, ex3_raw$pos$r)[, "1"]/ex3_raw$nonN                                        
table(ex3_raw$pos$region, ex3_raw$pos$s)[, "1"]/ex3_raw$nonN                                        
                                
}
\seealso{
See \link{observedMutations} for counting the number of observed mutations 
in a \code{data.frame}.
}
