#' seminr estimate_pls() function
#'
#' Estimates a pair of measurement and structural models using PLS-SEM, with optional estimation methods
#'
#' @param data A \code{dataframe} containing the manifest measurement items in named columns.
#'
#' The pair of measurement and structural models can optionally be specified as separate model objects
#'
#' @param measurement_model An optional \code{measurement_model} object representing the outer/measurement model,
#'   as generated by \code{constructs}.
#'
#' @param structural_model An optional \code{smMatrix} object representing the inner/structural model,
#'   as generated by \code{relationships}.
#'
#' The pair of measurement and structural models can also be specified as a single \code{specified_model} object
#'
#' @param model An optional \code{specified_model} object containing both the the outer/measurement and inner/structural models,
#'   as generated by \code{specify_model}.
#'
#' @param inner_weights Function that implements inner weighting scheme:
#'   \code{path_weighting} (default) or \code{path_factorial} can be used.
#'
#' @param missing Function that replaces missing values.
#'   \code{mean_replacement} is default.
#'
#' @param missing_value Value in dataset that indicates missing values.
#'   NA is used by default.
#'
#' @return A list of the estimated parameters for the SEMinR model including:
#'  \item{meanData}{A vector of the indicator means.}
#'  \item{sdData}{A vector of the indicator standard deviations}
#'  \item{mmMatrix}{A Matrix of the measurement model relations.}
#'  \item{smMatrix}{A Matrix of the structural model relations.}
#'  \item{constructs}{A vector of the construct names.}
#'  \item{mmVariables}{A vector of the indicator names.}
#'  \item{outer_loadings}{The matrix of estimated indicator loadings.}
#'  \item{outer_weights}{The matrix of estimated indicator weights.}
#'  \item{path_coef}{The matrix of estimated structural model relationships.}
#'  \item{iterations}{A numeric indicating the number of iterations required before the algorithm converged.}
#'  \item{weightDiff}{A numeric indicating the minimum weight difference between iterations of the algorithm.}
#'  \item{construct_scores}{A matrix of the estimated construct scores for the PLS model.}
#'  \item{rSquared}{A matrix of the estimated R Squared for each construct.}
#'  \item{inner_weights}{The inner weight estimation function.}
#'  \item{data}{A matrix of the data upon which the model was estimated (INcluding interactions.}
#'  \item{rawdata}{A matrix of the data upon which the model was estimated (EXcluding interactions.}
#'  \item{measurement_model}{The SEMinR measurement model specification.}
#'
#' @usage
#' estimate_pls(data,
#'              measurement_model = NULL, structural_model = NULL, model = NULL,
#'              inner_weights = path_weighting,
#'              missing = mean_replacement,
#'              missing_value = NA)
#'
#' @seealso \code{\link{specify_model}} \code{\link{relationships}} \code{\link{constructs}} \code{\link{paths}} \code{\link{interaction_term}}
#'          \code{\link{bootstrap_model}}
#'
#' @examples
#' mobi <- mobi
#'
#' #seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'              reflective("Image",        multi_items("IMAG", 1:5)),
#'              reflective("Expectation",  multi_items("CUEX", 1:3)),
#'              reflective("Quality",      multi_items("PERQ", 1:7)),
#'              reflective("Value",        multi_items("PERV", 1:2)),
#'              reflective("Satisfaction", multi_items("CUSA", 1:3)),
#'              reflective("Complaints",   single_item("CUSCO")),
#'              reflective("Loyalty",      multi_items("CUSL", 1:3))
#'            )
#' #seminr syntax for creating structural model
#' mobi_sm <- relationships(
#'   paths(from = "Image",        to = c("Expectation", "Satisfaction", "Loyalty")),
#'   paths(from = "Expectation",  to = c("Quality", "Value", "Satisfaction")),
#'   paths(from = "Quality",      to = c("Value", "Satisfaction")),
#'   paths(from = "Value",        to = c("Satisfaction")),
#'   paths(from = "Satisfaction", to = c("Complaints", "Loyalty")),
#'   paths(from = "Complaints",   to = "Loyalty")
#' )
#'
#' mobi_pls <- estimate_pls(data = mobi,
#'                          measurement_model = mobi_mm,
#'                          structural_model = mobi_sm,
#'                          missing = mean_replacement,
#'                          missing_value = NA)
#'
#' summary(mobi_pls)
#' plot_scores(mobi_pls)
#' @export
estimate_pls <- function(data, measurement_model=NULL, structural_model=NULL, model=NULL, inner_weights = path_weighting, missing = mean_replacement, missing_value = NA) {
  message("Generating the seminr model")
  data[data == missing_value] <- NA
  data <- missing(data)
  data <- stats::na.omit(data)
  rawdata <- data

  # Extract model specifications
  specified_model <- extract_models(model, measurement_model, structural_model)
  measurement_model <- specified_model$measurement_model
  structural_model <- specified_model$structural_model

  # Generate first order model if necessary
  HOCs <- HOCs_in_sm(measurement_model, structural_model)

  if ( length(HOCs)>0 ) {
    HOM <- prepare_higher_order_model(data = data,
                                      sm = structural_model,
                                      mm = measurement_model,
                                      inners = inner_weights,
                                      HOCs = HOCs)
    measurement_model <- HOM$mm
    structural_model <- HOM$sm
    data <- HOM$data
    first_stage_model <- HOM$first_stage_model
  }

  processed_measurements <- process_interactions(measurement_model, data, structural_model, inner_weights)
  mmMatrix <- processed_measurements$mmMatrix
  data <- processed_measurements$data

  # warning if the model is incorrectly specified
  warnings(mmMatrix, data, structural_model)

  # Make a named list of construct measurement_mode functions
  measurement_mode_scheme <- sapply(unique(c(structural_model[,1], structural_model[,2])), get_measure_mode, mmMatrix, USE.NAMES = TRUE)

  # Run the model in simplePLS
  seminr_model = seminr::simplePLS(obsData = data, smMatrix = structural_model, mmMatrix = mmMatrix, inner_weights = inner_weights, measurement_mode_scheme = measurement_mode_scheme)
  seminr_model$data <- data
  seminr_model$rawdata <- rawdata
  seminr_model$measurement_model <- measurement_model

  # Correct for Bias in Reflective models using PLS Consistent
  seminr_model <- model_consistent(seminr_model)

  if ( length(HOCs)>0 ) {
    # Append return list with first stage model and
    seminr_model$first_stage_model <- first_stage_model

    # Combine first and second stage measurement model matrices
    new_mm <- combine_first_order_second_order_matrices(model1 = first_stage_model, model2 = seminr_model, mmMatrix)
    seminr_model$outer_loadings <- new_mm$new_outer_loadings
    seminr_model$outer_weights <- new_mm$new_outer_weights
  }

  class(seminr_model) <- c("pls_model", "seminr_model")
  return(seminr_model)
}

