#############################################################################
## package 'secr'
## join.R

## join returns single-session object from list of inputs

## 2016-10-10 secr3
## 2017-01-29 telemetry fixes
## 2017-12-23
#############################################################################

join <- function (object, remove.dupl.sites = TRUE, tol = 0.001,
                  intervals = NULL, sessionlabels = NULL) {

    ####################################################################
    onesession <- function (sess) {
        ## form CH as a dataframe
        CH <- object[[sess]]
        newID <- animalID(CH)
        newocc <- occasion(CH) + before[sess]
        df <- data.frame(newID = newID, newocc = newocc, newtrap = newocc,   # dummy to hold place
                         alive = alive(CH), sess = rep(sess, length(newID)),
                         stringsAsFactors = FALSE)
        if (is.null(traps(CH)))
            df$newtrap <-  rep(1,nrow(df))
        else
            df$newtrap <- trap(CH)
        if (!is.null(xy(CH)))
            df[,c('x','y')] <- xy(CH)
        if (!is.null(signal(CH)))
            df[,'signal'] <- signal(CH)

        ## add all-zero (sighting-only) records as required,
        ## otherwise leave unchanged
        addzerodf (df, CH, sess)
    }
    ####################################################################

    condition.usage <- function (trp, i) {
        if (!is.null(trp)) {
            us <- matrix(0, nrow=nrow(trp), ncol=nnewocc)
            if ('telemetry' %in% detector(trp)) {
                occasions <- outputdetector == 'telemetry'
            }
            else {
                s1 <- c(1, cumsum(nocc)+1)[i]
                s2 <- cumsum(nocc)[i]
                occasions <- s1:s2
            }
            if (is.null(usage(trp)))
                us[,occasions] <- 1
            else
                us[,occasions] <- usage(trp)
            usage(trp) <- us
        }
        trp
    }
    ####################################################################
    ## preparing for merge when traps vary... 2015-10-29
    condition.sightings <- function (CH, i, type = 'Tu') {
        T <- attr(CH, type)
        if (!is.null(T)) {
            if (is.matrix(T)) {
                Tnew <- matrix(0, nrow=nrow(traps(CH)), ncol=nnewocc)
                s1 <- c(1, cumsum(nocc)+1)[i]
                s2 <- cumsum(nocc)[i]
                Tnew[,s1:s2] <- T
                attr(CH, type) <- Tnew
            }
        }
        CH
    }
    ####################################################################

    ## mainline
    if (!ms(object) | any(sapply(object, class) != 'capthist'))
        stop("requires multi-session capthist object or list of ",
             "single-session capthist")
    detectorlist <- lapply(object, expanddet)
    outputdetector <- unlist(detectorlist)

    nsession <- length(object)
    nocc <- sapply(object, ncol)
    names(nocc) <- NULL
    nnewocc <- sum(nocc)
    ## cumulative number of preceding occasions
    before <- c(0, cumsum(nocc)[-nsession])

    ##------------------------------------------------------------------
    ## combine capthist as one long dataframe
    df <- lapply(1:nsession, onesession)
    df <- do.call(rbind, df)
    n <- length(unique(df$newID))

    ##------------------------------------------------------------------
    ## resolve traps
    ## first check whether all the same (except usage)
    if (!is.null(traps(object))) {
        temptrp <- lapply(traps(object), function(x) {usage(x) <- NULL; x})
        sametrp <- all(sapply(temptrp[-1], identical, temptrp[[1]]))
        telemetrytrap <- function (ch) {
            if ('telemetry' %in% detector(traps(ch))) dim(ch)[3] else 0
        }
        if (sametrp & remove.dupl.sites) {
            newtraps <- temptrp[[1]]
            class(newtraps) <- c("traps", "data.frame")
            if (length(usage(traps(object))) > 0)
                usage(newtraps) <- do.call(cbind, usage(traps(object)))
            ## df$newtrap unchanged
        }
        else {
            temptrp <- traps(object)
            if ('telemetry' %in% outputdetector) {
                # drop all notional 'telemetry' traps and replace at end
                ttraps0 <- sapply(object, telemetrytrap)
                ttraps <- ttraps0[ttraps0>0]
                teltrapno <- ttraps[length(ttraps)] # use last
                df$newtrap[df$newtrap %in% ttraps] <- teltrapno
                dropteltrap <- function (trps, teltrap) {
                    if (teltrap>0) {
                        if (nrow(trps)==1)
                            NULL
                        else
                            subset(trps, (1:nrow(trps)) != teltrap)
                    }
                    else
                        trps
                }
                newteltrap <- subset(temptrp[[1]],1)
                temptrp <- mapply(dropteltrap, temptrp, ttraps0)
                rownames(newteltrap) <- teltrapno
                temptrp <- c(temptrp, list(newteltrap))
            }
            else {
                df$newtrap <- paste(df$newtrap,df$sess, sep=".")
            }

            temptrp <- mapply(condition.usage, temptrp, 1:length(temptrp), SIMPLIFY = FALSE)
            temptrp <- temptrp[!sapply(temptrp, is.null)]
            newtraps <- do.call(rbind, c(temptrp, renumber = FALSE, checkdetector = FALSE))
            detector(newtraps) <- outputdetector
            class(newtraps) <- c("traps", "data.frame")
        }
        if (all(outputdetector %in% .localstuff$polydetectors))
            df$newtrap <- factor(df$newtrap)
        else
            df$newtrap <- factor(df$newtrap, levels=rownames(newtraps))
    }
    else {
        sametrp <- FALSE
    }
    ##------------------------------------------------------------------
    ## ensure retain all occasions
    df$newocc <- factor(df$newocc, levels = 1:nnewocc)

    ##------------------------------------------------------------------
    ## construct new capthist matrix or array from positive detections
    tempnew <- table(df$newID, df$newocc, df$newtrap, useNA = "no")
    alivesign <- tapply(df$alive, list(df$newID,df$newocc,df$newtrap),all)
    alivesign[is.na(alivesign)] <- TRUE
    alivesign <- alivesign * 2 - 1
    tempnew <- tempnew * alivesign

    ##------------------------------------------------------------------
    ## pile on the attributes...
    class(tempnew) <- 'capthist'
    if (!is.null(traps(object))) traps(tempnew) <- newtraps
    session(tempnew) <- 1
    neworder <- order (df$newocc, df$newID, df$newtrap)

    ##------------------------------------------------------------------
    ## concatenate marking-and-resighting-occasion vectors
    tempmarkocc <- unlist(markocc(traps(object)))
    if (!is.null(tempmarkocc)) {
        names(tempmarkocc) <- NULL
        markocc(traps(tempnew)) <- tempmarkocc
    }

    ##------------------------------------------------------------------
    ## unmarked and nonID sightings
    ## not yet implemented for varying traps
    if (sametrp & remove.dupl.sites) {
        ## retain unmarked sightings and nonID sightings if present
        ## ignore if NULL
        Tu <- Tu(object)
        if (!is.null(Tu[[1]])) {
            if (!all(sapply(Tu, is.matrix)))
                Tu(tempnew) <- do.call(sum, Tu)
            else
                Tu(tempnew) <- do.call(cbind, Tu)
        }

        Tm <- Tm(object)
        if (!is.null(Tm[[1]])) {
            if (!all(sapply(Tm, is.matrix)))
                Tm(tempnew) <- do.call(sum, Tm)
            else
                Tm(tempnew) <- do.call(cbind, Tm)
        }
    }
    else {
        ## Tu, Tm not ready yet
        if (!is.null(Tu(object[[1]])) | !is.null(Tm(object[[1]])))
            stop ("join does not yet merge sighting matrices when traps vary")
    }

    ##------------------------------------------------------------------
    ## covariates, xy, signal attributes
    if (!is.null(covariates(object))) {
        tempcov <- do.call(rbind, covariates(object))
        if (!is.null(tempcov)) {
            IDcov <- unlist(lapply(object,rownames))
            ## use first match
            tempcov <- tempcov[match(rownames(tempnew), IDcov),,drop = FALSE]
            rownames(tempcov) <- rownames(tempnew)
            covariates(tempnew) <- tempcov
        }
    }

    ##------------------------------------------------------------------
    ## telemetry fixes

    if ('telemetry' %in% outputdetector) {
        oldtelem <- lapply(object, telemetryxy)
        telnames <- unique(unlist(lapply(oldtelem,names)))
        newtelem <- vector('list', length(telnames))
        names(newtelem) <- telnames
        for (id in telnames) {
            newtelem[[id]] <- do.call(rbind, lapply(oldtelem, '[[', id))
        }
        telemetryxy(tempnew) <- newtelem
    }

    ##------------------------------------------------------------------
    ## negotiate problem that all-zero histories have no xy, signal
    tempdf <- df[neworder,, drop = FALSE]
    if (!is.null(df$x)) {
        xy(tempnew) <- tempdf[!is.na(tempdf$newocc), c('x','y')]
    }
    if (!is.null(df$signal))
        signal(tempnew) <- tempdf[!is.na(tempdf$newocc),'signal']

    ##------------------------------------------------------------------
    ## purge duplicate sites, if requested
    if (remove.dupl.sites & !sametrp)
        tempnew <- reduce(tempnew, span=tol, dropunused = FALSE, verify = FALSE)

    ## remember previous structure, for MARK-style robust design
    tmpintervals <- unlist(sapply(nocc, function(x) c(1,rep(0,x-1))))[-1]
    if (!is.null(intervals)) {
        if (length(intervals) != sum(tmpintervals>0))
            stop("invalid intervals argument")
        tmpintervals[tmpintervals>0] <- intervals
    }
    else if (!is.null(attr(object, 'intervals'))) {
        attrintervals <- attr(object, 'intervals')
        if (length(attrintervals) != sum(tmpintervals>0))
            stop("invalid intervals attribute")
        tmpintervals[tmpintervals>0] <- attrintervals
    }
    if (is.null(sessionlabels)) sessionlabels <- sessionlabels(object)
    if (is.null(sessionlabels)) sessionlabels <- session(object)
    intervals(tempnew) <- tmpintervals
    sessionlabels(tempnew) <- sessionlabels

    ##------------------------------------------------------------------

    tempnew

}

unjoin <- function (object, intervals, ...) {
    if (missing(intervals) & is.null(attr(object,'intervals')))
        stop ("requires 'intervals' to define sessions")
    if (missing(intervals) )
        intervals <- attr(object,"intervals")
    session <- c(0,cumsum(intervals>0))+1
    nsess <- max(session)
    if (nsess<2) {
        warning ("intervals define only one session")
        return(object)
    }
    newobj <- vector(mode='list', length=nsess)
    for (sess in 1:nsess) {
        newobj[[sess]] <- subset(object, occasions = (session==sess), ...)
    }
    class (newobj) <- c('capthist', 'list')
    session(newobj) <- 1:nsess
    return(newobj)
}

