% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simSP.R
\name{simSP}
\alias{simSP}
\title{Similarity measure between snow profile pairs}
\usage{
simSP(
  ref,
  qw,
  gtype_distMat = sim2dist(grainSimilarity_evaluate(triag = FALSE)),
  type = "HerlaEtAl2021",
  nonMatchedSim = 0,
  nonMatchedThickness = 10,
  verbose = FALSE,
  returnDF = FALSE,
  apply_scalingFactor = FALSE
)
}
\arguments{
\item{ref}{snowprofile object 1}

\item{qw}{snowprofile object 2 (matched layers need to be on the same height grid of ref)}

\item{gtype_distMat}{a distance matrix that stores \strong{distance} information of grain types (\emph{Be careful} to convert
similarities, as in \link{grainSimilarity_evaluate}, into dissimilarities with \link{sim2dist}.)}

\item{type}{the similarity measure can be computed in several different ways (of sophistication). See Details section.
Possible choices
\itemize{
\item \code{simple}
\item \code{HerlaEtAl2021} (= \code{simple2})
\item \code{tsa_WLdetection} & \code{rta_WLdetection}
\item \code{layerwise} & \code{rta_scaling}
}}

\item{nonMatchedSim}{sets the similarity value of non-matched layers \verb{[0, 1]}. "indifference" = 0.5, penalty < 0.5.
Note that \link{dtwSP} sets the same value and overrides the default value in this function!}

\item{nonMatchedThickness}{If \code{NA}, every unique non-matched layer (i.e., contiguous resampled layers with identical properties)
contributes to the overall similarity by 1 x \code{nonMatchedSim}. In that case, 5cm of non-matched new snow has the same effect on
the overall similarity as 50cm of non-matched new snow. To make the effect of non-matched layers dependent on the layer thickness,
provide a positive number to \code{nonMatchedThickness}. For \code{nonMatchedThickness = 10}, every 10cm of a unique non-matched layer
contribute to the overall similarity by 1 x \code{nonMatchedSim}. So, 50cm of non-matched new snow would contribute 5 times stronger
than 5cm of non-matched new snow.
Note that \link{dtwSP} sets the same value and overrides the default value in this function!}

\item{verbose}{print similarities of different grain classes to console? default FALSE}

\item{returnDF}{additionally return the similarities of the grain classes as data.frame (analogously to verbose);
the return object then has the fields \verb{$sim} and \verb{$simDF}}

\item{apply_scalingFactor}{Only applicable to \code{type = layerwise}: \code{TRUE} or \code{FALSE}, see Details.}
}
\value{
Either a scalar similarity between \verb{[0, 1]} with 1 referring to the two profiles being identical, or
(if \code{returnDF} is TRUE) a list with the elements \verb{$sim} and \verb{$simDF}.
}
\description{
This function calculates a similarity measure for two snow profiles
that have been aligned onto the same height grid (either through DTW or resampling).
If one profile contains more layers than the other one, the layers with a non-matched height
represent missing layers and will be treated accordingly.
The similarity measure is compatible with top-down alignments and is symmetric with respect to its inputs, i.e.
\code{simSP(P1, P2) == simSP(P2, P1)}. \strong{Several different approaches of computing the measure have been implemented by now,
see Details below.}
}
\details{
The first several implementation types (\strong{simple}, \strong{HerlaEtAl2021}, \strong{tsa_WLdetection}, \strong{rta_WLdetection}) represent different flavours of the approach detailed in
Herla et al (2021). In essence, they are a simple approach to incorporate avalanche hazard relevant characteristics into the score by
computing the score as arithmetic mean of 4 different grain type classes:
\itemize{
\item weak layers (wl): SH and DH
\item new snow (pp): PP and DF
\item crusts (cr): MFcr and IF
\item bulk: the rest (i.e., predominantly RG, FC, FCxr --- MF falls also in here, will maybe be adjusted in future.)
}

Additionally, for classes wl and cr, vertical windows are computed to weigh layers more heavily that have no other wl or cr
grain types in their neighborhood.

Type \strong{simple} deviates from \emph{simple2} (= \emph{HerlaEtAl2021}) by computing the aforementioned vertical windows based on heuristic
depth ranges (i.e., Surface--30cm depth--80cm depth--150cm depth--Ground). It is otherwise identical to the \strong{simple2}
type, which computes as many numbers of equidistant vertical windows as number of wl or cr are present in the profile.

Type \strong{tsa_WLdetection} employs a similar approach as \emph{simple}, but it identifies weak layers (wl) based on the Threshold Sum Approach
(>= 5 TSA, lemons, german 'Nieten'). Therefore, the original profiles need to contain grain size information, which allows you to pre-compute the lemons
for all layers (additionally to the otherwise
necessary gain type and hardness information). It is thus more targeted to simulated profiles or detailed manual profiles of very high quality.
While the former two types neglect hardness information of wl and cr classes, this type does not.
Type \strong{rta_WLdetection} works analogous, but uses RTA instead of TSA and a threshold of >= 0.8.

Unlike the former types, \strong{layerwise} applies no weighting at all if used as per default. That means that the similarity of each individual layer
contributes equally to the overall similarity measure. It is, however, very flexible in that any custom scaling factor can be applied to each layer. The resulting similarity score is then computed by
\itemize{
\item simSP = sum(sim * scalingFactor) / sum(scalingFactor),
}

where the denominator ensures that the resulting score will be within \verb{[0, 1]}. If you want to explore your own scaling approach,
both input snow profiles need to contain a column called \verb{$layers$scalingFactor} that store the desired factor.
Type \strong{rta_scaling} is a special case of \code{layerwise}, where the scaling is determined by the relative lemons of each layer (RTA, see Monti et al 2013).

\strong{NOTE} that for all types that include TSA/RTA values, these values need to be computed \emph{prior to aligning} the profiles
(and therefore need to be present in the profiles provided to this function!)
}
\examples{

## first align two profiles, then assess the similarity of the aligned profiles
alignment <- dtwSP(SPpairs$A_modeled, SPpairs$A_manual)
SIM <- simSP(alignment$queryWarped, alignment$reference, verbose = TRUE)

## similarity of identical profiles
SIM <- simSP(alignment$queryWarped, alignment$queryWarped, verbose = TRUE)

## non-matched layers become apparent here:
alignment <- plotSPalignment(SPpairs$C_day1, SPpairs$C_day2, keep.alignment = TRUE,
                             rescale2refHS = FALSE, checkGlobalAlignment = FALSE)
simSP(alignment$queryWarped, alignment$reference, nonMatchedSim = 0.5)
## smaller similarity score due to 'penalty' of non-matched layers:
simSP(alignment$queryWarped, alignment$reference, nonMatchedSim = 0)
## even smaller similarity score due to higher impact of non-matched layer thickness:
simSP(alignment$queryWarped, alignment$reference, nonMatchedSim = 0, nonMatchedThickness = 1)

## detect WL based on lemons (instead of grain type alone):
P1 <- computeTSA(SPpairs$D_generalAlignment1)
P2 <- computeTSA(SPpairs$D_generalAlignment2)
alignment <- dtwSP(P1, P2, simType = "tsa_wldetection")
# sim based on WL-detection with TSA:
simSP(alignment$queryWarped, alignment$reference, type = "tsa_wldetection", verbose = TRUE)
# sim solely based on grain type, neglecting TSA information
simSP(alignment$queryWarped, alignment$reference, type = "simple", verbose = TRUE)

## RTA scaling type
P1 <- computeRTA(P1)
P2 <- computeRTA(P2)
alignment <- dtwSP(P1, P2, simType = "rta_scaling")
# sim based on scaling with RTA
simSP(alignment$queryWarped, alignment$reference, type = "rta_scaling")
# sim based on WL-detection with RTA
simSP(alignment$queryWarped, alignment$reference, type = "rta_wldetection")
# sim based on WL-detection with TSA
simSP(alignment$queryWarped, alignment$reference, type = "tsa_wldetection")

## layerwise similarity (i) unscaled...
simSP(alignment$queryWarped, alignment$reference, type = "layerwise", verbose = TRUE)

##... or (ii) with custom scaling factor (example only illustrative)
alignment$queryWarped$layers$scalingFactor <- 0.1
alignment$queryWarped$layers$scalingFactor[findPWL(alignment$queryWarped)] <- 1
alignment$reference$layers$scalingFactor <- 0.1
alignment$reference$layers$scalingFactor[findPWL(alignment$reference)] <- 1
simSP(alignment$queryWarped, alignment$reference, type = "layerwise",
      apply_scalingFactor = TRUE, verbose = TRUE)

}
\references{
Herla, F., Horton, S., Mair, P., & Haegeli, P. (2021). Snow profile alignment and similarity assessment for aggregating, clustering,
and evaluating of snowpack model output for avalanche forecasting. Geoscientific Model Development, 14(1), 239–258. https://doi.org/10.5194/gmd-14-239-2021

Monti, F., & Schweizer, J. (2013). A relative difference approach to detect potential weak layers within a snow profile.
Proceedings of the 2013 International Snow Science Workshop, {G}renoble, {F}rance, 339–343. Retrieved from https://arc.lib.montana.edu/snow-science/item.php?id=1861
}
