% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wilcoxtest.R
\name{wilcoxtest}
\alias{wilcoxtest}
\title{Calibrated Wilcoxon rank sum and signed rank tests}
\usage{
wilcoxtest(
  x,
  y = NULL,
  alternative = "two.sided",
  ties.break = "none",
  paired = FALSE,
  ...
)
}
\arguments{
\item{x, y}{two continuous variables.}

\item{alternative}{indicates the alternative hypothesis and must be one of "two.sided", "greater" or "less".}

\item{ties.break}{the method used to break ties in case there are ties in the x or y vectors. Can be \code{"none"} or \code{"random"}.}

\item{paired}{a logical value. If \code{paired=TRUE}, you must provide values for \code{x} and \code{y} (of same length)
and the paired test is implemented. If \code{paired=FALSE}, the paired test is implemented when \code{y} is null and
only \code{x} is provided and the two sample test (for independent variables) is implemented when both \code{x} and \code{y} are provided.}

\item{...}{it is possible to use a formula with or without specifying a dataset
from the commands \code{wilcoxtest(x~y)} or \code{wilcoxtest(x~y,data=dataset)} with dataset the name of the data.}
}
\value{
Returns the result of the test with its corresponding p-value and the value of the test statistic.
}
\description{
Compares the distribution between two random variables by testing if one variable tends to take larger (or smaller) values than the other. The test
works for independent and paired variables by using corrected versions of the Wilcoxon (or equivalently Mann-Whitney in the two-sample case) for one and two-sample tests.
}
\details{
For two independent samples, the null hypothesis for the corrected Wilcoxon (Mann-Whitney) test is: H0 Med(X-Y)=0 where Med represents the median.
The alternative is specified by the \code{alternative} argument: "\code{greater}" means that Med(X-Y)>0 and "\code{less}"
means that Med(X-Y)<0. The null hypothesis for the paired Wilcoxon test is: H0 Med(D1+D2)=0 where D1 is the difference
between X1 and Y1 taken on the same pair (same with D2 on a different pair). Both tests are asymptotically well calibrated in the sense that the rejection probability under the
null hypothesis is asymptotically equal to the level of the test.
}
\note{
The function can also be called using formulas: type \code{wilcoxtest(x~y,data)} with x the quantitative variable
and y a factor variable with two levels. The option \code{ties.break} handles ties in the Wilcoxon test. If \code{ties.break="none"} the ties are ignored, if
\code{ties.break="random"} they are randomly broken. For the Wilcoxon rank sum test the ties between the \code{x} and \code{y} are
detected and broken (but the ties inside the \code{x} and \code{y} vectors are not changed). For the signed rank test, the ties in the
vector \code{x-y} (or in the \code{x} vector in case \code{y=NULL}) are randomly broken.
}
\examples{
#Application on the Evans dataset
data(Evans)
#Description of this dataset is available in the lbreg package
with(Evans,wilcox.test(CHL[CDH==0],CHL[CDH==1]))
with(Evans,wilcoxtest(CHL[CDH==0],CHL[CDH==1]))
wilcoxtest(CHL~CDH,data=Evans) #using formulas
wilcoxtest(CHL~CDH,data=Evans,ties.break="random")
#the same test where ties are randomly broken

\donttest{
#For independent samples
n=100 #sample size
M=1000 #number of replications
testone=function(n){
X=runif(n,-0.5,0.5)
Y=rnorm(3*n,0,0.04)
list(test1=wilcoxtest(X,Y)$p.value,test2=wilcox.test(X,Y)$p.value)
#wilcox.test is the standard Wilcoxon test
}

#Simulation under the null hypothesis
#(note that P(X>Y)=0.5)
#Takes a few seconds to run
res1=res2=rep(NA,M)
for (i in 1:M)
{
result=testone(n)
res1[i]=result$test1
res2[i]=result$test2
}
mean(res1<0.05)
mean(res2<0.05)}
\donttest{
#For paired samples
#We use the value of the median of a Gamma distributed variable with shape
#parameter equal to 1/5 and scale parameter equal to 1. This value is
#computed from the command qgamma(shape=1/5, scale=1, 0.5)
n=100 #sample size
M=1000 #number of replications
testone=function(n){
D=rgamma(n,shape=1/10,scale=1)-qgamma(shape=1/5, scale=1, 0.5)/2
list(test1=wilcoxtest(D,ties.break = "random")$p.value,test2=wilcox.test(D)$p.value)
#wilcox.test is the standard paired Wilcoxon test
}
#Simulation under the null hypothesis
#(note that Med(D_1+D_2)=0)
#Takes a few seconds to run
for (i in 1:M)
{
result=testone(n)
res1[i]=result$test1
res2[i]=result$test2
}
mean(res1<0.05)
mean(res2<0.05)}
}
\seealso{
\code{\link{cortest}}, \code{\link{indeptest}}, \code{\link{mediantest}}, \code{\link{vartest}}.
}
\keyword{test}
