% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/topic-errors.R
\name{topic-error-chaining}
\alias{topic-error-chaining}
\title{Including contextual information with error chains}
\description{
Error chaining is a mechanism for providing contextual information when an error occurs. There are multiple situations in which you might be able to provide context that is helpful to quickly understand the cause or origin of an error:
\itemize{
\item Mentioning the \emph{high level context} in which a low level error arised. E.g. chaining a low-level HTTP error to a high-level download error.
\item Mentioning the \emph{pipeline step} in which a user error occured. This is a major use-case for NSE interfaces in the tidyverse, e.g. in dplyr, tidymodels or ggplot2.
\item Mentioning the \emph{iteration context} in which a user error occurred. For instance, the input file when processing documents, or the iteration number or key when running user code in a loop.
}

Here is an example of a chained error from dplyr that shows the pipeline step (\code{mutate()}) and the iteration context (group ID) in which a function called by the user failed:\if{html}{\out{<div class="sourceCode r">}}\preformatted{add <- function(x, y) x + y

mtcars |>
  dplyr::group_by(cyl) |>
  dplyr::mutate(new = add(disp, "foo"))
#> Error in `dplyr::mutate()`:
#> ! Problem while computing `new = add(disp, "foo")`.
#> i The error occurred in group 1: cyl = 4.
#> Caused by error in `x + y`:
#> ! non-numeric argument to binary operator
}\if{html}{\out{</div>}}

In all these cases, there are two errors in play, chained together:
\enumerate{
\item The \strong{causal error}, which interrupted the current course of action.
\item The \strong{contextual error}, which expresses higher-level information when something goes wrong.
}

There may be more than one contextual error in an error chain, but there is always only one causal error.
}
\section{Rethrowing errors}{
To create an error chain, you must first capture causal errors when they occur. We recommend using \code{try_fetch()} instead of \code{tryCatch()} or \code{withCallingHandlers()}.
\itemize{
\item Compared to \code{tryCatch()}, \code{try_fetch()} fully preserves the context of the error. This is important for debugging because it ensures complete backtraces are reported to users (e.g. via \code{last_error()}) and allows \code{options(error = recover)} to reach into the deepest error context.
\item Compared to \code{withCallingHandlers()}, which also preserves the error context, \code{try_fetch()} is able to catch stack overflow errors on R versions >= 4.2.0 (unreleased at the time of writing).
}

In practice, \code{try_fetch()} works just like \code{tryCatch()}. It takes pairs of error class names and handling functions. To chain an error, simply rethrow it from an error handler by passing it as \code{parent} argument.

In this example, we'll create a \code{with_} function. That is, a function that sets up some configuration (in this case, chained errors) before executing code supplied as input:\if{html}{\out{<div class="sourceCode r">}}\preformatted{with_chained_errors <- function(expr) \{
  try_fetch(
    expr,
    error = function(cnd) \{
      abort("Problem during step.", parent = cnd)
    \}
  )
\}

with_chained_errors(1 + "")
#> Error in `with_chained_errors()`:
#> ! Problem during step.
#> Caused by error in `1 + ""`:
#> ! non-numeric argument to binary operator
}\if{html}{\out{</div>}}

Typically, you'll use this error helper from another user-facing function.\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_verb <- function(expr) \{
  with_chained_errors(expr)
\}

my_verb(add(1, ""))
#> Error in `with_chained_errors()`:
#> ! Problem during step.
#> Caused by error in `x + y`:
#> ! non-numeric argument to binary operator
}\if{html}{\out{</div>}}

Altough we have created a chained error, the error call of the contextual error is not quite right. It mentions the name of the error helper instead of the name of the user-facing function.

If you've read \ifelse{html}{\link[=topic-error-call]{Including function calls in error messages}}{\link[=topic-error-call]{Including function calls in error messages}}, you may suspect that we need to pass a \code{call} argument to \code{abort()}. That's exactly what needs to happen to fix the call and backtrace issues:\if{html}{\out{<div class="sourceCode r">}}\preformatted{with_chained_errors <- function(expr, call = caller_env()) \{
  try_fetch(
    expr,
    error = function(cnd) \{
      abort("Problem during step.", parent = cnd, call = call)
    \}
  )
\}
}\if{html}{\out{</div>}}

Now that we've passed the caller environment as \code{call} argument, \code{abort()} automatically picks up the correspondin function call from the execution frame:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_verb(add(1, ""))
#> Error in `my_verb()`:
#> ! Problem during step.
#> Caused by error in `x + y`:
#> ! non-numeric argument to binary operator
}\if{html}{\out{</div>}}
\subsection{Side note about missing arguments}{

\code{my_verb()} is implemented with a lazy evaluation pattern. The user input kept unevaluated until the error chain context is set up. A downside of this arrangement is that missing argument errors are reported in the wrong context:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_verb()
#> Error in `my_verb()`:
#> ! Problem during step.
#> Caused by error in `withCallingHandlers()`:
#> ! argument "expr" is missing, with no default
}\if{html}{\out{</div>}}

To fix this, simply require these arguments before setting up the chained error context, for instance with the \code{check_required()} input checker exported from rlang:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_verb <- function(expr) \{
  check_required(expr)
  with_chained_errors(expr)
\}

my_verb()
#> Error in `my_verb()`:
#> ! `expr` is absent but must be supplied.
}\if{html}{\out{</div>}}
}
}

\section{Taking full ownership of a causal error}{
It is also possible to completely take ownership of a causal error and rethrow it with a more user-friendly error message. In this case, the original error is completely hidden from the end user. Opting for his approach instead of chaining should be carefully considered because hiding the causal error may deprive users from precious debugging information.
\itemize{
\item In general, hiding \emph{user errors} (e.g. dplyr inputs) in this way is likely a bad idea.
\item It may be appropriate to hide low-level errors, e.g. replacing HTTP errors by a high-level download error. Similarly, tidyverse packages like dplyr are replacing low-level vctrs errors with higher level errors of their own crafting.
\item Hiding causal errors indiscriminately should likely be avoided because it may suppress information about unexpected errors. In general, rethrowing an unchained errors should only be done with specific error classes.
}

To rethow an error without chaining it, and completely take over the causal error from the user point of view, fetch it with \code{try_fetch()} and throw a new error. The only difference with throwing a chained error is that the \code{parent} argument is set to \code{NA}. You could also omit the \code{parent} argument entirely, but passing \code{NA} lets \code{abort()} know it is rethrowing an error from a handler and that it should hide the corresponding error helpers in the backtrace.\if{html}{\out{<div class="sourceCode r">}}\preformatted{with_own_scalar_errors <- function(expr, call = caller_env()) \{
  try_fetch(
    expr,
    vctrs_error_scalar_type = function(cnd) \{
      abort(
        "Must supply a vector.",
        parent = NA,
        error = cnd,
        call = call
      )
    \}
  )
\}

my_verb <- function(expr) \{
  check_required(expr)
  with_own_scalar_errors(
    vctrs::vec_assert(expr)
  )
\}

my_verb(env())
#> Error in `my_verb()`:
#> ! Must supply a vector.
}\if{html}{\out{</div>}}

When a low-level error is overtaken, it is good practice to store it in the high-level error object, so that it can be inspected for debugging purposes. In the snippet above, we stored it in the \code{error} field. Here is one way of accessing the original error by subsetting the object returned by \code{last_error()}:\if{html}{\out{<div class="sourceCode r">}}\preformatted{rlang::last_error()$error
#> <error/vctrs_error_scalar_type>
#> Error in `my_verb()`:
#> ! `expr` must be a vector, not an environment.
#> ---
#> Backtrace:
#>  1. rlang my_verb(env())
}\if{html}{\out{</div>}}
}

\section{Case study: Mapping with chained errors}{
One good use case for chained errors is adding information about the iteration state when looping over a set of inputs. To illustrate this, we'll implement a version of \code{map()} / \code{lapply()} that chains an iteration error to any captured user error.

Here is a minimal implementation of \code{map()}:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_map <- function(.xs, .fn, ...) \{
  out <- new_list(length(.xs))

  for (i in seq_along(.xs)) \{
    out[[i]] <- .fn(.xs[[i]], ...)
  \}

  out
\}

list(1, 2) |> my_map(add, 100)
#> [[1]]
#> [1] 101
#> 
#> [[2]]
#> [1] 102
}\if{html}{\out{</div>}}

With this implementation, the user has no idea which iteration failed when an error occurs:\if{html}{\out{<div class="sourceCode r">}}\preformatted{list(1, "foo") |> my_map(add, 100)
#> Error in x + y: non-numeric argument to binary operator
}\if{html}{\out{</div>}}
\subsection{Rethrowing with iteration information}{

To improve on this we'll wrap the loop in a \code{try_fetch()} call that rethrow errors with iteration information. Make sure to call \code{try_fetch()} on the outside of the loop to avoid a massive performance hit:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_map <- function(.xs, .fn, ...) \{
  out <- new_list(length(.xs))
  i <- 0L

  try_fetch(
    for (i in seq_along(.xs)) \{
      out[[i]] <- .fn(.xs[[i]], ...)
    \},
    error = function(cnd) \{
      abort(
        sprintf("Problem while mapping element \%d.", i),
        parent = cnd
      )
    \}
  )

  out
\}
}\if{html}{\out{</div>}}

And that's it, the error chain created by the rethrowing handler now provides users with the number of the failing iteration:\if{html}{\out{<div class="sourceCode r">}}\preformatted{list(1, "foo") |> my_map(add, 100)
#> Error in `my_map()`:
#> ! Problem while mapping element 2.
#> Caused by error in `x + y`:
#> ! non-numeric argument to binary operator
}\if{html}{\out{</div>}}
}

\subsection{Dealing with errors thrown from the mapped function}{

One problem though, is that the user error call is not very informative when the error occurs immediately in the function supplied to \code{my_map()}:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_function <- function(x) \{
  if (!is_string(x)) \{
    abort("`x` must be a string.")
  \}
\}

list(1, "foo") |> my_map(my_function)
#> Error in `my_map()`:
#> ! Problem while mapping element 1.
#> Caused by error in `.fn()`:
#> ! `x` must be a string.
}\if{html}{\out{</div>}}

Functions have no names by themselves. Only the variable that refers to the function has a name. In this case, the mapped function is passed by argument to the variable \code{.fn}. So, when an error happens, this is the name that is reported to users.

One approach to fix this is to inspect the \code{call} field of the error. When we detect a \code{.fn} call, we replace it by the defused code supplied as \code{.fn} argument:\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_map <- function(.xs, .fn, ...) \{
  # Capture the defused code supplied as `.fn`
  fn_code <- substitute(.fn)

  out <- new_list(length(.xs))

  for (i in seq_along(.xs)) \{
    try_fetch(
      out[[i]] <- .fn(.xs[[i]], ...),
      error = function(cnd) \{
        # Inspect the `call` field to detect `.fn` calls
        if (is_call(cnd$call, ".fn")) \{
          # Replace `.fn` by the defused code.
          # Keep existing arguments.
          cnd$call[[1]] <- fn_code
        \}
        abort(
          sprintf("Problem while mapping element \%s.", i),
          parent = cnd
        )
      \}
    )
  \}

  out
\}
}\if{html}{\out{</div>}}

And voilà!\if{html}{\out{<div class="sourceCode r">}}\preformatted{list(1, "foo") |> my_map(my_function)
#> Error in `my_map()`:
#> ! Problem while mapping element 1.
#> Caused by error in `my_function()`:
#> ! `x` must be a string.
}\if{html}{\out{</div>}}
}
}

\keyword{internal}
