% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cnd.R
\name{cnd_signal}
\alias{cnd_signal}
\alias{cnd_abort}
\title{Signal a condition}
\usage{
cnd_signal(.cnd, ..., .msg = NULL, .call = FALSE, .mufflable = TRUE)

cnd_abort(.cnd, ..., .msg = NULL, .call = FALSE, .mufflable = FALSE)
}
\arguments{
\item{.cnd}{Either a condition object (see \code{\link[=new_cnd]{new_cnd()}}), or the
name of a s3 class from which a new condition will be created.}

\item{...}{Named data fields stored inside the condition
object. These dots are evaluated with \link[=dots_list]{explicit
splicing}.}

\item{.msg}{A string to override the condition's default message.}

\item{.call}{Whether to display the call of the frame in which the
condition is signalled. If \code{TRUE}, the call is stored in the
\code{call} field of the condition object: this field is displayed by
R when an error is issued. The call information is also stored in
the \code{.call} field in all cases.}

\item{.mufflable}{Whether to signal the condition with a muffling
restart. This is useful to let \code{\link[=inplace]{inplace()}} handlers muffle a
condition. It stops the condition from being passed to other
handlers when the inplace handler did not jump elsewhere. \code{TRUE}
by default for benign conditions, but \code{FALSE} for critical ones,
since in those cases execution should probably not be allowed to
continue normally.}
}
\description{
Signal a condition to handlers that have been established on the
stack. Conditions signalled with \code{cnd_signal()} are assumed to be
benign. Control flow can resume normally once the conditions has
been signalled (if no handler jumped somewhere else on the
evaluation stack). On the other hand, \code{cnd_abort()} treats the
condition as critical and will jump out of the distressed call
frame (see \code{\link[=rst_abort]{rst_abort()}}), unless a handler can deal with the
condition.
}
\details{
If \code{.critical} is \code{FALSE}, this function has no side effects beyond
calling handlers. In particular, execution will continue normally
after signalling the condition (unless a handler jumped somewhere
else via \code{\link[=rst_jump]{rst_jump()}} or by being \code{\link[=exiting]{exiting()}}). If \code{.critical} is
\code{TRUE}, the condition is signalled via \code{\link[base:stop]{base::stop()}} and the
program will terminate if no handler dealt with the condition by
jumping out of the distressed call frame.

\code{\link[=inplace]{inplace()}} handlers are called in turn when they decline to handle
the condition by returning normally. However, it is sometimes
useful for an inplace handler to produce a side effect (signalling
another condition, displaying a message, logging something, etc),
prevent the condition from being passed to other handlers, and
resume execution from the place where the condition was
signalled. The easiest way to accomplish this is by jumping to a
restart point (see \code{\link[=with_restarts]{with_restarts()}}) established by the signalling
function. If \code{.mufflable} is \code{TRUE}, a muffle restart is
established. This allows inplace handler to muffle a signalled
condition. See \code{\link[=rst_muffle]{rst_muffle()}} to jump to a muffling restart, and
the \code{muffle} argument of \code{\link[=inplace]{inplace()}} for creating a muffling
handler.
}
\examples{
# Creating a condition of type "foo"
cnd <- new_cnd("foo")

# If no handler capable of dealing with "foo" is established on the
# stack, signalling the condition has no effect:
cnd_signal(cnd)

# To learn more about establishing condition handlers, see
# documentation for with_handlers(), exiting() and inplace():
with_handlers(cnd_signal(cnd),
  foo = inplace(function(c) cat("side effect!\\n"))
)


# By default, cnd_signal() creates a muffling restart which allows
# inplace handlers to prevent a condition from being passed on to
# other handlers and to resume execution:
undesirable_handler <- inplace(function(c) cat("please don't call me\\n"))
muffling_handler <- inplace(function(c) {
  cat("muffling foo...\\n")
  rst_muffle(c)
})

with_handlers(foo = undesirable_handler,
  with_handlers(foo = muffling_handler, {
    cnd_signal("foo")
    "return value"
  }))


# You can signal a critical condition with cnd_abort(). Unlike
# cnd_signal() which has no side effect besides signalling the
# condition, cnd_abort() makes the program terminate with an error
# unless a handler can deal with the condition:
\dontrun{
cnd_abort(cnd)
}

# If you don't specify a .msg or .call, the default message/call
# (supplied to new_cnd()) are displayed. Otherwise, the ones
# supplied to cnd_abort() and cnd_signal() take precedence:
\dontrun{
critical <- new_cnd("my_error",
  .msg = "default 'my_error' msg",
  .call = quote(default(call))
)
cnd_abort(critical)
cnd_abort(critical, .msg = "overridden msg")

fn <- function(...) {
  cnd_abort(critical, .call = TRUE)
}
fn(arg = foo(bar))
}

# Note that by default a condition signalled with cnd_abort() does
# not have a muffle restart. That is because in most cases,
# execution should not continue after signalling a critical
# condition.
}
\seealso{
\code{\link[=abort]{abort()}}, \code{\link[=warn]{warn()}} and \code{\link[=inform]{inform()}} for signalling typical
R conditions. See \code{\link[=with_handlers]{with_handlers()}} for establishing condition
handlers.
}
