% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/j_query.R
\name{j_query}
\alias{j_query}
\alias{j_pivot}
\title{Query and pivot for JSON / NDJSON documents}
\usage{
j_query(
  data,
  path = "",
  object_names = "asis",
  as = "string",
  ...,
  data_type = j_data_type(data),
  path_type = j_path_type(path)
)

j_pivot(
  data,
  path = "",
  object_names = "asis",
  as = "string",
  ...,
  data_type = j_data_type(data),
  path_type = j_path_type(path)
)
}
\arguments{
\item{data}{a character() JSON string or NDJSON records, or an \emph{R}
object parsed to a JSON string using \code{jsonlite::toJSON()}.}

\item{path}{character(1) JSONpointer, JSONpath or JMESpath query
string.}

\item{object_names}{character(1) order \code{data} object elements
\code{"asis"} (default) or \code{"sort"} before filtering on \code{path}.}

\item{as}{character(1) return type. For \code{j_query()}, \code{"string"}
returns JSON / NDJSON strings; \code{"R"} parses JSON / NDJSON to R
using rules in \code{as_r()}. For \code{j_pivot()} (JSON only), use \code{as = "data.frame"} or \code{as = "tibble"} to coerce the result to a
data.frame or tibble.}

\item{...}{arguments for parsing NDJSON, or passed to \code{jsonlite::toJSON} when
\code{data} is not character-valued. For NDJSON,
\itemize{
\item Use \code{n_records = 2} to parse just the first two records of the
NDJSON document.
\item Use \code{verbose = TRUE} to obtain a progress bar when reading from a
connection (file or URL). Requires the cli package.
}

As an example for use with \code{jsonlite::toJSON()}
\itemize{
\item use \code{auto_unbox = TRUE} to automatically 'unbox' vectors of
length 1 to JSON scalar values.
}}

\item{data_type}{character(1) type of \code{data}; one of \code{"json"},
\code{"ndjson"}. Inferred from \code{data} using \code{j_data_type()}.}

\item{path_type}{character(1) type of \code{path}; one of
\code{"JSONpointer"}, \code{"JSONpath"}, \code{"JMESpath"}. Inferred from
\code{path} using \code{j_path_type()}.}
}
\description{
\code{j_query()} executes a query against a JSON or NDJSON
document, automatically inferring the type of \code{data} and
\code{path}.

\code{j_pivot()} transforms a JSON array-of-objects to an
object-of-arrays; this can be useful when forming a
column-based tibble from row-oriented JSON.
}
\details{
\code{j_pivot()} transforms an 'array-of-objects' (typical when the JSON
is a row-oriented representation of a table) to an
'object-of-arrays'. A simple example transforms an array of two
objects each with three fields \code{'[{"a": 1, "b": 2, "c": 3}, {"a": 4, "b": 5, "c": 6}]'} to an object with three fields, each a vector
of length 2 \code{'{"a": [1, 4], "b": [2, 5], "c": [3, 6]}'}. The
object-of-arrays representation corresponds closely to an \emph{R}
data.frame or tibble, as illustrated in the examples.

\code{j_pivot()} with JMESpath paths are especially useful for
transforming NDJSON to a \code{data.frame} or \code{tibble}
}
\examples{
json <- '{
  "locations": [
    {"name": "Seattle", "state": "WA"},
    {"name": "New York", "state": "NY"},
    {"name": "Bellevue", "state": "WA"},
    {"name": "Olympia", "state": "WA"}
  ]
}'

j_query(json, "/locations/0/name")             # JSONpointer
j_query(json, "$.locations[*].name", as = "R") # JSONpath
j_query(json, "locations[].state", as = "R")   # JMESpath

## a few NDJSON records from <https://www.gharchive.org/>
ndjson_file <-
    system.file(package = "rjsoncons", "extdata", "2023-02-08-0.json")
j_query(ndjson_file, "{id: id, type: type}")

j_pivot(json, "$.locations[?@.state=='WA']", as = "string")
j_pivot(json, "locations[?@.state=='WA']", as = "R")
j_pivot(json, "locations[?@.state=='WA']", as = "data.frame")
j_pivot(json, "locations[?@.state=='WA']", as = "tibble")

## use 'path' to pivot ndjson one record at at time
j_pivot(ndjson_file, "{id: id, type: type}", as = "data.frame")

## 'org' is a nested element; extract it
j_pivot(ndjson_file, "org", as = "data.frame")

## use j_pivot() to filter 'PushEvent' for organizations
path <- "[{id: id, type: type, org: org}]
             [?@.type == 'PushEvent' && @.org != null]"
j_pivot(ndjson_file, path, as = "data.frame")

## try also
##
##     j_pivot(ndjson_file, path, as = "tibble") |>
##         tidyr::unnest_wider("org", names_sep = ".")
}
