#' Create and upload data into a table in slides
#' @description This function takes in one single page element property and one dataset at each time
#' for uploading to a single slide
#' @inheritParams dataframe_convert
#' @param id ID of the presentation slide
#' @param page_element_property A list that contains a page element property. The page element is to be
#' generated by the page_element_property function in this package.
#' @importFrom assertthat assert_that is.string
#' @examples
#' \dontrun{
#' library(googleslides)
#' authorize()
#' slide_page <- page_element_property(['slide-id-2'])
#' create_data_table("presentation-slide-id", slide_page, iris[1:5,])
#' }
#' @export
create_data_table <- function(id=NULL, page_element_property=NULL, data=NULL, headers=TRUE){
  # Check to ensure validity of functions
  assert_that(is.string(id))
  assert_that(is.page_element_property(page_element_property))
  assert_that(is.data.frame(data))
  assert_that(is.logical(headers))

  # First create the table to get the objectId
  rows <- nrow(data)
  if(headers){
    rows <- rows + 1
  }
  columns <- ncol(data)

  # Create table
  requests_list <- add_create_table_request(page_element_property = page_element_property,
                                            rows = rows, columns = columns)
  result_list <- commit_to_slides(id, requests_list)

  # Get the object id
  object_id <- result_list$replies$createTable$objectId

  # Create the list to send to update the table - loop through dataset
  converted_data <- dataframe_convert(data, headers)
  row_no <- 1
  requests_list <- NULL
  while(row_no <= nrow(converted_data)){
    requests_list <- add_insert_text_request(requests_list, object_id = object_id,
                                             row_index = converted_data$row[row_no],
                                             column_index = converted_data$column[row_no],
                                             text = converted_data$value[row_no])
    row_no = row_no + 1
  }
  result_list <- commit_to_slides(id, requests_list)
  return(result_list)
}
