% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rgl.makemovie.R
\name{rgl.makemovie}
\alias{rgl.makemovie}
\title{Produce a movie from and 3d rgl scene}
\usage{
rgl.makemovie(
  frame = NULL,
  path = NULL,
  tmin = 0,
  tmax = 1,
  nframes = 60,
  fps = 60,
  output.path,
  output.filename,
  keep.frames = FALSE,
  quiet = TRUE,
  separator = .Platform$file.sep,
  ffmpeg.cmd = "ffmpeg",
  ffmpeg.opt = "-vcodec libx264 -crf 18 -pix_fmt yuv420p"
)
}
\arguments{
\item{frame}{optional function that plots or updates the 3D scene at a given time. This function must have exactly one argument, which specifies the time of the frame.}

\item{path}{optional list that specifies the motion of the camera at some discrete times. The list contains the following elements (for more details see \code{\link{rgl.camera}}):\cr\cr
\code{time} = optional n-vector of strictly monotonically increasing discrete times, required if and only if one of the following four arguments (position, direction, up, fov) are provided as matrices/vectors. If not given, equally spaced in times between tmin and tmax are assumed.\cr
\code{position} = optional argument specifying the camera position along the path. This argument must be one of three types: (1) A 3-element vector specifies a fixed camera position for the whole movie. (2) A n-by-3 matrix specifies n discrete camera positions at the exact times given in the \code{time} vector (see above). The code automatically generates a smooth function going through these n points. (3) A function f(t) of a single time variable t, which returns a 3-element vector, specifies the exact position at that time.\cr
\code{direction} = optional argument specifying the direction of the camera's optical axis. This argument can be of the same three types as the \code{position} argument.\cr
\code{up} = optional argument specifying the camera's up-direction. This argument can be of the same three types as the \code{position} argument.\cr
\code{fov} = optional argument specifying the field-of-view (FoV) in degrees. Similarly to the above arguments, this can be either a single number (fixed FoV), a n-element vector (specifying the Fov at the n discrete times), or a scalar function (specifying the FoV at any time t).\cr\cr}

\item{tmin}{physical time of first frame in the movie.}

\item{tmax}{physical time of last frame in the movie.}

\item{nframes}{number of frames in the movie. The time variable is sampled evenly between \code{tmin} and \code{tmax}.}

\item{fps}{number of frames per second}

\item{output.path}{character specifying the directory, where the movie and temporary frames are saved}

\item{output.filename}{movie filename without path. This filename should end on the extension '.mp4'.}

\item{keep.frames}{logical flag specifying whether the temporary directory with the individual frame files should be kept}

\item{quiet}{logical flag; if true, all console outputs produced by 'ffmpeg' are suppressed}

\item{separator}{filename separate of the system ('/' for Mac, Linux, Unix; '\' for Windows)}

\item{ffmpeg.cmd}{command used to call ffmpeg form a terminal. Normally, this is just 'ffmpeg'.}

\item{ffmpeg.opt}{optional arguments used with ffmpeg, such as compression and formatting options (see \url{https://www.ffmpeg.org/ffmpeg.html}).}
}
\value{
None
}
\description{
Generates an MP4-movie of a 3d rgl scene with time-dependent objects and/or a camera path. The routine has been developed and tested for MacOS and it requires on a working installation of ffmpeg.
}
\details{
Note that the frame width and height should be divisible by 2 for mp4 video compression to work.\cr
To accelerate the movie generation, it is possible to suppress the screen update by calling \code{\link{rgl.hold}} before calling \code{rgl.makemovie}.
}
\examples{

\donttest{
rgl.new(aspect=4/3, col='black', xlim=c(-4,4), ylim=c(-4,4), zlim=c(-4,4))
rgl::clear3d(type = "lights")
rgl::rgl.light(30,60,viewpoint.rel = FALSE)
}

# Make frame function
frame = function(t) {
  # t = time in seconds
  rgl.hold()
  if (t>0) {for (i in seq(3)) rgl::pop3d()}
  rgl.ball(0, 0, 0, 1, normals='improved', depth=6, png=system.file('earth.png', package='rglplus'),
           emission='#444466', rotation=rgl::rotationMatrix(t/86400*2*pi,0,0,1))
  alpha = seq(0,2*pi,length=360)+2*pi*t/43200
  alpha = c(alpha[1],rep(alpha[2:359],each=2),alpha[360])
  y = 3.168*cos(alpha)
  z = 3.168*sin(alpha)
  rgl.ball(0,y[1],z[1],0.05,col='red',emission='#aa0000')
  rgl::segments3d(0,y,z,col='red',alpha=seq(0,1,length=720))
  rgl.draw()
}

# Make path
path = list(position=c(10,10,0), up=c(0,0.5,1), fov = function(t) 40-t/8640)

# Produce movie
\dontrun{
rgl.makemovie(frame=frame, path=path, tmin=0, tmax=86400, output.path='~/testmovie',
              output.filename = 'movie.mp4', ffmpeg.cmd = 'ffmpeg', nframes=600)
}

}
\author{
Danail Obreschkow
}
