% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_MO2.R
\encoding{UTF-8}
\name{calc_MO2}
\alias{calc_MO2}
\title{Calculate metabolic rate}
\usage{
calc_MO2(duration, o2, o2_unit = "percent_a.s.", bin_width, vol, temp = 25,
  sal = 35, atm_pres = 1013.25, time, pH, good_data = TRUE)
}
\arguments{
\item{duration}{numeric vector of the timepoint for each observation (minutes).}

\item{o2}{numeric vector of O2 observations.}

\item{o2_unit}{a string describing the unit used to measure \code{o2}. Default is "percent_a.s." Options are from \code{\link{conv_o2}}.}

\item{bin_width}{numeric. A number defining how long of a period should be binned for each MO2 determination (minutes). If MO2 is to be calculated from one observation to the next (rather than binned observations), set \code{bin_width} to 0. To calculate one MO2 value from all observations, set \code{bin_width} to \code{Inf}.}

\item{vol}{volume of the respirometer (liter).}

\item{temp}{temperature (°C). Default is 25 °C.}

\item{sal}{salinity (psu). Default is 35 psu.}

\item{atm_pres}{atmospheric pressure (mbar). Default is 1013.25 mbar.}

\item{time}{(optional). Numeric vector of timestamp observations.}

\item{pH}{(optional). Numeric vector of pH observations.}

\item{good_data}{logical vector of whether O2 observations are "good" measurements and should be included in analysis. Linear regressions will not be fit over bins that include "bad" data. Bins will be split at bad data points. Default is that all observations are \code{TRUE}.}
}
\value{
A data frame is returned:
\describe{
\item{DUR_MEAN}{Mean duration of the bin (minutes).}
\item{DUR_RANGE}{Range of duration timepoints in the bin.}
\item{TIME_MEAN}{Exists only if the parameter \code{time} has values. Mean timestamp of the bin.}
\item{TIME_RANGE}{Exists only if the parameter \code{time} has values. Range of timestamps in the bin.}
\item{PH_MEAN}{Exists only if the parameter \code{pH} has values. Mean pH of the bin. Averaged using \code{mean_pH()}.}
\item{O2_MEAN}{Mean O2 value of the bin in the unit chosen by \code{o2_unit}).}
\item{O2_RANGE}{Range of O2 values in the bin.}
\item{MO2}{Metabolic rate (umol O2 / hour).}
\item{R2}{Coefficient of determination for the linear regression fit to calculate MO2.}
\item{N}{Number of observations in the bin.}
}
}
\description{
Calculates metabolic rate (MO2) given O2 measurements over time. Oxygen measurements are split into bins and MO2s are calculated from each bin (unless \code{bin_width} is set to \code{0}). The \code{bin_width} parameter defines the width of the bin in timed intervals (e.g. 15 minutes). Linear regressions are fit through each bin and the calculated MO2 is returned as the slope of the change in O2 over time.
}
\note{
If only beginning and ending O2 observations are known, consider using \code{\link{closed}}. Both functions will work fine, but \code{\link{closed}} is simpler.
}
\examples{
# get O2 data
file <- system.file('extdata', 'witrox_file.txt', package = 'respirometry')
o2_data <- na.omit(import_witrox(file, split_channels = TRUE)$CH_4)

# calculate MO2
(mo2_5_min <- calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = 5, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL))

# what if measurements from the 10 to 12 minute marks can't be trusted?
bad_data = o2_data$DURATION >= 10 & o2_data$DURATION <= 12
(mo2_5_min <- calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = 5, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL, good_data = !bad_data))

# easily make a Pcrit plot
plot(mo2_5_min$O2_MEAN, mo2_5_min$MO2)

# I want to express MO2 in mg per min instead.
(mo2_5_min$MO2 <- conv_resp_unit(value = mo2_5_min$MO2, from = 'umol_O2 / hr', to = 'mg_O2 / min'))

# just endpoint measurement:
calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = Inf, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL)

}
\seealso{
\code{\link{calc_b}}, \code{\link{closed}}, \code{\link{scale_MO2}}, \code{\link{conv_resp_unit}}
}
\author{
Matthew A. Birk, \email{matthewabirk@gmail.com}
}
