## **************************************************************************
## Probability density function(pdf) of Log-gamma distribution
dlog.gamma <- function (x, alpha, lambda,  log = FALSE)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(x)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (x <= 0))    
        stop("Invalid arguments")
    u <- exp(lambda * x)
    pdf <- exp(log(alpha) + log(lambda) + lambda * x - alpha * u)
    if(log)
        pdf <- log(pdf)
    return(pdf)
}
## **************************************************************************
## Cummulative distribution function(cdf) of Log-gamma distribution
plog.gamma <- function(q, alpha, lambda,  lower.tail = TRUE, log.p = FALSE)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(q)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (q <= 0))    
        stop("Invalid arguments")   
    u <- exp(lambda * q)
    cdf <- 1.0 - exp(- alpha * u)
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}
## **************************************************************************
## Quantile function of Log-gamma distribution
qlog.gamma <- function(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(p)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (p <= 0) || (p > 1))
        stop("Invalid arguments") 
    tmp<-log(1.0-p) 
    qtl<-  (1.0/lambda) * log(-tmp/alpha)
    if(!lower.tail) 
      qtl<-  (1.0/lambda) * log(-log(p)/alpha)
    if(log.p)
        qtl <- log(qtl)
    return(qtl)
}
## **************************************************************************
## Random variate generation from Log-gamma distribution
rlog.gamma <- function(n, alpha, lambda)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(n)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (n <= 0))    
        stop("Invalid arguments")
    u <- runif(n)
    tmp <- log(u) 
    return((1.0/lambda) * log(-tmp/alpha))
}
## **************************************************************************
## Reliability function of Log-gamma distribution 
slog.gamma <- function (x, alpha, lambda)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(x)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (x <= 0))    
        stop("Invalid arguments")
    return(exp(- alpha * exp(lambda * x)))  
  }
## **************************************************************************
## Hazard function of Log-gamma distribution
hlog.gamma <- function (x, alpha, lambda)
{
    if((!is.numeric(alpha)) || (!is.numeric(lambda)) || (!is.numeric(x)))
        stop("non-numeric argument to mathematical function")
    if((min(alpha) <= 0) || (min(lambda) <= 0) || (x <= 0))    
        stop("Invalid arguments")   
    return(alpha * lambda  * exp(lambda * x))  
}
## **************************************************************************
## Hazard rate average function of Log-gamma distribution
hra.log.gamma <- function(x, alpha, lambda)
{
  r <- slog.gamma(x, alpha, lambda)
  fra <- ((-1) * log(r))/x
  return(fra)
}
## **************************************************************************
## Conditional Hazard rate function of Log-gamma distribution
crf.log.gamma <- function(x, t=0, alpha, lambda)
{
    t <- t
    x <- x
    nume <- hlog.gamma(x+t, alpha, lambda)
    deno <- hlog.gamma(x, alpha, lambda)
    return(nume/deno)
}
## **************************************************************************
## Kolmogorov-Smirnov test (One-sample)for Log-gamma distribution
`ks.log.gamma` <-
function(x, alpha.est,lambda.est, 
alternative=c("less","two.sided","greater"),plot=FALSE, ...)
{
    alpha <- alpha.est 
    lambda <- lambda.est          
    res<-ks.test(x,plog.gamma, alpha, lambda, alternative=alternative)
    if(plot){
        plot(ecdf(x), do.points = FALSE, main = 'Empirical and Theoretical cdfs', 
            xlab = 'x', ylab = 'Fn(x)', ...)
        mini <- min(x)
        maxi <- max(x)
        t <- seq(mini, maxi, by = 0.01)
        y <- plog.gamma(t, alpha, lambda)
        lines(t, y, lwd = 2, col = 2)
    }
    return(res)
}
## **************************************************************************
## Quantile-Quantile(QQ) plot for Log-gamma distribution
qq.log.gamma <- function(x, alpha.est, lambda.est, main = ' ', line.qt = FALSE, ...)
{
    xlab <- 'Empirical quantiles'
    ylab <- 'Theoretical quantiles'
    alpha <- alpha.est 
    lambda <- lambda.est        
    n <- length(x)
    k <- seq(1, n, by = 1)
    P <- (k - 0.5)/n   
    limx <- c(min(x),  max(x))
    Finv <- qlog.gamma(P, alpha, lambda)
    quantiles <- sort(x)
    plot(quantiles, Finv, xlab = xlab, ylab = ylab, xlim = limx, 
         ylim = limx, main = main, col = 4, lwd = 2, ...)
    lines(c(0,limx), c(0,limx), col = 2, lwd = 2)
    if(line.qt){
        quant <- quantile(x)
        x1 <- quant[2]
        x2 <- quant[4]
        y1 <- qlog.gamma(0.25, alpha, lambda)
        y2 <- qlog.gamma(0.75, alpha, lambda)
        m <- ((y2-y1) / (x2-x1))
        inter <- y1 - (m * x1)
        abline(inter, m, col = 2, lwd = 2)
    }
    invisible(list(x = quantiles, y = Finv))
}
## **************************************************************************
## Probability-Probability(PP) plot for Log-gamma distribution
pp.log.gamma <- function(x, alpha.est, lambda.est, main=' ', line = FALSE, ...)
{
    xlab <- 'Empirical distribution function'
    ylab <- 'Theoretical distribution function'
    alpha <- alpha.est 
    lambda <- lambda.est              
    F <- plog.gamma(x,alpha, lambda)
    Pemp <- sort(F)
    n <- length(x)
    k <- seq(1, n, by = 1)
    Pteo <- (k - 0.5) / n
    plot(Pemp, Pteo, xlab = xlab, ylab = ylab, col = 4, 
         xlim = c(0, 1), ylim = c(0, 1), main = main, lwd = 2, ...)
    if(line)
        lines(c(0, 1), c(0, 1), col = 2, lwd = 2)
    Cor.Coeff <- cor(Pemp, Pteo)
    Determination.Coeff <- (Cor.Coeff^2) * 100
    return(list(Cor.Coeff = Cor.Coeff, Determination.Coeff = Determination.Coeff))
}
## **************************************************************************
## Akaike information criterium (AIC) and
## Schwartz/Bayesian information criterion (BIC) 
## for Log-gamma distribution
abic.log.gamma <- function(x, alpha.est, lambda.est)
{ 
    alpha <- alpha.est
    lambda <- lambda.est
    n <- length(x)
    p <- 2
    f <- dlog.gamma(x, alpha, lambda)
    l <- log(f)
    LogLik <- sum(l)
    AIC <- - 2 * LogLik  + 2 * p  
    BIC <- - 2 * LogLik + p * log(n)   
    return(list(LogLik = LogLik, AIC = AIC, BIC = BIC))
}
## **************************************************************************
