% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pair_extremes.R
\name{pair_extremes}
\alias{pair_extremes}
\alias{pair_extremes_vec}
\title{Pair extreme values and sort by the pairs}
\usage{
pair_extremes(
  data,
  col = NULL,
  unequal_method = "middle",
  num_pairings = 1,
  balance = "mean",
  shuffle_members = FALSE,
  shuffle_pairs = FALSE,
  factor_name = ifelse(num_pairings == 1, ".pair", ".pairing"),
  overwrite = FALSE
)

pair_extremes_vec(
  data,
  unequal_method = "middle",
  num_pairings = 1,
  balance = "mean",
  shuffle_members = FALSE,
  shuffle_pairs = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{col}{Column to create sorting factor by. When \code{NULL} and \code{`data`} is a \code{data.frame},
the row numbers are used.}

\item{unequal_method}{Method for dealing with an unequal number of rows
in \code{`data`}.

One of: \code{first}, \code{middle} or \code{last}

\subsection{first}{
The first group will have size \code{1}.

\strong{Example}:

The ordered column values:

\code{c(1, 2, 3, 4, 5)}

Creates the \strong{sorting factor}:

\code{c(}\strong{\code{1}}\code{, 2, 3, 3, 2)}

And are \strong{ordered as}:

\code{c(}\strong{\code{1}}\code{, 2, 5, 3, 4)}

}

\subsection{middle}{
The middle group will have size \code{1}.

\strong{Example}:

The ordered column values:

\code{c(1, 2, 3, 4, 5)}

Creates the \strong{sorting factor}:

\code{c(1, 3, }\strong{\code{2}}\code{, 3, 1)}

And are \strong{ordered as}:

\code{c(1, 5, } \strong{\code{3}}\code{, 2, 4)}

}
\subsection{last}{
The last group will have size \code{1}.

\strong{Example}:

The ordered column values:

\code{c(1, 2, 3, 4, 5)}

Creates the \strong{sorting factor}:

\code{c(1, 2, 2, 1, }\strong{\code{3}}\code{)}

And are \strong{ordered as}:

\code{c(1, 4, 2, 3,} \strong{\code{5}}\code{)}

}}

\item{num_pairings}{Number of pairings to perform (recursively). At least \code{1}.

Based on \code{`balance`}, the secondary pairings perform extreme pairing on either the
\emph{sum}, \emph{absolute difference}, \emph{min}, or \emph{max} of the pair elements.}

\item{balance}{What to balance pairs for in a given \emph{secondary} pairing.
Either \code{"mean"}, \code{"spread"}, \code{"min"}, or \code{"max"}.
Can be a single string used for all secondary pairings
or one for each secondary pairing (\code{`num_pairings` - 1}).

The first pairing always pairs the actual element values.

\subsection{mean}{
Pairs have similar means. The values in the pairs from the previous pairing
are aggregated with \code{`sum()`} and paired.
}
\subsection{spread}{
Pairs have similar spread (e.g. standard deviations). The values in the pairs from the previous pairing
are aggregated with \code{`sum(abs(diff()))`} and paired.
}
\subsection{min / max}{
Pairs have similar minimum / maximum values. The values in the pairs from the previous pairing
are aggregated with \code{`min()`} / \code{`max()`} and paired.
}}

\item{shuffle_members}{Whether to shuffle the pair members. (Logical)}

\item{shuffle_pairs}{Whether to shuffle the pairs. (Logical)}

\item{factor_name}{Name of new column with the sorting factor. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
The sorted \code{data.frame} (\code{tibble}) / \code{vector}.
Optionally with the sorting factor added.

When \code{`data`} is a \code{vector} and \code{`keep_factors`} is \code{FALSE},
the output will be a \code{vector}. Otherwise, a \code{data.frame}.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

The values are paired/grouped such that the highest and lowest values
form the first group, the second highest and the second lowest values
form the second group, and so on.
The values are then sorted by these groups/pairs.

When \code{`data`} has an uneven number of rows, the \code{`unequal_method`}
determines which group should have only \code{1} element.

The \code{*_vec()} version takes and returns a \code{vector}.

\strong{Example}:

The column values:

\code{c(1, 2, 3, 4, 5, 6)}

Creates the \strong{sorting factor}:

\code{c(1, 2, 3, 3, 2, 1)}

And are \strong{ordered as}:

\code{c(1, 6, 2, 5, 3, 4)}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "index" = 1:10,
  "A" = sample(1:10),
  "B" = runif(10),
  "C" = LETTERS[1:10],
  "G" = c(
    1, 1, 1, 2, 2,
    2, 3, 3, 3, 3
  ),
  stringsAsFactors = FALSE
)

# Pair extreme indices (row numbers)
pair_extremes(df)

# Pair extremes in each of the columns
pair_extremes(df, col = "A")$A
pair_extremes(df, col = "B")$B
pair_extremes(df, col = "C")$C

# Shuffle the members pair-wise
pair_extremes(df, col = "A", shuffle_members = TRUE)

# Shuffle the order of the pairs
pair_extremes(df, col = "A", shuffle_pairs = TRUE)

# Use recursive pairing
pair_extremes(df, col = "A", num_pairings = 2)

# Grouped by G
df \%>\%
  dplyr::select(G, A) \%>\% # For clarity
  dplyr::group_by(G) \%>\%
  pair_extremes(col = "A")

# Plot the extreme pairs
plot(
  x = 1:10,
  y = pair_extremes(df, col = "B")$B
)
# With shuffled pair members (run a few times)
plot(
  x = 1:10,
  y = pair_extremes(df, col = "B", shuffle_members = TRUE)$B
)
# With shuffled pairs (run a few times)
plot(
  x = rep(1:5, each = 2),
  y = pair_extremes(df, col = "B", shuffle_pairs = TRUE)$B
)
}
\seealso{
Other rearrange functions: 
\code{\link{center_max}()},
\code{\link{center_min}()},
\code{\link{closest_to}()},
\code{\link{furthest_from}()},
\code{\link{position_max}()},
\code{\link{position_min}()},
\code{\link{rev_windows}()},
\code{\link{roll_elements}()},
\code{\link{shuffle_hierarchy}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{rearrange functions}
