% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotRatematrix.R
\name{plotRatematrix}
\alias{plotRatematrix}
\title{Plot the distribution of evolutionary rate matrices}
\usage{
plotRatematrix(chain, p = NULL, colors = NULL, set.xlim = NULL,
  set.leg = NULL, l.cex = 0.7, ell.wd = 0.5, alphaOff = 1,
  alphaDiag = 1, alphaEll = 1, hpd = 100, show.zero = FALSE,
  n.lines = 50, n.points = 200, point.matrix = NULL,
  point.color = NULL, point.wd = 0.5)
}
\arguments{
\item{chain}{the posterior distribution of parameter estimates as produced by 'readMCMC' or samples from the prior using 'samplePrior'..}

\item{p}{a numeric vector with the regimes to be plotted. This parameter can be used to subset the rate regimes to be plotted as well as to control the order of the plotting. If 'NULL' (default), then all rate regimes are plotted in the same order as in the data.}

\item{colors}{a vector with colors for each rate regime with length equal to the number of regimes or to the number of regimes provided to the argument 'p'. If not provided the function will use pre-selected colors up to 8 regimes.}

\item{set.xlim}{user limits for the x axes. Need to be a vector with two elements, the minimum and the maximum.}

\item{set.leg}{user defined legends for the trait names. A character vector with same length as the number of traits in the model.}

\item{l.cex}{the 'cex' parameter for legends of the plot. See 'help(par)' for more information on 'cex'. Default is 0.7 .}

\item{ell.wd}{a number for the width of the ellipse lines. Default is 0.5 .}

\item{alphaOff}{a number between 0 and 1 with the transparency of the color used for the off-diagonal plots. Default is 1.}

\item{alphaDiag}{a number between 0 and 1 with the transparency of the color used for the diagonal plots. Default is 1.}

\item{alphaEll}{a number between 0 and 1 with the transparency of the color used for the lines of the ellipse plots. Using transparency in the lines might enhance the visualization of regions with more or less density of samples. Default is 1.}

\item{hpd}{a number between 0 and 100 to set the proportion of the highest posterior density (HPD) to be highlighted in the plot.}

\item{show.zero}{whether to plot a thin blue line showing the position of the 0 value on the histograms.}

\item{n.lines}{number of lines to be displayed in the ellipse plots. Default is 50 lines.}

\item{n.points}{number of points used to approximate the shape of the ellipses.}

\item{point.matrix}{optional argument. A list of variance-covariance matrices with length equal to p. If p=NULL then length need to be equal to the number of rate regimes fitted to the data. Each element of the list will be plotted as a single line on top of the distribution of parameter estimates.}

\item{point.color}{optional argument. A vector with color names for the matrices set in 'point.matrix'. The vector need to have same length as 'point.matrix. If not provided, the colors of the lines will be equal to the colors of the distribution (argument 'colors').}

\item{point.wd}{optional argument. The width of the lines plotted using 'point.matrix'. Default is 0.5 .}
}
\value{
A plate with a grid of plots with dimension equal to the number of traits fitted to the data.
}
\description{
Generates a plate with plots showing the posterior distribution of evolutionary rate matrices.
}
\details{
The function provides the option to plot a single evolutionary rate matrix on top of the posterior distribution of each regime as a vertical line on the upper-diagonal and diagonal histogram plots and as an ellipse on the lower-diagonal plots. This can be set using the argument 'point.matrix' (as well as the 'point.color' and 'point.wd' options). One can use this option to contrast the posterior with some point estimate or summary statistics. \cr
\cr
Colors can be provided either as color names recognized by R-base plot functions or in the HEX format. \cr
\cr
The lines showed by the ellipse plots (lower-diagonal) are a sample from the posterior distribution. The user can set the number of lines plotted using the argument 'n.lines'. Note that more lines will take more time to plot. \cr
\cr
The 'hpd' argument can be used to set some regions of the plot to be colored in white. For example, if 'hpd=95' the histograms will plot the region outside the 95\% HPD (Highest Posterior Density) in white and ellipse lines will only be showed if within this 95\% HPD interval. If the region chosen is too small (~10\% or lower), the plot might return an error. This happens because the function take random samples from the posterior distribution to plot as ellipse lines and exclude the samples that are outside the defined HPD interval. If this happens, try to choose a more inclusive percentage or increase the number of samples taken for the ellise lines (see argument 'n.lines') or repeat the plot until sucessful. [A better solution for this issue will be provided soon.] The default is 100 (no highlight is performed and ellipse lines are not restricted). \cr
\cr
The plots are divided into three groups. Upper-diagonal plots show histograms with the posterior distribution for the covariance values between each pairwise combination of the traits. Plot in the diagonal show histograms with the posterior distribution of evolutionary rates for each trait. Plots on the lower-diagonal slots show a collection of ellipses sampled from the posterior distribution of the model. Each ellipse line represents a bivariate distribution for the 95% CI given the pairwise correlation between the traits. \cr
\cr
Lower-diagonal plots are ideal to visualize the evolutionary correlation and variance between two traits. The orientation of the ellipses show whether there is a positive, negative or lack of correlation (horizontal or vertical orientation) between traits. The shape of the ellipses show the major axis of variation between traits. A 'cigar-shaped' ellipse indicates that one of the traits show faster evolutionary rates than the other, so one axis of variation is much larger than the other whereas a more circular (round) ellipse is a result of comparable rates of evolution between the two traits. A completely circular shape denotes lack of evolutionary correlation between two traits. It might help to understand the meaning of the ellipses lines by imagining each ellipse line marks the spread of the dots in a scatterplot with data generated with a particular covariance value (i.e., the covariance value the ellipse is representing).
}
\examples{
\donttest{
data( centrarchidae )
dt.range <- t( apply( centrarchidae$data, 2, range ) )
## The step size for the root value can be set given the range we need to sample from:
w_mu <- ( dt.range[,2] - dt.range[,1] ) / 10
par.sd <- cbind(c(0,0), sqrt( c(1,1) ))
prior <- makePrior(r=2, p=2, par.mu=dt.range, par.sd=par.sd)
handle <- ratematrixMCMC(data=centrarchidae$data, phy=centrarchidae$phy.map, prior=prior
                         , gen=50000, w_mu=w_mu, dir=tempdir())
posterior <- readMCMC(handle, burn = 0.2, thin = 10)
plotRatematrix( posterior )
plotRatematrix( posterior, colors = c("black","red"))
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5)
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5)
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5)
ref.matrix <- list( rbind(c(0.5,0),c(0,0.5)), rbind(c(0.5,0),c(0,0.5)) )
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5
               , point.matrix = ref.matrix)
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5
               , point.matrix = ref.matrix, point.color = "orange", point.wd = 3)
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5
               , point.matrix = ref.matrix, point.color = "orange", point.wd = 3
               , alphaEll = 0.05, n.lines = 2000)
plotRatematrix( posterior, colors = c("black","red"), alphaOff = 0.5, alphaDiag = 0.5
               , point.matrix = ref.matrix, point.color = "orange", point.wd = 3
               , alphaEll = 0.5, n.lines = 200, hpd = 90)
}
}
\author{
Daniel S. Caetano and Luke J. Harmon
}
