% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TRNG.Random.R, R/TRNGkind.R, R/TRNGseed.R,
%   R/TRNGjump.R, R/TRNGsplit.R, R/TRNG.Random.seed.R
\name{TRNG.Random}
\alias{TRNG.Random}
\alias{TRNGkind}
\alias{TRNGseed}
\alias{TRNGjump}
\alias{TRNGsplit}
\alias{TRNG.Random.seed}
\title{TRNG random number generation.}
\usage{
TRNGkind(kind = NULL)

TRNGseed(seed)

TRNGjump(steps)

TRNGsplit(p, s)

TRNG.Random.seed(engspec)
}
\arguments{
\item{kind}{Character string or \code{NULL}. If \code{kind} is not
\code{NULL}, it defines the TRNG random number engine to be used. Use
"default" for the \pkg{rTRNG} \link[=defaultKind]{default kind}
(\code{"yarn2"}).}

\item{seed}{Scalar integer seed, determining the internal state of the
current engine.}

\item{steps}{Number of steps to jump ahead.}

\item{p}{Number of subsequences to split the engine by.}

\item{s}{Index of the desired subsequence between \code{1} and \code{p}.}

\item{engspec}{Optional two-element character vector \code{c(kind, state)},
where the second element is a character representation of the current
engine's internal state and parameters.}
}
\value{
\code{TRNGkind} returns the TRNG kind selected before the call,
  invisibly if argument \code{kind} is not NULL.

\code{TRNG.Random.seed()} called with no arguments returns a two-element
character vector \code{c(kind, state)} fully representing the current state
of the engine in use. When argument \code{engspec = c(kind, state)} is
provided, it is used to set an engine of the given \code{kind} with internal
state and parameters restored from \code{state}.
}
\description{
The functions below allow setting and manipulating the current
  TRNG random number engine (pseudo-random number generator), similar to
  base-\R \link{Random}. The current engine is then used for generating
  random variates via any of the \code{r<dist>_trng} functions (e.g.,
  \code{\link{runif_trng}}).

\code{TRNGkind} allows to query or set the kind of TRNG engine
  in use. See \sQuote{Random number engines details} for the available
  engines.

\code{TRNGseed} specifies the seed for the current
  engine.

If the current engine is of a \emph{parallel} kind,
  \code{TRNGjump} advances its internal state without generating all
  intermediate steps.

If the current engine is of a \emph{parallel} kind,
  \code{TRNGsplit} updates its internal state and parameters in order to
  generate directly a subsequence obtained by decimation, producing every
  \code{s}th element starting from the \code{p}th.

\code{TRNG.Random.seed} allows to get a full representation of
  the current state of the engine in use, and to restore the current engine
  from such representation.
}
\section{Details}{

The TRNG C++ library provides a collection of random number engines
(pseudo-random number generators). In particular, compared to
\emph{conventional} engines working in a purely sequential manner,
\emph{parallel} engines can be manipulated via \code{jump} and \code{split}
operations. Jumping allows to advance the internal state by a number of steps
without generating all intermediate states, whereas split operations allow to
generate directly a subsequence obtained by decimating the original sequence.
Please consult the TRNG C++ library documentation (see \sQuote{References})
for an introduction to the concepts and details around (parallel) random
number generation and engines, including details about the state size and
period of the TRNG generators.


The current engine is an instance of one \link[=TRNG.Engine]{TRNG engine
class} provided by \pkg{rTRNG}, and is stored as \code{"TRNGengine"} global
option. If not explicitly set via \code{TRNGkind}, an engine of
\link[=defaultKind]{default kind} is implicitly created upon the first call
to any \code{TRNG*} or \code{r<dist>_trng} function. Note that the current
engine is not persistent across \R sessions. Function \code{TRNG.Random.seed}
can be used to extract and restore the current engine and its internal state.
}

\section{Random number engines details}{

\subsection{Parallel engines}{
  \describe{
    \item{\code{lcg64}}{
      Linear congruential generator with modulus \eqn{2^{64}}{2^64}.
    }
    \item{\code{lcg64_shift}}{
      Linear congruential generator with modulus \eqn{2^{64}}{2^64} and
      bit-shift transformation.
    }
    \item{\code{mrg2}, \code{mrg3}, \code{mrg4}, \code{mrg5}}{
      Multiple recurrence generators based on a linear feedback shift
      register sequence with prime modulus \eqn{2^{31}-1}{2^31-1}.
    }
    \item{\code{mrg3s}, \code{mrg5s}}{
      Multiple recurrence generators based on a linear feedback shift
      register with Sophie-Germain prime modulus.
    }
    \item{\code{yarn2}, \code{yarn3}, \code{yarn4}, \code{yarn5}}{
      YARN generators based on the delinearization of a linear feedback shift
      register sequence with prime modulus \eqn{2^{31}-1}{2^31-1}.
    }
    \item{\code{yarn3s}, \code{yarn5s}}{
      YARN generators based on the delinearization of a linear feedback shift
      register sequence with Sophie-Germain prime modulus.
    }
  }
}

\subsection{Conventional engines}{
  \describe{
    \item{\code{lagfib2plus_19937_64}, \code{lagfib4plus_19937_64}}{
      Lagged Fibonacci generator with \code{2} or \code{4} feedback taps and
      addition.
    }
    \item{\code{lagfib2xor_19937_64}, \code{lagfib4xor_19937_64}}{
      Lagged Fibonacci generator with \code{2} or \code{4} feedback taps and
      exclusive-or operation.
    }
    \item{\code{mt19937}}{
      Mersenne-Twister generating 32 random bit.
    }
    \item{\code{mt19937_64}}{
      Mersenne-Twister generating 64 random bit.
    }
  }
}
}

\examples{
## TRNG kind of the current engine
TRNGkind()
## set a specific TRNG kind
TRNGkind("yarn5s")
TRNGkind()
\dontrun{
  ## error if kind is not valid
  TRNGkind("invalid")
}
## set the deafult TRNG kind
TRNGkind("default")
TRNGkind()

## seed the current random number engine
TRNGseed(117)

## full representation of the current state of the engine in use
s <- TRNG.Random.seed()
s

## draw 10 random variates using the current engine
runif_trng(10)

## restore the engine and its internal state
TRNG.Random.seed(s)

## jump and draw the last 3 variates out of the 10 above
TRNGjump(7) # jump 7 steps ahead
runif_trng(3)

## restore the internal state, split and draw every 5th element starting from
## the 2nd
TRNG.Random.seed(s)
TRNGsplit(5, 2)
runif_trng(2)

## TRNGseed, TRNGjump and TRNGsplit can be combined with r<dist>_trng in c(...)
## as they return NULL
c(TRNGseed(117),
  TRNGjump(2), runif_trng(2),
  TRNGsplit(3,2), runif_trng(2))
}
\references{
Heiko Bauke, \emph{Tina's Random Number Generator Library}, Version
4.20, \url{https://numbercrunch.de/trng/trng.pdf}.
}
\seealso{
TRNG distributions:
\code{\link{rbinom_trng}}, \code{\link{rlnorm_trng}}, \code{\link{rnorm_trng}}, \code{\link{rpois_trng}}, \code{\link{runif_trng}}.
}
