% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/render_tabset.R
\name{render_tabset}
\alias{render_tabset}
\title{Dynamically Generate Tabset Panels in Quarto HTML Documents}
\usage{
render_tabset(
  data,
  tabset_vars,
  output_vars,
  sort = TRUE,
  layout = NULL,
  heading_levels = NULL,
  pills = FALSE,
  tabset_width = "default"
)
}
\arguments{
\item{data}{A data frame.}

\item{tabset_vars}{Columns to use as tabset labels. Internally passed
to the \code{select} argument of \code{\link[=subset]{subset()}}. Accepts raw column names,
strings, numbers and logical values.}

\item{output_vars}{Columns to display in each tabset panel. Internally
passed to the \code{select} argument of \code{\link[=subset]{subset()}}. Accepts raw column names,
strings, numbers and logical values.}

\item{sort}{Logical, whether to sort the \code{data} with \code{tabset_vars}.
The default is \code{TRUE}. If \code{FALSE}, the tabset will be output in the
original order of the \code{data}.}

\item{layout}{\code{NULL} or a character vector of length 1 for specifying layout
in tabset panel. If not \code{NULL}, \code{layout} must begin with at least three
or more repetitions of ":" (e.g. ":::"). Closing div (e.g. ":::") is
inserted automatically.
See for details: \url{https://quarto.org/docs/authoring/figures.html#complex-layouts}.}

\item{heading_levels}{\code{NULL} or a vector consisting of natural numbers
and missing values. The length is equal to the number of columns specified
in \code{tabset_vars}. This controls whether it is partially (or entirely)
displayed as normal header instead of tabset.
\itemize{
\item If \code{heading_levels} is a \code{NULL}, all output is tabset.
\item If \code{heading_levels} is a vector of positive natural number, the elements
of the vector correspond to the columns specified in \code{tabset_vars}.
\itemize{
\item If the element is integer, the tabset column is displayed as headers
with their level, not tabset. (e.g. 2 means h2 header).
Levels 1 to 6 are recommended. The reason is that quarto supports
headers up to 6. 7 and above will also work, but they are displayed
as normal text. In addition, considering the chapter format,
it is preferable to gradually increase the level, as in 1, 2 and 3.
\item If the element is NA, tabset is displayed.
}
}}

\item{pills}{Logical, use pills or not.
See \url{https://getbootstrap.com/docs/5.2/components/navs-tabs/#pills}
for details. If \code{heading_levels} is specified, this will be ignored.}

\item{tabset_width}{Character, one of "default", "fill" and "justified".
See \url{https://getbootstrap.com/docs/5.2/components/navs-tabs/#fill-and-justify}
for details. If \code{heading_levels} is specified, this will be ignored.}
}
\value{
\code{NULL} invisibly. This function outputs the markdown
that generates the \href{https://quarto.org/docs/output-formats/html-basics.html#tabsets}{tabset}
to stdout (console).
}
\description{
\code{render_tabset()} takes a data frame as input and outputs the markdown
that generates the \href{https://quarto.org/docs/output-formats/html-basics.html#tabsets}{tabset}
to stdout (console). \emph{\strong{Only works with Quarto HTML documents.}}
}
\details{
\itemize{
\item Write \verb{#| results: asis} at the beginning of the chunk or
\code{results='asis'} in the chunk options.
\item If multiple \code{tabset_vars} are given, create nested tabsets.
\item For columns specified in \code{output_vars}, columns of type list are output with
\code{\link[=print]{print()}} and normal columns are output with \code{\link[=cat]{cat()}}.
\item If \code{tabset_vars} or \code{output_vars} have "factor", "Date" and "POSIXt"
columns, they are converted internally to character. This is to prevent it
being displayed as numeric when \code{\link[=cat]{cat()}} is executed. If \code{sort = TRUE},
sorting is performed before conversion to string.
}
}
\section{Limitations}{

\itemize{
\item \code{layout} is intended for simplified use cases and
complex layouts may not work.
\item When outputting tables or figures that use JavaScript
(such as \code{{plotly}}, \code{{leaflet}}, \code{{DT}}, \code{{reactable}}, etc.),
it seems JavaScript dependencies need to be resolved.
A simple solution is to wrap the output in \code{\link[htmltools:builder]{htmltools::div()}}
and create a dummy plot in another chunk. See the Get started for details.
\item When \code{tabset_vars} and \code{output_vars} have the following columns,
they may not display well:
\itemize{
\item A column of type list contains a named vector or list
(This is for \code{output_vars}. \code{tabset_vars} must not contain list
columns).
\item Classes with their own printing methods,
such as "difftime", "ts", .etc.
}
\item When specifying a list-type column that includes ggplot objects in
\code{output_vars}, setting the chunk option \code{echo: fenced} may cause
the plots to not display correctly.
}
}

\examples{
# sample data
df <- data.frame(
  group1 = c(rep("A", 3), rep("B", 3)),
  group2 = rep(c("X", "Y", "Z"), 2),
  value1 = 1:6,
  value2 = letters[1:6]
)

# Here are examples of the output before it is converted to tabset.
# If you want it to actually work, in the .qmd file,
# set `results='asis'` in the chunk options or
# write `#| results: asis` at the beginning of the chunk.

# Basic usage
render_tabset(df, group1, value1)

# Nested tabset, two outputs side by side with a width of 1:1
render_tabset(
  df,
  c(group1, group2),
  c(value1, value2),
  layout = "::: {layout-ncol=2}"
)

# Use heading instead of tabset
render_tabset(
  df,
  c(group1, group2),
  value1,
  heading_levels = c(2, 3)
)
}
\references{
As this function is focused on quickly and dynamically
generating tabsets and chunks, it is difficult to customize it on a
chunk-by-chunk basis. The regular way to dynamically create chunks is
to use functions such as \code{\link[knitr:knit]{knitr::knit()}}, \code{\link[knitr:knit_child]{knitr::knit_child()}},
\code{\link[knitr:knit_expand]{knitr::knit_expand()}}, etc. For more information on these,
see the following links.
\itemize{
\item Heiss, Andrew. 2024. “Guide to Generating and Rendering Computational
Markdown Content Programmatically with Quarto.” November 4, 2024.
\doi{https://doi.org/10.59350/pa44j-cc302}.
\item \url{https://bookdown.org/yihui/rmarkdown-cookbook/child-document.html#child-document}
\item \url{https://bookdown.org/yihui/rmarkdown-cookbook/knit-expand.html}
}
}
