% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/feature_representation.R
\name{feature_representation}
\alias{feature_representation}
\alias{feature_representation,ConservationProblem,numeric-method}
\alias{feature_representation,ConservationProblem,matrix-method}
\alias{feature_representation,ConservationProblem,data.frame-method}
\alias{feature_representation,ConservationProblem,Spatial-method}
\alias{feature_representation,ConservationProblem,Raster-method}
\title{Feature representation}
\usage{
\S4method{feature_representation}{ConservationProblem,numeric}(x, solution)

\S4method{feature_representation}{ConservationProblem,matrix}(x, solution)

\S4method{feature_representation}{ConservationProblem,data.frame}(x, solution)

\S4method{feature_representation}{ConservationProblem,Spatial}(x, solution)

\S4method{feature_representation}{ConservationProblem,Raster}(x, solution)
}
\arguments{
\item{x}{\code{\link{ConservationProblem-class}} object.}

\item{solution}{\code{numeric}, \code{matrix}, \code{data.frame},
\code{\link[raster]{Raster-class}}, or \code{\link[sp]{Spatial-class}}
object. See the Details section for more information.}
}
\value{
\code{\link[tibble]{tibble}} object containing the amount
  (\code{"absolute_held"}) and proportion (\code{"relative_held"})
  of the distribution of each feature held in the solution. Here, each
  row contains data that pertain to a specific feature in a specific
  management zone (if multiple zones are present). This object
  contains the following columns:

  \describe{

  \item{feature}{\code{character} name of the feature.}

  \item{zone}{\code{character} name of the zone (not included when the
    argument to \code{x} contains only one management zone).}

  \item{absolute_held}{\code{numeric} total amount of each feature secured in
    the solution. If the problem contains multiple zones, then this
    column shows how well each feature is represented in a each
    zone.}

  \item{relative_held}{\code{numeric} proportion of the feature's
    distribution held in the solution. If the problem contains
    multiple zones, then this column shows how well each feature is
    represented in each zone.}

  }
}
\description{
Calculate how well features are represented in a solution.
}
\details{
Note that all arguments to \code{solution} must correspond
  to the planning unit data in the argument to \code{x} in terms
  of data representation, dimensionality, and spatial attributes (if
  applicable). This means that if the planning unit data in \code{x}
  is a \code{numeric} vector then the argument to \code{solution} must be a
  \code{numeric} vector with the same number of elements, if the planning
  unit data in \code{x} is a \code{\link[raster]{RasterLayer-class}} then the
  argument to \code{solution} must also be a
  \code{\link[raster]{RasterLayer-class}} with the same number of rows and
  columns and the same resolution, extent, and coordinate reference system,
  if the planning unit data in \code{x} is a \code{\link[sp]{Spatial-class}}
  object then the argument to \code{solution} must also be a
  \code{\link[sp]{Spatial-class}} object and have the same number of spatial
  features (e.g. polygons) and have the same coordinate reference system,
  if the planning units in \code{x} are a \code{data.frame} then the
  argument to \code{solution} must also be a \code{data.frame} with each
  column correspond to a different zone and each row correspond to
  a different planning unit, and values correspond to the allocations
  (e.g. values of zero or one).

  Solutions must have planning unit statuses set to missing (\code{NA})
  values for planning units that have missing (\code{NA}) cost data. For
  problems with multiple zones, this means that planning units must have
  missing (\code{NA}) allocation values in zones where they have missing
  (\code{NA}) cost data. In other words, planning units that have missing
  (\code{NA}) cost values in \code{x} should always have a missing
  (\code{NA}) value the argument to \code{solution}. If an argument is
  supplied to
  \code{solution} where this is not the case, then an error will be thrown.
  Please note that earlier versions of the \pkg{prioritizr}
  (prior to 4.0.4.1) required that such planning units always have zero
  values, but this has been changed to make the handling of missing values
  more consistent throughout the package.

  Additionally, note that when calculating the proportion of each feature
  represented in the solution, the denominator is calculated using all
  planning units---\strong{including any planning units with \code{NA} cost
  values in the argument to \code{x}}. This is exactly the same equation
  used when calculating relative targets for problems (e.g.
  \code{add_relative_targets}).
}
\examples{
# set seed for reproducibility
set.seed(500)

# load data
data(sim_pu_raster, sim_pu_polygons, sim_features, sim_pu_zones_stack,
     sim_pu_zones_polygons, sim_features_zones)


# create a simple conservation planning data set so we can see exactly
# how feature representation is calculated
pu <- data.frame(id = seq_len(10), cost = c(0.2, NA, runif(8)),
                 spp1 = runif(10), spp2 = c(rpois(9, 4), NA))

# create problem
p1 <- problem(pu, c("spp1", "spp2"), cost_column = "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions()

# create a solution
s1 <- data.frame(solution = c(1, NA, rep(c(1, 0), 4)))
print(s1)

# calculate feature representation
r1 <- feature_representation(p1, s1)
print(r1)

# verify that feature representation calculations are correct
all.equal(r1$absolute_held, c(sum(pu$spp1 * s1[[1]], na.rm = TRUE),
                              sum(pu$spp2 * s1[[1]], na.rm = TRUE)))
all.equal(r1$relative_held, c(sum(pu$spp1 * s1[[1]], na.rm = TRUE) /
                              sum(pu$spp1),
                              sum(pu$spp2 * s1[[1]], na.rm = TRUE) /
                              sum(pu$spp2, na.rm = TRUE)))

# solve the problem using an exact algorithm solver
s1_2 <- solve(p1)
print(s1_2)

# calculate feature representation in this solution
# note that we set missing values in the solution_1 explicitly to zero
r1_2 <- feature_representation(p1, s1_2[, "solution_1", drop = FALSE])
print(r1_2)

# build minimal conservation problem with raster data
p2 <- problem(sim_pu_raster, sim_features) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions()
\donttest{
# solve the problem
s2 <- solve(p2)

# print solution
print(s2)

# calculate feature representation in the solution
r2 <- feature_representation(p2, s2)
print(r2)

# plot solution
plot(s2, main = "solution", axes = FALSE, box = FALSE)
}
# build minimal conservation problem with spatial polygon data
p3 <- problem(sim_pu_polygons, sim_features, cost_column = "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.1) \%>\%
      add_binary_decisions()
\donttest{
# solve the problem
s3 <- solve(p3)

# print first six rows of the attribute table
print(head(s3))

# calculate feature representation in the solution
r3 <- feature_representation(p3, s3[, "solution_1"])
print(r3)

# plot solution
spplot(s3, zcol = "solution_1", main = "solution", axes = FALSE, box = FALSE)
}
# build multi-zone conservation problem with raster data
p4 <- problem(sim_pu_zones_stack, sim_features_zones) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5,
                                  ncol = 3)) \%>\%
      add_binary_decisions()
\donttest{
# solve the problem
s4 <- solve(p4)

# print solution
print(s4)

# calculate feature representation in the solution
r4 <- feature_representation(p4, s4)
print(r4)

# plot solution
plot(category_layer(s4), main = "solution", axes = FALSE, box = FALSE)
}
# build multi-zone conservation problem with spatial polygon data
p5 <- problem(sim_pu_zones_polygons, sim_features_zones,
              cost_column = c("cost_1", "cost_2", "cost_3")) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(matrix(runif(15, 0.1, 0.2), nrow = 5,
                                  ncol = 3)) \%>\%
      add_binary_decisions()
\donttest{
# solve the problem
s5 <- solve(p5)

# print first six rows of the attribute table
print(head(s5))

# calculate feature representation in the solution
r5 <- feature_representation(p5, s5[, c("solution_1_zone_1",
                                        "solution_1_zone_2",
                                        "solution_1_zone_3")])
print(r5)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s5$solution <- category_vector(s5@data[, c("solution_1_zone_1",
                                           "solution_1_zone_2",
                                           "solution_1_zone_3")])
s5$solution <- factor(s5$solution)

# plot solution
spplot(s5, zcol = "solution", main = "solution", axes = FALSE, box = FALSE)
}
}
\seealso{
\code{\link{problem}}, \code{\link{feature_abundances}}.
}
