context("Regression tests for rodent, ant, and plant output")

portal_data_path <- tempdir()

test_that("data generated by default setting is same", {
  data <- abundance(portal_data_path, level = 'Site',
                    type = "Rodents", plots = "all", unknowns = FALSE,
                    min_plots = 24, shape = "crosstab", time = "period") %>%
    dplyr::filter(period < 434)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "0cce45b6cf")
})

test_that("data generated by level = treatment, plots = longterm is same", {
  data <- abundance(portal_data_path, level = 'treatment', plots = "longterm") %>%
    dplyr::filter(period < 434)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "4381ec519f")
})

test_that("data generated by level = plot, time = newmoon, type = granivore, shape = flat is same", {
  data <- abundance(portal_data_path, level = 'plot', type = "granivores",
                    shape = "flat", time = "newmoon", na_drop = FALSE) %>%
    dplyr::filter(newmoonnumber < 465)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "20d3d2287c")

  sampled_newmoons <- abundance(portal_data_path, time = "all",
                                na_drop = FALSE, min_plots = 1) %>%
    dplyr::pull(newmoonnumber)
  data <- data %>%
    dplyr::filter(newmoonnumber %in% sampled_newmoons)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_equal(dim(data), c(155880, 5))
  expect_known_hash(data, "efbecf7764")
})

test_that("data generated by na_drop = FALSE, zero_drop = FALSE is same", {
  data <- abundance(portal_data_path, time = "date", na_drop = FALSE,
                    zero_drop = FALSE, min_plots = 1) %>%
    dplyr::filter(censusdate < as.Date("2015-01-01"))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_equal(dim(data), c(464, 22))
  expect_known_hash(is.na(data), "0294bfffde")
  data[is.na(data)] <- -999
  expect_known_hash(data, "37b36c0897")
  abundances <- data %>% dplyr::select(-censusdate)

  data <- abundance(portal_data_path, time = "newmoon", min_plots = 1,
                    include_unsampled = TRUE) %>%
    dplyr::filter(newmoonnumber < 465)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_equal(dim(data), c(464, 22))
  expect_known_hash(is.na(data), "b2d5abb360")
  data[is.na(data)] <- -999
  expect_known_hash(data, "f302fe980f")
  expect_equal(data %>% dplyr::select(-newmoonnumber),
               abundances)
})

test_that("data generated by unknowns = T, min_plots = 1 is same", {
  data <- abundance(portal_data_path, min_plots = 1, unknowns = TRUE) %>%
    dplyr::filter(period < 434) %>%
    dplyr::select(period, BA, DM, DO, DS, "NA", OL, OT, other, PB,
                  PE, PF, PH, PI, PL, PM, PP, RF, RM, RO, SF, SH, SO)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "2872e40782")
})

test_that("data generated by plots = c(4, 8, 10, 12) is same", {
  data <- summarize_rodent_data(path = portal_data_path, plots = c(4, 8, 10, 12),
                                na_drop = TRUE, zero_drop = FALSE) %>%
    dplyr::filter(period < 450)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "0c87bf6bd3")
})

test_that("biomass data generated by level = plot is same", {
  data <- biomass(portal_data_path, type = "rodents", level = "plot", na_drop = FALSE) %>%
    dplyr::filter(period < 434) %>%
    dplyr::mutate_if(is.numeric, round, 3)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "0db04ad878")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "0c14c940ab")
})

test_that("biomass data generated by min_plots = 1 is same", {
  data <- biomass(portal_data_path, type = "rodents", min_plots = 1) %>%
    dplyr::filter(period < 434) %>%
    dplyr::mutate_if(is.numeric, round, 3)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "0d12f11a2f")
})

test_that("data generated by default setting is same (plants)", {
  data <- plant_abundance(portal_data_path, level = 'Site',
                          type = "All", plots = "all", unknowns = FALSE,
                          correct_sp = TRUE, shape = "flat", na_drop = TRUE,
                          zero_drop = TRUE, min_quads = 1, effort = TRUE) %>%
    dplyr::filter(year < 2015) %>%
    dplyr::mutate(species = as.character(species))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "81500538a4")
})

test_that("data generated by type = Shrubs, unknowns = T, correct_sp = F is same (plants)", {
  data <- plant_abundance(portal_data_path, level = 'Site',
                          type = "Shrubs", plots = "all", unknowns = TRUE,
                          correct_sp = FALSE, shape = "flat", na_drop = TRUE,
                          zero_drop = TRUE, min_quads = 1, effort = TRUE) %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "6ff115fb81")
})

test_that("data generated by level = Plot, type = Annuals, plots = longterm is same (plants)", {
  data <- plant_abundance(portal_data_path, level = 'Plot',
                          type = "Annuals", plots = "longterm",
                          unknowns = TRUE, correct_sp = TRUE, shape = "flat",
                          na_drop = TRUE, zero_drop = TRUE, min_quads = 1, effort = TRUE) %>%
    dplyr::filter(year < 2015) %>%
    dplyr::mutate(species = as.character(species))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "71f8781911")
})

test_that("data generated by level = quadrat is same (plants)", {
  data <- plant_abundance(portal_data_path, level = 'quadrat') %>%
    dplyr::filter(year < 2015) %>%
    dplyr::mutate(species = as.character(species))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "1570e5d54b")
})

test_that("data generated by level = quadrat, shape = crosstab, output = cover is same (plants)", {
  data <- summarize_plant_data(portal_data_path, level = 'quadrat', shape = "crosstab",
                               output = "cover") %>%
    dplyr::filter(year == 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "ce34ad2830")
})

test_that("data generated by shape = crosstab is same (plants)", {
  selected_plants <- sort(c("carl line", "tria port", "amar palm", "atri acan", "atri cane", "atri eleg", "atri wrig", "chen frem", "chen sp", "bray dens", "sals kali", "euro lana", "tide lanu", "tide sp", "sper echi", "funa hart", "dich pulc", "yucc elat", "ambr arte", "ambr conf", "ambr sals", "bahi absi", "bahi bite", "bail mult", "bric eupa", "caly wrig", "chae stev", "chae eric", "cirs neom", "erig conc", "erig dive", "erig sp", "flou cern", "guti saro", "hapl sp", "hapl spin", "heli annu", "hete suba", "hapl tenu", "laen coul", "hapl grac", "mach tanr", "mala fend", "mala sp", "part inca", "pect papp", "pere nana", "pseu cane", "rafi neom", "sanv aber", "step exig", "dyss pent", "micr lene", "micr sp", "verb ence", "xant spin", "zinn pumi", "zinn gran", "zinn sp", "amsi inte", "amsi sp", "amsi tess", "cryp cras", "cryp micr", "cryp sp", "cryp sp2", "lapp redo", "nama hisp", "pect recu", "pect sp", "phac ariz", "plag ariz", "chor tene", "desc obtu", "desc pinn", "desc sp", "dith wisl", "lepi lasi", "lepi sp", "lesq gord", "sisy irio", "unkn must", "cyli fulg", "cyli sp", "opun basi", "opun poly", "opun sp", "comm erec", "cusc mitr", "cusc sp", "cusc tube", "ipom cost", "apod undu", "ephe trif", "crot cory", "euph exst", "euph fend", "euph micr", "euph serp", "euph serr", "euph sp", "erod cicu", "erod texa", "acac cons", "acac greg", "astr allo", "astr nutt", "astr sp", "cass lept", "crot pumi", "dale brac", "dale nana", "dale pogo", "dale sp", "hoff dens", "lupi brev",  "lupi conc", "lupi sp", "lupi sp2", "mimo acul", "pros glan", "pros sp", "pros velu", "cass bauh", "abut parv", "sida phys", "sida proc", "sida neom", "sida sp", "sida spin", "spha angu", "spha coul", "spha hast", "spha inca", "spha laxa", "spha sp", "prob parv", "moll cerv", "moll sp", "moll vert", "tali angu", "tali aura", "tali sp", "alli inca", "alli sp", "ammo chen", "boer cocc", "boer coul", "boer inte", "boer sp", "boer torr", "oeno prim", "oeno sp", "esch mexi", "lina texa", "plan purs", "aris adsc", "aris long", "aris sp", "aris hamu", "both barb", "bout aris", "bout sp", "pani ariz", "brom rube", "bout dact", "chlo virg", "bout barb", "bout erio", "tric cali", "sita hyst", "enne desv", "erag cili", "erag inte", "erag lehm", "erag arid", "erag sp", "erio lemm", "trid pulc", "hila muti", "lycu phle", "muhl port", "muhl sp", "pani hirt", "pani mili", "pani sp", "schi barb", "seta leuc", "seta macr", "spor cont", "trag bert", "annu gras", "annu gras1", "annu gras2", "unkn gras", "unkn gras2", "pere gras", "vulp octo", "eria diff", "gili mexi", "gili sinu", "lina bige", "lina sp", "poly twee", "erio aber", "erio poly", "rume alti", "rume angu", "rume hyme", "port parv", "port sp", "port suff", "andr occi", "delp sp", "lyci ande", "lyci torr", "sola elea", "sola rost", "papp vagi", "annu forb", "annu forb2", "annu forb3", "annu forb4", "annu forb5", "unkn forb", "unkn forb2", "pere forb", "unkn", "unkn shrb", "tetr coul", "kall cali", "kall gran", "kall hirs", "kall sp", "larr trid"))
  data <- plant_abundance(portal_data_path, shape = "crosstab") %>%
    dplyr::filter(year < 2015)
  expect_warning(data <- dplyr::select(data, dplyr::one_of(
    c("year", "season", "quads", selected_plants))))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "03d79118b6")
})

test_that("data generated by default setting is same (shrub_cover)", {
  data <- shrub_cover(path = portal_data_path, type = "Shrubs",
                      plots = "all", unknowns = FALSE,
                      correct_sp = TRUE) %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "77da4ca0e3")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "9e5849fa79")
})

test_that("data generated by default setting is same (ant colony_presence_absence)", {
  data <- colony_presence_absence(portal_data_path, level = "site") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "512fa0d867")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "8ce773ce81")
})

test_that("data generated by default setting is same (ant bait_presence_absence)", {
  data <- bait_presence_absence(portal_data_path, level = "Site") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "9b6a97c7b2")
})

test_that("data generated by level = 'stake' is same (ant colony_presence_absence)", {
  data <- colony_presence_absence(portal_data_path, level = "stake") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "2beb6917b2")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "4639dbfbf3")
})

test_that("data generated by level = 'stake' is same (ant bait_presence_absence)", {
  data <- bait_presence_absence(portal_data_path, level = "stake") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "8339137b84")
})

test_that("data generated by level = 'plot' is same (ant colony_presence_absence)", {
  data <- colony_presence_absence(portal_data_path, level = "plot") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "f9d41f2593")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "60e9306bfc")
})

test_that("data generated by level = 'plot' is same (ant bait_presence_absence)", {
  data <- bait_presence_absence(portal_data_path, level = "plot") %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]
  expect_known_hash(data, "7c217de3f2")
})

test_that("data generated by default setting is same (weather)", {
  data <- weather(path = portal_data_path) %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "7804c05069")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "b4c772e46f")
})

test_that("data generated by fill = TRUE is same (weather)", {
  data <- weather(fill = TRUE, path = portal_data_path) %>%
    dplyr::filter(year < 2015)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "2daddbffd9")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "bc40fd3fa0")
})

test_that("data generated by default setting is same (NDVI)", {
  data <- ndvi(path = portal_data_path) %>%
    dplyr::filter(date < as.Date("2015-01-01"))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  expect_known_hash(data, "418318fabc")
})

test_that("data generated by fill = TRUE is same (NDVI)", {
  data <- ndvi(path = portal_data_path, fill = TRUE) %>%
    dplyr::filter(date < as.Date("2015-01-01"))
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  data$ndvi <- round(data$ndvi, 4)
  expect_known_hash(data, "b3d2b60425")
})

test_that("get_future_moons returns identical table using sample input", {
  moons <- data.frame(newmoonnumber = c(1, 2),
                      newmoondate = c("1977-07-16", "1977-08-14"),
                      period = c(1, 2),
                      censusdate = c("1977-07-16", "1977-08-19"))

  newmoons <- get_future_moons(moons, num_future_moons = 10)
  attributes(newmoons) <- attributes(newmoons)[sort(names(attributes(newmoons)))]

  # correct for NAs in output
  expect_known_hash(is.na(newmoons), "1d6e5a1db8")
  newmoons$newmoondate[is.na(newmoons$newmoondate)] <- as.Date("0000-01-01")
  newmoons$period[is.na(newmoons$period)] <- -999999
  newmoons$censusdate[is.na(newmoons$censusdate)] <- as.Date("0000-01-01")
  expect_known_hash(newmoons, "aa0ddfd4ee")
})

test_that("data generated by default setting is same (phenocam)", {
  data <- phenocam(path = portal_data_path) %>%
    dplyr::filter(year < 2019)
  attributes(data) <- attributes(data)[sort(names(attributes(data)))]

  # correct for NAs in output
  expect_known_hash(is.na(data), "4c798df02b")
  data[is.na(data)] <- -999999
  expect_known_hash(data, "040fa9c6b1")
})
