% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/userdata.R
\name{userdata}
\alias{userdata}
\title{Facilities for making additional information to basic components}
\description{
When \acronym{POMP} basic components need information they can't get from parameters or covariates.
}
\details{
It can happen that one desires to pass information to one of the \acronym{POMP} model \dfn{basic components} (see \link[=pomp]{here for a definition of this term}) outside of the standard routes (i.e., via model parameters or covariates).
\pkg{pomp} provides facilities for this purpose.
We refer to the objects one wishes to pass in this way as \dfn{user data}.

The following will apply to every \link[=basic_components]{basic model component}.
For the sake of definiteness, however, we'll use the \code{rmeasure} component as an example.
To be even more specific, the measurement model we wish to implement is
\preformatted{
      y1 ~ Poisson(x1+theta),  y2 ~ Poisson(x2+theta),}
where \code{theta} is a parameter.
Although it would be very easy (and indeed far preferable) to include \code{theta} among the ordinary parameters (by including it in \code{params}), we will assume here that we have some reason for not wanting to do so.

Now, we have the choice of providing \code{rmeasure} in one of three ways:
\enumerate{
  \item as an \R function,
  \item as a C snippet, or
  \item as a procedure in an external, dynamically loaded library.
}
We'll deal with these three cases in turn.
}
\section{When the basic component is specified as an \R function}{

We can implement a simulator for the aforementioned measurement model so: \preformatted{
   f <- function (t, x, params, theta, ...) {
      y <- rpois(n=2,x[c("x1","x2")]+theta)
      setNames(y,c("y1","y2"))
   }}
So far, so good, but how do we get \code{theta} to this function?
We simply provide an additional argument to whichever \pkg{pomp} algorithm we are employing (e.g., \code{\link[=simulate-pomp]{simulate}}, \code{\link{pfilter}}, \code{\link{mif2}}, \code{\link{abc}}, etc.).
For example:
\preformatted{
    simulate(..., rmeasure = f, theta = 42, ...)
}
where the \code{\dots} represent the other \code{simulate} arguments we might want to supply.
When we do so, a message will be generated, informing us that \code{theta} is available for use by the \acronym{POMP} basic components.
This warning helps forestall accidental triggering of this facility due to typographical error.
}

\section{When the basic component is specified via a C snippet}{

A C snippet implementation of the aforementioned measurement model is:
\preformatted{
    f <- Csnippet("
     double theta = *(get_userdata_double(\"theta\"));
     y1 = rpois(x1+theta); y2 = rpois(x2+theta);
    ")}
Here, the call to \code{get_userdata_double} retrieves a \emph{pointer} to the stored value of \code{theta}.
Note the need to escape the quotes in the C snippet text.

It is possible to store and retrieve integer objects also, using \code{get_userdata_int}.

One must take care that one stores the user data with the appropriate storage type.
For example, it is wise to wrap floating point scalars and vectors with \code{as.double} and integers with \code{as.integer}.
In the present example, our call to simulate might look like
\preformatted{
    simulate(..., rmeasure = f, theta = as.double(42), ...)
}

Since the two functions \code{get_userdata_double} and \code{get_userdata_int} return pointers, it is trivial to pass vectors of double-precision and integers.

A simpler and more elegant approach is afforded by the \code{globals} argument (see below).
}

\section{When the basic component is specified via an external library}{


The rules are essentially the same as for C snippets.
\code{typedef} declarations for the \code{get_userdata_double} and \code{get_userdata_int} are given in the \file{pomp.h} header file and these two routines are registered so that they can be retrieved via a call to \code{R_GetCCallable}.
See the \href{https://cran.r-project.org/doc/manuals/R-exts.html}{Writing \R extensions manual} for more information.
}

\section{Setting \code{globals}}{


The use of the userdata facilities incurs a run-time cost.
It is faster and more elegant, when using C snippets, to put the needed objects directly into the C snippet library.
The \code{globals} argument does this.
See the example below.
}

\examples{
## The familiar Ricker example
## For some bizarre reason, we wish to pass 'phi' via the userdata facility.

## C snippet approach:

simulate(times=1:100,t0=0,
  phi=as.double(100),
  params=c(r=3.8,sigma=0.3,N.0=7),
  rprocess=discrete_time(
    step.fun=Csnippet("
      double e = (sigma > 0.0) ? rnorm(0,sigma) : 0.0;
      N = r*N*exp(-N+e);"
    ),
    delta.t=1
  ),
  rmeasure=Csnippet("
       double phi = *(get_userdata_double(\"phi\"));
       y = rpois(phi*N);"
  ),
  paramnames=c("r","sigma"),
  statenames="N",
  obsnames="y"
) -> rick1

## The same problem solved using 'globals':
simulate(times=1:100,t0=0,
  globals=Csnippet("static double phi = 100;"),
  params=c(r=3.8,sigma=0.3,N.0=7),
  rprocess=discrete_time(
    step.fun=Csnippet("
      double e = (sigma > 0.0) ? rnorm(0,sigma) : 0.0;
      N = r*N*exp(-N+e);"
    ),
    delta.t=1
  ),
  rmeasure=Csnippet("
       y = rpois(phi*N);"
  ),
  paramnames=c("r","sigma"),
  statenames="N",
  obsnames="y"
) -> rick2

## Finally, the R function approach:

simulate(times=1:100,t0=0,
  phi=100,
  params=c(r=3.8,sigma=0.3,N_0=7),
  rprocess=discrete_time(
    step.fun=function (r, N, sigma, ...) {
      e <- rnorm(n=1,mean=0,sd=sigma)
       c(N=r*N*exp(-N+e))
    },
    delta.t=1
  ),
  rmeasure=function(phi, N, ...) {
      c(y=rpois(n=1,lambda=phi*N))
  }
) -> rick3
}
\seealso{
More on implementing POMP models: 
\code{\link{Csnippet}},
\code{\link{accumulators}},
\code{\link{basic_components}},
\code{\link{covariate_table}()},
\code{\link{distributions}},
\code{\link{dmeasure_spec}},
\code{\link{dprocess_spec}},
\code{\link{parameter_trans}()},
\code{\link{pomp-package}},
\code{\link{prior_spec}},
\code{\link{rinit_spec}},
\code{\link{rmeasure_spec}},
\code{\link{rprocess_spec}},
\code{\link{skeleton_spec}},
\code{\link{transformations}}
}
\concept{implementation_info}
