## MIF algorithm functions

default.pomp.particles.fun <- function (Np, center, sd, ...) {
  matrix(
         data=rnorm(
           n=Np*length(center),
           mean=center,
           sd=sd
           ),
         nrow=length(center),
         ncol=Np,
         dimnames=list(
           names(center),
           NULL
           )
         )
}

mif.cooling <- function (factor, n) {   # default cooling schedule
  alpha <- factor^(n-1)
  list(alpha=alpha,gamma=alpha^2)
}

powerlaw.cooling <- function (init = 1, delta = 0.1, eps = (1-delta)/2, n) {
  m <- init
  if (n <= m) {                         # linear cooling regime
    alpha <- (m-n+1)/m
    gamma <- alpha
  } else {                              # power-law cooling regime
    alpha <- ((n/m)^(delta+eps))/n
    gamma <- ((n/m)^(delta+1))/n/n
  }
  list(alpha=alpha,gamma=gamma)
}

mif.internal <- function (object, Nmif,
                          start, pars, ivps,
                          particles,
                          rw.sd, 
                          Np, cooling.factor, var.factor, ic.lag,
                          weighted, tol, max.fail,
                          verbose, .ndone) {

  if (length(start)==0)
    stop(
         "mif error: ",sQuote("start")," must be specified if ",
         sQuote("coef(object)")," is NULL",
         call.=FALSE
         )
  start.names <- names(start)
  if (missing(start.names))
    stop("mif error: ",sQuote("start")," must be a named vector",call.=FALSE)

  if (missing(rw.sd))
    stop("mif error: ",sQuote("rw.sd")," must be specified",call.=FALSE)

  rw.names <- names(rw.sd)
  if (missing(rw.names) || any(rw.sd<0))
    stop("mif error: ",sQuote("rw.sd")," must be a named non-negative numerical vector",call.=FALSE)
  if (!all(rw.names%in%start.names))
    stop("mif error: all the names of ",sQuote("rw.sd")," must be names of ",sQuote("start"),call.=FALSE)
  rw.names <- names(rw.sd[rw.sd>0])
  if (length(rw.names) == 0)
    stop("mif error: ",sQuote("rw.sd")," must have one positive entry for each parameter to be estimated",call.=FALSE)

  if (missing(pars))
    stop("mif error: ",sQuote("pars")," must be specified",call.=FALSE)
  if (length(pars)==0)
    stop("mif error: at least one ordinary (non-IVP) parameter must be estimated",call.=FALSE)

  if (missing(ivps))
    stop("mif error: ",sQuote("ivps")," must be specified",call.=FALSE)

  if (
      !is.character(pars) ||
      !is.character(ivps) ||
      !all(pars%in%start.names) ||
      !all(ivps%in%start.names) ||
      any(pars%in%ivps) ||
      any(ivps%in%pars) ||
      !all(pars%in%rw.names) ||
      !all(ivps%in%rw.names)
      )
    stop(
         "mif error: ",
         sQuote("pars")," and ",sQuote("ivps"),
         " must be mutually disjoint subsets of ",
         sQuote("names(start)"),
         " and must have a positive random-walk SDs specified in ",
         sQuote("rw.sd"),
         call.=FALSE
         )

  if (!all(rw.names%in%c(pars,ivps))) {
    extra.rws <- rw.names[!(rw.names%in%c(pars,ivps))]
    warning(
            "mif warning: the variable(s) ",
            paste(extra.rws,collapse=", "),
            " have positive random-walk SDs specified, but are included in neither ",
            sQuote("pars")," nor ",sQuote("ivps"),
            ". These random walk SDs are ignored.",
            call.=FALSE
            )
  }
  rw.sd <- rw.sd[c(pars,ivps)]
  rw.names <- names(rw.sd)

  if (missing(particles))
    stop("mif error: ",sQuote("particles")," must be specified",call.=FALSE)
  
  if (missing(Np))
    stop("mif error: ",sQuote("Np")," must be specified",call.=FALSE)
  Np <- as.integer(Np)
  if ((length(Np)!=1)||(Np < 1))
    stop("mif error: ",sQuote("Np")," must be a positive integer",call.=FALSE)

  if (missing(ic.lag))
    stop("mif error: ",sQuote("ic.lag")," must be specified",call.=FALSE)
  ic.lag <- as.integer(ic.lag)
  if ((length(ic.lag)!=1)||(ic.lag < 1))
    stop("mif error: ",sQuote("ic.lag")," must be a positive integer",call.=FALSE)

  if (missing(cooling.factor))
    stop("mif error: ",sQuote("cooling.factor")," must be specified",call.=FALSE)
  if ((length(cooling.factor)!=1)||(cooling.factor < 0)||(cooling.factor>1))
    stop("mif error: ",sQuote("cooling.factor")," must be a number between 0 and 1",call.=FALSE)

  if (missing(var.factor))
    stop("mif error: ",sQuote("var.factor")," must be specified",call.=FALSE)
  if ((length(var.factor)!=1)||(var.factor < 0))
    stop("mif error: ",sQuote("var.factor")," must be a positive number",call.=FALSE)

  if (missing(Nmif))
    stop("mif error: ",sQuote("Nmif")," must be specified",call.=FALSE)
  Nmif <- as.integer(Nmif)
  if (Nmif<0)
    stop("mif error: ",sQuote("Nmif")," must be a positive integer",call.=FALSE)

  if (verbose) {
    cat("performing",Nmif,"MIF iteration(s) to estimate parameter(s)",
        paste(pars,collapse=", "))
    if (length(ivps)>0)
      cat(" and IVP(s)",paste(ivps,collapse=", "))
    cat(" using random-walk with SD\n")
    print(rw.sd)
    cat(
        "using",Np,"particles, variance factor",var.factor,
        "\ninitial condition smoothing lag",ic.lag,
        "and cooling factor",cooling.factor,"\n"
        )
  }

  theta <- start

  sigma <- rep(0,length(start))
  names(sigma) <- start.names

  rw.sd <- rw.sd[c(pars,ivps)]
  rw.names <- names(rw.sd)

  sigma[rw.names] <- rw.sd

  conv.rec <- matrix(
                     data=NA,
                     nrow=Nmif+1,
                     ncol=length(theta)+2,
                     dimnames=list(
                       seq(.ndone,.ndone+Nmif),
                       c('loglik','nfail',names(theta))
                       )
                     )
  conv.rec[1,] <- c(NA,NA,theta)

  if (!all(is.finite(theta[c(pars,ivps)]))) {
    stop(
         sQuote("mif"),
         " error: cannot estimate non-finite parameters: ",
         paste(
               c(pars,ivps)[!is.finite(theta[c(pars,ivps)])],
               collapse=","
               ),
         call.=FALSE
         )
  }
  
  obj <- as(object,"pomp")

  if (Nmif>0)
    tmp.mif <- new("mif",object,particles=particles,Np=Np) # only needed so that we can use the 'particles' method below
  else
    pfp <- obj

  for (n in seq_len(Nmif)) { # main loop

    ## compute the cooled sigma
    cool.sched <- try(
                      mif.cooling(cooling.factor,.ndone+n),
                      silent=FALSE
                      )
    if (inherits(cool.sched,'try-error'))
      stop("mif error: cooling schedule error",call.=FALSE)
    sigma.n <- sigma*cool.sched$alpha

    ## initialize the particles' parameter portion...
    P <- try(
             particles(tmp.mif,Np=Np,center=theta,sd=sigma.n*var.factor),
             silent=FALSE
             )
    if (inherits(P,'try-error'))
      stop("mif error: error in ",sQuote("particles"),call.=FALSE)

    ## run the particle filter
    pfp <- try(
               pfilter.internal(
                                object=obj,
                                params=P,
                                tol=tol,
                                max.fail=max.fail,
                                pred.mean=(n==Nmif),
                                pred.var=(weighted||(n==Nmif)),
                                filter.mean=TRUE,
                                save.states=FALSE,
                                .rw.sd=sigma.n[pars],
                                verbose=verbose
                                ),
               silent=FALSE
               )
    if (inherits(pfp,'try-error'))
      stop("mif error: error in ",sQuote("pfilter"),call.=FALSE)

    if (weighted) {           # MIF update rule
      v <- pfp$pred.var[pars,,drop=FALSE] # the prediction variance
      v1 <- cool.sched$gamma*(1+var.factor^2)*sigma[pars]^2
      theta.hat <- cbind(theta[pars],pfp$filter.mean[pars,,drop=FALSE])
      theta[pars] <- theta[pars]+colSums(apply(theta.hat,1,diff)/t(v))*v1
    } else {                  # unweighted (flat) average
      theta.hat <- pfp$filter.mean[pars,,drop=FALSE]
      theta[pars] <- rowMeans(theta.hat)
    }
    
    ## update the IVPs using fixed-lag smoothing
    theta[ivps] <- pfp$filter.mean[ivps,ic.lag]

    ## store a record of this iteration
    conv.rec[n+1,-c(1,2)] <- theta
    conv.rec[n,c(1,2)] <- c(pfp$loglik,pfp$nfail)

    if (verbose) cat("MIF iteration ",n," of ",Nmif," completed\n")

  }

  new(
      "mif",
      pfp,
      params=theta,
      ivps=ivps,
      pars=pars,
      Nmif=Nmif,
      particles=particles,
      var.factor=var.factor,
      ic.lag=ic.lag,
      cooling.factor=cooling.factor,
      random.walk.sd=sigma[rw.names],
      tol=tol,
      conv.rec=conv.rec
      )
}

setGeneric('mif',function(object,...)standardGeneric("mif"))

setMethod(
          "mif",
          signature=signature(object="pomp"),
          function (object, Nmif = 1,
                    start,
                    pars, ivps = character(0),
                    particles, rw.sd,
                    Np, ic.lag, var.factor, cooling.factor,
                    weighted = TRUE, tol = 1e-17, max.fail = 0,
                    verbose = getOption("verbose"), ...) {

            if (missing(start)) start <- coef(object)
            if (missing(rw.sd))
              stop("mif error: ",sQuote("rw.sd")," must be specified",call.=FALSE)
            if (missing(pars)) {
              rw.names <- names(rw.sd)[rw.sd>0]
              pars <- rw.names[!(rw.names%in%ivps)]
            }
            if (missing(Np))
              stop("mif error: ",sQuote("Np")," must be specified",call.=FALSE)
            if (missing(ic.lag))
              stop("mif error: ",sQuote("ic.lag")," must be specified",call.=FALSE)
            if (missing(var.factor))
              stop("mif error: ",sQuote("var.factor")," must be specified",call.=FALSE)
            if (missing(cooling.factor))
              stop("mif error: ",sQuote("cooling.factor")," must be specified",call.=FALSE)
            
            if (missing(particles)) {         # use default: normal distribution
              particles <- function (Np, center, sd, ...) {
                matrix(
                       data=rnorm(
                         n=Np*length(center),
                         mean=center,
                         sd=sd
                         ),
                       nrow=length(center),
                       ncol=Np,
                       dimnames=list(
                         names(center),
                         NULL
                         )
                       )
              }
            } else {
              particles <- match.fun(particles)
              if (!all(c('Np','center','sd','...')%in%names(formals(particles))))
                stop(
                     "mif error: ",
                     sQuote("particles"),
                     " must be a function of prototype ",
                     sQuote("particles(Np,center,sd,...)"),
                     call.=FALSE
                     )
            }
            
            mif.internal(
                         object=object,
                         Nmif=Nmif,
                         start=start,
                         pars=pars,
                         ivps=ivps,
                         particles=particles,
                         rw.sd=rw.sd,
                         Np=Np,
                         cooling.factor=cooling.factor,
                         var.factor=var.factor,
                         ic.lag=ic.lag,
                         weighted=weighted,
                         tol=tol,
                         max.fail=max.fail,
                         verbose=verbose,
                         .ndone=0
                         )

          }
          )


setMethod(
          "mif",
          signature=signature(object="pfilterd.pomp"),
          function (object, Nmif = 1,
                    start,
                    pars, ivps = character(0),
                    particles, rw.sd,
                    Np, ic.lag, var.factor, cooling.factor,
                    weighted = TRUE, tol, max.fail = 0,
                    verbose = getOption("verbose"), ...) {

            if (missing(start)) start <- coef(object)
            if (missing(rw.sd))
              stop("mif error: ",sQuote("rw.sd")," must be specified",call.=FALSE)
            if (missing(pars)) {
              rw.names <- names(rw.sd)[rw.sd>0]
              pars <- rw.names[!(rw.names%in%ivps)]
            }
            if (missing(Np)) Np <- object@Np
            if (missing(tol)) tol <- object@tol
            if (missing(ic.lag))
              stop("mif error: ",sQuote("ic.lag")," must be specified",call.=FALSE)
            if (missing(var.factor))
              stop("mif error: ",sQuote("var.factor")," must be specified",call.=FALSE)
            if (missing(cooling.factor))
              stop("mif error: ",sQuote("cooling.factor")," must be specified",call.=FALSE)
            
            if (missing(particles)) {         # use default: normal distribution
              particles <- default.pomp.particles.fun
            } else {
              particles <- match.fun(particles)
              if (!all(c('Np','center','sd','...')%in%names(formals(particles))))
                stop(
                     "mif error: ",
                     sQuote("particles"),
                     " must be a function of prototype ",
                     sQuote("particles(Np,center,sd,...)"),
                     call.=FALSE
                     )
            }
            
            mif.internal(
                         object=as(object,"pomp"),
                         Nmif=Nmif,
                         start=start,
                         pars=pars,
                         ivps=ivps,
                         particles=particles,
                         rw.sd=rw.sd,
                         Np=Np,
                         cooling.factor=cooling.factor,
                         var.factor=var.factor,
                         ic.lag=ic.lag,
                         weighted=weighted,
                         tol=tol,
                         max.fail=max.fail,
                         verbose=verbose,
                         .ndone=0
                         )
          }
          )

setMethod(
          "mif",
          signature=signature(object="mif"),
          function (object, Nmif,
                    start,
                    pars, ivps,
                    particles, rw.sd,
                    Np, ic.lag, var.factor, cooling.factor,
                    weighted = TRUE, tol, max.fail = 0,
                    verbose = getOption("verbose"), ...) {

            if (missing(Nmif)) Nmif <- object@Nmif
            if (missing(start)) start <- coef(object)
            if (missing(pars)) pars <- object@pars
            if (missing(ivps)) ivps <- object@ivps
            if (missing(particles)) particles <- object@particles
            if (missing(rw.sd)) rw.sd <- object@random.walk.sd
            if (missing(Np)) Np <- object@Np
            if (missing(ic.lag)) ic.lag <- object@ic.lag
            if (missing(var.factor)) var.factor <- object@var.factor
            if (missing(cooling.factor)) cooling.factor <- object@cooling.factor
            if (missing(tol)) tol <- object@tol

            mif.internal(
                         object=as(object,"pomp"),
                         Nmif=Nmif,
                         start=start,
                         pars=pars,
                         ivps=ivps,
                         particles=particles,
                         rw.sd=rw.sd,
                         Np=Np,
                         cooling.factor=cooling.factor,
                         var.factor=var.factor,
                         ic.lag=ic.lag,
                         weighted=weighted,
                         tol=tol,
                         max.fail=max.fail,
                         verbose=verbose,
                         .ndone=0
                         )
          }
          )

setMethod(
          'continue',
          signature=signature(object='mif'),
          function (object, Nmif = 1,
                    start,
                    pars, ivps,
                    particles, rw.sd,
                    Np, ic.lag, var.factor, cooling.factor,
                    weighted = TRUE, tol, max.fail = 0,
                    verbose = getOption("verbose"), ...) {

            ndone <- object@Nmif
            if (missing(start)) start <- coef(object)
            if (missing(pars)) pars <- object@pars
            if (missing(ivps)) ivps <- object@ivps
            if (missing(particles)) particles <- object@particles
            if (missing(rw.sd)) rw.sd <- object@random.walk.sd
            if (missing(Np)) Np <- object@Np
            if (missing(ic.lag)) ic.lag <- object@ic.lag
            if (missing(var.factor)) var.factor <- object@var.factor
            if (missing(cooling.factor)) cooling.factor <- object@cooling.factor
            if (missing(tol)) tol <- object@tol

            obj <- mif.internal(
                                object=as(object,"pomp"),
                                Nmif=Nmif,
                                start=start,
                                pars=pars,
                                ivps=ivps,
                                particles=particles,
                                rw.sd=rw.sd,
                                Np=Np,
                                cooling.factor=cooling.factor,
                                var.factor=var.factor,
                                ic.lag=ic.lag,
                                weighted=weighted,
                                tol=tol,
                                max.fail=max.fail,
                                verbose=verbose,
                                .ndone=ndone
                                )

            object@conv.rec[ndone+1,c('loglik','nfail')] <- obj@conv.rec[1,c('loglik','nfail')]
            obj@conv.rec <- rbind(
                                  object@conv.rec,
                                  obj@conv.rec[-1,colnames(object@conv.rec)]
                                  )
            obj@Nmif <- as.integer(ndone+Nmif)

            obj
          }
          )

mif.profile.design <- function (object, profile, lower, upper, nprof, ivps, 
                                rw.sd, Np, ic.lag, var.factor, cooling.factor, ...)
  {
    if (missing(profile)) profile <- list()
    if (missing(lower)) lower <- numeric(0)
    if (missing(upper)) upper <- lower
    if (length(lower)!=length(upper))
      stop(sQuote("lower")," and ",sQuote("upper")," must be of the same length")
    pars <- names(lower)
    if (missing(ivps)) ivps <- character(0)
    Np <- as.integer(Np)

    pd <- do.call(profile.design,c(profile,list(lower=lower,upper=upper,nprof=nprof)))

    object <- as(object,"pomp")

    pp <- coef(object)
    idx <- !(names(pp)%in%names(pd))
    if (any(idx)) pd <- cbind(pd,as.list(pp[idx]))
    
    ans <- vector(mode="list",length=nrow(pd))
    for (k in seq_len(nrow(pd))) {
      ans[[k]] <- list(
                       mf=mif(
                         object,
                         Nmif=0,
                         start=unlist(pd[k,]),
                         pars=pars,
                         ivps=ivps,
                         rw.sd=rw.sd,
                         Np=Np,
                         ic.lag=ic.lag,
                         var.factor=var.factor,
                         cooling.factor=cooling.factor,
                         ...
                         )
                       )
    }

    ans
  }
