% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_agent.R
\name{create_agent}
\alias{create_agent}
\title{Create a \strong{pointblank} \emph{agent} object}
\usage{
create_agent(
  tbl = NULL,
  tbl_name = NULL,
  label = NULL,
  actions = NULL,
  end_fns = NULL,
  embed_report = FALSE,
  lang = NULL,
  locale = NULL,
  read_fn = NULL
)
}
\arguments{
\item{tbl}{The input table. This can be a data frame, a tibble, a \code{tbl_dbi}
object, or a \code{tbl_spark} object. Alternatively, an expression can be
supplied to serve as instructions on how to retrieve the target table at
interrogation-time. There are two ways to specify an association to a
target table: (1) as a table-prep formula, which is a right-hand side (RHS)
formula expression (e.g., \verb{~ \{ <table reading code>\}}), or (2) as a
function (e.g., \verb{function() \{ <table reading code>\}}).}

\item{tbl_name}{A optional name to assign to the input table object. If no
value is provided, a name will be generated based on whatever information
is available. This table name will be displayed in the header area of the
agent report generated by printing the \emph{agent} or calling
\code{\link[=get_agent_report]{get_agent_report()}}.}

\item{label}{An optional label for the validation plan. If no value is
provided, a label will be generated based on the current system time.
Markdown can be used here to make the label more visually appealing (it
will appear in the header area of the agent report).}

\item{actions}{A option to include a list with threshold levels so that all
validation steps can react accordingly when exceeding the set levels. This
is to be created with the \code{\link[=action_levels]{action_levels()}} helper function. Should an
action levels list be used for a specific validation step, the default set
specified here will be overridden.}

\item{end_fns}{A list of expressions that should be invoked at the end of an
interrogation. Each expression should be in the form of a one-sided R
formula, so overall this construction should be used: \verb{end_fns = list(~ <R statements>, ~ <R statements>, ...)}. An example of a function included in
\strong{pointblank} that can be sensibly used here is \code{\link[=email_blast]{email_blast()}}, which
sends an email of the validation report (based on a sending condition).}

\item{embed_report}{An option to embed a \strong{gt}-based validation report into
the \code{ptblank_agent} object. If \code{FALSE} (the default) then the table object
will be not generated and available with the \emph{agent} upon returning from
the interrogation.}

\item{lang}{The language to use for automatic creation of briefs (short
descriptions for each validation step) and for the \emph{agent report} (a
summary table that provides the validation plan and the results from the
interrogation. By default, \code{NULL} will create English (\code{"en"}) text. Other
options include French (\code{"fr"}), German (\code{"de"}), Italian (\code{"it"}), Spanish
(\code{"es"}), Portuguese (\code{"pt"}), Turkish (\code{"tr"}), Chinese (\code{"zh"}), Russian
(\code{"ru"}), Polish (\code{"pl"}), Danish (\code{"da"}), Swedish (\code{"sv"}), and Dutch
(\code{"nl"}).}

\item{locale}{An optional locale ID to use for formatting values in the
\emph{agent report} summary table according the locale's rules. Examples include
\code{"en_US"} for English (United States) and \code{"fr_FR"} for French (France);
more simply, this can be a language identifier without a country
designation, like "es" for Spanish (Spain, same as \code{"es_ES"}).}

\item{read_fn}{The \code{read_fn} argument is deprecated. Instead, supply a
table-prep formula or function to \code{tbl}.}
}
\value{
A \code{ptblank_agent} object.
}
\description{
The \code{create_agent()} function creates an \emph{agent} object, which is used in a
\emph{data quality reporting} workflow. The overall aim of this workflow is to
generate useful reporting information for assessing the level of data quality
for the target table. We can supply as many validation functions as the user
wishes to write, thereby increasing the level of validation coverage for that
table. The \emph{agent} assigned by the \code{create_agent()} call takes validation
functions (e.g., \code{\link[=col_vals_between]{col_vals_between()}}, \code{\link[=rows_distinct]{rows_distinct()}}, etc.), which
translate to discrete validation steps (each one is numbered and will later
provide its own set of results). This process is known as developing a
\emph{validation plan}.

The validation functions, when called on an \emph{agent}, are merely instructions
up to the point the \code{\link[=interrogate]{interrogate()}} function is called. That kicks off the
process of the \emph{agent} acting on the \emph{validation plan} and getting results
for each step. Once the interrogation process is complete, we can say that
the \emph{agent} has intel. Calling the \emph{agent} itself will result in a reporting
table. This reporting of the interrogation can also be accessed with the
\code{\link[=get_agent_report]{get_agent_report()}} function, where there are more reporting options.
}
\section{Supported Input Tables}{

The types of data tables that are officially supported are:
\itemize{
\item data frames (\code{data.frame}) and tibbles (\code{tbl_df})
\item Spark DataFrames (\code{tbl_spark})
\item the following database tables (\code{tbl_dbi}):
\itemize{
\item \emph{PostgreSQL} tables (using the \code{RPostgres::Postgres()} as driver)
\item \emph{MySQL} tables (with \code{RMySQL::MySQL()})
\item \emph{Microsoft SQL Server} tables (via \strong{odbc})
\item \emph{BigQuery} tables (using \code{bigrquery::bigquery()})
\item \emph{DuckDB} tables (through \code{duckdb::duckdb()})
\item \emph{SQLite} (with \code{RSQLite::SQLite()})
}
}

Other database tables may work to varying degrees but they haven't been
formally tested (so be mindful of this when using unsupported backends with
\strong{pointblank}).
}

\section{The Use of an Agent for Validation Is Just One Option of Several}{


There are a few validation workflows and using an \emph{agent} is the one that
provides the most options. It is probably the best choice for assessing the
state of data quality since it yields detailed reporting, has options for
further exploration of root causes, and allows for granular definition of
actions to be taken based on the severity of validation failures (e.g.,
emailing, logging, etc.).

Different situations, however, call for different validation workflows. You
use validation functions (the same ones you would with an \emph{agent}) directly
on the data. This acts as a sort of data filter in that the input table will
become output data (without modification), but there may be warnings, errors,
or other side effects that you can define if validation fails. Basically,
instead of this

\if{html}{\out{<div class="sourceCode r">}}\preformatted{create_agent(tbl = small_table) \%>\% rows_distinct() \%>\% interrogate()
}\if{html}{\out{</div>}}

you would use this:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% rows_distinct()
}\if{html}{\out{</div>}}

This results in an error (with the default failure threshold settings),
displaying the reason for the error in the console. Notably, the data is not
passed though.

We can use variants of the validation functions, the \emph{test} (\verb{test_*()}) and
\emph{expectation} (\verb{expect_*()}) versions, directly on the data for different
workflows. The first returns to us a logical value. So this

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% test_rows_distinct()
}\if{html}{\out{</div>}}

returns \code{FALSE} instead of an error.

In a unit testing scenario, we can use \emph{expectation} functions exactly as we
would with \strong{testthat}'s library of \verb{expect_*()} functions:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% expect_rows_distinct()
}\if{html}{\out{</div>}}

This test of \code{small_table} would be counted as a failure.
}

\section{The Agent Report}{


While printing an \emph{agent} (a \code{ptblank_agent} object) will display its
reporting in the Viewer, we can alternatively use the \code{\link[=get_agent_report]{get_agent_report()}} to
take advantage of other options (e.g., overriding the language, modifying the
arrangement of report rows, etc.), and to return the report as independent
objects. For example, with the \code{display_table = TRUE} option (the default),
\code{\link[=get_agent_report]{get_agent_report()}} will return a \code{ptblank_agent_report} object. If
\code{display_table} is set to \code{FALSE}, we'll get a data frame back instead.

Exporting the report as standalone HTML file can be accomplished by using the
\code{\link[=export_report]{export_report()}} function. This function can accept either the
\code{ptblank_agent} object or the \code{ptblank_agent_report} as input. Each HTML
document written to disk in this way is self-contained and easily viewable in
a web browser.
}

\section{Data Products Obtained from an Agent}{


A very detailed list object, known as an x-list, can be obtained by using the
\code{\link[=get_agent_x_list]{get_agent_x_list()}} function on the \emph{agent}. This font of information can be
taken as a whole, or, broken down by the step number (with the \code{i} argument).

Sometimes it is useful to see which rows were the failing ones. By using the
\code{\link[=get_data_extracts]{get_data_extracts()}} function on the \emph{agent}, we either get a list of
tibbles (for those steps that have data extracts) or one tibble if the
validation step is specified with the \code{i} argument.

The target data can be split into pieces that represent the 'pass' and 'fail'
portions with the \code{\link[=get_sundered_data]{get_sundered_data()}} function. A primary requirement is an
agent that has had \code{\link[=interrogate]{interrogate()}} called on it. In addition, the validation
steps considered for this data splitting need to be those that operate on
values down a column (e.g., the \verb{col_vals_*()} functions or \code{\link[=conjointly]{conjointly()}}).
With these in-consideration validation steps, rows with no failing test units
across all validation steps comprise the 'pass' data piece, and rows with at
least one failing test unit across the same series of validations constitute
the 'fail' piece.

If we just need to know whether all validations completely passed (i.e., all
steps had no failing test units), the \code{\link[=all_passed]{all_passed()}} function could be used
on the \emph{agent}. However, in practice, it's not often the case that all data
validation steps are free from any failing units.
}

\section{YAML}{


A \strong{pointblank} agent can be written to YAML with \code{\link[=yaml_write]{yaml_write()}} and the
resulting YAML can be used to regenerate an agent (with \code{\link[=yaml_read_agent]{yaml_read_agent()}})
or interrogate the target table (via \code{\link[=yaml_agent_interrogate]{yaml_agent_interrogate()}}). Here is an
example of how a complex call of \code{create_agent()} is expressed in R code and
in the corresponding YAML representation.

R statement:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{create_agent(
  tbl = ~ small_table,
  tbl_name = "small_table",
  label = "An example.",
  actions = action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35,
    fns = list(notify = ~ email_blast(
      x,
      to = "joe_public@example.com",
      from = "pb_notif@example.com",
      msg_subject = "Table Validation",
      credentials = blastula::creds_key(
        id = "smtp2go"
      )
    ))
  ), 
  end_fns = list(
    ~ beepr::beep(2),
    ~ Sys.sleep(1)
  ), 
  embed_report = TRUE,
  lang = "fr", 
  locale = "fr_CA"
)
}\if{html}{\out{</div>}}

YAML representation:

\if{html}{\out{<div class="sourceCode yaml">}}\preformatted{type: agent
tbl: ~small_table
tbl_name: small_table
label: An example.
lang: fr
locale: fr_CA
actions:
  warn_fraction: 0.1
  stop_fraction: 0.25
  notify_fraction: 0.35
  fns:
    notify: ~email_blast(x, to = "joe_public@example.com",
      from = "pb_notif@example.com",
      msg_subject = "Table Validation",
      credentials = blastula::creds_key(id = "smtp2go"))
end_fns:
- ~beepr::beep(2)
- ~Sys.sleep(1)
embed_report: true
steps: []
}\if{html}{\out{</div>}}

In practice, this YAML file will be shorter since arguments with default
values won't be written to YAML when using \code{\link[=yaml_write]{yaml_write()}} (though it is
acceptable to include them with their default when generating the YAML by
other means). The only requirement for writing the YAML representation of an
\emph{agent} is having \code{tbl} specified as table-prep formula.

What typically follows this chunk of YAML is a \code{steps} part, and that
corresponds to the addition of validation steps via validation functions.
Help articles for each validation function have a \emph{YAML} section that
describes how a given validation function is translated to YAML.

Should you need to preview the transformation of an \emph{agent} to YAML (without
any committing anything to disk), use the \code{\link[=yaml_agent_string]{yaml_agent_string()}} function. If
you already have a \code{.yml} file that holds an \emph{agent}, you can get a glimpse
of the R expressions that are used to regenerate that agent with
\code{\link[=yaml_agent_show_exprs]{yaml_agent_show_exprs()}}.
}

\section{Writing an Agent to Disk}{


An \emph{agent} object can be written to disk with the \code{\link[=x_write_disk]{x_write_disk()}} function.
This can be useful for keeping a history of validations and generating views
of data quality over time. Agents are stored in the serialized RDS format and
can be easily retrieved with the \code{\link[=x_read_disk]{x_read_disk()}} function.

It's recommended that table-prep formulas are supplied to the \code{tbl}
argument of \code{create_agent()}. In this way, when an \emph{agent} is read from disk
through \code{\link[=x_read_disk]{x_read_disk()}}, it can be reused to access the target table (which
may change, hence the need to use an expression for this).
}

\section{Combining Several Agents in a \emph{multiagent} Object}{


Multiple \emph{agent} objects can be part of a \emph{multiagent} object, and two
functions can be used for this: \code{\link[=create_multiagent]{create_multiagent()}} and
\code{\link[=read_disk_multiagent]{read_disk_multiagent()}}. By gathering multiple agents that have performed
interrogations in the past, we can get a \emph{multiagent} report showing how data
quality evolved over time. This use case is interesting for data quality
monitoring and management, and, the reporting (which can be customized with
\code{\link[=get_multiagent_report]{get_multiagent_report()}}) is robust against changes in validation steps for
a given target table.
}

\section{Examples}{

\subsection{Creating an agent, adding a validation plan, and interrogating}{

Let's walk through a data quality analysis of an extremely small table. It's
actually called \code{small_table} and we can find it as a dataset in this
package.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## # A tibble: 13 × 8
##    date_time           date           a b             c      d e     f    
##    <dttm>              <date>     <int> <chr>     <dbl>  <dbl> <lgl> <chr>
##  1 2016-01-04 11:00:00 2016-01-04     2 1-bcd-345     3  3423. TRUE  high 
##  2 2016-01-04 00:32:00 2016-01-04     3 5-egh-163     8 10000. TRUE  low  
##  3 2016-01-05 13:32:00 2016-01-05     6 8-kdg-938     3  2343. TRUE  high 
##  4 2016-01-06 17:23:00 2016-01-06     2 5-jdo-903    NA  3892. FALSE mid  
##  5 2016-01-09 12:36:00 2016-01-09     8 3-ldm-038     7   284. TRUE  low  
##  6 2016-01-11 06:15:00 2016-01-11     4 2-dhe-923     4  3291. TRUE  mid  
##  7 2016-01-15 18:46:00 2016-01-15     7 1-knw-093     3   843. TRUE  high 
##  8 2016-01-17 11:27:00 2016-01-17     4 5-boe-639     2  1036. FALSE low  
##  9 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
## 10 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
## 11 2016-01-26 20:07:00 2016-01-26     4 2-dmx-010     7   834. TRUE  low  
## 12 2016-01-28 02:51:00 2016-01-28     2 7-dmx-010     8   108. FALSE low  
## 13 2016-01-30 11:23:00 2016-01-30     1 3-dka-303    NA  2230. TRUE  high
}\if{html}{\out{</div>}}

We ought to think about what's tolerable in terms of data quality so let's
designate proportional failure thresholds to the \code{warn}, \code{stop}, and \code{notify}
states using \code{\link[=action_levels]{action_levels()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{al <- 
  action_levels(
      warn_at = 0.10,
      stop_at = 0.25,
    notify_at = 0.35
  )
}\if{html}{\out{</div>}}

Now create a pointblank \code{agent} object and give it the \code{al} object (which
serves as a default for all validation steps which can be overridden). The
static thresholds provided by \code{al} will make the reporting a bit more useful.
We also provide a target table and we'll use \code{pointblank::small_table}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <- 
  create_agent(
    tbl = pointblank::small_table,
    tbl_name = "small_table",
    label = "`create_agent()` example.",
    actions = al
  )
}\if{html}{\out{</div>}}

Then, as with any \code{agent} object, we can add steps to the validation plan by
using as many validation functions as we want. then, we use \code{\link[=interrogate]{interrogate()}}
to actually perform the validations and gather intel.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <-
  agent \%>\% 
  col_exists(columns = vars(date, date_time)) \%>\%
  col_vals_regex(
    columns = vars(b),
    regex = "[0-9]-[a-z]\{3\}-[0-9]\{3\}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(columns = vars(d), value = 100) \%>\%
  col_vals_lte(columns = vars(c), value = 5) \%>\%
  col_vals_between(
    columns = vars(c),
    left = vars(a), right = vars(d),
    na_pass = TRUE
  ) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

The \code{agent} object can be printed to see the validation report in the
Viewer.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent
}\if{html}{\out{</div>}}

\if{html}{

\out{
<img src="https://raw.githubusercontent.com/rich-iannone/pointblank/main/images/man_create_agent_1.png" style="width:100\%;">
}
}

If we want to make use of more report display options, we can alternatively
use the \code{\link[=get_agent_report]{get_agent_report()}} function.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{report <-
  get_agent_report(
    agent = agent,
    arrange_by = "severity",
    title = "Validation of `small_table`"
  )

report
}\if{html}{\out{</div>}}

\if{html}{

\out{
<img src="https://raw.githubusercontent.com/rich-iannone/pointblank/main/images/man_create_agent_2.png" style="width:100\%;">
}
}
}

\subsection{Post-interrogation operations}{

We can use the \code{agent} object with a variety of functions to get at more
of the information collected during interrogation.

We can see from the validation report that Step 4 (which used the
\code{\link[=rows_distinct]{rows_distinct()}} validation function) had two test units, corresponding to
duplicated rows, that failed. We can see those rows with
\code{\link[=get_data_extracts]{get_data_extracts()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent \%>\% get_data_extracts(i = 4)
}\if{html}{\out{</div>}}

\preformatted{## # A tibble: 2 × 8
##   date_time           date           a b            c     d e     f    
##   <dttm>              <date>     <int> <chr>    <dbl> <dbl> <lgl> <chr>
## 1 2016-01-20 04:30:00 2016-01-20     3 5-bce-6…     9  838. FALSE high 
## 2 2016-01-20 04:30:00 2016-01-20     3 5-bce-6…     9  838. FALSE high}



We can get an x-list for the entire validation process (7 steps), or, just
for the 4th step with \code{\link[=get_agent_x_list]{get_agent_x_list()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{xl_step_4 <- agent \%>\% get_agent_x_list(i = 4)
}\if{html}{\out{</div>}}

And then we can peruse the different parts of the list. Let's get the
fraction of test units that failed.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{xl_step_4$f_failed
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] 0.15385
}\if{html}{\out{</div>}}

An x-list not specific to any step will have way more information and a
slightly different structure. See \code{help(get_agent_x_list)} for more info.
}
}

\section{Function ID}{

1-2
}

\seealso{
Other Planning and Prep: 
\code{\link{action_levels}()},
\code{\link{create_informant}()},
\code{\link{db_tbl}()},
\code{\link{draft_validation}()},
\code{\link{file_tbl}()},
\code{\link{scan_data}()},
\code{\link{tbl_get}()},
\code{\link{tbl_source}()},
\code{\link{tbl_store}()},
\code{\link{validate_rmd}()}
}
\concept{Planning and Prep}
