#' fit_phenology fits parameters to timeseries.
#' @title Fits the phenology parameters to timeseries.
#' @author Marc Girondot
#' @return Return a list of with data and result
#' @param data A dataset generated by add_format
#' @param parametersfixed Set of fixed parameters
#' @param parametersfit Set of parameters to be fitted
#' @param trace If 1, displays the progression of fit; 0 is silent (don't be afraid if it is long !)
#' @param maxit Number of iterations for search before checking if it converges. If it does not converge, will continue. Default 500.
#' @param method_incertitude 2 [default] is the correct one from a statistical point of view;\cr
#'                           0 is an aproximate method more rapid and using less memory;\cr
#'                           1 is an alternative more rapid but biased.
#' @param zero_counts example c(TRUE, TRUE, FALSE) indicates whether the zeros have 
#'                    been recorder for each of these timeseries. Defaut is TRUE for all.
#' @param hessian If FALSE does not estimate se of parameters
#' @param help If TRUE, an help is displayed
#' @description Function of the package phenology to fit parameters to timeseries.\cr
#' To fit data, the syntaxe is :\cr
#' Result<-fit_phenology(data=dataset, parametersfit=par, parametersfixed=pfixed, trace=1, method_incertitude=2, zero_counts=TRUE, hessian=TRUE)\cr
#' or if no parameter is fixed :\cr
#' Result<-fit_phenology(data=dataset, parametersfit=par)\cr
#' or\cr
#' fit_phenology(help=TRUE) to have this help !\cr
#' Add trace=1 [default] to have information on the fit progression or trace=0 to hide information on the fit progression.\cr
#' method_incertitude=2 [default] is the correct one from a statistical point of view.\cr
#' method_incertitude=0 is an aproximate method more rapid and using less memory.\cr
#' method_incertitude=1 is an alternative more rapid but potentially biased.\cr
#' zero_counts=c(TRUE, TRUE, FALSE) indicates whether the zeros have been recorded for each of these timeseries. Defaut is TRUE for all.\cr
#' hessian=FALSE does not estimate se of parameters.
#' @examples
#' library(phenology)
#' # Read a file with data
#' # Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", , header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_format(origin=NULL, add=Gratiot, name="Complete", reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, parametersfixed=NULL)
#' # Run the optimisation
#' # result_Gratiot<-fit_phenology(data=data_Gratiot, parametersfit=parg, parametersfixed=NULL, trace=1)
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' output<-plot_phenology(result=result_Gratiot, pdf=FALSE)
#' @export


fit_phenology <-
function(data=NULL, parametersfit=NULL, parametersfixed=NA, trace=1, maxit=500, method_incertitude=2, zero_counts=TRUE, hessian=TRUE, help=FALSE) {
if ((help)||(is.null(data))||(is.null(parametersfit))) {
	cat("To fit data, the syntaxe is :\n")
	cat("Result<-fit_phenology(data=dataset, parametersfit=par, parametersfixed=pfixed, trace=1,\n")
	cat("+      method_incertitude=2, zero_counts=TRUE, hessian=TRUE)\n")
	cat("or if no parameter is fixed :\n")
	cat("Result<-fit_phenology(data=dataset, parametersfit=par)\n")
	cat("or\n")	
	cat("fit_phenology(help=TRUE) to have this help !\n")
	cat("Add trace=1 [default] to have information on the fit progression.\n")
	cat("or trace=0 to hide information on the fit progression.\n")
	cat("method_incertitude=2 [default] is the correct one from a statistical point of view.\n")
	cat("method_incertitude=0 is an aproximate method more rapid.\n")
	cat("method_incertitude=1 is an alternative more rapid but biased.\n")
	cat("zero_counts=c(TRUE, TRUE, FALSE) indicates whether the zeros have\n")
	cat("been recorder for each of these timeseries. Defaut is TRUE for all.\n")
	cat("hessian=FALSE does not estimate se of parameters.\n")
} else {

#.phenology.env<- NULL
#rm(.phenology.env)

if (is.null(parametersfixed)) {parametersfixed<-NA}

if (class(data)!="phenologydata") {
  cat("Data must be formated first using the function add_format().\n")
  return()
}

	
	if (length(zero_counts)==1) {zero_counts<-rep(zero_counts, length(data))}
	if (length(zero_counts)!=length(data)) {
		print("zero_counts parameter must be TRUE (the zeros are used for all timeseries) or FALSE (the zeros are not used for all timeseries) or with the same number of logical values (TRUE or FALSE) than the number of series analyzed.")
		return()
	}


	repeat {
		resul<-optim(parametersfit, .Lnegbin, pt=list(data=data, fixed=parametersfixed, incertitude=method_incertitude, zerocounts=zero_counts) , method="BFGS",control=list(trace=trace, REPORT=1, maxit=maxit),hessian=FALSE)
		if (resul$convergence==0) break
		parametersfit<-resul$par
		print("Convergence is not acheived. Optimization continues !")
	}
	
	resfit<-resul$par
	resfit[substr(names(resfit), 1, 4)=="Peak"]<-abs(resfit[substr(names(resfit), 1, 4)=="Peak"])
	resfit["Theta"]<-abs(resfit["Theta"])
	resfit["PMinE"]<-abs(resfit["PMinE"])
	resfit["PMinB"]<-abs(resfit["PMinB"])
	resfit["Flat"]<-abs(resfit["Flat"])
	resfit[substr(names(resfit), 1, 6)=="Length"]<-abs(resfit[substr(names(resfit), 1, 6)=="Length"])
	resfit[substr(names(resfit), 1, 3)=="Min"]<-abs(resfit[substr(names(resfit), 1, 3)=="Min"])
	resfit[substr(names(resfit), 1, 3)=="Max"]<-abs(resfit[substr(names(resfit), 1, 3)=="Max"])
	resfit<-resfit[!is.na(resfit)]
	cat("Fit done!\n")
	cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
	if (hessian) {
	cat("Estimation of the standard error of parameters. Be patient please.\n")
	
	resul<-optim(resfit, .Lnegbin, pt=list(data=data, fixed=parametersfixed, incertitude=method_incertitude, zerocounts=zero_counts), method="BFGS",control=list(trace=0, REPORT=1, maxit=10),hessian=TRUE)

	resfit<-resul$par

	mathessian<-resul$hessian
	inversemathessian=try(solve(mathessian), silent=TRUE)
	if (substr(inversemathessian[1], 1, 5)=="Error") {
		print("Error in the fit; probably one or more parameters are not estimable.")
		print("Standard errors cannot be estimated.")
		res_se<-rep(NA, length(resfit))
	
	} else {
		res_se_diag=diag(inversemathessian)
		
		res_se <- rep(NA, length(resfit))
		res_se[res_se_diag>=0]<-sqrt(res_se_diag[res_se_diag>=0])

	}
	} else {
	
		print("Standard errors are not estimated.")
		res_se<-rep(NA, length(resfit))
	
	}
		
	names(res_se)<-names(resfit)
	
	resul$se<-res_se
	
	resul$parametersfixed<-parametersfixed
	
	resul$method_incertitude<-method_incertitude
	
	resul$zero_counts<-zero_counts
	
	resul$data<-data
	
		
for(kl in 1:length(res_se)) {
	if (is.na(res_se[kl])) {
		cat(paste(names(resfit[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE= NaN\n", sep=""))
	} else {
		cat(paste(names(resfit[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE=", format(res_se[kl], , digits=max(3, trunc(log10(res_se[kl]))+4)), "\n", sep=""))
	}
}

dtout <- list()

for (kl in 1:length(resul$data)) {

cat(paste("Series: ", names(resul$data[kl]), "\n", sep=""))

# la date de référence est resul$data[[kl]][1, "Date"]-resul$data[[kl]][1, "ordinal"]+1
ref <- resul$data[[kl]][1, "Date"]-resul$data[[kl]][1, "ordinal"]+1
intdtout <- c(reference=ref)

par <- .format_par(c(resfit, parametersfixed), names(resul$data[kl]))
sepfixed <- parametersfixed[strtrim(names(parametersfixed), 3)=="sd#"]
names(sepfixed) <- substring(names(sepfixed), 4)
se <- c(res_se, sepfixed)

d1 <- ref+par["Peak"]
cat(paste("Peak: ", d1, "\n", sep=""))
intdtout <- c(intdtout, Peak=as.numeric(d1))
if (!is.na(se["Peak"])) {
	d2 <- d1-2*se["Peak"]
	d3 <- d1+2*se["Peak"]
	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, PeakCI1=as.numeric(d2), PeakCI2=as.numeric(d3))
} else {
	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, PeakCI1=NA, PeakCI2=NA)
}

d1 <- ref+par["Begin"]
cat(paste("Begin: ", d1, "\n", sep=""))
intdtout <- c(intdtout, Begin=as.numeric(d1))
# pour l'intervalle de confiance, il faut modifier soit directement
# Begin
# Peak Length
# Peak LengthB
d2 <- NULL
d3 <- NULL
if (!is.na(se["Begin"])) {
	d2 <- ref+par["Begin"]-2*se["Begin"]
	d3 <- ref+par["Begin"]+2*se["Begin"]
} else {
	sel <- 0
	l <- NA
	if (!is.na(se["Length"])) {
		l <- par["Length"]
		sel <- se["Length"]
	} else {
		if (!is.na(se["LengthB"])) {
			l <- par["LengthB"]
			sel <- se["LengthB"]
		}
	}
	if (!is.na(se["Peak"])) {
		d2 <- ref+par["Peak"]-2*se["Peak"]-l-2*sel
		d3 <- ref+par["Peak"]+2*se["Peak"]-l+2*sel
	} else {
		d2 <- ref+par["Peak"]-l-2*sel
		d3 <- ref+par["Peak"]-l+2*sel
	}
}
if (!is.null(d2)) {
	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, BeginCI1=as.numeric(d2), BeginCI2=as.numeric(d3))
} else {
	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, BeginCI1=NA, BeginCI2=NA)
}


d1 <- ref+par["End"]
cat(paste("End: ", d1, "\n", sep=""))
intdtout <- c(intdtout, End=as.numeric(d1))
# pour l'intervalle de confiance, il faut modifier soit directement
# End
# Peak Length
# Peak LengthE
d2 <- NULL
d3 <- NULL
if (!is.na(se["End"])) {
	d2 <- ref+par["End"]-2*se["End"]
	d3 <- ref+par["End"]+2*se["End"]
} else {
	sel <- 0
	l <- NA
	if (!is.na(se["Length"])) {
		l <- par["Length"]
		sel <- se["Length"]
	} else {
		if (!is.na(se["LengthE"])) {
			l <- par["LengthE"]
			sel <- se["LengthE"]
		}
	}
	if (!is.na(se["Peak"])) {
		d2 <- ref+par["Peak"]-2*se["Peak"]+l-2*sel
		d3 <- ref+par["Peak"]+2*se["Peak"]+l+2*sel
	} else {
		d2 <- ref+par["Peak"]+l-2*sel
		d3 <- ref+par["Peak"]+l+2*sel
	}
}
if (!is.null(d2)) {
	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, EndCI1=as.numeric(d2), EndCI2=as.numeric(d3))
} else {
	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, EndCI1=NA, EndCI2=NA)
}

dtout <- c(dtout, list(intdtout))

}

names(dtout) <- names(resul$data)

resul$Dates <- dtout

class(resul) <- "phenology"
	
cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
cat(paste("Parameters=", format(length(resul$par), digits=max(3, trunc(log10(length(resul$par)))+4)), "\n", sep=""))
cat(paste("AIC=", format(2*resul$value+2*length(resul$par), digits=max(3, trunc(log10(2*resul$value+2*length(resul$par)))+4)), "\n", sep=""))

	
growlnotify('Fit is done!')
	

return(resul)

}
	
}
