% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exports.R
\name{par_apply}
\alias{par_apply}
\title{Run a Task in Parallel}
\usage{
par_apply(backend = NULL, x, margin, fun, ...)
}
\arguments{
\item{backend}{An object of class \code{\link{Backend}} as returned by the
\code{\link[=start_backend]{start_backend()}} function. It can also be \code{NULL} to run the task
sequentially via \code{\link[base:apply]{base::apply()}}. The default value is \code{NULL}.}

\item{x}{An array to pass to the \code{fun} function.}

\item{margin}{A numeric vector indicating the dimensions of \code{x} the
\code{fun} function should be applied over. For example, for a matrix,
\code{margin = 1} indicates applying \code{fun} rows-wise, \code{margin = 2}
indicates applying \code{fun} columns-wise, and \code{margin = c(1, 2)}
indicates applying \code{fun} element-wise. Named dimensions are also
possible depending on \code{x}. See \code{\link[parallel:clusterApply]{parallel::parApply()}} and
\code{\link[base:apply]{base::apply()}} for more details.}

\item{fun}{A function to apply to \code{x} according to the \code{margin}.}

\item{...}{Additional arguments to pass to the \code{fun} function.}
}
\value{
The dimensions of the output vary according to the \code{margin} argument. Consult
the documentation of \code{\link[base:apply]{base::apply()}} for a detailed explanation on how the
output is structured.
}
\description{
This function can be used to run a task in parallel. The task is executed in
parallel on the specified backend, similar to \code{\link[parallel:clusterApply]{parallel::parApply()}}. If
\code{backend = NULL}, the task is executed sequentially using \code{\link[base:apply]{base::apply()}}.
See the \strong{Details} section for more information on how this function works.
}
\details{
This function uses the \code{\link{UserApiConsumer}} class that acts like an
interface for the developer API of the \code{\link{parabar}} package.
}
\examples{
\donttest{

# Define a simple task.
task <- function(x) {
    # Perform computations.
    Sys.sleep(0.01)

    # Return the result.
    mean(x)
}

# Define a matrix for the task.
x <- matrix(rnorm(100^2, mean = 10, sd = 0.5), nrow = 100, ncol = 100)

# Start an asynchronous backend.
backend <- start_backend(cores = 2, cluster_type = "psock", backend_type = "async")

# Run a task in parallel over the rows of `x`.
results <- par_apply(backend, x = x, margin = 1, fun = task)

# Run a task in parallel over the columns of `x`.
results <- par_apply(backend, x = x, margin = 2, fun = task)

# The task can also be run over all elements of `x` using `margin = c(1, 2)`.
# Improper dimensions will throw an error.
try(par_apply(backend, x = x, margin = c(1, 2, 3), fun = task))

# Disable progress tracking.
set_option("progress_track", FALSE)

# Run a task in parallel.
results <- par_apply(backend, x = x, margin = 1, fun = task)

# Enable progress tracking.
set_option("progress_track", TRUE)

# Change the progress bar options.
configure_bar(type = "modern", format = "[:bar] :percent")

# Run a task in parallel.
results <- par_apply(backend, x = x, margin = 1, fun = task)

# Stop the backend.
stop_backend(backend)

# Start a synchronous backend.
backend <- start_backend(cores = 2, cluster_type = "psock", backend_type = "sync")

# Run a task in parallel.
results <- par_apply(backend, x = x, margin = 1, fun = task)

# Disable progress tracking to remove the warning that progress is not supported.
set_option("progress_track", FALSE)

# Run a task in parallel.
results <- par_apply(backend, x = x, margin = 1, fun = task)

# Stop the backend.
stop_backend(backend)

# Run the task using the `base::lapply` (i.e., non-parallel).
results <- par_apply(NULL, x = x, margin = 1, fun = task)

}

}
\seealso{
\code{\link[=start_backend]{start_backend()}}, \code{\link[=peek]{peek()}}, \code{\link[=export]{export()}},
\code{\link[=evaluate]{evaluate()}}, \code{\link[=clear]{clear()}}, \code{\link[=configure_bar]{configure_bar()}},
\code{\link[=par_sapply]{par_sapply()}}, \code{\link[=par_lapply]{par_lapply()}}, \code{\link[=stop_backend]{stop_backend()}},
\code{\link[=set_option]{set_option()}}, \code{\link[=get_option]{get_option()}}, \code{\link{Options}},
\code{\link{UserApiConsumer}}, and \code{\link{Service}}.
}
