% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/openSTARS.R
\docType{package}
\name{openSTARS}
\alias{openSTARS}
\title{openSTARS: An Open Source Implementation of the 'ArcGIS' Toolbox 'STARS'.}
\description{
openSTARS provides functions to prepare data so that it can be imported by
the \code{\link[SSN]{SSN}} package for spatial modelling on stream networks.
'GRASS GIS 7.0' (or greater) with installed addons r.stream.basins,
r.stream.distance, r.stream.order, and r.hydrodem is needed.
}
\section{obs_data.csv}{

Artificial observation data with arbitrary measurements.
}

\section{elev_ned30m.tif}{

South-West Wake county National Elevation Data 30m.
}

\section{sites_nc.shp}{

Arbitrary sites along rivers in North Carolina.
}

\section{streams.shp}{

Rivers in North Carolina.
}

\section{geology.shp}{

Geological data.
}

\section{landuse_r.tif}{

Land use date in North Carolina.
}

\section{lakes.shp}{

Artificial lakes (not at topologically correct locations)
}

\section{pointsources.shp}{

Artificial point sources.
}

\examples{
\donttest{
# Initiate and setup GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
if(.Platform$OS.type == "windows"){
  grass_program_path = "c:/Program Files/GRASS GIS 7.6"
  } else {
  grass_program_path = "/usr/lib/grass78/"
  }

setup_grass_environment(dem = dem_path, 
                        gisBase = grass_program_path,      
                        remove_GISRC = TRUE,
                        override = TRUE
                        )
gmeta()

# Load files into GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
sites_path <- system.file("extdata", "nc", "sites_nc.shp", package = "openSTARS")
streams_path <- system.file("extdata", "nc", "streams.shp", package = "openSTARS")
preds_v_path <- system.file("extdata", "nc", "pointsources.shp", package = "openSTARS")
preds_r_path <- system.file("extdata", "nc", "landuse_r.tif", package = "openSTARS")

import_data(dem = dem_path, sites = sites_path, streams = streams_path,
            predictor_vector = preds_v_path, predictor_raster = preds_r_path)

# Derive streams from DEM
# burn in 'streams' 10 meters
derive_streams(burn = 10, accum_threshold = 700, condition = TRUE, clean = TRUE)

# Check and correct complex confluences (there are no complex confluences in this
# example date set; set accum_threshold in derive_streams to a smaller value
# to create complex confluences)
cj <- check_compl_confluences()
if(cj){
  correct_compl_confluences()
}

# calculate slope as potential predictor
execGRASS("r.slope.aspect", flags = c("overwrite","quiet"),
          parameters = list(
            elevation = "dem",
            slope = "slope"
          ))

# Prepare edges
calc_edges()
calc_attributes_edges(input_raster = c("slope", "landuse_r"),
                      stat_rast = c("max", "percent"),
                      attr_name_rast = c("maxSlo", "luse"),
                      input_vector = "pointsources", stat_vect = "count",
                      attr_name_vect = "psource")

# Prepare site
calc_sites()

# Usually, only one of the following methods is needed. The exact one takes
# longer to run
# approximate potential predictor variables for each site based on edge values
calc_attributes_sites_approx(input_attr_name = c("maxSlo", "lusep_1", "lusep_2",
                                                 "lusep_3", "lusep_4", "lusep_5",
                                                 "lusep_6", "lusep_7"),
                             output_attr_name = c("maxSloA","luse1A", "luse2A",
                                                  "luse_3A", "luse4A", "luse5A",
                                                  "luse6A", "luse7A"),
                             stat = c("max", rep("percent", 7)))

# exact potential predictor variables for each site based on catchments
calc_attributes_sites_exact(input_raster = c("slope", "landuse_r"),
                            attr_name_rast = c("maxSloEx", "luseE"),
                            stat_rast = c("max", "percent"))

# Plot data
library(sp)
dem <- readRAST("dem", ignore.stderr = TRUE, plugin = FALSE)
sites <- readVECT("sites", ignore.stderr = TRUE)
sites_orig <-  readVECT("sites_o", ignore.stderr = TRUE)
edges <- readVECT("edges", ignore.stderr = TRUE)
plot(dem, col = terrain.colors(20))
lines(edges, col = "blue")
points(sites_orig, pch = 4)
cols <- colorRampPalette(c("blue", "red"))(length(sites$H2OArea))[rank(sites$H2OArea)]
points(sites, pch = 16, col = cols)

# Write data to SSN Folder
ssn_dir <- file.path(tempdir(), "nc.ssn")
export_ssn(ssn_dir, delete_directory = TRUE)

# Check if all files are ok
library(SSN)
check_ssn(ssn_dir)

# Load into SSN-package
ssn_obj <- importSSN(ssn_dir, o.write = TRUE)
print(ssn_obj)
}


#Datasets shipped with openSTARS

}
