% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{ctdDecimate}
\alias{ctdDecimate}
\title{Decimate a CTD profile}
\usage{
ctdDecimate(x, p = 1, method = "boxcar", rule = 1, e = 1.5,
  debug = getOption("oceDebug"))
}
\arguments{
\item{x}{A \code{ctd} object, i.e. one inheriting from \code{\link{ctd-class}}.}

\item{p}{pressure increment, or vector of pressures.  In the first case,
pressures from 0dbar to the rounded maximum pressure are used, incrementing by
\code{p} dbars.  If a vector of pressures is given, interpolation is done to
these pressures.}

\item{method}{the method to be used for calculating decimated values.  This may
be a function or a string naming a built-in method.  The built-in methods are
\code{"boxcar"} (based on a local average), \code{"approx"} (based on linear
interpolation between neighboring points, using \code{\link{approx}}
with the \code{rule} argument specified here), \code{"lm"} (based on local
regression, with \code{e} setting the size of the local region), \code{"rr"}
(for the Reiniger and Ross method, carried out with \code{\link{oce.approx}})
and \code{"unesco"} (for the UNESCO method, carried out with.
\code{\link{oce.approx}}.  If \code{method} is a function, then it must take
three arguments, the first being pressure, the second being an arbitrary
variable in another column of the data, and the third being a vector of target
pressures at which the calculation is carried out, and the return value must be
a vector.  See \dQuote{Examples}.}

\item{rule}{an integer that is passed to \code{\link{approx}}, in the
case where \code{method} is \code{"approx"}. Note that the default value
for \code{rule} is 1, which will inhibit extrapolation beyond the observed
pressure range. This is a change from the behaviour previous to May 8, 2017,
when a \code{rule} of 2 was used (without stating so as an argument).}

\item{e}{is an expansion coefficient used to calculate the local neighbourhoods
for the \code{"boxcar"} and \code{"lm"} methods.  If \code{e=1}, then the
neighbourhood for the i-th pressure extends from the (\code{i-1})-th pressure to
the (\code{i+1})-th pressure.  At the endpoints it is assumed that the outside
bin is of the same pressure range as the first inside bin.  For other values of
\code{e}, the neighbourhood is expanded linearly in each direction.  If the
\code{"lm"} method produces warnings about "prediction from a rank-deficient
fit", a larger value of \code{"e"} should be used.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
An object of \code{\link{ctd-class}}, with pressures that are as set by
the \code{"p"} parameter and all other properties modified appropriately.
}
\description{
Interpolate a CTD profile to specified pressure values.
}
\details{
The \code{"approx"} method is best for bottle data, in which the usual task is
to interpolate from a coarse sampling grid to a finer one. For CTD data, the
\code{"boxcar"} method is the more common choice, because the task is normally
to sub-sample, and some degree of smoothing is usually desired.  (The
\code{"lm"} method is quite slow, and the results are similar to those of the
boxcar method.)

Note that a sort of numerical cabeling effect can result from this procedure,
but it can be avoided as follows

\preformatted{
xd <- ctdDecimate(x)
xd[["sigmaTheta"]] <- swSigmaTheta(xd[["salinity"]],xd[["temperature"]],xd[["pressure"]])
}
}
\section{A note about flags}{

Data-quality flags contained within the original object are ignored by this
function, and the returned value contains no such flags.  This is because such
flags represent an assessment of the original data, not of quantities derived
from those data.  This function produces a warning to this effect. The
recommended practice is to use \code{\link{handleFlags}} or some other means to
deal with flags before calling the present function.
}

\examples{
library(oce)
data(ctd)
plotProfile(ctd, "salinity", ylim=c(10, 0))
p <- seq(0, 45, 1)
ctd2 <- ctdDecimate(ctd, p=p)
lines(ctd2[["salinity"]], ctd2[["pressure"]], col="blue")
p <- seq(0, 45, 1)
ctd3 <- ctdDecimate(ctd, p=p, method=function(x, y, xout)
                    predict(smooth.spline(x, y, df=30), xout)$y)
lines(ctd3[["salinity"]], ctd3[["pressure"]], col="red")


}
\references{
R.F. Reiniger and C.K. Ross, 1968.  A method of interpolation with
application to oceanographic data.  \emph{Deep Sea Research}, \bold{15},
185-193.
}
\seealso{
The documentation for \code{\link{ctd-class}} explains the structure of
CTD objects, and also outlines the other functions dealing with them.

Other things related to \code{ctd} data: \code{\link{[[,ctd-method}},
  \code{\link{[[<-,ctd-method}}, \code{\link{as.ctd}},
  \code{\link{cnvName2oceName}}, \code{\link{ctd-class}},
  \code{\link{ctdFindProfiles}}, \code{\link{ctdRaw}},
  \code{\link{ctdTrim}}, \code{\link{ctd}},
  \code{\link{handleFlags,ctd-method}},
  \code{\link{oceNames2whpNames}},
  \code{\link{oceUnits2whpUnits}},
  \code{\link{plot,ctd-method}}, \code{\link{plotProfile}},
  \code{\link{plotScan}}, \code{\link{plotTS}},
  \code{\link{read.ctd.itp}}, \code{\link{read.ctd.odf}},
  \code{\link{read.ctd.sbe}},
  \code{\link{read.ctd.woce.other}},
  \code{\link{read.ctd.woce}}, \code{\link{read.ctd}},
  \code{\link{subset,ctd-method}},
  \code{\link{summary,ctd-method}},
  \code{\link{woceNames2oceNames}},
  \code{\link{woceUnit2oceUnit}}, \code{\link{write.ctd}}
}
\author{
Dan Kelley
}
