#
# This CLI application allows to engage in text-based conversations
# by interacting with ChatGPT.
# You can send messages or questions to the chatbot via the command line
# interface (CLI) and receive responses generated by the AI model.
# Additionally, you have the flexibility to adjust various parameters
# that govern the behavior of the chatbot.
#


# ---- settings -----

settings <- new.env()
settings$model <- "gpt-3.5-turbo"
settings$temperature <- 0.7
settings$n <- 1
settings$max_tokens <- 50
settings$presence_penalty <- 0
settings$frequency_penalty <- 0


# ---- helpers ----

display_help <- function() {
  cli::cat_line("Available commands:", col = "green")
  cli::cat_line(" /model new_model_name")
  cli::cat_line(" /temperature new_temperature_value")
  cli::cat_line(" /n new_n_value")
  cli::cat_line(" /max_tokens new_max_tokens_value")
  cli::cat_line(" /presence_penalty new_presence_penalty_value")
  cli::cat_line(" /frequency_penalty new_frequency_penalty_value")
  cli::cat_line(" /settings - display current settings")
  cli::cat_line(" /help - print this information")
  cli::cat_line(cli::style_bold(" /quit or /exit - terminate program"))
  cli::cat_line()
}

# dump `settings` to console
display_settings <- function() {
  n_size <- max(nchar(names(settings))) + 1
  fmt <- paste0(" %-", n_size, "s: %s")

  cli::cat_line("Current settings:", col = "green")
  for (n in names(settings)) {
    cli::cat_line(sprintf(fmt, n, settings[[n]]))
  }
  cli::cat_line()
}

# test if the string is a command
is_cmd <- function(txt) substr(txt, 1, 1) == "/"

# process command string
exec_cmd <- function(txt) {
  cmd <- unlist(strsplit(substring(txt, 2), "\\s+", perl = TRUE))
  switch(cmd[1],
    # display info
    help = display_help(),
    settings = display_settings(),

    # settings (integers)
    n =,
    max_tokens =
    {
      settings[[cmd[1]]] <- as.integer(cmd[2])
      display_settings()
    },
    # settings (floats)
    temperature =,
    presence_penalty =,
    frequency_penalty =
    {
      settings[[cmd[1]]] <- as.double(cmd[2])
      display_settings()
    },

    # default
    {
     cli::cat_line("Unknown command '", cmd[1], "'", col = "red")
     display_help()
    }
  )
}


# ---- main ----

# display help
display_help()

# start conversation
dialog <- NULL
repeat {
  # get text from user
  text <- ""
  while(!nchar(text)) {
    text <- readline(cli::col_blue("you: "))
  }

  # check if the text is a command, if so, process it
  if (is_cmd(text)) {
    if (text %in% c("/exit", "/quit")) return(invisible(NULL))
    exec_cmd(text)
    next
  }
  # create dialog data.frame from `text`
  dialog_user <- oaii::dialog_df(text)

  # send request
  log <- utils::capture.output({
    res_content <- oaii::chat_request(
      api_key,
      oaii::merge_dialog_df(dialog, dialog_user),
      model = settings$model,
      temperature = settings$temperature,
      n = settings$n,
      max_tokens = settings$max_tokens,
      presence_penalty = settings$presence_penalty,
      frequency_penalty = settings$frequency_penalty
    )
  }, type = "message")

  # process response
  if (oaii::is_error(res_content)) {
    # display all logs
    for (n in seq_along(log)) {
      cli::cat_line(log[n], col = "red")
    }
  }
  else {
    # fetch answer(s) from response as data.frame
    dialog_ai <- oaii::chat_fetch_messages(res_content)
    # save conversation
    dialog <- oaii::merge_dialog_df(dialog, dialog_user, dialog_ai)
    # display answers
    for(n in seq(NROW(dialog_ai))) {
      cat(cli::col_blue("ai: "))
      cli::cat_line(dialog_ai[n, "content"])
    }
  }
}
